/***************************************************************************
 *                                                                         *
 * Project:  norGIS ALKIS Import                                           *
 * Purpose:  SQL-Funktionen für ALKIS                                      *
 * Author:   Jürgen E. Fischer <jef@norbit.de>                             *
 *                                                                         *
 ***************************************************************************
 * Copyright (c) 2012-2023, Jürgen E. Fischer <jef@norbit.de>              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

CREATE FUNCTION pg_temp.alkis_set_schema(t TEXT) RETURNS varchar AS $$
DECLARE
	i integer;
BEGIN
	IF NOT EXISTS (SELECT 1 FROM pg_namespace WHERE nspname=t) THEN
		BEGIN
			EXECUTE 'CREATE SCHEMA ' || quote_ident(t);
			RAISE NOTICE 'Schema % angelegt.', t;
		EXCEPTION WHEN duplicate_schema OR unique_violation THEN
			-- skip
		END;
	END IF;

	PERFORM set_config('search_path', quote_ident(t) || ', ' || current_setting('search_path'), false);

	IF t <> current_schema() THEN
		RAISE EXCEPTION 'Nicht in Schema % gewechselt.', t;
	END IF;

	RETURN 'Aktuelles Schema ' || t || '.';
END
$$ LANGUAGE plpgsql;

SET search_path = :"parent_schema", :"postgis_schema", public;
SELECT pg_temp.alkis_set_schema(:'alkis_schema');

-- Table/View/Sequence löschen, wenn vorhanden
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_dropobject(t TEXT) RETURNS varchar AS $$
DECLARE
	c RECORD;
	s varchar;
	r varchar;
	d varchar;
	i integer;
	tn varchar;
BEGIN
	-- drop objects
	FOR c IN SELECT relkind,relname
		FROM pg_catalog.pg_class
		JOIN pg_catalog.pg_namespace ON pg_class.relnamespace=pg_namespace.oid
		WHERE pg_catalog.pg_namespace.nspname=current_schema() AND pg_class.relname=t
		ORDER BY relkind
	LOOP
		IF c.relkind = 'v' THEN
			r := alkis_string_append(r, 'Sicht ' || c.relname || ' gelöscht.');
			EXECUTE 'DROP VIEW ' || c.relname || ' CASCADE';
		ELSIF c.relkind = 'r' THEN
			r := alkis_string_append(r, 'Tabelle ' || c.relname || ' gelöscht.');
			EXECUTE 'DROP TABLE ' || c.relname || ' CASCADE';
		ELSIF c.relkind = 'S' THEN
			r := alkis_string_append(r, 'Sequenz ' || c.relname || ' gelöscht.');
			EXECUTE 'DROP SEQUENCE ' || c.relname;
		ELSIF c.relkind <> 'i' THEN
			r := alkis_string_append(r, 'Typ ' || c.table_type || '.' || c.table_name || ' unerwartet.');
		END IF;
	END LOOP;

	FOR c IN SELECT indexname FROM pg_catalog.pg_indexes WHERE schemaname=current_schema() AND indexname=t
	LOOP
		r := alkis_string_append(r, 'Index ' || c.indexname || ' gelöscht.');
		EXECUTE 'DROP INDEX ' || c.indexname;
	END LOOP;

	FOR c IN SELECT proname,proargtypes
		FROM pg_catalog.pg_proc
		JOIN pg_catalog.pg_namespace ON pg_proc.pronamespace=pg_namespace.oid
		WHERE pg_namespace.nspname=current_schema() AND pg_proc.proname=t
	LOOP
		r := alkis_string_append(r, 'Funktion ' || c.proname || ' gelöscht.');

		s := 'DROP FUNCTION ' || c.proname || '(';
		d := '';

		FOR i IN array_lower(c.proargtypes,1)..array_upper(c.proargtypes,1) LOOP
			SELECT typname INTO tn FROM pg_catalog.pg_type WHERE oid=c.proargtypes[i];
			s := s || d || tn;
			d := ',';
		END LOOP;

		s := s || ')';

		EXECUTE s;
	END LOOP;

	FOR c IN SELECT relname,conname
		FROM pg_catalog.pg_constraint
		JOIN pg_catalog.pg_class ON pg_constraint.conrelid=pg_class.oid
		JOIN pg_catalog.pg_namespace ON pg_constraint.connamespace=pg_namespace.oid
		WHERE pg_namespace.nspname=current_schema() AND pg_constraint.conname=t
	LOOP
		r := alkis_string_append(r, 'Constraint ' || c.conname || ' von ' || c.relname || ' gelöscht.');
		EXECUTE 'ALTER TABLE ' || c.relname || ' DROP CONSTRAINT ' || c.conname;
	END LOOP;

	FOR c IN SELECT tgname,relname
		FROM pg_catalog.pg_trigger
		JOIN pg_catalog.pg_class ON pg_trigger.tgrelid=pg_class.oid
		JOIN pg_catalog.pg_namespace ON pg_class.relnamespace=pg_namespace.oid
		WHERE pg_namespace.nspname=current_schema() AND pg_trigger.tgname=t
	LOOP
		r := alkis_string_append(r, 'Trigger ' || c.tgname || ' von ' || c.relname || ' gelöscht.');
		EXECUTE 'DROP TRIGGER ' || c.tgname || ' ON ' || c.relname;
	END LOOP;

	RETURN r;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION :"parent_schema".alkis_string_append(varchar, varchar) RETURNS varchar AS $$
	SELECT CASE WHEN $1='' OR $1 LIKE E'%\n' THEN $1 ELSE coalesce($1||E'\n','') END || coalesce($2, '');
$$ LANGUAGE 'sql' IMMUTABLE;

-- Alle ALKIS-Tabellen löschen
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_drop() RETURNS varchar AS $$
DECLARE
	c RECORD;
	r VARCHAR;
BEGIN
	-- drop tables & views
	FOR c IN SELECT table_type,table_name FROM information_schema.tables
		   WHERE table_schema=current_schema() AND table_type='BASE TABLE'
		     AND ( table_name IN ('aa_advstandardmodell','aa_aktivitaet','aa_anlassart','aa_anlassart_benutzungsauftrag','aa_antrag','aa_antragsgebiet','aa_art_themendefinition','aa_besonderemeilensteinkategorie','aa_gefuehrteobjektart','aa_meilenstein','aa_nas_ausgabeform','aa_projektsteuerung','aa_themendimension','aa_vorgang','aa_weiteremodellart','ap_darstellung','ap_dateityp_3d','ap_fpo','ap_horizontaleausrichtung','ap_kpo_3d','ap_lpo','ap_lto','ap_ppo','ap_pto','ap_vertikaleausrichtung','au_koerperobjekt_3d','au_mehrfachlinienobjekt_3d','au_punkthaufenobjekt_3d','au_trianguliertesoberflaechenobjekt_3d','au_umringobjekt_3d','ax_abbaugut_bergbaubetrieb','ax_abbaugut_tagebaugrubesteinbruch','ax_abgabeversion','ax_abgeleitetehoehenlinie','ax_abschlussflaeche3d','ax_abschnitt','ax_administrative_funktion','ax_anderefestlegungnachstrassenrecht','ax_anderefestlegungnachwasserrecht','ax_anrede_person','ax_anschrift','ax_anzahlderstreckengleise','ax_archaeologischertyp_historischesbauwerkoderhistorischee','ax_art_adressat_auszug','ax_art_baublock','ax_art_bauwerkspunkt','ax_art_bereichzeitlich','ax_art_dammwalldeich','ax_art_einrichtungenfuerdenschiffsverkehr','ax_art_einrichtunginoeffentlichenbereichen','ax_art_flaeche3d','ax_art_flugverkehr','ax_art_flugverkehrsanlage','ax_art_gebaeudepunkt','ax_art_gewaessermerkmal','ax_art_gleis','ax_art_heilquellegasquelle','ax_art_punkt3d','ax_art_punktkennung','ax_art_punktwolke3d','ax_art_reservierung','ax_art_schifffahrtsliniefaehrverkehr','ax_art_strassenverkehrsanlage','ax_art_strukturlinie3d','ax_art_verband','ax_art_verbandsgemeinde','ax_art_wegpfadsteig','ax_artderbebauung_flaechebesondererfunktionalerpraegung','ax_artderbebauung_flaechegemischternutzung','ax_artderbebauung_siedlungsflaeche','ax_artderbebauung_wohnbauflaeche','ax_artderfestlegung_anderefestlegungnachstrassenrecht','ax_artderfestlegung_anderefestlegungnachwasserrecht','ax_artderfestlegung_bauraumoderbodenordnungsrecht','ax_artderfestlegung_denkmalschutzrecht','ax_artderfestlegung_forstrecht','ax_artderfestlegung_klassifizierungnachstrassenrecht','ax_artderfestlegung_klassifizierungnachwasserrecht','ax_artderfestlegung_naturumweltoderbodenschutzrecht','ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc','ax_artderfestlegung_schutzgebietnachwasserrecht','ax_artderfestlegung_sonstigesrecht','ax_artderflurstuecksgrenze_besondereflurstuecksgrenze','ax_artdergebietsgrenze_gebietsgrenze','ax_artdergewaesserstationierungsachse','ax_artderrechtsgemeinschaft_namensnummer','ax_artdesnullpunktes_nullpunkt','ax_artdespolders','ax_ast','ax_aufnahmepunkt','ax_ausgabemedium_benutzer','ax_auspraegung','ax_bahnhofskategorie_bahnverkehrsanlage','ax_bahnkategorie','ax_bahnkategorie_bahnstrecke','ax_bahnkategorie_bahnverkehrsanlage','ax_bahnkategorie_gleis','ax_bahnkategorie_seilbahnschwebebahn','ax_bahnstrecke','ax_bahnverkehr','ax_bahnverkehrsanlage','ax_bauart_bauteil','ax_baublock','ax_bauraumoderbodenordnungsrecht','ax_bauraumoderbodenordnungsrechtgrundbuch','ax_bauteil','ax_bauteil3d','ax_bauweise_gebaeude','ax_bauwerk3d','ax_bauwerkimgewaesserbereich','ax_bauwerkimverkehrsbereich','ax_bauwerkoderanlagefuerindustrieundgewerbe','ax_bauwerkoderanlagefuersportfreizeitunderholung','ax_bauwerksfunktion_bauwerkimgewaesserbereich','ax_bauwerksfunktion_bauwerkimverkehrsbereich','ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer','ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde','ax_bauwerksfunktion_leitung','ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun','ax_bauwerksfunktion_transportanlage','ax_bauwerksfunktion_turm','ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk','ax_bedeutung_grablochderbodenschaetzung','ax_befahrbarkeit_fahrwegachse','ax_befestigung_fahrwegachse','ax_befestigung_wegpfadsteig','ax_behoerde','ax_bemerkungzurabmarkung_grenzpunkt','ax_benutzer','ax_benutzergruppemitzugriffskontrolle','ax_benutzergruppenba','ax_berechnungsmethode','ax_bergbaubetrieb','ax_beschaffenheit_besonderegebaeudelinie','ax_besonderefahrstreifen','ax_besondereflurstuecksgrenze','ax_besonderefunktion_forstrecht','ax_besonderegebaeudelinie','ax_besondererbauwerkspunkt','ax_besonderergebaeudepunkt','ax_besonderertopographischerpunkt','ax_besondereverkehrsbedeutung','ax_bewertung','ax_bewuchs_vegetationsmerkmal','ax_bezeichnung_verwaltungsgemeinschaft','ax_bezugspunktdach','ax_blattart_buchungsblatt','ax_blattart_historischesflurstueck','ax_bodenart_bodenschaetzung','ax_bodenart_musterundvergleichsstueck','ax_bodenflaeche3d','ax_bodenschaetzung','ax_bodenstufe','ax_boeschungkliff','ax_boeschungsflaeche','ax_buchungsart_anlieger','ax_buchungsart_buchungsstelle','ax_buchungsblatt','ax_buchungsblattbezirk','ax_buchungsstelle','ax_bundesland','ax_dachflaeche3d','ax_dachform','ax_dachgeschossausbau_gebaeude','ax_dammwalldeich','ax_darstellung_gebaeudeausgestaltung','ax_dateitypraster_textur3d','ax_datenerhebung','ax_datenerhebung_punktort','ax_datenerhebung_schwere','ax_datenerhebung3d','ax_datenformat_benutzer','ax_denkmalschutzrecht','ax_dhmgitter','ax_dienststelle','ax_duene','ax_eigentuemerart_namensnummer','ax_einrichtungenfuerdenschiffsverkehr','ax_einrichtunginoeffentlichenbereichen','ax_einschnitt','ax_elektrifizierung','ax_entstehungsart','ax_ergebnisderueberpruefung_bauwerkeeinrichtungenundsonsti','ax_ergebnisderueberpruefung_dammwalldeich','ax_ergebnisderueberpruefung_klassifizierungnachwasserrecht','ax_ergebnisderueberpruefung_tatsaechlichenutzung','ax_fahrbahnachse','ax_fahrbahntrennung_strasse','ax_fahrwegachse','ax_felsenfelsblockfelsnadel','ax_fenster3d','ax_firstlinie','ax_flaeche3d','ax_flaechebesondererfunktionalerpraegung','ax_flaechegemischternutzung','ax_fliessgewaesser','ax_flugverkehr','ax_flugverkehrsanlage','ax_flurstueck','ax_flurstueckgrundbuch','ax_foerdergut_industrieundgewerbeflaeche','ax_forstrecht','ax_fortfuehrungsauftrag','ax_fortfuehrungsfall','ax_fortfuehrungsfallgrundbuch','ax_fortfuehrungsnachweisdeckblatt','ax_friedhof','ax_funktion_bahnverkehr','ax_funktion_bauwerk','ax_funktion_bergbaubetrieb','ax_funktion_dammwalldeich','ax_funktion_einschnitt','ax_funktion_fahrbahnachse','ax_funktion_flaechebesondererfunktionalerpraegung','ax_funktion_flaechegemischternutzung','ax_funktion_fliessgewaesser','ax_funktion_flugverkehr','ax_funktion_friedhof','ax_funktion_gebaeudeinstallation3d','ax_funktion_gehoelz','ax_funktion_gewaesserachse','ax_funktion_hafenbecken','ax_funktion_industrieundgewerbeflaeche','ax_funktion_lagefestpunkt','ax_funktion_meer','ax_funktion_platz','ax_funktion_polder','ax_funktion_referenzstationspunkt','ax_funktion_schiffsverkehr','ax_funktion_schutzgebietnachwasserrecht','ax_funktion_schwerefestpunkt','ax_funktion_siedlungsflaeche','ax_funktion_sportfreizeitunderholungsflaeche','ax_funktion_stehendesgewaesser','ax_funktion_strasse','ax_funktion_strassenachse','ax_funktion_tagebaugrubesteinbruch','ax_funktion_unlandvegetationsloseflaeche','ax_funktion_untergeordnetesgewaesser','ax_funktion_vegetationsmerkmal','ax_funktion_weg','ax_funktion_wegachse','ax_funktion_wohnbauflaeche','ax_gebaeude','ax_gebaeudeausgestaltung','ax_gebaeudefunktion','ax_gebaeudeinstallation3d','ax_gebiet_bundesland','ax_gebiet_kreis','ax_gebiet_nationalstaat','ax_gebiet_regierungsbezirk','ax_gebiet_verwaltungsgemeinschaft','ax_gebietsgrenze','ax_gehoelz','ax_gemarkung','ax_gemarkungsteilflur','ax_gemeinde','ax_gemeindeteil','ax_genauigkeitsstufe_punktort','ax_genauigkeitsstufe_schwere','ax_georeferenziertegebaeudeadresse','ax_gewaesserachse','ax_gewaessermerkmal','ax_gewaesserstationierungsachse','ax_gewann','ax_gleis','ax_gnsstauglichkeit','ax_grablochderbodenschaetzung','ax_grenzpunkt','ax_grenzuebergang','ax_gruendederausgesetztenabmarkung_grenzpunkt','ax_hafen','ax_hafenbecken','ax_hafenkategorie_hafen','ax_halde','ax_heide','ax_heilquellegasquelle','ax_historischesbauwerkoderhistorischeeinrichtung','ax_historischesflurstueck','ax_historischesflurstueckalb','ax_historischesflurstueckohneraumbezug','ax_hoehenfestpunkt','ax_hoehenlinie','ax_hoehleneingang','ax_horizontfreiheit_grenzpunkt','ax_horizontfreiheit_netzpunkt','ax_hydrologischesmerkmal_fliessgewaesser','ax_hydrologischesmerkmal_gewaesserachse','ax_hydrologischesmerkmal_gewaessermerkmal','ax_hydrologischesmerkmal_heilquellegasquelle','ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri','ax_hydrologischesmerkmal_stehendesgewaesser','ax_hydrologischesmerkmal_untergeordnetesgewaesser','ax_impliziteloeschungderreservierung','ax_industrieundgewerbeflaeche','ax_insel','ax_internationalebedeutung_strasse','ax_kanal','ax_klassifikation_hierarchiestufe3d_lagefestpunkt','ax_klassifikation_ordnung_lagefestpunkt','ax_klassifikation_wertigkeit_lagefestpunkt','ax_klassifizierung_bewertung','ax_klassifizierungnachstrassenrecht','ax_klassifizierungnachwasserrecht','ax_kleinraeumigerlandschaftsteil','ax_klimastufe','ax_kommunalesgebiet','ax_kommunalesteilgebiet','ax_kondominium','ax_konstruktionsmerkmalbauart_schleuse','ax_koordinatenstatus_punktort','ax_kreisregion','ax_lagebezeichnungkatalogeintrag','ax_lagebezeichnungmithausnummer','ax_lagebezeichnungmitpseudonummer','ax_lagebezeichnungohnehausnummer','ax_lagefestpunkt','ax_lagergut_halde','ax_lagergut_industrieundgewerbeflaeche','ax_lagezurerdoberflaeche_gebaeude','ax_lagezurerdoberflaeche_transportanlage','ax_lagezurerdoberflaeche_untergeordnetesgewaesser','ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk','ax_lagezuroberflaeche_gleis','ax_landschaft','ax_landschaftstyp','ax_landwirtschaft','ax_leitung','ax_letzteabgabeart','ax_li_processstep_bodenhoehe_description','ax_li_processstep_dachhoehe_source','ax_li_processstep_mitdatenerhebung_description','ax_li_processstep_ohnedatenerhebung_description','ax_li_processstep_punktort_description','ax_li_processstep3d_description','ax_liniendarstellung_topographischelinie','ax_marke','ax_markierung_wegachse','ax_markierung_wegpfadsteig','ax_material_materialeigenschaft_material3d','ax_material3d','ax_meer','ax_merkmal_musterundvergleichsstueck','ax_messmethode_schwere','ax_moor','ax_musterundvergleichsstueck','ax_namensnummer','ax_nationalstaat','ax_naturumweltoderbodenschutzrecht','ax_netzknoten','ax_null_listenelement3d','ax_nullpunkt','ax_nutzung','ax_nutzung_flugverkehr','ax_nutzung_hafen','ax_nutzung_hafenbecken','ax_nutzung_stehendesgewaesser','ax_nutzung_wald','ax_nutzungsart_bodenschaetzung','ax_nutzungsart_musterundvergleichsstueck','ax_obererbezugspunkt','ax_oberflaechenmaterial_flugverkehrsanlage','ax_oberflaechenmaterial_strasse','ax_oberflaechenmaterial_unlandvegetationsloseflaeche','ax_ordnung_hoehenfestpunkt','ax_ordnung_schwerefestpunkt','ax_ortslage','ax_person','ax_personengruppe','ax_platz','ax_polder','ax_primaerenergie_industrieundgewerbeflaeche','ax_produkt_transportanlage','ax_punkt3d','ax_punktkennunguntergegangen','ax_punktkennungvergleichend','ax_punktortag','ax_punktortau','ax_punktortta','ax_punktstabilitaet','ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet','ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung','ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand','ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes','ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra','ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi','ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt','ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili','ax_punktwolke3d','ax_qualitaet_hauskoordinate','ax_rechtszustand_schutzzone','ax_referenzstationspunkt','ax_regierungsbezirk','ax_reservierung','ax_reservierungsart_reservierungsauftrag','ax_rolledergemeinde','ax_schifffahrtskategorie','ax_schifffahrtskategorie_gewaesserstationierungsachse','ax_schifffahrtskategorie_kanal','ax_schifffahrtsliniefaehrverkehr','ax_schiffsverkehr','ax_schleuse','ax_schutzgebietnachnaturumweltoderbodenschutzrecht','ax_schutzgebietnachwasserrecht','ax_schutzzone','ax_schwere','ax_schwereanomalie_schwere_art','ax_schwerebezugssystem_schwere','ax_schwerefestpunkt','ax_schwerestatus_schwere','ax_seilbahnschwebebahn','ax_selektionsmasstab_benutzungsauftrag','ax_sicherungspunkt','ax_sickerstrecke','ax_siedlungsflaeche','ax_skizze','ax_skizzenart_skizze','ax_soll','ax_sonstigeangaben_bodenschaetzung','ax_sonstigeangaben_musterundvergleichsstueck','ax_sonstigervermessungspunkt','ax_sonstigesbauwerkodersonstigeeinrichtung','ax_sonstigesrecht','ax_speicherinhalt_vorratsbehaelterspeicherbauwerk','ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung','ax_sportfreizeitunderholungsflaeche','ax_spurweite','ax_stehendesgewaesser','ax_strasse','ax_strassenachse','ax_strassenverkehr','ax_strassenverkehrsanlage','ax_strukturlinie3d','ax_sumpf','ax_tagebaugrubesteinbruch','ax_tagesabschnitt','ax_tauglichkeit_gcg','ax_testgelaende','ax_textur3d','ax_tidemerkmal_meer','ax_topographischelinie','ax_transportanlage','ax_tuer3d','ax_turm','ax_typ_textur3d','ax_unlandvegetationsloseflaeche','ax_untererbezugspunkt','ax_untergeordnetesgewaesser','ax_ursprung','ax_vegetationsmerkmal','ax_vegetationsmerkmal_gehoelz','ax_vegetationsmerkmal_landwirtschaft','ax_vegetationsmerkmal_wald','ax_verarbeitungsart_einrichtungsauftrag','ax_verarbeitungsart_fortfuehrungsauftrag','ax_verband','ax_verkehrsdienst_bahnstrecke','ax_verkehrsdienst_bahnverkehrsanlage','ax_vertrauenswuerdigkeit_punktort','ax_vertrauenswuerdigkeit_schwere','ax_vertretung','ax_verwaltung','ax_verwaltungsgemeinschaft','ax_verwendeteobjekte_abgeleitetehoehenlinie','ax_verwendeteobjekte_dhmgitter','ax_vorratsbehaelterspeicherbauwerk','ax_wald','ax_wandflaeche3d','ax_wasserlauf','ax_wasserspiegelhoehe','ax_wasserverhaeltnisse','ax_weg','ax_wegpfadsteig','ax_weitere_gebaeudefunktion','ax_weiterenutzung_tatsaechlichenutzung','ax_widmung_kanal','ax_widmung_stehendesgewaesser','ax_widmung_strasse','ax_widmung_wasserlauf','ax_wirtschaftlicheeinheit','ax_wirtschaftsart','ax_wirtschaftsartgrundbuch','ax_wohnbauflaeche','ax_wohnplatz','ax_zone_schutzzone','ax_zustand','ax_zustand_bahnverkehr','ax_zustand_bahnverkehrsanlage','ax_zustand_bauwerkimgewaesserbereich','ax_zustand_bauwerkimverkehrsbereich','ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe','ax_zustand_bauwerkoderanlagefuersportfreizeitunderholung','ax_zustand_bergbaubetrieb','ax_zustand_boeschungkliff','ax_zustand_einrichtungenfuerdenschiffsverkehr','ax_zustand_flaechebesondererfunktionalerpraegung','ax_zustand_flaechegemischternutzung','ax_zustand_flugverkehr','ax_zustand_flugverkehrsanlage','ax_zustand_friedhof','ax_zustand_gebaeude','ax_zustand_gebietsgrenze','ax_zustand_gewaesserachse','ax_zustand_halde','ax_zustand_hoehleneingang','ax_zustand_industrieundgewerbeflaeche','ax_zustand_kanal','ax_zustand_naturumweltoderbodenschutzrecht','ax_zustand_schiffsverkehr','ax_zustand_schleuse','ax_zustand_sportfreizeitunderholungsflaeche','ax_zustand_stehendesgewaesser','ax_zustand_strasse','ax_zustand_tagebaugrubesteinbruch','ax_zustand_turm','ax_zustand_vegetationsmerkmal','ax_zustand_wald','ax_zustand_wohnbauflaeche','ax_zustandsstufe','ci_rolecode','lb_binnengewaesser','lb_blattform_holzigevegetation','lb_eis','lb_eisart_eis','lb_festgestein','lb_fliesseigenschaft_binnengewaesser','lb_gewaesserart_binnengewaesser','lb_hochbau','lb_holzigevegetation','lb_krautigevegetation','lb_lockermaterial','lb_meer','lb_meerart_meer','lb_oberflaechenmaterial_lockermaterial','lb_tiefbau','lb_vegetationsmerkmal_holzigevegetation','lb_vegetationsmerkmal_krautigevegetation','lb_wasserfuehrung_binnengewaesser','lb_wassersaettigung_holzigevegetation','lb_wassersaettigung_krautigevegetation','lb_wassersaettigung_lockermaterial','ln_abbau','ln_abbaugut_abbau','ln_aquakulturundfischereiwirtschaft','ln_art_abbau','ln_art_aquakulturundfischereiwirtschaft','ln_art_forstwirtschaft','ln_art_freiluftundnaherholung','ln_art_freizeitanlage','ln_art_gewerblichedienstleistungen','ln_art_industrieundverarbeitendesgewerbe','ln_art_versorgungundentsorgung','ln_art_wasserwirtschaft','ln_artderbestattungsflaeche_bestattung','ln_artderbetriebsflaeche_landwirtschaft','ln_artdesparkplatzes_strassenundwegeverkehr','ln_bahnverkehr','ln_bestattung','ln_bewirtschaftung_landwirtschaft','ln_flugverkehr','ln_foerdergut_abbau','ln_forstwirtschaft','ln_freiluftundnaherholung','ln_freizeitanlage','ln_funktion_bahnverkehr','ln_funktion_flugverkehr','ln_funktion_kulturundunterhaltung','ln_funktion_lagerung','ln_funktion_oeffentlicheeinrichtungen','ln_funktion_schiffsverkehr','ln_funktion_schutzanlage','ln_funktion_strassenundwegeverkehr','ln_funktion_wasserwirtschaft','ln_fussgaengerzone_strassenundwegeverkehr','ln_gewerblichedienstleistungen','ln_hafenkategorie_schiffsverkehr','ln_industrieundverarbeitendesgewerbe','ln_kulturundunterhaltung','ln_lagergut_lagerung','ln_lagerung','ln_landwirtschaft','ln_nutzung_flugverkehr','ln_nutzung_schiffsverkehr','ln_oberflaeche_lagerung','ln_oeffentlicheeinrichtungen','ln_ohnenutzung','ln_primaerenergie_versorgungundentsorgung','ln_schiffsverkehr','ln_schutzanlage','ln_sportanlage','ln_sportart_sportanlage','ln_strassenundwegeverkehr','ln_typ_schutzanlage','ln_versorgungundentsorgung','ln_wasserwirtschaft','ln_wohnnutzung','ln_zeitlichkeit_wohnnutzung','ln_zuflusssteuerung_schutzanlage','ln_zustand_abbau','ln_zustand_bahnverkehr','ln_zustand_flugverkehr','ln_zustand_gewerblichedienstleistungen','ln_zustand_industrieundverarbeitendesgewerbe','ln_zustand_kulturundunterhaltung','ln_zustand_lagerung','ln_zustand_oeffentlicheeinrichtungen','ln_zustand_schiffsverkehr','ln_zustand_strassenundwegeverkehr','ln_zustand_versorgungundentsorgung','ln_zustand_wasserwirtschaft','ln_zustand_wohnnutzung','nas_filter_capabilities','operation')
			   OR table_name IN ('alkis_beziehungen','delete','alkis_version') )
		   ORDER BY table_type DESC LOOP
		IF c.table_type = 'VIEW' THEN
			r := alkis_string_append(r, 'Sicht ' || c.table_name || ' gelöscht.');
			EXECUTE 'DROP VIEW ' || c.table_name || ' CASCADE';
		ELSIF c.table_type = 'BASE TABLE' THEN
			r := alkis_string_append(r, 'Tabelle ' || c.table_name || ' gelöscht.');
			EXECUTE 'DROP TABLE ' || c.table_name || ' CASCADE';
		ELSE
			r := alkis_string_append(r, 'Typ ' || c.table_type || '.' || c.table_name || ' unerwartet.');
		END IF;
	END LOOP;

	-- clean geometry_columns
	DELETE FROM geometry_columns
		WHERE f_table_schema=current_schema()
		AND ( f_table_name IN ('aa_advstandardmodell','aa_aktivitaet','aa_anlassart','aa_anlassart_benutzungsauftrag','aa_antrag','aa_antragsgebiet','aa_art_themendefinition','aa_besonderemeilensteinkategorie','aa_gefuehrteobjektart','aa_meilenstein','aa_nas_ausgabeform','aa_projektsteuerung','aa_themendimension','aa_vorgang','aa_weiteremodellart','ap_darstellung','ap_dateityp_3d','ap_fpo','ap_horizontaleausrichtung','ap_kpo_3d','ap_lpo','ap_lto','ap_ppo','ap_pto','ap_vertikaleausrichtung','au_koerperobjekt_3d','au_mehrfachlinienobjekt_3d','au_punkthaufenobjekt_3d','au_trianguliertesoberflaechenobjekt_3d','au_umringobjekt_3d','ax_abbaugut_bergbaubetrieb','ax_abbaugut_tagebaugrubesteinbruch','ax_abgabeversion','ax_abgeleitetehoehenlinie','ax_abschlussflaeche3d','ax_abschnitt','ax_administrative_funktion','ax_anderefestlegungnachstrassenrecht','ax_anderefestlegungnachwasserrecht','ax_anrede_person','ax_anschrift','ax_anzahlderstreckengleise','ax_archaeologischertyp_historischesbauwerkoderhistorischee','ax_art_adressat_auszug','ax_art_baublock','ax_art_bauwerkspunkt','ax_art_bereichzeitlich','ax_art_dammwalldeich','ax_art_einrichtungenfuerdenschiffsverkehr','ax_art_einrichtunginoeffentlichenbereichen','ax_art_flaeche3d','ax_art_flugverkehr','ax_art_flugverkehrsanlage','ax_art_gebaeudepunkt','ax_art_gewaessermerkmal','ax_art_gleis','ax_art_heilquellegasquelle','ax_art_punkt3d','ax_art_punktkennung','ax_art_punktwolke3d','ax_art_reservierung','ax_art_schifffahrtsliniefaehrverkehr','ax_art_strassenverkehrsanlage','ax_art_strukturlinie3d','ax_art_verband','ax_art_verbandsgemeinde','ax_art_wegpfadsteig','ax_artderbebauung_flaechebesondererfunktionalerpraegung','ax_artderbebauung_flaechegemischternutzung','ax_artderbebauung_siedlungsflaeche','ax_artderbebauung_wohnbauflaeche','ax_artderfestlegung_anderefestlegungnachstrassenrecht','ax_artderfestlegung_anderefestlegungnachwasserrecht','ax_artderfestlegung_bauraumoderbodenordnungsrecht','ax_artderfestlegung_denkmalschutzrecht','ax_artderfestlegung_forstrecht','ax_artderfestlegung_klassifizierungnachstrassenrecht','ax_artderfestlegung_klassifizierungnachwasserrecht','ax_artderfestlegung_naturumweltoderbodenschutzrecht','ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc','ax_artderfestlegung_schutzgebietnachwasserrecht','ax_artderfestlegung_sonstigesrecht','ax_artderflurstuecksgrenze_besondereflurstuecksgrenze','ax_artdergebietsgrenze_gebietsgrenze','ax_artdergewaesserstationierungsachse','ax_artderrechtsgemeinschaft_namensnummer','ax_artdesnullpunktes_nullpunkt','ax_artdespolders','ax_ast','ax_aufnahmepunkt','ax_ausgabemedium_benutzer','ax_auspraegung','ax_bahnhofskategorie_bahnverkehrsanlage','ax_bahnkategorie','ax_bahnkategorie_bahnstrecke','ax_bahnkategorie_bahnverkehrsanlage','ax_bahnkategorie_gleis','ax_bahnkategorie_seilbahnschwebebahn','ax_bahnstrecke','ax_bahnverkehr','ax_bahnverkehrsanlage','ax_bauart_bauteil','ax_baublock','ax_bauraumoderbodenordnungsrecht','ax_bauraumoderbodenordnungsrechtgrundbuch','ax_bauteil','ax_bauteil3d','ax_bauweise_gebaeude','ax_bauwerk3d','ax_bauwerkimgewaesserbereich','ax_bauwerkimverkehrsbereich','ax_bauwerkoderanlagefuerindustrieundgewerbe','ax_bauwerkoderanlagefuersportfreizeitunderholung','ax_bauwerksfunktion_bauwerkimgewaesserbereich','ax_bauwerksfunktion_bauwerkimverkehrsbereich','ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer','ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde','ax_bauwerksfunktion_leitung','ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun','ax_bauwerksfunktion_transportanlage','ax_bauwerksfunktion_turm','ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk','ax_bedeutung_grablochderbodenschaetzung','ax_befahrbarkeit_fahrwegachse','ax_befestigung_fahrwegachse','ax_befestigung_wegpfadsteig','ax_behoerde','ax_bemerkungzurabmarkung_grenzpunkt','ax_benutzer','ax_benutzergruppemitzugriffskontrolle','ax_benutzergruppenba','ax_berechnungsmethode','ax_bergbaubetrieb','ax_beschaffenheit_besonderegebaeudelinie','ax_besonderefahrstreifen','ax_besondereflurstuecksgrenze','ax_besonderefunktion_forstrecht','ax_besonderegebaeudelinie','ax_besondererbauwerkspunkt','ax_besonderergebaeudepunkt','ax_besonderertopographischerpunkt','ax_besondereverkehrsbedeutung','ax_bewertung','ax_bewuchs_vegetationsmerkmal','ax_bezeichnung_verwaltungsgemeinschaft','ax_bezugspunktdach','ax_blattart_buchungsblatt','ax_blattart_historischesflurstueck','ax_bodenart_bodenschaetzung','ax_bodenart_musterundvergleichsstueck','ax_bodenflaeche3d','ax_bodenschaetzung','ax_bodenstufe','ax_boeschungkliff','ax_boeschungsflaeche','ax_buchungsart_anlieger','ax_buchungsart_buchungsstelle','ax_buchungsblatt','ax_buchungsblattbezirk','ax_buchungsstelle','ax_bundesland','ax_dachflaeche3d','ax_dachform','ax_dachgeschossausbau_gebaeude','ax_dammwalldeich','ax_darstellung_gebaeudeausgestaltung','ax_dateitypraster_textur3d','ax_datenerhebung','ax_datenerhebung_punktort','ax_datenerhebung_schwere','ax_datenerhebung3d','ax_datenformat_benutzer','ax_denkmalschutzrecht','ax_dhmgitter','ax_dienststelle','ax_duene','ax_eigentuemerart_namensnummer','ax_einrichtungenfuerdenschiffsverkehr','ax_einrichtunginoeffentlichenbereichen','ax_einschnitt','ax_elektrifizierung','ax_entstehungsart','ax_ergebnisderueberpruefung_bauwerkeeinrichtungenundsonsti','ax_ergebnisderueberpruefung_dammwalldeich','ax_ergebnisderueberpruefung_klassifizierungnachwasserrecht','ax_ergebnisderueberpruefung_tatsaechlichenutzung','ax_fahrbahnachse','ax_fahrbahntrennung_strasse','ax_fahrwegachse','ax_felsenfelsblockfelsnadel','ax_fenster3d','ax_firstlinie','ax_flaeche3d','ax_flaechebesondererfunktionalerpraegung','ax_flaechegemischternutzung','ax_fliessgewaesser','ax_flugverkehr','ax_flugverkehrsanlage','ax_flurstueck','ax_flurstueckgrundbuch','ax_foerdergut_industrieundgewerbeflaeche','ax_forstrecht','ax_fortfuehrungsauftrag','ax_fortfuehrungsfall','ax_fortfuehrungsfallgrundbuch','ax_fortfuehrungsnachweisdeckblatt','ax_friedhof','ax_funktion_bahnverkehr','ax_funktion_bauwerk','ax_funktion_bergbaubetrieb','ax_funktion_dammwalldeich','ax_funktion_einschnitt','ax_funktion_fahrbahnachse','ax_funktion_flaechebesondererfunktionalerpraegung','ax_funktion_flaechegemischternutzung','ax_funktion_fliessgewaesser','ax_funktion_flugverkehr','ax_funktion_friedhof','ax_funktion_gebaeudeinstallation3d','ax_funktion_gehoelz','ax_funktion_gewaesserachse','ax_funktion_hafenbecken','ax_funktion_industrieundgewerbeflaeche','ax_funktion_lagefestpunkt','ax_funktion_meer','ax_funktion_platz','ax_funktion_polder','ax_funktion_referenzstationspunkt','ax_funktion_schiffsverkehr','ax_funktion_schutzgebietnachwasserrecht','ax_funktion_schwerefestpunkt','ax_funktion_siedlungsflaeche','ax_funktion_sportfreizeitunderholungsflaeche','ax_funktion_stehendesgewaesser','ax_funktion_strasse','ax_funktion_strassenachse','ax_funktion_tagebaugrubesteinbruch','ax_funktion_unlandvegetationsloseflaeche','ax_funktion_untergeordnetesgewaesser','ax_funktion_vegetationsmerkmal','ax_funktion_weg','ax_funktion_wegachse','ax_funktion_wohnbauflaeche','ax_gebaeude','ax_gebaeudeausgestaltung','ax_gebaeudefunktion','ax_gebaeudeinstallation3d','ax_gebiet_bundesland','ax_gebiet_kreis','ax_gebiet_nationalstaat','ax_gebiet_regierungsbezirk','ax_gebiet_verwaltungsgemeinschaft','ax_gebietsgrenze','ax_gehoelz','ax_gemarkung','ax_gemarkungsteilflur','ax_gemeinde','ax_gemeindeteil','ax_genauigkeitsstufe_punktort','ax_genauigkeitsstufe_schwere','ax_georeferenziertegebaeudeadresse','ax_gewaesserachse','ax_gewaessermerkmal','ax_gewaesserstationierungsachse','ax_gewann','ax_gleis','ax_gnsstauglichkeit','ax_grablochderbodenschaetzung','ax_grenzpunkt','ax_grenzuebergang','ax_gruendederausgesetztenabmarkung_grenzpunkt','ax_hafen','ax_hafenbecken','ax_hafenkategorie_hafen','ax_halde','ax_heide','ax_heilquellegasquelle','ax_historischesbauwerkoderhistorischeeinrichtung','ax_historischesflurstueck','ax_historischesflurstueckalb','ax_historischesflurstueckohneraumbezug','ax_hoehenfestpunkt','ax_hoehenlinie','ax_hoehleneingang','ax_horizontfreiheit_grenzpunkt','ax_horizontfreiheit_netzpunkt','ax_hydrologischesmerkmal_fliessgewaesser','ax_hydrologischesmerkmal_gewaesserachse','ax_hydrologischesmerkmal_gewaessermerkmal','ax_hydrologischesmerkmal_heilquellegasquelle','ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri','ax_hydrologischesmerkmal_stehendesgewaesser','ax_hydrologischesmerkmal_untergeordnetesgewaesser','ax_impliziteloeschungderreservierung','ax_industrieundgewerbeflaeche','ax_insel','ax_internationalebedeutung_strasse','ax_kanal','ax_klassifikation_hierarchiestufe3d_lagefestpunkt','ax_klassifikation_ordnung_lagefestpunkt','ax_klassifikation_wertigkeit_lagefestpunkt','ax_klassifizierung_bewertung','ax_klassifizierungnachstrassenrecht','ax_klassifizierungnachwasserrecht','ax_kleinraeumigerlandschaftsteil','ax_klimastufe','ax_kommunalesgebiet','ax_kommunalesteilgebiet','ax_kondominium','ax_konstruktionsmerkmalbauart_schleuse','ax_koordinatenstatus_punktort','ax_kreisregion','ax_lagebezeichnungkatalogeintrag','ax_lagebezeichnungmithausnummer','ax_lagebezeichnungmitpseudonummer','ax_lagebezeichnungohnehausnummer','ax_lagefestpunkt','ax_lagergut_halde','ax_lagergut_industrieundgewerbeflaeche','ax_lagezurerdoberflaeche_gebaeude','ax_lagezurerdoberflaeche_transportanlage','ax_lagezurerdoberflaeche_untergeordnetesgewaesser','ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk','ax_lagezuroberflaeche_gleis','ax_landschaft','ax_landschaftstyp','ax_landwirtschaft','ax_leitung','ax_letzteabgabeart','ax_li_processstep_bodenhoehe_description','ax_li_processstep_dachhoehe_source','ax_li_processstep_mitdatenerhebung_description','ax_li_processstep_ohnedatenerhebung_description','ax_li_processstep_punktort_description','ax_li_processstep3d_description','ax_liniendarstellung_topographischelinie','ax_marke','ax_markierung_wegachse','ax_markierung_wegpfadsteig','ax_material_materialeigenschaft_material3d','ax_material3d','ax_meer','ax_merkmal_musterundvergleichsstueck','ax_messmethode_schwere','ax_moor','ax_musterundvergleichsstueck','ax_namensnummer','ax_nationalstaat','ax_naturumweltoderbodenschutzrecht','ax_netzknoten','ax_null_listenelement3d','ax_nullpunkt','ax_nutzung','ax_nutzung_flugverkehr','ax_nutzung_hafen','ax_nutzung_hafenbecken','ax_nutzung_stehendesgewaesser','ax_nutzung_wald','ax_nutzungsart_bodenschaetzung','ax_nutzungsart_musterundvergleichsstueck','ax_obererbezugspunkt','ax_oberflaechenmaterial_flugverkehrsanlage','ax_oberflaechenmaterial_strasse','ax_oberflaechenmaterial_unlandvegetationsloseflaeche','ax_ordnung_hoehenfestpunkt','ax_ordnung_schwerefestpunkt','ax_ortslage','ax_person','ax_personengruppe','ax_platz','ax_polder','ax_primaerenergie_industrieundgewerbeflaeche','ax_produkt_transportanlage','ax_punkt3d','ax_punktkennunguntergegangen','ax_punktkennungvergleichend','ax_punktortag','ax_punktortau','ax_punktortta','ax_punktstabilitaet','ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet','ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung','ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand','ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes','ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra','ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi','ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt','ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili','ax_punktwolke3d','ax_qualitaet_hauskoordinate','ax_rechtszustand_schutzzone','ax_referenzstationspunkt','ax_regierungsbezirk','ax_reservierung','ax_reservierungsart_reservierungsauftrag','ax_rolledergemeinde','ax_schifffahrtskategorie','ax_schifffahrtskategorie_gewaesserstationierungsachse','ax_schifffahrtskategorie_kanal','ax_schifffahrtsliniefaehrverkehr','ax_schiffsverkehr','ax_schleuse','ax_schutzgebietnachnaturumweltoderbodenschutzrecht','ax_schutzgebietnachwasserrecht','ax_schutzzone','ax_schwere','ax_schwereanomalie_schwere_art','ax_schwerebezugssystem_schwere','ax_schwerefestpunkt','ax_schwerestatus_schwere','ax_seilbahnschwebebahn','ax_selektionsmasstab_benutzungsauftrag','ax_sicherungspunkt','ax_sickerstrecke','ax_siedlungsflaeche','ax_skizze','ax_skizzenart_skizze','ax_soll','ax_sonstigeangaben_bodenschaetzung','ax_sonstigeangaben_musterundvergleichsstueck','ax_sonstigervermessungspunkt','ax_sonstigesbauwerkodersonstigeeinrichtung','ax_sonstigesrecht','ax_speicherinhalt_vorratsbehaelterspeicherbauwerk','ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung','ax_sportfreizeitunderholungsflaeche','ax_spurweite','ax_stehendesgewaesser','ax_strasse','ax_strassenachse','ax_strassenverkehr','ax_strassenverkehrsanlage','ax_strukturlinie3d','ax_sumpf','ax_tagebaugrubesteinbruch','ax_tagesabschnitt','ax_tauglichkeit_gcg','ax_testgelaende','ax_textur3d','ax_tidemerkmal_meer','ax_topographischelinie','ax_transportanlage','ax_tuer3d','ax_turm','ax_typ_textur3d','ax_unlandvegetationsloseflaeche','ax_untererbezugspunkt','ax_untergeordnetesgewaesser','ax_ursprung','ax_vegetationsmerkmal','ax_vegetationsmerkmal_gehoelz','ax_vegetationsmerkmal_landwirtschaft','ax_vegetationsmerkmal_wald','ax_verarbeitungsart_einrichtungsauftrag','ax_verarbeitungsart_fortfuehrungsauftrag','ax_verband','ax_verkehrsdienst_bahnstrecke','ax_verkehrsdienst_bahnverkehrsanlage','ax_vertrauenswuerdigkeit_punktort','ax_vertrauenswuerdigkeit_schwere','ax_vertretung','ax_verwaltung','ax_verwaltungsgemeinschaft','ax_verwendeteobjekte_abgeleitetehoehenlinie','ax_verwendeteobjekte_dhmgitter','ax_vorratsbehaelterspeicherbauwerk','ax_wald','ax_wandflaeche3d','ax_wasserlauf','ax_wasserspiegelhoehe','ax_wasserverhaeltnisse','ax_weg','ax_wegpfadsteig','ax_weitere_gebaeudefunktion','ax_weiterenutzung_tatsaechlichenutzung','ax_widmung_kanal','ax_widmung_stehendesgewaesser','ax_widmung_strasse','ax_widmung_wasserlauf','ax_wirtschaftlicheeinheit','ax_wirtschaftsart','ax_wirtschaftsartgrundbuch','ax_wohnbauflaeche','ax_wohnplatz','ax_zone_schutzzone','ax_zustand','ax_zustand_bahnverkehr','ax_zustand_bahnverkehrsanlage','ax_zustand_bauwerkimgewaesserbereich','ax_zustand_bauwerkimverkehrsbereich','ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe','ax_zustand_bauwerkoderanlagefuersportfreizeitunderholung','ax_zustand_bergbaubetrieb','ax_zustand_boeschungkliff','ax_zustand_einrichtungenfuerdenschiffsverkehr','ax_zustand_flaechebesondererfunktionalerpraegung','ax_zustand_flaechegemischternutzung','ax_zustand_flugverkehr','ax_zustand_flugverkehrsanlage','ax_zustand_friedhof','ax_zustand_gebaeude','ax_zustand_gebietsgrenze','ax_zustand_gewaesserachse','ax_zustand_halde','ax_zustand_hoehleneingang','ax_zustand_industrieundgewerbeflaeche','ax_zustand_kanal','ax_zustand_naturumweltoderbodenschutzrecht','ax_zustand_schiffsverkehr','ax_zustand_schleuse','ax_zustand_sportfreizeitunderholungsflaeche','ax_zustand_stehendesgewaesser','ax_zustand_strasse','ax_zustand_tagebaugrubesteinbruch','ax_zustand_turm','ax_zustand_vegetationsmerkmal','ax_zustand_wald','ax_zustand_wohnbauflaeche','ax_zustandsstufe','ci_rolecode','lb_binnengewaesser','lb_blattform_holzigevegetation','lb_eis','lb_eisart_eis','lb_festgestein','lb_fliesseigenschaft_binnengewaesser','lb_gewaesserart_binnengewaesser','lb_hochbau','lb_holzigevegetation','lb_krautigevegetation','lb_lockermaterial','lb_meer','lb_meerart_meer','lb_oberflaechenmaterial_lockermaterial','lb_tiefbau','lb_vegetationsmerkmal_holzigevegetation','lb_vegetationsmerkmal_krautigevegetation','lb_wasserfuehrung_binnengewaesser','lb_wassersaettigung_holzigevegetation','lb_wassersaettigung_krautigevegetation','lb_wassersaettigung_lockermaterial','ln_abbau','ln_abbaugut_abbau','ln_aquakulturundfischereiwirtschaft','ln_art_abbau','ln_art_aquakulturundfischereiwirtschaft','ln_art_forstwirtschaft','ln_art_freiluftundnaherholung','ln_art_freizeitanlage','ln_art_gewerblichedienstleistungen','ln_art_industrieundverarbeitendesgewerbe','ln_art_versorgungundentsorgung','ln_art_wasserwirtschaft','ln_artderbestattungsflaeche_bestattung','ln_artderbetriebsflaeche_landwirtschaft','ln_artdesparkplatzes_strassenundwegeverkehr','ln_bahnverkehr','ln_bestattung','ln_bewirtschaftung_landwirtschaft','ln_flugverkehr','ln_foerdergut_abbau','ln_forstwirtschaft','ln_freiluftundnaherholung','ln_freizeitanlage','ln_funktion_bahnverkehr','ln_funktion_flugverkehr','ln_funktion_kulturundunterhaltung','ln_funktion_lagerung','ln_funktion_oeffentlicheeinrichtungen','ln_funktion_schiffsverkehr','ln_funktion_schutzanlage','ln_funktion_strassenundwegeverkehr','ln_funktion_wasserwirtschaft','ln_fussgaengerzone_strassenundwegeverkehr','ln_gewerblichedienstleistungen','ln_hafenkategorie_schiffsverkehr','ln_industrieundverarbeitendesgewerbe','ln_kulturundunterhaltung','ln_lagergut_lagerung','ln_lagerung','ln_landwirtschaft','ln_nutzung_flugverkehr','ln_nutzung_schiffsverkehr','ln_oberflaeche_lagerung','ln_oeffentlicheeinrichtungen','ln_ohnenutzung','ln_primaerenergie_versorgungundentsorgung','ln_schiffsverkehr','ln_schutzanlage','ln_sportanlage','ln_sportart_sportanlage','ln_strassenundwegeverkehr','ln_typ_schutzanlage','ln_versorgungundentsorgung','ln_wasserwirtschaft','ln_wohnnutzung','ln_zeitlichkeit_wohnnutzung','ln_zuflusssteuerung_schutzanlage','ln_zustand_abbau','ln_zustand_bahnverkehr','ln_zustand_flugverkehr','ln_zustand_gewerblichedienstleistungen','ln_zustand_industrieundverarbeitendesgewerbe','ln_zustand_kulturundunterhaltung','ln_zustand_lagerung','ln_zustand_oeffentlicheeinrichtungen','ln_zustand_schiffsverkehr','ln_zustand_strassenundwegeverkehr','ln_zustand_versorgungundentsorgung','ln_zustand_wasserwirtschaft','ln_zustand_wohnnutzung','nas_filter_capabilities','operation') OR f_table_name IN ('alkis_beziehungen','delete') );

	RETURN r;
END;
$$ LANGUAGE plpgsql;

-- Alle ALKIS-Tabellen leeren
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_clean() RETURNS varchar AS $$
DECLARE
	c RECORD;
	r VARCHAR;
BEGIN
	-- clean tables
	FOR c IN SELECT table_name FROM information_schema.tables
		   WHERE table_schema=current_schema()
		     AND (table_name IN ('aa_aktivitaet','aa_antrag','aa_antragsgebiet','aa_meilenstein','aa_projektsteuerung','aa_vorgang','ap_darstellung','ap_fpo','ap_kpo_3d','ap_lpo','ap_lto','ap_ppo','ap_pto','au_koerperobjekt_3d','au_mehrfachlinienobjekt_3d','au_punkthaufenobjekt_3d','au_trianguliertesoberflaechenobjekt_3d','au_umringobjekt_3d','ax_abgeleitetehoehenlinie','ax_abschlussflaeche3d','ax_abschnitt','ax_anderefestlegungnachstrassenrecht','ax_anderefestlegungnachwasserrecht','ax_anschrift','ax_ast','ax_aufnahmepunkt','ax_bahnstrecke','ax_bahnverkehr','ax_bahnverkehrsanlage','ax_baublock','ax_bauraumoderbodenordnungsrecht','ax_bauraumoderbodenordnungsrechtgrundbuch','ax_bauteil','ax_bauteil3d','ax_bauwerk3d','ax_bauwerkimgewaesserbereich','ax_bauwerkimverkehrsbereich','ax_bauwerkoderanlagefuerindustrieundgewerbe','ax_bauwerkoderanlagefuersportfreizeitunderholung','ax_benutzer','ax_benutzergruppemitzugriffskontrolle','ax_benutzergruppenba','ax_bergbaubetrieb','ax_besondereflurstuecksgrenze','ax_besonderegebaeudelinie','ax_besondererbauwerkspunkt','ax_besonderergebaeudepunkt','ax_besonderertopographischerpunkt','ax_bewertung','ax_bodenflaeche3d','ax_bodenschaetzung','ax_boeschungkliff','ax_boeschungsflaeche','ax_buchungsblatt','ax_buchungsblattbezirk','ax_buchungsstelle','ax_bundesland','ax_dachflaeche3d','ax_dammwalldeich','ax_denkmalschutzrecht','ax_dhmgitter','ax_dienststelle','ax_duene','ax_einrichtungenfuerdenschiffsverkehr','ax_einrichtunginoeffentlichenbereichen','ax_einschnitt','ax_fahrbahnachse','ax_fahrwegachse','ax_felsenfelsblockfelsnadel','ax_fenster3d','ax_firstlinie','ax_flaeche3d','ax_flaechebesondererfunktionalerpraegung','ax_flaechegemischternutzung','ax_fliessgewaesser','ax_flugverkehr','ax_flugverkehrsanlage','ax_flurstueck','ax_flurstueckgrundbuch','ax_forstrecht','ax_fortfuehrungsfall','ax_fortfuehrungsfallgrundbuch','ax_fortfuehrungsnachweisdeckblatt','ax_friedhof','ax_gebaeude','ax_gebaeudeausgestaltung','ax_gebaeudeinstallation3d','ax_gebiet_bundesland','ax_gebiet_kreis','ax_gebiet_nationalstaat','ax_gebiet_regierungsbezirk','ax_gebiet_verwaltungsgemeinschaft','ax_gebietsgrenze','ax_gehoelz','ax_gemarkung','ax_gemarkungsteilflur','ax_gemeinde','ax_gemeindeteil','ax_georeferenziertegebaeudeadresse','ax_gewaesserachse','ax_gewaessermerkmal','ax_gewaesserstationierungsachse','ax_gewann','ax_gleis','ax_grablochderbodenschaetzung','ax_grenzpunkt','ax_grenzuebergang','ax_hafen','ax_hafenbecken','ax_halde','ax_heide','ax_heilquellegasquelle','ax_historischesbauwerkoderhistorischeeinrichtung','ax_historischesflurstueck','ax_historischesflurstueckalb','ax_historischesflurstueckohneraumbezug','ax_hoehenfestpunkt','ax_hoehenlinie','ax_hoehleneingang','ax_industrieundgewerbeflaeche','ax_insel','ax_kanal','ax_klassifizierungnachstrassenrecht','ax_klassifizierungnachwasserrecht','ax_kleinraeumigerlandschaftsteil','ax_kommunalesgebiet','ax_kommunalesteilgebiet','ax_kondominium','ax_kreisregion','ax_lagebezeichnungkatalogeintrag','ax_lagebezeichnungmithausnummer','ax_lagebezeichnungmitpseudonummer','ax_lagebezeichnungohnehausnummer','ax_lagefestpunkt','ax_landschaft','ax_landwirtschaft','ax_leitung','ax_material3d','ax_meer','ax_moor','ax_musterundvergleichsstueck','ax_namensnummer','ax_nationalstaat','ax_naturumweltoderbodenschutzrecht','ax_netzknoten','ax_nullpunkt','ax_ortslage','ax_person','ax_personengruppe','ax_platz','ax_polder','ax_punkt3d','ax_punktkennunguntergegangen','ax_punktkennungvergleichend','ax_punktortag','ax_punktortau','ax_punktortta','ax_punktwolke3d','ax_referenzstationspunkt','ax_regierungsbezirk','ax_reservierung','ax_schifffahrtsliniefaehrverkehr','ax_schiffsverkehr','ax_schleuse','ax_schutzgebietnachnaturumweltoderbodenschutzrecht','ax_schutzgebietnachwasserrecht','ax_schutzzone','ax_schwere','ax_schwerefestpunkt','ax_seilbahnschwebebahn','ax_sicherungspunkt','ax_sickerstrecke','ax_siedlungsflaeche','ax_skizze','ax_soll','ax_sonstigervermessungspunkt','ax_sonstigesbauwerkodersonstigeeinrichtung','ax_sonstigesrecht','ax_sportfreizeitunderholungsflaeche','ax_stehendesgewaesser','ax_strasse','ax_strassenachse','ax_strassenverkehr','ax_strassenverkehrsanlage','ax_strukturlinie3d','ax_sumpf','ax_tagebaugrubesteinbruch','ax_tagesabschnitt','ax_testgelaende','ax_textur3d','ax_topographischelinie','ax_transportanlage','ax_tuer3d','ax_turm','ax_unlandvegetationsloseflaeche','ax_untergeordnetesgewaesser','ax_vegetationsmerkmal','ax_verband','ax_vertretung','ax_verwaltung','ax_verwaltungsgemeinschaft','ax_vorratsbehaelterspeicherbauwerk','ax_wald','ax_wandflaeche3d','ax_wasserlauf','ax_wasserspiegelhoehe','ax_weg','ax_wegpfadsteig','ax_wirtschaftlicheeinheit','ax_wohnbauflaeche','ax_wohnplatz','lb_binnengewaesser','lb_eis','lb_festgestein','lb_hochbau','lb_holzigevegetation','lb_krautigevegetation','lb_lockermaterial','lb_meer','lb_tiefbau','ln_abbau','ln_aquakulturundfischereiwirtschaft','ln_bahnverkehr','ln_bestattung','ln_flugverkehr','ln_forstwirtschaft','ln_freiluftundnaherholung','ln_freizeitanlage','ln_gewerblichedienstleistungen','ln_industrieundverarbeitendesgewerbe','ln_kulturundunterhaltung','ln_lagerung','ln_landwirtschaft','ln_oeffentlicheeinrichtungen','ln_ohnenutzung','ln_schiffsverkehr','ln_schutzanlage','ln_sportanlage','ln_strassenundwegeverkehr','ln_versorgungundentsorgung','ln_wasserwirtschaft','ln_wohnnutzung') OR table_name IN ('alkis_beziehungen','delete'))
		   ORDER BY table_type DESC LOOP
		r := alkis_string_append(r, 'Tabelle ' || c.table_name || ' geleert.');
		EXECUTE 'DELETE FROM ' || c.table_name;
	END LOOP;

	RETURN r;
END;
$$ LANGUAGE plpgsql;

-- Alle ALKIS-Tabellen erben
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_inherit(parent varchar) RETURNS varchar AS $$
DECLARE
	tab RECORD;
	ind RECORD;
	r VARCHAR;
	nt INTEGER;
	ni INTEGER;
	nv INTEGER;
BEGIN
	nt := 0;
	ni := 0;
	nv := 0;

	-- inherit tables
	FOR tab IN
		SELECT c.oid, c.relname, obj_description(c.oid) AS description
		FROM pg_catalog.pg_class c
		JOIN pg_catalog.pg_namespace n ON n.oid=c.relnamespace AND n.nspname=parent
		WHERE pg_get_userbyid(c.relowner)=current_user AND c.relkind='r'
		  AND NOT EXISTS (
			SELECT *
			FROM pg_catalog.pg_class c1
			JOIN pg_catalog.pg_namespace n1 ON n1.oid=c1.relnamespace AND n1.nspname=current_schema()
			WHERE c1.relname=c.relname
		  )
	LOOP
		IF tab.description LIKE 'FeatureType:%' OR tab.description LIKE 'BASE:%' THEN
			nt := nt + 1;
			EXECUTE 'CREATE TABLE ' || quote_ident(tab.relname) || '() INHERITS (' || quote_ident(parent) || '.' || quote_ident(tab.relname) || ')';
			RAISE NOTICE 'Tabelle % abgeleitet.', tab.relname;

			FOR ind IN
				SELECT c.relname, replace(pg_get_indexdef(i.indexrelid), 'ON '||quote_ident(parent)||'.', 'ON ') AS sql
				FROM pg_catalog.pg_index i
				JOIN pg_catalog.pg_class c ON c.oid=i.indexrelid
				WHERE i.indrelid=tab.oid
			LOOP
				ni := ni + 1;
				EXECUTE ind.sql;
			END LOOP;
		ELSE
			nv := nv + 1;
			EXECUTE 'CREATE VIEW ' || quote_ident(tab.relname) || ' AS SELECT * FROM ' || quote_ident(parent) || '.' || quote_ident(tab.relname);
		END IF;
	END LOOP;

	RETURN nt || ' Tabellen mit ' || ni || ' Indizes abgeleitet und ' || nv || ' Sichten erzeugt.';
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION :"parent_schema".alkis_create_bsrs(id INTEGER) RETURNS varchar AS $$
DECLARE
	n INTEGER;
BEGIN
	SELECT count(*) INTO n FROM spatial_ref_sys WHERE srid=id;
	IF n=1 THEN
		RETURN NULL;
	END IF;

	IF NOT has_table_privilege('spatial_ref_sys', 'INSERT') THEN
		RAISE EXCEPTION 'Darf fehlendes Koordinatensystem % nicht einfügen.', id;
	END IF;

	IF id=131466 THEN
		-- DE_DHDN_3GK2_BW100
		INSERT INTO spatial_ref_sys(srid,auth_name,auth_srid,srtext,proj4text)
			SELECT
				131466,auth_name,131466
				,replace(replace(srtext,'PARAMETER["false_easting",2500000]','PARAMETER["false_easting",500000]'),'"EPSG","31466"','"EPSG","131466"')
				,replace(proj4text,'+x_0=2500000','+x_0=500000')
			FROM spatial_ref_sys
			WHERE srid=31466
			  AND NOT EXISTS (SELECT * FROM spatial_ref_sys WHERE srid=131466);
		RETURN 'Koordinatensystem '||id||' angelegt.';
	END IF;

	IF id=131467 THEN
		-- DE_DHDN_3GK3_BW100
		INSERT INTO spatial_ref_sys(srid,auth_name,auth_srid,srtext,proj4text)
			SELECT
				131467,auth_name,131467
				,replace(replace(srtext,'PARAMETER["false_easting",3500000]','PARAMETER["false_easting",500000]'),'"EPSG","31467"','"EPSG","131467"')
				,replace(proj4text,'+x_0=3500000','+x_0=500000')
			FROM spatial_ref_sys
			WHERE srid=31467
			  AND NOT EXISTS (SELECT * FROM spatial_ref_sys WHERE srid=131467);
		RETURN 'Koordinatensystem '||id||' angelegt.';
	END IF;

	IF id=131468 THEN
		-- DE_DHDN_3GK4_BY120
		INSERT INTO spatial_ref_sys(srid,auth_name,auth_srid,srtext,proj4text)
			SELECT
				131468,auth_name,131468
				,replace(replace(srtext,'PARAMETER["false_easting",4500000]','PARAMETER["false_easting",500000]'),'"EPSG","31468"','"EPSG","131468"')
				,replace(proj4text,'+x_0=4500000','+x_0=500000')
			FROM spatial_ref_sys
			WHERE srid=31468
			  AND NOT EXISTS (SELECT * FROM spatial_ref_sys WHERE srid=131468);
		RETURN 'Koordinatensystem '||id||' angelegt.';
	END IF;

	RAISE EXCEPTION 'Nicht erwartetes Koordinatensystem %.', id;
END;
$$ LANGUAGE plpgsql;

-- Alle ALKIS-Tabellen leeren
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_delete() RETURNS varchar AS $$
DECLARE
	c RECORD;
	r varchar;
BEGIN
	-- drop views
	FOR c IN
		SELECT table_name
		FROM information_schema.tables
		WHERE table_schema=current_schema() AND table_type='BASE TABLE'
		  AND ( table_name IN ('aa_advstandardmodell','aa_aktivitaet','aa_anlassart','aa_anlassart_benutzungsauftrag','aa_antrag','aa_antragsgebiet','aa_art_themendefinition','aa_besonderemeilensteinkategorie','aa_gefuehrteobjektart','aa_meilenstein','aa_nas_ausgabeform','aa_projektsteuerung','aa_themendimension','aa_vorgang','aa_weiteremodellart','ap_darstellung','ap_dateityp_3d','ap_fpo','ap_horizontaleausrichtung','ap_kpo_3d','ap_lpo','ap_lto','ap_ppo','ap_pto','ap_vertikaleausrichtung','au_koerperobjekt_3d','au_mehrfachlinienobjekt_3d','au_punkthaufenobjekt_3d','au_trianguliertesoberflaechenobjekt_3d','au_umringobjekt_3d','ax_abbaugut_bergbaubetrieb','ax_abbaugut_tagebaugrubesteinbruch','ax_abgabeversion','ax_abgeleitetehoehenlinie','ax_abschlussflaeche3d','ax_abschnitt','ax_administrative_funktion','ax_anderefestlegungnachstrassenrecht','ax_anderefestlegungnachwasserrecht','ax_anrede_person','ax_anschrift','ax_anzahlderstreckengleise','ax_archaeologischertyp_historischesbauwerkoderhistorischee','ax_art_adressat_auszug','ax_art_baublock','ax_art_bauwerkspunkt','ax_art_bereichzeitlich','ax_art_dammwalldeich','ax_art_einrichtungenfuerdenschiffsverkehr','ax_art_einrichtunginoeffentlichenbereichen','ax_art_flaeche3d','ax_art_flugverkehr','ax_art_flugverkehrsanlage','ax_art_gebaeudepunkt','ax_art_gewaessermerkmal','ax_art_gleis','ax_art_heilquellegasquelle','ax_art_punkt3d','ax_art_punktkennung','ax_art_punktwolke3d','ax_art_reservierung','ax_art_schifffahrtsliniefaehrverkehr','ax_art_strassenverkehrsanlage','ax_art_strukturlinie3d','ax_art_verband','ax_art_verbandsgemeinde','ax_art_wegpfadsteig','ax_artderbebauung_flaechebesondererfunktionalerpraegung','ax_artderbebauung_flaechegemischternutzung','ax_artderbebauung_siedlungsflaeche','ax_artderbebauung_wohnbauflaeche','ax_artderfestlegung_anderefestlegungnachstrassenrecht','ax_artderfestlegung_anderefestlegungnachwasserrecht','ax_artderfestlegung_bauraumoderbodenordnungsrecht','ax_artderfestlegung_denkmalschutzrecht','ax_artderfestlegung_forstrecht','ax_artderfestlegung_klassifizierungnachstrassenrecht','ax_artderfestlegung_klassifizierungnachwasserrecht','ax_artderfestlegung_naturumweltoderbodenschutzrecht','ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc','ax_artderfestlegung_schutzgebietnachwasserrecht','ax_artderfestlegung_sonstigesrecht','ax_artderflurstuecksgrenze_besondereflurstuecksgrenze','ax_artdergebietsgrenze_gebietsgrenze','ax_artdergewaesserstationierungsachse','ax_artderrechtsgemeinschaft_namensnummer','ax_artdesnullpunktes_nullpunkt','ax_artdespolders','ax_ast','ax_aufnahmepunkt','ax_ausgabemedium_benutzer','ax_auspraegung','ax_bahnhofskategorie_bahnverkehrsanlage','ax_bahnkategorie','ax_bahnkategorie_bahnstrecke','ax_bahnkategorie_bahnverkehrsanlage','ax_bahnkategorie_gleis','ax_bahnkategorie_seilbahnschwebebahn','ax_bahnstrecke','ax_bahnverkehr','ax_bahnverkehrsanlage','ax_bauart_bauteil','ax_baublock','ax_bauraumoderbodenordnungsrecht','ax_bauraumoderbodenordnungsrechtgrundbuch','ax_bauteil','ax_bauteil3d','ax_bauweise_gebaeude','ax_bauwerk3d','ax_bauwerkimgewaesserbereich','ax_bauwerkimverkehrsbereich','ax_bauwerkoderanlagefuerindustrieundgewerbe','ax_bauwerkoderanlagefuersportfreizeitunderholung','ax_bauwerksfunktion_bauwerkimgewaesserbereich','ax_bauwerksfunktion_bauwerkimverkehrsbereich','ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer','ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde','ax_bauwerksfunktion_leitung','ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun','ax_bauwerksfunktion_transportanlage','ax_bauwerksfunktion_turm','ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk','ax_bedeutung_grablochderbodenschaetzung','ax_befahrbarkeit_fahrwegachse','ax_befestigung_fahrwegachse','ax_befestigung_wegpfadsteig','ax_behoerde','ax_bemerkungzurabmarkung_grenzpunkt','ax_benutzer','ax_benutzergruppemitzugriffskontrolle','ax_benutzergruppenba','ax_berechnungsmethode','ax_bergbaubetrieb','ax_beschaffenheit_besonderegebaeudelinie','ax_besonderefahrstreifen','ax_besondereflurstuecksgrenze','ax_besonderefunktion_forstrecht','ax_besonderegebaeudelinie','ax_besondererbauwerkspunkt','ax_besonderergebaeudepunkt','ax_besonderertopographischerpunkt','ax_besondereverkehrsbedeutung','ax_bewertung','ax_bewuchs_vegetationsmerkmal','ax_bezeichnung_verwaltungsgemeinschaft','ax_bezugspunktdach','ax_blattart_buchungsblatt','ax_blattart_historischesflurstueck','ax_bodenart_bodenschaetzung','ax_bodenart_musterundvergleichsstueck','ax_bodenflaeche3d','ax_bodenschaetzung','ax_bodenstufe','ax_boeschungkliff','ax_boeschungsflaeche','ax_buchungsart_anlieger','ax_buchungsart_buchungsstelle','ax_buchungsblatt','ax_buchungsblattbezirk','ax_buchungsstelle','ax_bundesland','ax_dachflaeche3d','ax_dachform','ax_dachgeschossausbau_gebaeude','ax_dammwalldeich','ax_darstellung_gebaeudeausgestaltung','ax_dateitypraster_textur3d','ax_datenerhebung','ax_datenerhebung_punktort','ax_datenerhebung_schwere','ax_datenerhebung3d','ax_datenformat_benutzer','ax_denkmalschutzrecht','ax_dhmgitter','ax_dienststelle','ax_duene','ax_eigentuemerart_namensnummer','ax_einrichtungenfuerdenschiffsverkehr','ax_einrichtunginoeffentlichenbereichen','ax_einschnitt','ax_elektrifizierung','ax_entstehungsart','ax_ergebnisderueberpruefung_bauwerkeeinrichtungenundsonsti','ax_ergebnisderueberpruefung_dammwalldeich','ax_ergebnisderueberpruefung_klassifizierungnachwasserrecht','ax_ergebnisderueberpruefung_tatsaechlichenutzung','ax_fahrbahnachse','ax_fahrbahntrennung_strasse','ax_fahrwegachse','ax_felsenfelsblockfelsnadel','ax_fenster3d','ax_firstlinie','ax_flaeche3d','ax_flaechebesondererfunktionalerpraegung','ax_flaechegemischternutzung','ax_fliessgewaesser','ax_flugverkehr','ax_flugverkehrsanlage','ax_flurstueck','ax_flurstueckgrundbuch','ax_foerdergut_industrieundgewerbeflaeche','ax_forstrecht','ax_fortfuehrungsauftrag','ax_fortfuehrungsfall','ax_fortfuehrungsfallgrundbuch','ax_fortfuehrungsnachweisdeckblatt','ax_friedhof','ax_funktion_bahnverkehr','ax_funktion_bauwerk','ax_funktion_bergbaubetrieb','ax_funktion_dammwalldeich','ax_funktion_einschnitt','ax_funktion_fahrbahnachse','ax_funktion_flaechebesondererfunktionalerpraegung','ax_funktion_flaechegemischternutzung','ax_funktion_fliessgewaesser','ax_funktion_flugverkehr','ax_funktion_friedhof','ax_funktion_gebaeudeinstallation3d','ax_funktion_gehoelz','ax_funktion_gewaesserachse','ax_funktion_hafenbecken','ax_funktion_industrieundgewerbeflaeche','ax_funktion_lagefestpunkt','ax_funktion_meer','ax_funktion_platz','ax_funktion_polder','ax_funktion_referenzstationspunkt','ax_funktion_schiffsverkehr','ax_funktion_schutzgebietnachwasserrecht','ax_funktion_schwerefestpunkt','ax_funktion_siedlungsflaeche','ax_funktion_sportfreizeitunderholungsflaeche','ax_funktion_stehendesgewaesser','ax_funktion_strasse','ax_funktion_strassenachse','ax_funktion_tagebaugrubesteinbruch','ax_funktion_unlandvegetationsloseflaeche','ax_funktion_untergeordnetesgewaesser','ax_funktion_vegetationsmerkmal','ax_funktion_weg','ax_funktion_wegachse','ax_funktion_wohnbauflaeche','ax_gebaeude','ax_gebaeudeausgestaltung','ax_gebaeudefunktion','ax_gebaeudeinstallation3d','ax_gebiet_bundesland','ax_gebiet_kreis','ax_gebiet_nationalstaat','ax_gebiet_regierungsbezirk','ax_gebiet_verwaltungsgemeinschaft','ax_gebietsgrenze','ax_gehoelz','ax_gemarkung','ax_gemarkungsteilflur','ax_gemeinde','ax_gemeindeteil','ax_genauigkeitsstufe_punktort','ax_genauigkeitsstufe_schwere','ax_georeferenziertegebaeudeadresse','ax_gewaesserachse','ax_gewaessermerkmal','ax_gewaesserstationierungsachse','ax_gewann','ax_gleis','ax_gnsstauglichkeit','ax_grablochderbodenschaetzung','ax_grenzpunkt','ax_grenzuebergang','ax_gruendederausgesetztenabmarkung_grenzpunkt','ax_hafen','ax_hafenbecken','ax_hafenkategorie_hafen','ax_halde','ax_heide','ax_heilquellegasquelle','ax_historischesbauwerkoderhistorischeeinrichtung','ax_historischesflurstueck','ax_historischesflurstueckalb','ax_historischesflurstueckohneraumbezug','ax_hoehenfestpunkt','ax_hoehenlinie','ax_hoehleneingang','ax_horizontfreiheit_grenzpunkt','ax_horizontfreiheit_netzpunkt','ax_hydrologischesmerkmal_fliessgewaesser','ax_hydrologischesmerkmal_gewaesserachse','ax_hydrologischesmerkmal_gewaessermerkmal','ax_hydrologischesmerkmal_heilquellegasquelle','ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri','ax_hydrologischesmerkmal_stehendesgewaesser','ax_hydrologischesmerkmal_untergeordnetesgewaesser','ax_impliziteloeschungderreservierung','ax_industrieundgewerbeflaeche','ax_insel','ax_internationalebedeutung_strasse','ax_kanal','ax_klassifikation_hierarchiestufe3d_lagefestpunkt','ax_klassifikation_ordnung_lagefestpunkt','ax_klassifikation_wertigkeit_lagefestpunkt','ax_klassifizierung_bewertung','ax_klassifizierungnachstrassenrecht','ax_klassifizierungnachwasserrecht','ax_kleinraeumigerlandschaftsteil','ax_klimastufe','ax_kommunalesgebiet','ax_kommunalesteilgebiet','ax_kondominium','ax_konstruktionsmerkmalbauart_schleuse','ax_koordinatenstatus_punktort','ax_kreisregion','ax_lagebezeichnungkatalogeintrag','ax_lagebezeichnungmithausnummer','ax_lagebezeichnungmitpseudonummer','ax_lagebezeichnungohnehausnummer','ax_lagefestpunkt','ax_lagergut_halde','ax_lagergut_industrieundgewerbeflaeche','ax_lagezurerdoberflaeche_gebaeude','ax_lagezurerdoberflaeche_transportanlage','ax_lagezurerdoberflaeche_untergeordnetesgewaesser','ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk','ax_lagezuroberflaeche_gleis','ax_landschaft','ax_landschaftstyp','ax_landwirtschaft','ax_leitung','ax_letzteabgabeart','ax_li_processstep_bodenhoehe_description','ax_li_processstep_dachhoehe_source','ax_li_processstep_mitdatenerhebung_description','ax_li_processstep_ohnedatenerhebung_description','ax_li_processstep_punktort_description','ax_li_processstep3d_description','ax_liniendarstellung_topographischelinie','ax_marke','ax_markierung_wegachse','ax_markierung_wegpfadsteig','ax_material_materialeigenschaft_material3d','ax_material3d','ax_meer','ax_merkmal_musterundvergleichsstueck','ax_messmethode_schwere','ax_moor','ax_musterundvergleichsstueck','ax_namensnummer','ax_nationalstaat','ax_naturumweltoderbodenschutzrecht','ax_netzknoten','ax_null_listenelement3d','ax_nullpunkt','ax_nutzung','ax_nutzung_flugverkehr','ax_nutzung_hafen','ax_nutzung_hafenbecken','ax_nutzung_stehendesgewaesser','ax_nutzung_wald','ax_nutzungsart_bodenschaetzung','ax_nutzungsart_musterundvergleichsstueck','ax_obererbezugspunkt','ax_oberflaechenmaterial_flugverkehrsanlage','ax_oberflaechenmaterial_strasse','ax_oberflaechenmaterial_unlandvegetationsloseflaeche','ax_ordnung_hoehenfestpunkt','ax_ordnung_schwerefestpunkt','ax_ortslage','ax_person','ax_personengruppe','ax_platz','ax_polder','ax_primaerenergie_industrieundgewerbeflaeche','ax_produkt_transportanlage','ax_punkt3d','ax_punktkennunguntergegangen','ax_punktkennungvergleichend','ax_punktortag','ax_punktortau','ax_punktortta','ax_punktstabilitaet','ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet','ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung','ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand','ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes','ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra','ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi','ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt','ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili','ax_punktwolke3d','ax_qualitaet_hauskoordinate','ax_rechtszustand_schutzzone','ax_referenzstationspunkt','ax_regierungsbezirk','ax_reservierung','ax_reservierungsart_reservierungsauftrag','ax_rolledergemeinde','ax_schifffahrtskategorie','ax_schifffahrtskategorie_gewaesserstationierungsachse','ax_schifffahrtskategorie_kanal','ax_schifffahrtsliniefaehrverkehr','ax_schiffsverkehr','ax_schleuse','ax_schutzgebietnachnaturumweltoderbodenschutzrecht','ax_schutzgebietnachwasserrecht','ax_schutzzone','ax_schwere','ax_schwereanomalie_schwere_art','ax_schwerebezugssystem_schwere','ax_schwerefestpunkt','ax_schwerestatus_schwere','ax_seilbahnschwebebahn','ax_selektionsmasstab_benutzungsauftrag','ax_sicherungspunkt','ax_sickerstrecke','ax_siedlungsflaeche','ax_skizze','ax_skizzenart_skizze','ax_soll','ax_sonstigeangaben_bodenschaetzung','ax_sonstigeangaben_musterundvergleichsstueck','ax_sonstigervermessungspunkt','ax_sonstigesbauwerkodersonstigeeinrichtung','ax_sonstigesrecht','ax_speicherinhalt_vorratsbehaelterspeicherbauwerk','ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung','ax_sportfreizeitunderholungsflaeche','ax_spurweite','ax_stehendesgewaesser','ax_strasse','ax_strassenachse','ax_strassenverkehr','ax_strassenverkehrsanlage','ax_strukturlinie3d','ax_sumpf','ax_tagebaugrubesteinbruch','ax_tagesabschnitt','ax_tauglichkeit_gcg','ax_testgelaende','ax_textur3d','ax_tidemerkmal_meer','ax_topographischelinie','ax_transportanlage','ax_tuer3d','ax_turm','ax_typ_textur3d','ax_unlandvegetationsloseflaeche','ax_untererbezugspunkt','ax_untergeordnetesgewaesser','ax_ursprung','ax_vegetationsmerkmal','ax_vegetationsmerkmal_gehoelz','ax_vegetationsmerkmal_landwirtschaft','ax_vegetationsmerkmal_wald','ax_verarbeitungsart_einrichtungsauftrag','ax_verarbeitungsart_fortfuehrungsauftrag','ax_verband','ax_verkehrsdienst_bahnstrecke','ax_verkehrsdienst_bahnverkehrsanlage','ax_vertrauenswuerdigkeit_punktort','ax_vertrauenswuerdigkeit_schwere','ax_vertretung','ax_verwaltung','ax_verwaltungsgemeinschaft','ax_verwendeteobjekte_abgeleitetehoehenlinie','ax_verwendeteobjekte_dhmgitter','ax_vorratsbehaelterspeicherbauwerk','ax_wald','ax_wandflaeche3d','ax_wasserlauf','ax_wasserspiegelhoehe','ax_wasserverhaeltnisse','ax_weg','ax_wegpfadsteig','ax_weitere_gebaeudefunktion','ax_weiterenutzung_tatsaechlichenutzung','ax_widmung_kanal','ax_widmung_stehendesgewaesser','ax_widmung_strasse','ax_widmung_wasserlauf','ax_wirtschaftlicheeinheit','ax_wirtschaftsart','ax_wirtschaftsartgrundbuch','ax_wohnbauflaeche','ax_wohnplatz','ax_zone_schutzzone','ax_zustand','ax_zustand_bahnverkehr','ax_zustand_bahnverkehrsanlage','ax_zustand_bauwerkimgewaesserbereich','ax_zustand_bauwerkimverkehrsbereich','ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe','ax_zustand_bauwerkoderanlagefuersportfreizeitunderholung','ax_zustand_bergbaubetrieb','ax_zustand_boeschungkliff','ax_zustand_einrichtungenfuerdenschiffsverkehr','ax_zustand_flaechebesondererfunktionalerpraegung','ax_zustand_flaechegemischternutzung','ax_zustand_flugverkehr','ax_zustand_flugverkehrsanlage','ax_zustand_friedhof','ax_zustand_gebaeude','ax_zustand_gebietsgrenze','ax_zustand_gewaesserachse','ax_zustand_halde','ax_zustand_hoehleneingang','ax_zustand_industrieundgewerbeflaeche','ax_zustand_kanal','ax_zustand_naturumweltoderbodenschutzrecht','ax_zustand_schiffsverkehr','ax_zustand_schleuse','ax_zustand_sportfreizeitunderholungsflaeche','ax_zustand_stehendesgewaesser','ax_zustand_strasse','ax_zustand_tagebaugrubesteinbruch','ax_zustand_turm','ax_zustand_vegetationsmerkmal','ax_zustand_wald','ax_zustand_wohnbauflaeche','ax_zustandsstufe','ci_rolecode','lb_binnengewaesser','lb_blattform_holzigevegetation','lb_eis','lb_eisart_eis','lb_festgestein','lb_fliesseigenschaft_binnengewaesser','lb_gewaesserart_binnengewaesser','lb_hochbau','lb_holzigevegetation','lb_krautigevegetation','lb_lockermaterial','lb_meer','lb_meerart_meer','lb_oberflaechenmaterial_lockermaterial','lb_tiefbau','lb_vegetationsmerkmal_holzigevegetation','lb_vegetationsmerkmal_krautigevegetation','lb_wasserfuehrung_binnengewaesser','lb_wassersaettigung_holzigevegetation','lb_wassersaettigung_krautigevegetation','lb_wassersaettigung_lockermaterial','ln_abbau','ln_abbaugut_abbau','ln_aquakulturundfischereiwirtschaft','ln_art_abbau','ln_art_aquakulturundfischereiwirtschaft','ln_art_forstwirtschaft','ln_art_freiluftundnaherholung','ln_art_freizeitanlage','ln_art_gewerblichedienstleistungen','ln_art_industrieundverarbeitendesgewerbe','ln_art_versorgungundentsorgung','ln_art_wasserwirtschaft','ln_artderbestattungsflaeche_bestattung','ln_artderbetriebsflaeche_landwirtschaft','ln_artdesparkplatzes_strassenundwegeverkehr','ln_bahnverkehr','ln_bestattung','ln_bewirtschaftung_landwirtschaft','ln_flugverkehr','ln_foerdergut_abbau','ln_forstwirtschaft','ln_freiluftundnaherholung','ln_freizeitanlage','ln_funktion_bahnverkehr','ln_funktion_flugverkehr','ln_funktion_kulturundunterhaltung','ln_funktion_lagerung','ln_funktion_oeffentlicheeinrichtungen','ln_funktion_schiffsverkehr','ln_funktion_schutzanlage','ln_funktion_strassenundwegeverkehr','ln_funktion_wasserwirtschaft','ln_fussgaengerzone_strassenundwegeverkehr','ln_gewerblichedienstleistungen','ln_hafenkategorie_schiffsverkehr','ln_industrieundverarbeitendesgewerbe','ln_kulturundunterhaltung','ln_lagergut_lagerung','ln_lagerung','ln_landwirtschaft','ln_nutzung_flugverkehr','ln_nutzung_schiffsverkehr','ln_oberflaeche_lagerung','ln_oeffentlicheeinrichtungen','ln_ohnenutzung','ln_primaerenergie_versorgungundentsorgung','ln_schiffsverkehr','ln_schutzanlage','ln_sportanlage','ln_sportart_sportanlage','ln_strassenundwegeverkehr','ln_typ_schutzanlage','ln_versorgungundentsorgung','ln_wasserwirtschaft','ln_wohnnutzung','ln_zeitlichkeit_wohnnutzung','ln_zuflusssteuerung_schutzanlage','ln_zustand_abbau','ln_zustand_bahnverkehr','ln_zustand_flugverkehr','ln_zustand_gewerblichedienstleistungen','ln_zustand_industrieundverarbeitendesgewerbe','ln_zustand_kulturundunterhaltung','ln_zustand_lagerung','ln_zustand_oeffentlicheeinrichtungen','ln_zustand_schiffsverkehr','ln_zustand_strassenundwegeverkehr','ln_zustand_versorgungundentsorgung','ln_zustand_wasserwirtschaft','ln_zustand_wohnnutzung','nas_filter_capabilities','operation')
			OR table_name IN ('alkis_beziehungen','delete') )
	LOOP
		r := alkis_string_append(r, c.table_name || ' wurde geleert.');
		EXECUTE 'TRUNCATE '||c.table_name;
	END LOOP;

	RETURN r;
END;
$$ LANGUAGE plpgsql;

-- Übersicht erzeugen, die alle alkis_beziehungen mit den Typen der beteiligen ALKIS-Objekte versieht
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_mviews() RETURNS varchar AS $$
DECLARE
	sql TEXT;
	delim TEXT;
	c RECORD;
BEGIN
	SELECT alkis_dropobject('vbeziehungen') INTO sql;
	SELECT alkis_dropobject('vobjekte') INTO sql;

	delim := '';
	sql := 'CREATE VIEW vobjekte AS ';

	FOR c IN SELECT table_name FROM information_schema.columns
		   WHERE column_name='gml_id'
		     AND table_name IN ('aa_advstandardmodell','aa_aktivitaet','aa_anlassart','aa_anlassart_benutzungsauftrag','aa_antrag','aa_antragsgebiet','aa_art_themendefinition','aa_besonderemeilensteinkategorie','aa_gefuehrteobjektart','aa_meilenstein','aa_nas_ausgabeform','aa_projektsteuerung','aa_themendimension','aa_vorgang','aa_weiteremodellart','ap_darstellung','ap_dateityp_3d','ap_fpo','ap_horizontaleausrichtung','ap_kpo_3d','ap_lpo','ap_lto','ap_ppo','ap_pto','ap_vertikaleausrichtung','au_koerperobjekt_3d','au_mehrfachlinienobjekt_3d','au_punkthaufenobjekt_3d','au_trianguliertesoberflaechenobjekt_3d','au_umringobjekt_3d','ax_abbaugut_bergbaubetrieb','ax_abbaugut_tagebaugrubesteinbruch','ax_abgabeversion','ax_abgeleitetehoehenlinie','ax_abschlussflaeche3d','ax_abschnitt','ax_administrative_funktion','ax_anderefestlegungnachstrassenrecht','ax_anderefestlegungnachwasserrecht','ax_anrede_person','ax_anschrift','ax_anzahlderstreckengleise','ax_archaeologischertyp_historischesbauwerkoderhistorischee','ax_art_adressat_auszug','ax_art_baublock','ax_art_bauwerkspunkt','ax_art_bereichzeitlich','ax_art_dammwalldeich','ax_art_einrichtungenfuerdenschiffsverkehr','ax_art_einrichtunginoeffentlichenbereichen','ax_art_flaeche3d','ax_art_flugverkehr','ax_art_flugverkehrsanlage','ax_art_gebaeudepunkt','ax_art_gewaessermerkmal','ax_art_gleis','ax_art_heilquellegasquelle','ax_art_punkt3d','ax_art_punktkennung','ax_art_punktwolke3d','ax_art_reservierung','ax_art_schifffahrtsliniefaehrverkehr','ax_art_strassenverkehrsanlage','ax_art_strukturlinie3d','ax_art_verband','ax_art_verbandsgemeinde','ax_art_wegpfadsteig','ax_artderbebauung_flaechebesondererfunktionalerpraegung','ax_artderbebauung_flaechegemischternutzung','ax_artderbebauung_siedlungsflaeche','ax_artderbebauung_wohnbauflaeche','ax_artderfestlegung_anderefestlegungnachstrassenrecht','ax_artderfestlegung_anderefestlegungnachwasserrecht','ax_artderfestlegung_bauraumoderbodenordnungsrecht','ax_artderfestlegung_denkmalschutzrecht','ax_artderfestlegung_forstrecht','ax_artderfestlegung_klassifizierungnachstrassenrecht','ax_artderfestlegung_klassifizierungnachwasserrecht','ax_artderfestlegung_naturumweltoderbodenschutzrecht','ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc','ax_artderfestlegung_schutzgebietnachwasserrecht','ax_artderfestlegung_sonstigesrecht','ax_artderflurstuecksgrenze_besondereflurstuecksgrenze','ax_artdergebietsgrenze_gebietsgrenze','ax_artdergewaesserstationierungsachse','ax_artderrechtsgemeinschaft_namensnummer','ax_artdesnullpunktes_nullpunkt','ax_artdespolders','ax_ast','ax_aufnahmepunkt','ax_ausgabemedium_benutzer','ax_auspraegung','ax_bahnhofskategorie_bahnverkehrsanlage','ax_bahnkategorie','ax_bahnkategorie_bahnstrecke','ax_bahnkategorie_bahnverkehrsanlage','ax_bahnkategorie_gleis','ax_bahnkategorie_seilbahnschwebebahn','ax_bahnstrecke','ax_bahnverkehr','ax_bahnverkehrsanlage','ax_bauart_bauteil','ax_baublock','ax_bauraumoderbodenordnungsrecht','ax_bauraumoderbodenordnungsrechtgrundbuch','ax_bauteil','ax_bauteil3d','ax_bauweise_gebaeude','ax_bauwerk3d','ax_bauwerkimgewaesserbereich','ax_bauwerkimverkehrsbereich','ax_bauwerkoderanlagefuerindustrieundgewerbe','ax_bauwerkoderanlagefuersportfreizeitunderholung','ax_bauwerksfunktion_bauwerkimgewaesserbereich','ax_bauwerksfunktion_bauwerkimverkehrsbereich','ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer','ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde','ax_bauwerksfunktion_leitung','ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun','ax_bauwerksfunktion_transportanlage','ax_bauwerksfunktion_turm','ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk','ax_bedeutung_grablochderbodenschaetzung','ax_befahrbarkeit_fahrwegachse','ax_befestigung_fahrwegachse','ax_befestigung_wegpfadsteig','ax_behoerde','ax_bemerkungzurabmarkung_grenzpunkt','ax_benutzer','ax_benutzergruppemitzugriffskontrolle','ax_benutzergruppenba','ax_berechnungsmethode','ax_bergbaubetrieb','ax_beschaffenheit_besonderegebaeudelinie','ax_besonderefahrstreifen','ax_besondereflurstuecksgrenze','ax_besonderefunktion_forstrecht','ax_besonderegebaeudelinie','ax_besondererbauwerkspunkt','ax_besonderergebaeudepunkt','ax_besonderertopographischerpunkt','ax_besondereverkehrsbedeutung','ax_bewertung','ax_bewuchs_vegetationsmerkmal','ax_bezeichnung_verwaltungsgemeinschaft','ax_bezugspunktdach','ax_blattart_buchungsblatt','ax_blattart_historischesflurstueck','ax_bodenart_bodenschaetzung','ax_bodenart_musterundvergleichsstueck','ax_bodenflaeche3d','ax_bodenschaetzung','ax_bodenstufe','ax_boeschungkliff','ax_boeschungsflaeche','ax_buchungsart_anlieger','ax_buchungsart_buchungsstelle','ax_buchungsblatt','ax_buchungsblattbezirk','ax_buchungsstelle','ax_bundesland','ax_dachflaeche3d','ax_dachform','ax_dachgeschossausbau_gebaeude','ax_dammwalldeich','ax_darstellung_gebaeudeausgestaltung','ax_dateitypraster_textur3d','ax_datenerhebung','ax_datenerhebung_punktort','ax_datenerhebung_schwere','ax_datenerhebung3d','ax_datenformat_benutzer','ax_denkmalschutzrecht','ax_dhmgitter','ax_dienststelle','ax_duene','ax_eigentuemerart_namensnummer','ax_einrichtungenfuerdenschiffsverkehr','ax_einrichtunginoeffentlichenbereichen','ax_einschnitt','ax_elektrifizierung','ax_entstehungsart','ax_ergebnisderueberpruefung_bauwerkeeinrichtungenundsonsti','ax_ergebnisderueberpruefung_dammwalldeich','ax_ergebnisderueberpruefung_klassifizierungnachwasserrecht','ax_ergebnisderueberpruefung_tatsaechlichenutzung','ax_fahrbahnachse','ax_fahrbahntrennung_strasse','ax_fahrwegachse','ax_felsenfelsblockfelsnadel','ax_fenster3d','ax_firstlinie','ax_flaeche3d','ax_flaechebesondererfunktionalerpraegung','ax_flaechegemischternutzung','ax_fliessgewaesser','ax_flugverkehr','ax_flugverkehrsanlage','ax_flurstueck','ax_flurstueckgrundbuch','ax_foerdergut_industrieundgewerbeflaeche','ax_forstrecht','ax_fortfuehrungsauftrag','ax_fortfuehrungsfall','ax_fortfuehrungsfallgrundbuch','ax_fortfuehrungsnachweisdeckblatt','ax_friedhof','ax_funktion_bahnverkehr','ax_funktion_bauwerk','ax_funktion_bergbaubetrieb','ax_funktion_dammwalldeich','ax_funktion_einschnitt','ax_funktion_fahrbahnachse','ax_funktion_flaechebesondererfunktionalerpraegung','ax_funktion_flaechegemischternutzung','ax_funktion_fliessgewaesser','ax_funktion_flugverkehr','ax_funktion_friedhof','ax_funktion_gebaeudeinstallation3d','ax_funktion_gehoelz','ax_funktion_gewaesserachse','ax_funktion_hafenbecken','ax_funktion_industrieundgewerbeflaeche','ax_funktion_lagefestpunkt','ax_funktion_meer','ax_funktion_platz','ax_funktion_polder','ax_funktion_referenzstationspunkt','ax_funktion_schiffsverkehr','ax_funktion_schutzgebietnachwasserrecht','ax_funktion_schwerefestpunkt','ax_funktion_siedlungsflaeche','ax_funktion_sportfreizeitunderholungsflaeche','ax_funktion_stehendesgewaesser','ax_funktion_strasse','ax_funktion_strassenachse','ax_funktion_tagebaugrubesteinbruch','ax_funktion_unlandvegetationsloseflaeche','ax_funktion_untergeordnetesgewaesser','ax_funktion_vegetationsmerkmal','ax_funktion_weg','ax_funktion_wegachse','ax_funktion_wohnbauflaeche','ax_gebaeude','ax_gebaeudeausgestaltung','ax_gebaeudefunktion','ax_gebaeudeinstallation3d','ax_gebiet_bundesland','ax_gebiet_kreis','ax_gebiet_nationalstaat','ax_gebiet_regierungsbezirk','ax_gebiet_verwaltungsgemeinschaft','ax_gebietsgrenze','ax_gehoelz','ax_gemarkung','ax_gemarkungsteilflur','ax_gemeinde','ax_gemeindeteil','ax_genauigkeitsstufe_punktort','ax_genauigkeitsstufe_schwere','ax_georeferenziertegebaeudeadresse','ax_gewaesserachse','ax_gewaessermerkmal','ax_gewaesserstationierungsachse','ax_gewann','ax_gleis','ax_gnsstauglichkeit','ax_grablochderbodenschaetzung','ax_grenzpunkt','ax_grenzuebergang','ax_gruendederausgesetztenabmarkung_grenzpunkt','ax_hafen','ax_hafenbecken','ax_hafenkategorie_hafen','ax_halde','ax_heide','ax_heilquellegasquelle','ax_historischesbauwerkoderhistorischeeinrichtung','ax_historischesflurstueck','ax_historischesflurstueckalb','ax_historischesflurstueckohneraumbezug','ax_hoehenfestpunkt','ax_hoehenlinie','ax_hoehleneingang','ax_horizontfreiheit_grenzpunkt','ax_horizontfreiheit_netzpunkt','ax_hydrologischesmerkmal_fliessgewaesser','ax_hydrologischesmerkmal_gewaesserachse','ax_hydrologischesmerkmal_gewaessermerkmal','ax_hydrologischesmerkmal_heilquellegasquelle','ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri','ax_hydrologischesmerkmal_stehendesgewaesser','ax_hydrologischesmerkmal_untergeordnetesgewaesser','ax_impliziteloeschungderreservierung','ax_industrieundgewerbeflaeche','ax_insel','ax_internationalebedeutung_strasse','ax_kanal','ax_klassifikation_hierarchiestufe3d_lagefestpunkt','ax_klassifikation_ordnung_lagefestpunkt','ax_klassifikation_wertigkeit_lagefestpunkt','ax_klassifizierung_bewertung','ax_klassifizierungnachstrassenrecht','ax_klassifizierungnachwasserrecht','ax_kleinraeumigerlandschaftsteil','ax_klimastufe','ax_kommunalesgebiet','ax_kommunalesteilgebiet','ax_kondominium','ax_konstruktionsmerkmalbauart_schleuse','ax_koordinatenstatus_punktort','ax_kreisregion','ax_lagebezeichnungkatalogeintrag','ax_lagebezeichnungmithausnummer','ax_lagebezeichnungmitpseudonummer','ax_lagebezeichnungohnehausnummer','ax_lagefestpunkt','ax_lagergut_halde','ax_lagergut_industrieundgewerbeflaeche','ax_lagezurerdoberflaeche_gebaeude','ax_lagezurerdoberflaeche_transportanlage','ax_lagezurerdoberflaeche_untergeordnetesgewaesser','ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk','ax_lagezuroberflaeche_gleis','ax_landschaft','ax_landschaftstyp','ax_landwirtschaft','ax_leitung','ax_letzteabgabeart','ax_li_processstep_bodenhoehe_description','ax_li_processstep_dachhoehe_source','ax_li_processstep_mitdatenerhebung_description','ax_li_processstep_ohnedatenerhebung_description','ax_li_processstep_punktort_description','ax_li_processstep3d_description','ax_liniendarstellung_topographischelinie','ax_marke','ax_markierung_wegachse','ax_markierung_wegpfadsteig','ax_material_materialeigenschaft_material3d','ax_material3d','ax_meer','ax_merkmal_musterundvergleichsstueck','ax_messmethode_schwere','ax_moor','ax_musterundvergleichsstueck','ax_namensnummer','ax_nationalstaat','ax_naturumweltoderbodenschutzrecht','ax_netzknoten','ax_null_listenelement3d','ax_nullpunkt','ax_nutzung','ax_nutzung_flugverkehr','ax_nutzung_hafen','ax_nutzung_hafenbecken','ax_nutzung_stehendesgewaesser','ax_nutzung_wald','ax_nutzungsart_bodenschaetzung','ax_nutzungsart_musterundvergleichsstueck','ax_obererbezugspunkt','ax_oberflaechenmaterial_flugverkehrsanlage','ax_oberflaechenmaterial_strasse','ax_oberflaechenmaterial_unlandvegetationsloseflaeche','ax_ordnung_hoehenfestpunkt','ax_ordnung_schwerefestpunkt','ax_ortslage','ax_person','ax_personengruppe','ax_platz','ax_polder','ax_primaerenergie_industrieundgewerbeflaeche','ax_produkt_transportanlage','ax_punkt3d','ax_punktkennunguntergegangen','ax_punktkennungvergleichend','ax_punktortag','ax_punktortau','ax_punktortta','ax_punktstabilitaet','ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet','ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung','ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand','ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes','ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra','ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi','ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt','ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili','ax_punktwolke3d','ax_qualitaet_hauskoordinate','ax_rechtszustand_schutzzone','ax_referenzstationspunkt','ax_regierungsbezirk','ax_reservierung','ax_reservierungsart_reservierungsauftrag','ax_rolledergemeinde','ax_schifffahrtskategorie','ax_schifffahrtskategorie_gewaesserstationierungsachse','ax_schifffahrtskategorie_kanal','ax_schifffahrtsliniefaehrverkehr','ax_schiffsverkehr','ax_schleuse','ax_schutzgebietnachnaturumweltoderbodenschutzrecht','ax_schutzgebietnachwasserrecht','ax_schutzzone','ax_schwere','ax_schwereanomalie_schwere_art','ax_schwerebezugssystem_schwere','ax_schwerefestpunkt','ax_schwerestatus_schwere','ax_seilbahnschwebebahn','ax_selektionsmasstab_benutzungsauftrag','ax_sicherungspunkt','ax_sickerstrecke','ax_siedlungsflaeche','ax_skizze','ax_skizzenart_skizze','ax_soll','ax_sonstigeangaben_bodenschaetzung','ax_sonstigeangaben_musterundvergleichsstueck','ax_sonstigervermessungspunkt','ax_sonstigesbauwerkodersonstigeeinrichtung','ax_sonstigesrecht','ax_speicherinhalt_vorratsbehaelterspeicherbauwerk','ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung','ax_sportfreizeitunderholungsflaeche','ax_spurweite','ax_stehendesgewaesser','ax_strasse','ax_strassenachse','ax_strassenverkehr','ax_strassenverkehrsanlage','ax_strukturlinie3d','ax_sumpf','ax_tagebaugrubesteinbruch','ax_tagesabschnitt','ax_tauglichkeit_gcg','ax_testgelaende','ax_textur3d','ax_tidemerkmal_meer','ax_topographischelinie','ax_transportanlage','ax_tuer3d','ax_turm','ax_typ_textur3d','ax_unlandvegetationsloseflaeche','ax_untererbezugspunkt','ax_untergeordnetesgewaesser','ax_ursprung','ax_vegetationsmerkmal','ax_vegetationsmerkmal_gehoelz','ax_vegetationsmerkmal_landwirtschaft','ax_vegetationsmerkmal_wald','ax_verarbeitungsart_einrichtungsauftrag','ax_verarbeitungsart_fortfuehrungsauftrag','ax_verband','ax_verkehrsdienst_bahnstrecke','ax_verkehrsdienst_bahnverkehrsanlage','ax_vertrauenswuerdigkeit_punktort','ax_vertrauenswuerdigkeit_schwere','ax_vertretung','ax_verwaltung','ax_verwaltungsgemeinschaft','ax_verwendeteobjekte_abgeleitetehoehenlinie','ax_verwendeteobjekte_dhmgitter','ax_vorratsbehaelterspeicherbauwerk','ax_wald','ax_wandflaeche3d','ax_wasserlauf','ax_wasserspiegelhoehe','ax_wasserverhaeltnisse','ax_weg','ax_wegpfadsteig','ax_weitere_gebaeudefunktion','ax_weiterenutzung_tatsaechlichenutzung','ax_widmung_kanal','ax_widmung_stehendesgewaesser','ax_widmung_strasse','ax_widmung_wasserlauf','ax_wirtschaftlicheeinheit','ax_wirtschaftsart','ax_wirtschaftsartgrundbuch','ax_wohnbauflaeche','ax_wohnplatz','ax_zone_schutzzone','ax_zustand','ax_zustand_bahnverkehr','ax_zustand_bahnverkehrsanlage','ax_zustand_bauwerkimgewaesserbereich','ax_zustand_bauwerkimverkehrsbereich','ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe','ax_zustand_bauwerkoderanlagefuersportfreizeitunderholung','ax_zustand_bergbaubetrieb','ax_zustand_boeschungkliff','ax_zustand_einrichtungenfuerdenschiffsverkehr','ax_zustand_flaechebesondererfunktionalerpraegung','ax_zustand_flaechegemischternutzung','ax_zustand_flugverkehr','ax_zustand_flugverkehrsanlage','ax_zustand_friedhof','ax_zustand_gebaeude','ax_zustand_gebietsgrenze','ax_zustand_gewaesserachse','ax_zustand_halde','ax_zustand_hoehleneingang','ax_zustand_industrieundgewerbeflaeche','ax_zustand_kanal','ax_zustand_naturumweltoderbodenschutzrecht','ax_zustand_schiffsverkehr','ax_zustand_schleuse','ax_zustand_sportfreizeitunderholungsflaeche','ax_zustand_stehendesgewaesser','ax_zustand_strasse','ax_zustand_tagebaugrubesteinbruch','ax_zustand_turm','ax_zustand_vegetationsmerkmal','ax_zustand_wald','ax_zustand_wohnbauflaeche','ax_zustandsstufe','ci_rolecode','lb_binnengewaesser','lb_blattform_holzigevegetation','lb_eis','lb_eisart_eis','lb_festgestein','lb_fliesseigenschaft_binnengewaesser','lb_gewaesserart_binnengewaesser','lb_hochbau','lb_holzigevegetation','lb_krautigevegetation','lb_lockermaterial','lb_meer','lb_meerart_meer','lb_oberflaechenmaterial_lockermaterial','lb_tiefbau','lb_vegetationsmerkmal_holzigevegetation','lb_vegetationsmerkmal_krautigevegetation','lb_wasserfuehrung_binnengewaesser','lb_wassersaettigung_holzigevegetation','lb_wassersaettigung_krautigevegetation','lb_wassersaettigung_lockermaterial','ln_abbau','ln_abbaugut_abbau','ln_aquakulturundfischereiwirtschaft','ln_art_abbau','ln_art_aquakulturundfischereiwirtschaft','ln_art_forstwirtschaft','ln_art_freiluftundnaherholung','ln_art_freizeitanlage','ln_art_gewerblichedienstleistungen','ln_art_industrieundverarbeitendesgewerbe','ln_art_versorgungundentsorgung','ln_art_wasserwirtschaft','ln_artderbestattungsflaeche_bestattung','ln_artderbetriebsflaeche_landwirtschaft','ln_artdesparkplatzes_strassenundwegeverkehr','ln_bahnverkehr','ln_bestattung','ln_bewirtschaftung_landwirtschaft','ln_flugverkehr','ln_foerdergut_abbau','ln_forstwirtschaft','ln_freiluftundnaherholung','ln_freizeitanlage','ln_funktion_bahnverkehr','ln_funktion_flugverkehr','ln_funktion_kulturundunterhaltung','ln_funktion_lagerung','ln_funktion_oeffentlicheeinrichtungen','ln_funktion_schiffsverkehr','ln_funktion_schutzanlage','ln_funktion_strassenundwegeverkehr','ln_funktion_wasserwirtschaft','ln_fussgaengerzone_strassenundwegeverkehr','ln_gewerblichedienstleistungen','ln_hafenkategorie_schiffsverkehr','ln_industrieundverarbeitendesgewerbe','ln_kulturundunterhaltung','ln_lagergut_lagerung','ln_lagerung','ln_landwirtschaft','ln_nutzung_flugverkehr','ln_nutzung_schiffsverkehr','ln_oberflaeche_lagerung','ln_oeffentlicheeinrichtungen','ln_ohnenutzung','ln_primaerenergie_versorgungundentsorgung','ln_schiffsverkehr','ln_schutzanlage','ln_sportanlage','ln_sportart_sportanlage','ln_strassenundwegeverkehr','ln_typ_schutzanlage','ln_versorgungundentsorgung','ln_wasserwirtschaft','ln_wohnnutzung','ln_zeitlichkeit_wohnnutzung','ln_zuflusssteuerung_schutzanlage','ln_zustand_abbau','ln_zustand_bahnverkehr','ln_zustand_flugverkehr','ln_zustand_gewerblichedienstleistungen','ln_zustand_industrieundverarbeitendesgewerbe','ln_zustand_kulturundunterhaltung','ln_zustand_lagerung','ln_zustand_oeffentlicheeinrichtungen','ln_zustand_schiffsverkehr','ln_zustand_strassenundwegeverkehr','ln_zustand_versorgungundentsorgung','ln_zustand_wasserwirtschaft','ln_zustand_wohnnutzung','nas_filter_capabilities','operation')
		     AND table_schema=current_schema
	LOOP
		sql := sql || delim || 'SELECT gml_id,beginnt,endet,''' || c.table_name || ''' AS table_name FROM ' || c.table_name;
		delim := ' UNION ';
	END LOOP;

	EXECUTE sql;

	CREATE VIEW vbeziehungen AS
		SELECT beziehung_von,(SELECT DISTINCT table_name FROM vobjekte WHERE gml_id=beziehung_von) AS typ_von
			,beziehungsart
			,beziehung_zu,(SELECT DISTINCT table_name FROM vobjekte WHERE gml_id=beziehung_zu) AS typ_zu
		FROM alkis_beziehungen;

	RETURN 'ALKIS-Views erzeugt.';
END;
$$ LANGUAGE plpgsql;

-- Wenn die Datenbank MIT Historie angelegt wurde, kann nach dem Laden hiermit aufgeräumt werden.
CREATE OR REPLACE FUNCTION :"parent_schema".alkis_delete_all_endet() RETURNS void AS $$
DECLARE
	c RECORD;
BEGIN
	-- In allen Tabellen die Objekte löschen, die ein Ende-Datum haben
	FOR c IN
		SELECT table_name
		FROM information_schema.columns a
		WHERE a.column_name='endet' AND a.is_updatable='YES' AND table_schema=current_schema()
		ORDER BY table_name
	LOOP
		EXECUTE 'DELETE FROM ' || c.table_name || ' WHERE NOT endet IS NULL';
		-- RAISE NOTICE 'Lösche ''endet'' in: %', c.table_name;
	END LOOP;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION :"parent_schema".alkis_exception() RETURNS void AS $$
BEGIN
	RAISE EXCEPTION 'raising deliberate exception';
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION :"parent_schema".alkis_hist_check() RETURNS varchar AS $$
DECLARE
	c RECORD;
	n INTEGER;
	r VARCHAR;
BEGIN
	FOR c IN SELECT table_name FROM information_schema.tables WHERE table_schema=current_schema() AND table_name IN ('aa_advstandardmodell','aa_aktivitaet','aa_anlassart','aa_anlassart_benutzungsauftrag','aa_antrag','aa_antragsgebiet','aa_art_themendefinition','aa_besonderemeilensteinkategorie','aa_gefuehrteobjektart','aa_meilenstein','aa_nas_ausgabeform','aa_projektsteuerung','aa_themendimension','aa_vorgang','aa_weiteremodellart','ap_darstellung','ap_dateityp_3d','ap_fpo','ap_horizontaleausrichtung','ap_kpo_3d','ap_lpo','ap_lto','ap_ppo','ap_pto','ap_vertikaleausrichtung','au_koerperobjekt_3d','au_mehrfachlinienobjekt_3d','au_punkthaufenobjekt_3d','au_trianguliertesoberflaechenobjekt_3d','au_umringobjekt_3d','ax_abbaugut_bergbaubetrieb','ax_abbaugut_tagebaugrubesteinbruch','ax_abgabeversion','ax_abgeleitetehoehenlinie','ax_abschlussflaeche3d','ax_abschnitt','ax_administrative_funktion','ax_anderefestlegungnachstrassenrecht','ax_anderefestlegungnachwasserrecht','ax_anrede_person','ax_anschrift','ax_anzahlderstreckengleise','ax_archaeologischertyp_historischesbauwerkoderhistorischee','ax_art_adressat_auszug','ax_art_baublock','ax_art_bauwerkspunkt','ax_art_bereichzeitlich','ax_art_dammwalldeich','ax_art_einrichtungenfuerdenschiffsverkehr','ax_art_einrichtunginoeffentlichenbereichen','ax_art_flaeche3d','ax_art_flugverkehr','ax_art_flugverkehrsanlage','ax_art_gebaeudepunkt','ax_art_gewaessermerkmal','ax_art_gleis','ax_art_heilquellegasquelle','ax_art_punkt3d','ax_art_punktkennung','ax_art_punktwolke3d','ax_art_reservierung','ax_art_schifffahrtsliniefaehrverkehr','ax_art_strassenverkehrsanlage','ax_art_strukturlinie3d','ax_art_verband','ax_art_verbandsgemeinde','ax_art_wegpfadsteig','ax_artderbebauung_flaechebesondererfunktionalerpraegung','ax_artderbebauung_flaechegemischternutzung','ax_artderbebauung_siedlungsflaeche','ax_artderbebauung_wohnbauflaeche','ax_artderfestlegung_anderefestlegungnachstrassenrecht','ax_artderfestlegung_anderefestlegungnachwasserrecht','ax_artderfestlegung_bauraumoderbodenordnungsrecht','ax_artderfestlegung_denkmalschutzrecht','ax_artderfestlegung_forstrecht','ax_artderfestlegung_klassifizierungnachstrassenrecht','ax_artderfestlegung_klassifizierungnachwasserrecht','ax_artderfestlegung_naturumweltoderbodenschutzrecht','ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc','ax_artderfestlegung_schutzgebietnachwasserrecht','ax_artderfestlegung_sonstigesrecht','ax_artderflurstuecksgrenze_besondereflurstuecksgrenze','ax_artdergebietsgrenze_gebietsgrenze','ax_artdergewaesserstationierungsachse','ax_artderrechtsgemeinschaft_namensnummer','ax_artdesnullpunktes_nullpunkt','ax_artdespolders','ax_ast','ax_aufnahmepunkt','ax_ausgabemedium_benutzer','ax_auspraegung','ax_bahnhofskategorie_bahnverkehrsanlage','ax_bahnkategorie','ax_bahnkategorie_bahnstrecke','ax_bahnkategorie_bahnverkehrsanlage','ax_bahnkategorie_gleis','ax_bahnkategorie_seilbahnschwebebahn','ax_bahnstrecke','ax_bahnverkehr','ax_bahnverkehrsanlage','ax_bauart_bauteil','ax_baublock','ax_bauraumoderbodenordnungsrecht','ax_bauraumoderbodenordnungsrechtgrundbuch','ax_bauteil','ax_bauteil3d','ax_bauweise_gebaeude','ax_bauwerk3d','ax_bauwerkimgewaesserbereich','ax_bauwerkimverkehrsbereich','ax_bauwerkoderanlagefuerindustrieundgewerbe','ax_bauwerkoderanlagefuersportfreizeitunderholung','ax_bauwerksfunktion_bauwerkimgewaesserbereich','ax_bauwerksfunktion_bauwerkimverkehrsbereich','ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer','ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde','ax_bauwerksfunktion_leitung','ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun','ax_bauwerksfunktion_transportanlage','ax_bauwerksfunktion_turm','ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk','ax_bedeutung_grablochderbodenschaetzung','ax_befahrbarkeit_fahrwegachse','ax_befestigung_fahrwegachse','ax_befestigung_wegpfadsteig','ax_behoerde','ax_bemerkungzurabmarkung_grenzpunkt','ax_benutzer','ax_benutzergruppemitzugriffskontrolle','ax_benutzergruppenba','ax_berechnungsmethode','ax_bergbaubetrieb','ax_beschaffenheit_besonderegebaeudelinie','ax_besonderefahrstreifen','ax_besondereflurstuecksgrenze','ax_besonderefunktion_forstrecht','ax_besonderegebaeudelinie','ax_besondererbauwerkspunkt','ax_besonderergebaeudepunkt','ax_besonderertopographischerpunkt','ax_besondereverkehrsbedeutung','ax_bewertung','ax_bewuchs_vegetationsmerkmal','ax_bezeichnung_verwaltungsgemeinschaft','ax_bezugspunktdach','ax_blattart_buchungsblatt','ax_blattart_historischesflurstueck','ax_bodenart_bodenschaetzung','ax_bodenart_musterundvergleichsstueck','ax_bodenflaeche3d','ax_bodenschaetzung','ax_bodenstufe','ax_boeschungkliff','ax_boeschungsflaeche','ax_buchungsart_anlieger','ax_buchungsart_buchungsstelle','ax_buchungsblatt','ax_buchungsblattbezirk','ax_buchungsstelle','ax_bundesland','ax_dachflaeche3d','ax_dachform','ax_dachgeschossausbau_gebaeude','ax_dammwalldeich','ax_darstellung_gebaeudeausgestaltung','ax_dateitypraster_textur3d','ax_datenerhebung','ax_datenerhebung_punktort','ax_datenerhebung_schwere','ax_datenerhebung3d','ax_datenformat_benutzer','ax_denkmalschutzrecht','ax_dhmgitter','ax_dienststelle','ax_duene','ax_eigentuemerart_namensnummer','ax_einrichtungenfuerdenschiffsverkehr','ax_einrichtunginoeffentlichenbereichen','ax_einschnitt','ax_elektrifizierung','ax_entstehungsart','ax_ergebnisderueberpruefung_bauwerkeeinrichtungenundsonsti','ax_ergebnisderueberpruefung_dammwalldeich','ax_ergebnisderueberpruefung_klassifizierungnachwasserrecht','ax_ergebnisderueberpruefung_tatsaechlichenutzung','ax_fahrbahnachse','ax_fahrbahntrennung_strasse','ax_fahrwegachse','ax_felsenfelsblockfelsnadel','ax_fenster3d','ax_firstlinie','ax_flaeche3d','ax_flaechebesondererfunktionalerpraegung','ax_flaechegemischternutzung','ax_fliessgewaesser','ax_flugverkehr','ax_flugverkehrsanlage','ax_flurstueck','ax_flurstueckgrundbuch','ax_foerdergut_industrieundgewerbeflaeche','ax_forstrecht','ax_fortfuehrungsauftrag','ax_fortfuehrungsfall','ax_fortfuehrungsfallgrundbuch','ax_fortfuehrungsnachweisdeckblatt','ax_friedhof','ax_funktion_bahnverkehr','ax_funktion_bauwerk','ax_funktion_bergbaubetrieb','ax_funktion_dammwalldeich','ax_funktion_einschnitt','ax_funktion_fahrbahnachse','ax_funktion_flaechebesondererfunktionalerpraegung','ax_funktion_flaechegemischternutzung','ax_funktion_fliessgewaesser','ax_funktion_flugverkehr','ax_funktion_friedhof','ax_funktion_gebaeudeinstallation3d','ax_funktion_gehoelz','ax_funktion_gewaesserachse','ax_funktion_hafenbecken','ax_funktion_industrieundgewerbeflaeche','ax_funktion_lagefestpunkt','ax_funktion_meer','ax_funktion_platz','ax_funktion_polder','ax_funktion_referenzstationspunkt','ax_funktion_schiffsverkehr','ax_funktion_schutzgebietnachwasserrecht','ax_funktion_schwerefestpunkt','ax_funktion_siedlungsflaeche','ax_funktion_sportfreizeitunderholungsflaeche','ax_funktion_stehendesgewaesser','ax_funktion_strasse','ax_funktion_strassenachse','ax_funktion_tagebaugrubesteinbruch','ax_funktion_unlandvegetationsloseflaeche','ax_funktion_untergeordnetesgewaesser','ax_funktion_vegetationsmerkmal','ax_funktion_weg','ax_funktion_wegachse','ax_funktion_wohnbauflaeche','ax_gebaeude','ax_gebaeudeausgestaltung','ax_gebaeudefunktion','ax_gebaeudeinstallation3d','ax_gebiet_bundesland','ax_gebiet_kreis','ax_gebiet_nationalstaat','ax_gebiet_regierungsbezirk','ax_gebiet_verwaltungsgemeinschaft','ax_gebietsgrenze','ax_gehoelz','ax_gemarkung','ax_gemarkungsteilflur','ax_gemeinde','ax_gemeindeteil','ax_genauigkeitsstufe_punktort','ax_genauigkeitsstufe_schwere','ax_georeferenziertegebaeudeadresse','ax_gewaesserachse','ax_gewaessermerkmal','ax_gewaesserstationierungsachse','ax_gewann','ax_gleis','ax_gnsstauglichkeit','ax_grablochderbodenschaetzung','ax_grenzpunkt','ax_grenzuebergang','ax_gruendederausgesetztenabmarkung_grenzpunkt','ax_hafen','ax_hafenbecken','ax_hafenkategorie_hafen','ax_halde','ax_heide','ax_heilquellegasquelle','ax_historischesbauwerkoderhistorischeeinrichtung','ax_historischesflurstueck','ax_historischesflurstueckalb','ax_historischesflurstueckohneraumbezug','ax_hoehenfestpunkt','ax_hoehenlinie','ax_hoehleneingang','ax_horizontfreiheit_grenzpunkt','ax_horizontfreiheit_netzpunkt','ax_hydrologischesmerkmal_fliessgewaesser','ax_hydrologischesmerkmal_gewaesserachse','ax_hydrologischesmerkmal_gewaessermerkmal','ax_hydrologischesmerkmal_heilquellegasquelle','ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri','ax_hydrologischesmerkmal_stehendesgewaesser','ax_hydrologischesmerkmal_untergeordnetesgewaesser','ax_impliziteloeschungderreservierung','ax_industrieundgewerbeflaeche','ax_insel','ax_internationalebedeutung_strasse','ax_kanal','ax_klassifikation_hierarchiestufe3d_lagefestpunkt','ax_klassifikation_ordnung_lagefestpunkt','ax_klassifikation_wertigkeit_lagefestpunkt','ax_klassifizierung_bewertung','ax_klassifizierungnachstrassenrecht','ax_klassifizierungnachwasserrecht','ax_kleinraeumigerlandschaftsteil','ax_klimastufe','ax_kommunalesgebiet','ax_kommunalesteilgebiet','ax_kondominium','ax_konstruktionsmerkmalbauart_schleuse','ax_koordinatenstatus_punktort','ax_kreisregion','ax_lagebezeichnungkatalogeintrag','ax_lagebezeichnungmithausnummer','ax_lagebezeichnungmitpseudonummer','ax_lagebezeichnungohnehausnummer','ax_lagefestpunkt','ax_lagergut_halde','ax_lagergut_industrieundgewerbeflaeche','ax_lagezurerdoberflaeche_gebaeude','ax_lagezurerdoberflaeche_transportanlage','ax_lagezurerdoberflaeche_untergeordnetesgewaesser','ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk','ax_lagezuroberflaeche_gleis','ax_landschaft','ax_landschaftstyp','ax_landwirtschaft','ax_leitung','ax_letzteabgabeart','ax_li_processstep_bodenhoehe_description','ax_li_processstep_dachhoehe_source','ax_li_processstep_mitdatenerhebung_description','ax_li_processstep_ohnedatenerhebung_description','ax_li_processstep_punktort_description','ax_li_processstep3d_description','ax_liniendarstellung_topographischelinie','ax_marke','ax_markierung_wegachse','ax_markierung_wegpfadsteig','ax_material_materialeigenschaft_material3d','ax_material3d','ax_meer','ax_merkmal_musterundvergleichsstueck','ax_messmethode_schwere','ax_moor','ax_musterundvergleichsstueck','ax_namensnummer','ax_nationalstaat','ax_naturumweltoderbodenschutzrecht','ax_netzknoten','ax_null_listenelement3d','ax_nullpunkt','ax_nutzung','ax_nutzung_flugverkehr','ax_nutzung_hafen','ax_nutzung_hafenbecken','ax_nutzung_stehendesgewaesser','ax_nutzung_wald','ax_nutzungsart_bodenschaetzung','ax_nutzungsart_musterundvergleichsstueck','ax_obererbezugspunkt','ax_oberflaechenmaterial_flugverkehrsanlage','ax_oberflaechenmaterial_strasse','ax_oberflaechenmaterial_unlandvegetationsloseflaeche','ax_ordnung_hoehenfestpunkt','ax_ordnung_schwerefestpunkt','ax_ortslage','ax_person','ax_personengruppe','ax_platz','ax_polder','ax_primaerenergie_industrieundgewerbeflaeche','ax_produkt_transportanlage','ax_punkt3d','ax_punktkennunguntergegangen','ax_punktkennungvergleichend','ax_punktortag','ax_punktortau','ax_punktortta','ax_punktstabilitaet','ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet','ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung','ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand','ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes','ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra','ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi','ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt','ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili','ax_punktwolke3d','ax_qualitaet_hauskoordinate','ax_rechtszustand_schutzzone','ax_referenzstationspunkt','ax_regierungsbezirk','ax_reservierung','ax_reservierungsart_reservierungsauftrag','ax_rolledergemeinde','ax_schifffahrtskategorie','ax_schifffahrtskategorie_gewaesserstationierungsachse','ax_schifffahrtskategorie_kanal','ax_schifffahrtsliniefaehrverkehr','ax_schiffsverkehr','ax_schleuse','ax_schutzgebietnachnaturumweltoderbodenschutzrecht','ax_schutzgebietnachwasserrecht','ax_schutzzone','ax_schwere','ax_schwereanomalie_schwere_art','ax_schwerebezugssystem_schwere','ax_schwerefestpunkt','ax_schwerestatus_schwere','ax_seilbahnschwebebahn','ax_selektionsmasstab_benutzungsauftrag','ax_sicherungspunkt','ax_sickerstrecke','ax_siedlungsflaeche','ax_skizze','ax_skizzenart_skizze','ax_soll','ax_sonstigeangaben_bodenschaetzung','ax_sonstigeangaben_musterundvergleichsstueck','ax_sonstigervermessungspunkt','ax_sonstigesbauwerkodersonstigeeinrichtung','ax_sonstigesrecht','ax_speicherinhalt_vorratsbehaelterspeicherbauwerk','ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung','ax_sportfreizeitunderholungsflaeche','ax_spurweite','ax_stehendesgewaesser','ax_strasse','ax_strassenachse','ax_strassenverkehr','ax_strassenverkehrsanlage','ax_strukturlinie3d','ax_sumpf','ax_tagebaugrubesteinbruch','ax_tagesabschnitt','ax_tauglichkeit_gcg','ax_testgelaende','ax_textur3d','ax_tidemerkmal_meer','ax_topographischelinie','ax_transportanlage','ax_tuer3d','ax_turm','ax_typ_textur3d','ax_unlandvegetationsloseflaeche','ax_untererbezugspunkt','ax_untergeordnetesgewaesser','ax_ursprung','ax_vegetationsmerkmal','ax_vegetationsmerkmal_gehoelz','ax_vegetationsmerkmal_landwirtschaft','ax_vegetationsmerkmal_wald','ax_verarbeitungsart_einrichtungsauftrag','ax_verarbeitungsart_fortfuehrungsauftrag','ax_verband','ax_verkehrsdienst_bahnstrecke','ax_verkehrsdienst_bahnverkehrsanlage','ax_vertrauenswuerdigkeit_punktort','ax_vertrauenswuerdigkeit_schwere','ax_vertretung','ax_verwaltung','ax_verwaltungsgemeinschaft','ax_verwendeteobjekte_abgeleitetehoehenlinie','ax_verwendeteobjekte_dhmgitter','ax_vorratsbehaelterspeicherbauwerk','ax_wald','ax_wandflaeche3d','ax_wasserlauf','ax_wasserspiegelhoehe','ax_wasserverhaeltnisse','ax_weg','ax_wegpfadsteig','ax_weitere_gebaeudefunktion','ax_weiterenutzung_tatsaechlichenutzung','ax_widmung_kanal','ax_widmung_stehendesgewaesser','ax_widmung_strasse','ax_widmung_wasserlauf','ax_wirtschaftlicheeinheit','ax_wirtschaftsart','ax_wirtschaftsartgrundbuch','ax_wohnbauflaeche','ax_wohnplatz','ax_zone_schutzzone','ax_zustand','ax_zustand_bahnverkehr','ax_zustand_bahnverkehrsanlage','ax_zustand_bauwerkimgewaesserbereich','ax_zustand_bauwerkimverkehrsbereich','ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe','ax_zustand_bauwerkoderanlagefuersportfreizeitunderholung','ax_zustand_bergbaubetrieb','ax_zustand_boeschungkliff','ax_zustand_einrichtungenfuerdenschiffsverkehr','ax_zustand_flaechebesondererfunktionalerpraegung','ax_zustand_flaechegemischternutzung','ax_zustand_flugverkehr','ax_zustand_flugverkehrsanlage','ax_zustand_friedhof','ax_zustand_gebaeude','ax_zustand_gebietsgrenze','ax_zustand_gewaesserachse','ax_zustand_halde','ax_zustand_hoehleneingang','ax_zustand_industrieundgewerbeflaeche','ax_zustand_kanal','ax_zustand_naturumweltoderbodenschutzrecht','ax_zustand_schiffsverkehr','ax_zustand_schleuse','ax_zustand_sportfreizeitunderholungsflaeche','ax_zustand_stehendesgewaesser','ax_zustand_strasse','ax_zustand_tagebaugrubesteinbruch','ax_zustand_turm','ax_zustand_vegetationsmerkmal','ax_zustand_wald','ax_zustand_wohnbauflaeche','ax_zustandsstufe','ci_rolecode','lb_binnengewaesser','lb_blattform_holzigevegetation','lb_eis','lb_eisart_eis','lb_festgestein','lb_fliesseigenschaft_binnengewaesser','lb_gewaesserart_binnengewaesser','lb_hochbau','lb_holzigevegetation','lb_krautigevegetation','lb_lockermaterial','lb_meer','lb_meerart_meer','lb_oberflaechenmaterial_lockermaterial','lb_tiefbau','lb_vegetationsmerkmal_holzigevegetation','lb_vegetationsmerkmal_krautigevegetation','lb_wasserfuehrung_binnengewaesser','lb_wassersaettigung_holzigevegetation','lb_wassersaettigung_krautigevegetation','lb_wassersaettigung_lockermaterial','ln_abbau','ln_abbaugut_abbau','ln_aquakulturundfischereiwirtschaft','ln_art_abbau','ln_art_aquakulturundfischereiwirtschaft','ln_art_forstwirtschaft','ln_art_freiluftundnaherholung','ln_art_freizeitanlage','ln_art_gewerblichedienstleistungen','ln_art_industrieundverarbeitendesgewerbe','ln_art_versorgungundentsorgung','ln_art_wasserwirtschaft','ln_artderbestattungsflaeche_bestattung','ln_artderbetriebsflaeche_landwirtschaft','ln_artdesparkplatzes_strassenundwegeverkehr','ln_bahnverkehr','ln_bestattung','ln_bewirtschaftung_landwirtschaft','ln_flugverkehr','ln_foerdergut_abbau','ln_forstwirtschaft','ln_freiluftundnaherholung','ln_freizeitanlage','ln_funktion_bahnverkehr','ln_funktion_flugverkehr','ln_funktion_kulturundunterhaltung','ln_funktion_lagerung','ln_funktion_oeffentlicheeinrichtungen','ln_funktion_schiffsverkehr','ln_funktion_schutzanlage','ln_funktion_strassenundwegeverkehr','ln_funktion_wasserwirtschaft','ln_fussgaengerzone_strassenundwegeverkehr','ln_gewerblichedienstleistungen','ln_hafenkategorie_schiffsverkehr','ln_industrieundverarbeitendesgewerbe','ln_kulturundunterhaltung','ln_lagergut_lagerung','ln_lagerung','ln_landwirtschaft','ln_nutzung_flugverkehr','ln_nutzung_schiffsverkehr','ln_oberflaeche_lagerung','ln_oeffentlicheeinrichtungen','ln_ohnenutzung','ln_primaerenergie_versorgungundentsorgung','ln_schiffsverkehr','ln_schutzanlage','ln_sportanlage','ln_sportart_sportanlage','ln_strassenundwegeverkehr','ln_typ_schutzanlage','ln_versorgungundentsorgung','ln_wasserwirtschaft','ln_wohnnutzung','ln_zeitlichkeit_wohnnutzung','ln_zuflusssteuerung_schutzanlage','ln_zustand_abbau','ln_zustand_bahnverkehr','ln_zustand_flugverkehr','ln_zustand_gewerblichedienstleistungen','ln_zustand_industrieundverarbeitendesgewerbe','ln_zustand_kulturundunterhaltung','ln_zustand_lagerung','ln_zustand_oeffentlicheeinrichtungen','ln_zustand_schiffsverkehr','ln_zustand_strassenundwegeverkehr','ln_zustand_versorgungundentsorgung','ln_zustand_wasserwirtschaft','ln_zustand_wohnnutzung','nas_filter_capabilities','operation')
	LOOP
		EXECUTE 'SELECT count(*) FROM ' || c.table_name || ' WHERE endet IS NULL GROUP BY gml_id HAVING count(*)>1' INTO n;
		IF n>1 THEN
			r := alkis_string_append(r, c.table_name || ': ' || n || ' Objekte, die in mehreren Versionen nicht beendet sind.');
		END IF;

		EXECUTE 'SELECT count(*) FROM ' || c.table_name || ' WHERE beginnt>=endet' INTO n;
		IF n>1 THEN
			r := alkis_string_append(r, c.table_name || ': ' || n || ' Objekte mit ungültiger Lebensdauer.');
		END IF;

		EXECUTE 'SELECT count(*)'
			|| ' FROM ' || c.table_name || ' a'
			|| ' JOIN ' || c.table_name || ' b ON a.gml_id=b.gml_id AND a.ogc_fid<>b.ogc_fid AND a.beginnt<b.endet AND a.endet>b.beginnt'
			INTO n;
		IF n>0 THEN
			r := alkis_string_append(r, c.table_name || ': ' || n || ' Lebensdauerüberschneidungen.');
		END IF;
	END LOOP;

	RETURN coalesce(r,'Keine Fehler gefunden.');
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION :"parent_schema".alkis_bufferline(g geometry,offs float8) RETURNS geometry AS $$
BEGIN
	BEGIN
		RETURN st_buffer(g,offs,'endcap=flat');
	EXCEPTION WHEN OTHERS THEN
		IF geometrytype(g) = 'LINESTRING' THEN
			DECLARE
				g0 GEOMETRY;
				g1 GEOMETRY;
				g2 GEOMETRY;
			BEGIN
				SELECT alkis_offsetcurve(g,offs,'') INTO g0;
				SELECT st_reverse( alkis_offsetcurve(g,-offs,'') ) INTO g1;

				g2 := st_makepolygon( st_linemerge( st_collect(
					ARRAY[
						g0, st_makeline( st_endpoint(g0), st_startpoint(g1) ),
						g1, st_makeline( st_endpoint(g1), st_startpoint(g0) )
					]
				) ) );

				IF geometrytype(g2) <> 'POLYGON' THEN
					RAISE EXCEPTION 'alkis_bufferline: POLYGON expected, % found', geometrytype(g2);
				END IF;

				RETURN g2;
			END;
		ELSE
			RAISE EXCEPTION 'alkis_bufferline: LINESTRING expected, % found', geometrytype(g);
		END IF;
	END;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION pg_temp.create_accum() RETURNS void AS $$
BEGIN
  CREATE AGGREGATE alkis_accum (anycompatiblearray) (
    sfunc = array_cat,
    stype = anycompatiblearray,
    initcond = '{}'
  );
EXCEPTION
  WHEN duplicate_function THEN
    -- pass
  WHEN OTHERS THEN
    BEGIN
      CREATE AGGREGATE alkis_accum (anyarray) (
        sfunc = array_cat,
        stype = anyarray,
        initcond = '{}'
      );
    EXCEPTION
      WHEN duplicate_function THEN
        -- pass
    END;
END;
$$ LANGUAGE plpgsql;

SELECT pg_temp.create_accum();

-- Löschsatz verarbeiten (MIT Historie)
-- context='delete'        => "endet" auf Datenstand (pg_temp.deletadate) setzen
-- context='replace'       => "endet" des ersetzten auf "beginnt" des neuen Objekts setzen
-- context='update'        => "endet" auf übergebene Zeit setzen und "anlass" festhalten
CREATE OR REPLACE FUNCTION :"alkis_schema".delete_feature_hist() RETURNS TRIGGER AS $$
DECLARE
	n INTEGER;
	beginnt TEXT;
	s TEXT;
BEGIN
	NEW.context := coalesce(lower(NEW.context),'delete');

	IF length(NEW.featureid)=32 THEN
		beginnt := substr(NEW.featureid, 17, 4) || '-'
			|| substr(NEW.featureid, 21, 2) || '-'
			|| substr(NEW.featureid, 23, 2) || 'T'
			|| substr(NEW.featureid, 26, 2) || ':'
			|| substr(NEW.featureid, 28, 2) || ':'
			|| substr(NEW.featureid, 30, 2) || 'Z'
			;
	ELSIF length(NEW.featureid)=16 THEN
		-- Ältestes nicht gelöschtes Objekt
		EXECUTE 'SELECT min(beginnt) FROM ' || NEW.typename
			|| ' WHERE gml_id=''' || NEW.featureid || ''''
			|| ' AND endet IS NULL'
			INTO beginnt;

		IF beginnt IS NULL THEN
			RAISE EXCEPTION '%: Keinen Kandidaten zum Löschen gefunden.', NEW.featureid;
		END IF;
	ELSE
		RAISE EXCEPTION '%: Identifikator gescheitert.', NEW.featureid;
	END IF;

	IF NEW.context='delete' THEN
		SELECT endet INTO NEW.endet FROM pg_temp.deletedate;

	ELSIF NEW.context='update' THEN
		IF NEW.endet IS NULL THEN
			RAISE EXCEPTION '%: Endedatum nicht gesetzt', NEW.featureid;
		END IF;

	ELSIF NEW.context='replace' THEN
		NEW.safetoignore := lower(NEW.safetoignore);
		IF NEW.safetoignore IS NULL THEN
			RAISE EXCEPTION '%: safeToIgnore nicht gesetzt.', NEW.featureid;
		ELSIF NEW.safetoignore<>'true' AND NEW.safetoignore<>'false' THEN
			RAISE EXCEPTION '%: safeToIgnore ''%'' ungültig (''true'' oder ''false'' erwartet).', NEW.featureid, NEW.safetoignore;
		END IF;

		IF length(NEW.replacedby)=32 AND NEW.replacedby<>NEW.featureid THEN
			NEW.endet := substr(NEW.replacedby, 17, 4) || '-'
				  || substr(NEW.replacedby, 21, 2) || '-'
				  || substr(NEW.replacedby, 23, 2) || 'T'
				  || substr(NEW.replacedby, 26, 2) || ':'
				  || substr(NEW.replacedby, 28, 2) || ':'
				  || substr(NEW.replacedby, 30, 2) || 'Z'
				  ;
		END IF;

		IF NEW.endet IS NULL THEN
			-- Beginn des ersten Nachfolgeobjektes
			EXECUTE 'SELECT min(beginnt) FROM ' || NEW.typename || ' a'
				|| ' WHERE gml_id=''' || substr(NEW.replacedby, 1, 16) || ''''
				|| ' AND beginnt>''' || beginnt || ''''
				INTO NEW.endet;
		ELSE
			EXECUTE 'SELECT count(*) FROM ' || NEW.typename
				|| ' WHERE gml_id=''' || substr(NEW.replacedby, 1, 16) || ''''
				|| ' AND beginnt=''' || NEW.endet || ''''
				INTO n;
			IF n<>1 THEN
				RAISE EXCEPTION '%: Ersatzobjekt % % nicht gefunden.', NEW.featureid, NEW.replacedby, NEW.endet;
			END IF;
		END IF;

		IF NEW.endet IS NULL THEN
			-- Abbrechen, wenn Austausch nicht ignoriert werden
			-- darf, aber nicht wenn ein Objekt (sinnloserweise?)
			-- gegen selbst getauscht werden soll.
			IF NEW.safetoignore='false' AND NEW.featureid<>NEW.replacedby THEN
				RAISE EXCEPTION '%: Beginn des Ersatzobjekts % nicht gefunden.', NEW.featureid, NEW.replacedby;
				-- RAISE NOTICE '%: Beginn des ersetzenden Objekts % nicht gefunden.', NEW.featureid, NEW.replacedby;
			END IF;

			NEW.ignored=true;
			RETURN NEW;
		END IF;

	ELSE
		RAISE EXCEPTION '%: Ungültiger Kontext % (''delete'', ''replace'' oder ''update'' erwartet).', NEW.featureid, NEW.context;

	END IF;

	s := 'UPDATE ' || NEW.typename || ' SET endet=''' || NEW.endet || '''';

	IF NEW.context='update' AND NEW.anlass IS NOT NULL THEN
		s := s || ',anlass=array_cat(anlass,''{' || array_to_string(NEW.anlass,',') || '}'')';
	END IF;

	s := s || ' WHERE gml_id=''' || substr(NEW.featureid, 1, 16) || ''''
	       || ' AND beginnt=''' || beginnt || ''''
	       ;
	EXECUTE s;
	GET DIAGNOSTICS n = ROW_COUNT;
	-- RAISE NOTICE 'SQL[%]:%', n, s;
	IF n<>1 THEN
		IF n=0 THEN
			s := 'SELECT count(*),min(beginnt) FROM ' || NEW.typename || ' WHERE gml_id=''' || substr(NEW.featureid, 1, 16) || ''' AND endet IS NULL';
			EXECUTE s INTO n, beginnt;
			IF (n=0 AND NEW.context IN ('delete','update')) OR (n=1 AND NEW.context='replace') THEN
				RAISE NOTICE '%: Kein Objekt gefunden [%:%]', NEW.featureid, NEW.context, n;
				NEW.ignored=true;
				RETURN NEW;
			ELSIF n=2 AND beginnt IS NOT NULL THEN
				s := 'UPDATE ' || NEW.typename || ' a SET endet=''' || NEW.endet || '''';

				IF NEW.anlass IS NOT NULL THEN
					s := s || ',anlass=array_cat(anlass,''{' || array_to_string(NEW.anlass,',') || '}'')';
				END IF;

				s := s || ' WHERE gml_id=''' || substr(NEW.featureid, 1, 16) || ''''
				       || ' AND beginnt=''' || beginnt || ''''
				       ;
				EXECUTE s;
				GET DIAGNOSTICS n = ROW_COUNT;
				-- RAISE NOTICE 'SQL[%]:%', n, s;
				IF n<>1 THEN
					RAISE EXCEPTION '%: Aktualisierung des Vorgängerobjekts von % schlug fehl [%:%]', NEW.featureid, beginnt, NEW.context, n;
				END IF;
			ELSE
				RAISE NOTICE '%: Kein eindeutiges Vorgängerobjekt gefunden [%:%]', NEW.featureid, NEW.context, n;
				RETURN NEW;
			END IF;
		ELSE
			RAISE EXCEPTION '%: % schlug fehl [%]', NEW.featureid, NEW.context, n;
		END IF;
	END IF;

	NEW.ignored := false;
	RETURN NEW;
END;
$$ LANGUAGE plpgsql SET search_path = :"alkis_schema", public;

-- Abwandlung der Hist-Version als Kill-Version.
-- Die "gml_id" muss in der Datenbank das Format character(16) haben.
-- Dies kann auch Abgabeart 3100 verarbeiten. Historische Objekte werden aber sofort entfernt.
CREATE OR REPLACE FUNCTION :"alkis_schema".delete_feature_kill() RETURNS TRIGGER AS $$
DECLARE
	n INTEGER;
	beginnt TEXT;
	s TEXT;
BEGIN
	NEW.context := coalesce(lower(NEW.context),'delete');

	IF length(NEW.featureid)=32 THEN
		beginnt := substr(NEW.featureid, 17, 4) || '-'
			|| substr(NEW.featureid, 21, 2) || '-'
			|| substr(NEW.featureid, 23, 2) || 'T'
			|| substr(NEW.featureid, 26, 2) || ':'
			|| substr(NEW.featureid, 28, 2) || ':'
			|| substr(NEW.featureid, 30, 2) || 'Z'
			;
	ELSIF length(NEW.featureid)=16 THEN
		-- Ältestes nicht gelöschtes Objekt
		EXECUTE 'SELECT min(beginnt) FROM ' || NEW.typename
			|| ' WHERE gml_id=''' || NEW.featureid || ''''
			|| ' AND endet IS NULL'
			INTO beginnt;

		IF beginnt IS NULL THEN
			RAISE EXCEPTION '%: Keinen Kandidaten zum Löschen gefunden.', NEW.featureid;
		END IF;
	ELSE
		RAISE EXCEPTION '%: Identifikator gescheitert.', NEW.featureid;
	END IF;

	IF NEW.context='replace' THEN
		NEW.safetoignore := lower(NEW.safetoignore);
		IF NEW.safetoignore IS NULL THEN
			RAISE EXCEPTION '%: safeToIgnore nicht gesetzt.', NEW.featureid;
		ELSIF NEW.safetoignore<>'true' AND NEW.safetoignore<>'false' THEN
			RAISE EXCEPTION '%: safeToIgnore ''%'' ungültig (''true'' oder ''false'' erwartet).', NEW.featureid, NEW.safetoignore;
		END IF;

		IF NEW.featureid=NEW.replacedby THEN
			NEW.ignored := true;
			RETURN NEW;
		END IF;

	ELSIF NEW.context NOT IN ('delete', 'update') THEN
		RAISE EXCEPTION '%: Ungültiger Kontext % (''delete'', ''replace'' oder ''update'' erwartet).', NEW.featureid, NEW.context;

	END IF;

	s := 'DELETE FROM ' || NEW.typename || ' WHERE gml_id=''' || substr(NEW.featureid, 1, 16) || ''' AND beginnt=''' || beginnt || '''';
	EXECUTE s;
	GET DIAGNOSTICS n = ROW_COUNT;
	-- RAISE NOTICE 'SQL[%]:%', n, s;
	IF n=1 THEN
		NEW.ignored := false;
	ELSE
		RAISE NOTICE '%: % schlug fehl ignoriert [%]', NEW.featureid, NEW.context, n;
		NEW.ignored := true;
	END IF;

	RETURN NEW;
END;
$$ LANGUAGE plpgsql SET search_path = :"alkis_schema", public;
