/***************************************************************************
 *                                                                         *
 * Projekt:  norGIS ALKIS Import                                           *
 * Purpose:  ALKIS-Schema ggf. migrieren                                   *
 * Author:   Jürgen E. Fischer <jef@norbit.de>                             *
 *                                                                         *
 ***************************************************************************
 * Copyright (c) 2012-2023, Jürgen E. Fischer <jef@norbit.de>              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

SET client_encoding = 'UTF8';
SET search_path = :"postgis_schema", public;

\i alkis-compat.sql

-- Stored Procedures laden
\i alkis-functions.sql

--
-- Datenbankmigration
--

CREATE FUNCTION pg_temp.alkis_rename_table(t TEXT) RETURNS varchar AS $$
BEGIN
	PERFORM alkis_dropobject(t || '_');

	EXECUTE 'ALTER TABLE ' || t || ' RENAME TO ' || t || '_';

	PERFORM alkis_dropobject(t || '_geom_idx');                -- < GDAL 1.11
	PERFORM alkis_dropobject(t || '_wkb_geometry_geom_idx');   -- >= GDAL 1.11

	RETURN t || ' umbenannt - INHALT MANUELL MIGRIEREN.';
EXCEPTION WHEN OTHERS THEN
	RETURN '';
END;
$$ LANGUAGE plpgsql;

CREATE FUNCTION pg_temp.alkis_update_schema() RETURNS varchar AS $$
DECLARE
	c RECORD;
	s INTEGER;
	v_n INTEGER;
	i INTEGER;
	ver INTEGER;
	r TEXT;
BEGIN
	r := NULL;

	--
	-- ALKIS-Schema
	--
	SELECT count(*) INTO v_n FROM information_schema.columns
		WHERE table_schema=current_schema()
		  AND table_name='ax_flurstueck'
		  AND column_name='sonstigesmodell';
	IF v_n=0 THEN
		RAISE EXCEPTION 'Modell zu alt für Migration.';
	END IF;

	BEGIN
		SELECT version INTO ver FROM alkis_version;

	EXCEPTION WHEN OTHERS THEN
		ver := 0;
		CREATE TABLE alkis_version(version INTEGER);
		INSERT INTO alkis_version(version) VALUES (ver);

		BEGIN ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD name varchar; EXCEPTION WHEN OTHERS THEN END;
	END;

	RAISE NOTICE 'ALKIS-Schema-Version: %', ver;

	IF ver<1 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 1';

		PERFORM alkis_dropobject('ax_tatsaechlichenutzung');
		PERFORM alkis_dropobject('ax_klassifizierung');
		PERFORM alkis_dropobject('ax_ausfuehrendestellen');
		PERFORM alkis_dropobject('v_eigentuemer');
		PERFORM alkis_dropobject('v_haeuser');
		PERFORM alkis_dropobject('v_schutzgebietnachwasserrecht');
		PERFORM alkis_dropobject('v_schutzgebietnachnaturumweltoderbodenschutzrecht');

		ALTER TABLE ax_flurstueck ALTER angabenzumabschnittnummeraktenzeichen TYPE varchar[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER ortsteil TYPE varchar;
		ALTER TABLE ax_historischesflurstueck ALTER zaehler TYPE varchar;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER zaehler TYPE varchar;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER rechtsbehelfsverfahren TYPE varchar;
		ALTER TABLE ax_gebaeude ALTER geschossflaeche TYPE double precision;
		ALTER TABLE ax_gebaeude ALTER grundflaeche TYPE double precision;
		ALTER TABLE ax_gebaeude ALTER umbauterraum TYPE double precision;
		ALTER TABLE ax_bodenschaetzung ALTER bodenzahlodergruenlandgrundzahl TYPE varchar;
		ALTER TABLE ax_bodenschaetzung ALTER ackerzahlodergruenlandzahl TYPE varchar;
		ALTER TABLE ax_grablochderbodenschaetzung ALTER bodenzahlodergruenlandgrundzahl TYPE varchar;
		ALTER TABLE ax_gemarkungsteilflur ADD gehoertzu character(16)[];
		CREATE INDEX ax_gemarkungsteilflur_ghz ON ax_gemarkungsteilflur USING gin (gehoertzu);
		ALTER TABLE ax_dienststelle ADD kennung varchar;
		ALTER TABLE ax_wohnplatz ADD zweitname varchar;
		ALTER TABLE ax_baublock ADD art integer;

		-- gml_id: varchar => character(16)
		s := 0;
		i := 0;
		FOR c IN
			SELECT table_name
			FROM information_schema.columns a
			WHERE a.table_schema=current_schema()
			  AND (a.table_name LIKE 'ax_%' OR a.table_name LIKE 'ap_%')
			  AND a.column_name='gml_id'
			  AND a.data_type='character varying'
		LOOP
			-- RAISE NOTICE '%', 'UPDATE ' || c.table_name || ' SET gml_id=substring(gml_id,1,16) WHERE length(gml_id)>16';
			EXECUTE 'UPDATE ' || c.table_name || ' SET gml_id=substring(gml_id,1,16) WHERE length(gml_id)>16';
			GET DIAGNOSTICS v_n = ROW_COUNT;
			s := s + v_n;

			-- RAISE NOTICE '%', 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN gml_id TYPE character(16)';
			EXECUTE 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN gml_id TYPE character(16)';
			i := i + 1;
		END LOOP;

		IF i > 0 OR s > 0 THEN
			r := alkis_string_append(r, i || ' Tabellen mit ' || s || ' lange AAA-Identifikatoren geändert.');
		END IF;

		-- land, gemarkungsnummer, gemeinde, regierungsbezirk, bezirk, kreis, schluesselgesamt: integer => varchar
		i := 0;
		FOR c IN
			SELECT table_name, column_name
			FROM information_schema.columns a
			WHERE a.table_schema=current_schema()
			  AND a.table_name LIKE 'ax_%'
			  AND a.column_name IN ('land','gemarkungsnummer','gemeinde','regierungsbezirk','bezirk','kreis','schluesselgesamt')
			  AND a.data_type='integer'
		LOOP
			-- RAISE NOTICE '%', 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN ' || c.column_name || ' TYPE character varying';
			EXECUTE 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN ' || c.column_name || ' TYPE character varying';
			i := i + 1;
		END LOOP;

		IF i > 0 THEN
			r := alkis_string_append(r, i || ' Spalten angepaßt (integer->character varying).');
		END IF;

		-- Relationen: varchar => character(16) bzw. varchar[] => character(16)[]
		i := 0;
		FOR c IN
			WITH RECURSIVE
				element(name,base) AS (
					SELECT name,unnest(name||abgeleitet_aus) AS base
					FROM alkis_elemente
				UNION
					SELECT a.name,unnest(b.abgeleitet_aus) AS base
					FROM element a
					JOIN alkis_elemente b ON a.base=b.name
				),
				relation(element,bezeichnung,kennung) AS (
					SELECT element,bezeichnung,kennung FROM alkis_relationsart
					UNION
					SELECT b.element,a.bezeichnung,a.kennung FROM alkis_relationsart a JOIN relation b ON a.element=lower(b.bezeichnung)
				)
			SELECT col.table_name,col.column_name,col.udt_name
			FROM element t
			JOIN relation a ON t.base=a.element
			JOIN information_schema.columns col ON lower(t.name)=col.table_name AND lower(a.bezeichnung)=col.column_name
			WHERE col.udt_name IN ('_varchar','varchar')
		LOOP
			IF c.udt_name='_varchar' THEN
				-- RAISE NOTICE '%', 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN ' || c.column_name || ' TYPE character(16)[]';
				EXECUTE 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN ' || c.column_name || ' TYPE character(16)[]';
				i := i + 1;
			ELSE
				-- RAISE NOTICE '%', 'UPDATE ' || c.table_name || ' SET ' || c.column_name || '=' || 'substring('||c.column_name||',1,16) WHERE length('||c.column_name||')>16';
				EXECUTE 'UPDATE ' || c.table_name || ' SET ' || c.column_name || '=' || 'substring('||c.column_name||',1,16) WHERE length('||c.column_name||')>16';

				-- RAISE NOTICE '%', 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN ' || c.column_name || ' TYPE character(16)';
				EXECUTE 'ALTER TABLE ' || c.table_name || ' ALTER COLUMN ' || c.column_name || ' TYPE character(16)';
				i := i + 1;
			END IF;
		END LOOP;

		IF i > 0 THEN
			r := alkis_string_append(r, i || ' Spalten angepaßt (varchar->character(16)).');
		END IF;

		UPDATE alkis_version SET version=1;
	END IF;

	IF ver<2 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 2';

		-- Indizes ergänzen
		CREATE UNIQUE INDEX ax_sicherungspunkt_gmlid ON ax_sicherungspunkt USING btree (gml_id,beginnt);
		CREATE INDEX ax_sicherungspunkt_bsa ON ax_sicherungspunkt USING btree (beziehtsichauf);
		CREATE INDEX ax_sicherungspunkt_ghz ON ax_sicherungspunkt USING btree (gehoertzu);

		-- drop identifier
		i := 0;
		FOR c IN
			SELECT table_name
			FROM information_schema.columns a
			WHERE a.table_schema=current_schema()
			  AND (a.table_name LIKE 'ax_%' OR a.table_name LIKE 'ap_%')
			  AND a.column_name='identifier'
		LOOP
			-- RAISE NOTICE '%', 'ALTER TABLE ' || c.table_name || ' DROP COLUMN identifier';
			EXECUTE 'ALTER TABLE ' || c.table_name || ' DROP COLUMN identifier';
			i := i + 1;
		END LOOP;

		IF i > 0 THEN
			r := alkis_string_append(r, i || ' identifier-Spalten gelöscht.');
		END IF;

		UPDATE alkis_version SET version=2;
	END IF;

	IF ver<3 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 3';

		ALTER TABLE ax_fortfuehrungsfall ALTER zeigtaufaltesflurstueck TYPE character(20)[];
		ALTER TABLE ax_fortfuehrungsfall ALTER zeigtaufneuesflurstueck TYPE character(20)[];

		UPDATE alkis_version SET version=3;
	END IF;

	IF ver<4 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 4';

		BEGIN
			ALTER TABLE ax_lagebezeichnungmithausnummer ADD unverschluesselt varchar;
			ALTER TABLE ax_lagebezeichnungmitpseudonummer ADD unverschluesselt varchar;
		EXCEPTION WHEN OTHERS THEN
			--
		END;

		UPDATE alkis_version SET version=4;
	END IF;

	IF ver<5 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 5';

		DROP INDEX delete_fid;
		CREATE INDEX delete_fid ON "delete"(featureid);

		UPDATE alkis_version SET version=5;
	END IF;

	IF ver<6 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 6';

		CREATE INDEX ap_ppo_art ON ap_ppo USING btree (art);
		CREATE INDEX ap_lpo_art ON ap_lpo USING btree (art);

		UPDATE alkis_version SET version=6;
	END IF;

	IF ver<7 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 7';

		ALTER TABLE ax_gebaeude ADD gebaeudekennzeichen varchar;
		ALTER TABLE ax_gebaeude RENAME baujahr TO baujahr_;
		ALTER TABLE ax_gebaeude ADD baujahr integer[];
		UPDATE ax_gebaeude SET baujahr=ARRAY[baujahr_];
		ALTER TABLE ax_gebaeude DROP baujahr_;

		UPDATE alkis_version SET version=7;
	END IF;

	IF ver<8 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 8';

		BEGIN
			ALTER TABLE ax_tagesabschnitt DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_tagesabschnitt RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_tagesabschnitt','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_tagesabschnitt SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_tagesabschnitt DROP wkb_geometry_ CASCADE;

			CREATE INDEX ax_tagesabschnitt_geom_idx ON ax_tagesabschnitt USING gist(wkb_geometry);
		END;

		UPDATE alkis_version SET version=8;
	END IF;

	IF ver<9 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 9';

		BEGIN
			ALTER TABLE ax_topographischelinie DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_topographischelinie RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_topographischelinie','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_topographischelinie SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_topographischelinie DROP wkb_geometry_ CASCADE;

			CREATE INDEX ax_topographischelinie_geom_idx ON ax_topographischelinie USING gist(wkb_geometry);
		END;

		UPDATE alkis_version SET version=9;
	END IF;

	IF ver<10 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 10';

		i := 0;
		FOR c IN
			SELECT table_name
			FROM information_schema.columns a
			WHERE a.table_schema=current_schema()
			  AND (a.table_name LIKE 'ax_%' OR a.table_name LIKE 'ap_%' OR a.table_name LIKE 'ks_%')
			  AND a.column_name='anlass'
			  AND a.data_type='character varying'
		LOOP
			EXECUTE 'ALTER TABLE ' || c.table_name || ' RENAME anlass TO anlass_';
			EXECUTE 'ALTER TABLE ' || c.table_name || ' ADD anlass varchar[]';
			EXECUTE 'UPDATE ' || c.table_name || ' SET anlass=ARRAY[anlass_]';
			EXECUTE 'ALTER TABLE ' || c.table_name || ' DROP anlass_';
			i := i + 1;
		END LOOP;

		IF i > 0 THEN
			r := alkis_string_append(r, i || ' anlass-Spalten angepaßt (character varying->character varying[])');
		END IF;

		UPDATE alkis_version SET version=10;
	END IF;

	IF ver<11 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 11';

		EXECUTE 'ALTER TABLE "delete" RENAME anlass TO anlass_';
		EXECUTE 'ALTER TABLE "delete" ADD anlass varchar[]';
		EXECUTE 'UPDATE "delete" SET anlass=ARRAY[anlass_]';
		EXECUTE 'ALTER TABLE "delete" DROP anlass_';

		IF i > 0 THEN
			r := alkis_string_append(r, 'Spalte delete.anlass angepaßt (character varying->character varying[])');
		END IF;

		UPDATE alkis_version SET version=11;
	END IF;

	IF ver<12 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 12';

		BEGIN
			ALTER TABLE ks_einrichtunginoeffentlichenbereichen ADD oberflaechenmaterial integer;
			ALTER TABLE ks_einrichtunginoeffentlichenbereichen ADD material integer[];
			ALTER TABLE ks_einrichtunginoeffentlichenbereichen ADD bezeichnung varchar;
			ALTER TABLE ks_einrichtunginoeffentlichenbereichen ADD zustand integer;

		EXCEPTION WHEN OTHERS THEN
			CREATE TABLE ks_einrichtunginoeffentlichenbereichen (
				ogc_fid                 serial NOT NULL,
				gml_id                  character(16) NOT NULL,
				beginnt                 character(20),
				endet                   character(20),
				advstandardmodell       varchar[],
				sonstigesmodell         varchar[],
				anlass                  varchar[],
				art                     varchar,
				oberflaechenmaterial    integer,
				material                integer[],
				bezeichnung             varchar,
				zustand                 integer,
				PRIMARY KEY (ogc_fid)
			);

			PERFORM AddGeometryColumn('ks_einrichtunginoeffentlichenbereichen','wkb_geometry',find_srid('','ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
		END;

		CREATE INDEX ks_einrichtunginoeffentlichenbereichen_geom_idx ON ks_einrichtunginoeffentlichenbereichen USING gist (wkb_geometry);

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_bauwerkanlagenfuerverundentsorgung'));

		CREATE TABLE ks_bauwerkanlagenfuerverundentsorgung (
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			art                     integer,
			bezeichnung             varchar,
			zustand                 integer,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_bauwerkanlagenfuerverundentsorgung','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'POINT',2);

		CREATE INDEX ks_bauwerkanlagenfuerverundentsorgung_geom_idx ON ks_bauwerkanlagenfuerverundentsorgung USING gist (wkb_geometry);

		ALTER TABLE ks_sonstigesbauwerk ADD advstandardmodell varchar[];
		ALTER TABLE ks_sonstigesbauwerk ADD bezeichnung varchar;

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_einrichtungimstrassenverkehr'));

		CREATE TABLE ks_einrichtungimstrassenverkehr(
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			art                     integer,
			oberflaechenmaterial    integer,
			bezeichnung             varchar,
			zustand                 integer,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_einrichtungimstrassenverkehr','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);

		CREATE INDEX ks_einrichtungimstrassenverkehr_geom_idx ON ks_einrichtungimstrassenverkehr USING gist (wkb_geometry);

		BEGIN
			ALTER TABLE ks_verkehrszeichen ADD gefahrzeichen integer[];
			ALTER TABLE ks_verkehrszeichen ADD vorschriftzeichen integer[];
			ALTER TABLE ks_verkehrszeichen ADD richtzeichen integer[];

			ALTER TABLE ks_verkehrszeichen RENAME verkehrseinrichtung TO verkehrseinrichtung_;
			ALTER TABLE ks_verkehrszeichen ADD verkehrseinrichtung integer[];
			UPDATE ks_verkehrszeichen SET verkehrseinrichtung=ARRAY[verkehrseinrichtung_];
			ALTER TABLE ks_verkehrszeichen DROP verkehrseinrichtung_;

			ALTER TABLE ks_verkehrszeichen ADD zusatzzeichen integer[];
			ALTER TABLE ks_verkehrszeichen ADD bezeichnung varchar;
		EXCEPTION WHEN OTHERS THEN
			CREATE TABLE ks_verkehrszeichen (
				ogc_fid                 serial NOT NULL,
				gml_id                  character(16) NOT NULL,
				beginnt                 character(20),
				endet                   character(20),
				advstandardmodell       varchar[],
				sonstigesmodell         varchar[],
				anlass                  varchar[],
				gefahrzeichen           integer[],
				vorschriftzeichen       integer[],
				richtzeichen            integer[],
				verkehrseinrichtung     integer[],
				zusatzzeichen           integer[],
				bezeichnung             varchar,
				PRIMARY KEY (ogc_fid)
			);

			PERFORM AddGeometryColumn('ks_verkehrszeichen','wkb_geometry',find_srid('','ax_flurstueck','wkb_geometry'),'POINT',2);

			CREATE INDEX ks_verkehrszeichen_geom_idx ON ks_verkehrszeichen USING gist (wkb_geometry);
		END;

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_einrichtungimbahnverkehr'));

		CREATE TABLE ks_einrichtungimbahnverkehr(
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			art                     integer,
			bezeichnung             varchar,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_einrichtungimbahnverkehr','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);

		CREATE INDEX ks_einrichtungimbahnverkehr_geom_idx ON ks_einrichtungimbahnverkehr USING gist (wkb_geometry);

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_bauwerkimgewaesserbereich'));

		CREATE TABLE ks_bauwerkimgewaesserbereich (
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			bauwerksfunktion        integer,
			bezeichnung             varchar,
			zustand                 integer,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_bauwerkimgewaesserbereich','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'LINESTRING',2);

		CREATE INDEX ks_bauwerkimgewaesserbereich_geom_idx ON ks_bauwerkimgewaesserbereich USING gist (wkb_geometry);

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_vegetationsmerkmal'));

		CREATE TABLE ks_vegetationsmerkmal (
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			bewuchs                 integer,
			zustand                 integer,
			breitedesobjekts        double precision,
			name                    varchar,
			bezeichnung             varchar,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_vegetationsmerkmal','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);

		CREATE INDEX ks_vegetationsmerkmal_geom_idx ON ks_vegetationsmerkmal USING gist (wkb_geometry);

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_bauraumoderbodenordnungsrecht'));

		CREATE TABLE ks_bauraumoderbodenordnungsrecht (
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			artderfestlegung        integer,
			bezeichnung             varchar,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_bauraumoderbodenordnungsrecht','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);

		CREATE INDEX ks_bauraumoderbodenordnungsrecht_geom_idx ON ks_vegetationsmerkmal USING gist (wkb_geometry);

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ks_kommunalerbesitz'));

		CREATE TABLE ks_kommunalerbesitz (
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			zustaendigkeit          varchar,
			nutzung                 varchar,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ks_kommunalerbesitz','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);

		CREATE INDEX ks_kommunalerbesitz_geom_idx ON ks_vegetationsmerkmal USING gist (wkb_geometry);

		UPDATE alkis_version SET version=12;
	END IF;

	IF ver<13 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 13';

		r := alkis_string_append(r, pg_temp.alkis_rename_table('ax_landschaft'));

		CREATE TABLE ax_landschaft(
			ogc_fid                 serial NOT NULL,
			gml_id                  character(16) NOT NULL,
			beginnt                 character(20),
			endet                   character(20),
			advstandardmodell       varchar[],
			sonstigesmodell         varchar[],
			anlass                  varchar[],
			landschaftstyp          integer,
			name                    varchar,
			PRIMARY KEY (ogc_fid)
		);

		PERFORM AddGeometryColumn('ax_landschaft','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2); -- POINT/LINESTRING

		CREATE INDEX ax_landschaft_geom_idx   ON ax_landschaft USING gist (wkb_geometry);
		CREATE UNIQUE INDEX ax_landschaft_gml ON ax_landschaft USING btree (gml_id,beginnt);

		UPDATE alkis_version SET version=13;
	END IF;

	IF ver<14 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 14';

		PERFORM alkis_dropobject('ks_bauraumoderbodenordnungsrecht_geom_idx');
		CREATE INDEX ks_bauraumoderbodenordnungsrecht_geom_idx ON ks_bauraumoderbodenordnungsrecht USING gist (wkb_geometry);

		PERFORM alkis_dropobject('ks_kommunalerbesitz_geom_idx');
		CREATE INDEX ks_kommunalerbesitz_geom_idx ON ks_kommunalerbesitz USING gist (wkb_geometry);

		UPDATE alkis_version SET version=14;
	END IF;

	IF ver<15 THEN
		RAISE NOTICE 'Migriere auf Schema-Version 15';

		-- aa_aktivitaet.art => art: character(16) => character varying
		ALTER TABLE aa_aktivitaet ALTER art TYPE character varying;
		ALTER TABLE aa_aktivitaet ALTER art SET NOT NULL;
		ALTER TABLE aa_aktivitaet ALTER beginnt SET NOT NULL;
		ALTER TABLE aa_aktivitaet ALTER status SET NOT NULL;
		ALTER TABLE aa_aktivitaet ADD anlass character varying[];
		ALTER TABLE aa_aktivitaet ADD erlaeuterung character varying;
		ALTER TABLE aa_aktivitaet ADD istteilvon character(16)[];
		ALTER TABLE aa_aktivitaet ADD zeigtaufexternes_art character varying[];
		ALTER TABLE aa_aktivitaet ADD zeigtaufexternes_name character varying[];
		ALTER TABLE aa_aktivitaet ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2d578cc8_196f_46b3_b8e0_66e5a0338b7b ON aa_aktivitaet USING btree (endet);
		CREATE INDEX alkis_59bcc1e2_30d5_46bf_b306_3873fcc849ba ON aa_aktivitaet USING btree (gml_id, beginnt);
		CREATE INDEX alkis_659bbfd8_37f7_4c4f_8557_75ed73427057 ON aa_aktivitaet USING btree (status);
		CREATE INDEX alkis_9aae1167_d2ce_4a30_a9e2_8627524d2984 ON aa_aktivitaet USING gin (istteilvon);

		ALTER TABLE aa_antrag ALTER antragunterbrochen SET NOT NULL;
		-- aa_antrag.art => art: character(16) => character varying
		ALTER TABLE aa_antrag ALTER art TYPE character varying;
		ALTER TABLE aa_antrag ALTER art SET NOT NULL;
		ALTER TABLE aa_antrag ALTER beginnt SET NOT NULL;
		ALTER TABLE aa_antrag ALTER kennzeichen SET NOT NULL;
		ALTER TABLE aa_antrag ALTER verweistauf SET NOT NULL;
		ALTER TABLE aa_antrag ADD anlass character varying[];
		ALTER TABLE aa_antrag ADD erlaeuterungzumstatus character varying;
		ALTER TABLE aa_antrag ADD istteilvon character(16)[];
		ALTER TABLE aa_antrag ADD zeigtaufexternes_art character varying[];
		ALTER TABLE aa_antrag ADD zeigtaufexternes_name character varying[];
		ALTER TABLE aa_antrag ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_b193c3f2_4bfc_4203_ae8f_09f9f97fe0e9 ON aa_antrag USING btree (bearbeitungsstatus);
		CREATE INDEX alkis_7ccd6052_c941_47f6_9b02_8507c6354a94 ON aa_antrag USING btree (endet);
		CREATE INDEX alkis_fc76ab2e_0294_4666_84f1_c6232056c9bf ON aa_antrag USING btree (gebiet);
		CREATE INDEX alkis_4ab44846_c317_4231_8b43_eb40e95e4f0a ON aa_antrag USING btree (gml_id, beginnt);
		CREATE INDEX alkis_c36ccbf5_a1e9_45bc_a6c1_90d2618a04f7 ON aa_antrag USING gin (istteilvon);
		CREATE INDEX alkis_3386c374_f521_4e84_9f86_618a6c85dc67 ON aa_antrag USING gin (verweistauf);

		ALTER TABLE aa_antragsgebiet ALTER beginnt SET NOT NULL;
		-- aa_antragsgebiet.wkb_geometry => wkb_geometry: geometry(Polygon,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE aa_antragsgebiet DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE aa_antragsgebiet RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('aa_antragsgebiet','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE aa_antragsgebiet SET wkb_geometry=wkb_geometry_;
			ALTER TABLE aa_antragsgebiet DROP wkb_geometry_ CASCADE;
			CREATE INDEX aa_antragsgebiet_wkb_geometry_idx ON aa_antragsgebiet USING gist(wkb_geometry);
		END;
		ALTER TABLE aa_antragsgebiet ADD anlass character varying[];
		ALTER TABLE aa_antragsgebiet ADD hatdirektunten character(16)[];
		ALTER TABLE aa_antragsgebiet ADD istabgeleitetaus character(16)[];
		ALTER TABLE aa_antragsgebiet ADD istteilvon character(16)[];
		ALTER TABLE aa_antragsgebiet ADD traegtbeizu character(16)[];
		ALTER TABLE aa_antragsgebiet ADD zeigtaufexternes_art character varying[];
		ALTER TABLE aa_antragsgebiet ADD zeigtaufexternes_name character varying[];
		ALTER TABLE aa_antragsgebiet ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4ff31191_d6e7_4f5a_aa51_39ebb7fd4830 ON aa_antragsgebiet USING btree (endet);
		CREATE INDEX alkis_0d2da273_581b_45bf_8cc4_d7d3f928837d ON aa_antragsgebiet USING btree (gml_id, beginnt);
		CREATE INDEX alkis_6d486f97_4c37_4a43_a473_4f4128d854a1 ON aa_antragsgebiet USING gin (hatdirektunten);
		CREATE INDEX alkis_dbe486cf_4661_4e0c_b89a_43405c71e854 ON aa_antragsgebiet USING gin (istabgeleitetaus);
		CREATE INDEX alkis_9819200b_8169_4143_bec4_181890f8edd3 ON aa_antragsgebiet USING gin (istteilvon);
		CREATE INDEX alkis_766469cc_e0df_4b38_a864_0fffc20c3ea7 ON aa_antragsgebiet USING gin (traegtbeizu);

		ALTER TABLE aa_meilenstein ALTER beginnt SET NOT NULL;
		-- aa_meilenstein.vonantrag => vonantrag: character(16) => character(16)[]
		ALTER TABLE aa_meilenstein ALTER vonantrag TYPE character(16)[] USING CASE WHEN vonantrag IS NULL THEN NULL ELSE ARRAY[vonantrag] END;
		-- aa_meilenstein.vonvorgang => vonvorgang: character(16) => character(16)[]
		ALTER TABLE aa_meilenstein ALTER vonvorgang TYPE character(16)[] USING CASE WHEN vonvorgang IS NULL THEN NULL ELSE ARRAY[vonvorgang] END;
		ALTER TABLE aa_meilenstein ADD anlass character varying[];
		ALTER TABLE aa_meilenstein ADD istteilvon character(16)[];
		ALTER TABLE aa_meilenstein ADD kategorie character varying;
		ALTER TABLE aa_meilenstein ADD vonaktivitaet character(16)[];
		ALTER TABLE aa_meilenstein ADD wer character(16);
		ALTER TABLE aa_meilenstein ADD zeigtaufexternes_art character varying[];
		ALTER TABLE aa_meilenstein ADD zeigtaufexternes_name character varying[];
		ALTER TABLE aa_meilenstein ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_6fd3f59b_77d0_4ec2_81ac_3c9eb2915093 ON aa_meilenstein USING btree (endet);
		CREATE INDEX alkis_7f41ad24_0da4_4ab3_b328_e7d8a45dae8d ON aa_meilenstein USING btree (gml_id, beginnt);
		CREATE INDEX alkis_1b8a0d0b_761e_494d_981d_c04cec8bd849 ON aa_meilenstein USING btree (wer);
		CREATE INDEX alkis_f5010937_d163_4090_90e2_f2a066075fe2 ON aa_meilenstein USING gin (istteilvon);
		CREATE INDEX alkis_a408f318_9e14_4f93_9614_98a58de63a9b ON aa_meilenstein USING gin (vonaktivitaet);
		CREATE INDEX alkis_9bb84517_f345_45c2_8dcc_73e54306f54b ON aa_meilenstein USING gin (vonantrag);
		CREATE INDEX alkis_0c12fdc1_461c_4984_ac8c_6510440e7235 ON aa_meilenstein USING gin (vonvorgang);

		-- aa_projektsteuerung.anlassdesprozesses => anlassdesprozesses: character(6) => character varying[]
		ALTER TABLE aa_projektsteuerung ALTER anlassdesprozesses TYPE character varying[] USING CASE WHEN anlassdesprozesses IS NULL THEN NULL ELSE ARRAY[anlassdesprozesses] END;
		ALTER TABLE aa_projektsteuerung ALTER anlassdesprozesses SET NOT NULL;
		-- aa_projektsteuerung.art => art: character(16) => character varying
		ALTER TABLE aa_projektsteuerung ALTER art TYPE character varying;
		ALTER TABLE aa_projektsteuerung ALTER art SET NOT NULL;
		ALTER TABLE aa_projektsteuerung ALTER beginnt SET NOT NULL;
		ALTER TABLE aa_projektsteuerung ALTER enthaelt SET NOT NULL;
		ALTER TABLE aa_projektsteuerung ADD anlass character varying[];
		ALTER TABLE aa_projektsteuerung ADD istteilvon character(16)[];
		ALTER TABLE aa_projektsteuerung ADD parameterwert character varying;
		ALTER TABLE aa_projektsteuerung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE aa_projektsteuerung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE aa_projektsteuerung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_ca5e8d86_d7ad_4582_b2dd_46b771d0bab1 ON aa_projektsteuerung USING btree (endet);
		CREATE INDEX alkis_af16f507_98af_498e_b199_ba833ab8c6f1 ON aa_projektsteuerung USING btree (gml_id, beginnt);
		CREATE INDEX alkis_39a525e1_a473_48bd_aa1a_bce73e05cf77 ON aa_projektsteuerung USING gin (enthaelt);
		CREATE INDEX alkis_e9c97980_1bfc_4966_b97f_83ffcbdb357c ON aa_projektsteuerung USING gin (istteilvon);

		-- aa_vorgang.art => art: character(16) => character varying
		ALTER TABLE aa_vorgang ALTER art TYPE character varying;
		ALTER TABLE aa_vorgang ALTER art SET NOT NULL;
		ALTER TABLE aa_vorgang ALTER bearbeitbardurch SET NOT NULL;
		ALTER TABLE aa_vorgang ALTER beginnt SET NOT NULL;
		ALTER TABLE aa_vorgang ALTER status SET NOT NULL;
		ALTER TABLE aa_vorgang ADD anlass character varying[];
		ALTER TABLE aa_vorgang ADD erlaeuterung character varying;
		ALTER TABLE aa_vorgang ADD istteilvon character(16)[];
		ALTER TABLE aa_vorgang ADD synchronisiertmit character(16);
		ALTER TABLE aa_vorgang ADD zeigtaufexternes_art character varying[];
		ALTER TABLE aa_vorgang ADD zeigtaufexternes_name character varying[];
		ALTER TABLE aa_vorgang ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_506dd4d2_1a45_43a3_9b51_56d970718496 ON aa_vorgang USING btree (bearbeitbardurch);
		CREATE INDEX alkis_cac97355_f732_4f50_854e_b62c3243a9a6 ON aa_vorgang USING btree (endet);
		CREATE INDEX alkis_4f7a1db3_1399_4edc_ab6c_6c029a70eb74 ON aa_vorgang USING btree (gml_id, beginnt);
		CREATE INDEX alkis_7f2120ab_0a56_46b6_af44_1db9bffd41c9 ON aa_vorgang USING btree (status);
		CREATE INDEX alkis_0b3f7ddf_31c3_4db5_8384_86afb585ddaf ON aa_vorgang USING btree (synchronisiertmit);
		CREATE INDEX alkis_90fda40c_070f_4030_bd2b_c6e16510374d ON aa_vorgang USING gin (enthaelt);
		CREATE INDEX alkis_326ad559_9bfb_42d3_adc2_176841b83ddd ON aa_vorgang USING gin (istteilvon);

		ALTER TABLE ap_darstellung ALTER beginnt SET NOT NULL;
		-- ap_darstellung.positionierungsregel => positionierungsregel: integer => character varying
		ALTER TABLE ap_darstellung ALTER positionierungsregel TYPE character varying USING positionierungsregel::varchar;
		ALTER TABLE ap_darstellung ADD istteilvon character(16)[];
		ALTER TABLE ap_darstellung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ap_darstellung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ap_darstellung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_bd33d1d7_ed33_4110_8e9a_8253721fa8ee ON ap_darstellung USING gin (istteilvon);

		ALTER TABLE ap_lpo ALTER beginnt SET NOT NULL;
		ALTER TABLE ap_lpo ADD hatdirektunten character(16)[];
		ALTER TABLE ap_lpo ADD istabgeleitetaus character(16)[];
		ALTER TABLE ap_lpo ADD istteilvon character(16)[];
		ALTER TABLE ap_lpo ADD traegtbeizu character(16)[];
		ALTER TABLE ap_lpo ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ap_lpo ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ap_lpo ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_dcad2a31_02c9_4c5d_bc70_4e039f5a7bc4 ON ap_lpo USING gin (hatdirektunten);
		CREATE INDEX alkis_4ad5172a_7092_4268_9a56_5218bf3a5c6a ON ap_lpo USING gin (istabgeleitetaus);
		CREATE INDEX alkis_2dd26091_115c_4206_8a2d_98f4b43dfc87 ON ap_lpo USING gin (istteilvon);
		CREATE INDEX alkis_58075be7_f312_44ed_8974_e1a83d8f58d2 ON ap_lpo USING gin (traegtbeizu);

		ALTER TABLE ap_lto ALTER beginnt SET NOT NULL;
		ALTER TABLE ap_lto ALTER fontsperrung SET NOT NULL;
		ALTER TABLE ap_lto ALTER horizontaleausrichtung SET NOT NULL;
		ALTER TABLE ap_lto ALTER skalierung SET NOT NULL;
		ALTER TABLE ap_lto ALTER vertikaleausrichtung SET NOT NULL;
		-- ap_lto.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ap_lto DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ap_lto RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ap_lto','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ap_lto SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ap_lto DROP wkb_geometry_ CASCADE;
			CREATE INDEX ap_lto_wkb_geometry_idx ON ap_lto USING gist(wkb_geometry);
		END;
		ALTER TABLE ap_lto ADD hatdirektunten character(16)[];
		ALTER TABLE ap_lto ADD istabgeleitetaus character(16)[];
		ALTER TABLE ap_lto ADD istteilvon character(16)[];
		ALTER TABLE ap_lto ADD traegtbeizu character(16)[];
		ALTER TABLE ap_lto ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ap_lto ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ap_lto ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4a8057aa_55ab_4714_9b55_a9e0f50738dc ON ap_lto USING gin (hatdirektunten);
		CREATE INDEX alkis_fba62a13_2e92_4074_8a59_eb89f9d0bcbb ON ap_lto USING gin (istabgeleitetaus);
		CREATE INDEX alkis_489e017f_7df8_4e1e_b679_f9f68c640fdd ON ap_lto USING gin (istteilvon);
		CREATE INDEX alkis_3917c24d_0821_498f_b9b0_7e5131e2609e ON ap_lto USING gin (traegtbeizu);

		ALTER TABLE ap_ppo ALTER beginnt SET NOT NULL;
		ALTER TABLE ap_ppo ADD hatdirektunten character(16)[];
		ALTER TABLE ap_ppo ADD istabgeleitetaus character(16)[];
		ALTER TABLE ap_ppo ADD istteilvon character(16)[];
		ALTER TABLE ap_ppo ADD traegtbeizu character(16)[];
		ALTER TABLE ap_ppo ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ap_ppo ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ap_ppo ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_3c2d61c2_ba0b_4224_8d07_2c26b8eaecb7 ON ap_ppo USING gin (hatdirektunten);
		CREATE INDEX alkis_3e18a97c_11c8_4f38_a916_8958563b7112 ON ap_ppo USING gin (istabgeleitetaus);
		CREATE INDEX alkis_7317cf3d_9834_45e7_a78d_06665a615de2 ON ap_ppo USING gin (istteilvon);
		CREATE INDEX alkis_0602817f_edd4_46e4_9b75_2629069bcbda ON ap_ppo USING gin (traegtbeizu);

		ALTER TABLE ap_pto ALTER beginnt SET NOT NULL;
		ALTER TABLE ap_pto ALTER fontsperrung SET NOT NULL;
		ALTER TABLE ap_pto ALTER horizontaleausrichtung SET NOT NULL;
		ALTER TABLE ap_pto ALTER skalierung SET NOT NULL;
		ALTER TABLE ap_pto ALTER vertikaleausrichtung SET NOT NULL;
		-- ap_pto.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ap_pto DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ap_pto RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ap_pto','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ap_pto SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ap_pto DROP wkb_geometry_ CASCADE;
			CREATE INDEX ap_pto_wkb_geometry_idx ON ap_pto USING gist(wkb_geometry);
		END;
		ALTER TABLE ap_pto ADD hatdirektunten character(16)[];
		ALTER TABLE ap_pto ADD istabgeleitetaus character(16)[];
		ALTER TABLE ap_pto ADD istteilvon character(16)[];
		ALTER TABLE ap_pto ADD traegtbeizu character(16)[];
		ALTER TABLE ap_pto ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ap_pto ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ap_pto ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_db273ead_24b0_4aa7_a7e8_57c9f52b151e ON ap_pto USING gin (hatdirektunten);
		CREATE INDEX alkis_2d1af67d_18fa_462f_a42f_d11c6dbaa82b ON ap_pto USING gin (istabgeleitetaus);
		CREATE INDEX alkis_4811f6b4_058e_478e_b1fa_bf8344c56de7 ON ap_pto USING gin (istteilvon);
		CREATE INDEX alkis_e42aa703_06d8_4ec2_adc1_2e30682cbe61 ON ap_pto USING gin (traegtbeizu);

		ALTER TABLE ax_anderefestlegungnachwasserrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_anderefestlegungnachwasserrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD nummer character varying;
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_address character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_role character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD statement character varying;
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_anderefestlegungnachwasserrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_3e13c5de_73bd_4485_8de4_c7cdf4b0b57f ON ax_anderefestlegungnachwasserrecht USING btree (endet);
		CREATE INDEX alkis_25da7d4d_f1e2_46f6_8881_ea2aed035f1e ON ax_anderefestlegungnachwasserrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_115541ac_5b9c_4cb5_b5d8_8923f85cde9d ON ax_anderefestlegungnachwasserrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_56f48b65_1491_48ea_8c92_d409b88d26e2 ON ax_anderefestlegungnachwasserrecht USING gin (istteilvon);
		CREATE INDEX alkis_1be8fb52_e0fc_4e68_9bad_f5f940ec882d ON ax_anderefestlegungnachwasserrecht USING gin (traegtbeizu);

		ALTER TABLE ax_anschrift ALTER beginnt SET NOT NULL;
		-- ax_anschrift.fax => fax: character varying => character varying[]
		ALTER TABLE ax_anschrift ALTER fax TYPE character varying[] USING CASE WHEN fax IS NULL THEN NULL ELSE ARRAY[fax] END;
		ALTER TABLE ax_anschrift RENAME organisationname TO processstep_organisationname;
		-- ax_anschrift.organisationname => processstep_organisationname: character varying => character varying[]
		ALTER TABLE ax_anschrift ALTER processstep_organisationname TYPE character varying[] USING CASE WHEN processstep_organisationname IS NULL THEN NULL ELSE ARRAY[processstep_organisationname] END;
		ALTER TABLE ax_anschrift ALTER ort_post SET NOT NULL;
		-- ax_anschrift.telefon => telefon: character varying => character varying[]
		ALTER TABLE ax_anschrift ALTER telefon TYPE character varying[] USING CASE WHEN telefon IS NULL THEN NULL ELSE ARRAY[telefon] END;
		ALTER TABLE ax_anschrift ADD ax_li_processstep_ohnedatenerhebung_description character varying[];
		ALTER TABLE ax_anschrift ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_anschrift ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_anschrift ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_anschrift ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_anschrift ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_anschrift ADD istteilvon character(16)[];
		ALTER TABLE ax_anschrift ADD ort_amtlichesortsnamensverzeichnis character varying;
		ALTER TABLE ax_anschrift ADD processstep_address character varying[];
		ALTER TABLE ax_anschrift ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_anschrift ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_anschrift ADD processstep_datetime character(20)[];
		ALTER TABLE ax_anschrift ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_anschrift ADD processstep_individualname character varying[];
		ALTER TABLE ax_anschrift ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_anschrift ADD processstep_phone character varying[];
		ALTER TABLE ax_anschrift ADD processstep_positionname character varying[];
		ALTER TABLE ax_anschrift ADD processstep_rationale character varying[];
		ALTER TABLE ax_anschrift ADD processstep_role character varying[];
		ALTER TABLE ax_anschrift ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_anschrift ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_anschrift ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_anschrift ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_anschrift ADD statement character varying;
		ALTER TABLE ax_anschrift ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_anschrift ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_anschrift ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_7b2b63da_0a74_49a7_9ba5_8bca43f0bcbc ON ax_anschrift USING btree (endet);
		CREATE INDEX alkis_5f545065_d5fb_4a3c_8b1a_ec9db7018664 ON ax_anschrift USING gin (istteilvon);

		ALTER TABLE ax_aufnahmepunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_aufnahmepunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_aufnahmepunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_aufnahmepunkt ALTER vermarkung_marke SET NOT NULL;
		ALTER TABLE ax_aufnahmepunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_aufnahmepunkt ADD beziehtsichauf character(16)[];
		ALTER TABLE ax_aufnahmepunkt ADD gehoertzu character(16)[];
		ALTER TABLE ax_aufnahmepunkt ADD haengtan character(16)[];
		ALTER TABLE ax_aufnahmepunkt ADD hatidentitaet character(16)[];
		ALTER TABLE ax_aufnahmepunkt ADD horizontfreiheit integer;
		ALTER TABLE ax_aufnahmepunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_aufnahmepunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_aufnahmepunkt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_aufnahmepunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_45827218_8025_4d64_862a_2cd3e320ee01 ON ax_aufnahmepunkt USING btree (endet);
		CREATE INDEX alkis_c469e649_e99a_460f_afa8_2288d5017b0c ON ax_aufnahmepunkt USING gin (bestehtaus);
		CREATE INDEX alkis_11127394_be07_44b5_816f_767a3e100ee3 ON ax_aufnahmepunkt USING gin (beziehtsichauf);
		CREATE INDEX alkis_78ee36ff_d872_42c9_a41d_305966f3112c ON ax_aufnahmepunkt USING gin (gehoertzu);
		CREATE INDEX alkis_8074699e_56c3_4ad1_85b0_e72a61b6da9a ON ax_aufnahmepunkt USING gin (haengtan);
		CREATE INDEX alkis_acabc7da_f6bf_4b93_8466_f4c35a398cce ON ax_aufnahmepunkt USING gin (hatidentitaet);
		CREATE INDEX alkis_50c03631_a9e3_4ac3_90ef_4f1c4eb7fa37 ON ax_aufnahmepunkt USING gin (istteilvon);

		-- ax_bahnverkehr.bahnkategorie => bahnkategorie: integer => integer[]
		ALTER TABLE ax_bahnverkehr ALTER bahnkategorie TYPE integer[] USING CASE WHEN bahnkategorie IS NULL THEN NULL ELSE ARRAY[bahnkategorie] END;
		ALTER TABLE ax_bahnverkehr ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bahnverkehr DROP bezeichnung;
		ALTER TABLE ax_bahnverkehr ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_bahnverkehr ADD gemeinde character varying;
		ALTER TABLE ax_bahnverkehr ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bahnverkehr ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bahnverkehr ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bahnverkehr ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bahnverkehr ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bahnverkehr ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bahnverkehr ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bahnverkehr ADD istteilvon character(16)[];
		ALTER TABLE ax_bahnverkehr ADD kreis character varying;
		ALTER TABLE ax_bahnverkehr ADD lage character varying;
		ALTER TABLE ax_bahnverkehr ADD land character varying;
		ALTER TABLE ax_bahnverkehr ADD processstep_address character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bahnverkehr ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_individualname character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_phone character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_positionname character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_rationale character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_role character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bahnverkehr ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bahnverkehr ADD regierungsbezirk character varying;
		ALTER TABLE ax_bahnverkehr ADD statement character varying;
		ALTER TABLE ax_bahnverkehr ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bahnverkehr ADD unverschluesselt character varying;
		ALTER TABLE ax_bahnverkehr ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bahnverkehr ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bahnverkehr ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_185545e9_1d5d_41ae_978d_7905e2df5e04 ON ax_bahnverkehr USING btree (endet);
		CREATE INDEX alkis_0165e719_59cf_4997_be39_42deff6ebe7c ON ax_bahnverkehr USING gin (hatdirektunten);
		CREATE INDEX alkis_56f98b09_eff3_4018_a29e_1fcded48cd05 ON ax_bahnverkehr USING gin (istabgeleitetaus);
		CREATE INDEX alkis_12e90b67_954b_4b32_a2b1_3ecd3a6843cd ON ax_bahnverkehr USING gin (istteilvon);
		CREATE INDEX alkis_aa4a5e33_7b7a_45e3_a78d_37d76e3d5b98 ON ax_bahnverkehr USING gin (traegtbeizu);

		ALTER TABLE ax_bahnverkehrsanlage ALTER bahnhofskategorie SET NOT NULL;
		-- ax_bahnverkehrsanlage.bahnkategorie => bahnkategorie: integer => integer[]
		ALTER TABLE ax_bahnverkehrsanlage ALTER bahnkategorie TYPE integer[] USING CASE WHEN bahnkategorie IS NULL THEN NULL ELSE ARRAY[bahnkategorie] END;
		ALTER TABLE ax_bahnverkehrsanlage ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bahnverkehrsanlage ADD bezeichnung character varying;
		ALTER TABLE ax_bahnverkehrsanlage ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bahnverkehrsanlage ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bahnverkehrsanlage ADD istteilvon character(16)[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_address character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_individualname character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_phone character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_positionname character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_rationale character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_role character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD statement character varying;
		ALTER TABLE ax_bahnverkehrsanlage ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bahnverkehrsanlage ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_bahnverkehrsanlage ADD zustand integer;
		CREATE INDEX alkis_9c23cb0d_7ad6_4f73_a1ed_87b77cd3e6b6 ON ax_bahnverkehrsanlage USING btree (endet);
		CREATE INDEX alkis_66ea059d_7fab_4f3b_8459_639cf3ce35a0 ON ax_bahnverkehrsanlage USING gin (hatdirektunten);
		CREATE INDEX alkis_51cc82d4_7608_4056_bea2_ad83369eb777 ON ax_bahnverkehrsanlage USING gin (istabgeleitetaus);
		CREATE INDEX alkis_1a0b1518_1a53_4232_acf6_c18aa3641fe0 ON ax_bahnverkehrsanlage USING gin (istteilvon);
		CREATE INDEX alkis_b014825d_36f2_474e_b053_a45bfed037ff ON ax_bahnverkehrsanlage USING gin (traegtbeizu);

		ALTER TABLE ax_baublock ALTER baublockbezeichnung SET NOT NULL;
		ALTER TABLE ax_baublock ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_baublock ADD hatdirektunten character(16)[];
		ALTER TABLE ax_baublock ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_baublock ADD istteilvon character(16)[];
		ALTER TABLE ax_baublock ADD traegtbeizu character(16)[];
		ALTER TABLE ax_baublock ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_baublock ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_baublock ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_a3abf190_f565_40ab_838c_ddb9a03ddd53 ON ax_baublock USING btree (endet);
		CREATE INDEX alkis_6190f146_21fe_4596_a2ac_89c97b45f539 ON ax_baublock USING gin (hatdirektunten);
		CREATE INDEX alkis_30f049d4_2b5a_4863_8b1e_1a6e4aa156ef ON ax_baublock USING gin (istabgeleitetaus);
		CREATE INDEX alkis_8cef5106_e093_4370_a1c6_9903b8e8c94e ON ax_baublock USING gin (istteilvon);
		CREATE INDEX alkis_747be46a_e029_484e_9e9f_4f2e94c39917 ON ax_baublock USING gin (traegtbeizu);

		ALTER TABLE ax_bauraumoderbodenordnungsrecht RENAME art TO zeigtaufexternes_art;
		-- ax_bauraumoderbodenordnungsrecht.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ALTER beginnt SET NOT NULL;
		-- ax_bauraumoderbodenordnungsrecht.datumanordnung => datumanordnung: character varying => date
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ALTER datumanordnung TYPE date USING to_date(datumanordnung, 'YYYY-mm-dd');
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD datumabgabe date;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD datumbesitzeinweisung date;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD datumrechtskraeftig date;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_address character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_role character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD statement character varying;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD veraenderungohneruecksprache character varying;
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bauraumoderbodenordnungsrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_3be8dc35_2cc8_4574_a90b_3de1cdc65088 ON ax_bauraumoderbodenordnungsrecht USING btree (endet);
		CREATE INDEX alkis_e764099b_f9c2_4512_ae96_e07191e23d38 ON ax_bauraumoderbodenordnungsrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_e1916c1a_51bb_4069_ba19_38adf0129654 ON ax_bauraumoderbodenordnungsrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_1b09786c_4a4f_4140_a865_b36cbe459c4f ON ax_bauraumoderbodenordnungsrecht USING gin (istteilvon);
		CREATE INDEX alkis_aed62756_da4c_48b1_8c7a_831392fc6121 ON ax_bauraumoderbodenordnungsrecht USING gin (traegtbeizu);

		ALTER TABLE ax_bauteil DROP anzahlderunterirdischengeschosse;
		ALTER TABLE ax_bauteil ALTER bauart SET NOT NULL;
		ALTER TABLE ax_bauteil ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bauteil ADD baujahr integer[];
		ALTER TABLE ax_bauteil ADD durchfahrtshoehe double precision;
		ALTER TABLE ax_bauteil ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bauteil ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bauteil ADD istteilvon character(16)[];
		ALTER TABLE ax_bauteil ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bauteil ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bauteil ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bauteil ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_8527c5ec_6ae1_480c_a1bd_c653a87995df ON ax_bauteil USING btree (endet);
		CREATE INDEX alkis_bd68e2ae_f7e0_474a_8118_af6303abc205 ON ax_bauteil USING gin (hatdirektunten);
		CREATE INDEX alkis_5406fcb3_e999_49bb_b6b7_9efec19f25b5 ON ax_bauteil USING gin (istabgeleitetaus);
		CREATE INDEX alkis_6535e3a5_05c0_487e_a204_ca575278a43a ON ax_bauteil USING gin (istteilvon);
		CREATE INDEX alkis_f88ae71b_1f86_4d15_9075_84085a5a5f03 ON ax_bauteil USING gin (traegtbeizu);

		ALTER TABLE ax_bauwerkimgewaesserbereich ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_bauwerkimgewaesserbereich ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD bezeichnung character varying;
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD istteilvon character(16)[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_address character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_individualname character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_phone character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_positionname character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_rationale character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_role character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD statement character varying;
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bauwerkimgewaesserbereich ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_fbf9b933_085b_41fb_9589_965bff86caee ON ax_bauwerkimgewaesserbereich USING btree (endet);
		CREATE INDEX alkis_7787a6de_f3dc_4952_b3fd_21c126144804 ON ax_bauwerkimgewaesserbereich USING gin (hatdirektunten);
		CREATE INDEX alkis_f84d31dd_1471_427a_8c22_6fc8ad3fbb54 ON ax_bauwerkimgewaesserbereich USING gin (istabgeleitetaus);
		CREATE INDEX alkis_4663cb0a_9791_4734_bb9d_185795098da5 ON ax_bauwerkimgewaesserbereich USING gin (istteilvon);
		CREATE INDEX alkis_8a1acbf1_16a2_4817_823c_273c65dd95f5 ON ax_bauwerkimgewaesserbereich USING gin (traegtbeizu);

		ALTER TABLE ax_bauwerkimverkehrsbereich ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_bauwerkimverkehrsbereich ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD bezeichnung character varying;
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD breitedesobjekts double precision;
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD durchfahrtshoehe double precision;
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD istteilvon character(16)[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_address character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_individualname character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_phone character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_positionname character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_rationale character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_role character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD statement character varying;
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bauwerkimverkehrsbereich ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_44d1f505_ca56_4519_86b3_82fb254664db ON ax_bauwerkimverkehrsbereich USING btree (endet);
		CREATE INDEX alkis_eea0e06c_36eb_4ed4_a18c_9a693618269e ON ax_bauwerkimverkehrsbereich USING gin (hatdirektunten);
		CREATE INDEX alkis_e3383eef_c0ce_4154_9c27_e3f37b9e1ac8 ON ax_bauwerkimverkehrsbereich USING gin (istabgeleitetaus);
		CREATE INDEX alkis_ad1155bb_27fb_430b_8f19_726dfa2bcf57 ON ax_bauwerkimverkehrsbereich USING gin (istteilvon);
		CREATE INDEX alkis_9c873c43_19aa_4e0a_9d90_fd5a673e5492 ON ax_bauwerkimverkehrsbereich USING gin (traegtbeizu);

		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD bezeichnung character varying;
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD istteilvon character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_address character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_individualname character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_phone character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_positionname character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_rationale character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_role character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD statement character varying;
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuerindustrieundgewerbe ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_6c998569_f93a_4f8d_9ea3_610906a73319 ON ax_bauwerkoderanlagefuerindustrieundgewerbe USING btree (endet);
		CREATE INDEX alkis_2b351a9a_d4d7_4a8f_b100_e6ec88c43f87 ON ax_bauwerkoderanlagefuerindustrieundgewerbe USING gin (hatdirektunten);
		CREATE INDEX alkis_715b534a_b938_4af6_baa4_6277b2c72f82 ON ax_bauwerkoderanlagefuerindustrieundgewerbe USING gin (istabgeleitetaus);
		CREATE INDEX alkis_a548da01_17cc_49c3_8580_6824af9fb18f ON ax_bauwerkoderanlagefuerindustrieundgewerbe USING gin (istteilvon);
		CREATE INDEX alkis_fa81bd04_4d46_4e20_9344_0b7073dbfcf6 ON ax_bauwerkoderanlagefuerindustrieundgewerbe USING gin (traegtbeizu);

		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD breitedesobjekts double precision;
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD istteilvon character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_address character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_individualname character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_phone character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_positionname character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_rationale character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_role character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD statement character varying;
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bauwerkoderanlagefuersportfreizeitunderholung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2baa2cdc_b720_4c52_9e9c_c0f94a6d2750 ON ax_bauwerkoderanlagefuersportfreizeitunderholung USING btree (endet);
		CREATE INDEX alkis_4c658a91_6526_4a52_a9e1_61d247b6c5ba ON ax_bauwerkoderanlagefuersportfreizeitunderholung USING gin (hatdirektunten);
		CREATE INDEX alkis_6963db31_c66c_4e9d_a324_9ebd9481dcb2 ON ax_bauwerkoderanlagefuersportfreizeitunderholung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_1751ef17_71de_4c87_8962_68fd310f5557 ON ax_bauwerkoderanlagefuersportfreizeitunderholung USING gin (istteilvon);
		CREATE INDEX alkis_7b9c2b2a_3e8e_4c3e_90d4_f140fffc8642 ON ax_bauwerkoderanlagefuersportfreizeitunderholung USING gin (traegtbeizu);

		ALTER TABLE ax_bergbaubetrieb ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bergbaubetrieb ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_bergbaubetrieb ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bergbaubetrieb ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bergbaubetrieb ADD istteilvon character(16)[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_address character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_individualname character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_phone character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_positionname character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_rationale character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_role character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD statement character varying;
		ALTER TABLE ax_bergbaubetrieb ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bergbaubetrieb ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bergbaubetrieb ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_8ab53362_3a88_4ad1_aa63_9a6308eab663 ON ax_bergbaubetrieb USING btree (endet);
		CREATE INDEX alkis_a05a1aa9_7c1f_4edc_8633_1916bd72c7c4 ON ax_bergbaubetrieb USING gin (hatdirektunten);
		CREATE INDEX alkis_0224cc73_89a9_4c83_b4dc_f586a5b4cffe ON ax_bergbaubetrieb USING gin (istabgeleitetaus);
		CREATE INDEX alkis_4fea7ef6_4f7b_43ab_ab77_60f4ba1feb97 ON ax_bergbaubetrieb USING gin (istteilvon);
		CREATE INDEX alkis_addc85cf_daec_4607_bbb5_17f38bd1288e ON ax_bergbaubetrieb USING gin (traegtbeizu);

		ALTER TABLE ax_besondereflurstuecksgrenze ALTER artderflurstuecksgrenze SET NOT NULL;
		ALTER TABLE ax_besondereflurstuecksgrenze ALTER beginnt SET NOT NULL;
		-- ax_besondereflurstuecksgrenze.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_besondereflurstuecksgrenze DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_besondereflurstuecksgrenze RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_besondereflurstuecksgrenze','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_besondereflurstuecksgrenze SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_besondereflurstuecksgrenze DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_besondereflurstuecksgrenze_wkb_geometry_idx ON ax_besondereflurstuecksgrenze USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_besondereflurstuecksgrenze ADD hatdirektunten character(16)[];
		ALTER TABLE ax_besondereflurstuecksgrenze ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_besondereflurstuecksgrenze ADD istteilvon character(16)[];
		ALTER TABLE ax_besondereflurstuecksgrenze ADD traegtbeizu character(16)[];
		ALTER TABLE ax_besondereflurstuecksgrenze ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_besondereflurstuecksgrenze ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_besondereflurstuecksgrenze ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_9dc43c8f_2c0e_4d30_a830_3650a4c5cf75 ON ax_besondereflurstuecksgrenze USING btree (endet);
		CREATE INDEX alkis_fd2d6ddf_b109_4ad5_9b6d_5ccdeb676d92 ON ax_besondereflurstuecksgrenze USING gin (hatdirektunten);
		CREATE INDEX alkis_beff5c16_3a95_4ae8_b34f_6907e00b9fab ON ax_besondereflurstuecksgrenze USING gin (istabgeleitetaus);
		CREATE INDEX alkis_a73f83bd_2ed2_4985_944c_a1b3f59009eb ON ax_besondereflurstuecksgrenze USING gin (istteilvon);
		CREATE INDEX alkis_5109b1e2_f0a8_4c0b_9620_18906c586cf8 ON ax_besondereflurstuecksgrenze USING gin (traegtbeizu);

		ALTER TABLE ax_besonderegebaeudelinie ALTER beginnt SET NOT NULL;
		UPDATE ax_besonderegebaeudelinie SET beschaffenheit=ARRAY[]::integer[] WHERE beschaffenheit IS NULL;
		ALTER TABLE ax_besonderegebaeudelinie ALTER beschaffenheit SET NOT NULL;
		ALTER TABLE ax_besonderegebaeudelinie ADD hatdirektunten character(16)[];
		ALTER TABLE ax_besonderegebaeudelinie ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_besonderegebaeudelinie ADD istteilvon character(16)[];
		ALTER TABLE ax_besonderegebaeudelinie ADD traegtbeizu character(16)[];
		ALTER TABLE ax_besonderegebaeudelinie ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_besonderegebaeudelinie ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_besonderegebaeudelinie ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e58c842d_76b6_4baa_8d62_6f19ba077d2f ON ax_besonderegebaeudelinie USING btree (endet);
		CREATE INDEX alkis_1e91b566_7f96_47be_a445_489fe0d41455 ON ax_besonderegebaeudelinie USING gin (hatdirektunten);
		CREATE INDEX alkis_a041d288_b0fd_4f28_be8d_b4261a6c56d9 ON ax_besonderegebaeudelinie USING gin (istabgeleitetaus);
		CREATE INDEX alkis_a830c043_30b8_4d53_8168_e2df881368bc ON ax_besonderegebaeudelinie USING gin (istteilvon);
		CREATE INDEX alkis_b71ea13c_fa6c_4dfc_9287_2332d3009525 ON ax_besonderegebaeudelinie USING gin (traegtbeizu);

		ALTER TABLE ax_besondererbauwerkspunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_besondererbauwerkspunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_besondererbauwerkspunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_besondererbauwerkspunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_besondererbauwerkspunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_besondererbauwerkspunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_besondererbauwerkspunkt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_besondererbauwerkspunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_9acb2b7b_a00a_42a0_8ae2_5ceda2f35327 ON ax_besondererbauwerkspunkt USING btree (endet);
		CREATE INDEX alkis_9eea4ad7_2b08_4e4f_a6f4_0c6996dcd2a4 ON ax_besondererbauwerkspunkt USING gin (bestehtaus);
		CREATE INDEX alkis_6c138329_2333_4e97_9563_4ab8f51ca5a5 ON ax_besondererbauwerkspunkt USING gin (istteilvon);

		-- ax_besonderergebaeudepunkt.art => art: character varying => integer
		ALTER TABLE ax_besonderergebaeudepunkt ALTER art TYPE integer USING art::int;
		ALTER TABLE ax_besonderergebaeudepunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_besonderergebaeudepunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_besonderergebaeudepunkt RENAME name TO zeigtaufexternes_name;
		ALTER TABLE ax_besonderergebaeudepunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_besonderergebaeudepunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_besonderergebaeudepunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_besonderergebaeudepunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_besonderergebaeudepunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_83c0a5cf_2ec2_4516_9912_4bc81642db42 ON ax_besonderergebaeudepunkt USING btree (endet);
		CREATE INDEX alkis_471b89e0_9121_4139_aa5c_43ba723abc29 ON ax_besonderergebaeudepunkt USING gin (bestehtaus);
		CREATE INDEX alkis_6e49cc8a_1fb1_4a9e_8365_edab82e2a4c1 ON ax_besonderergebaeudepunkt USING gin (istteilvon);

		ALTER TABLE ax_besondererhoehenpunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_besondererhoehenpunkt ALTER besonderebedeutung SET NOT NULL;
		-- ax_besondererhoehenpunkt.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_besondererhoehenpunkt DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_besondererhoehenpunkt RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_besondererhoehenpunkt','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_besondererhoehenpunkt SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_besondererhoehenpunkt DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_besondererhoehenpunkt_wkb_geometry_idx ON ax_besondererhoehenpunkt USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_besondererhoehenpunkt ADD ax_dqerfassungsmethodebesondererhoehenpunkt integer;
		ALTER TABLE ax_besondererhoehenpunkt ADD erfassungbesondererhoehenpunkt_datetime character(20);
		ALTER TABLE ax_besondererhoehenpunkt ADD hatdirektunten character(16)[];
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_datetime character(20)[];
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_evaluationmethoddescription character varying;
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_evaluationmethodtype character varying;
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_evaluationprocedure character varying;
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_measuredescription character varying;
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_measureidentification character varying;
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_nameofmeasure character varying[];
		ALTER TABLE ax_besondererhoehenpunkt ADD hoehengenauigkeit_result character varying[];
		ALTER TABLE ax_besondererhoehenpunkt ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_besondererhoehenpunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_besondererhoehenpunkt ADD traegtbeizu character(16)[];
		ALTER TABLE ax_besondererhoehenpunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_besondererhoehenpunkt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_besondererhoehenpunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_10ea255e_ec36_49c3_b306_a8bf89c3fc3d ON ax_besondererhoehenpunkt USING btree (endet);
		CREATE INDEX alkis_c549c779_3e80_415c_a828_c1b80679ef72 ON ax_besondererhoehenpunkt USING gin (hatdirektunten);
		CREATE INDEX alkis_b5c305d1_c4b7_4ced_96f5_47138a2aa46e ON ax_besondererhoehenpunkt USING gin (istabgeleitetaus);
		CREATE INDEX alkis_55109b6d_a350_4ce0_baa0_57b3aac06d88 ON ax_besondererhoehenpunkt USING gin (istteilvon);
		CREATE INDEX alkis_93d66305_1791_482a_9c9d_25c7955f30c6 ON ax_besondererhoehenpunkt USING gin (traegtbeizu);

		ALTER TABLE ax_besonderertopographischerpunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_besonderertopographischerpunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_besonderertopographischerpunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_besonderertopographischerpunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_besonderertopographischerpunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_besonderertopographischerpunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_besonderertopographischerpunkt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_besonderertopographischerpunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_67cb24ef_f675_4abf_ac33_7ae4dc75ad57 ON ax_besonderertopographischerpunkt USING btree (endet);
		CREATE INDEX alkis_0b285432_09d1_4df2_8acf_4739e1139a0a ON ax_besonderertopographischerpunkt USING gin (bestehtaus);
		CREATE INDEX alkis_361f41bc_c31c_42c9_82d8_eef4c1711994 ON ax_besonderertopographischerpunkt USING gin (istteilvon);

		ALTER TABLE ax_bewertung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bewertung ALTER klassifizierung SET NOT NULL;
		ALTER TABLE ax_bewertung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bewertung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bewertung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bewertung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bewertung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bewertung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bewertung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bewertung ADD istteilvon character(16)[];
		ALTER TABLE ax_bewertung ADD processstep_address character varying[];
		ALTER TABLE ax_bewertung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bewertung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bewertung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bewertung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bewertung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bewertung ADD processstep_individualname character varying[];
		ALTER TABLE ax_bewertung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bewertung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bewertung ADD processstep_phone character varying[];
		ALTER TABLE ax_bewertung ADD processstep_positionname character varying[];
		ALTER TABLE ax_bewertung ADD processstep_rationale character varying[];
		ALTER TABLE ax_bewertung ADD processstep_role character varying[];
		ALTER TABLE ax_bewertung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bewertung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bewertung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bewertung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bewertung ADD statement character varying;
		ALTER TABLE ax_bewertung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bewertung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bewertung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bewertung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_87fb9190_30fc_4ac2_890e_f1066d85b873 ON ax_bewertung USING btree (endet);
		CREATE INDEX alkis_c16a72f3_17ed_419d_8e6e_dbf9c29d0d6c ON ax_bewertung USING gin (hatdirektunten);
		CREATE INDEX alkis_e5d2522d_3c80_4a50_9887_a6cb7da65faf ON ax_bewertung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_daddb948_bfd0_4108_be7b_cb82c3dc77ec ON ax_bewertung USING gin (istteilvon);
		CREATE INDEX alkis_1450698a_3081_4669_a1c5_e662fc9ed29b ON ax_bewertung USING gin (traegtbeizu);

		ALTER TABLE ax_bodenschaetzung RENAME art TO zeigtaufexternes_art;
		-- ax_bodenschaetzung.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_bodenschaetzung ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_bodenschaetzung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bodenschaetzung ALTER bodenart SET NOT NULL;
		ALTER TABLE ax_bodenschaetzung ALTER kulturart SET NOT NULL;
		ALTER TABLE ax_bodenschaetzung RENAME name TO zeigtaufexternes_name;
		-- ax_bodenschaetzung.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_bodenschaetzung ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_bodenschaetzung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_bodenschaetzung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_bodenschaetzung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_bodenschaetzung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_bodenschaetzung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_bodenschaetzung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_bodenschaetzung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_bodenschaetzung ADD istteilvon character(16)[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_address character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_individualname character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_phone character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_positionname character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_rationale character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_role character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_bodenschaetzung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_bodenschaetzung ADD statement character varying;
		ALTER TABLE ax_bodenschaetzung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_bodenschaetzung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_154bc81a_fbda_496f_bb4e_e2e145608f67 ON ax_bodenschaetzung USING btree (endet);
		CREATE INDEX alkis_b058e450_f586_4c25_a653_3712fac8a3f9 ON ax_bodenschaetzung USING gin (hatdirektunten);
		CREATE INDEX alkis_0984a710_c22c_4476_8322_a402c404bb3c ON ax_bodenschaetzung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_54caaeed_6c8b_4bf3_be05_be55410ab745 ON ax_bodenschaetzung USING gin (istteilvon);
		CREATE INDEX alkis_9c161b79_15b1_460c_a7e5_e627a605711b ON ax_bodenschaetzung USING gin (traegtbeizu);

		ALTER TABLE ax_boeschungkliff ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_boeschungkliff ADD bestehtaus character(16)[];
		ALTER TABLE ax_boeschungkliff ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_boeschungkliff ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_boeschungkliff ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_boeschungkliff ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_boeschungkliff ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_boeschungkliff ADD istteilvon character(16)[];
		ALTER TABLE ax_boeschungkliff ADD name character varying;
		ALTER TABLE ax_boeschungkliff ADD processstep_address character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_datetime character(20)[];
		ALTER TABLE ax_boeschungkliff ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_individualname character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_organisationname character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_phone character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_positionname character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_rationale character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_role character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_boeschungkliff ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_boeschungkliff ADD statement character varying;
		ALTER TABLE ax_boeschungkliff ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_boeschungkliff ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_boeschungkliff ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_boeschungkliff ADD zustand integer;
		CREATE INDEX alkis_9797aa3f_af91_4219_93ce_bae08ede27f2 ON ax_boeschungkliff USING btree (endet);
		CREATE INDEX alkis_8f59fb77_7230_4ec7_a5f9_096e6f67acfb ON ax_boeschungkliff USING gin (bestehtaus);
		CREATE INDEX alkis_a13db4e5_77df_47d7_a3df_eb7d12079720 ON ax_boeschungkliff USING gin (istteilvon);

		ALTER TABLE ax_boeschungsflaeche ALTER beginnt SET NOT NULL;
		-- ax_boeschungsflaeche.istteilvon => istteilvon: character(16) => character(16)[]
		ALTER TABLE ax_boeschungsflaeche ALTER istteilvon TYPE character(16)[] USING CASE WHEN istteilvon IS NULL THEN NULL ELSE ARRAY[istteilvon] END;
		ALTER TABLE ax_boeschungsflaeche ADD hatdirektunten character(16)[];
		ALTER TABLE ax_boeschungsflaeche ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_boeschungsflaeche ADD traegtbeizu character(16)[];
		ALTER TABLE ax_boeschungsflaeche ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_boeschungsflaeche ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_boeschungsflaeche ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_boeschungsflaeche' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_boeschungsflaeche USING btree (istteilvon)';
		CREATE INDEX alkis_bacfa673_743d_439e_a495_b451abf9b13b ON ax_boeschungsflaeche USING btree (endet);
		CREATE INDEX alkis_303c3669_1366_4619_8c07_f299c82611e5 ON ax_boeschungsflaeche USING gin (hatdirektunten);
		CREATE INDEX alkis_5a612553_f658_4f02_bbd3_91ce7dfd18e5 ON ax_boeschungsflaeche USING gin (istabgeleitetaus);
		CREATE INDEX alkis_b90bf03f_8462_428a_9f34_8a45da1a77d7 ON ax_boeschungsflaeche USING gin (istteilvon);
		CREATE INDEX alkis_4416a2b4_2f5d_4252_b58a_0092ae23ea94 ON ax_boeschungsflaeche USING gin (traegtbeizu);

		ALTER TABLE ax_buchungsblatt RENAME art TO zeigtaufexternes_art;
		-- ax_buchungsblatt.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_buchungsblatt ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_buchungsblatt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_buchungsblatt ALTER bezirk SET NOT NULL;
		-- ax_buchungsblatt.blattart => blattart: character varying => integer
		ALTER TABLE ax_buchungsblatt ALTER blattart TYPE integer USING blattart::int;
		ALTER TABLE ax_buchungsblatt ALTER blattart SET NOT NULL;
		ALTER TABLE ax_buchungsblatt ALTER buchungsblattkennzeichen SET NOT NULL;
		ALTER TABLE ax_buchungsblatt ALTER buchungsblattnummermitbuchstabenerweiterung SET NOT NULL;
		ALTER TABLE ax_buchungsblatt ALTER land SET NOT NULL;
		ALTER TABLE ax_buchungsblatt ADD istteilvon character(16)[];
		ALTER TABLE ax_buchungsblatt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_buchungsblatt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_fb82bd5d_aa48_448f_8ea0_5131bfde22fd ON ax_buchungsblatt USING btree (endet);
		CREATE INDEX alkis_e52deb37_704a_487d_be0e_3cbc5637bfe0 ON ax_buchungsblatt USING gin (istteilvon);

		ALTER TABLE ax_buchungsblattbezirk ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_buchungsblattbezirk ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_buchungsblattbezirk ALTER bezirk SET NOT NULL;
		ALTER TABLE ax_buchungsblattbezirk DROP gehoertzu;
		ALTER TABLE ax_buchungsblattbezirk ALTER land SET NOT NULL;
		ALTER TABLE ax_buchungsblattbezirk ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_buchungsblattbezirk RENAME stelle TO gehoertzu_stelle;
		ALTER TABLE ax_buchungsblattbezirk ADD administrativefunktion integer[];
		ALTER TABLE ax_buchungsblattbezirk ADD gehoertzu_land character varying;
		ALTER TABLE ax_buchungsblattbezirk ADD istteilvon character(16)[];
		ALTER TABLE ax_buchungsblattbezirk ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_buchungsblattbezirk ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_buchungsblattbezirk ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2cf634ae_780c_4b4f_af12_87f1c219b70b ON ax_buchungsblattbezirk USING btree (endet);
		CREATE INDEX alkis_cf6bb3e4_045b_44b7_9361_aaa14ec3d258 ON ax_buchungsblattbezirk USING gin (istteilvon);

		ALTER TABLE ax_buchungsstelle ALTER beginnt SET NOT NULL;
		-- ax_buchungsstelle.beschreibungdesumfangsderbuchung => beschreibungdesumfangsderbuchung: character(1) => character varying
		ALTER TABLE ax_buchungsstelle ALTER beschreibungdesumfangsderbuchung TYPE character varying;
		ALTER TABLE ax_buchungsstelle ALTER buchungsart SET NOT NULL;
		ALTER TABLE ax_buchungsstelle ALTER istbestandteilvon SET NOT NULL;
		ALTER TABLE ax_buchungsstelle ALTER laufendenummer SET NOT NULL;
		ALTER TABLE ax_buchungsstelle ADD istteilvon character(16)[];
		ALTER TABLE ax_buchungsstelle ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_buchungsstelle ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_buchungsstelle ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_buchungsstelle ADD zeitpunktdereintragung date;
		CREATE INDEX alkis_33b89c13_234f_4094_9dd7_8167c4a6ac41 ON ax_buchungsstelle USING btree (endet);
		CREATE INDEX alkis_ae09a09f_12ee_431d_ab58_5a08e3ba2716 ON ax_buchungsstelle USING gin (istteilvon);

		ALTER TABLE ax_bundesland ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_bundesland ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_bundesland ALTER land SET NOT NULL;
		ALTER TABLE ax_bundesland ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_bundesland RENAME stelle TO istamtsbezirkvon_stelle;
		-- ax_bundesland.stelle => istamtsbezirkvon_stelle: character varying => character varying[]
		ALTER TABLE ax_bundesland ALTER istamtsbezirkvon_stelle TYPE character varying[] USING CASE WHEN istamtsbezirkvon_stelle IS NULL THEN NULL ELSE ARRAY[istamtsbezirkvon_stelle] END;
		ALTER TABLE ax_bundesland ADD administrativefunktion integer[];
		ALTER TABLE ax_bundesland ADD istamtsbezirkvon_land character varying[];
		ALTER TABLE ax_bundesland ADD istteilvon character(16)[];
		ALTER TABLE ax_bundesland ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_bundesland ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_bundesland ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_20de57d7_48a1_4b74_8457_a39eb839d9b5 ON ax_bundesland USING btree (endet);
		CREATE INDEX alkis_177b9d11_962d_430e_9ef2_e886b6263d44 ON ax_bundesland USING gin (istteilvon);

		ALTER TABLE ax_dammwalldeich ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_dammwalldeich ADD bezeichnung character varying;
		ALTER TABLE ax_dammwalldeich ADD hatdirektunten character(16)[];
		ALTER TABLE ax_dammwalldeich ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_dammwalldeich ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_dammwalldeich ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_dammwalldeich ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_dammwalldeich ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_dammwalldeich ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_dammwalldeich ADD istteilvon character(16)[];
		ALTER TABLE ax_dammwalldeich ADD objekthoehe double precision;
		ALTER TABLE ax_dammwalldeich ADD processstep_address character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_datetime character(20)[];
		ALTER TABLE ax_dammwalldeich ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_individualname character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_organisationname character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_phone character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_positionname character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_rationale character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_role character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_dammwalldeich ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_dammwalldeich ADD statement character varying;
		ALTER TABLE ax_dammwalldeich ADD traegtbeizu character(16)[];
		ALTER TABLE ax_dammwalldeich ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_dammwalldeich ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_dammwalldeich ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_c046d253_4ea4_4fa3_a70c_c04e36955815 ON ax_dammwalldeich USING btree (endet);
		CREATE INDEX alkis_ed19fd19_d53d_4336_9608_a764bd2e6e0b ON ax_dammwalldeich USING gin (hatdirektunten);
		CREATE INDEX alkis_641f2425_b248_4d57_b6cb_b8411bff3354 ON ax_dammwalldeich USING gin (istabgeleitetaus);
		CREATE INDEX alkis_86a39d4f_bf32_49ba_9f9b_fa6d9413b46a ON ax_dammwalldeich USING gin (istteilvon);
		CREATE INDEX alkis_e1137839_7ab5_4b77_893e_dd5bf6328446 ON ax_dammwalldeich USING gin (traegtbeizu);

		ALTER TABLE ax_denkmalschutzrecht RENAME art TO zeigtaufexternes_art;
		-- ax_denkmalschutzrecht.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_denkmalschutzrecht ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_denkmalschutzrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_denkmalschutzrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_denkmalschutzrecht ADD bezeichnung character varying;
		ALTER TABLE ax_denkmalschutzrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_denkmalschutzrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_denkmalschutzrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_address character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_role character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD statement character varying;
		ALTER TABLE ax_denkmalschutzrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_denkmalschutzrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_denkmalschutzrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4d1bf09c_115b_41cb_83ae_bc608848ca56 ON ax_denkmalschutzrecht USING btree (endet);
		CREATE INDEX alkis_3d0cbc45_7ba5_4867_9298_8a052312e3de ON ax_denkmalschutzrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_e822c652_d8a3_455b_9821_87777be09d87 ON ax_denkmalschutzrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_f89a4c4d_70cb_440f_9fee_3e961d35a58b ON ax_denkmalschutzrecht USING gin (istteilvon);
		CREATE INDEX alkis_90dd0d36_5b10_40ff_a8f3_d43453a773b1 ON ax_denkmalschutzrecht USING gin (traegtbeizu);

		ALTER TABLE ax_dienststelle ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_dienststelle ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_dienststelle ALTER land SET NOT NULL;
		ALTER TABLE ax_dienststelle ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_dienststelle ALTER stelle SET NOT NULL;
		ALTER TABLE ax_dienststelle ADD administrativefunktion integer[];
		ALTER TABLE ax_dienststelle ADD istteilvon character(16)[];
		ALTER TABLE ax_dienststelle ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_dienststelle ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_dienststelle ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_7d0bd967_aa79_47c3_9cc2_0a3114aa5fa5 ON ax_dienststelle USING btree (endet);
		CREATE INDEX alkis_eeae2144_b738_478b_88ba_e87d61a8baab ON ax_dienststelle USING btree (hat);
		CREATE INDEX alkis_3bc4d2c7_cd43_416a_b6a0_4130e5525d07 ON ax_dienststelle USING gin (istteilvon);

		ALTER TABLE ax_duene ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_duene ADD hatdirektunten character(16)[];
		ALTER TABLE ax_duene ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_duene ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_duene ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_duene ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_duene ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_duene ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_duene ADD istteilvon character(16)[];
		ALTER TABLE ax_duene ADD processstep_address character varying[];
		ALTER TABLE ax_duene ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_duene ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_duene ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_duene ADD processstep_datetime character(20)[];
		ALTER TABLE ax_duene ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_duene ADD processstep_individualname character varying[];
		ALTER TABLE ax_duene ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_duene ADD processstep_organisationname character varying[];
		ALTER TABLE ax_duene ADD processstep_phone character varying[];
		ALTER TABLE ax_duene ADD processstep_positionname character varying[];
		ALTER TABLE ax_duene ADD processstep_rationale character varying[];
		ALTER TABLE ax_duene ADD processstep_role character varying[];
		ALTER TABLE ax_duene ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_duene ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_duene ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_duene ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_duene ADD statement character varying;
		ALTER TABLE ax_duene ADD traegtbeizu character(16)[];
		ALTER TABLE ax_duene ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_duene ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_duene ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_a30b3ca7_ffef_4471_ac89_8f48a3bb31bb ON ax_duene USING btree (endet);
		CREATE INDEX alkis_36c30b5b_f958_44cb_ba62_309f6f970132 ON ax_duene USING gin (hatdirektunten);
		CREATE INDEX alkis_5edf69d7_2bf5_4b7e_bfe9_55664ff16e0f ON ax_duene USING gin (istabgeleitetaus);
		CREATE INDEX alkis_819e01f3_be87_4127_a103_9e705b1d4fa4 ON ax_duene USING gin (istteilvon);
		CREATE INDEX alkis_ba54a5c0_dec4_4fcd_8642_1e2793e7109f ON ax_duene USING gin (traegtbeizu);

		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ALTER art SET NOT NULL;
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ALTER beginnt SET NOT NULL;
		-- ax_einrichtungenfuerdenschiffsverkehr.kilometerangabe => kilometerangabe: character varying => double precision
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ALTER kilometerangabe TYPE double precision USING kilometerangabe::double precision;
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD bezeichnung character varying;
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD hatdirektunten character(16)[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD istteilvon character(16)[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_address character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_datetime character(20)[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_individualname character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_organisationname character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_phone character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_positionname character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_rationale character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_role character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD statement character varying;
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD traegtbeizu character(16)[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_einrichtungenfuerdenschiffsverkehr ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_bf834453_e06d_41a0_bee3_00ceaf4e83d5 ON ax_einrichtungenfuerdenschiffsverkehr USING btree (endet);
		CREATE INDEX alkis_9bc5ab35_7e5a_4129_a0bd_37fd1adbbb0f ON ax_einrichtungenfuerdenschiffsverkehr USING gin (hatdirektunten);
		CREATE INDEX alkis_1989f906_6c91_4f8c_bf9d_5e58d356f514 ON ax_einrichtungenfuerdenschiffsverkehr USING gin (istabgeleitetaus);
		CREATE INDEX alkis_d485d94e_2790_4266_8822_ab26c213e62e ON ax_einrichtungenfuerdenschiffsverkehr USING gin (istteilvon);
		CREATE INDEX alkis_3c8e3463_8847_41d8_b9b5_9d7d4620f8ff ON ax_einrichtungenfuerdenschiffsverkehr USING gin (traegtbeizu);

		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ALTER art SET NOT NULL;
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ALTER beginnt SET NOT NULL;
		-- ax_einrichtunginoeffentlichenbereichen.kilometerangabe => kilometerangabe: character varying => double precision
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ALTER kilometerangabe TYPE double precision USING kilometerangabe::double precision;
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD hatdirektunten character(16)[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD istteilvon character(16)[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_address character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_datetime character(20)[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_individualname character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_organisationname character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_phone character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_positionname character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_rationale character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_role character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD statement character varying;
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD traegtbeizu character(16)[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_einrichtunginoeffentlichenbereichen ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_226537cd_6adb_4302_a125_42b3302abd77 ON ax_einrichtunginoeffentlichenbereichen USING btree (endet);
		CREATE INDEX alkis_22bb6df7_b98d_4a02_9f26_96677d9662aa ON ax_einrichtunginoeffentlichenbereichen USING gin (hatdirektunten);
		CREATE INDEX alkis_d704c80f_e841_486b_994b_6c6f5f9b5427 ON ax_einrichtunginoeffentlichenbereichen USING gin (istabgeleitetaus);
		CREATE INDEX alkis_941168c8_8d32_42a9_ba5b_24460a676dd6 ON ax_einrichtunginoeffentlichenbereichen USING gin (istteilvon);
		CREATE INDEX alkis_c323cc57_005e_4fff_beee_607e0fe5a6d0 ON ax_einrichtunginoeffentlichenbereichen USING gin (traegtbeizu);

		ALTER TABLE ax_felsenfelsblockfelsnadel ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD bezeichnung character varying;
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD hatdirektunten character(16)[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD istteilvon character(16)[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD objekthoehe double precision;
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_address character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_datetime character(20)[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_individualname character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_organisationname character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_phone character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_positionname character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_rationale character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_role character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD statement character varying;
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD traegtbeizu character(16)[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_felsenfelsblockfelsnadel ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e6d6932a_8fca_4c45_bc9f_dfd853945cbc ON ax_felsenfelsblockfelsnadel USING btree (endet);
		CREATE INDEX alkis_251b30b0_7926_42d6_b154_9cb67fef3dfc ON ax_felsenfelsblockfelsnadel USING gin (hatdirektunten);
		CREATE INDEX alkis_8e8e4602_4168_4bb3_882a_d9f33430af9b ON ax_felsenfelsblockfelsnadel USING gin (istabgeleitetaus);
		CREATE INDEX alkis_86d6c488_657a_4893_9209_819b92673229 ON ax_felsenfelsblockfelsnadel USING gin (istteilvon);
		CREATE INDEX alkis_d269714c_448b_4c16_9702_a4b3541d7150 ON ax_felsenfelsblockfelsnadel USING gin (traegtbeizu);

		ALTER TABLE ax_firstlinie RENAME art TO zeigtaufexternes_art;
		-- ax_firstlinie.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_firstlinie ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_firstlinie ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_firstlinie RENAME uri TO zeigtaufexternes_uri;
		-- ax_firstlinie.uri => zeigtaufexternes_uri: character varying => character varying[]
		ALTER TABLE ax_firstlinie ALTER zeigtaufexternes_uri TYPE character varying[] USING CASE WHEN zeigtaufexternes_uri IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_uri] END;
		-- ax_firstlinie.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_firstlinie DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_firstlinie RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_firstlinie','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_firstlinie SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_firstlinie DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_firstlinie_wkb_geometry_idx ON ax_firstlinie USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_firstlinie ADD hatdirektunten character(16)[];
		ALTER TABLE ax_firstlinie ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_firstlinie ADD istteilvon character(16)[];
		ALTER TABLE ax_firstlinie ADD traegtbeizu character(16)[];
		ALTER TABLE ax_firstlinie ADD zeigtaufexternes_name character varying[];
		CREATE INDEX alkis_a81a3d7e_ffc8_473a_9782_2c4aad862225 ON ax_firstlinie USING btree (endet);
		CREATE INDEX alkis_68ecdb08_6e43_4193_ad1f_e9c2b9d0a56b ON ax_firstlinie USING gin (hatdirektunten);
		CREATE INDEX alkis_22c7fc14_570b_4555_b321_8ffbd5765b8f ON ax_firstlinie USING gin (istabgeleitetaus);
		CREATE INDEX alkis_1c90850d_6c1a_4387_bafe_9dff7f00a6dc ON ax_firstlinie USING gin (istteilvon);
		CREATE INDEX alkis_cc53cd4a_b12c_469b_b749_4828140ec9a8 ON ax_firstlinie USING gin (traegtbeizu);

		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD istteilvon character(16)[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_address character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_individualname character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_phone character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_positionname character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_rationale character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_role character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD statement character varying;
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_flaechebesondererfunktionalerpraegung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_ab792357_45a5_41a1_a434_377f600ba902 ON ax_flaechebesondererfunktionalerpraegung USING btree (endet);
		CREATE INDEX alkis_c757afa3_2915_409f_a4cf_09036cf42af7 ON ax_flaechebesondererfunktionalerpraegung USING gin (hatdirektunten);
		CREATE INDEX alkis_5532c029_24fc_4d92_8bc4_832f407dcb1e ON ax_flaechebesondererfunktionalerpraegung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_862beefa_6132_4fd3_b1a9_5d9079b029e2 ON ax_flaechebesondererfunktionalerpraegung USING gin (istteilvon);
		CREATE INDEX alkis_d85d7526_d456_45db_84bb_d5034c7da23f ON ax_flaechebesondererfunktionalerpraegung USING gin (traegtbeizu);

		ALTER TABLE ax_flaechegemischternutzung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_flaechegemischternutzung ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_flaechegemischternutzung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_flaechegemischternutzung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_flaechegemischternutzung ADD istteilvon character(16)[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_address character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_individualname character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_phone character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_positionname character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_rationale character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_role character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD statement character varying;
		ALTER TABLE ax_flaechegemischternutzung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_flaechegemischternutzung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_flaechegemischternutzung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_676e630d_d2fa_4842_903d_18204789903e ON ax_flaechegemischternutzung USING btree (endet);
		CREATE INDEX alkis_cd643cb6_5c46_40c2_92a8_76aeb7b8de55 ON ax_flaechegemischternutzung USING gin (hatdirektunten);
		CREATE INDEX alkis_27ab6964_0eba_4dec_b6a9_aa869dd9c3cb ON ax_flaechegemischternutzung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_4764023d_223e_47fd_a00b_a9f32ee11223 ON ax_flaechegemischternutzung USING gin (istteilvon);
		CREATE INDEX alkis_df5d6f81_9fcf_4f69_ad26_bea5e2c10866 ON ax_flaechegemischternutzung USING gin (traegtbeizu);

		ALTER TABLE ax_fliessgewaesser ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_fliessgewaesser RENAME name TO zeigtaufexternes_name;
		-- ax_fliessgewaesser.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_fliessgewaesser ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_fliessgewaesser ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_fliessgewaesser ADD gemeinde character varying;
		ALTER TABLE ax_fliessgewaesser ADD hatdirektunten character(16)[];
		ALTER TABLE ax_fliessgewaesser ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_fliessgewaesser ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_fliessgewaesser ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_fliessgewaesser ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_fliessgewaesser ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_fliessgewaesser ADD hydrologischesmerkmal integer;
		ALTER TABLE ax_fliessgewaesser ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_fliessgewaesser ADD istteilvon character(16)[];
		ALTER TABLE ax_fliessgewaesser ADD kreis character varying;
		ALTER TABLE ax_fliessgewaesser ADD lage character varying;
		ALTER TABLE ax_fliessgewaesser ADD land character varying;
		ALTER TABLE ax_fliessgewaesser ADD processstep_address character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_datetime character(20)[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_individualname character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_organisationname character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_phone character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_positionname character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_rationale character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_role character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_fliessgewaesser ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_fliessgewaesser ADD regierungsbezirk character varying;
		ALTER TABLE ax_fliessgewaesser ADD statement character varying;
		ALTER TABLE ax_fliessgewaesser ADD traegtbeizu character(16)[];
		ALTER TABLE ax_fliessgewaesser ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_fliessgewaesser ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_bd2f3148_e944_45e3_8b41_dc94fd3e3b39 ON ax_fliessgewaesser USING btree (endet);
		CREATE INDEX alkis_41aacf9f_f09e_4f28_bc4b_fb5fd3e05fa9 ON ax_fliessgewaesser USING gin (hatdirektunten);
		CREATE INDEX alkis_7340a489_10e6_4e54_bc69_ec0663c3f0e6 ON ax_fliessgewaesser USING gin (istabgeleitetaus);
		CREATE INDEX alkis_325f3576_e2c7_41f5_8dad_4ddd2f547ad9 ON ax_fliessgewaesser USING gin (istteilvon);
		CREATE INDEX alkis_6ee4a06a_f55f_4ba2_9708_c3cebeae808e ON ax_fliessgewaesser USING gin (traegtbeizu);

		ALTER TABLE ax_flugverkehr ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_flugverkehr RENAME name TO zeigtaufexternes_name;
		-- ax_flugverkehr.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_flugverkehr ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_flugverkehr ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_flugverkehr ADD gemeinde character varying;
		ALTER TABLE ax_flugverkehr ADD hatdirektunten character(16)[];
		ALTER TABLE ax_flugverkehr ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_flugverkehr ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_flugverkehr ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_flugverkehr ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_flugverkehr ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_flugverkehr ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_flugverkehr ADD istteilvon character(16)[];
		ALTER TABLE ax_flugverkehr ADD kreis character varying;
		ALTER TABLE ax_flugverkehr ADD lage character varying;
		ALTER TABLE ax_flugverkehr ADD land character varying;
		ALTER TABLE ax_flugverkehr ADD processstep_address character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_datetime character(20)[];
		ALTER TABLE ax_flugverkehr ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_individualname character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_organisationname character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_phone character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_positionname character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_rationale character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_role character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_flugverkehr ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_flugverkehr ADD regierungsbezirk character varying;
		ALTER TABLE ax_flugverkehr ADD statement character varying;
		ALTER TABLE ax_flugverkehr ADD traegtbeizu character(16)[];
		ALTER TABLE ax_flugverkehr ADD unverschluesselt character varying;
		ALTER TABLE ax_flugverkehr ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_flugverkehr ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_525ef6bf_aa47_4463_a688_22688a1c4cad ON ax_flugverkehr USING btree (endet);
		CREATE INDEX alkis_33398a2a_863e_4a7c_ab08_e0dbb0da76d1 ON ax_flugverkehr USING gin (hatdirektunten);
		CREATE INDEX alkis_d78a37c4_b5d4_4363_aa0d_697c334d77b2 ON ax_flugverkehr USING gin (istabgeleitetaus);
		CREATE INDEX alkis_65a6e0f1_03d2_42b4_bde4_0920125c428a ON ax_flugverkehr USING gin (istteilvon);
		CREATE INDEX alkis_34c58017_d2d9_475d_901b_79a567a2159f ON ax_flugverkehr USING gin (traegtbeizu);

		ALTER TABLE ax_flugverkehrsanlage ALTER art SET NOT NULL;
		ALTER TABLE ax_flugverkehrsanlage ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_flugverkehrsanlage ADD bezeichnung character varying;
		ALTER TABLE ax_flugverkehrsanlage ADD breitedesobjekts double precision;
		ALTER TABLE ax_flugverkehrsanlage ADD hatdirektunten character(16)[];
		ALTER TABLE ax_flugverkehrsanlage ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_flugverkehrsanlage ADD istteilvon character(16)[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_address character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_datetime character(20)[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_individualname character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_organisationname character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_phone character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_positionname character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_rationale character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_role character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD statement character varying;
		ALTER TABLE ax_flugverkehrsanlage ADD traegtbeizu character(16)[];
		ALTER TABLE ax_flugverkehrsanlage ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_flugverkehrsanlage ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e42a4d9e_2067_43e8_b06d_6e27bd5d1015 ON ax_flugverkehrsanlage USING btree (endet);
		CREATE INDEX alkis_17fb735c_1f0a_4070_8d59_97e3ae529c67 ON ax_flugverkehrsanlage USING gin (hatdirektunten);
		CREATE INDEX alkis_04b4748f_57ca_4fe4_9971_20c714b444bb ON ax_flugverkehrsanlage USING gin (istabgeleitetaus);
		CREATE INDEX alkis_0b881eae_9d3f_4a38_884b_c85652d95d60 ON ax_flugverkehrsanlage USING gin (istteilvon);
		CREATE INDEX alkis_81881d87_9a91_4e81_8746_9325dbf1e361 ON ax_flugverkehrsanlage USING gin (traegtbeizu);

		PERFORM alkis_dropobject('v_eigentuemer');
		PERFORM alkis_dropobject('v_haeuser');
		PERFORM alkis_dropobject('ax_flurstueck_flsnr');
		PERFORM alkis_dropobject('alkis_flsnrk');
		PERFORM alkis_dropobject('alkis_flsnr');

		ALTER TABLE ax_flurstueck ALTER amtlicheflaeche SET NOT NULL;
		ALTER TABLE ax_flurstueck ALTER beginnt SET NOT NULL;
		-- ax_flurstueck.flurstueckskennzeichen => flurstueckskennzeichen: character(20) => character varying
		ALTER TABLE ax_flurstueck ALTER flurstueckskennzeichen TYPE character varying;
		ALTER TABLE ax_flurstueck ALTER flurstueckskennzeichen SET NOT NULL;
		ALTER TABLE ax_flurstueck ALTER gemarkungsnummer SET NOT NULL;
		ALTER TABLE ax_flurstueck RENAME gemeinde TO gemeindezugehoerigkeit_gemeinde;
		ALTER TABLE ax_flurstueck ALTER istgebucht SET NOT NULL;
		ALTER TABLE ax_flurstueck RENAME kreis TO gemeindezugehoerigkeit_kreis;
		ALTER TABLE ax_flurstueck ALTER land SET NOT NULL;
		ALTER TABLE ax_flurstueck RENAME name TO zeigtaufexternes_name;
		ALTER TABLE ax_flurstueck RENAME regierungsbezirk TO gemeindezugehoerigkeit_regierungsbezirk;
		ALTER TABLE ax_flurstueck RENAME stelle TO zustaendigestelle_stelle;
		-- ax_flurstueck.stelle => zustaendigestelle_stelle: character varying => character varying[]
		ALTER TABLE ax_flurstueck ALTER zustaendigestelle_stelle TYPE character varying[] USING CASE WHEN zustaendigestelle_stelle IS NULL THEN NULL ELSE ARRAY[zustaendigestelle_stelle] END;
		-- ax_flurstueck.zaehler => zaehler: integer => character varying
		ALTER TABLE ax_flurstueck ALTER zaehler TYPE character varying USING zaehler::varchar;
		ALTER TABLE ax_flurstueck ALTER zaehler SET NOT NULL;
		-- ax_flurstueck.zeitpunktderentstehung => zeitpunktderentstehung: character varying => date
		ALTER TABLE ax_flurstueck ALTER zeitpunktderentstehung TYPE date USING to_date(zeitpunktderentstehung, 'YYYY-mm-dd');
		ALTER TABLE ax_flurstueck ADD angabenzumabschnittstelle character varying[];
		ALTER TABLE ax_flurstueck ADD gemeindezugehoerigkeit_gemeindeteil character varying;
		ALTER TABLE ax_flurstueck ADD gemeindezugehoerigkeit_land character varying;
		ALTER TABLE ax_flurstueck ADD hatdirektunten character(16)[];
		ALTER TABLE ax_flurstueck ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_flurstueck ADD istteilvon character(16)[];
		PERFORM AddGeometryColumn('ax_flurstueck','objektkoordinaten',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'POINT',2);
		ALTER TABLE ax_flurstueck ADD traegtbeizu character(16)[];
		ALTER TABLE ax_flurstueck ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_flurstueck ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_flurstueck ADD zustaendigestelle_land character varying[];
		CREATE INDEX alkis_27e07827_6d01_43f4_ad5f_8dea320a2740 ON ax_flurstueck USING btree (endet);
		CREATE INDEX alkis_cb85f565_7819_42bb_8803_5bbafa698aee ON ax_flurstueck USING gin (hatdirektunten);
		CREATE INDEX alkis_1f28a686_b0cc_4e60_a4ce_bbc792087c27 ON ax_flurstueck USING gin (istabgeleitetaus);
		CREATE INDEX alkis_91dd21ec_509f_4b3d_92ae_ba53318da7b7 ON ax_flurstueck USING gin (istteilvon);
		CREATE INDEX alkis_69f6e6f3_3ff8_4001_b2df_b3e0e9b146be ON ax_flurstueck USING gin (traegtbeizu);
		CREATE INDEX alkis_b4a717b0_cbce_4dfe_8665_c995ceaca879 ON ax_flurstueck USING gist (objektkoordinaten);

		ALTER TABLE ax_forstrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_forstrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_forstrecht ADD bezeichnung character varying;
		ALTER TABLE ax_forstrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_forstrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_forstrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_forstrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_forstrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_forstrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_forstrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_forstrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_forstrecht ADD processstep_address character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_forstrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_role character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_forstrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_forstrecht ADD statement character varying;
		ALTER TABLE ax_forstrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_forstrecht ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_forstrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_forstrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_ad777bd3_c408_4486_a5c1_20f8a1860a73 ON ax_forstrecht USING btree (endet);
		CREATE INDEX alkis_9d75f975_be8e_4153_8a60_127223f35dc8 ON ax_forstrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_34647045_7b35_4045_b7ec_bb398e15a775 ON ax_forstrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_d29c4644_2af7_4753_96df_f684e7611417 ON ax_forstrecht USING gin (istteilvon);
		CREATE INDEX alkis_61b1563b_d105_456f_9543_7045bc073e70 ON ax_forstrecht USING gin (traegtbeizu);

		ALTER TABLE ax_fortfuehrungsfall ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_fortfuehrungsfall ALTER fortfuehrungsfallnummer SET NOT NULL;
		-- ax_fortfuehrungsfall.laufendenummer => laufendenummer: integer => character varying
		ALTER TABLE ax_fortfuehrungsfall ALTER laufendenummer TYPE character varying USING laufendenummer::varchar;
		-- ax_fortfuehrungsfall.ueberschriftimfortfuehrungsnachweis => ueberschriftimfortfuehrungsnachweis: integer[] => character varying[]
		ALTER TABLE ax_fortfuehrungsfall RENAME ueberschriftimfortfuehrungsnachweis TO ueberschriftimfortfuehrungsnachweis_;
		ALTER TABLE ax_fortfuehrungsfall ADD ueberschriftimfortfuehrungsnachweis character varying[];
		UPDATE ax_fortfuehrungsfall SET ueberschriftimfortfuehrungsnachweis=ARRAY(SELECT unnest(ueberschriftimfortfuehrungsnachweis_)::text);
		ALTER TABLE ax_fortfuehrungsfall DROP ueberschriftimfortfuehrungsnachweis_;
		ALTER TABLE ax_fortfuehrungsfall ALTER ueberschriftimfortfuehrungsnachweis SET NOT NULL;
		ALTER TABLE ax_fortfuehrungsfall RENAME uri TO zeigtaufexternes_uri;
		-- ax_fortfuehrungsfall.uri => zeigtaufexternes_uri: character varying => character varying[]
		ALTER TABLE ax_fortfuehrungsfall ALTER zeigtaufexternes_uri TYPE character varying[] USING CASE WHEN zeigtaufexternes_uri IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_uri] END;
		-- ax_fortfuehrungsfall.zeigtaufaltesflurstueck => zeigtaufaltesflurstueck: character(20)[] => character varying[]
		ALTER TABLE ax_fortfuehrungsfall ALTER zeigtaufaltesflurstueck TYPE character varying[];
		-- ax_fortfuehrungsfall.zeigtaufneuesflurstueck => zeigtaufneuesflurstueck: character(20)[] => character varying[]
		ALTER TABLE ax_fortfuehrungsfall ALTER zeigtaufneuesflurstueck TYPE character varying[];
		ALTER TABLE ax_fortfuehrungsfall ADD anmerkungfuerdennotar character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD dienststellenlogo_uri character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD enthaeltewp_uri character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD fortfuehrungsmitteilunganeigentuemerantragsteller character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD istteilvon character(16)[];
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_artderausgabe character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_datumderausgabe character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_dienststelle character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_plzort character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_strassehausnummer character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_telefon character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD verweistauf_uri character varying;
		ALTER TABLE ax_fortfuehrungsfall ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_fortfuehrungsfall ADD zeigtaufexternes_name character varying[];
		CREATE INDEX alkis_cbbdd51c_a38c_4d94_af5c_6cbff266a691 ON ax_fortfuehrungsfall USING btree (endet);
		CREATE INDEX alkis_d6e173d1_f1e0_46a4_9673_d5e695c4d2c2 ON ax_fortfuehrungsfall USING btree (gml_id, beginnt);
		CREATE INDEX alkis_ae04c997_ccd8_4671_ba12_846a59be4cfb ON ax_fortfuehrungsfall USING gin (istteilvon);

		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER beginnt SET NOT NULL;
		-- ax_fortfuehrungsnachweisdeckblatt.beziehtsichauf => beziehtsichauf: character(16) => character(16)[]
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER beziehtsichauf TYPE character(16)[] USING CASE WHEN beziehtsichauf IS NULL THEN NULL ELSE ARRAY[beziehtsichauf] END;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER beziehtsichauf SET NOT NULL;
		-- ax_fortfuehrungsnachweisdeckblatt.erstelltam => erstelltam: character varying => date
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER erstelltam TYPE date USING to_date(erstelltam, 'YYYY-mm-dd');
		-- ax_fortfuehrungsnachweisdeckblatt.fortfuehrungsentscheidungam => fortfuehrungsentscheidungam: character varying => date
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER fortfuehrungsentscheidungam TYPE date USING to_date(fortfuehrungsentscheidungam, 'YYYY-mm-dd');
		-- ax_fortfuehrungsnachweisdeckblatt.laufendenummer => laufendenummer: integer => character varying
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER laufendenummer TYPE character varying USING laufendenummer::varchar;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER laufendenummer SET NOT NULL;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER titel SET NOT NULL;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt RENAME uri TO zeigtaufexternes_uri;
		-- ax_fortfuehrungsnachweisdeckblatt.uri => zeigtaufexternes_uri: character varying => character varying[]
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ALTER zeigtaufexternes_uri TYPE character varying[] USING CASE WHEN zeigtaufexternes_uri IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_uri] END;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD artderausgabe character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD auszugfuer_adressat character varying[];
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD auszugfuer_art integer[];
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD auszugfuer_datum date[];
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD bekanntgabeanbeteiligteam date;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD datumderausgabe character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD dienststelle character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD dienststellenlogo_uri character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD eignungspruefungam date;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD eignungspruefungvon character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD enthaeltewp_uri character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD erstelltvon character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD fnnummer_gemarkungsnummer character varying NOT NULL;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD fnnummer_land character varying NOT NULL;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD fortfuehrungsmitteilungenerzeugt date;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD grundbuchmitteilungfuerfinanzamt character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD ingemarkung_gemarkungsnummer character varying NOT NULL;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD ingemarkung_land character varying NOT NULL;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD istteilvon character(16)[];
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD plzort character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD strassehausnummer character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD telefon character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD verwaltungsaktjn character varying;
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_fortfuehrungsnachweisdeckblatt ADD zeigtaufexternes_name character varying[];
		CREATE INDEX alkis_89c5800e_434d_4dd0_b6ec_e04dbca6ffc4 ON ax_fortfuehrungsnachweisdeckblatt USING btree (endet);
		CREATE INDEX alkis_28654c37_5796_479c_97cc_cecce4fcd9cd ON ax_fortfuehrungsnachweisdeckblatt USING btree (gml_id, beginnt);
		CREATE INDEX alkis_4d2d1453_d6fe_4660_8756_808594eee875 ON ax_fortfuehrungsnachweisdeckblatt USING gin (beziehtsichauf);
		CREATE INDEX alkis_d73d151c_1e36_4692_8eb2_25426dc6b62b ON ax_fortfuehrungsnachweisdeckblatt USING gin (istteilvon);

		ALTER TABLE ax_friedhof ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_friedhof ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_friedhof ADD hatdirektunten character(16)[];
		ALTER TABLE ax_friedhof ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_friedhof ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_friedhof ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_friedhof ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_friedhof ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_friedhof ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_friedhof ADD istteilvon character(16)[];
		ALTER TABLE ax_friedhof ADD processstep_address character varying[];
		ALTER TABLE ax_friedhof ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_friedhof ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_friedhof ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_friedhof ADD processstep_datetime character(20)[];
		ALTER TABLE ax_friedhof ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_friedhof ADD processstep_individualname character varying[];
		ALTER TABLE ax_friedhof ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_friedhof ADD processstep_organisationname character varying[];
		ALTER TABLE ax_friedhof ADD processstep_phone character varying[];
		ALTER TABLE ax_friedhof ADD processstep_positionname character varying[];
		ALTER TABLE ax_friedhof ADD processstep_rationale character varying[];
		ALTER TABLE ax_friedhof ADD processstep_role character varying[];
		ALTER TABLE ax_friedhof ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_friedhof ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_friedhof ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_friedhof ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_friedhof ADD statement character varying;
		ALTER TABLE ax_friedhof ADD traegtbeizu character(16)[];
		ALTER TABLE ax_friedhof ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_friedhof ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_friedhof ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4adcc6b0_3cd5_48d3_81d5_3eb0fd719420 ON ax_friedhof USING btree (endet);
		CREATE INDEX alkis_598f426c_1cd4_45b8_86bf_28cd24b9e0b9 ON ax_friedhof USING gin (hatdirektunten);
		CREATE INDEX alkis_e54a16d9_8067_4b08_bb2f_1499dc6ed30f ON ax_friedhof USING gin (istabgeleitetaus);
		CREATE INDEX alkis_f9ec9f90_3e8d_4188_aeb9_a23bc5df801d ON ax_friedhof USING gin (istteilvon);
		CREATE INDEX alkis_c775f8d9_73c1_4796_834a_0e9ed18a38f1 ON ax_friedhof USING gin (traegtbeizu);

		ALTER TABLE ax_gebaeude RENAME art TO zeigtaufexternes_art;
		-- ax_gebaeude.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_gebaeude ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_gebaeude ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gebaeude RENAME description TO processstep_ax_li_processstep_mitdatenerhebung_description;
		-- ax_gebaeude.description => processstep_ax_li_processstep_mitdatenerhebung_description: integer => character varying[]
		ALTER TABLE ax_gebaeude ALTER processstep_ax_li_processstep_mitdatenerhebung_description TYPE character varying[] USING CASE WHEN processstep_ax_li_processstep_mitdatenerhebung_description IS NULL THEN NULL ELSE ARRAY[processstep_ax_li_processstep_mitdatenerhebung_description::int] END;
		ALTER TABLE ax_gebaeude ALTER gebaeudefunktion SET NOT NULL;
		ALTER TABLE ax_gebaeude RENAME individualname TO processstep_individualname;
		-- ax_gebaeude.individualname => processstep_individualname: character varying => character varying[]
		ALTER TABLE ax_gebaeude ALTER processstep_individualname TYPE character varying[] USING CASE WHEN processstep_individualname IS NULL THEN NULL ELSE ARRAY[processstep_individualname] END;
		ALTER TABLE ax_gebaeude DROP qualitaetsangaben;
		ALTER TABLE ax_gebaeude ADD anteil integer[];
		ALTER TABLE ax_gebaeude ADD hatdirektunten character(16)[];
		ALTER TABLE ax_gebaeude ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_gebaeude ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_gebaeude ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_gebaeude ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_gebaeude ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_gebaeude ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_gebaeude ADD istteilvon character(16)[];
		ALTER TABLE ax_gebaeude ADD nutzung integer[];
		ALTER TABLE ax_gebaeude ADD processstep_address character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_datetime character(20)[];
		ALTER TABLE ax_gebaeude ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_organisationname character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_phone character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_positionname character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_rationale character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_role character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_gebaeude ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_gebaeude ADD statement character varying;
		ALTER TABLE ax_gebaeude ADD traegtbeizu character(16)[];
		ALTER TABLE ax_gebaeude ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gebaeude ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_5dd50074_eca8_47b8_ab8c_4d490829111b ON ax_gebaeude USING btree (endet);
		CREATE INDEX alkis_dd0811b0_0440_46d2_a336_7e241e3874b0 ON ax_gebaeude USING gin (hatdirektunten);
		CREATE INDEX alkis_089146da_1f80_4a11_bb93_544fda3369f2 ON ax_gebaeude USING gin (istabgeleitetaus);
		CREATE INDEX alkis_d44b9d64_4abd_4ba0_98bd_8742c88c1927 ON ax_gebaeude USING gin (istteilvon);
		CREATE INDEX alkis_c6877f2d_9c71_42c7_8892_04bc76f18c50 ON ax_gebaeude USING gin (traegtbeizu);

		ALTER TABLE ax_gebaeudeausgestaltung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gebaeudeausgestaltung ALTER darstellung SET NOT NULL;
		ALTER TABLE ax_gebaeudeausgestaltung ALTER zeigtauf SET NOT NULL;
		ALTER TABLE ax_gebaeudeausgestaltung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_gebaeudeausgestaltung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_gebaeudeausgestaltung ADD istteilvon character(16)[];
		ALTER TABLE ax_gebaeudeausgestaltung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_gebaeudeausgestaltung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gebaeudeausgestaltung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gebaeudeausgestaltung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_3c37ba21_f94a_4826_b785_4b52445cd81d ON ax_gebaeudeausgestaltung USING btree (endet);
		CREATE INDEX alkis_71b39464_2f8b_45c4_b80e_20106c1f4b95 ON ax_gebaeudeausgestaltung USING btree (zeigtauf);
		CREATE INDEX alkis_c52fd0f5_bdf4_45ba_b5a3_b86deb479890 ON ax_gebaeudeausgestaltung USING gin (hatdirektunten);
		CREATE INDEX alkis_19cb724f_ef08_4e39_b4bc_d027d31596c5 ON ax_gebaeudeausgestaltung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_5dcd38bd_7ce0_4a3d_bae6_84e5f4e1dcd7 ON ax_gebaeudeausgestaltung USING gin (istteilvon);
		CREATE INDEX alkis_cab5b3c9_6617_4771_85a6_379230be110d ON ax_gebaeudeausgestaltung USING gin (traegtbeizu);

		ALTER TABLE ax_gehoelz ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gehoelz ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_gehoelz ADD hatdirektunten character(16)[];
		ALTER TABLE ax_gehoelz ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_gehoelz ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_gehoelz ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_gehoelz ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_gehoelz ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_gehoelz ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_gehoelz ADD istteilvon character(16)[];
		ALTER TABLE ax_gehoelz ADD processstep_address character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_datetime character(20)[];
		ALTER TABLE ax_gehoelz ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_individualname character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_organisationname character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_phone character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_positionname character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_rationale character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_role character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_gehoelz ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_gehoelz ADD statement character varying;
		ALTER TABLE ax_gehoelz ADD traegtbeizu character(16)[];
		ALTER TABLE ax_gehoelz ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gehoelz ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gehoelz ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_105a4e12_0883_4d66_abad_9c23523ab26f ON ax_gehoelz USING btree (endet);
		CREATE INDEX alkis_438572db_a448_45d5_a9f1_a2c06a67de47 ON ax_gehoelz USING gin (hatdirektunten);
		CREATE INDEX alkis_2c59d705_2eea_4d69_9917_3d29748034b1 ON ax_gehoelz USING gin (istabgeleitetaus);
		CREATE INDEX alkis_12a8f628_dcdc_483a_b7ff_b5cb675e72dc ON ax_gehoelz USING gin (istteilvon);
		CREATE INDEX alkis_f60b50c7_f357_4725_9bc4_2106baed130e ON ax_gehoelz USING gin (traegtbeizu);

		ALTER TABLE ax_gelaendekante RENAME art TO zeigtaufexternes_art;
		-- ax_gelaendekante.art => zeigtaufexternes_art: integer => character varying[]
		ALTER TABLE ax_gelaendekante ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art::int] END;
		ALTER TABLE ax_gelaendekante ALTER artdergelaendekante SET NOT NULL;
		ALTER TABLE ax_gelaendekante RENAME ax_dqerfassungsmethode TO erfassung_ax_dqerfassungsmethode;
		ALTER TABLE ax_gelaendekante ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gelaendekante RENAME identifikation TO erfassung_identifikation;
		-- ax_gelaendekante.istteilvon => istteilvon: character(16) => character(16)[]
		ALTER TABLE ax_gelaendekante ALTER istteilvon TYPE character(16)[] USING CASE WHEN istteilvon IS NULL THEN NULL ELSE ARRAY[istteilvon] END;
		-- ax_gelaendekante.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_gelaendekante DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_gelaendekante RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_gelaendekante','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_gelaendekante SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_gelaendekante DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_gelaendekante_wkb_geometry_idx ON ax_gelaendekante USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_gelaendekante ADD erfassung_datetime character(20);
		ALTER TABLE ax_gelaendekante ADD hatdirektunten character(16)[];
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_datetime character(20)[];
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_evaluationmethoddescription character varying;
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_evaluationmethodtype character varying;
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_evaluationprocedure character varying;
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_measuredescription character varying;
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_measureidentification character varying;
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_nameofmeasure character varying[];
		ALTER TABLE ax_gelaendekante ADD hoehengenauigkeit_result character varying[];
		ALTER TABLE ax_gelaendekante ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_gelaendekante ADD traegtbeizu character(16)[];
		ALTER TABLE ax_gelaendekante ADD ursprung integer;
		ALTER TABLE ax_gelaendekante ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gelaendekante ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_gelaendekante' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_gelaendekante USING btree (istteilvon)';
		CREATE INDEX alkis_0db30901_776c_44b6_aedd_124f25701111 ON ax_gelaendekante USING btree (endet);
		CREATE INDEX alkis_6c7c6055_990c_4a97_82e3_16d84ef87380 ON ax_gelaendekante USING gin (hatdirektunten);
		CREATE INDEX alkis_b366adc0_4ddc_4674_bec8_8cc395e62070 ON ax_gelaendekante USING gin (istabgeleitetaus);
		CREATE INDEX alkis_0b32692d_cee9_4dbb_9964_6ece25bc33e0 ON ax_gelaendekante USING gin (istteilvon);
		CREATE INDEX alkis_8c4a4019_f040_4f7a_aef1_18a88593a626 ON ax_gelaendekante USING gin (traegtbeizu);

		ALTER TABLE ax_gemarkung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gemarkung ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_gemarkung ALTER gemarkungsnummer SET NOT NULL;
		ALTER TABLE ax_gemarkung ALTER land SET NOT NULL;
		ALTER TABLE ax_gemarkung ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_gemarkung RENAME stelle TO istamtsbezirkvon_stelle;
		-- ax_gemarkung.stelle => istamtsbezirkvon_stelle: character varying => character varying[]
		ALTER TABLE ax_gemarkung ALTER istamtsbezirkvon_stelle TYPE character varying[] USING CASE WHEN istamtsbezirkvon_stelle IS NULL THEN NULL ELSE ARRAY[istamtsbezirkvon_stelle] END;
		ALTER TABLE ax_gemarkung ADD administrativefunktion integer[];
		ALTER TABLE ax_gemarkung ADD istamtsbezirkvon_land character varying[];
		ALTER TABLE ax_gemarkung ADD istteilvon character(16)[];
		ALTER TABLE ax_gemarkung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gemarkung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gemarkung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_84241d11_8817_4ff0_8e98_e1501bf6b52e ON ax_gemarkung USING btree (endet);
		CREATE INDEX alkis_4d5d6399_579d_443b_a52d_d2432e9742f0 ON ax_gemarkung USING gin (istteilvon);

		ALTER TABLE ax_gemarkungsteilflur ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gemarkungsteilflur ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_gemarkungsteilflur DROP gehoertzu;
		-- ax_gemarkungsteilflur.gemarkung => gemarkung: integer => character varying
		ALTER TABLE ax_gemarkungsteilflur ALTER gemarkung TYPE character varying USING gemarkung::varchar;
		ALTER TABLE ax_gemarkungsteilflur ALTER gemarkung SET NOT NULL;
		-- ax_gemarkungsteilflur.gemarkungsteilflur => gemarkungsteilflur: integer => character varying
		ALTER TABLE ax_gemarkungsteilflur ALTER gemarkungsteilflur TYPE character varying USING gemarkungsteilflur::varchar;
		ALTER TABLE ax_gemarkungsteilflur ALTER gemarkungsteilflur SET NOT NULL;
		ALTER TABLE ax_gemarkungsteilflur ALTER land SET NOT NULL;
		ALTER TABLE ax_gemarkungsteilflur ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_gemarkungsteilflur ADD administrativefunktion integer[];
		ALTER TABLE ax_gemarkungsteilflur ADD gehoertzu_land character varying[];
		ALTER TABLE ax_gemarkungsteilflur ADD gehoertzu_stelle character varying[];
		ALTER TABLE ax_gemarkungsteilflur ADD istteilvon character(16)[];
		ALTER TABLE ax_gemarkungsteilflur ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gemarkungsteilflur ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gemarkungsteilflur ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_f1b119b8_e7fc_40c5_8d7d_0c1cf2187a20 ON ax_gemarkungsteilflur USING btree (endet);
		CREATE INDEX alkis_4c10aa55_816e_451f_a803_b2a2b2789ddf ON ax_gemarkungsteilflur USING gin (istteilvon);

		ALTER TABLE ax_gemeinde ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gemeinde ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_gemeinde ALTER gemeinde SET NOT NULL;
		ALTER TABLE ax_gemeinde DROP istamtsbezirkvon;
		ALTER TABLE ax_gemeinde ALTER kreis SET NOT NULL;
		ALTER TABLE ax_gemeinde ALTER land SET NOT NULL;
		ALTER TABLE ax_gemeinde ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_gemeinde RENAME stelle TO istamtsbezirkvon_stelle;
		-- ax_gemeinde.stelle => istamtsbezirkvon_stelle: character varying => character varying[]
		ALTER TABLE ax_gemeinde ALTER istamtsbezirkvon_stelle TYPE character varying[] USING CASE WHEN istamtsbezirkvon_stelle IS NULL THEN NULL ELSE ARRAY[istamtsbezirkvon_stelle] END;
		ALTER TABLE ax_gemeinde ADD administrativefunktion integer[];
		ALTER TABLE ax_gemeinde ADD gemeindeteil character varying;
		ALTER TABLE ax_gemeinde ADD istamtsbezirkvon_land character varying[];
		ALTER TABLE ax_gemeinde ADD istteilvon character(16)[];
		ALTER TABLE ax_gemeinde ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gemeinde ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gemeinde ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_f3dfa217_6d68_4070_8ee0_3da328cda109 ON ax_gemeinde USING btree (endet);
		CREATE INDEX alkis_b01ea41d_e68d_4918_8270_686bc9f72f7c ON ax_gemeinde USING gin (istteilvon);

		-- ax_gemeindeteil.administrativefunktion => administrativefunktion: integer => integer[]
		ALTER TABLE ax_gemeindeteil ALTER administrativefunktion TYPE integer[] USING CASE WHEN administrativefunktion IS NULL THEN NULL ELSE ARRAY[administrativefunktion] END;
		ALTER TABLE ax_gemeindeteil ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gemeindeteil ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_gemeindeteil ALTER gemeinde SET NOT NULL;
		-- ax_gemeindeteil.gemeindeteil => gemeindeteil: integer => character varying
		ALTER TABLE ax_gemeindeteil ALTER gemeindeteil TYPE character varying USING gemeindeteil::varchar;
		ALTER TABLE ax_gemeindeteil ALTER kreis SET NOT NULL;
		ALTER TABLE ax_gemeindeteil ALTER land SET NOT NULL;
		ALTER TABLE ax_gemeindeteil ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_gemeindeteil ADD istteilvon character(16)[];
		ALTER TABLE ax_gemeindeteil ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gemeindeteil ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gemeindeteil ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2806019d_869e_4b2c_a4d5_b7240d6922a3 ON ax_gemeindeteil USING btree (endet);
		CREATE INDEX alkis_746bdc1a_bafc_4b0e_85cf_75121a9513e3 ON ax_gemeindeteil USING gin (istteilvon);

		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER gemeinde SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER hatauch SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER hausnummer SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER kreis SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER land SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER ortsteil SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER qualitaetsangaben SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER regierungsbezirk SET NOT NULL;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ALTER strassenschluessel SET NOT NULL;
		-- ax_georeferenziertegebaeudeadresse.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_georeferenziertegebaeudeadresse DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_georeferenziertegebaeudeadresse RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_georeferenziertegebaeudeadresse','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_georeferenziertegebaeudeadresse SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_georeferenziertegebaeudeadresse DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_georeferenziertegebaeudeadresse_wkb_geometry_idx ON ax_georeferenziertegebaeudeadresse USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD datensatznummer character varying;
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD hatdirektunten character(16)[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD istteilvon character(16)[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD traegtbeizu character(16)[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_georeferenziertegebaeudeadresse ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_6a7da82f_53be_4bcf_9040_d37b21f682ca ON ax_georeferenziertegebaeudeadresse USING btree (endet);
		CREATE INDEX alkis_4f405a2b_46f7_472f_920f_5dc3e48fa01a ON ax_georeferenziertegebaeudeadresse USING btree (hatauch);
		CREATE INDEX alkis_4d4fb4d2_c8eb_42fd_bd33_f2f2a5e88c3b ON ax_georeferenziertegebaeudeadresse USING gin (hatdirektunten);
		CREATE INDEX alkis_e9b60326_e972_49fd_828a_e9f0e2dee591 ON ax_georeferenziertegebaeudeadresse USING gin (istabgeleitetaus);
		CREATE INDEX alkis_f193fa1d_d0be_47c7_a046_3caa5b60b34f ON ax_georeferenziertegebaeudeadresse USING gin (istteilvon);
		CREATE INDEX alkis_4e38af04_b6cb_45f2_8d64_e5953f776ee8 ON ax_georeferenziertegebaeudeadresse USING gin (traegtbeizu);

		ALTER TABLE ax_gewaessermerkmal ALTER art SET NOT NULL;
		ALTER TABLE ax_gewaessermerkmal ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gewaessermerkmal ADD bezeichnung character varying;
		ALTER TABLE ax_gewaessermerkmal ADD hatdirektunten character(16)[];
		ALTER TABLE ax_gewaessermerkmal ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD hydrologischesmerkmal integer;
		ALTER TABLE ax_gewaessermerkmal ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_gewaessermerkmal ADD istteilvon character(16)[];
		ALTER TABLE ax_gewaessermerkmal ADD objekthoehe double precision;
		ALTER TABLE ax_gewaessermerkmal ADD processstep_address character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_datetime character(20)[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_individualname character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_organisationname character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_phone character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_positionname character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_rationale character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_role character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD statement character varying;
		ALTER TABLE ax_gewaessermerkmal ADD traegtbeizu character(16)[];
		ALTER TABLE ax_gewaessermerkmal ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gewaessermerkmal ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_0f8b3b81_6117_4684_95c9_a8659129edf5 ON ax_gewaessermerkmal USING btree (endet);
		CREATE INDEX alkis_2bcbc336_3e89_4196_9301_e1545e4805eb ON ax_gewaessermerkmal USING gin (hatdirektunten);
		CREATE INDEX alkis_503f4fd4_adcd_4b98_a09e_0100c99bcc6c ON ax_gewaessermerkmal USING gin (istabgeleitetaus);
		CREATE INDEX alkis_054f22db_8fe1_4714_9129_29f86aba6cd5 ON ax_gewaessermerkmal USING gin (istteilvon);
		CREATE INDEX alkis_67557914_4355_4bbe_a578_b5e2ba5ec40b ON ax_gewaessermerkmal USING gin (traegtbeizu);

		-- ax_gleis.bahnkategorie => bahnkategorie: integer => integer[]
		ALTER TABLE ax_gleis ALTER bahnkategorie TYPE integer[] USING CASE WHEN bahnkategorie IS NULL THEN NULL ELSE ARRAY[bahnkategorie] END;
		ALTER TABLE ax_gleis ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_gleis ADD hatdirektunten character(16)[];
		ALTER TABLE ax_gleis ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_gleis ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_gleis ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_gleis ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_gleis ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_gleis ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_gleis ADD istteilvon character(16)[];
		ALTER TABLE ax_gleis ADD processstep_address character varying[];
		ALTER TABLE ax_gleis ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_gleis ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_gleis ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_gleis ADD processstep_datetime character(20)[];
		ALTER TABLE ax_gleis ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_gleis ADD processstep_individualname character varying[];
		ALTER TABLE ax_gleis ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_gleis ADD processstep_organisationname character varying[];
		ALTER TABLE ax_gleis ADD processstep_phone character varying[];
		ALTER TABLE ax_gleis ADD processstep_positionname character varying[];
		ALTER TABLE ax_gleis ADD processstep_rationale character varying[];
		ALTER TABLE ax_gleis ADD processstep_role character varying[];
		ALTER TABLE ax_gleis ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_gleis ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_gleis ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_gleis ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_gleis ADD statement character varying;
		ALTER TABLE ax_gleis ADD traegtbeizu character(16)[];
		ALTER TABLE ax_gleis ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_gleis ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_gleis ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_df69597d_b2a8_417b_ac8d_8c71461dca45 ON ax_gleis USING btree (endet);
		CREATE INDEX alkis_a0ea9a4f_a0e0_49ec_8b04_002ed5edb8a1 ON ax_gleis USING gin (hatdirektunten);
		CREATE INDEX alkis_44120a9d_9fbe_4fbd_ba8a_c0988a84c9e4 ON ax_gleis USING gin (istabgeleitetaus);
		CREATE INDEX alkis_69b4a829_34f0_4fb7_a0fe_c3dcb14c5559 ON ax_gleis USING gin (istteilvon);
		CREATE INDEX alkis_9a270b76_b72c_41bf_bd98_440f92d9eab8 ON ax_gleis USING gin (traegtbeizu);

		ALTER TABLE ax_grablochderbodenschaetzung RENAME art TO zeigtaufexternes_art;
		-- ax_grablochderbodenschaetzung.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_grablochderbodenschaetzung ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		UPDATE ax_grablochderbodenschaetzung SET bedeutung=ARRAY[]::integer[] WHERE bedeutung IS NULL;
		ALTER TABLE ax_grablochderbodenschaetzung ALTER bedeutung SET NOT NULL;
		ALTER TABLE ax_grablochderbodenschaetzung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_grablochderbodenschaetzung RENAME name TO zeigtaufexternes_name;
		-- ax_grablochderbodenschaetzung.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_grablochderbodenschaetzung ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		UPDATE ax_grablochderbodenschaetzung SET nummerdesgrablochs='' WHERE nummerdesgrablochs='';
		ALTER TABLE ax_grablochderbodenschaetzung ALTER nummerdesgrablochs SET NOT NULL;
		ALTER TABLE ax_grablochderbodenschaetzung RENAME nummerierungsbezirk TO kennziffer_nummerierungsbezirk;
		-- ax_grablochderbodenschaetzung.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_grablochderbodenschaetzung DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_grablochderbodenschaetzung RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_grablochderbodenschaetzung','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_grablochderbodenschaetzung SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_grablochderbodenschaetzung DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_grablochderbodenschaetzung_wkb_geometry_idx ON ax_grablochderbodenschaetzung USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_grablochderbodenschaetzung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD ingemarkung_gemarkungsnummer character varying;
		ALTER TABLE ax_grablochderbodenschaetzung ADD ingemarkung_land character varying;
		ALTER TABLE ax_grablochderbodenschaetzung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD istteilvon character(16)[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD kennziffer_gemarkungsnummer character varying;
		ALTER TABLE ax_grablochderbodenschaetzung ADD kennziffer_land character varying;
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_address character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_individualname character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_phone character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_positionname character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_rationale character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_role character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD statement character varying;
		ALTER TABLE ax_grablochderbodenschaetzung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_grablochderbodenschaetzung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_5b2b6142_7abb_48ff_8d58_106393cd4241 ON ax_grablochderbodenschaetzung USING btree (endet);
		CREATE INDEX alkis_652a92aa_edb7_4f76_9c64_9dd2b45afb06 ON ax_grablochderbodenschaetzung USING btree (gehoertzu);
		CREATE INDEX alkis_7359ddc6_6160_479b_8bb0_b40ccf6d2b57 ON ax_grablochderbodenschaetzung USING gin (hatdirektunten);
		CREATE INDEX alkis_e9f0ba2b_4842_4ea3_817b_f01b61400c0d ON ax_grablochderbodenschaetzung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_c076f5f0_4cbb_4419_b14b_e58c568b20f2 ON ax_grablochderbodenschaetzung USING gin (istteilvon);
		CREATE INDEX alkis_170ea0ce_94ba_49a1_bc50_4b861c4e2648 ON ax_grablochderbodenschaetzung USING gin (traegtbeizu);

		ALTER TABLE ax_grenzpunkt ALTER abmarkung_marke SET NOT NULL;
		ALTER TABLE ax_grenzpunkt RENAME art TO zeigtaufexternes_art;
		-- ax_grenzpunkt.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_grenzpunkt ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_grenzpunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_grenzpunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_grenzpunkt RENAME name TO zeigtaufexternes_name;
		ALTER TABLE ax_grenzpunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_grenzpunkt ADD ausgesetzteabmarkung_land character varying;
		ALTER TABLE ax_grenzpunkt ADD ausgesetzteabmarkung_stelle character varying;
		ALTER TABLE ax_grenzpunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_grenzpunkt ADD beziehtsichaufsfp character(16)[];
		ALTER TABLE ax_grenzpunkt ADD gehoertzulfp character(16)[];
		ALTER TABLE ax_grenzpunkt ADD gruendederausgesetztenabmarkung integer;
		ALTER TABLE ax_grenzpunkt ADD horizontfreiheit integer;
		ALTER TABLE ax_grenzpunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_grenzpunkt ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_grenzpunkt ADD zwischenmarke character varying;
		CREATE INDEX alkis_a789f1d1_b7f0_4316_bbf7_124ee0b167ba ON ax_grenzpunkt USING btree (endet);
		CREATE INDEX alkis_8d8c9e94_83ef_4b69_b64e_3fdf276e891a ON ax_grenzpunkt USING gin (bestehtaus);
		CREATE INDEX alkis_76ca051f_269c_48a0_96d9_5cea5efaa5ad ON ax_grenzpunkt USING gin (beziehtsichaufsfp);
		CREATE INDEX alkis_f2d59f31_e2ea_4366_b531_d63097656111 ON ax_grenzpunkt USING gin (gehoertzulfp);
		CREATE INDEX alkis_4d2c815c_eec3_42ba_8029_10da23d51e73 ON ax_grenzpunkt USING gin (istteilvon);

		ALTER TABLE ax_hafenbecken ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_hafenbecken RENAME name TO zeigtaufexternes_name;
		-- ax_hafenbecken.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_hafenbecken ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_hafenbecken ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_hafenbecken ADD gemeinde character varying;
		ALTER TABLE ax_hafenbecken ADD hatdirektunten character(16)[];
		ALTER TABLE ax_hafenbecken ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_hafenbecken ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_hafenbecken ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_hafenbecken ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_hafenbecken ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_hafenbecken ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_hafenbecken ADD istteilvon character(16)[];
		ALTER TABLE ax_hafenbecken ADD kreis character varying;
		ALTER TABLE ax_hafenbecken ADD lage character varying;
		ALTER TABLE ax_hafenbecken ADD land character varying;
		ALTER TABLE ax_hafenbecken ADD processstep_address character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_datetime character(20)[];
		ALTER TABLE ax_hafenbecken ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_individualname character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_organisationname character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_phone character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_positionname character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_rationale character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_role character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_hafenbecken ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_hafenbecken ADD regierungsbezirk character varying;
		ALTER TABLE ax_hafenbecken ADD statement character varying;
		ALTER TABLE ax_hafenbecken ADD traegtbeizu character(16)[];
		ALTER TABLE ax_hafenbecken ADD unverschluesselt character varying;
		ALTER TABLE ax_hafenbecken ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_hafenbecken ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_6a1cdcfb_3ce9_4f09_9990_20154ac143d1 ON ax_hafenbecken USING btree (endet);
		CREATE INDEX alkis_ef5f5fb9_81a7_4676_83f3_efe99b1418fe ON ax_hafenbecken USING gin (hatdirektunten);
		CREATE INDEX alkis_dde1982b_9cd9_46fd_a914_36da84a0f083 ON ax_hafenbecken USING gin (istabgeleitetaus);
		CREATE INDEX alkis_4188d97a_1f7e_43eb_8d1b_b0233bf40a69 ON ax_hafenbecken USING gin (istteilvon);
		CREATE INDEX alkis_40fd7535_72d0_4643_8cc4_7da0fd24ba4b ON ax_hafenbecken USING gin (traegtbeizu);

		ALTER TABLE ax_halde ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_halde ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_halde ADD hatdirektunten character(16)[];
		ALTER TABLE ax_halde ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_halde ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_halde ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_halde ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_halde ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_halde ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_halde ADD istteilvon character(16)[];
		ALTER TABLE ax_halde ADD processstep_address character varying[];
		ALTER TABLE ax_halde ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_halde ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_halde ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_halde ADD processstep_datetime character(20)[];
		ALTER TABLE ax_halde ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_halde ADD processstep_individualname character varying[];
		ALTER TABLE ax_halde ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_halde ADD processstep_organisationname character varying[];
		ALTER TABLE ax_halde ADD processstep_phone character varying[];
		ALTER TABLE ax_halde ADD processstep_positionname character varying[];
		ALTER TABLE ax_halde ADD processstep_rationale character varying[];
		ALTER TABLE ax_halde ADD processstep_role character varying[];
		ALTER TABLE ax_halde ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_halde ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_halde ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_halde ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_halde ADD statement character varying;
		ALTER TABLE ax_halde ADD traegtbeizu character(16)[];
		ALTER TABLE ax_halde ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_halde ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_halde ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_17fe5a12_ed35_4473_b66c_f83a8cf86991 ON ax_halde USING btree (endet);
		CREATE INDEX alkis_69cf0daa_db19_4c14_ab44_9ff8e68300fd ON ax_halde USING gin (hatdirektunten);
		CREATE INDEX alkis_7388a385_fc73_41a4_b2dc_39849efeef0a ON ax_halde USING gin (istabgeleitetaus);
		CREATE INDEX alkis_6a9cfa1e_75cf_40b7_96ac_29fb60cf23b0 ON ax_halde USING gin (istteilvon);
		CREATE INDEX alkis_e7458aec_e719_4367_bbda_e284d3033877 ON ax_halde USING gin (traegtbeizu);

		ALTER TABLE ax_heide ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_heide ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_heide ADD hatdirektunten character(16)[];
		ALTER TABLE ax_heide ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_heide ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_heide ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_heide ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_heide ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_heide ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_heide ADD istteilvon character(16)[];
		ALTER TABLE ax_heide ADD processstep_address character varying[];
		ALTER TABLE ax_heide ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_heide ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_heide ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_heide ADD processstep_datetime character(20)[];
		ALTER TABLE ax_heide ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_heide ADD processstep_individualname character varying[];
		ALTER TABLE ax_heide ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_heide ADD processstep_organisationname character varying[];
		ALTER TABLE ax_heide ADD processstep_phone character varying[];
		ALTER TABLE ax_heide ADD processstep_positionname character varying[];
		ALTER TABLE ax_heide ADD processstep_rationale character varying[];
		ALTER TABLE ax_heide ADD processstep_role character varying[];
		ALTER TABLE ax_heide ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_heide ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_heide ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_heide ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_heide ADD statement character varying;
		ALTER TABLE ax_heide ADD traegtbeizu character(16)[];
		ALTER TABLE ax_heide ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_heide ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_heide ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e2be50f0_fc8e_452d_9648_1f2dbaeab41f ON ax_heide USING btree (endet);
		CREATE INDEX alkis_c2028b06_c80e_44f2_86d3_079f8f33b7a3 ON ax_heide USING gin (hatdirektunten);
		CREATE INDEX alkis_89bf0187_99ae_4ead_a0ef_16d307f4a028 ON ax_heide USING gin (istabgeleitetaus);
		CREATE INDEX alkis_22296889_7275_4fa1_b628_8daf53775c42 ON ax_heide USING gin (istteilvon);
		CREATE INDEX alkis_5ed93688_e3b8_46ca_9333_f741f26534f7 ON ax_heide USING gin (traegtbeizu);

		UPDATE ax_heilquellegasquelle SET art=-1 WHERE art IS NULL;
		ALTER TABLE ax_heilquellegasquelle ALTER art SET NOT NULL;
		ALTER TABLE ax_heilquellegasquelle ALTER beginnt SET NOT NULL;
		-- ax_heilquellegasquelle.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_heilquellegasquelle DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_heilquellegasquelle RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_heilquellegasquelle','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_heilquellegasquelle SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_heilquellegasquelle DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_heilquellegasquelle_wkb_geometry_idx ON ax_heilquellegasquelle USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_heilquellegasquelle ADD hatdirektunten character(16)[];
		ALTER TABLE ax_heilquellegasquelle ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD hydrologischesmerkmal integer;
		ALTER TABLE ax_heilquellegasquelle ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_heilquellegasquelle ADD istteilvon character(16)[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_address character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_datetime character(20)[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_individualname character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_organisationname character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_phone character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_positionname character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_rationale character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_role character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD statement character varying;
		ALTER TABLE ax_heilquellegasquelle ADD traegtbeizu character(16)[];
		ALTER TABLE ax_heilquellegasquelle ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_heilquellegasquelle ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_574253b4_e2e4_47e7_821f_ff9b721efff6 ON ax_heilquellegasquelle USING btree (endet);
		CREATE INDEX alkis_bc5620c9_be32_498b_b1d2_1ac0555337ad ON ax_heilquellegasquelle USING gin (hatdirektunten);
		CREATE INDEX alkis_bb4dc703_8b0e_4ab1_af96_9ea07a8d8159 ON ax_heilquellegasquelle USING gin (istabgeleitetaus);
		CREATE INDEX alkis_3e949eab_0036_48c4_9cec_e3511a7064b6 ON ax_heilquellegasquelle USING gin (istteilvon);
		CREATE INDEX alkis_92ed52f9_0e40_4fa4_b3ac_7a40f8e4a80a ON ax_heilquellegasquelle USING gin (traegtbeizu);

		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ALTER archaeologischertyp SET NOT NULL;
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD istteilvon character(16)[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_address character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_individualname character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_phone character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_positionname character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_rationale character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_role character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD statement character varying;
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_historischesbauwerkoderhistorischeeinrichtung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_c713ca44_47e5_4807_a556_f0619c55b775 ON ax_historischesbauwerkoderhistorischeeinrichtung USING btree (endet);
		CREATE INDEX alkis_7685acf3_5bf5_4687_a5c4_456ad669a364 ON ax_historischesbauwerkoderhistorischeeinrichtung USING gin (hatdirektunten);
		CREATE INDEX alkis_cd63491f_edff_498d_83cc_a5dd81b69cf1 ON ax_historischesbauwerkoderhistorischeeinrichtung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_8e73bf26_62f4_47b7_8aa8_d944b9772c9b ON ax_historischesbauwerkoderhistorischeeinrichtung USING gin (istteilvon);
		CREATE INDEX alkis_61daf965_a86a_46b6_be7b_9189ba11a892 ON ax_historischesbauwerkoderhistorischeeinrichtung USING gin (traegtbeizu);

		ALTER TABLE ax_historischesflurstueck ALTER amtlicheflaeche SET NOT NULL;
		ALTER TABLE ax_historischesflurstueck RENAME art TO zeigtaufexternes_art;
		ALTER TABLE ax_historischesflurstueck ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_historischesflurstueck RENAME bezirk TO buchungsblattbezirk_bezirk;
		-- ax_historischesflurstueck.bezirk => buchungsblattbezirk_bezirk: character varying => character varying[]
		ALTER TABLE ax_historischesflurstueck ALTER buchungsblattbezirk_bezirk TYPE character varying[] USING CASE WHEN buchungsblattbezirk_bezirk IS NULL THEN NULL ELSE ARRAY[buchungsblattbezirk_bezirk] END;
		-- ax_historischesflurstueck.blattart => blattart: integer => integer[]
		ALTER TABLE ax_historischesflurstueck ALTER blattart TYPE integer[] USING CASE WHEN blattart IS NULL THEN NULL ELSE ARRAY[blattart] END;
		-- ax_historischesflurstueck.buchungsart => buchungsart: character varying => character varying[]
		ALTER TABLE ax_historischesflurstueck ALTER buchungsart TYPE character varying[] USING CASE WHEN buchungsart IS NULL THEN NULL ELSE ARRAY[buchungsart] END;
		-- ax_historischesflurstueck.flurstueckskennzeichen => flurstueckskennzeichen: character(20) => character varying
		ALTER TABLE ax_historischesflurstueck ALTER flurstueckskennzeichen TYPE character varying;
		ALTER TABLE ax_historischesflurstueck ALTER flurstueckskennzeichen SET NOT NULL;
		ALTER TABLE ax_historischesflurstueck ALTER gemarkungsnummer SET NOT NULL;
		ALTER TABLE ax_historischesflurstueck RENAME gemeinde TO gemeindezugehoerigkeit_gemeinde;
		ALTER TABLE ax_historischesflurstueck RENAME kreis TO gemeindezugehoerigkeit_kreis;
		ALTER TABLE ax_historischesflurstueck ALTER land SET NOT NULL;
		-- ax_historischesflurstueck.laufendenummerderbuchungsstelle => laufendenummerderbuchungsstelle: character varying => character varying[]
		ALTER TABLE ax_historischesflurstueck ALTER laufendenummerderbuchungsstelle TYPE character varying[] USING CASE WHEN laufendenummerderbuchungsstelle IS NULL THEN NULL ELSE ARRAY[laufendenummerderbuchungsstelle] END;
		ALTER TABLE ax_historischesflurstueck RENAME name TO zeigtaufexternes_name;
		ALTER TABLE ax_historischesflurstueck RENAME regierungsbezirk TO gemeindezugehoerigkeit_regierungsbezirk;
		ALTER TABLE ax_historischesflurstueck DROP vorgaengerflurstueckskennzeichen;
		ALTER TABLE ax_historischesflurstueck ALTER zaehler SET NOT NULL;
		-- ax_historischesflurstueck.zeitpunktderentstehung => zeitpunktderentstehung: character(10) => date
		ALTER TABLE ax_historischesflurstueck ALTER zeitpunktderentstehung TYPE date USING to_date(zeitpunktderentstehung, 'YYYY-mm-dd');
		-- ax_historischesflurstueck.zeitpunktderhistorisierung => zeitpunktderhistorisierung: character(10) => date
		ALTER TABLE ax_historischesflurstueck ALTER zeitpunktderhistorisierung TYPE date USING to_date(zeitpunktderhistorisierung, 'YYYY-mm-dd');
		ALTER TABLE ax_historischesflurstueck ADD angabenzumabschnittbemerkung character varying[];
		ALTER TABLE ax_historischesflurstueck ADD angabenzumabschnittflurstueck character varying[];
		ALTER TABLE ax_historischesflurstueck ADD angabenzumabschnittnummeraktenzeichen character varying[];
		ALTER TABLE ax_historischesflurstueck ADD angabenzumabschnittstelle character varying[];
		ALTER TABLE ax_historischesflurstueck ADD buchungsblattbezirk_land character varying[];
		ALTER TABLE ax_historischesflurstueck ADD flaechedesabschnitts double precision[];
		ALTER TABLE ax_historischesflurstueck ADD flurstuecksfolge character varying;
		ALTER TABLE ax_historischesflurstueck ADD gemeindezugehoerigkeit_gemeindeteil character varying;
		ALTER TABLE ax_historischesflurstueck ADD gemeindezugehoerigkeit_land character varying;
		ALTER TABLE ax_historischesflurstueck ADD hatdirektunten character(16)[];
		ALTER TABLE ax_historischesflurstueck ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_historischesflurstueck ADD istteilvon character(16)[];
		ALTER TABLE ax_historischesflurstueck ADD kennungschluessel character varying[];
		PERFORM AddGeometryColumn('ax_historischesflurstueck','objektkoordinaten',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'POINT',2);
		ALTER TABLE ax_historischesflurstueck ADD traegtbeizu character(16)[];
		ALTER TABLE ax_historischesflurstueck ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2e76a0c0_6f40_4a62_8f6d_be5ef7fb08a4 ON ax_historischesflurstueck USING btree (endet);
		CREATE INDEX alkis_ee8a169d_9bda_4331_9542_1b147a2ae208 ON ax_historischesflurstueck USING gin (hatdirektunten);
		CREATE INDEX alkis_5fe471a0_5572_427e_82c5_04a3e916657f ON ax_historischesflurstueck USING gin (istabgeleitetaus);
		CREATE INDEX alkis_64957e28_b6af_4d7c_8b4a_3532973b613e ON ax_historischesflurstueck USING gin (istteilvon);
		CREATE INDEX alkis_4525c6f7_c122_4977_9aef_533777d6d25e ON ax_historischesflurstueck USING gin (traegtbeizu);
		CREATE INDEX alkis_d448bc6e_a481_4a59_a79c_b4d8ab2991d2 ON ax_historischesflurstueck USING gist (objektkoordinaten);

		ALTER TABLE ax_historischesflurstueckalb DROP abweichenderrechtszustand;
		ALTER TABLE ax_historischesflurstueckalb ALTER amtlicheflaeche SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckalb ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckalb RENAME bezirk TO buchungsblattbezirk_bezirk;
		-- ax_historischesflurstueckalb.bezirk => buchungsblattbezirk_bezirk: character varying => character varying[]
		ALTER TABLE ax_historischesflurstueckalb ALTER buchungsblattbezirk_bezirk TYPE character varying[] USING CASE WHEN buchungsblattbezirk_bezirk IS NULL THEN NULL ELSE ARRAY[buchungsblattbezirk_bezirk] END;
		-- ax_historischesflurstueckalb.blattart => blattart: integer => integer[]
		ALTER TABLE ax_historischesflurstueckalb ALTER blattart TYPE integer[] USING CASE WHEN blattart IS NULL THEN NULL ELSE ARRAY[blattart] END;
		-- ax_historischesflurstueckalb.flurstueckskennzeichen => flurstueckskennzeichen: character(20) => character varying
		ALTER TABLE ax_historischesflurstueckalb ALTER flurstueckskennzeichen TYPE character varying;
		ALTER TABLE ax_historischesflurstueckalb ALTER flurstueckskennzeichen SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckalb ALTER gemarkungsnummer SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckalb DROP gemeinde;
		ALTER TABLE ax_historischesflurstueckalb ALTER land SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckalb RENAME name TO zeigtaufexternes_name;
		ALTER TABLE ax_historischesflurstueckalb DROP rechtsbehelfsverfahren;
		-- ax_historischesflurstueckalb.zaehler => zaehler: integer => character varying
		ALTER TABLE ax_historischesflurstueckalb ALTER zaehler TYPE character varying USING zaehler::varchar;
		ALTER TABLE ax_historischesflurstueckalb ALTER zaehler SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckalb DROP zeitpunktderentstehung;
		-- ax_historischesflurstueckalb.zeitpunktderentstehungdesbezugsflurstuecks => zeitpunktderentstehungdesbezugsflurstuecks: character varying => date
		ALTER TABLE ax_historischesflurstueckalb ALTER zeitpunktderentstehungdesbezugsflurstuecks TYPE date USING to_date(zeitpunktderentstehungdesbezugsflurstuecks, 'YYYY-mm-dd');
		ALTER TABLE ax_historischesflurstueckalb DROP zweifelhafterflurstuecksnachweis;
		ALTER TABLE ax_historischesflurstueckalb ADD buchungsblattbezirk_land character varying[];
		ALTER TABLE ax_historischesflurstueckalb ADD istteilvon character(16)[];
		PERFORM AddGeometryColumn('ax_historischesflurstueckalb','objektkoordinaten',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'POINT',2);
		ALTER TABLE ax_historischesflurstueckalb ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_historischesflurstueckalb ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_historischesflurstueckalb' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_historischesflurstueckalb USING btree (nachfolgerflurstueckskennzeichen)';
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_historischesflurstueckalb' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_historischesflurstueckalb USING btree (vorgaengerflurstueckskennzeichen)';
		CREATE INDEX alkis_21db3780_12d8_48f1_8e13_f7b893e1dacc ON ax_historischesflurstueckalb USING btree (endet);
		CREATE INDEX alkis_b80f6956_7814_44c9_980e_45b8fa85bb23 ON ax_historischesflurstueckalb USING btree (flurstueckskennzeichen);
		CREATE INDEX alkis_4ae16a49_f4cb_46ad_be1a_62db75c6981d ON ax_historischesflurstueckalb USING gin (istteilvon);
		CREATE INDEX alkis_b49146d2_be2c_4c28_a45e_f1ac1b1983f6 ON ax_historischesflurstueckalb USING gin (nachfolgerflurstueckskennzeichen);
		CREATE INDEX alkis_10099ea3_e629_4a56_9b2c_3df8916327d8 ON ax_historischesflurstueckalb USING gin (vorgaengerflurstueckskennzeichen);
		CREATE INDEX alkis_626ef53e_3f57_48b3_b87f_a70eedc796a8 ON ax_historischesflurstueckalb USING gist (objektkoordinaten);

		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER amtlicheflaeche SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER beginnt SET NOT NULL;
		-- ax_historischesflurstueckohneraumbezug.flurstueckskennzeichen => flurstueckskennzeichen: character(20) => character varying
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER flurstueckskennzeichen TYPE character varying;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER flurstueckskennzeichen SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER gemarkungsnummer SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckohneraumbezug RENAME gemeinde TO gemeindezugehoerigkeit_gemeinde;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER land SET NOT NULL;
		ALTER TABLE ax_historischesflurstueckohneraumbezug RENAME name TO zeigtaufexternes_name;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER zaehler SET NOT NULL;
		-- ax_historischesflurstueckohneraumbezug.zeitpunktderentstehung => zeitpunktderentstehung: character varying => date
		ALTER TABLE ax_historischesflurstueckohneraumbezug ALTER zeitpunktderentstehung TYPE date USING to_date(zeitpunktderentstehung, 'YYYY-mm-dd');
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD angabenzumabschnittbemerkung character varying[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD angabenzumabschnittflurstueck character varying[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD angabenzumabschnittnummeraktenzeichen character varying[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD angabenzumabschnittstelle character varying[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD flaechedesabschnitts double precision[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD flurstuecksfolge character varying;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD gemeindezugehoerigkeit_gemeindeteil character varying;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD gemeindezugehoerigkeit_kreis character varying;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD gemeindezugehoerigkeit_land character varying;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD gemeindezugehoerigkeit_regierungsbezirk character varying;
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD istteilvon character(16)[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD kennungschluessel character varying[];
		PERFORM AddGeometryColumn('ax_historischesflurstueckohneraumbezug','objektkoordinaten',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'POINT',2);
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_historischesflurstueckohneraumbezug ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_historischesflurstueckohneraumbezug' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_historischesflurstueckohneraumbezug USING btree (nachfolgerflurstueckskennzeichen)';
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_historischesflurstueckohneraumbezug' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_historischesflurstueckohneraumbezug USING btree (vorgaengerflurstueckskennzeichen)';
		CREATE INDEX alkis_52405507_de46_42fb_86da_44c0a630f3b9 ON ax_historischesflurstueckohneraumbezug USING btree (endet);
		CREATE INDEX alkis_a13c748f_cb0a_4327_a59e_e5b12a707285 ON ax_historischesflurstueckohneraumbezug USING btree (gml_id, beginnt);
		CREATE INDEX alkis_d3ae2a2b_b690_468e_ac81_25ec6bf9c808 ON ax_historischesflurstueckohneraumbezug USING gin (istteilvon);
		CREATE INDEX alkis_68bd32fd_f199_4f7b_8c9e_ab1690c3c8cd ON ax_historischesflurstueckohneraumbezug USING gin (nachfolgerflurstueckskennzeichen);
		CREATE INDEX alkis_ead6e84b_eb1e_481a_a60b_cf86beba8404 ON ax_historischesflurstueckohneraumbezug USING gin (vorgaengerflurstueckskennzeichen);
		CREATE INDEX alkis_895edadb_aa82_41cd_b6f4_e568925beb4e ON ax_historischesflurstueckohneraumbezug USING gist (objektkoordinaten);

		ALTER TABLE ax_hoehenlinie ALTER beginnt SET NOT NULL;
		UPDATE ax_hoehenlinie SET hoehevonhoehenlinie=0.0 WHERE hoehevonhoehenlinie IS NULL;
		ALTER TABLE ax_hoehenlinie ALTER hoehevonhoehenlinie SET NOT NULL;
		-- ax_hoehenlinie.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_hoehenlinie DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_hoehenlinie RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_hoehenlinie','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_hoehenlinie SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_hoehenlinie DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_hoehenlinie_wkb_geometry_idx ON ax_hoehenlinie USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_hoehenlinie ADD hatdirektunten character(16)[];
		ALTER TABLE ax_hoehenlinie ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_hoehenlinie ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_hoehenlinie ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_hoehenlinie ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_hoehenlinie ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_hoehenlinie ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_hoehenlinie ADD istteilvon character(16)[];
		ALTER TABLE ax_hoehenlinie ADD processstep_address character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_datetime character(20)[];
		ALTER TABLE ax_hoehenlinie ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_individualname character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_organisationname character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_phone character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_positionname character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_rationale character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_role character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_hoehenlinie ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_hoehenlinie ADD statement character varying;
		ALTER TABLE ax_hoehenlinie ADD traegtbeizu character(16)[];
		ALTER TABLE ax_hoehenlinie ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_hoehenlinie ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_hoehenlinie ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_f3492c5c_8c1e_43a2_ad96_a0ddd4fd2b92 ON ax_hoehenlinie USING btree (endet);
		CREATE INDEX alkis_4f4ca521_321a_405a_a0a4_2602c7a9b527 ON ax_hoehenlinie USING gin (hatdirektunten);
		CREATE INDEX alkis_5b278176_3b87_417c_ae5d_8c8df0c622ef ON ax_hoehenlinie USING gin (istabgeleitetaus);
		CREATE INDEX alkis_395cd9ae_a8af_49ec_b98d_bbf2dd4ab556 ON ax_hoehenlinie USING gin (istteilvon);
		CREATE INDEX alkis_6fcd3afe_b777_42b6_83f4_e352b7d0153b ON ax_hoehenlinie USING gin (traegtbeizu);

		ALTER TABLE ax_hoehleneingang ALTER beginnt SET NOT NULL;
		-- ax_hoehleneingang.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_hoehleneingang DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_hoehleneingang RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_hoehleneingang','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_hoehleneingang SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_hoehleneingang DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_hoehleneingang_wkb_geometry_idx ON ax_hoehleneingang USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_hoehleneingang ADD hatdirektunten character(16)[];
		ALTER TABLE ax_hoehleneingang ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_hoehleneingang ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_hoehleneingang ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_hoehleneingang ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_hoehleneingang ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_hoehleneingang ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_hoehleneingang ADD istteilvon character(16)[];
		ALTER TABLE ax_hoehleneingang ADD processstep_address character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_datetime character(20)[];
		ALTER TABLE ax_hoehleneingang ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_individualname character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_organisationname character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_phone character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_positionname character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_rationale character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_role character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_hoehleneingang ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_hoehleneingang ADD statement character varying;
		ALTER TABLE ax_hoehleneingang ADD traegtbeizu character(16)[];
		ALTER TABLE ax_hoehleneingang ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_hoehleneingang ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_hoehleneingang ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_hoehleneingang ADD zustand integer;
		CREATE INDEX alkis_fd75254d_1a1f_4c06_8ee7_2b7a4e0cca80 ON ax_hoehleneingang USING btree (endet);
		CREATE INDEX alkis_e343c236_fc7f_49cf_8768_0715dd4311f3 ON ax_hoehleneingang USING gin (hatdirektunten);
		CREATE INDEX alkis_dfc6aaba_7f66_4432_ad51_61c1c16d5e4b ON ax_hoehleneingang USING gin (istabgeleitetaus);
		CREATE INDEX alkis_0e012b3b_a2f1_4d54_a554_2b25b3d86787 ON ax_hoehleneingang USING gin (istteilvon);
		CREATE INDEX alkis_aac50656_1b06_451d_bef1_a91030a92847 ON ax_hoehleneingang USING gin (traegtbeizu);

		ALTER TABLE ax_industrieundgewerbeflaeche ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_industrieundgewerbeflaeche ADD bezeichnung character varying;
		ALTER TABLE ax_industrieundgewerbeflaeche ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_industrieundgewerbeflaeche ADD hatdirektunten character(16)[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD istteilvon character(16)[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_address character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_datetime character(20)[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_individualname character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_organisationname character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_phone character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_positionname character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_rationale character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_role character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD statement character varying;
		ALTER TABLE ax_industrieundgewerbeflaeche ADD traegtbeizu character(16)[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_industrieundgewerbeflaeche ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_30ddf2c0_b7d0_408f_b29f_8b5713721cff ON ax_industrieundgewerbeflaeche USING btree (endet);
		CREATE INDEX alkis_2561ffa4_4672_4647_b3f3_305ae7e45f36 ON ax_industrieundgewerbeflaeche USING gin (hatdirektunten);
		CREATE INDEX alkis_c915450c_f6a1_4e87_a777_9d3beb81d325 ON ax_industrieundgewerbeflaeche USING gin (istabgeleitetaus);
		CREATE INDEX alkis_6ea74ee3_5936_4298_ba61_a0a8e63a724a ON ax_industrieundgewerbeflaeche USING gin (istteilvon);
		CREATE INDEX alkis_4dfc8869_285c_463b_b8ce_c2340b380023 ON ax_industrieundgewerbeflaeche USING gin (traegtbeizu);

		ALTER TABLE ax_klassifizierungnachstrassenrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_klassifizierungnachstrassenrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_address character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_role character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD statement character varying;
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_klassifizierungnachstrassenrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_3cebb3a0_eb04_4707_b379_999224ee807e ON ax_klassifizierungnachstrassenrecht USING btree (endet);
		CREATE INDEX alkis_03800c22_e4d6_4abe_b7ec_5c5938bc02cc ON ax_klassifizierungnachstrassenrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_dcdbf8e8_d1a6_4b77_afed_225b671f4a87 ON ax_klassifizierungnachstrassenrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_c683a542_a7c0_443f_b42b_b1db81a262f4 ON ax_klassifizierungnachstrassenrecht USING gin (istteilvon);
		CREATE INDEX alkis_f00113db_f002_40c4_a835_93a99ef8e47e ON ax_klassifizierungnachstrassenrecht USING gin (traegtbeizu);

		ALTER TABLE ax_klassifizierungnachwasserrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_klassifizierungnachwasserrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD bezeichnung character varying;
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_address character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_role character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD statement character varying;
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_klassifizierungnachwasserrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_9aeb5a89_93d0_45a9_8eaa_fc9357182830 ON ax_klassifizierungnachwasserrecht USING btree (endet);
		CREATE INDEX alkis_2d8bbf9d_d3df_47d4_a76a_7f0f967d46ba ON ax_klassifizierungnachwasserrecht USING btree (gml_id, beginnt);
		CREATE INDEX alkis_0e8751e9_6d74_4e1e_8a68_eca131a07fc6 ON ax_klassifizierungnachwasserrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_5cbb09c3_507f_4978_953f_933c2e5905ff ON ax_klassifizierungnachwasserrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_255d83d1_aad0_45ea_b30c_63f274328712 ON ax_klassifizierungnachwasserrecht USING gin (istteilvon);
		CREATE INDEX alkis_f599f774_0855_48f8_a47f_20f56c1bc592 ON ax_klassifizierungnachwasserrecht USING gin (traegtbeizu);

		ALTER TABLE ax_kleinraeumigerlandschaftsteil ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ALTER landschaftstyp SET NOT NULL;
		UPDATE ax_kleinraeumigerlandschaftsteil SET name='' WHERE name IS NULL;
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ALTER name SET NOT NULL;
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD hatdirektunten character(16)[];
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD istteilvon character(16)[];
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD traegtbeizu character(16)[];
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_kleinraeumigerlandschaftsteil ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_aaa44559_4549_4a5f_ac62_7dc9d79c6e2d ON ax_kleinraeumigerlandschaftsteil USING btree (endet);
		CREATE INDEX alkis_60e5db53_8409_41a8_9709_469c74a2fa61 ON ax_kleinraeumigerlandschaftsteil USING gin (hatdirektunten);
		CREATE INDEX alkis_5d2837e6_6e03_442a_86f6_288c89be29a2 ON ax_kleinraeumigerlandschaftsteil USING gin (istabgeleitetaus);
		CREATE INDEX alkis_f082c6b7_bb50_49d2_8ad3_48b2b5aa8cdd ON ax_kleinraeumigerlandschaftsteil USING gin (istteilvon);
		CREATE INDEX alkis_81ebf052_364a_451f_a930_c8d4d7593aac ON ax_kleinraeumigerlandschaftsteil USING gin (traegtbeizu);

		ALTER TABLE ax_kommunalesgebiet ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_kommunalesgebiet ALTER gemeinde SET NOT NULL;
		ALTER TABLE ax_kommunalesgebiet ALTER kreis SET NOT NULL;
		ALTER TABLE ax_kommunalesgebiet ALTER land SET NOT NULL;
		ALTER TABLE ax_kommunalesgebiet ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_kommunalesgebiet ADD gemeindeteil character varying;
		ALTER TABLE ax_kommunalesgebiet ADD hatdirektunten character(16)[];
		ALTER TABLE ax_kommunalesgebiet ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_kommunalesgebiet ADD istteilvon character(16)[];
		ALTER TABLE ax_kommunalesgebiet ADD traegtbeizu character(16)[];
		ALTER TABLE ax_kommunalesgebiet ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_kommunalesgebiet ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_kommunalesgebiet ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_1cfddb27_ab66_4adf_bf2d_a18817ba05c9 ON ax_kommunalesgebiet USING btree (endet);
		CREATE INDEX alkis_f72ba4ef_31e5_48bc_b9ed_555eb354926c ON ax_kommunalesgebiet USING gin (hatdirektunten);
		CREATE INDEX alkis_1fba25e6_844f_4f82_879f_8fe8c7674d26 ON ax_kommunalesgebiet USING gin (istabgeleitetaus);
		CREATE INDEX alkis_e476e65b_3c84_4986_bb35_473a5cf4f855 ON ax_kommunalesgebiet USING gin (istteilvon);
		CREATE INDEX alkis_6976efd3_8be9_47ac_8abc_87aeb5b9eb52 ON ax_kommunalesgebiet USING gin (traegtbeizu);

		ALTER TABLE ax_kreisregion ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_kreisregion ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_kreisregion ALTER kreis SET NOT NULL;
		ALTER TABLE ax_kreisregion ALTER land SET NOT NULL;
		ALTER TABLE ax_kreisregion ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_kreisregion ADD administrativefunktion integer[];
		ALTER TABLE ax_kreisregion ADD istamtsbezirkvon_land character varying[];
		ALTER TABLE ax_kreisregion ADD istamtsbezirkvon_stelle character varying[];
		ALTER TABLE ax_kreisregion ADD istteilvon character(16)[];
		ALTER TABLE ax_kreisregion ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_kreisregion ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_kreisregion ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_b05dab96_50d6_4f12_ab63_79a14847acd0 ON ax_kreisregion USING btree (endet);
		CREATE INDEX alkis_b9e0fded_55c1_469c_9df7_ded942784245 ON ax_kreisregion USING gin (istteilvon);

		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER gemeinde SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER kreis SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER lage SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER land SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ADD administrativefunktion integer[];
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ADD istteilvon character(16)[];
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ADD kennung character varying;
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_lagebezeichnungkatalogeintrag ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_a32d8702_8f6f_4f75_aa0e_0bbc8e17527d ON ax_lagebezeichnungkatalogeintrag USING btree (endet);
		CREATE INDEX alkis_955990c3_7604_4265_9dc7_c9dceecc0889 ON ax_lagebezeichnungkatalogeintrag USING gin (istteilvon);

		ALTER TABLE ax_lagebezeichnungmithausnummer ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungmithausnummer ALTER hausnummer SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungmithausnummer ADD istteilvon character(16)[];
		ALTER TABLE ax_lagebezeichnungmithausnummer ADD ortsteil character varying;
		ALTER TABLE ax_lagebezeichnungmithausnummer ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_lagebezeichnungmithausnummer ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_lagebezeichnungmithausnummer ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_fab89e96_865d_4b1d_aeda_892b9be50b01 ON ax_lagebezeichnungmithausnummer USING btree (endet);
		CREATE INDEX alkis_46949710_71d5_423a_b3eb_0ab971bf32b0 ON ax_lagebezeichnungmithausnummer USING gin (istteilvon);

		ALTER TABLE ax_lagebezeichnungmitpseudonummer ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungmitpseudonummer ALTER pseudonummer SET NOT NULL;
		ALTER TABLE ax_lagebezeichnungmitpseudonummer ADD istteilvon character(16)[];
		ALTER TABLE ax_lagebezeichnungmitpseudonummer ADD ortsteil character varying;
		ALTER TABLE ax_lagebezeichnungmitpseudonummer ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_lagebezeichnungmitpseudonummer ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_lagebezeichnungmitpseudonummer ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_bcb24e0a_b08d_49b3_816d_5a53d654799f ON ax_lagebezeichnungmitpseudonummer USING btree (endet);
		CREATE INDEX alkis_fb50f465_d0e1_4569_886f_15ece7313d6c ON ax_lagebezeichnungmitpseudonummer USING gin (istteilvon);

		ALTER TABLE ax_lagebezeichnungohnehausnummer ALTER beginnt SET NOT NULL;
		-- ax_lagebezeichnungohnehausnummer.gehoertzu => gehoertzu: character varying[] => character(16)[]
		ALTER TABLE ax_lagebezeichnungohnehausnummer ALTER gehoertzu TYPE character(16)[];
		ALTER TABLE ax_lagebezeichnungohnehausnummer ADD istteilvon character(16)[];
		ALTER TABLE ax_lagebezeichnungohnehausnummer ADD ortsteil character varying;
		ALTER TABLE ax_lagebezeichnungohnehausnummer ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_lagebezeichnungohnehausnummer ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_lagebezeichnungohnehausnummer ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_9c0d232d_ff0e_4f0f_bbee_b5bcb6e18d95 ON ax_lagebezeichnungohnehausnummer USING btree (endet);
		CREATE INDEX alkis_7edcc632_43ed_4d1f_884d_f0517772f66e ON ax_lagebezeichnungohnehausnummer USING gin (istteilvon);

		ALTER TABLE ax_landschaft ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_landschaft ALTER landschaftstyp SET NOT NULL;
		ALTER TABLE ax_landschaft ALTER name SET NOT NULL;
		ALTER TABLE ax_landschaft ADD hatdirektunten character(16)[];
		ALTER TABLE ax_landschaft ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_landschaft ADD istteilvon character(16)[];
		ALTER TABLE ax_landschaft ADD traegtbeizu character(16)[];
		ALTER TABLE ax_landschaft ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_landschaft ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_landschaft ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_3cbdf208_321c_4f34_8cfb_4071f3aec31e ON ax_landschaft USING btree (endet);
		CREATE INDEX alkis_b24755e6_a148_4704_9849_57b6ed734953 ON ax_landschaft USING gin (hatdirektunten);
		CREATE INDEX alkis_f6d2e868_a46e_4030_a9dc_8f2f5cbbbf89 ON ax_landschaft USING gin (istabgeleitetaus);
		CREATE INDEX alkis_789242f5_865b_481e_bbe0_5bfc4c20ee4e ON ax_landschaft USING gin (istteilvon);
		CREATE INDEX alkis_88ed0571_435e_4764_9a71_3aa6fa56b0bd ON ax_landschaft USING gin (traegtbeizu);

		ALTER TABLE ax_landwirtschaft ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_landwirtschaft ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_landwirtschaft ADD hatdirektunten character(16)[];
		ALTER TABLE ax_landwirtschaft ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_landwirtschaft ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_landwirtschaft ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_landwirtschaft ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_landwirtschaft ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_landwirtschaft ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_landwirtschaft ADD istteilvon character(16)[];
		ALTER TABLE ax_landwirtschaft ADD processstep_address character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_datetime character(20)[];
		ALTER TABLE ax_landwirtschaft ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_individualname character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_organisationname character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_phone character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_positionname character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_rationale character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_role character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_landwirtschaft ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_landwirtschaft ADD statement character varying;
		ALTER TABLE ax_landwirtschaft ADD traegtbeizu character(16)[];
		ALTER TABLE ax_landwirtschaft ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_landwirtschaft ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_landwirtschaft ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_c1ab5c1e_edd9_4a7a_b74e_b37f37dbf28a ON ax_landwirtschaft USING btree (endet);
		CREATE INDEX alkis_f269a606_a42b_44ba_bf33_aa77822302f1 ON ax_landwirtschaft USING gin (hatdirektunten);
		CREATE INDEX alkis_7d57e7a8_60f9_47eb_9251_471534724675 ON ax_landwirtschaft USING gin (istabgeleitetaus);
		CREATE INDEX alkis_a533bc3e_9301_4014_92a5_971b456cad08 ON ax_landwirtschaft USING gin (istteilvon);
		CREATE INDEX alkis_86f68f6d_17ba_4911_8bba_40dcfc679ff6 ON ax_landwirtschaft USING gin (traegtbeizu);

		ALTER TABLE ax_leitung ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_leitung ALTER beginnt SET NOT NULL;
		-- ax_leitung.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_leitung DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_leitung RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_leitung','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_leitung SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_leitung DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_leitung_wkb_geometry_idx ON ax_leitung USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_leitung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_leitung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_leitung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_leitung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_leitung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_leitung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_leitung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_leitung ADD istteilvon character(16)[];
		ALTER TABLE ax_leitung ADD name character varying;
		ALTER TABLE ax_leitung ADD processstep_address character varying[];
		ALTER TABLE ax_leitung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_leitung ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_leitung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_leitung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_leitung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_leitung ADD processstep_individualname character varying[];
		ALTER TABLE ax_leitung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_leitung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_leitung ADD processstep_phone character varying[];
		ALTER TABLE ax_leitung ADD processstep_positionname character varying[];
		ALTER TABLE ax_leitung ADD processstep_rationale character varying[];
		ALTER TABLE ax_leitung ADD processstep_role character varying[];
		ALTER TABLE ax_leitung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_leitung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_leitung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_leitung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_leitung ADD statement character varying;
		ALTER TABLE ax_leitung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_leitung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_leitung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_leitung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2a4c1617_2135_41ee_b20d_48b9f995da7a ON ax_leitung USING btree (endet);
		CREATE INDEX alkis_4c9de9ae_ec2a_4462_a425_3d77faeeada0 ON ax_leitung USING gin (hatdirektunten);
		CREATE INDEX alkis_cc932a19_f766_441d_a864_46de46348136 ON ax_leitung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_94c765cd_ffe0_4c71_b871_7e66f170a1fb ON ax_leitung USING gin (istteilvon);
		CREATE INDEX alkis_a56ad157_0b34_4e64_95fc_e9190ec9b989 ON ax_leitung USING gin (traegtbeizu);

		ALTER TABLE ax_meer ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_meer RENAME name TO zeigtaufexternes_name;
		-- ax_meer.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_meer ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_meer ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_meer ADD gemeinde character varying;
		ALTER TABLE ax_meer ADD hatdirektunten character(16)[];
		ALTER TABLE ax_meer ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_meer ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_meer ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_meer ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_meer ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_meer ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_meer ADD istteilvon character(16)[];
		ALTER TABLE ax_meer ADD kreis character varying;
		ALTER TABLE ax_meer ADD lage character varying;
		ALTER TABLE ax_meer ADD land character varying;
		ALTER TABLE ax_meer ADD processstep_address character varying[];
		ALTER TABLE ax_meer ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_meer ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_meer ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_meer ADD processstep_datetime character(20)[];
		ALTER TABLE ax_meer ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_meer ADD processstep_individualname character varying[];
		ALTER TABLE ax_meer ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_meer ADD processstep_organisationname character varying[];
		ALTER TABLE ax_meer ADD processstep_phone character varying[];
		ALTER TABLE ax_meer ADD processstep_positionname character varying[];
		ALTER TABLE ax_meer ADD processstep_rationale character varying[];
		ALTER TABLE ax_meer ADD processstep_role character varying[];
		ALTER TABLE ax_meer ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_meer ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_meer ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_meer ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_meer ADD regierungsbezirk character varying;
		ALTER TABLE ax_meer ADD statement character varying;
		ALTER TABLE ax_meer ADD traegtbeizu character(16)[];
		ALTER TABLE ax_meer ADD unverschluesselt character varying;
		ALTER TABLE ax_meer ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_meer ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_meer ADD zweitname character varying[];
		CREATE INDEX alkis_4543998c_970d_42f0_88a0_7be8859921e9 ON ax_meer USING btree (endet);
		CREATE INDEX alkis_17243af4_b1fc_4fa3_aa0a_23864bab0d12 ON ax_meer USING gin (hatdirektunten);
		CREATE INDEX alkis_df56a5aa_af33_43a6_8eee_d1751f2c3921 ON ax_meer USING gin (istabgeleitetaus);
		CREATE INDEX alkis_ee41b43a_62cf_426c_adc3_e0cd5a61f815 ON ax_meer USING gin (istteilvon);
		CREATE INDEX alkis_15e1a2dd_e814_476a_bd3b_e07c80d5ba75 ON ax_meer USING gin (traegtbeizu);

		ALTER TABLE ax_moor ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_moor ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_moor ADD hatdirektunten character(16)[];
		ALTER TABLE ax_moor ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_moor ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_moor ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_moor ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_moor ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_moor ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_moor ADD istteilvon character(16)[];
		ALTER TABLE ax_moor ADD processstep_address character varying[];
		ALTER TABLE ax_moor ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_moor ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_moor ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_moor ADD processstep_datetime character(20)[];
		ALTER TABLE ax_moor ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_moor ADD processstep_individualname character varying[];
		ALTER TABLE ax_moor ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_moor ADD processstep_organisationname character varying[];
		ALTER TABLE ax_moor ADD processstep_phone character varying[];
		ALTER TABLE ax_moor ADD processstep_positionname character varying[];
		ALTER TABLE ax_moor ADD processstep_rationale character varying[];
		ALTER TABLE ax_moor ADD processstep_role character varying[];
		ALTER TABLE ax_moor ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_moor ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_moor ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_moor ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_moor ADD statement character varying;
		ALTER TABLE ax_moor ADD traegtbeizu character(16)[];
		ALTER TABLE ax_moor ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_moor ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_moor ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e3d6066c_9898_47ab_b5e0_2f99adecafab ON ax_moor USING btree (endet);
		CREATE INDEX alkis_e6fade99_823a_42f0_ad32_3224b2cfdf1e ON ax_moor USING gin (hatdirektunten);
		CREATE INDEX alkis_3bc327e4_ad4c_4c0a_a61d_4cc3d4e50cd8 ON ax_moor USING gin (istabgeleitetaus);
		CREATE INDEX alkis_c0189944_b6be_4713_9a90_a04bb3d84455 ON ax_moor USING gin (istteilvon);
		CREATE INDEX alkis_1f975089_6687_465f_b2d1_391a4590c3ae ON ax_moor USING gin (traegtbeizu);

		ALTER TABLE ax_musterlandesmusterundvergleichsstueck RENAME art TO zeigtaufexternes_art;
		-- ax_musterlandesmusterundvergleichsstueck.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ALTER beginnt SET NOT NULL;
		-- ax_musterlandesmusterundvergleichsstueck.entstehungsartoderklimastufewasserverhaeltnisse => entstehungsartoderklimastufewasserverhaeltnisse: integer => integer[]
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ALTER entstehungsartoderklimastufewasserverhaeltnisse TYPE integer[] USING CASE WHEN entstehungsartoderklimastufewasserverhaeltnisse IS NULL THEN NULL ELSE ARRAY[entstehungsartoderklimastufewasserverhaeltnisse] END;
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ALTER merkmal SET NOT NULL;
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck RENAME name TO zeigtaufexternes_name;
		-- ax_musterlandesmusterundvergleichsstueck.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		-- ax_musterlandesmusterundvergleichsstueck.sonstigeangaben => sonstigeangaben: integer => integer[]
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ALTER sonstigeangaben TYPE integer[] USING CASE WHEN sonstigeangaben IS NULL THEN NULL ELSE ARRAY[sonstigeangaben] END;
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD hatdirektunten character(16)[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD istteilvon character(16)[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_address character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_datetime character(20)[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_individualname character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_organisationname character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_phone character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_positionname character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_rationale character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_role character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD statement character varying;
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD traegtbeizu character(16)[];
		ALTER TABLE ax_musterlandesmusterundvergleichsstueck ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_103f4faa_e89a_433b_98e8_bb05fb0b191d ON ax_musterlandesmusterundvergleichsstueck USING btree (endet);
		CREATE INDEX alkis_cc96c3cf_cd2c_4794_8738_58a6d7d43b8b ON ax_musterlandesmusterundvergleichsstueck USING gin (hatdirektunten);
		CREATE INDEX alkis_40cb070a_1444_44e9_a67d_ec4df1beb208 ON ax_musterlandesmusterundvergleichsstueck USING gin (istabgeleitetaus);
		CREATE INDEX alkis_00b5d3cc_156e_4959_b163_3da418f07300 ON ax_musterlandesmusterundvergleichsstueck USING gin (istteilvon);
		CREATE INDEX alkis_abc9c642_10c9_4454_be3b_1e7d988da253 ON ax_musterlandesmusterundvergleichsstueck USING gin (traegtbeizu);

		ALTER TABLE ax_namensnummer ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_namensnummer ALTER istbestandteilvon SET NOT NULL;
		-- ax_namensnummer.laufendenummernachdin1421 => laufendenummernachdin1421: character(16) => character varying
		ALTER TABLE ax_namensnummer ALTER laufendenummernachdin1421 TYPE character varying;
		ALTER TABLE ax_namensnummer ADD istteilvon character(16)[];
		ALTER TABLE ax_namensnummer ADD strichblattnummer integer;
		ALTER TABLE ax_namensnummer ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_namensnummer ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_namensnummer ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e25bcb78_4520_4e13_bba2_c3f11b5a3a61 ON ax_namensnummer USING btree (endet);
		CREATE INDEX alkis_22eb0d2c_a5ba_4d53_beca_b406b28c04de ON ax_namensnummer USING gin (istteilvon);

		ALTER TABLE ax_naturumweltoderbodenschutzrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD bezeichnung character varying;
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_address character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_role character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD statement character varying;
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_naturumweltoderbodenschutzrecht ADD zustand integer;
		CREATE INDEX alkis_4ae7e4cd_bda8_4c7b_a108_750e249b41cd ON ax_naturumweltoderbodenschutzrecht USING btree (endet);
		CREATE INDEX alkis_1cabbd81_1099_4b10_88a2_a966f5d608f4 ON ax_naturumweltoderbodenschutzrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_68f0036f_b644_423f_8c4d_aa341659d00b ON ax_naturumweltoderbodenschutzrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_6920bb42_2447_4402_96da_7c5f6db1935a ON ax_naturumweltoderbodenschutzrecht USING gin (istteilvon);
		CREATE INDEX alkis_f946250d_a834_4b6a_9630_85b1b06b1810 ON ax_naturumweltoderbodenschutzrecht USING gin (traegtbeizu);

		ALTER TABLE ax_person ALTER beginnt SET NOT NULL;
		-- ax_person.geburtsdatum => geburtsdatum: character varying => date
		ALTER TABLE ax_person ALTER geburtsdatum TYPE date USING to_date(geburtsdatum, 'YYYY-mm-dd');
		ALTER TABLE ax_person ALTER nachnameoderfirma SET NOT NULL;
		ALTER TABLE ax_person ADD ax_li_processstep_ohnedatenerhebung_description character varying[];
		ALTER TABLE ax_person ADD beruf character varying;
		ALTER TABLE ax_person ADD haushaltsstellelandesgrundbesitz character varying;
		ALTER TABLE ax_person ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_person ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_person ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_person ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_person ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_person ADD istteilvon character(16)[];
		ALTER TABLE ax_person ADD processstep_address character varying[];
		ALTER TABLE ax_person ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_person ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_person ADD processstep_datetime character(20)[];
		ALTER TABLE ax_person ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_person ADD processstep_individualname character varying[];
		ALTER TABLE ax_person ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_person ADD processstep_organisationname character varying[];
		ALTER TABLE ax_person ADD processstep_phone character varying[];
		ALTER TABLE ax_person ADD processstep_positionname character varying[];
		ALTER TABLE ax_person ADD processstep_rationale character varying[];
		ALTER TABLE ax_person ADD processstep_role character varying[];
		ALTER TABLE ax_person ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_person ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_person ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_person ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_person ADD sonstigeeigenschaften character varying;
		ALTER TABLE ax_person ADD statement character varying;
		ALTER TABLE ax_person ADD wohnortodersitz character varying;
		ALTER TABLE ax_person ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_person ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_person ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_f967b26e_6fe3_4852_a8db_62b31666ff4b ON ax_person USING btree (endet);
		CREATE INDEX alkis_a4c94835_2c94_438a_8128_f39ba667c7b5 ON ax_person USING gin (istteilvon);

		ALTER TABLE ax_platz ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_platz RENAME name TO zeigtaufexternes_name;
		-- ax_platz.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_platz ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_platz ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_platz ADD hatdirektunten character(16)[];
		ALTER TABLE ax_platz ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_platz ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_platz ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_platz ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_platz ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_platz ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_platz ADD istteilvon character(16)[];
		ALTER TABLE ax_platz ADD processstep_address character varying[];
		ALTER TABLE ax_platz ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_platz ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_platz ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_platz ADD processstep_datetime character(20)[];
		ALTER TABLE ax_platz ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_platz ADD processstep_individualname character varying[];
		ALTER TABLE ax_platz ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_platz ADD processstep_organisationname character varying[];
		ALTER TABLE ax_platz ADD processstep_phone character varying[];
		ALTER TABLE ax_platz ADD processstep_positionname character varying[];
		ALTER TABLE ax_platz ADD processstep_rationale character varying[];
		ALTER TABLE ax_platz ADD processstep_role character varying[];
		ALTER TABLE ax_platz ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_platz ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_platz ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_platz ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_platz ADD statement character varying;
		ALTER TABLE ax_platz ADD strassenschluessel character varying;
		ALTER TABLE ax_platz ADD traegtbeizu character(16)[];
		ALTER TABLE ax_platz ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_platz ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4331d124_88f6_4e23_ae75_205086c3cc86 ON ax_platz USING btree (endet);
		CREATE INDEX alkis_765c7df4_27b8_4100_965f_37f5e6594666 ON ax_platz USING gin (hatdirektunten);
		CREATE INDEX alkis_06ee8c51_169f_4257_b47b_af14d1f39ca8 ON ax_platz USING gin (istabgeleitetaus);
		CREATE INDEX alkis_51a349ea_0dfb_4dad_a7de_012e9aec9434 ON ax_platz USING gin (istteilvon);
		CREATE INDEX alkis_04d95c70_7cc5_4e07_abbd_440eaf94af93 ON ax_platz USING gin (traegtbeizu);

		ALTER TABLE ax_punktkennunguntergegangen ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_punktkennunguntergegangen ALTER punktkennung SET NOT NULL;
		ALTER TABLE ax_punktkennunguntergegangen ADD istteilvon character(16)[];
		ALTER TABLE ax_punktkennunguntergegangen ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_punktkennunguntergegangen ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_punktkennunguntergegangen ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_a9c12b65_4c8a_4568_a8e2_2d830b2a986c ON ax_punktkennunguntergegangen USING btree (endet);
		CREATE INDEX alkis_5abb7525_2be4_429d_a136_17ccaf456ec2 ON ax_punktkennunguntergegangen USING btree (gml_id, beginnt);
		CREATE INDEX alkis_3278d65b_7ece_4af8_8a57_564218abded1 ON ax_punktkennunguntergegangen USING gin (istteilvon);

		ALTER TABLE ax_punktortag RENAME art TO zeigtaufexternes_art;
		ALTER TABLE ax_punktortag DROP ax_datenerhebung_punktort;
		ALTER TABLE ax_punktortag ALTER beginnt SET NOT NULL;
		-- ax_punktortag.istteilvon => istteilvon: character(16) => character(16)[]
		ALTER TABLE ax_punktortag ALTER istteilvon TYPE character(16)[] USING CASE WHEN istteilvon IS NULL THEN NULL ELSE ARRAY[istteilvon] END;
		ALTER TABLE ax_punktortag RENAME name TO zeigtaufexternes_name;
		-- ax_punktortag.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_punktortag DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_punktortag RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_punktortag','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_punktortag SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_punktortag DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_punktortag_wkb_geometry_idx ON ax_punktortag USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_punktortag ADD genauigkeitswert_datetime character(20)[];
		ALTER TABLE ax_punktortag ADD genauigkeitswert_evaluationmethoddescription character varying;
		ALTER TABLE ax_punktortag ADD genauigkeitswert_evaluationmethodtype character varying;
		ALTER TABLE ax_punktortag ADD genauigkeitswert_evaluationprocedure character varying;
		ALTER TABLE ax_punktortag ADD genauigkeitswert_measuredescription character varying;
		ALTER TABLE ax_punktortag ADD genauigkeitswert_measureidentification character varying;
		ALTER TABLE ax_punktortag ADD genauigkeitswert_nameofmeasure character varying[];
		ALTER TABLE ax_punktortag ADD genauigkeitswert_result character varying[];
		ALTER TABLE ax_punktortag ADD hatdirektunten character(16)[];
		ALTER TABLE ax_punktortag ADD herkunft_source_source_description character varying[];
		ALTER TABLE ax_punktortag ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_punktortag ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_punktortag ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_punktortag ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_punktortag ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_punktortag ADD lagezuverlaessigkeit character varying;
		ALTER TABLE ax_punktortag ADD processstep_address character varying[];
		ALTER TABLE ax_punktortag ADD processstep_ax_li_processstep_punktort_description character varying[];
		ALTER TABLE ax_punktortag ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_punktortag ADD processstep_datetime character(20)[];
		ALTER TABLE ax_punktortag ADD processstep_description character varying[];
		ALTER TABLE ax_punktortag ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_punktortag ADD processstep_individualname character varying[];
		ALTER TABLE ax_punktortag ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_punktortag ADD processstep_organisationname character varying[];
		ALTER TABLE ax_punktortag ADD processstep_phone character varying[];
		ALTER TABLE ax_punktortag ADD processstep_positionname character varying[];
		ALTER TABLE ax_punktortag ADD processstep_rationale character varying[];
		ALTER TABLE ax_punktortag ADD processstep_role character varying[];
		ALTER TABLE ax_punktortag ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_punktortag ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_punktortag ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_punktortag ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_punktortag ADD statement character varying;
		ALTER TABLE ax_punktortag ADD traegtbeizu character(16)[];
		ALTER TABLE ax_punktortag ADD ueberpruefungsdatum date;
		ALTER TABLE ax_punktortag ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_punktortag' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_punktortag USING btree (istteilvon)';
		CREATE INDEX alkis_7611c1a4_2ef3_494c_a214_1affcb79103a ON ax_punktortag USING btree (endet);
		CREATE INDEX alkis_517810ac_6763_45ee_97aa_256346f1fe43 ON ax_punktortag USING gin (hatdirektunten);
		CREATE INDEX alkis_a28e5561_e73a_49f1_a785_a9813c40513b ON ax_punktortag USING gin (istabgeleitetaus);
		CREATE INDEX alkis_bf58512b_27b5_4641_840c_78fdfa838ba8 ON ax_punktortag USING gin (istteilvon);
		CREATE INDEX alkis_0b0b1e8a_876a_4c33_bf0a_75577699e308 ON ax_punktortag USING gin (traegtbeizu);

		ALTER TABLE ax_punktortau DROP ax_datenerhebung_punktort;
		ALTER TABLE ax_punktortau ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_punktortau RENAME individualname TO processstep_individualname;
		-- ax_punktortau.individualname => processstep_individualname: character varying => character varying[]
		ALTER TABLE ax_punktortau ALTER processstep_individualname TYPE character varying[] USING CASE WHEN processstep_individualname IS NULL THEN NULL ELSE ARRAY[processstep_individualname] END;
		-- ax_punktortau.istteilvon => istteilvon: character(16) => character(16)[]
		ALTER TABLE ax_punktortau ALTER istteilvon TYPE character(16)[] USING CASE WHEN istteilvon IS NULL THEN NULL ELSE ARRAY[istteilvon] END;
		ALTER TABLE ax_punktortau RENAME name TO zeigtaufexternes_name;
		-- ax_punktortau.wkb_geometry => wkb_geometry: geometry(PointZ,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_punktortau DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_punktortau RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_punktortau','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',3);
			UPDATE ax_punktortau SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_punktortau DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_punktortau_wkb_geometry_idx ON ax_punktortau USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_punktortau ADD genauigkeitswert_datetime character(20)[];
		ALTER TABLE ax_punktortau ADD genauigkeitswert_evaluationmethoddescription character varying;
		ALTER TABLE ax_punktortau ADD genauigkeitswert_evaluationmethodtype character varying;
		ALTER TABLE ax_punktortau ADD genauigkeitswert_evaluationprocedure character varying;
		ALTER TABLE ax_punktortau ADD genauigkeitswert_measuredescription character varying;
		ALTER TABLE ax_punktortau ADD genauigkeitswert_measureidentification character varying;
		ALTER TABLE ax_punktortau ADD genauigkeitswert_nameofmeasure character varying[];
		ALTER TABLE ax_punktortau ADD genauigkeitswert_result character varying[];
		ALTER TABLE ax_punktortau ADD hatdirektunten character(16)[];
		ALTER TABLE ax_punktortau ADD herkunft_source_source_description character varying[];
		ALTER TABLE ax_punktortau ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_punktortau ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_punktortau ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_punktortau ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_punktortau ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_punktortau ADD lagezuverlaessigkeit character varying;
		ALTER TABLE ax_punktortau ADD processstep_address character varying[];
		ALTER TABLE ax_punktortau ADD processstep_ax_li_processstep_punktort_description character varying[];
		ALTER TABLE ax_punktortau ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_punktortau ADD processstep_datetime character(20)[];
		ALTER TABLE ax_punktortau ADD processstep_description character varying[];
		ALTER TABLE ax_punktortau ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_punktortau ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_punktortau ADD processstep_organisationname character varying[];
		ALTER TABLE ax_punktortau ADD processstep_phone character varying[];
		ALTER TABLE ax_punktortau ADD processstep_positionname character varying[];
		ALTER TABLE ax_punktortau ADD processstep_rationale character varying[];
		ALTER TABLE ax_punktortau ADD processstep_role character varying[];
		ALTER TABLE ax_punktortau ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_punktortau ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_punktortau ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_punktortau ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_punktortau ADD statement character varying;
		ALTER TABLE ax_punktortau ADD traegtbeizu character(16)[];
		ALTER TABLE ax_punktortau ADD ueberpruefungsdatum date;
		ALTER TABLE ax_punktortau ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_punktortau ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_punktortau' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_punktortau USING btree (istteilvon)';
		CREATE INDEX alkis_f512fd00_cdb6_4b78_9804_b37c2170bbfe ON ax_punktortau USING btree (endet);
		CREATE INDEX alkis_f636c79e_8a6b_4db5_8e70_c6fe8773c857 ON ax_punktortau USING gin (hatdirektunten);
		CREATE INDEX alkis_69d8ec8b_748b_4c60_b9d8_213e752f0965 ON ax_punktortau USING gin (istabgeleitetaus);
		CREATE INDEX alkis_b5176aa7_69d7_404b_b2a7_93df5960ec5a ON ax_punktortau USING gin (istteilvon);
		CREATE INDEX alkis_ddd73802_16ff_41a5_8275_c44afe3b826f ON ax_punktortau USING gin (traegtbeizu);

		ALTER TABLE ax_punktortta RENAME art TO zeigtaufexternes_art;
		ALTER TABLE ax_punktortta DROP ax_datenerhebung_punktort;
		ALTER TABLE ax_punktortta ALTER beginnt SET NOT NULL;
		-- ax_punktortta.istteilvon => istteilvon: character(16) => character(16)[]
		ALTER TABLE ax_punktortta ALTER istteilvon TYPE character(16)[] USING CASE WHEN istteilvon IS NULL THEN NULL ELSE ARRAY[istteilvon] END;
		ALTER TABLE ax_punktortta RENAME name TO zeigtaufexternes_name;
		-- ax_punktortta.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_punktortta DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_punktortta RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_punktortta','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_punktortta SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_punktortta DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_punktortta_wkb_geometry_idx ON ax_punktortta USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_punktortta ADD genauigkeitswert_datetime character(20)[];
		ALTER TABLE ax_punktortta ADD genauigkeitswert_evaluationmethoddescription character varying;
		ALTER TABLE ax_punktortta ADD genauigkeitswert_evaluationmethodtype character varying;
		ALTER TABLE ax_punktortta ADD genauigkeitswert_evaluationprocedure character varying;
		ALTER TABLE ax_punktortta ADD genauigkeitswert_measuredescription character varying;
		ALTER TABLE ax_punktortta ADD genauigkeitswert_measureidentification character varying;
		ALTER TABLE ax_punktortta ADD genauigkeitswert_nameofmeasure character varying[];
		ALTER TABLE ax_punktortta ADD genauigkeitswert_result character varying[];
		ALTER TABLE ax_punktortta ADD hatdirektunten character(16)[];
		ALTER TABLE ax_punktortta ADD herkunft_source_source_description character varying[];
		ALTER TABLE ax_punktortta ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_punktortta ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_punktortta ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_punktortta ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_punktortta ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_punktortta ADD lagezuverlaessigkeit character varying;
		ALTER TABLE ax_punktortta ADD processstep_address character varying[];
		ALTER TABLE ax_punktortta ADD processstep_ax_li_processstep_punktort_description character varying[];
		ALTER TABLE ax_punktortta ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_punktortta ADD processstep_datetime character(20)[];
		ALTER TABLE ax_punktortta ADD processstep_description character varying[];
		ALTER TABLE ax_punktortta ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_punktortta ADD processstep_individualname character varying[];
		ALTER TABLE ax_punktortta ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_punktortta ADD processstep_organisationname character varying[];
		ALTER TABLE ax_punktortta ADD processstep_phone character varying[];
		ALTER TABLE ax_punktortta ADD processstep_positionname character varying[];
		ALTER TABLE ax_punktortta ADD processstep_rationale character varying[];
		ALTER TABLE ax_punktortta ADD processstep_role character varying[];
		ALTER TABLE ax_punktortta ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_punktortta ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_punktortta ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_punktortta ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_punktortta ADD statement character varying;
		ALTER TABLE ax_punktortta ADD traegtbeizu character(16)[];
		ALTER TABLE ax_punktortta ADD ueberpruefungsdatum date;
		ALTER TABLE ax_punktortta ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_punktortta' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_punktortta USING btree (istteilvon)';
		CREATE INDEX alkis_7b3c63dc_e6bf_44a8_b9ac_209ca8a3d8e8 ON ax_punktortta USING gin (hatdirektunten);
		CREATE INDEX alkis_d0154547_40df_4b9d_a9b0_07f71d0ebcdb ON ax_punktortta USING gin (istabgeleitetaus);
		CREATE INDEX alkis_60968e58_e1b1_4a6f_8a0a_86cf1bb2261a ON ax_punktortta USING gin (istteilvon);
		CREATE INDEX alkis_e026cd5c_783d_4551_b5c9_5511040b8608 ON ax_punktortta USING gin (traegtbeizu);

		ALTER TABLE ax_regierungsbezirk ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_regierungsbezirk ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_regierungsbezirk ALTER land SET NOT NULL;
		ALTER TABLE ax_regierungsbezirk ALTER regierungsbezirk SET NOT NULL;
		ALTER TABLE ax_regierungsbezirk ALTER schluesselgesamt SET NOT NULL;
		ALTER TABLE ax_regierungsbezirk ADD administrativefunktion integer[];
		ALTER TABLE ax_regierungsbezirk ADD istteilvon character(16)[];
		ALTER TABLE ax_regierungsbezirk ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_regierungsbezirk ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_regierungsbezirk ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4fb4cd36_d1b6_48cb_81c8_1950e100487f ON ax_regierungsbezirk USING btree (endet);
		CREATE INDEX alkis_1574bc43_c060_413a_a7e6_f64ff1b5dabb ON ax_regierungsbezirk USING gin (istteilvon);

		-- ax_reservierung.ablaufderreservierung => ablaufderreservierung: character varying => date
		ALTER TABLE ax_reservierung ALTER ablaufderreservierung TYPE date USING to_date(ablaufderreservierung, 'YYYY-mm-dd');
		ALTER TABLE ax_reservierung ALTER art SET NOT NULL;
		ALTER TABLE ax_reservierung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_reservierung ALTER land SET NOT NULL;
		ALTER TABLE ax_reservierung ALTER nummer SET NOT NULL;
		ALTER TABLE ax_reservierung ALTER stelle SET NOT NULL;
		ALTER TABLE ax_reservierung ADD buchungsblattbezirk_bezirk character varying;
		ALTER TABLE ax_reservierung ADD buchungsblattbezirk_land character varying;
		ALTER TABLE ax_reservierung ADD flur_gemarkung character varying;
		ALTER TABLE ax_reservierung ADD flur_gemarkungsteilflur character varying;
		ALTER TABLE ax_reservierung ADD flur_land character varying;
		ALTER TABLE ax_reservierung ADD gemarkung_gemarkungsnummer character varying;
		ALTER TABLE ax_reservierung ADD gemarkung_land character varying;
		ALTER TABLE ax_reservierung ADD istteilvon character(16)[];
		ALTER TABLE ax_reservierung ADD nummerierungsbezirk character varying;
		ALTER TABLE ax_reservierung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_reservierung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_reservierung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_15e95ff5_73fb_435a_9ee0_d37870971803 ON ax_reservierung USING btree (endet);
		CREATE INDEX alkis_d47d4d1b_e253_4f42_8efc_4f035d1dc137 ON ax_reservierung USING btree (gml_id, beginnt);
		CREATE INDEX alkis_c4805830_11f4_4c1a_9bf5_6b69956cab56 ON ax_reservierung USING gin (istteilvon);

		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ALTER beginnt SET NOT NULL;
		-- ax_schifffahrtsliniefaehrverkehr.wkb_geometry => wkb_geometry: geometry(LineString,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_schifffahrtsliniefaehrverkehr DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_schifffahrtsliniefaehrverkehr RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_schifffahrtsliniefaehrverkehr','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_schifffahrtsliniefaehrverkehr SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_schifffahrtsliniefaehrverkehr DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_schifffahrtsliniefaehrverkehr_wkb_geometry_idx ON ax_schifffahrtsliniefaehrverkehr USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD hatdirektunten character(16)[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD istteilvon character(16)[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_address character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_datetime character(20)[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_individualname character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_organisationname character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_phone character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_positionname character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_rationale character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_role character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD statement character varying;
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD traegtbeizu character(16)[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_schifffahrtsliniefaehrverkehr ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_c41d3455_9fdb_4da6_93cd_e8867a6f5173 ON ax_schifffahrtsliniefaehrverkehr USING btree (endet);
		CREATE INDEX alkis_060ec5e4_d37e_42cc_ba8f_31189f2a9694 ON ax_schifffahrtsliniefaehrverkehr USING gin (hatdirektunten);
		CREATE INDEX alkis_00166e68_d67b_44ec_a8f9_740a184756e7 ON ax_schifffahrtsliniefaehrverkehr USING gin (istabgeleitetaus);
		CREATE INDEX alkis_8656686a_cf4b_431e_95f1_96c86cc9b709 ON ax_schifffahrtsliniefaehrverkehr USING gin (istteilvon);
		CREATE INDEX alkis_d8830234_e1f3_4c2f_87e8_5211659332e0 ON ax_schifffahrtsliniefaehrverkehr USING gin (traegtbeizu);

		ALTER TABLE ax_schiffsverkehr ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_schiffsverkehr RENAME name TO zeigtaufexternes_name;
		-- ax_schiffsverkehr.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_schiffsverkehr ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_schiffsverkehr ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_schiffsverkehr ADD gemeinde character varying;
		ALTER TABLE ax_schiffsverkehr ADD hatdirektunten character(16)[];
		ALTER TABLE ax_schiffsverkehr ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_schiffsverkehr ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_schiffsverkehr ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_schiffsverkehr ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_schiffsverkehr ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_schiffsverkehr ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_schiffsverkehr ADD istteilvon character(16)[];
		ALTER TABLE ax_schiffsverkehr ADD kreis character varying;
		ALTER TABLE ax_schiffsverkehr ADD lage character varying;
		ALTER TABLE ax_schiffsverkehr ADD land character varying;
		ALTER TABLE ax_schiffsverkehr ADD processstep_address character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_datetime character(20)[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_individualname character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_organisationname character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_phone character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_positionname character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_rationale character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_role character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_schiffsverkehr ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_schiffsverkehr ADD regierungsbezirk character varying;
		ALTER TABLE ax_schiffsverkehr ADD statement character varying;
		ALTER TABLE ax_schiffsverkehr ADD traegtbeizu character(16)[];
		ALTER TABLE ax_schiffsverkehr ADD unverschluesselt character varying;
		ALTER TABLE ax_schiffsverkehr ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_schiffsverkehr ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_4dffe632_8503_4ce8_bb02_4121d18a8f09 ON ax_schiffsverkehr USING btree (endet);
		CREATE INDEX alkis_2df433ae_bcea_4416_b253_df663dba0e51 ON ax_schiffsverkehr USING gin (hatdirektunten);
		CREATE INDEX alkis_31f49200_66f2_4543_8c6d_f48d99db67d1 ON ax_schiffsverkehr USING gin (istabgeleitetaus);
		CREATE INDEX alkis_1d59b629_560a_4753_a3a3_f7ec97213327 ON ax_schiffsverkehr USING gin (istteilvon);
		CREATE INDEX alkis_e0a23b09_a876_459b_8064_64b9cceadd9d ON ax_schiffsverkehr USING gin (traegtbeizu);

		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD bestehtaus character(16)[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD nummerdesschutzgebietes character varying;
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_address character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_role character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD statement character varying;
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_schutzgebietnachnaturumweltoderbodenschutzrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_f5bab60b_9905_4412_b350_ab63cfe117a5 ON ax_schutzgebietnachnaturumweltoderbodenschutzrecht USING btree (endet);
		CREATE INDEX alkis_463dc50b_11c2_457e_814e_e15733ff1069 ON ax_schutzgebietnachnaturumweltoderbodenschutzrecht USING gin (bestehtaus);
		CREATE INDEX alkis_a75245ac_310e_430f_b4ef_e1dc958d8d32 ON ax_schutzgebietnachnaturumweltoderbodenschutzrecht USING gin (istteilvon);

		ALTER TABLE ax_schutzgebietnachwasserrecht RENAME art TO zeigtaufexternes_art;
		ALTER TABLE ax_schutzgebietnachwasserrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_schutzgebietnachwasserrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD bestehtaus character(16)[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD funktion integer;
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_address character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_role character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD statement character varying;
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_schutzgebietnachwasserrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_afe65b3f_ff2e_4621_91d8_f3382bb9795d ON ax_schutzgebietnachwasserrecht USING btree (endet);
		CREATE INDEX alkis_e8ce640c_bcc2_4a9a_aa75_fca123659174 ON ax_schutzgebietnachwasserrecht USING gin (bestehtaus);
		CREATE INDEX alkis_959b21fe_fe3c_474e_b80c_791aeba90cff ON ax_schutzgebietnachwasserrecht USING gin (istteilvon);

		ALTER TABLE ax_schutzzone RENAME art TO zeigtaufexternes_art;
		ALTER TABLE ax_schutzzone ALTER beginnt SET NOT NULL;
		-- ax_schutzzone.istteilvon => istteilvon: character(16) => character(16)[]
		ALTER TABLE ax_schutzzone ALTER istteilvon TYPE character(16)[] USING CASE WHEN istteilvon IS NULL THEN NULL ELSE ARRAY[istteilvon] END;
		ALTER TABLE ax_schutzzone ALTER zone SET NOT NULL;
		ALTER TABLE ax_schutzzone ADD hatdirektunten character(16)[];
		ALTER TABLE ax_schutzzone ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_schutzzone ADD nummerderschutzzone character varying;
		ALTER TABLE ax_schutzzone ADD rechtszustand integer;
		ALTER TABLE ax_schutzzone ADD traegtbeizu character(16)[];
		ALTER TABLE ax_schutzzone ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_schutzzone ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_schutzzone' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_schutzzone USING btree (istteilvon)';
		CREATE INDEX alkis_c5ffeeb0_c531_41a3_902f_7d078e9cae30 ON ax_schutzzone USING btree (endet);
		CREATE INDEX alkis_3d5c780d_90a7_4cdc_b3dd_8633402df859 ON ax_schutzzone USING gin (hatdirektunten);
		CREATE INDEX alkis_a707ffa1_1d01_4d7d_b463_51e1087d7f74 ON ax_schutzzone USING gin (istabgeleitetaus);
		CREATE INDEX alkis_b954697f_05fb_484e_bf91_312009dc34bf ON ax_schutzzone USING gin (istteilvon);
		CREATE INDEX alkis_64fff255_dc50_4417_8085_7a9c9353f105 ON ax_schutzzone USING gin (traegtbeizu);

		ALTER TABLE ax_seilbahnschwebebahn ALTER bahnkategorie SET NOT NULL;
		ALTER TABLE ax_seilbahnschwebebahn ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_seilbahnschwebebahn ADD hatdirektunten character(16)[];
		ALTER TABLE ax_seilbahnschwebebahn ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_seilbahnschwebebahn ADD istteilvon character(16)[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_address character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_datetime character(20)[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_individualname character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_organisationname character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_phone character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_positionname character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_rationale character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_role character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD statement character varying;
		ALTER TABLE ax_seilbahnschwebebahn ADD traegtbeizu character(16)[];
		ALTER TABLE ax_seilbahnschwebebahn ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_seilbahnschwebebahn ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_088f8a74_d5b2_4231_a370_26cbbf956cbe ON ax_seilbahnschwebebahn USING btree (endet);
		CREATE INDEX alkis_7decde0d_5b31_4733_a6a3_9df348829b36 ON ax_seilbahnschwebebahn USING gin (hatdirektunten);
		CREATE INDEX alkis_900fb3c6_70bb_4871_9974_c8f0ba7f7eb5 ON ax_seilbahnschwebebahn USING gin (istabgeleitetaus);
		CREATE INDEX alkis_de97f000_0e80_437a_87f1_667d42b9d6dd ON ax_seilbahnschwebebahn USING gin (istteilvon);
		CREATE INDEX alkis_8b3c1784_b74d_4309_93eb_4f4a44aec97c ON ax_seilbahnschwebebahn USING gin (traegtbeizu);

		ALTER TABLE ax_sicherungspunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_sicherungspunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_sicherungspunkt RENAME name TO zeigtaufexternes_name;
		-- ax_sicherungspunkt.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_sicherungspunkt ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_sicherungspunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_sicherungspunkt ALTER vermarkung_marke SET NOT NULL;
		ALTER TABLE ax_sicherungspunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_sicherungspunkt ADD hat character(16)[];
		ALTER TABLE ax_sicherungspunkt ADD horizontfreiheit integer;
		ALTER TABLE ax_sicherungspunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_sicherungspunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_sicherungspunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_b2595bb3_98d8_4249_99da_930fe2c01250 ON ax_sicherungspunkt USING btree (endet);
		CREATE INDEX alkis_37c09f70_6231_44fa_91e8_a940b95d5d08 ON ax_sicherungspunkt USING gin (bestehtaus);
		CREATE INDEX alkis_6cd88406_9ab6_44e5_943c_6afd28b6b06e ON ax_sicherungspunkt USING gin (hat);
		CREATE INDEX alkis_7f7a29e2_a0c3_40c0_9d74_6329887001c0 ON ax_sicherungspunkt USING gin (istteilvon);

		ALTER TABLE ax_soll ALTER beginnt SET NOT NULL;
		-- ax_soll.wkb_geometry => wkb_geometry: geometry(Polygon,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_soll DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_soll RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_soll','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_soll SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_soll DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_soll_wkb_geometry_idx ON ax_soll USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_soll ADD hatdirektunten character(16)[];
		ALTER TABLE ax_soll ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_soll ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_soll ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_soll ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_soll ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_soll ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_soll ADD istteilvon character(16)[];
		ALTER TABLE ax_soll ADD processstep_address character varying[];
		ALTER TABLE ax_soll ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_soll ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_soll ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_soll ADD processstep_datetime character(20)[];
		ALTER TABLE ax_soll ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_soll ADD processstep_individualname character varying[];
		ALTER TABLE ax_soll ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_soll ADD processstep_organisationname character varying[];
		ALTER TABLE ax_soll ADD processstep_phone character varying[];
		ALTER TABLE ax_soll ADD processstep_positionname character varying[];
		ALTER TABLE ax_soll ADD processstep_rationale character varying[];
		ALTER TABLE ax_soll ADD processstep_role character varying[];
		ALTER TABLE ax_soll ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_soll ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_soll ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_soll ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_soll ADD statement character varying;
		ALTER TABLE ax_soll ADD traegtbeizu character(16)[];
		ALTER TABLE ax_soll ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_soll ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_soll ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_96f17510_595d_4693_9ccf_922f5fda6ba0 ON ax_soll USING btree (endet);
		CREATE INDEX alkis_ac8cebf5_dfdb_45b9_ae2d_c32781f04851 ON ax_soll USING gin (hatdirektunten);
		CREATE INDEX alkis_6b59edb9_5a53_4523_922a_2cc26811015a ON ax_soll USING gin (istabgeleitetaus);
		CREATE INDEX alkis_fcbef8ec_05e3_41d7_807a_42c288911f64 ON ax_soll USING gin (istteilvon);
		CREATE INDEX alkis_0adb38f5_f174_4806_9737_749efd8b2461 ON ax_soll USING gin (traegtbeizu);

		ALTER TABLE ax_sonstigervermessungspunkt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_sonstigervermessungspunkt RENAME land TO zustaendigestelle_land;
		ALTER TABLE ax_sonstigervermessungspunkt RENAME stelle TO zustaendigestelle_stelle;
		ALTER TABLE ax_sonstigervermessungspunkt ALTER vermarkung_marke SET NOT NULL;
		ALTER TABLE ax_sonstigervermessungspunkt ADD bestehtaus character(16)[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD beziehtsichauf character(16)[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD gehoertzu character(16)[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD horizontfreiheit integer;
		ALTER TABLE ax_sonstigervermessungspunkt ADD istteilvon character(16)[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD verbundenmit character(16)[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_sonstigervermessungspunkt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_24e3cbe0_a123_4b5e_8f7e_37d4f9d7f9d1 ON ax_sonstigervermessungspunkt USING btree (endet);
		CREATE INDEX alkis_b3c85d35_9ca5_418e_941e_ff85e2ef9de8 ON ax_sonstigervermessungspunkt USING gin (bestehtaus);
		CREATE INDEX alkis_5c3ba3db_6103_423d_adf0_54b22421b2f6 ON ax_sonstigervermessungspunkt USING gin (beziehtsichauf);
		CREATE INDEX alkis_1bec07e9_0d6a_4133_b859_dcf0c6d15d84 ON ax_sonstigervermessungspunkt USING gin (gehoertzu);
		CREATE INDEX alkis_ad99f1f6_1f4d_4cf8_a14f_8c4008b5dcc6 ON ax_sonstigervermessungspunkt USING gin (istteilvon);
		CREATE INDEX alkis_66eebbe8_25b9_4115_aaf2_929f0cbb9007 ON ax_sonstigervermessungspunkt USING gin (verbundenmit);

		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung RENAME description TO processstep_ax_li_processstep_mitdatenerhebung_description;
		-- ax_sonstigesbauwerkodersonstigeeinrichtung.description => processstep_ax_li_processstep_mitdatenerhebung_description: integer => character varying[]
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ALTER processstep_ax_li_processstep_mitdatenerhebung_description TYPE character varying[] USING CASE WHEN processstep_ax_li_processstep_mitdatenerhebung_description IS NULL THEN NULL ELSE ARRAY[processstep_ax_li_processstep_mitdatenerhebung_description::int] END;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung DROP gehoertzubauwerk;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD bezeichnung character varying;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_abschnitt character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_ast character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_bahnverkehrsanlage character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_bauwerkimgewaesserbereich character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_bauwerkimverkehrsbereich character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_bauwerkoderanlagefuerindustrieundgewer character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_bauwerkoderanlagefuersportfreizeitunde character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_einrichtungenfuerdenschiffsverkehr character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_einrichtunginoeffentlichenbereichen character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_flugverkehrsanlage character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_gewaessermerkmal character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_gewaesserstationierungsachse character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_gleis character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_grenzuebergang character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_hafen character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_heilquellegasquelle character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_historischesbauwerkoderhistorischeeinr character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_leitung character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_netzknoten character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_nullpunkt character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_ortslage character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_polder character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_schifffahrtsliniefaehrverkehr character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_schleuse character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_seilbahnschwebebahn character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_sickerstrecke character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_sonstigesbauwerkodersonstigeeinrichtun character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_strassenverkehrsanlage character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_testgelaende character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_transportanlage character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_turm character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_untergeordnetesgewaesser character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_vegetationsmerkmal character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_vorratsbehaelterspeicherbauwerk character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_wasserspiegelhoehe character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD gehoertzubauwerk_ax_wegpfadsteig character(16);
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD hatdirektunten character(16)[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD hydrologischesmerkmal integer;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD istteilvon character(16)[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD objekthoehe double precision;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_address character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_individualname character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_phone character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_positionname character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_rationale character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_role character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD statement character varying;
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD traegtbeizu character(16)[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_sonstigesbauwerkodersonstigeeinrichtung ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_sonstigesbauwerkodersonstigeeinrichtung' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk)';
		CREATE INDEX alkis_883c12ad_45ba_4b06_9a9c_98b0aff853f3 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (endet);
		CREATE INDEX alkis_cc40e634_bd89_4bb4_9c24_32648a5f94fb ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_abschnitt);
		CREATE INDEX alkis_e0df5d9f_e7ee_4a31_b572_0cb7f74a1eae ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_ast);
		CREATE INDEX alkis_eb61156c_4cb3_4007_ac4d_c29a0ffb670a ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_bahnverkehrsanlage);
		CREATE INDEX alkis_01126df7_aba1_4c72_ab27_a2666a9cd602 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_bauwerkimgewaesserbereich);
		CREATE INDEX alkis_301c30db_c3fd_48e3_a0c7_afde999f03d5 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_bauwerkimverkehrsbereich);
		CREATE INDEX alkis_e43cf3ea_416e_4421_92b6_8acaa82e7442 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_bauwerkoderanlagefuerindustrieundgewer);
		CREATE INDEX alkis_353588fe_303d_49c6_95ad_c18d54633bd3 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_bauwerkoderanlagefuersportfreizeitunde);
		CREATE INDEX alkis_eb7ab46e_0ead_4c15_bb75_e89cf520c807 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_einrichtungenfuerdenschiffsverkehr);
		CREATE INDEX alkis_b37b8956_7fd1_4cb8_a01f_8472add24123 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_einrichtunginoeffentlichenbereichen);
		CREATE INDEX alkis_bb5ffe1b_1ee6_4cfd_a1b1_134532f66c45 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_flugverkehrsanlage);
		CREATE INDEX alkis_4e04bc35_7622_4b38_9993_ef4fd7507f3c ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_gewaessermerkmal);
		CREATE INDEX alkis_96416332_352c_4331_9bca_b236224be490 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_gewaesserstationierungsachse);
		CREATE INDEX alkis_fbe2b696_d3c9_44ec_a0f7_6f5094508183 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_gleis);
		CREATE INDEX alkis_b3934314_1da7_41e2_a627_b9010db385c3 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_grenzuebergang);
		CREATE INDEX alkis_db244b10_933b_48dd_aa05_928618f43513 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_hafen);
		CREATE INDEX alkis_850ef963_6031_4f1c_ae66_cbb52bac7ef4 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_heilquellegasquelle);
		CREATE INDEX alkis_04529a66_7309_4b95_b8ee_27bdb52673e5 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_historischesbauwerkoderhistorischeeinr);
		CREATE INDEX alkis_9f05bdca_95e3_4103_baa0_7694576dac5c ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_leitung);
		CREATE INDEX alkis_730e962a_1798_4cf3_8254_9b7141bd97ef ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_netzknoten);
		CREATE INDEX alkis_e91fed58_96b3_4533_92f2_5fab821c7642 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_nullpunkt);
		CREATE INDEX alkis_8609fcbc_1f5a_4f30_b896_7df1b269de47 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_ortslage);
		CREATE INDEX alkis_e7918d29_4225_4e34_9a07_56fbbd035734 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_polder);
		CREATE INDEX alkis_689a51e1_58c8_4003_a3f2_dc4ed556a997 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_schifffahrtsliniefaehrverkehr);
		CREATE INDEX alkis_4b12de8c_2a09_4d37_85b8_0ae60a82f064 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_schleuse);
		CREATE INDEX alkis_a053989e_549a_4414_8cbe_903af58ddbae ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_seilbahnschwebebahn);
		CREATE INDEX alkis_4c480257_894c_4cde_888a_25390d0204ff ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_sickerstrecke);
		CREATE INDEX alkis_ab06253f_a213_4bd9_81fd_622e059b443c ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_sonstigesbauwerkodersonstigeeinrichtun);
		CREATE INDEX alkis_714b88ee_b924_4615_b42a_ab76c4a2d93e ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_strassenverkehrsanlage);
		CREATE INDEX alkis_a71b74f9_f986_426b_ba09_131d7ab2df18 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_testgelaende);
		CREATE INDEX alkis_8a80344a_e6bf_4b34_9d5b_62f877b74efa ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_transportanlage);
		CREATE INDEX alkis_6dcc77cc_4167_4129_ba6a_49d4157ca39e ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_turm);
		CREATE INDEX alkis_e904c709_ceb3_48ce_beae_ae924378e8de ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_untergeordnetesgewaesser);
		CREATE INDEX alkis_993ad727_bbfc_4396_875a_543fe84a5446 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_vegetationsmerkmal);
		CREATE INDEX alkis_262d9d21_c56a_46d0_a1ea_b85401da694a ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_vorratsbehaelterspeicherbauwerk);
		CREATE INDEX alkis_12763e6c_6666_4302_a556_b2aeff74b421 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_wasserspiegelhoehe);
		CREATE INDEX alkis_d77010ed_e406_4fbc_afb1_30daf7e9380c ON ax_sonstigesbauwerkodersonstigeeinrichtung USING btree (gehoertzubauwerk_ax_wegpfadsteig);
		CREATE INDEX alkis_1052e3f5_0211_4a48_8367_9ee4c8d23583 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING gin (hatdirektunten);
		CREATE INDEX alkis_14835e4c_416a_4055_a29d_f91941366c5e ON ax_sonstigesbauwerkodersonstigeeinrichtung USING gin (istabgeleitetaus);
		CREATE INDEX alkis_60af1f78_0dea_4af5_aa8a_d743f7c0f113 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING gin (istteilvon);
		CREATE INDEX alkis_61d462e4_12cb_499b_98cb_db9cf40914d2 ON ax_sonstigesbauwerkodersonstigeeinrichtung USING gin (traegtbeizu);

		ALTER TABLE ax_sonstigesrecht RENAME art TO zeigtaufexternes_art;
		-- ax_sonstigesrecht.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_sonstigesrecht ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_sonstigesrecht ALTER artderfestlegung SET NOT NULL;
		ALTER TABLE ax_sonstigesrecht ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_sonstigesrecht DROP characterstring;
		ALTER TABLE ax_sonstigesrecht DROP funktion;
		ALTER TABLE ax_sonstigesrecht ADD hatdirektunten character(16)[];
		ALTER TABLE ax_sonstigesrecht ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_sonstigesrecht ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_sonstigesrecht ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_sonstigesrecht ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_sonstigesrecht ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_sonstigesrecht ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_sonstigesrecht ADD istteilvon character(16)[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_address character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_datetime character(20)[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_individualname character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_organisationname character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_phone character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_positionname character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_rationale character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_role character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_sonstigesrecht ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_sonstigesrecht ADD statement character varying;
		ALTER TABLE ax_sonstigesrecht ADD traegtbeizu character(16)[];
		ALTER TABLE ax_sonstigesrecht ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_sonstigesrecht ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_84893f48_bc81_4120_b017_f44c923220a9 ON ax_sonstigesrecht USING btree (endet);
		CREATE INDEX alkis_f337a134_f9fe_4f7a_8671_26e9692278e7 ON ax_sonstigesrecht USING gin (hatdirektunten);
		CREATE INDEX alkis_8cdffee8_a012_4097_9b8b_6c460d28c893 ON ax_sonstigesrecht USING gin (istabgeleitetaus);
		CREATE INDEX alkis_093b8ba3_c327_4309_a7e0_217b2890e561 ON ax_sonstigesrecht USING gin (istteilvon);
		CREATE INDEX alkis_46ff8bff_0b5e_402e_8927_a87413008743 ON ax_sonstigesrecht USING gin (traegtbeizu);

		ALTER TABLE ax_sportfreizeitunderholungsflaeche ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD bezeichnung character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD hatdirektunten character(16)[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD istteilvon character(16)[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_address character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_datetime character(20)[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_individualname character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_organisationname character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_phone character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_positionname character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_rationale character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_role character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD statement character varying;
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD traegtbeizu character(16)[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_sportfreizeitunderholungsflaeche ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_84013e64_abc0_44ca_a057_0e060c19d697 ON ax_sportfreizeitunderholungsflaeche USING btree (endet);
		CREATE INDEX alkis_bea3119c_aff6_4f45_8cb3_cde242751fc6 ON ax_sportfreizeitunderholungsflaeche USING gin (hatdirektunten);
		CREATE INDEX alkis_c7068084_90bb_453c_9b2e_d78cc82e09d0 ON ax_sportfreizeitunderholungsflaeche USING gin (istabgeleitetaus);
		CREATE INDEX alkis_3cc2532c_af74_4cfe_a5d6_2de406ce4c44 ON ax_sportfreizeitunderholungsflaeche USING gin (istteilvon);
		CREATE INDEX alkis_47da9a40_3481_4142_92c4_2e201b5560fc ON ax_sportfreizeitunderholungsflaeche USING gin (traegtbeizu);

		ALTER TABLE ax_stehendesgewaesser ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_stehendesgewaesser RENAME name TO zeigtaufexternes_name;
		-- ax_stehendesgewaesser.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_stehendesgewaesser ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_stehendesgewaesser ADD bezeichnung character varying;
		ALTER TABLE ax_stehendesgewaesser ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_stehendesgewaesser ADD gemeinde character varying;
		ALTER TABLE ax_stehendesgewaesser ADD hatdirektunten character(16)[];
		ALTER TABLE ax_stehendesgewaesser ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_stehendesgewaesser ADD istteilvon character(16)[];
		ALTER TABLE ax_stehendesgewaesser ADD kreis character varying;
		ALTER TABLE ax_stehendesgewaesser ADD lage character varying;
		ALTER TABLE ax_stehendesgewaesser ADD land character varying;
		ALTER TABLE ax_stehendesgewaesser ADD processstep_address character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_datetime character(20)[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_individualname character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_organisationname character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_phone character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_positionname character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_rationale character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_role character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD regierungsbezirk character varying;
		ALTER TABLE ax_stehendesgewaesser ADD schifffahrtskategorie integer;
		ALTER TABLE ax_stehendesgewaesser ADD statement character varying;
		ALTER TABLE ax_stehendesgewaesser ADD traegtbeizu character(16)[];
		ALTER TABLE ax_stehendesgewaesser ADD widmung integer;
		ALTER TABLE ax_stehendesgewaesser ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_stehendesgewaesser ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_114f4d4e_785e_4f97_9805_94b1675a5388 ON ax_stehendesgewaesser USING btree (endet);
		CREATE INDEX alkis_194915c0_b58c_45ad_a03b_cedaa569a854 ON ax_stehendesgewaesser USING gin (hatdirektunten);
		CREATE INDEX alkis_365c8a1e_613d_4a28_8d63_1df37a32525e ON ax_stehendesgewaesser USING gin (istabgeleitetaus);
		CREATE INDEX alkis_287507be_09ad_4ec6_894a_96d40475c742 ON ax_stehendesgewaesser USING gin (istteilvon);
		CREATE INDEX alkis_d4c1dbc5_f13c_4217_80c1_be01be1af93d ON ax_stehendesgewaesser USING gin (traegtbeizu);

		ALTER TABLE ax_strassenverkehr ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_strassenverkehr RENAME name TO zeigtaufexternes_name;
		-- ax_strassenverkehr.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_strassenverkehr ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_strassenverkehr ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_strassenverkehr ADD hatdirektunten character(16)[];
		ALTER TABLE ax_strassenverkehr ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_strassenverkehr ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_strassenverkehr ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_strassenverkehr ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_strassenverkehr ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_strassenverkehr ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_strassenverkehr ADD istteilvon character(16)[];
		ALTER TABLE ax_strassenverkehr ADD processstep_address character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_datetime character(20)[];
		ALTER TABLE ax_strassenverkehr ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_individualname character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_organisationname character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_phone character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_positionname character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_rationale character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_role character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_strassenverkehr ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_strassenverkehr ADD statement character varying;
		ALTER TABLE ax_strassenverkehr ADD traegtbeizu character(16)[];
		ALTER TABLE ax_strassenverkehr ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_strassenverkehr ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_035fef35_dc12_4eff_afae_20b5cd44971c ON ax_strassenverkehr USING btree (endet);
		CREATE INDEX alkis_5fd84952_8486_4da7_968d_2ccb41ece4a6 ON ax_strassenverkehr USING gin (hatdirektunten);
		CREATE INDEX alkis_7b234dc8_b08a_4723_a034_dd673ac1fd60 ON ax_strassenverkehr USING gin (istabgeleitetaus);
		CREATE INDEX alkis_7b29b31a_9b7e_4ec0_aa72_5ddc9dd0046c ON ax_strassenverkehr USING gin (istteilvon);
		CREATE INDEX alkis_5f7fb978_caa9_459a_a3d0_d7382fc76b59 ON ax_strassenverkehr USING gin (traegtbeizu);

		ALTER TABLE ax_strassenverkehrsanlage ALTER art SET NOT NULL;
		ALTER TABLE ax_strassenverkehrsanlage ALTER beginnt SET NOT NULL;
		-- ax_strassenverkehrsanlage.bezeichnung => bezeichnung: character varying => character varying[]
		ALTER TABLE ax_strassenverkehrsanlage ALTER bezeichnung TYPE character varying[] USING CASE WHEN bezeichnung IS NULL THEN NULL ELSE ARRAY[bezeichnung] END;
		ALTER TABLE ax_strassenverkehrsanlage ADD hatdirektunten character(16)[];
		ALTER TABLE ax_strassenverkehrsanlage ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_strassenverkehrsanlage ADD istteilvon character(16)[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_address character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_datetime character(20)[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_individualname character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_organisationname character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_phone character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_positionname character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_rationale character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_role character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD statement character varying;
		ALTER TABLE ax_strassenverkehrsanlage ADD strassenschluessel character varying;
		ALTER TABLE ax_strassenverkehrsanlage ADD traegtbeizu character(16)[];
		ALTER TABLE ax_strassenverkehrsanlage ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_strassenverkehrsanlage ADD zweitname character varying;
		CREATE INDEX alkis_c8e20472_6df6_4af4_87dc_986a9993d323 ON ax_strassenverkehrsanlage USING btree (endet);
		CREATE INDEX alkis_36dc8e9e_673d_4e57_b887_f844ebbe2e83 ON ax_strassenverkehrsanlage USING gin (hatdirektunten);
		CREATE INDEX alkis_21d036cb_89ff_4260_9e7a_c5945ab772a1 ON ax_strassenverkehrsanlage USING gin (istabgeleitetaus);
		CREATE INDEX alkis_c38b8084_3809_4f32_8ec3_9f746fa0f97d ON ax_strassenverkehrsanlage USING gin (istteilvon);
		CREATE INDEX alkis_f568fb3f_d5d4_4165_8b89_af906cca8434 ON ax_strassenverkehrsanlage USING gin (traegtbeizu);

		ALTER TABLE ax_sumpf ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_sumpf ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_sumpf ADD hatdirektunten character(16)[];
		ALTER TABLE ax_sumpf ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_sumpf ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_sumpf ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_sumpf ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_sumpf ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_sumpf ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_sumpf ADD istteilvon character(16)[];
		ALTER TABLE ax_sumpf ADD processstep_address character varying[];
		ALTER TABLE ax_sumpf ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_sumpf ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_sumpf ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_sumpf ADD processstep_datetime character(20)[];
		ALTER TABLE ax_sumpf ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_sumpf ADD processstep_individualname character varying[];
		ALTER TABLE ax_sumpf ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_sumpf ADD processstep_organisationname character varying[];
		ALTER TABLE ax_sumpf ADD processstep_phone character varying[];
		ALTER TABLE ax_sumpf ADD processstep_positionname character varying[];
		ALTER TABLE ax_sumpf ADD processstep_rationale character varying[];
		ALTER TABLE ax_sumpf ADD processstep_role character varying[];
		ALTER TABLE ax_sumpf ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_sumpf ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_sumpf ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_sumpf ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_sumpf ADD statement character varying;
		ALTER TABLE ax_sumpf ADD traegtbeizu character(16)[];
		ALTER TABLE ax_sumpf ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_sumpf ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_sumpf ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e13e5165_e423_4dd6_a4a0_003a4aa63fe3 ON ax_sumpf USING btree (endet);
		CREATE INDEX alkis_3565e92e_31bf_4bec_9f56_ec8d205e7deb ON ax_sumpf USING gin (hatdirektunten);
		CREATE INDEX alkis_19421cea_36fe_448b_a928_a3d082f40c2c ON ax_sumpf USING gin (istabgeleitetaus);
		CREATE INDEX alkis_04549102_fe12_49f5_b243_8830d78f350f ON ax_sumpf USING gin (istteilvon);
		CREATE INDEX alkis_6a2708de_6ae7_42f3_b4ab_db95c0a321a5 ON ax_sumpf USING gin (traegtbeizu);

		ALTER TABLE ax_tagebaugrubesteinbruch ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_tagebaugrubesteinbruch ADD bezeichnung character varying;
		ALTER TABLE ax_tagebaugrubesteinbruch ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_tagebaugrubesteinbruch ADD hatdirektunten character(16)[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD istteilvon character(16)[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_address character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_datetime character(20)[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_individualname character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_organisationname character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_phone character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_positionname character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_rationale character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_role character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD statement character varying;
		ALTER TABLE ax_tagebaugrubesteinbruch ADD traegtbeizu character(16)[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_tagebaugrubesteinbruch ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_2315e942_97e5_4fb0_9e62_f4350c30bf7e ON ax_tagebaugrubesteinbruch USING btree (endet);
		CREATE INDEX alkis_c9a61a4d_9304_4718_a136_ab188e243401 ON ax_tagebaugrubesteinbruch USING gin (hatdirektunten);
		CREATE INDEX alkis_f18c2976_07c7_46d1_bec4_b5a14b3df883 ON ax_tagebaugrubesteinbruch USING gin (istabgeleitetaus);
		CREATE INDEX alkis_40947f73_e89d_4ec2_83a3_8481e105129d ON ax_tagebaugrubesteinbruch USING gin (istteilvon);
		CREATE INDEX alkis_b7b0dc34_98aa_4abc_bddc_3d7dab48c8f5 ON ax_tagebaugrubesteinbruch USING gin (traegtbeizu);

		ALTER TABLE ax_tagesabschnitt ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_tagesabschnitt ALTER tagesabschnittsnummer SET NOT NULL;
		ALTER TABLE ax_tagesabschnitt ADD hatdirektunten character(16)[];
		ALTER TABLE ax_tagesabschnitt ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_tagesabschnitt ADD istteilvon character(16)[];
		ALTER TABLE ax_tagesabschnitt ADD traegtbeizu character(16)[];
		ALTER TABLE ax_tagesabschnitt ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_tagesabschnitt ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_tagesabschnitt ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_36378475_d0a7_4cb4_ab48_5261618285a5 ON ax_tagesabschnitt USING btree (endet);
		CREATE INDEX alkis_5dbd83fa_41d1_429f_9ce9_12e27c101ed7 ON ax_tagesabschnitt USING gin (hatdirektunten);
		CREATE INDEX alkis_2d93029a_7dc9_487f_bc4f_6b2f6c6222a8 ON ax_tagesabschnitt USING gin (istabgeleitetaus);
		CREATE INDEX alkis_2b9e3857_8d14_4dd8_9ddb_d0bbe42dc347 ON ax_tagesabschnitt USING gin (istteilvon);
		CREATE INDEX alkis_0d20d5b7_c5ff_4bde_bcd5_569a939034de ON ax_tagesabschnitt USING gin (traegtbeizu);

		ALTER TABLE ax_topographischelinie ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_topographischelinie ALTER liniendarstellung SET NOT NULL;
		ALTER TABLE ax_topographischelinie ADD hatdirektunten character(16)[];
		ALTER TABLE ax_topographischelinie ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_topographischelinie ADD istteilvon character(16)[];
		ALTER TABLE ax_topographischelinie ADD traegtbeizu character(16)[];
		ALTER TABLE ax_topographischelinie ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_topographischelinie ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_topographischelinie ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_c2e1f847_c243_418b_98aa_efed9e52493d ON ax_topographischelinie USING btree (endet);
		CREATE INDEX alkis_1fb5ff91_0f28_4d33_85bd_150704d0d034 ON ax_topographischelinie USING gin (hatdirektunten);
		CREATE INDEX alkis_fec26ad5_5a4b_4fc3_80e4_24de613549e9 ON ax_topographischelinie USING gin (istabgeleitetaus);
		CREATE INDEX alkis_bf4e8d3d_e0b3_4db3_91f3_88264834ea46 ON ax_topographischelinie USING gin (istteilvon);
		CREATE INDEX alkis_9b10d6ff_7129_4674_9f22_7245e8ee840b ON ax_topographischelinie USING gin (traegtbeizu);

		ALTER TABLE ax_transportanlage RENAME art TO zeigtaufexternes_art;
		-- ax_transportanlage.art => zeigtaufexternes_art: character varying => character varying[]
		ALTER TABLE ax_transportanlage ALTER zeigtaufexternes_art TYPE character varying[] USING CASE WHEN zeigtaufexternes_art IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_art] END;
		ALTER TABLE ax_transportanlage ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_transportanlage ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_transportanlage RENAME name TO zeigtaufexternes_name;
		-- ax_transportanlage.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_transportanlage ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_transportanlage ADD hatdirektunten character(16)[];
		ALTER TABLE ax_transportanlage ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_transportanlage ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_transportanlage ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_transportanlage ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_transportanlage ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_transportanlage ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_transportanlage ADD istteilvon character(16)[];
		ALTER TABLE ax_transportanlage ADD processstep_address character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_datetime character(20)[];
		ALTER TABLE ax_transportanlage ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_individualname character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_organisationname character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_phone character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_positionname character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_rationale character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_role character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_transportanlage ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_transportanlage ADD statement character varying;
		ALTER TABLE ax_transportanlage ADD traegtbeizu character(16)[];
		ALTER TABLE ax_transportanlage ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_efbb80ce_d49c_44ac_8a9f_5d9c6d7930ba ON ax_transportanlage USING btree (endet);
		CREATE INDEX alkis_4b662244_3302_4fcd_86ff_563499a71be7 ON ax_transportanlage USING gin (hatdirektunten);
		CREATE INDEX alkis_aff025f8_5167_4803_89e5_d9b5cdd4e9e5 ON ax_transportanlage USING gin (istabgeleitetaus);
		CREATE INDEX alkis_922e24ee_a8b4_4bb5_8d12_e3d3b8aac44d ON ax_transportanlage USING gin (istteilvon);
		CREATE INDEX alkis_564ac042_f731_499f_9beb_56ed026ccc49 ON ax_transportanlage USING gin (traegtbeizu);

		-- ax_turm.bauwerksfunktion => bauwerksfunktion: integer => integer[]
		ALTER TABLE ax_turm ALTER bauwerksfunktion TYPE integer[] USING CASE WHEN bauwerksfunktion IS NULL THEN NULL ELSE ARRAY[bauwerksfunktion] END;
		ALTER TABLE ax_turm ALTER bauwerksfunktion SET NOT NULL;
		ALTER TABLE ax_turm ALTER beginnt SET NOT NULL;
		-- ax_turm.zeigtauf => zeigtauf: character(16) => character(16)[]
		ALTER TABLE ax_turm ALTER zeigtauf TYPE character(16)[] USING CASE WHEN zeigtauf IS NULL THEN NULL ELSE ARRAY[zeigtauf] END;
		ALTER TABLE ax_turm ADD hatdirektunten character(16)[];
		ALTER TABLE ax_turm ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_turm ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_turm ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_turm ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_turm ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_turm ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_turm ADD istteilvon character(16)[];
		ALTER TABLE ax_turm ADD objekthoehe double precision;
		ALTER TABLE ax_turm ADD processstep_address character varying[];
		ALTER TABLE ax_turm ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_turm ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_turm ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_turm ADD processstep_datetime character(20)[];
		ALTER TABLE ax_turm ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_turm ADD processstep_individualname character varying[];
		ALTER TABLE ax_turm ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_turm ADD processstep_organisationname character varying[];
		ALTER TABLE ax_turm ADD processstep_phone character varying[];
		ALTER TABLE ax_turm ADD processstep_positionname character varying[];
		ALTER TABLE ax_turm ADD processstep_rationale character varying[];
		ALTER TABLE ax_turm ADD processstep_role character varying[];
		ALTER TABLE ax_turm ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_turm ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_turm ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_turm ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_turm ADD statement character varying;
		ALTER TABLE ax_turm ADD traegtbeizu character(16)[];
		ALTER TABLE ax_turm ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_turm ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_turm ADD zeigtaufexternes_uri character varying[];
		PERFORM alkis_dropobject(ix.relname) FROM pg_class cl JOIN pg_namespace ns ON ns.oid=cl.relnamespace JOIN pg_index ind ON cl.oid = ind.indrelid JOIN pg_class ix ON ix.oid = ind.indexrelid WHERE ns.nspname=current_schema() AND cl.relname='ax_turm' AND pg_get_indexdef(ind.indexrelid) LIKE 'CREATE INDEX % ON ax_turm USING btree (zeigtauf)';
		CREATE INDEX alkis_97d04fc1_bdf6_4978_b376_5678d5e7455c ON ax_turm USING btree (endet);
		CREATE INDEX alkis_4a6977a6_28a6_43d0_a542_1706c03dfc99 ON ax_turm USING gin (hatdirektunten);
		CREATE INDEX alkis_8ef79b33_5de7_4202_8d06_c6860adff24b ON ax_turm USING gin (istabgeleitetaus);
		CREATE INDEX alkis_4c793ced_3e9e_40fc_8c47_d23b3c7ea556 ON ax_turm USING gin (istteilvon);
		CREATE INDEX alkis_bfbcbc0f_2950_45f2_b125_b81a64260f2a ON ax_turm USING gin (traegtbeizu);
		CREATE INDEX alkis_dcf731b3_9941_47cc_a3cc_225a88d4b71c ON ax_turm USING gin (zeigtauf);

		ALTER TABLE ax_unlandvegetationsloseflaeche ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD hatdirektunten character(16)[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD istteilvon character(16)[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_address character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_datetime character(20)[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_individualname character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_organisationname character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_phone character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_positionname character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_rationale character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_role character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD statement character varying;
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD traegtbeizu character(16)[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_unlandvegetationsloseflaeche ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_a39b050a_9fee_4fcb_bb00_aa33fe2d6b61 ON ax_unlandvegetationsloseflaeche USING btree (endet);
		CREATE INDEX alkis_cddc9c34_6b34_4e83_80f7_aca4efd73b47 ON ax_unlandvegetationsloseflaeche USING gin (hatdirektunten);
		CREATE INDEX alkis_5d97999c_2d89_473c_a993_8ebb73eee425 ON ax_unlandvegetationsloseflaeche USING gin (istabgeleitetaus);
		CREATE INDEX alkis_1b344fe5_034a_4fce_af66_2a9ccc35022f ON ax_unlandvegetationsloseflaeche USING gin (istteilvon);
		CREATE INDEX alkis_832f882b_41b5_42ec_8c56_79d5f4fcf6c6 ON ax_unlandvegetationsloseflaeche USING gin (traegtbeizu);

		ALTER TABLE ax_untergeordnetesgewaesser ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_untergeordnetesgewaesser ADD hatdirektunten character(16)[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD istteilvon character(16)[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_address character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_datetime character(20)[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_individualname character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_organisationname character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_phone character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_positionname character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_rationale character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_role character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD statement character varying;
		ALTER TABLE ax_untergeordnetesgewaesser ADD traegtbeizu character(16)[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_untergeordnetesgewaesser ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_747c62de_7ab8_4886_a034_b0201eec47eb ON ax_untergeordnetesgewaesser USING btree (endet);
		CREATE INDEX alkis_0825db75_73fb_45d1_a2eb_77a3d312039c ON ax_untergeordnetesgewaesser USING gin (hatdirektunten);
		CREATE INDEX alkis_fe184775_a077_4a06_8ded_e963669edd39 ON ax_untergeordnetesgewaesser USING gin (istabgeleitetaus);
		CREATE INDEX alkis_01b41756_e4fd_4042_b6dc_01f10db36055 ON ax_untergeordnetesgewaesser USING gin (istteilvon);
		CREATE INDEX alkis_7c92750a_9a48_4b52_8f49_9f5c4394233e ON ax_untergeordnetesgewaesser USING gin (traegtbeizu);

		ALTER TABLE ax_vegetationsmerkmal ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_vegetationsmerkmal ADD bezeichnung character varying;
		ALTER TABLE ax_vegetationsmerkmal ADD breitedesobjekts double precision;
		ALTER TABLE ax_vegetationsmerkmal ADD funktion integer;
		ALTER TABLE ax_vegetationsmerkmal ADD hatdirektunten character(16)[];
		ALTER TABLE ax_vegetationsmerkmal ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_vegetationsmerkmal ADD istteilvon character(16)[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_address character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_datetime character(20)[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_individualname character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_organisationname character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_phone character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_positionname character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_rationale character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_role character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD statement character varying;
		ALTER TABLE ax_vegetationsmerkmal ADD traegtbeizu character(16)[];
		ALTER TABLE ax_vegetationsmerkmal ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_vegetationsmerkmal ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_012bf8c0_0caa_4dc9_87de_a8dec4e65fb5 ON ax_vegetationsmerkmal USING btree (endet);
		CREATE INDEX alkis_21fef265_8ad0_4938_9082_c5110dab0793 ON ax_vegetationsmerkmal USING gin (hatdirektunten);
		CREATE INDEX alkis_bdd4f932_09d6_438a_974b_d2330dd00daf ON ax_vegetationsmerkmal USING gin (istabgeleitetaus);
		CREATE INDEX alkis_e9ec1be4_44a1_4c6a_a765_e614cb89030e ON ax_vegetationsmerkmal USING gin (istteilvon);
		CREATE INDEX alkis_9c9886b5_ca51_419c_ba61_57b7c40c097d ON ax_vegetationsmerkmal USING gin (traegtbeizu);

		ALTER TABLE ax_vertretung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_vertretung ALTER haengtan SET NOT NULL;
		ALTER TABLE ax_vertretung ADD angabenzurvertretung character varying;
		ALTER TABLE ax_vertretung ADD artdervertretung character varying[];
		ALTER TABLE ax_vertretung ADD beginndervertretung date;
		ALTER TABLE ax_vertretung ADD endedervertretung date;
		ALTER TABLE ax_vertretung ADD istteilvon character(16)[];
		ALTER TABLE ax_vertretung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_vertretung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_vertretung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_8a8c8269_6895_47f6_9969_3d795eaa5f36 ON ax_vertretung USING btree (endet);
		CREATE INDEX alkis_32c51271_cafa_4acc_b7e0_b76532b3956f ON ax_vertretung USING btree (gml_id, beginnt);
		CREATE INDEX alkis_f35ba914_2f60_4073_a705_51fa489c3e02 ON ax_vertretung USING btree (haengtan);
		CREATE INDEX alkis_ed1318ee_b553_4740_848a_95d8441965f8 ON ax_vertretung USING gin (beziehtsichauf);
		CREATE INDEX alkis_36443c57_839b_45aa_a0d5_6570ce8339ee ON ax_vertretung USING gin (istteilvon);
		CREATE INDEX alkis_6cd5876b_988e_4e67_af1f_389510d66a45 ON ax_vertretung USING gin (vertritt);

		ALTER TABLE ax_verwaltung ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_verwaltung ALTER haengtan SET NOT NULL;
		ALTER TABLE ax_verwaltung ADD ax_li_processstep_ohnedatenerhebung_description character varying[];
		ALTER TABLE ax_verwaltung ADD beginnderbestellung date;
		ALTER TABLE ax_verwaltung ADD bestellungsbeschluss character varying;
		ALTER TABLE ax_verwaltung ADD endederbestellung date;
		ALTER TABLE ax_verwaltung ADD grundakt character varying;
		ALTER TABLE ax_verwaltung ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_verwaltung ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_verwaltung ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_verwaltung ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_verwaltung ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_verwaltung ADD istteilvon character(16)[];
		ALTER TABLE ax_verwaltung ADD notariat character varying;
		ALTER TABLE ax_verwaltung ADD processstep_address character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_datetime character(20)[];
		ALTER TABLE ax_verwaltung ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_individualname character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_organisationname character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_phone character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_positionname character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_rationale character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_role character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_verwaltung ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_verwaltung ADD statement character varying;
		ALTER TABLE ax_verwaltung ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_verwaltung ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_verwaltung ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_c58d6105_d099_40bc_85e7_2e20d50a5fef ON ax_verwaltung USING btree (endet);
		CREATE INDEX alkis_963197f8_987c_40fe_acbd_41f79968c7e3 ON ax_verwaltung USING btree (gml_id, beginnt);
		CREATE INDEX alkis_80432bab_4461_4c74_983f_ea1636669116 ON ax_verwaltung USING btree (haengtan);
		CREATE INDEX alkis_f50a3b63_14bc_4e12_9919_0a9dd0feb815 ON ax_verwaltung USING gin (beziehtsichauf);
		CREATE INDEX alkis_9d5c5caf_acf8_4e86_b187_7c63cc45fea1 ON ax_verwaltung USING gin (istteilvon);

		ALTER TABLE ax_verwaltungsgemeinschaft ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_verwaltungsgemeinschaft ALTER bezeichnung SET NOT NULL;
		ALTER TABLE ax_verwaltungsgemeinschaft ALTER kreis SET NOT NULL;
		ALTER TABLE ax_verwaltungsgemeinschaft ALTER land SET NOT NULL;
		ALTER TABLE ax_verwaltungsgemeinschaft ALTER schluesselgesamt SET NOT NULL;
		-- ax_verwaltungsgemeinschaft.verwaltungsgemeinschaft => verwaltungsgemeinschaft: integer => character varying
		ALTER TABLE ax_verwaltungsgemeinschaft ALTER verwaltungsgemeinschaft TYPE character varying USING verwaltungsgemeinschaft::varchar;
		ALTER TABLE ax_verwaltungsgemeinschaft ALTER verwaltungsgemeinschaft SET NOT NULL;
		ALTER TABLE ax_verwaltungsgemeinschaft ADD administrativefunktion integer[];
		ALTER TABLE ax_verwaltungsgemeinschaft ADD bestehtaus character(16)[];
		ALTER TABLE ax_verwaltungsgemeinschaft ADD istteilvon character(16)[];
		ALTER TABLE ax_verwaltungsgemeinschaft ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_verwaltungsgemeinschaft ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_verwaltungsgemeinschaft ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_f39424b1_f43f_426b_ab8b_b2f7ef9b6469 ON ax_verwaltungsgemeinschaft USING btree (endet);
		CREATE INDEX alkis_e1604d84_e100_4500_bad4_82ecc1d29542 ON ax_verwaltungsgemeinschaft USING btree (gml_id, beginnt);
		CREATE INDEX alkis_1aec869b_8c49_48a6_be74_aa3fe63245ba ON ax_verwaltungsgemeinschaft USING gin (bestehtaus);
		CREATE INDEX alkis_036218c4_f443_4572_bb0e_518bb5d952ae ON ax_verwaltungsgemeinschaft USING gin (istteilvon);

		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD hatdirektunten character(16)[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD istteilvon character(16)[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD objekthoehe double precision;
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_address character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_datetime character(20)[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_individualname character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_organisationname character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_phone character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_positionname character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_rationale character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_role character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD statement character varying;
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD traegtbeizu character(16)[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_vorratsbehaelterspeicherbauwerk ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_de92ad47_6165_48b6_9509_7986ebd38111 ON ax_vorratsbehaelterspeicherbauwerk USING btree (endet);
		CREATE INDEX alkis_cfba6705_a90c_4800_a38a_fbdf3dbde172 ON ax_vorratsbehaelterspeicherbauwerk USING gin (hatdirektunten);
		CREATE INDEX alkis_09b0b001_36fc_43a6_b1fd_be43b5b7c8da ON ax_vorratsbehaelterspeicherbauwerk USING gin (istabgeleitetaus);
		CREATE INDEX alkis_c749ee2f_2a37_43fc_9c7a_7e927c2c7036 ON ax_vorratsbehaelterspeicherbauwerk USING gin (istteilvon);
		CREATE INDEX alkis_cc265066_cd12_4bd9_9626_b50cc7c75b46 ON ax_vorratsbehaelterspeicherbauwerk USING gin (traegtbeizu);

		ALTER TABLE ax_wald ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_wald ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_wald ADD hatdirektunten character(16)[];
		ALTER TABLE ax_wald ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_wald ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_wald ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_wald ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_wald ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_wald ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_wald ADD istteilvon character(16)[];
		ALTER TABLE ax_wald ADD processstep_address character varying[];
		ALTER TABLE ax_wald ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_wald ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_wald ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_wald ADD processstep_datetime character(20)[];
		ALTER TABLE ax_wald ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_wald ADD processstep_individualname character varying[];
		ALTER TABLE ax_wald ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_wald ADD processstep_organisationname character varying[];
		ALTER TABLE ax_wald ADD processstep_phone character varying[];
		ALTER TABLE ax_wald ADD processstep_positionname character varying[];
		ALTER TABLE ax_wald ADD processstep_rationale character varying[];
		ALTER TABLE ax_wald ADD processstep_role character varying[];
		ALTER TABLE ax_wald ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_wald ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_wald ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_wald ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_wald ADD statement character varying;
		ALTER TABLE ax_wald ADD traegtbeizu character(16)[];
		ALTER TABLE ax_wald ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_wald ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_wald ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_a4e1f3ad_cb2d_4ddb_8e98_92182bfaf008 ON ax_wald USING btree (endet);
		CREATE INDEX alkis_b78a8b34_278d_4ccb_8e5b_7ac874a1631e ON ax_wald USING gin (hatdirektunten);
		CREATE INDEX alkis_457773d3_5bc0_44d9_9908_cde9ccef4e3d ON ax_wald USING gin (istabgeleitetaus);
		CREATE INDEX alkis_b10a76fe_77df_4b90_9f49_9d8f34cfbd5a ON ax_wald USING gin (istteilvon);
		CREATE INDEX alkis_3f475463_06c1_444c_97ae_eba26591ddcf ON ax_wald USING gin (traegtbeizu);

		ALTER TABLE ax_wasserspiegelhoehe ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_wasserspiegelhoehe ALTER hoehedeswasserspiegels SET NOT NULL;
		-- ax_wasserspiegelhoehe.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_wasserspiegelhoehe DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_wasserspiegelhoehe RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_wasserspiegelhoehe','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_wasserspiegelhoehe SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_wasserspiegelhoehe DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_wasserspiegelhoehe_wkb_geometry_idx ON ax_wasserspiegelhoehe USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_wasserspiegelhoehe ADD hatdirektunten character(16)[];
		ALTER TABLE ax_wasserspiegelhoehe ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_wasserspiegelhoehe ADD istteilvon character(16)[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_address character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_datetime character(20)[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_individualname character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_organisationname character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_phone character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_positionname character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_rationale character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_role character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD statement character varying;
		ALTER TABLE ax_wasserspiegelhoehe ADD traegtbeizu character(16)[];
		ALTER TABLE ax_wasserspiegelhoehe ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_wasserspiegelhoehe ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_e14036cb_e8fc_4e6e_81cd_3174b230bddc ON ax_wasserspiegelhoehe USING btree (endet);
		CREATE INDEX alkis_11696f21_6b91_49bc_abfc_78edbf64463a ON ax_wasserspiegelhoehe USING gin (hatdirektunten);
		CREATE INDEX alkis_a7857584_3df2_4d25_a061_7ea50f2f466b ON ax_wasserspiegelhoehe USING gin (istabgeleitetaus);
		CREATE INDEX alkis_8819c8fb_2344_4177_b6c8_77c4df9e27eb ON ax_wasserspiegelhoehe USING gin (istteilvon);
		CREATE INDEX alkis_630279b3_c271_4db7_971c_ee5d13914651 ON ax_wasserspiegelhoehe USING gin (traegtbeizu);

		ALTER TABLE ax_weg ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_weg RENAME name TO zeigtaufexternes_name;
		-- ax_weg.name => zeigtaufexternes_name: character varying => character varying[]
		ALTER TABLE ax_weg ALTER zeigtaufexternes_name TYPE character varying[] USING CASE WHEN zeigtaufexternes_name IS NULL THEN NULL ELSE ARRAY[zeigtaufexternes_name] END;
		ALTER TABLE ax_weg ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_weg ADD hatdirektunten character(16)[];
		ALTER TABLE ax_weg ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_weg ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_weg ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_weg ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_weg ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_weg ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_weg ADD istteilvon character(16)[];
		ALTER TABLE ax_weg ADD processstep_address character varying[];
		ALTER TABLE ax_weg ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_weg ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_weg ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_weg ADD processstep_datetime character(20)[];
		ALTER TABLE ax_weg ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_weg ADD processstep_individualname character varying[];
		ALTER TABLE ax_weg ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_weg ADD processstep_organisationname character varying[];
		ALTER TABLE ax_weg ADD processstep_phone character varying[];
		ALTER TABLE ax_weg ADD processstep_positionname character varying[];
		ALTER TABLE ax_weg ADD processstep_rationale character varying[];
		ALTER TABLE ax_weg ADD processstep_role character varying[];
		ALTER TABLE ax_weg ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_weg ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_weg ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_weg ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_weg ADD statement character varying;
		ALTER TABLE ax_weg ADD traegtbeizu character(16)[];
		ALTER TABLE ax_weg ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_weg ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_5167164d_afc5_4662_8930_e2baaf5c88c8 ON ax_weg USING btree (endet);
		CREATE INDEX alkis_8f4ef0f8_b3b9_4c9e_a225_02c03ddf1960 ON ax_weg USING gin (hatdirektunten);
		CREATE INDEX alkis_64be3114_2960_4b4f_beb7_b2b56d7f14f6 ON ax_weg USING gin (istabgeleitetaus);
		CREATE INDEX alkis_5d311977_50b6_46e4_9d65_77eafbac6fad ON ax_weg USING gin (istteilvon);
		CREATE INDEX alkis_084ac247_05fc_42d1_81a3_09bd008eef16 ON ax_weg USING gin (traegtbeizu);

		ALTER TABLE ax_wegpfadsteig ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_wegpfadsteig ADD befestigung integer;
		ALTER TABLE ax_wegpfadsteig ADD bezeichnung character varying;
		ALTER TABLE ax_wegpfadsteig ADD breitedesverkehrsweges integer;
		ALTER TABLE ax_wegpfadsteig ADD hatdirektunten character(16)[];
		ALTER TABLE ax_wegpfadsteig ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_wegpfadsteig ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_wegpfadsteig ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_wegpfadsteig ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_wegpfadsteig ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_wegpfadsteig ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_wegpfadsteig ADD istteilvon character(16)[];
		ALTER TABLE ax_wegpfadsteig ADD markierung integer;
		ALTER TABLE ax_wegpfadsteig ADD processstep_address character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_datetime character(20)[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_individualname character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_organisationname character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_phone character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_positionname character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_rationale character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_role character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_wegpfadsteig ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_wegpfadsteig ADD statement character varying;
		ALTER TABLE ax_wegpfadsteig ADD strassenschluessel character varying;
		ALTER TABLE ax_wegpfadsteig ADD traegtbeizu character(16)[];
		ALTER TABLE ax_wegpfadsteig ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_wegpfadsteig ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_wegpfadsteig ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_wegpfadsteig ADD zweitname character varying;
		CREATE INDEX alkis_206d1c7c_de8a_4d01_8e87_526d5bf4b45e ON ax_wegpfadsteig USING btree (endet);
		CREATE INDEX alkis_7f545e57_def4_4e50_8017_53cb027fd393 ON ax_wegpfadsteig USING gin (hatdirektunten);
		CREATE INDEX alkis_97f3fa52_64e9_412f_a69d_b55bd3f81ac8 ON ax_wegpfadsteig USING gin (istabgeleitetaus);
		CREATE INDEX alkis_9825a239_8d09_4d3e_9c69_b8a752555053 ON ax_wegpfadsteig USING gin (istteilvon);
		CREATE INDEX alkis_dd494fad_512c_43ac_9d4a_81bd07bafb03 ON ax_wegpfadsteig USING gin (traegtbeizu);

		ALTER TABLE ax_wirtschaftlicheeinheit ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_wirtschaftlicheeinheit ADD bestehtaus character(16)[];
		ALTER TABLE ax_wirtschaftlicheeinheit ADD istteilvon character(16)[];
		ALTER TABLE ax_wirtschaftlicheeinheit ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_wirtschaftlicheeinheit ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_wirtschaftlicheeinheit ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_27ba03e1_1a12_42e7_8bcb_c63daf7c46da ON ax_wirtschaftlicheeinheit USING btree (endet);
		CREATE INDEX alkis_1c5856cf_6b52_4b71_997d_2f3b3de31cd7 ON ax_wirtschaftlicheeinheit USING btree (gml_id, beginnt);
		CREATE INDEX alkis_a83f5929_6259_47b1_8450_4611ca0dd4e0 ON ax_wirtschaftlicheeinheit USING gin (bestehtaus);
		CREATE INDEX alkis_0b8dcdd0_67bc_4276_90f0_755ad2c220de ON ax_wirtschaftlicheeinheit USING gin (istteilvon);

		ALTER TABLE ax_wohnbauflaeche ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_wohnbauflaeche ADD datumderletztenueberpruefung character(20);
		ALTER TABLE ax_wohnbauflaeche ADD hatdirektunten character(16)[];
		ALTER TABLE ax_wohnbauflaeche ADD herkunft_source_source_ax_datenerhebung character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD herkunft_source_source_scaledenominator character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD herkunft_source_source_sourceextent character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD herkunft_source_source_sourcereferencesystem character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD herkunft_source_source_sourcestep character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_wohnbauflaeche ADD istteilvon character(16)[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_address character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_ax_datenerhebung character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_ax_li_processstep_mitdatenerhebung_description character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_contactinstructions character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_datetime character(20)[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_hoursofservice character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_individualname character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_onlineresource character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_organisationname character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_phone character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_positionname character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_rationale character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_role character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_scaledenominator character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_sourceextent character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_sourcereferencesystem character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD processstep_sourcestep character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD statement character varying;
		ALTER TABLE ax_wohnbauflaeche ADD traegtbeizu character(16)[];
		ALTER TABLE ax_wohnbauflaeche ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD zeigtaufexternes_uri character varying[];
		ALTER TABLE ax_wohnbauflaeche ADD zweitname character varying;
		CREATE INDEX alkis_21ca05fa_f292_40c0_8a83_6b7232096251 ON ax_wohnbauflaeche USING btree (endet);
		CREATE INDEX alkis_28b4994b_27bb_4917_b991_0108060ed58f ON ax_wohnbauflaeche USING gin (hatdirektunten);
		CREATE INDEX alkis_86a47a7d_de6b_4dd8_8f5a_9ed021054dcc ON ax_wohnbauflaeche USING gin (istabgeleitetaus);
		CREATE INDEX alkis_06e406a8_f7cb_4776_a47d_3ac28701f94c ON ax_wohnbauflaeche USING gin (istteilvon);
		CREATE INDEX alkis_36f5abb5_9328_467a_b2eb_8265eabc3e72 ON ax_wohnbauflaeche USING gin (traegtbeizu);

		ALTER TABLE ax_wohnplatz ALTER beginnt SET NOT NULL;
		ALTER TABLE ax_wohnplatz ALTER name SET NOT NULL;
		-- ax_wohnplatz.wkb_geometry => wkb_geometry: geometry(Point,25832) => geometry(Geometry,25832)
		BEGIN
			ALTER TABLE ax_wohnplatz DROP CONSTRAINT enforce_geotype_wkb_geometry;
		EXCEPTION WHEN OTHERS THEN
			ALTER TABLE ax_wohnplatz RENAME wkb_geometry TO wkb_geometry_;
			PERFORM AddGeometryColumn('ax_wohnplatz','wkb_geometry',find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'),'GEOMETRY',2);
			UPDATE ax_wohnplatz SET wkb_geometry=wkb_geometry_;
			ALTER TABLE ax_wohnplatz DROP wkb_geometry_ CASCADE;
			CREATE INDEX ax_wohnplatz_wkb_geometry_idx ON ax_wohnplatz USING gist(wkb_geometry);
		END;
		ALTER TABLE ax_wohnplatz ADD bezeichnung character varying[];
		ALTER TABLE ax_wohnplatz ADD einwohnerzahl character varying;
		ALTER TABLE ax_wohnplatz ADD hatdirektunten character(16)[];
		ALTER TABLE ax_wohnplatz ADD istabgeleitetaus character(16)[];
		ALTER TABLE ax_wohnplatz ADD istteilvon character(16)[];
		ALTER TABLE ax_wohnplatz ADD traegtbeizu character(16)[];
		ALTER TABLE ax_wohnplatz ADD zeigtaufexternes_art character varying[];
		ALTER TABLE ax_wohnplatz ADD zeigtaufexternes_name character varying[];
		ALTER TABLE ax_wohnplatz ADD zeigtaufexternes_uri character varying[];
		CREATE INDEX alkis_dda8a583_40fb_492e_942e_5fced40b27c0 ON ax_wohnplatz USING btree (endet);
		CREATE INDEX alkis_d6c698c1_1646_4a56_b19e_139b3abe5cf1 ON ax_wohnplatz USING gin (hatdirektunten);
		CREATE INDEX alkis_675567e1_fbfc_4c21_b097_89ff9dd37f95 ON ax_wohnplatz USING gin (istabgeleitetaus);
		CREATE INDEX alkis_bbe1f5ef_025e_4336_8881_eb1037e663d7 ON ax_wohnplatz USING gin (istteilvon);
		CREATE INDEX alkis_c29cef1e_2cb8_4681_a702_18331d2c97d6 ON ax_wohnplatz USING gin (traegtbeizu);


		CREATE TABLE aa_advstandardmodell (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE aa_advstandardmodell IS 'Alias: "AA_AdVStandardModell", UML-Typ: Enumeration';
		INSERT INTO aa_advstandardmodell (wert,beschreibung,dokumentation) VALUES
		('Basis-DLM', 'BasisLandschaftsModell', ''),
		('DFGM', 'Festpunktmodell', ''),
		('DGM2', 'DigitalesGelaendemodell2', ''),
		('DGM25', 'DigitalesGelaendemodell25', ''),
		('DGM5', 'DigitalesGelaendemodell5', ''),
		('DGM50', 'Digitales Gelaendemodell50', ''),
		('DKKM1000', 'KatasterkartenModell1000', ''),
		('DKKM2000', 'KatasterkartenModell2000', ''),
		('DKKM500', 'KatasterkartenModell500', ''),
		('DKKM5000', 'KatasterkartenModell5000', ''),
		('DLKM', 'LiegenschaftskatasterModell', ''),
		('DLM1000', 'LandschaftsModell1000', ''),
		('DLM250', 'LandschaftsModell250', ''),
		('DLM50', 'LandschaftsModell50', ''),
		('DTK10', 'TopographischeKarte10', ''),
		('DTK100', 'TopographischeKarte100', ''),
		('DTK1000', 'TopographischeKarte1000', ''),
		('DTK25', 'TopographischeKarte25', ''),
		('DTK250', 'TopographischeKarte250', ''),
		('DTK50', 'TopographischeKarte50', '');
		CREATE TABLE aa_nas_ausgabeform (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE aa_nas_ausgabeform IS 'Alias: "AA_NAS_Ausgabeform", UML-Typ: Enumeration';
		INSERT INTO aa_nas_ausgabeform (wert,beschreibung,dokumentation) VALUES
		('application/xml', NULL, ''),
		('application/gzip', NULL, ''),
		('application/zip', NULL, '');
		CREATE TABLE nas_filter_capabilities (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE nas_filter_capabilities IS 'Alias: "NAS_Filter_Capabilities", UML-Typ: Enumeration';
		INSERT INTO nas_filter_capabilities (wert,beschreibung,dokumentation) VALUES
		('PropertyIsOfType', NULL, ''),
		('multiplePropertyValues', NULL, ''),
		('XlinkPropertyPath_leafOnly', NULL, ''),
		('transparentXlinks', NULL, ''),
		('XlinkPropertyPath', NULL, ''),
		('XlinkPropertyName', NULL, '');
		CREATE TABLE aa_themendimension (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE aa_themendimension IS 'Alias: "AA_Themendimension", UML-Typ: Enumeration';
		INSERT INTO aa_themendimension (wert,beschreibung,dokumentation) VALUES
		('1000', 'Punkt-Linien-Thema (Dimension 1)', ''),
		('2000', 'Topologiethema (Dimension 2)', '');
		CREATE TABLE aa_art_themendefinition (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE aa_art_themendefinition IS 'Alias: "AA_Art_Themendefinition", UML-Typ: Enumeration';
		INSERT INTO aa_art_themendefinition (wert,beschreibung,dokumentation) VALUES
		('1000', 'alleObjekte', 'Diese Werteart bedeutet eine zwingende Themenbildung. Dabei sind alle in der Themendefinition genannten Objektarten Bestandteil des Themas und die Objektarten teilen sich stets die Geometrien.'),
		('2000', 'anwenderdefiniert', 'Die anwenderdefinierteThemenbildung wird im Erhebungsprozess gesetzt, wenn aus fachlicher Sicht eine Identität zwischen zwei oder mehreren Objektarten zum Ausdruck gebracht werden soll, wie z. B. zwischen Flurstücksgrenze und Gebäudelinie.');
		CREATE TABLE operation (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE operation IS 'Alias: "Operation", UML-Typ: Enumeration';
		INSERT INTO operation (wert,beschreibung,dokumentation) VALUES
		('Lock', NULL, ''),
		('Insert', NULL, ''),
		('Delete', NULL, ''),
		('Unlock', NULL, ''),
		('Reserve', NULL, ''),
		('Replace', NULL, ''),
		('Query', NULL, '');
		CREATE TABLE ap_horizontaleausrichtung (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ap_horizontaleausrichtung IS 'Alias: "AP_HorizontaleAusrichtung", UML-Typ: Enumeration';
		INSERT INTO ap_horizontaleausrichtung (wert,beschreibung,dokumentation) VALUES
		('linksbündig', NULL, 'Text linksbündig am Textpunkt bzw. am ersten Punkt der Linie.'),
		('rechtsbündig', NULL, 'Text rechtsbündig am Textpunkt bzw. am letzten Punkt der Linie.'),
		('zentrisch', NULL, 'Text zentriert am Textpunkt bzw. in der Mitte der Textstandlinie.');
		CREATE TABLE ap_vertikaleausrichtung (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ap_vertikaleausrichtung IS 'Alias: "AP_VertikaleAusrichtung", UML-Typ: Enumeration';
		INSERT INTO ap_vertikaleausrichtung (wert,beschreibung,dokumentation) VALUES
		('Basis', NULL, 'Textgeometrie bezieht sich auf die Basis- bzw. Grundlinie der Buchstaben.'),
		('Mitte', NULL, 'Textgeometrie bezieht sich auf die Mittellinie der Buchstaben.'),
		('oben', NULL, 'Textgeometrie bezieht sich auf die Oberlinie der Großbuchstaben.');
		CREATE TABLE ap_dateityp_3d (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ap_dateityp_3d IS 'Alias: "AP_DateiTyp_3D", UML-Typ: Enumeration';
		INSERT INTO ap_dateityp_3d (wert,beschreibung,dokumentation) VALUES
		('1000', 'VRML97', ''),
		('2000', '3DSmax', ''),
		('3000', 'autocad DXF', ''),
		('4000', 'autocad DWG', ''),
		('5000', 'shockwave3D', ''),
		('6000', 'X3D', ''),
		('9990', 'unbekannt', '');
		CREATE TABLE ax_artdesnullpunktes_nullpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdesnullpunktes_nullpunkt IS 'Alias: "AX_ArtDesNullpunktes_Nullpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_artdesnullpunktes_nullpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zentraler Nullpunkt', 'Zentraler Nullpunkt ist die Bezeichnung für einen i. d. R. mit O gekennzeichneten Nullpunkt in einem Netzknoten.'),
		('2000', 'Nullpunkt', 'Nullpunkt ist der Anfangs- oder Endpunkt von einem Abschnitt oder Ast.'),
		('3000', 'Fiktiver Nullpunkt', 'Fiktiver Nullpunkt ist ein Nullpunkt, der verhindert, dass zwei verschiedene Äste in einem Nullpunkt beginnen und in einem anderen Nullpunkt wieder zusammenlaufen.');
		CREATE TABLE ax_li_processstep_mitdatenerhebung_description (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_li_processstep_mitdatenerhebung_description IS 'Alias: "AX_LI_ProcessStep_MitDatenerhebung_Description", UML-Typ: Enumeration';
		INSERT INTO ax_li_processstep_mitdatenerhebung_description (wert,beschreibung,dokumentation) VALUES
		('Erhebung', NULL, '');
		CREATE TABLE ax_datenerhebung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_datenerhebung IS 'Alias: "AX_Datenerhebung", UML-Typ: Enumeration';
		INSERT INTO ax_datenerhebung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Aus Katastervermessung ermittelt', ''),
		('1100', 'Aufgrund Anforderungen mit Netzanschluss ermittelt', ''),
		('1200', 'Aufgrund Anforderungen mit Bezug zur Flurstücksgrenze ermittelt', ''),
		('1900', 'Aus sonstiger Vermessung ermittelt', ''),
		('2000', 'Aus Luftbildmessung oder Fernerkundungsdaten ermittelt', ''),
		('4000', 'Aus Katasterunterlagen und Karten für graphische Zwecke ermittelt', ''),
		('4100', 'Aus Katasterzahlen für graphische Zwecke ermittelt', ''),
		('4200', 'Aus Katasterkarten digitalisiert', ''),
		('4210', 'Aus Katasterkarten digitalisiert, Kartenmaßstab M größer gleich 1 zu 1000', ''),
		('4220', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 1000 größer M größer gleich 1 zu 2000', ''),
		('4230', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 2000 größer M größer gleich 1 zu 3000', ''),
		('4240', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 3000 größer M größer gleich 1 zu 5000', ''),
		('4250', 'Aus Katasterkarten  digitalisiert, Kartenmaßstab 1 zu 5000 größer M', ''),
		('4300', 'Aus sonstigen Unterlagen digitalisiert', ''),
		('4310', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab M größer gleich 1 zu 1000', ''),
		('4320', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 1000 größer M größer gleich 1 zu 2000', ''),
		('4330', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 2000 größer M größer gleich 1 zu 3000', ''),
		('4340', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 3000 größer M größer gleich 1 zu 5000', ''),
		('4350', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 5000 größer M', ''),
		('4360', 'Aus sonstigen Unterlagen digitalisiert, mit sonstigen geometrischen Bedingungen und bzw. oder Homogenisierung (M größer gleich 1 zu 1000)', ''),
		('4370', 'Aus sonstigen Unterlagen digitalisiert, mit Berechnung oder Abstandsbedingung (M größer gleich 1 zu 1000)', ''),
		('4380', 'Aus sonstigen Unterlagen digitalisiert, mit sonstigen geometrischen Bedingungen und bzw. oder Homogenisierung (M kleiner 1 zu 1000)', ''),
		('4390', 'Aus sonstigen Unterlagen digitalisiert, mit Berechnung oder Abstandsbedingungen (M kleiner 1 zu 1000)', ''),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung IS 'Alias: "AX_Sportart_BauwerkOderAnlageFuerSportFreizeitUndErholung", UML-Typ: Enumeration';
		INSERT INTO ax_sportart_bauwerkoderanlagefuersportfreizeitunderholung (wert,beschreibung,dokumentation) VALUES
		('1010', 'Ballsport', 'Ballsport bedeutet, dass ein Spielfeld oder Stadion zur Ausübung des Ballsports genutzt wird.'),
		('1011', 'Fußball', 'Fußball bedeutet, dass ein Spielfeld oder Stadion zum Fußball spielen genutzt wird.'),
		('1020', 'Leichtathletik', 'Leichtathletik bedeutet, dass ein Spielfeld oder Stadion zur Ausübung verschiedener Leichtathletikdisziplinen genutzt wird.'),
		('1030', 'Tennis', 'Tennis bedeutet, dass ein Spielfeld oder Stadion zum Tennis spielen genutzt wird.'),
		('1040', 'Reiten', 'Reiten bedeutet, dass ein Stadion oder eine Rennbahn zur  Ausübung des Reitsports genutzt wird.'),
		('1050', 'Schwimmen', 'Schwimmen bedeutet, dass ein Stadion zum  Schwimmen  genutzt wird.'),
		('1060', 'Skisport', 'Skisport bedeutet, dass ein Stadion zur Ausübung des  Skisports genutzt wird.'),
		('1070', 'Eislaufsport, Rollschuhlaufen', 'Eislaufsport, Rollschuhlaufen bedeutet, dass ein Stadion zur  Ausübung des Eislaufsports oder des Rollschuhlaufens genutzt  wird.'),
		('1071', 'Eislaufsport', ''),
		('1072', 'Rollschuhlaufen', ''),
		('1080', 'Skating', 'Skating bedeutet, dass eine Laufbahn zum Skaten genutzt wird.'),
		('1090', 'Motorrennsport', 'Motorrennsport bedeutet, dass eine Rennbahn zur Ausübung  des Motorrennsports genutzt wird.'),
		('1100', 'Radsport', 'Radsport bedeutet, dass ein Stadion oder eine Rennbahn zur  Ausübung des Radsports genutzt wird.'),
		('1110', 'Pferderennsport', 'Pferderennsport bedeutet, dass eine Rennbahn zur Ausübung  des Pferderennsports genutzt wird.'),
		('1115', 'Hunderennsport', 'Hunderennsport bedeutet, dass eine Rennbahn zur Ausübung des Hunderennsports genutzt wird.');
		CREATE TABLE ax_lagezurerdoberflaeche_transportanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagezurerdoberflaeche_transportanlage IS 'Alias: "AX_LageZurErdoberflaeche_Transportanlage", UML-Typ: Enumeration';
		INSERT INTO ax_lagezurerdoberflaeche_transportanlage (wert,beschreibung,dokumentation) VALUES
		('1200', 'Unter der Erdoberfläche', 'Unter der Erdoberfläche bedeutet, dass sich die Transportanlage unter der Erdoberfläche befindet.'),
		('1400', 'Aufgeständert', 'Aufgeständert bedeutet, dass eine Transportanlage durch Tragwerke (Stützen, Pfeiler) über das Niveau der Erdoberfläche geführt wird, um z.B. Hindernisse zu überwinden.'),
		('1700', 'Unter der Wasseroberfläche', 'Unter der Wasseroberfläche bedeutet, dass sich eine Transportanlage unter der Wasseroberfläche befindet.');
		CREATE TABLE ax_produkt_transportanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_produkt_transportanlage IS 'Alias: "AX_Produkt_Transportanlage", UML-Typ: Enumeration';
		INSERT INTO ax_produkt_transportanlage (wert,beschreibung,dokumentation) VALUES
		('1110', 'Erdöl', 'Erdöl ist ein flüssiges und brennbares Kohlenwasserstoffgemisch, das gefördert oder transportiert wird.'),
		('1120', 'Gas', 'Gas ist eine gasförmige und brennbare Materie, die gefördert oder transportiert wird.'),
		('1130', 'Wasser', 'Wasser ist die chemische Verbindung von Wasserstoff mit Sauerstoff, die gefördert oder transportiert wird.'),
		('1131', 'Trinkwasser', 'Trinkwasser ist ein für den menschlichen Genuss und Gebrauch geeignetes Wasser, das transportiert wird.'),
		('1132', 'Brauchwasser', 'Brauchwasser ist ein für technische u.a. Zwecke verwendetes Wasser, das transportiert wird.'),
		('1133', 'Abwasser', 'Abwasser ist ein verunreinigtes Wasser aus Haushaltungen, Gewerbe- und Industriebetrieben sowie Niederschlagswasser, das transportiert wird.'),
		('1140', 'Fernwärme', '');
		CREATE TABLE ax_bauwerksfunktion_turm (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_turm IS 'Alias: "AX_Bauwerksfunktion_Turm", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_turm (wert,beschreibung,dokumentation) VALUES
		('1001', 'Wasserturm', 'Wasserturm ist ein hochgelegenes Bauwerk mit einem Behälter, in dem Wasser für die Wasserversorgung und Konstanthaltung des Wasserdruckes gespeichert wird.'),
		('1002', 'Kirchturm, Glockenturm', 'Kirchturm, Glockenturm ist ein freistehender Turm, der die Glockenstube mit den Glocken aufnimmt.'),
		('1003', 'Aussichtsturm', 'Aussichtsturm ist ein Bauwerk, das ausschließlich der Fernsicht dient.'),
		('1004', 'Kontrollturm', 'Kontrollturm (Tower) ist ein Bauwerk auf dem Fluggelände, in dem die für die Lenkung und Überwachung des Flugverkehrs erforderlichen Anlagen und Einrichtungen untergebracht sind.'),
		('1005', 'Kühlturm', 'Kühlturm ist eine turmartige Kühlanlage (Nass- oder Trockenkühlturm), in der erwärmtes Kühlwasser insbesondere von Kraftwerken rückgekühlt wird.'),
		('1006', 'Leuchtturm', 'Leuchtturm ist ein als Schifffahrtszeichen dienender hoher Turm, ausgerüstet mit einem starken Leuchtfeuer verschiedener Kennungen an der Turmspitze und mit anderen, der Schifffahrt dienenden Signalen.'),
		('1007', 'Feuerwachturm', 'Feuerwachturm ist ein Turm, der zum Erkennen von Gefahren (Feuer) dient.'),
		('1008', 'Sende-, Funkturm, Fernmeldeturm', 'Sende-, Funkturm, Fernmeldeturm ist ein Bauwerk, ausgerüstet mit Sende - und Empfangsantennen zum Übertragen und Empfangen von Nachrichten aller Arten von Telekommunikation.'),
		('1009', 'Stadt-, Torturm', 'Stadtturm ist ein historischer Turm, der das Stadtbild prägt. Torturm ist der auf einem Tor stehende Turm, wobei das Tor allein stehen oder in eine Befestigungsanlage eingebunden sein kann.'),
		('1010', 'Förderturm', 'Förderturm ist ein Turm über einem Schacht. An Förderseile, die über Seilscheiben im Turm geführt werden, werden Lasten in den Schacht gesenkt oder aus dem Schacht gehoben.'),
		('1011', 'Bohrturm', 'Bohrturm ist ein zur Gewinnung von Erdöl und Erdgas verwendetes, meist aus einer Stahlkonstruktion bestehendes Gerüst, in dem das Bohrgestänge aufgehängt ist.'),
		('1012', 'Schloss-, Burgturm', 'Schloss-, Burgturm ist ein Turm innerhalb einer Schloss- bzw. einer Burganlage, auch Bergfried genannt.'),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', 'Nach Quellenlage nicht zu spezifizieren bedeutet, dass zum Zeitpunkt der Erhebung keine Funktion zuweisbar war.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Funktion bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri IS 'Alias: "AX_HydrologischesMerkmal_SonstigesBauwerkOderSonstigeEinrichtung", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_sonstigesbauwerkodersonstigeeinri (wert,beschreibung,dokumentation) VALUES
		('1000', 'Ständig Wasser führend', 'Ständig Wasser führend heißt, dass der Brunnen ganzjährig Wasser führt.'),
		('2000', 'Nicht ständig Wasser führend', 'Nicht ständig Wasser führend heißt, dass der Brunnen nicht ganzjährig Wasser führt.'),
		('3000', 'Trocken, versiegt', 'Trocken, versiegt heißt, dass der Brunnen ganzjährig kein Wasser führt.');
		CREATE TABLE ax_zustand_turm (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_turm IS 'Alias: "AX_Zustand_Turm", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_turm (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich der Turm nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('2200', 'Verfallen, zerstört', 'Verfallen, zerstört bedeutet, dass sich der ursprüngliche Zustand des Turmes durch menschliche oder zeitliche Einwirkungen so verändert hat, dass eine Nutzung nicht mehr möglich ist.');
		CREATE TABLE ax_art_heilquellegasquelle (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_heilquellegasquelle IS 'Alias: "AX_Art_HeilquelleGasquelle", UML-Typ: Enumeration';
		INSERT INTO ax_art_heilquellegasquelle (wert,beschreibung,dokumentation) VALUES
		('4010', 'Heilquelle', ''),
		('4020', 'Gasquelle, Mofette', '');
		CREATE TABLE ax_bauwerksfunktion_transportanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_transportanlage IS 'Alias: "AX_Bauwerksfunktion_Transportanlage", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_transportanlage (wert,beschreibung,dokumentation) VALUES
		('1101', 'Rohrleitung, Pipeline', 'Rohrleitung, Pipeline ist ein langgestreckter Hohlkörper zum Transport von Flüssigkeiten und Gasen.'),
		('1102', 'Förderband, Bandstraße', 'Förderband, Bandstraße ist ein mechanisch bewegtes Band zum Transport von Gütern.'),
		('1103', 'Pumpe', 'Pumpe ist eine Vorrichtung zum An-, Absaugen oder Injizieren von Flüssigkeiten oder Gasen; Verdichtungsstation  für Gase.');
		CREATE TABLE ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk IS 'Alias: "AX_LageZurErdoberflaeche_VorratsbehaelterSpeicherbauwerk", UML-Typ: Enumeration';
		INSERT INTO ax_lagezurerdoberflaeche_vorratsbehaelterspeicherbauwerk (wert,beschreibung,dokumentation) VALUES
		('1200', 'Unter der Erdoberfläche', 'Unter der Erdoberfläche bedeutet, dass sich der Vorratsbehälter bzw. das Speicherbauwerk unter der Erdoberfläche befindet.'),
		('1400', 'Aufgeständert', 'Aufgeständert bedeutet, dass der Vorratsbehälter bzw. das  Speicherbauwerk auf Stützen steht.');
		CREATE TABLE ax_speicherinhalt_vorratsbehaelterspeicherbauwerk (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_speicherinhalt_vorratsbehaelterspeicherbauwerk IS 'Alias: "AX_Speicherinhalt_VorratsbehaelterSpeicherbauwerk", UML-Typ: Enumeration';
		INSERT INTO ax_speicherinhalt_vorratsbehaelterspeicherbauwerk (wert,beschreibung,dokumentation) VALUES
		('1000', 'Erze', 'Erze bedeutet, dass die in der Natur vorkommenden, metallhaltigen Mineralien und Mineralgemische gelagert werden.'),
		('1100', 'Treib- und Brennstoffe', 'Treib- und Brennstoffe bedeutet, dass die in der Natur vorkommenden brennbaren organischen und anorganischen Substanzen gelagert werden.'),
		('1110', 'Erdöl', 'Erdöl ist ein flüssiges und brennbares Kohlenwasserstoffgemisch, das gelagert wird.'),
		('1120', 'Gas', 'Gas ist eine gasförmige und brennbare Materie, die  gespeichert wird.'),
		('1130', 'Wasser', 'Wasser ist die chemische Verbindung von Sauerstoff und Wasserstoff, die gespeichert wird.'),
		('1140', 'Gülle', ''),
		('1200', 'Baustoffe', 'Baustoffe sind sämtliche im Bauwesen verwendete Materialien, die gelagert werden.'),
		('1300', 'Chemikalien', 'Chemikalien sind Werkstoffe organischen oder anorganischen Ursprungs, die gespeichert werden.'),
		('1500', 'Lebensmittel', 'Lebensmittel sind alle Nahrungs- und Genussmittel für Menschen, die in Vorratsbehältern zwischengelagert werden.'),
		('1501', 'Getreide', 'Getreide sind die trockenen Kornfrüchte (Weizen, Roggen, Gerste, Hafer), die in Vorratsbehältern zwischengelagert werden.'),
		('1510', 'Futtermittel', 'Futtermittel sind die als Tiernahrung dienenden pflanzlichen oder tierischen Stoffe, die in Vorratsbehältern zwischengelagert werden.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass der Speicherinhalt bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer IS 'Alias: "AX_Bauwerksfunktion_BauwerkOderAnlageFuerIndustrieUndGewerbe", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_bauwerkoderanlagefuerindustrieundgewer (wert,beschreibung,dokumentation) VALUES
		('1210', 'Klärbecken', 'Klärbecken ist ein künstlich errichtetes Becken oder eine Geländevertiefung, in der Feststoffe aus einer Flüssigkeit ausgefällt werden.'),
		('1215', 'Biogasanlage', ''),
		('1220', 'Windrad', 'Windrad ist ein mit Flügeln besetztes Rad, das durch Wind in Rotation versetzt wird und mit Hilfe eines eingebauten  Generators elektrische Energie erzeugt.'),
		('1230', 'Solarzellen', 'Solarzellen sind Flächenelemente aus Halbleitern, die die Energie der Sonnenstrahlen in elektrische Energie umwandeln.'),
		('1240', 'Wasserrad', 'Wasserrad ist ein mit Schaufeln oder Zellen besetztes Rad, das die Energie des strömenden Wassers zum Antrieb,  besonders von Mühlen, ausnutzt oder zum Schöpfen von  Wasser (Schöpfrad) genutzt wird.'),
		('1250', 'Mast', 'Mast ist eine senkrecht stehende Konstruktion mit stützender oder tragender Funktion.'),
		('1251', 'Freileitungsmast', 'Freileitungsmast ist ein Mast, an dem Hochspannungsleitungen befestigt sind.'),
		('1260', 'Funkmast', 'Funkmast ist ein Mast mit Vorrichtungen zum Empfangen, Umformen und Weitersenden von elektromagnetischen Wellen.'),
		('1270', 'Antenne', 'Antenne ist eine Vorrichtung zum Empfang oder zur Ausstrahlung elektromagnetischer Wellen.'),
		('1280', 'Radioteleskop', 'Radioteleskop ist ein Bauwerk mit einer Parabolantenne für den Empfang von elektromagnetischer Strahlung aus dem Weltall.'),
		('1290', 'Schornstein, Schlot, Esse', 'Schornstein, Schlot, Esse ist ein freistehend senkrecht  hochgeführter Abzugskanal für die Rauchgase einer  Feuerungsanlage oder für andere Abgase.'),
		('1310', 'Stollenmundloch', 'Stollenmundloch ist der Eingang eines unterirdischen Gangs, der annähernd horizontal von der Erdoberfläche in das Gebirge führt.'),
		('1320', 'Schachtöffnung', 'Schachtöffnung ist der Eingang auf der Erdoberfläche zu einem Schacht.'),
		('1330', 'Kran', 'Kran ist eine Vorrichtung, die aus einer fahrbaren oder ortsfesten Konstruktion besteht und die zum Heben von Lasten benutzt wird.'),
		('1331', 'Drehkran', ''),
		('1332', 'Portalkran', ''),
		('1333', 'Laufkran, Brückenlaufkran', ''),
		('1340', 'Trockendock', 'Trockendock ist eine Anlage in Werften und Häfen, in der das Schiff zum Ausbessern aus dem Wasser genommen wird.'),
		('1350', 'Hochofen', 'Hochofen ist ein hoher Schachtofen zum Schmelzen von Eisenerz.'),
		('1360', 'Merkzeichen, Merkstein', ''),
		('1370', 'Hydrant', ''),
		('1371', 'Oberflurhydrant', ''),
		('1372', 'Unterflurhydrant', ''),
		('1380', 'Schieberkappe', ''),
		('1390', 'Einsteigeschacht', ''),
		('1400', 'Umformer', ''),
		('1700', 'Bergbaubetrieb', ''),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Bauwerkfunktion bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_art_einrichtunginoeffentlichenbereichen (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_einrichtunginoeffentlichenbereichen IS 'Alias: "AX_Art_EinrichtungInOeffentlichenBereichen", UML-Typ: Enumeration';
		INSERT INTO ax_art_einrichtunginoeffentlichenbereichen (wert,beschreibung,dokumentation) VALUES
		('1100', 'Kommunikationseinrichtung', ''),
		('1110', 'Fernsprechhäuschen', ''),
		('1120', 'Briefkasten', ''),
		('1130', 'Notrufeinrichtung', ''),
		('1140', 'Feuermelder', ''),
		('1150', 'Polizeirufsäule', ''),
		('1200', 'Kabelkasten, Schaltkasten', ''),
		('1300', 'Verkehrszeichen', ''),
		('1310', 'Verkehrsampel', ''),
		('1320', 'Freistehende Hinweistafel, -zeichen', ''),
		('1330', 'Wegweiser von besonderer Bedeutung', ''),
		('1340', 'Freistehende Warntafel', ''),
		('1350', 'Bushaltestelle', ''),
		('1400', 'Markierungshinweise, -steine', ''),
		('1410', 'Kilometerstein, -tafel', 'Kilometerstein, -tafel ist ein Punkt mit einem festen Wert im Netz der Autobahnen oder Schienenbahnen der in der Örtlichkeit durch eine Markierung (z. B. Kilometerstein) repräsentiert wird.'),
		('1420', 'Ortsdurchfahrtsstein', ''),
		('1430', 'Fischereigrenzstein', ''),
		('1500', 'Bahnübergang, Schranke', ''),
		('1510', 'Tor', ''),
		('1600', 'Laterne, Kandelaber', ''),
		('1610', 'Gaslaterne', ''),
		('1620', 'Laterne, elektrisch', ''),
		('1630', 'Gaskandelaber', ''),
		('1640', 'Kandelaber, elektrisch', ''),
		('1650', 'Hängende Lampe', ''),
		('1700', 'Säule, Werbefläche', ''),
		('1710', 'Leuchtsäule', ''),
		('1910', 'Fahnenmast', ''),
		('2100', 'Straßensinkkasten', ''),
		('2200', 'Müllbox', ''),
		('2300', 'Kehrichtgrube', ''),
		('2400', 'Uhr', ''),
		('2500', 'Richtscheinwerfer', ''),
		('2600', 'Flutlichtmast', ''),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Art bekannt, aber in der Attributwertliste nicht aufgeführt ist.');
		CREATE TABLE ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde IS 'Alias: "AX_Bauwerksfunktion_BauwerkOderAnlageFuerSportFreizeitUndErholung", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_bauwerkoderanlagefuersportfreizeitunde (wert,beschreibung,dokumentation) VALUES
		('1410', 'Spielfeld', 'Spielfeld ist eine abgegrenzte, markierte Fläche, auf der die Sportart unmittelbar ausgeübt wird, z.B. die einzelnen  Fußballfelder (Hauptplatz und Trainingsplätze) einer größeren Anlage. Die zusammenhängenden Spielflächen innerhalb einer Tennisanlage werden zu einem Spielfeld  zusammengefasst'),
		('1411', 'Hartplatz', ''),
		('1412', 'Rasenplatz', ''),
		('1420', 'Rennbahn, Laufbahn, Geläuf', 'Rennbahn, Laufbahn, Geläuf ist eine je nach Art des  Rennens verschiedenartig gestaltete Strecke (oval, gerade,  kurvig), auf der das Rennen stattfindet.'),
		('1430', 'Zuschauertribüne', 'Zuschauertribüne ist ein großes Gerüst oder ein festes, meist überdachtes Bauwerk mit ansteigenden Sitz- oder Stehplatzreihen für Zuschauer.'),
		('1431', 'Zuschauertribüne, überdacht', 'Zuschauertribüne, überdacht bedeutet, dass Zuschauertribüne mit einer Dachfläche ausgestattet ist.'),
		('1432', 'Zuschauertribüne, nicht überdacht', 'Zuschauertribüne, nicht überdacht bedeutet, dass die Zuschauertribüne keine Dachfläche besitzt.'),
		('1440', 'Stadion', 'Stadion ist ein Bauwerk mit Tribünen und entsprechenden  Einrichtungen zur Ausübung von bestimmten Sportarten.'),
		('1450', 'Schwimmbecken', 'Schwimmbecken ist ein mit Wasser gefülltes Becken zum Schwimmen oder Baden.'),
		('1460', 'Liegewiese', ''),
		('1470', 'Sprungschanze (Anlauf)', 'Sprungschanze (Anlauf) ist eine Anlage zum Skispringen mit einer stark abschüssigen, in einem Absprungtisch endenden Bahn zum Anlauf nehmen.'),
		('1480', 'Schießanlage', 'Schießanlage ist eine Anlage mit Schießbahnen für  Schießübungen oder sportliche Wettbewerbe.'),
		('1490', 'Gradierwerk', 'Gradierwerk ist ein mit Reisig bedecktes Gerüst, über das Sole rieselt, die durch erhöhte Verdunstung konzentriert wird.'),
		('1510', 'Wildgehege', 'Wildgehege ist ein eingezäuntes Areal, in dem Wild waidgerecht betreut wird oder beobachtet werden kann.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Bauwerkfunktion bekannt, aber in der Attributwertliste nicht aufgeführt ist.');
		CREATE TABLE ax_archaeologischertyp_historischesbauwerkoderhistorischee (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_archaeologischertyp_historischesbauwerkoderhistorischee IS 'Alias: "AX_ArchaeologischerTyp_HistorischesBauwerkOderHistorischeEinrichtung", UML-Typ: Enumeration';
		INSERT INTO ax_archaeologischertyp_historischesbauwerkoderhistorischee (wert,beschreibung,dokumentation) VALUES
		('1000', 'Grab', 'Grab ist eine künstlich geschaffene Bestattungsstätte unter, auf oder über der  Erdoberfläche.'),
		('1010', 'Großsteingrab (Dolmen, Hünenbett)', 'Großsteingrab (Dolmen, Hünenbett) ist ein Grab mit Steineinbau, d. h. es ist ein aus großen Steinen (z.B. Findlingen) errichteter Grabbau.'),
		('1020', 'Grabhügel (Hügelgrab)', 'Grabhügel (Hügelgrab) ist ein meist runder oder ovaler Hügel, der über einer ur- oder frühgeschichtlichen Bestattung aus Erde aufgeschüttet oder aus Plaggen aufgeschichtet  wurde.'),
		('1100', 'Historische Wasserleitung', 'Historische Wasserleitung ist ein meist offenes System von Gräben, Kunstgräben und Kanälen, in dem Wasser transportiert wird.'),
		('1110', 'Aquädukt', 'Aquädukt ist ein brückenartiges Steinbauwerk zur Überführung von Freispiegel-Wasserleitungen mit  natürlichem Gefälle über Täler oder andere Bodenunebenheiten.'),
		('1200', 'Befestigung (Wall, Graben)', 'Befestigung (Wall, Graben) ist ein aus Erde aufgeschütteter Grenz-, Schutz- oder Stadtwall. Zu der Befestigung (Wall) zählen auch Limes und Landwehr.'),
		('1210', 'Wachturm (römisch), Warte', 'Wachtturm (römisch), Warte ist ein allein oder in Verbindung mit einem Befestigungssystem (Limes) stehender Beobachtungsturm.'),
		('1300', 'Steinmal', 'Steinmal ist eine kultische oder rechtliche Kennzeichnung, bestehend aus einzelnen oder Gruppen von Steinen.'),
		('1400', 'Befestigung (Burgruine)', 'Befestigung (Burgruine) ist eine künstliche Anlage zur Sicherung von Leben und Gut.'),
		('1410', 'Burg (Fliehburg, Ringwall)', 'Burg (Fliehburg, Ringwall) ist eine ur- oder frühgeschichtliche runde, ovale oder an Gegebenheiten des Geländes (Böschungskanten) angepasste Befestigungsanlage, die aus einem Erdwall mit oder ohne Holzeinbauten besteht.'),
		('1420', 'Schanze', 'Schanze ist eine mittelalterliche oder neuzeitliche, in der Regel geschlossene, quadratische, rechteckige oder sternförmige Wallanlage mit Außengraben.'),
		('1430', 'Lager', 'Lager ist die Bezeichnung für ein befestigtes Truppenlager in der Römer- oder in der Neuzeit (z.B. bei Belagerungen im 30 jährigen Krieg).'),
		('1500', 'Historische Mauer', 'Historische Mauer ist eine Mauer mit kulturgeschichtlicher Bedeutung.'),
		('1510', 'Stadtmauer', ''),
		('1520', 'Sonstige historische Mauer', ''),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass der archäologische Typ bekannt, aber nicht in der Attributwertliste aufgeführt ist');
		CREATE TABLE ax_hydrologischesmerkmal_heilquellegasquelle (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_heilquellegasquelle IS 'Alias: "AX_HydrologischesMerkmal_HeilquelleGasquelle", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_heilquellegasquelle (wert,beschreibung,dokumentation) VALUES
		('4000', 'Ständig schüttend', ''),
		('5000', 'Nicht ständig schüttend', '');
		CREATE TABLE ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe IS 'Alias: "AX_Zustand_BauwerkOderAnlageFuerIndustrieUndGewerbe", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_bauwerkoderanlagefuerindustrieundgewerbe (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich Bauwerk oder Anlage für Industrie und Gewerbe nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('2200', 'Verfallen, zerstört', 'Verfallen, zerstört bedeutet, dass sich der ursprüngliche Zustand von Bauwerk oder Anlage für Industrie und Gewerbe durch menschliche oder zeitliche Einwirkungen so verändert hat, dass eine Nutzung nicht mehr möglich ist.'),
		('4100', 'Offen', 'Offen bedeutet, dass Bauwerk oder Anlage für Industrie und Gewerbe allgemein zugänglich ist.'),
		('4200', 'Verschlossen', 'Verschlossen bedeutet, dass Bauwerk oder Anlage für Industrie und Gewerbe nicht allgemein zugänglich ist.');
		CREATE TABLE ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun IS 'Alias: "AX_Bauwerksfunktion_SonstigesBauwerkOderSonstigeEinrichtung", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_sonstigesbauwerkodersonstigeeinrichtun (wert,beschreibung,dokumentation) VALUES
		('1610', 'Überdachung', ''),
		('1611', 'Carport', ''),
		('1620', 'Treppe', 'Treppe ist ein stufenförmiges Bauwerk zur Überwindung von Höhenunterschieden.'),
		('1621', 'Freitreppe', ''),
		('1622', 'Rolltreppe', ''),
		('1630', 'Treppenunterkante', ''),
		('1640', 'Kellereingang', 'Kellereingang ist der Eingang zu einem unterirdischen Vorratsraum außerhalb von Gebäuden.'),
		('1641', 'Kellereingang, offen', 'Kellereingang, offen ist der offene Eingang zu einem unterirdischen Vorratsraum außerhalb von Gebäuden.'),
		('1642', 'Kellereingang, geschlossen', 'Kellereingang, geschlossen ist der geschlossene Eingang zu einem unterirdischen Vorratsraum außerhalb von Gebäuden.'),
		('1650', 'Rampe', ''),
		('1670', 'Terrasse', 'Es werden nur unterkellerte Terrassen erfasst.'),
		('1700', 'Mauer', 'Mauer ist ein freistehendes, langgestrecktes Bauwerk, das aus Natur- bzw. Kunststeinen oder anderen Materialien besteht.'),
		('1701', 'Mauerkante, rechts', ''),
		('1702', 'Mauerkante, links', ''),
		('1703', 'Mauermitte', ''),
		('1720', 'Stützmauer', 'Stützmauer ist eine zum Stützen von Erdreich dienende Mauer.'),
		('1721', 'Stützmauer, rechts', ''),
		('1722', 'Stützmauer, links', ''),
		('1723', 'Stützmauermitte', ''),
		('1740', 'Zaun', 'Zaun ist eine Abgrenzung oder Einfriedung aus Holz- oder Metallstäben oder aus Draht bzw. Drahtgeflecht.'),
		('1750', 'Gedenkstätte, Denkmal, Denkstein, Standbild', 'Gedenkstätte, Denkmal, Denkstein, Standbild ist ein zum Gedenken errichtete Anlage oder Bauwerk an eine Person, ein Ereignis oder eine plastische Darstellung.'),
		('1760', 'Bildstock, Wegekreuz, Gipfelkreuz', 'Bildstock, Wegekreuz, Gipfelkreuz ist ein frei stehendes Mal aus Holz oder Stein, das in einem tabernakelartigen Aufbau ein Kruzifix oder eine Heiligendarstellung enthält und als Andachtsbild,  als Erinnerung an Verstorbene oder als Sühnemal errichtet wurde;  ist ein errichtetes Kreuz z.B. an Wegen; ist ein Kreuz auf dem Gipfel eines Berges.'),
		('1761', 'Bildstock', ''),
		('1762', 'Wegekreuz', ''),
		('1763', 'Gipfelkreuz', ''),
		('1770', 'Meilenstein, historischer Grenzstein', 'Meilenstein, historischer Grenzstein sind Steine von kulturgeschichtlicher  Bedeutung, die als Meilensteine (Entfernungsangaben in Meilen) am Rande einer Straße aufgestellt sind oder als Grenzsteine vergangene Eigentumsverhältnisse dokumentieren.'),
		('1780', 'Brunnen', 'Brunnen ist eine Anlage zur Gewinnung von Grundwasser bzw. ein architektonisch ausgestaltetes Bauwerk mit Becken zum Auffangen von Wasser.'),
		('1781', 'Brunnen (Trinkwasserversorgung)', 'Brunnen (Trinkwasserversorgung) bedeutet, dass in dem Brunnen ausschließlich Trinkwasser gewonnen wird.'),
		('1782', 'Springbrunnen, Zierbrunnen', ''),
		('1783', 'Ziehbrunnen', ''),
		('1790', 'Spundwand', 'Spundwand ist ein Sicherungsbauwerk (wasserdichte Wand) aus miteinander verbundenen schmalen, langen Holz-, Stahl- oder Stahlbetonbohlen zum Schutz gegen das Außenwasser. Die Bohlen werden horizontal hinter Pfählen (Bohlwand) oder vertikal als Spundwand eingebaut und meist rückwärtig verankert.'),
		('1791', 'Höckerlinie', 'Höckerlinie bezeichnet die ehemalige Panzersperre Westwall.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Bauwerksfunktion bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_funktion_bauwerk (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_bauwerk IS 'Alias: "AX_Funktion_Bauwerk", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_bauwerk (wert,beschreibung,dokumentation) VALUES
		('1000', 'Hochwasser-, Sturmflutschutz', 'Hochwasser-, Sturmflutschutz bedeutet, dass das Bauwerk dem Schutz vor Hochwasser bzw. Sturmflut dient.'),
		('2000', 'Lärmschutz', 'Lärmschutz bedeutet, dass das Bauwerk dem Schutz vor Lärmemissionen dient.');
		CREATE TABLE ax_bauwerksfunktion_leitung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_leitung IS 'Alias: "AX_Bauwerksfunktion_Leitung", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_leitung (wert,beschreibung,dokumentation) VALUES
		('1110', 'Freileitung', 'Freileitung ist eine aus einem oder mehreren Drähten oder  Fasern hergestellte oberirdische Leitung zum Transport von  elektrischer Energie und zur Übertragung von elektrischen  Signalen.'),
		('1111', 'Erdkabel', '');
		CREATE TABLE ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk IS 'Alias: "AX_Bauwerksfunktion_VorratsbehaelterSpeicherbauwerk", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_vorratsbehaelterspeicherbauwerk (wert,beschreibung,dokumentation) VALUES
		('1201', 'Silo', 'Silo ist ein Großraumbehälter zum Speichern von Schüttgütern (Getreide, Erz, Zement, Sand) oder Gärfutter (gehäckseltes Grüngut).'),
		('1202', 'Fülltrichter', ''),
		('1203', 'Bunker', 'Bunker ist ein Bauwerk, in dem Schüttgut gelagert wird.'),
		('1204', 'Getreideheber', ''),
		('1205', 'Tank', 'Tank ist ein Behälter, in dem Flüssigkeiten gelagert oder Gase gespeichert werden.'),
		('1206', 'Gasometer', 'Gasometer ist ein Vorratsbehälter für Gas.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Bauwerkfunktion bekannt, aber  in der Attributwertliste nicht aufgeführt ist.');
		CREATE TABLE ax_befestigung_wegpfadsteig (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_befestigung_wegpfadsteig IS 'Alias: "AX_Befestigung_WegPfadSteig", UML-Typ: Enumeration';
		INSERT INTO ax_befestigung_wegpfadsteig (wert,beschreibung,dokumentation) VALUES
		('1000', 'Befestigt', 'Befestigt bedeutet, dass Weg, Pfad, Steig mit einem festen Unterbau versehen und ganzjährig befahrbar bzw. begehbar ist.'),
		('2000', 'Unbefestigt', 'Unbefestigt bedeutet, dass Weg, Pfad, Steig nicht mit einem festen Unterbau versehen und nicht ganzjährig befahrbar bzw. begehbar ist.');
		CREATE TABLE ax_oberflaechenmaterial_flugverkehrsanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_oberflaechenmaterial_flugverkehrsanlage IS 'Alias: "AX_Oberflaechenmaterial_Flugverkehrsanlage", UML-Typ: Enumeration';
		INSERT INTO ax_oberflaechenmaterial_flugverkehrsanlage (wert,beschreibung,dokumentation) VALUES
		('1210', 'Gras, Rasen', 'Gras, Rasen bedeutet, dass die Oberfläche von Flugverkehrsanlage mit Gras bewachsen ist.'),
		('1220', 'Beton', 'Beton bedeutet, dass die Oberfläche von Flugverkehrsanlage aus Beton besteht.'),
		('1230', 'Bitumen, Asphalt', 'Bitumen, Asphalt bedeutet, dass die Oberfläche von Flugverkehrsanlage aus Bitumen bzw. Asphalt besteht.');
		CREATE TABLE ax_art_gleis (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_gleis IS 'Alias: "AX_Art_Gleis", UML-Typ: Enumeration';
		INSERT INTO ax_art_gleis (wert,beschreibung,dokumentation) VALUES
		('1200', 'Drehscheibe', '');
		CREATE TABLE ax_bahnkategorie_gleis (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bahnkategorie_gleis IS 'Alias: "AX_Bahnkategorie_Gleis", UML-Typ: Enumeration';
		INSERT INTO ax_bahnkategorie_gleis (wert,beschreibung,dokumentation) VALUES
		('1100', 'Eisenbahn', ''),
		('1102', 'Güterverkehr', ''),
		('1104', 'S-Bahn', ''),
		('1200', 'Stadtbahn', ''),
		('1201', 'Straßenbahn', ''),
		('1202', 'U-Bahn', ''),
		('1300', 'Bergbahn, Seilbahn', ''),
		('1301', 'Zahnradbahn', ''),
		('1302', 'Standseilbahn', ''),
		('1400', 'Museumsbahn', ''),
		('1500', 'Bahn im Freizeitpark', ''),
		('1600', 'Magnetschwebebahn', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_art_strassenverkehrsanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_strassenverkehrsanlage IS 'Alias: "AX_Art_Strassenverkehrsanlage", UML-Typ: Enumeration';
		INSERT INTO ax_art_strassenverkehrsanlage (wert,beschreibung,dokumentation) VALUES
		('1000', 'Fahrbahn', ''),
		('1010', 'Fahrbahnbegrenzungslinie', ''),
		('1011', 'Fahrbahnbegrenzungslinie, überdeckt', ''),
		('2000', 'Furt', 'Furt ist eine zum Überqueren geeignete Stelle in einem Gewässer.'),
		('3000', 'Autobahnknoten', 'Autobahnknoten ist ein höhengleicher oder höhenungleicher Knoten, der sich aus der verkehrlichen Verknüpfung zweier Autobahnen sowie an Anschlussstellen mit dem nachgeordneten Straßennetz ergibt.'),
		('3001', 'Kreuz', 'Kreuz ist ein vierarmiger Knotenpunkt in mehreren Ebenen in dem sich zwei Autobahnen kreuzen.'),
		('3002', 'Dreieck', 'Dreieck ist eine Einmündung einer Autobahn in eine durchgehende Autobahn.'),
		('3003', 'Anschlussstelle', 'Anschlussstelle ist die verkehrliche Verknüpfung der Autobahn mit dem nachgeordneten Straßennetz.'),
		('4000', 'Platz', 'Platz ist eine ebene, befestigte oder unbefestigte Fläche.'),
		('5330', 'Raststätte', 'Raststätte ist eine Anlage an Verkehrsstraßen mit Bauwerken und Einrichtungen zur Versorgung und 
		Erholung von Reisenden.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Art bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_markierung_wegpfadsteig (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_markierung_wegpfadsteig IS 'Alias: "AX_Markierung_WegPfadSteig", UML-Typ: Enumeration';
		INSERT INTO ax_markierung_wegpfadsteig (wert,beschreibung,dokumentation) VALUES
		('1401', 'Gekennzeichneter Wanderweg', 'Gekennzeichneter Wanderweg ist ein als Wanderweg eindeutig markierter Weg.'),
		('1402', 'Gekennzeichneter Radwanderweg', 'Gekennzeichneter Radwanderweg ist ein als Radwanderweg eindeutig markierter Weg.'),
		('1403', 'Gekennzeichnete Skaterstrecke', 'Gekennzeichnete Skaterstrecke ist ein als Skaterstrecke eindeutig markierter Weg.');
		CREATE TABLE ax_bahnhofskategorie_bahnverkehrsanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bahnhofskategorie_bahnverkehrsanlage IS 'Alias: "AX_Bahnhofskategorie_Bahnverkehrsanlage", UML-Typ: Enumeration';
		INSERT INTO ax_bahnhofskategorie_bahnverkehrsanlage (wert,beschreibung,dokumentation) VALUES
		('1010', 'Bahnhof', 'Bahnhof ist eine Anlage im Netz der Schienenbahnen und der Seilbahnen (Bahnkategorie Schwebebahn) zur Abwicklung des Personen- und Güterverkehrs entsprechend der Angaben des Betreibers.'),
		('1020', 'Haltestelle', 'Haltestelle ist eine Anlage im Netz der Schienenbahnen und der Seilbahnen (Bahnkategorie Schwebebahn) zur Abwicklung des Personen- und Güterverkehrs entsprechend der Angaben des Betreibers.'),
		('1030', 'Haltepunkt', 'Haltepunkt ist eine Anlage im Netz der Schienenbahnen und der Seilbahnen (Bahnkategorie Schwebebahn) zur Abwicklung des Personen- und Güterverkehrs entsprechend der Angaben des Betreibers.');
		CREATE TABLE ax_bahnkategorie_seilbahnschwebebahn (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bahnkategorie_seilbahnschwebebahn IS 'Alias: "AX_Bahnkategorie_SeilbahnSchwebebahn", UML-Typ: Enumeration';
		INSERT INTO ax_bahnkategorie_seilbahnschwebebahn (wert,beschreibung,dokumentation) VALUES
		('2100', 'Luftseilbahn, Großkabinenbahn', 'Luftseilbahn, Großkabinenbahn ist die Bezeichnung für eine Seilbahn, die Personen befördert und Güter transportiert. Die Kabinen und Transporteinrichtungen werden an einem Zugseil über ein Tragseil fortbewegt.'),
		('2200', 'Kabinenbahn, Umlaufseilbahn', 'Kabinenbahn, Umlaufseilbahn ist die Bezeichnung für eine Seilbahn zur Beförderung von Personen und zum Transport von Gütern. Die Wagen oder Kabinen sind an einem umlaufenden Seil festgeklemmt.'),
		('2300', 'Sessellift', 'Sessellift ist die Bezeichnung für eine Seilbahn zur Beförderung von Personen in Sitzen ohne Kabinenverkleidung.'),
		('2400', 'Ski-, Schlepplift', 'Ski-, Schlepplift ist die Bezeichnung für eine Seilbahn, mit der Skifahrer stehend den Berg hinauf gezogen werden.'),
		('2500', 'Schwebebahn', 'Schwebebahn ist die Bezeichnung für eine Bahn, bei der elektrisch angetriebene Fahrzeuge unter einer Fahrschiene hängen.'),
		('2600', 'Materialseilbahn', 'Materialseilbahn ist die Bezeichnung für eine Seilbahn, die ausschließlich Güter transportiert.');
		CREATE TABLE ax_zustand_bahnverkehrsanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_bahnverkehrsanlage IS 'Alias: "AX_Zustand_Bahnverkehrsanlage", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_bahnverkehrsanlage (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich die Bahnverkehrsanlage nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('4000', 'Im Bau', 'Im Bau bedeutet, dass die Bahnverkehrsanlage noch nicht fertiggestellt ist.');
		CREATE TABLE ax_zustand_bauwerkimgewaesserbereich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_bauwerkimgewaesserbereich IS 'Alias: "AX_Zustand_BauwerkImGewaesserbereich", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_bauwerkimgewaesserbereich (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich Bauwerk im Gewässerbereich nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('4000', 'Im Bau', 'Im Bau bedeutet, dass Bauwerk im Gewässerbereich noch nicht fertiggestellt ist.');
		CREATE TABLE ax_art_wegpfadsteig (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_wegpfadsteig IS 'Alias: "AX_Art_WegPfadSteig", UML-Typ: Enumeration';
		INSERT INTO ax_art_wegpfadsteig (wert,beschreibung,dokumentation) VALUES
		('1103', 'Fußweg', 'Fußweg ist ein Weg, der auf Grund seines Ausbauzustandes nur von Fußgängern zu begehen ist.'),
		('1105', 'Karren- und Ziehweg', 'Karrenweg ist ein Weg im Gebirge, der meist sehr steil ist und nur mit einem Gespann befahren werden kann. Ziehweg ist ein Weg, der der Holzabfuhr im Gebirge dient.'),
		('1106', 'Radweg', 'Radweg ist ein Weg, der als besonders gekennzeichneter und abgegrenzter Teil einer Straße oder mit selbständiger Linienführung für den Fahrradverkehr bestimmt ist'),
		('1107', 'Reitweg', 'Reitweg ist ein besonders ausgebauter Weg, auf dem ausschließlich das Reiten zugelassen ist.'),
		('1108', 'Wattenweg', ''),
		('1109', '(Kletter-)Steig im Gebirge', '(Kletter-)Steig im Gebirge ist ein stellenweise mit Drahtseilen gesicherter Pfad, der zur Überwindung besonders steiler Stellen mit Leitern versehen sein kann.'),
		('1110', 'Rad- und Fußweg', 'Rad- und Fußweg ist ein Weg, der als besonders gekennzeichneter und abgegrenzter Teil einer Straße oder mit selbständiger Linienführung ausschließlich für den Fahrrad- und Fußgängerverkehr bestimmt ist.'),
		('1111', 'Skaterstrecke', 'Skaterstrecke ist ein für Skater besonders ausgebauter asphaltierter Weg.');
		CREATE TABLE ax_lagezuroberflaeche_gleis (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagezuroberflaeche_gleis IS 'Alias: "AX_LageZurOberflaeche_Gleis", UML-Typ: Enumeration';
		INSERT INTO ax_lagezuroberflaeche_gleis (wert,beschreibung,dokumentation) VALUES
		('1200', 'Unter der Erdoberfläche', ''),
		('1400', 'Aufgeständert', '');
		CREATE TABLE ax_art_flugverkehrsanlage (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_flugverkehrsanlage IS 'Alias: "AX_Art_Flugverkehrsanlage", UML-Typ: Enumeration';
		INSERT INTO ax_art_flugverkehrsanlage (wert,beschreibung,dokumentation) VALUES
		('1310', 'Startbahn, Landebahn', 'Startbahn, Landebahn ist eine Fläche, auf der Flugzeuge starten bzw. landen.'),
		('1320', 'Zurollbahn, Taxiway', 'Zurollbahn, Taxiway ist ein Verbindungsweg zwischen den Terminals bzw. dem Vorfeld und der Start- und/oder Landebahn.'),
		('1330', 'Vorfeld', 'Vorfeld ist ein Bereich, in dem Flugzeuge abgefertigt und abgestellt werden.'),
		('5520', 'Verkehrslandeplatz', 'Verkehrslandeplatz ist ein Flugplatz, der in der Luftfahrtkarte 1:500000 (ICAO) als solcher ausgewiesen ist.'),
		('5531', 'Hubschrauberlandeplatz', 'Hubschrauberlandeplatz ist ein Landeplatz, der in der Luftfahrtkarte 1:500000 (ICAO) als solcher ausgewiesen ist.'),
		('5540', 'Landeplatz, Sonderlandeplatz', 'Landeplatz, Sonderlandeplatz ist eine Fläche, die in der Luftfahrtkarte 1:500000 (ICAO) als Landeplatz, Sonderlandeplatz ausgewiesen ist.'),
		('5550', 'Segelfluggelände', 'Segelfluggelände ist eine Fläche, die in der Luftfahrtkarte 1:500000 (ICAO) als Segelfluggelände ausgewiesen ist.');
		CREATE TABLE ax_bauwerksfunktion_bauwerkimverkehrsbereich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_bauwerkimverkehrsbereich IS 'Alias: "AX_Bauwerksfunktion_BauwerkImVerkehrsbereich", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_bauwerkimverkehrsbereich (wert,beschreibung,dokumentation) VALUES
		('1800', 'Brücke', 'Brücke ist ein Bauwerk zum Zweck der Überführung eines Verkehrsweges über einen anderen Verkehrsweg  oder über ein Gewässer sowie zur Überführung über ein tieferliegendes Gelände.'),
		('1801', 'Mehrstöckige Brücke', 'Mehrstöckige Brücke ist eine Brücke, die mit Verkehrswegen in mehreren Etagen ausgestattet ist.'),
		('1802', 'Bogenbrücke', 'Bogenbrücke ist eine Brücke, bei der das Tragwerk aus Bögen besteht.'),
		('1803', 'Fachwerkbrücke', 'Fachwerkbrücke ist eine Brücke, bei der das Tragwerk aus starr zusammengesetzten Tragbalken (Holz oder Metall) besteht.'),
		('1804', 'Hängebrücke', 'Hängebrücke ist eine Brücke, bei der das Tragwerk von Hängegurten (Kabel) an einem oder mehreren Pylonen gehalten wird.'),
		('1805', 'Pontonbrücke', 'Pontonbrücke ist eine Behelfsbrücke, die sich aus kastenförmigen Schwimmkörpern zusammensetzt.'),
		('1806', 'Drehbrücke', 'Drehbrücke ist eine Brücke, bei der sich das Tragwerk um einen senkrechten Zapfen (Königsstuhl) dreht.'),
		('1807', 'Hebebrücke', 'Hebebrücke ist eine Brücke, bei der das Tragwerk an Seilen oder Ketten emporgehoben wird.'),
		('1808', 'Zugbrücke', 'Zugbrücke ist eine Brücke, bei der das Tragwerk um eine waagerechte Achse hochgeklappt wird.'),
		('1810', 'Landebrücke', ''),
		('1820', 'Steg', 'Steg ist ein Bauwerk, das Fußgängern den Übergang über ein Gewässer ermöglicht.'),
		('1830', 'Hochbahn, Hochstraße', 'Hochbahn, Hochstraße ist ein brückenartiges, aufgeständertes Verkehrsbauwerk.'),
		('1840', 'Brückenpfeiler', ''),
		('1845', 'Widerlager', ''),
		('1850', 'Strompfeiler', ''),
		('1870', 'Tunnel, Unterführung', 'Tunnel, Unterführung ist ein künstlich angelegtes unterirdisches Bauwerk, das im Verlauf von Verkehrswegen durch Bergmassive oder unter Flussläufen, Meerengen, städt. Bebauungen u. a. hindurchführt.'),
		('1880', 'Schutzgalerie, Einhausung', 'Schutzgalerie, Einhausung ist eine bauliche Einrichtung an Verkehrswegen zum Schutz gegen Lawinen, Schneeverwehungen, Steinschlägen sowie zum Schutz  gegen Emission. 
Schutzgalerien sind einseitige Überbauungen an Verkehrswegen,  Einhausungen umschließen die Verkehrswege meist vollständig.'),
		('1890', 'Schleusenkammer', 'Schleusenkammer ist eine Einrichtung zur Überführung von Wasserfahrzeugen zwischen Gewässern mit unterschiedlichen Wasserspiegelhöhen.'),
		('1900', 'Durchfahrt', 'Durchfahrt ist eine Stelle, an der mit Fahrzeugen durch ein Bauwerk (z.B. ein Turm, eine Mauer) hindurch gefahren werden kann.'),
		('1910', 'Anflugbefeuerung', ''),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Bauwerksfunktion bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_bauwerksfunktion_bauwerkimgewaesserbereich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauwerksfunktion_bauwerkimgewaesserbereich IS 'Alias: "AX_Bauwerksfunktion_BauwerkImGewaesserbereich", UML-Typ: Enumeration';
		INSERT INTO ax_bauwerksfunktion_bauwerkimgewaesserbereich (wert,beschreibung,dokumentation) VALUES
		('2010', 'Durchlass', 'Durchlass ist ein Bauwerk, in dem ein Gewässer unter einem auf der Erdoberfläche liegenden Hindernis (Verkehrsweg, Siedlungsfläche) hindurchgeführt wird oder unter der Erdoberfläche in freier Feldlage oder abgedeckt (verdolt) auf der Erdoberfläche verläuft.'),
		('2011', 'Rohrdurchlass', 'Rohrdurchlass ist ein Bauwerk zur Unterführung eines Gewässers unter einem Verkehrsweg.'),
		('2012', 'Düker', 'Düker ist ein Kreuzungsbauwerk, in dem ein Gewässer unter einem anderen Gewässer, einem Geländeeinschnitt oder einem tieferliegenden Hindernis unter Druck hindurchgeleitet wird.'),
		('2020', 'Rückhaltebecken', ''),
		('2030', 'Staumauer', 'Staumauer ist ein aus Mauerwerk oder Beton bestehendes Absperrbauwerk zur Erzeugung eines Staus.'),
		('2040', 'Staudamm', 'Staudamm ist ein meist aus natürlichen Baustoffen, meist aufgeschüttetes Absperrbauwerk zur Erzeugung eines Staus.'),
		('2050', 'Wehr', 'Wehr ist ein festes oder mit beweglichen Teilen ausgestattetes Bauwerk im Flussbett zur Regulierung des Wasserabflusses.'),
		('2060', 'Sicherheitstor', 'Sicherheitstor ist ein Bauwerk zum Abschließen von Kanalstrecken, um bei Schäden das Auslaufen der gesamten Kanalhaltung zu verhindern.'),
		('2070', 'Siel', 'Siel ist ein Bauwerk mit Verschlusseinrichtung (gegen rückströmendes Wasser) zum Durchleiten eines oberirdischen Gewässers durch einen Deich.'),
		('2080', 'Sperrwerk', 'Sperrwerk ist ein Bauwerk in einem Tideflussgewässer mit Verschlusseinrichtung zum Absperren bestimmter Tiden, vor allem zum Schutz gegen Sturmfluten auch bei Tidehäfen.'),
		('2090', 'Schöpfwerk', 'Schöpfwerk ist eine Anlage, in der Pumpen Wasser einem höher gelegenen Vorfluter zuführen, u.a. zur künstlichen Entwässerung von landwirtschaftlich genutzten Flächen.'),
		('2110', 'Fischtreppe', ''),
		('2120', 'Pegel', 'Pegel ist eine Messeinrichtung zur Feststellung des Wasserstandes von Gewässern.'),
		('2130', 'Uferbefestigung', 'Uferbefestigung ist eine Anlage zum Schutze des Ufers.'),
		('2131', 'Wellenbrecher, Buhne', 'Wellenbrecher, Buhne ist ein ins Meer oder in den Fluss hinein angelegtes Bauwerk zum Uferschutz aus Buschwerk, Holz, Stein, Stahlbeton oder Asphalt.'),
		('2132', 'Lahnung', 'Lahnung ist ein ins Meer hineingebauter Damm aus Buschwerk, das zwischen Pfahlreihen fest eingepackt und verschnürt ist. Mehrere netzartig angelegte Dämme dienen zur Landgewinnung (z. B. im Watt) indem die Schlickablagerung gefördert wird.'),
		('2133', 'Hafendamm, Mole', 'Hafendamm, Mole ist ein in das Wasser vorgestreckter Steindamm, der eine Hafeneinfahrt begrenzt und das Hafenbecken vor Strömung und Wellenschlag schützt.'),
		('2134', 'Höft', 'Höft ist eine vorspringende Ecke bei Kaimauern in einem Hafen.'),
		('2135', 'Deckwerk', 'Deckwerk ist ein geböschter Uferschutz an Schardeichen (Deiche ohne Vorland).'),
		('2136', 'Ufermauer, Kaimauer', 'Ufermauer, Kaimauer ist eine Mauer entlang der Uferlinie eines Gewässers zum Schutz des Ufers bzw. eine Uferbefestigung im Hafengelände zum Anlegen von Schiffen.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Bauwerksfunktion bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_art_einrichtungenfuerdenschiffsverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_einrichtungenfuerdenschiffsverkehr IS 'Alias: "AX_Art_EinrichtungenFuerDenSchiffsverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_art_einrichtungenfuerdenschiffsverkehr (wert,beschreibung,dokumentation) VALUES
		('1410', 'Bake', 'Bake ist ein festgegründetes pfahl- oder gittermastartiges Schifffahrtszeichen mit Kennung durch Form oder Form und Farbe.'),
		('1420', 'Leuchtfeuer', 'Leuchtfeuer sind Anlagen, die ein Feuer tragen, das über den ganzen Horizont oder in festgelegten Sektoren oder Richtungen gezeigt wird und die bei Tage als Körperzeichen dienen.'),
		('1430', 'Kilometerstein', 'Kilometerstein ist ein Punkt mit einem festen Wert im Netz der Gewässer, der in der Örtlichkeit durch eine Markierung (z.B. Kilometerstein) repräsentiert wird.'),
		('1440', 'Tafel an Gewässern', ''),
		('1450', 'Pricke', ''),
		('1460', 'Anleger', 'Anleger ist eine feste oder schwimmende Einrichtung zum Anlegen von Schiffen.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Art bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_zustand_bauwerkimverkehrsbereich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_bauwerkimverkehrsbereich IS 'Alias: "AX_Zustand_BauwerkImVerkehrsbereich", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_bauwerkimverkehrsbereich (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich Bauwerk im Verkehrsbereich nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.');
		CREATE TABLE ax_artdergewaesserachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdergewaesserachse IS 'Alias: "AX_ArtDerGewaesserachse", UML-Typ: Enumeration';
		INSERT INTO ax_artdergewaesserachse (wert,beschreibung,dokumentation) VALUES
		('1000', 'Gewässerachse der WSV', 'Gewässerachse der WSV ist eine Gewässerachse, deren Geometrie unverändert aus den Unterlagen der Wasser- und Schifffahrtsverwaltung übernommen wurde.'),
		('2000', 'Genäherte Mittellinie in Gewässern', 'Genäherte Mittellinie in Gewässern ist eine Gewässerachse, die den Spezifikationen der Richtlinie der Gebiets- und Gewässerverschlüsselung der Länderarbeitsgemeinschaft Wasser (LAWA) entspricht.'),
		('3001', 'Fiktive Verbindung in Fließgewässern', 'Fiktive Verbindung in Fließgewässern ist eine Gewässerachse, die ein einmündendes Gewässer mit der Gewässerachse des aufnehmenden Fließgewässers verbindet.'),
		('3002', 'Fiktive Verbindung in Seen und Teichen', 'Fiktive Verbindung in Seen und Teichen ist eine hydrologisch sinnvolle Verbindungslinie in stehenden Gewässern, die für den Aufbau eines geschlossenen topologischen Gewässernetzes benötigt wird.');
		CREATE TABLE ax_art_schifffahrtsliniefaehrverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_schifffahrtsliniefaehrverkehr IS 'Alias: "AX_Art_SchifffahrtslinieFaehrverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_art_schifffahrtsliniefaehrverkehr (wert,beschreibung,dokumentation) VALUES
		('1710', 'Autofährverkehr', 'Autofährverkehr ist ein in der Regel nach festem Fahrplan über Flüsse, Seen, Kanäle, Meerengen oder Meeresarme stattfindender Schiffsverkehr zwischen zwei Anlegestellen speziell für Fahrzeuge des Straßenverkehrs.'),
		('1720', 'Eisenbahnfährverkehr', 'Eisenbahnfährverkehr ist ein in der Regel nach festem Fahrplan über Flüsse, Seen, Kanäle, Meerengen oder Meeresarme stattfindender Schiffsverkehr zwischen zwei Anlegestellen speziell für Fahrzeuge des Schienenverkehrs.'),
		('1730', 'Personenfährverkehr', 'Personenfährverkehr ist ein in der Regel nach festem Fahrplan über Flüsse, Seen, Kanäle, Meerengen oder Meeresarme stattfindender Schiffsverkehr zwischen zwei Anlegestellen für Personenbeförderung.'),
		('1740', 'Linienverkehr', 'Linienverkehr ist die auf einer festgelegten Route nach einem festen Fahrplan verkehrende Güter- und Personenschifffahrt.');
		CREATE TABLE ax_zustand_schleuse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_schleuse IS 'Alias: "AX_Zustand_Schleuse", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_schleuse (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich die Schleuse nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.');
		CREATE TABLE ax_nutzung_hafen (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_nutzung_hafen IS 'Alias: "AX_Nutzung_Hafen", UML-Typ: Enumeration';
		INSERT INTO ax_nutzung_hafen (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zivil', 'Zivil bedeutet, dass Hafen privaten oder öffentlichen Zwecken dient und nicht militärisch genutzt wird.'),
		('2000', 'Militärisch', 'Militärisch bedeutet, dass Hafen nur von Streitkräften genutzt wird.'),
		('3000', 'Teils zivil, teils militärisch', 'Teils zivil, teils militärisch bedeutet, dass Hafen sowohl zivil als auch militärisch genutzt wird.');
		CREATE TABLE ax_konstruktionsmerkmalbauart_schleuse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_konstruktionsmerkmalbauart_schleuse IS 'Alias: "AX_KonstruktionsmerkmalBauart_Schleuse", UML-Typ: Enumeration';
		INSERT INTO ax_konstruktionsmerkmalbauart_schleuse (wert,beschreibung,dokumentation) VALUES
		('1010', 'Schiffshebewerk', 'Schiffshebewerk ist ein Bauwerk zum Überwinden einer Fallstufe (in Binnenwasserstraßen und Kanälen) mit Förderung der Schiffe in einem Trog.'),
		('1020', 'Kammerschleuse', 'Kammerschleuse ist ein Bauwerk zum Überwinden einer Fallstufe, in dem durch Füllen oder Leeren der Schleusenkammer Schiffe gehoben oder gesenkt werden.');
		CREATE TABLE ax_hafenkategorie_hafen (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hafenkategorie_hafen IS 'Alias: "AX_Hafenkategorie_Hafen", UML-Typ: Enumeration';
		INSERT INTO ax_hafenkategorie_hafen (wert,beschreibung,dokumentation) VALUES
		('1010', 'Containerhafen', 'Containerhafen ist ein Hafen mit speziellen Einrichtungen (z. B. Verladebrücken) für den Umschlag von genormten Containern.'),
		('1020', 'Ölhafen', 'Ölhafen ist ein Hafen mit speziellen Einrichtungen (z. B. Tankanlagen) für den Umschlag von Rohöl und den daraus verarbeiteten Produkten.'),
		('1030', 'Fischereihafen', 'Fischereihafen ist ein Hafen mit speziellen Einrichtungen (z. B. Kühlhäuser) für den Umschlag von frisch gefangenem Fisch.'),
		('1040', 'Sporthafen, Yachthafen', 'Sporthafen, Yachthafen ist ein Hafen für Sport- und Freizeitschiffe.'),
		('1050', 'Fährhafen', 'Fährhafen ist ein Hafen zum Anlegen von Fährschiffen.'),
		('1060', 'Stückguthafen', 'Stückguthafen ist ein Hafen, in dem nur Stückgüter umgeschlagen werden.'),
		('1070', 'Hafen für Massengüter', 'Hafen für Massengüter ist ein Hafen, in dem Massengüter umgeschlagen werden.');
		CREATE TABLE ax_art_gewaessermerkmal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_gewaessermerkmal IS 'Alias: "AX_Art_Gewaessermerkmal", UML-Typ: Enumeration';
		INSERT INTO ax_art_gewaessermerkmal (wert,beschreibung,dokumentation) VALUES
		('1610', 'Quelle', 'Quelle ist eine natürliche, örtlich begrenzte Austrittsstelle von Wasser.'),
		('1620', 'Wasserfall', 'Wasserfall ist ein senkrechter oder nahezu senkrechter Absturz eines Wasserlaufs, der über eine oder mehrere natürliche Stufen verlaufen kann.'),
		('1630', 'Stromschnelle', 'Stromschnelle ist eine Flussstrecke mit höherer Strömungsgeschwindigkeit durch ein besonders starkes Gefälle sowie oft auch geringerer Wassertiefe.'),
		('1640', 'Sandbank', 'Sandbank ist eine vegetationslose Sand- oder Kiesablagerung auf dem Meeresboden oder in Flüssen, die durch Brandung oder Strömung aufgebaut wird.'),
		('1650', 'Watt', 'Watt ist ein aus Sand oder Schlick bestehender Boden an flachen Gezeitenküsten und in Mündungstrichtern von Flüssen, der bei Ebbe ganz oder teilweise trocken fällt und bei Flut vom sogenannten Wattenmeer überspült wird.'),
		('1660', 'Priel', 'Priel ist eine natürliche Rinne im Watt, die auch bei Ebbe Wasser führt.'),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_hydrologischesmerkmal_untergeordnetesgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_untergeordnetesgewaesser IS 'Alias: "AX_HydrologischesMerkmal_UntergeordnetesGewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_untergeordnetesgewaesser (wert,beschreibung,dokumentation) VALUES
		('2000', 'Nicht ständig Wasser führend', ''),
		('3000', 'Trocken, versiegt', '');
		CREATE TABLE ax_lagezurerdoberflaeche_untergeordnetesgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagezurerdoberflaeche_untergeordnetesgewaesser IS 'Alias: "AX_LageZurErdoberflaeche_UntergeordnetesGewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_lagezurerdoberflaeche_untergeordnetesgewaesser (wert,beschreibung,dokumentation) VALUES
		('1800', 'Verrohrt, unterirdisch, bedeckt', ''),
		('1810', 'Verdolt', '');
		CREATE TABLE ax_artdespolders (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdespolders IS 'Alias: "AX_ArtDesPolders", UML-Typ: Enumeration';
		INSERT INTO ax_artdespolders (wert,beschreibung,dokumentation) VALUES
		('1000', 'Sommerpolder', 'Sommerpolder (auch Überlaufpolder) ist ein Polder, der durch einen Überlauf- oder Sommerdeich nur gegen niedrige Hochwasser geschützt ist. Der Deich ist so ausgelegt, dass er zeitweise überströmt werden kann.'),
		('2000', 'Entlastungspolder', 'Entlastungspolder (auch Speicherpolder) ist ein Becken, das durch Zurückhalten von Wasser das Schöpfwerk, das Siel, den Vorfluter und/oder die Sperrwerksanlage entlastet.'),
		('3000', 'Flutungspolder', 'Flutungspolder ist eine eingedeichte, meist landwirtschaftlich genutzte Fläche, die beim Eintreten eines kritischen Wasserstandes zur Entlastung der Deiche genutzt wird.');
		CREATE TABLE ax_funktion_polder (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_polder IS 'Alias: "AX_Funktion_Polder", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_polder (wert,beschreibung,dokumentation) VALUES
		(' 7410', 'Gesteuert = 7410', ''),
		(' 7420', 'Ungesteuert = 7420', '');
		CREATE TABLE ax_funktion_untergeordnetesgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_untergeordnetesgewaesser IS 'Alias: "AX_Funktion_UntergeordnetesGewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_untergeordnetesgewaesser (wert,beschreibung,dokumentation) VALUES
		('1010', 'Graben', ''),
		('1011', 'Grabenkante, rechts', ''),
		('1012', 'Grabenkante, links', ''),
		('1013', 'Grabenmitte', ''),
		('1020', 'Fleet', ''),
		('1030', 'Bach', ''),
		('1040', 'Teich', '');
		CREATE TABLE ax_hydrologischesmerkmal_gewaessermerkmal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_gewaessermerkmal IS 'Alias: "AX_HydrologischesMerkmal_Gewaessermerkmal", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_gewaessermerkmal (wert,beschreibung,dokumentation) VALUES
		('2000', 'Nicht ständig Wasser führend', 'Nicht ständig Wasser führend heißt, dass die Quelle nicht ganzjährig Wasser führt.');
		CREATE TABLE ax_funktion_vegetationsmerkmal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_vegetationsmerkmal IS 'Alias: "AX_Funktion_Vegetationsmerkmal", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_vegetationsmerkmal (wert,beschreibung,dokumentation) VALUES
		('1000', 'Windschutz', '');
		CREATE TABLE ax_zustand_vegetationsmerkmal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_vegetationsmerkmal IS 'Alias: "AX_Zustand_Vegetationsmerkmal", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_vegetationsmerkmal (wert,beschreibung,dokumentation) VALUES
		('5000', 'Nass', 'Nass bezeichnet eine Vegetationsfläche, die aufgrund besonderer Bodenbeschaffenheit ganzjährig wassergesättigt ist, zeitweise auch unter Wasser stehen kann.');
		CREATE TABLE ax_bewuchs_vegetationsmerkmal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bewuchs_vegetationsmerkmal IS 'Alias: "AX_Bewuchs_Vegetationsmerkmal", UML-Typ: Enumeration';
		INSERT INTO ax_bewuchs_vegetationsmerkmal (wert,beschreibung,dokumentation) VALUES
		('1011', 'Nadelbaum', 'Nadelbaum beschreibt die Zugehörigkeit eines einzeln stehenden Baumes zur Gruppe der Nadelhölzer.'),
		('1012', 'Laubbaum', 'Laubbaum beschreibt die Zugehörigkeit eines einzeln stehenden Baumes zur Gruppe der Laubhölzer.'),
		('1021', 'Baumbestand, Laubholz', 'Baumbestand, Laubholz beschreibt den Bewuchs einer Vegetationsfläche mit Laubbäumen.'),
		('1022', 'Baumbestand, Nadelholz', 'Baumbestand, Nadelholz beschreibt den Bewuchs einer Vegetationsfläche mit Nadelbäumen.'),
		('1023', 'Baumbestand, Laub- und Nadelholz', 'Baumbestand, Laub- und Nadelholz beschreibt den Bewuchs einer Vegetationsfläche mit Laub- und Nadelbäumen.'),
		('1100', 'Hecke', 'Hecke besteht aus einer Reihe dicht beieinander stehender, meist wildwachsender Sträucher.'),
		('1101', 'Heckenkante, rechts', ''),
		('1102', 'Heckenkante, links', ''),
		('1103', 'Heckenmitte', ''),
		('1210', 'Baumreihe, Laubholz', 'Laubholz beschreibt die Zugehörigkeit einer Baumreihe zur Gruppe der Laubhölzer.'),
		('1220', 'Baumreihe, Nadelholz', 'Nadelholz beschreibt die Zugehörigkeit einer Baumreihe zur Gruppe der Nadelhölzer.'),
		('1230', 'Baumreihe, Laub- und Nadelholz', 'Laub- und Nadelholz beschreibt den Bewuchs einer Baumreihe mit Laub- und Nadelbäumen.'),
		('1250', 'Gehölz', 'Gehölz ist eine Fläche, die mit einzelnen Bäumen, Baumgruppen, Büschen, Hecken und Sträuchern bestockt ist.'),
		('1260', 'Gebüsch', 'Gebüsch beschreibt den Bewuchs einer Vegetationsfläche mit Holzpflanzen, deren Sprossen sich nahe der Bodenoberfläche verzweigen.'),
		('1300', 'Schneise', 'Schneise ist eine künstlich angelegte Waldeinteilungslinie zur dauerhaften Begrenzung forstlicher Wirtschaftsflächen (räumliche Ordnung), die in der Regel geradlinig verläuft.'),
		('1400', 'Röhricht, Schilf', 'Röhricht, Schilf beschreibt den Bewuchs einer Vegetations- oder Wasserfläche mit Schilfrohr- und schilfrohrähnlichen Pflanzen.'),
		('1500', 'Gras', 'Gras beschreibt den Bewuchs einer Vegetationsfläche mit schlanken, krautigen einkeimblättrigen Blütenpflanzen.'),
		('1510', 'Rain', ''),
		('1600', 'Zierfläche', ''),
		('1700', 'Korbweide', ''),
		('1800', 'Reet', 'Reet bezeichnet eine ständig oder zeitweise unter Wasser stehende und mit Reet bewachsene Fläche.'),
		('1900', 'Streuobst', 'Streuobst beschreibt den Bewuchs einer Fläche mit Obstbäumen.');
		CREATE TABLE ax_eigentuemerart_namensnummer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_eigentuemerart_namensnummer IS 'Alias: "AX_Eigentuemerart_Namensnummer", UML-Typ: Enumeration';
		INSERT INTO ax_eigentuemerart_namensnummer (wert,beschreibung,dokumentation) VALUES
		('1000', 'Natürliche Personen', ''),
		('1100', 'Natürliche Person - Alleineigentum oder Ehepartner', ''),
		('1200', 'Natürliche Person - Wohnsitz im Land', ''),
		('1300', 'Natürliche Person - Wohnsitz außerhalb des Landes', ''),
		('1500', 'Natürliche Person - Gemeinschaftseigentum', ''),
		('2000', 'Juristische Personen', ''),
		('2100', 'Gemeinnützige Bau-, Wohnungs- oder Siedlungsgesellschaft oder -genossenschaft einschließlich Heimstätte', ''),
		('2200', 'Sonstige gemeinnützige Institution (Träger von Krankenhäusern, Altenheimen usw.)', ''),
		('2300', 'Privates Wohnungsunternehmen, private Baugesellschaft u.ä.', ''),
		('2400', 'Kreditinstitut', ''),
		('2500', 'Versicherungsunternehmen', ''),
		('2900', 'Andere Unternehmen, Gesellschaften usw.', ''),
		('3000', 'Körperschaften', ''),
		('3100', 'Stiftung', ''),
		('4000', 'Kirchliches Eigentum', ''),
		('4100', 'Evangelische Kirche', ''),
		('4200', 'Katholische Kirche', ''),
		('4900', 'Andere Kirchen, Religionsgemeinschaften usw.', ''),
		('5100', 'Bundesrepublik Deutschland', ''),
		('5101', 'Bundesrepublik Deutschland, Bundesstraßenverwaltung', ''),
		('5102', 'Bundesrepublik Deutschland, Bundeswehrverwaltung', ''),
		('5103', 'Bundesrepublik Deutschland, Forstverwaltung', ''),
		('5104', 'Bundesrepublik Deutschland, Finanzverwaltung', ''),
		('5105', 'Bundesrepublik Deutschland, Zivilschutz', ''),
		('5106', 'Bundesrepublik Deutschland, Wasserstraßenverwaltung', ''),
		('5107', 'Bundesrepublik Deutschland, Bundeseisenbahnvermögen', ''),
		('5210', 'Eigentum des Volkes nach DDR-Recht', ''),
		('5220', 'Eigentum der Genossenschaften und deren Einrichtungen', ''),
		('5230', 'Eigentum der gesellschaftlichen Organisationen und deren Einrichtungen', ''),
		('5240', 'Kommunale Gebietskörperschaften nach DDR-Recht', ''),
		('5300', 'Ausländischer Staat', ''),
		('5400', 'Kreis', ''),
		('5500', 'Gemeinde', ''),
		('5600', 'Kommunale Gebietskörperschaften', ''),
		('5700', 'Andere Gebietskörperschaften, Regionalverbände usw.', ''),
		('5800', 'Zweckverbände, Kommunale Betriebe', ''),
		('5920', 'Eigenes Bundesland', ''),
		('5921', 'Eigenes Bundesland, Denkmalpflege', ''),
		('5922', 'Eigenes Bundesland, Domänenverwaltung', ''),
		('5923', 'Eigenes Bundesland, Eichverwaltung', ''),
		('5924', 'Eigenes Bundesland, Finanzverwaltung', ''),
		('5925', 'Eigenes Bundesland, Forstverwaltung', ''),
		('5926', 'Eigenes Bundesland, Gesundheitswesen', ''),
		('5927', 'Eigenes Bundesland, Polizeiverwaltung', ''),
		('5928', 'Eigenes Bundesland, innere Verwaltung', ''),
		('5929', 'Eigenes Bundesland, Justizverwaltung', ''),
		('5930', 'Eigenes Bundesland, Kultusverwaltung', ''),
		('5931', 'Eigenes Bundesland, Landespflanzenschutzverwaltung', ''),
		('5932', 'Eigenes Bundesland, Arbeitsverwaltung', ''),
		('5933', 'Eigenes Bundesland, Sozialwesen', ''),
		('5934', 'Eigenes Bundesland, Landesbetrieb Straßen und Verkehr', ''),
		('5935', 'Eigenes Bundesland, Umweltverwaltung', ''),
		('5936', 'Eigenes Bundesland, Vermessungs- und Katasterverwaltung', ''),
		('5937', 'Eigenes Bundesland, Wasserwirtschaftsverwaltung', ''),
		('5938', 'Eigenes Bundesland, Wirtschaftsverwaltung', ''),
		('5939', 'Eigenes Bundesland, Liegenschafts- und Baubetreuung (LBB)', ''),
		('6000', 'Anderes Bundesland (allg.)', ''),
		('6001', 'Schleswig-Holstein', ''),
		('6002', 'Hamburg', ''),
		('6003', 'Niedersachsen', ''),
		('6004', 'Bremen', ''),
		('6005', 'Nordrhein-Westfalen', ''),
		('6006', 'Hessen', ''),
		('6007', 'Rheinland-Pfalz', ''),
		('6008', 'Baden-Württemberg', ''),
		('6009', 'Bayern', ''),
		('6010', 'Saarland', ''),
		('6011', 'Berlin', ''),
		('6012', 'Brandenburg', ''),
		('6013', 'Mecklenburg-Vorpommern', ''),
		('6014', 'Sachsen', ''),
		('6015', 'Sachsen-Anhalt', ''),
		('6016', 'Thüringen', ''),
		('7100', 'Deutsche Bahn AG', ''),
		('8000', 'Herrenlos', ''),
		('9000', 'Eigentümer unbekannt', '');
		CREATE TABLE ax_li_processstep_ohnedatenerhebung_description (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_li_processstep_ohnedatenerhebung_description IS 'Alias: "AX_LI_ProcessStep_OhneDatenerhebung_Description", UML-Typ: Enumeration';
		INSERT INTO ax_li_processstep_ohnedatenerhebung_description (wert,beschreibung,dokumentation) VALUES
		('Erhebung', NULL, '');
		CREATE TABLE ax_blattart_buchungsblatt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_blattart_buchungsblatt IS 'Alias: "AX_Blattart_Buchungsblatt", UML-Typ: Enumeration';
		INSERT INTO ax_blattart_buchungsblatt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Grundbuchblatt', 'Ein Grundbuchblatt ist ein Buchungsblatt, das die Buchung im Grundbuch enthält.'),
		('2000', 'Katasterblatt', 'Ein Katasterblatt ist ein Buchungsblatt, das die Buchung im Liegenschaftskataster enthält.'),
		('3000', 'Pseudoblatt', 'Ein Pseudoblatt ist ein Buchungsblatt, das die Buchung, die bereits vor Eintrag im Grundbuch Rechtskraft erlangt hat, enthält (z.B. Übernahme von Flurbereinigungsverfahren, Umlegungsverfahren).'),
		('4000', 'Erwerberblatt', 'Ein Erwerberblatt ist ein Buchungsblatt, das die Buchung, die bereits im Liegenschaftskataster, aber noch nicht im Grundbuch gebucht ist, enthält (Buchungsvorschlag für die Grundbuchverwaltung).
Pseudoblatt und Erwerberblatt werden nach Eintragung in das Grundbuch historisch.'),
		('5000', 'Fiktives Blatt', 'Das fiktive Blatt enthält die aufgeteilten Grundstücke und Rechte als Ganzes. Es bildet um die Miteigentumsanteile eine fachliche Klammer.');
		CREATE TABLE ax_anrede_person (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_anrede_person IS 'Alias: "AX_Anrede_Person", UML-Typ: Enumeration';
		INSERT INTO ax_anrede_person (wert,beschreibung,dokumentation) VALUES
		('1000', 'Frau', ''),
		('2000', 'Herr', ''),
		('3000', 'Firma', '');
		CREATE TABLE ax_artderrechtsgemeinschaft_namensnummer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderrechtsgemeinschaft_namensnummer IS 'Alias: "AX_ArtDerRechtsgemeinschaft_Namensnummer", UML-Typ: Enumeration';
		INSERT INTO ax_artderrechtsgemeinschaft_namensnummer (wert,beschreibung,dokumentation) VALUES
		('1000', 'Erbengemeinschaft', ''),
		('2000', 'Gütergemeinschaft', ''),
		('3000', 'BGB-Gesellschaft', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_buchungsart_buchungsstelle (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_buchungsart_buchungsstelle IS 'Alias: "AX_Buchungsart_Buchungsstelle", UML-Typ: Enumeration';
		INSERT INTO ax_buchungsart_buchungsstelle (wert,beschreibung,dokumentation) VALUES
		('1100', 'Grundstück', 'Das Grundstück ist ein räumlich abgegrenzter Teil der Erdoberfläche, der auf einem besonderen Blatt, dem Grundbuchblatt, für sich allein oder auf einem gemeinschaftlichen Grundbuchblatt unter einer eindeutigen Nummer des Bestandsverzeichnisses eingetragen ist (Grundstück im Rechtssinn). Das Grundstück besteht aus einem oder mehreren Flurstücken.'),
		('1101', 'Aufgeteiltes Grundstück WEG', 'Ein aufgeteiltes GrundstückWEG ist die Zusammenfassung  aller in Wohnungs- oder Teileigentum aufgeteilten Anteile eines Grundstücks. Es handelt sich daher um eine Buchungsart für das Fiktive Blatt.'),
		('1102', 'Aufgeteiltes Grundstück Par. 3 Abs. 4 GBO', 'Ein aufgeteiltes Grundstück nach Par. 3  Abs. 4 GBO ist die Zusammenfassung  aller dienenden Miteigentumsanteile eines Grundstücks (Miteigentumsanteil nach  § 3 Abs. 4 GBO).  Es handelt sich daher um eine Buchungsart für das Fiktive Blatt.'),
		('1200', 'Ungetrennter Hofraum', 'Zu einem ungetrennten Hofraum gehören Grundstücke, die zwar in ihren Außengrenzen, nicht aber bezüglich der daran bestehenden Anteile vermessen und katastermäßig erfasst sind. Im Grundbuch werden die nicht ausgemessenen , einzelnen Grundstücksflächen als Anteil an einem ungetrennten Hofraum ausgewiesen. Bis zur Vermessung der einzelnen Grundstücksflächen und Aufnahme in das Liegenschaftsbuch gilt das Gebäudesteuerbuch als amtliches Verzeichnis nach Par. 2 Abs. 2 GBO. Diese Nummer des Gebäudesteuerbuchs ist bis zur Vermessung im Grundbuch eingetragen. Für ungetrennte Hofräume existieren zwei miteinander korrespondierende Eintragungen:
In einem Grundbuchblatt sind alle ungetrennten Hofräume eines Grundbuchbezirks verzeichnet. Im zweiten Grundbuchblatt ist im Bestandsverzeichnis dann der Anteil an einem bestimmten ungetrennten Hofraum eingetragen.
Nach der katasterlichen Erfassung wird die bisherige Eintragung in beiden Grundbüchern gelöscht und das vermessenene Grundstück als normales Grundstück gebucht.'),
		('1301', 'Wohnungs-/Teileigentum', 'Das Wohnungseigentum kann nach Par. 3 Wohnungseigentumsgesetz (WEG) durch Vertrag der Miteigentümer oder nach Par. 8 WEG durch Erklärung des Eigentümers 
begründet werden.
Das entstehende Wohnungseigentum (Teileigentum) ist echtes Eigentum bürgerlichen Rechts in Form einer rechtlichen Verbindung von Miteigentum an Grundstück und Gebäude mit Sondereigentum an einer Wohnung bzw. Teileigentum an nicht zu Wohnzwecken dienenden Räumen.'),
		('1302', 'Miteigentum Par. 3 Abs. 4 GBO', 'Ein Miteigentum nach Par. 3 Abs. 4 der Grundbuchordnung (GBO) ist ein Miteigentum an einem dienenden Grundstück. Ist das Grundstück im wirtschaftlichen Sinn als Zubehör mehrerer anderer Grundstücke anzusehen und steht es im Miteigentum dieser Grundstücke (Bruchteilseigentum nach Par. 1008 ff des Bürgerlichen Gesetzbuchs (BGB), muss das Grundstück nicht in einem separaten Grundbuch geführt werden. Vielmehr wird das dienende Grundstück in ideellen Miteigentumsanteilen auf den Grundbuchblättern der herrschenden Grundstücke gebucht.'),
		('1303', 'Anteil am ungetrennten Hofraum', 'Hierbei handelt es sich um die Buchung des Anteils am ungetrennten Hofraum.'),
		('1401', 'Aufgeteilter Anteil Wohnungs-/Teileigentum', 'Hier wurde der mit dem Sondereigentum verbundene Miteigentumsanteil (Wohnungs-/Teileigentum) nochmals unterteilt. Die vorgenommene Grundbucheintragung deutet auf  eine Untergemeinschaft innerhalb der Gesamtgemeinschaft hin. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('1402', 'Aufgeteilter Anteil Miteigentum Par. 3 Abs. 4 GBO', 'Hier wurde der Miteigentumsanteil nach Par. 3 (4) GBO nochmals unterteilt. Die vorgenommene Grundbucheintragung deutet auf  eine Untergemeinschaft innerhalb der Gesamtgemeinschaft hin. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('1403', 'Aufgeteilter Anteil am ungetrennten Hofraum', 'Hier wurde der Anteil an ungeteiltem Hofraum nochmals unterteilt. Die vorgenommene Grundbucheintragung deutet auf  eine Untergemeinschaft innerhalb der Gesamtgemeinschaft hin. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('1501', 'Anteil an Wohnungs-/Teileigentumsanteil', 'Hier wird der Anteil an dem Wohnungs-/Teileigentumsanteil im Grundbuch eingetragen.'),
		('1502', 'Anteil an Miteigentumsanteil Par. 3 Abs. 4 GBO', 'Hier wird der Anteil an dem Miteigentumsanteil nach Par. 3 Abs. 4 GBO im Grundbuch eingetragen'),
		('1503', 'Anteil am Anteil zum ungetrennten Hofraum', 'Hier wird der Anteil an dem Anteil an dem ungetrennten Hofraum im Grundbuch eingetragen.'),
		('2101', 'Erbbaurecht', 'Veräußerliches und vererbliches grundstücksgleiches Recht, auf oder unter der Erdoberfläche eines (in der Regel) fremden Grundstücks ein Bauwerk zu haben.'),
		('2102', 'Untererbbaurecht', 'Untererbbaurecht ist das Erbbaurecht an einem Erbbaurecht. Hier ist der Belastungsgegenstand nicht das Grundstück, sondern das auf diesem lastenden Erbbaurecht.'),
		('2103', 'Gebäudeeigentum', 'Das Gebäudeeigentum an einem Grundstück als Ganzes zur Errichtung und  Nutzung eines Gebäudes.'),
		('2104', 'Fischereirecht', 'Fischereirecht ist die Befugnis, in einem Binnengewässer (See, Teich, Fluß, Bach) Fische, Krebse und andere nutzbare Wassertiere (z.B. Muscheln, Frösche), die nicht Gegenstand des Jagdrechts sind, zu hegen und sich anzueignen.'),
		('2105', 'Bergwerksrecht', 'Bergwerksrecht ist das ausschließliche Recht, in einem bestimmten Feld die in der Bewilligung bezeichneten Bodenschätze aufzusuchen und zu gewinnen (Par.9 I, Par. 8 BBergG vom 13.08.1980, BGBl. I 1310).'),
		('2106', 'Nutzungsrecht', 'Hierunter sind alle Nutzungsrechte zu verstehen, die im Bestandsverzeichnis eingetragen werden, unabhängig von ihrer öffentlich- oder privatrechtlichen Natur. Die nähere Bezeichnung des Nutzungsrechts ergibt sich aus dem Attribut Buchungstext.'),
		('2107', 'Realgewerberecht', 'Hierbei handelt es sich um die frei veräußerliche und vedrerbliche Befugnis zum Betrieb eines bestimmten Gewerbes, die mit dem Besitz einer Liegenschaft verbunden sein  kann aber nicht zwingend an ein bestimmtes Grundstücks gebunden sein  muss. Die nähere Bezeichnung des Nutzungsrechts ergibt sich aus dem Attribut Buchungstext.'),
		('2108', 'Gemeinderecht', 'Gemeinderecht ist das Recht zur Nutzung eines gemeinschftlichen Grundstücks. Die näheren Angaben zu diesem Recht sind in privatrechtlichen Verträgen enthalten.'),
		('2109', 'Stavenrecht', 'Ist ein mit dem Erbbaurecht vergleichbares Recht in  den nordfriesischen Küstenregionen.'),
		('2110', 'Hauberge', ''),
		('2201', 'Aufgeteiltes Erbbaurecht WEG', 'Diese Buchungsart ist die Zusammenfassung aller Anteile eines Erbbaurechts, die  auf mehreren Grundbuchblättern gebucht sind. Es handelt sich hier um eine Buchungsart für das Fiktive Blatt.'),
		('2202', 'Aufgeteiltes Untererbbaurecht WEG', 'Diese Buchungsart ist die Zusammenfassung aller Anteile eines Untererbbaurechts, die  auf mehreren Grundbuchblättern gebucht sind. Es handelt sich hier um eine Buchungsart für das Fiktive Blatt.'),
		('2203', 'Aufgeteiltes Recht Par. 3 Abs. 4 GBO', 'Diese Buchungsart ist die Zusammenfassung aller dienenden Miteigentumsanteile eines Erbbaurechts. Es handelt sich hier um eine Buchungsart für das Fiktive Blatt.'),
		('2204', 'Aufgeteiltes Recht, Körperschaft', 'Diese Buchungsart ist die Zusammenfassung aller auf den Grundbuchblättern der herrschenden Grundstücke gebuchten Nutzanteile an einer Körperschaft. Es handelt sich hier um eine Buchungsart für das Fiktive Blatt. Eine Körperschaft besteht aus einem Verband von Mitgliedern, deren Mitgliedschaft an landesrechtliche (meistens  altrechtliche ) und persönliche Merkmale gebunden ist; die  Mitglieder haben das Recht zur Nutzung des Grundstücks in einem bestimmten Umfang (z.B. Körperschaftswaldungen).'),
		('2205', 'Aufgeteiltes Gebäudeeigentum', 'Diese Buchungsart ist die Zusammenfassung aller Anteile eines Gebäudeeigentums, die  auf mehreren Grundbuchblättern gebucht sind. Es handelt sich hier um eine  Buchungsart für das Fiktive Blatt.'),
		('2301', 'Wohnungs-/Teilerbbaurecht', 'Wohnungs-/Teilerbaurechte können nach Par. 30 WEG unter Anwendung der Par. 3, 8 WEG begründet werden, wobei an die Stelle des Miteigentums am Grundstück die Mitberechtigung nach Bruchteilen an einem Erbbaurecht tritt, mit welchem das Sondereigentum an der Wohnung bzw. den nicht zu Wohnzwecken dienenden Räumen verbunden wird.'),
		('2302', 'Wohnungs-/Teiluntererbbaurecht', 'Wohnungs-/Teiluntererbbaurecht ist die Aufteilung eines Untererbbaurechts analog Par. 30 WEG.'),
		('2303', 'Erbbaurechtsanteil Par. 3 Abs. 4 GBO', 'Ein Erbbaurechtsanteil nach Par. 3 Abs. 4 der Grundbuchordnung (GBO) ist ein Miteigentum an einem dienenden Erbbaurecht.'),
		('2304', 'Anteiliges Recht, Körperschaft', 'Eintragung eines Anteils an dem  Recht - Körperschaft nach Par. 9 GBO im Grundbuch des jeweils herrschenden Grundstückes, dabei besteht die Körperschaft aus einem Verband von Mitgliedern, deren Mitgliedschaft  an gebietliche und persönliche Merkmale geknüpft ist  (z. B. Körperschaftswaldungen).'),
		('2305', 'Anteil am Gebäudeeigentum', 'Ist der Anteil des Nutzungsberechtigten für die Nutzung des auf dem Grundstück stehenden Gebäudes.'),
		('2401', 'Aufgeteilter Anteil Wohnungs-/Teilerbbaurecht', 'Hierbei wurde der Anteil an einem Wohnungs-/Teilerbbaurecht nochmals unterteilt. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('2402', 'Aufgeteilter Anteil Wohnungs-/Teiluntererbbaurecht', 'Hierbei wurde der Anteil an einem Wohnungs-/Teiluntererbbaurecht nochmals unterteilt. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('2403', 'Aufgeteilter Erbbaurechtsanteil Par. 3 Abs. 4 GBO', 'Hierbei wurde der Anteil an einem Erbbaurechtsanteil nochmals unterteilt. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('2404', 'Aufgeteiltes anteiliges Recht, Körperschaft', 'Hierbei wurde der Anteil an einem anteiligem Recht Körperschaft nochmals unterteilt. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('2405', 'Aufgeteilter Anteil am Gebäudeeigentum', 'Hierbei wurde der Anteil an einem Gebäudeeigentum nochmals unterteilt. Es handelt sich um eine Buchungsart für das Fiktive Blatt.'),
		('2501', 'Anteil am Wohnungs-/Teilerbbaurechtsanteil', 'Hier wird der Anteil an dem Wohnungs-/Teilerbbaurechtsanteil im Grundbuch eingetragen.'),
		('2502', 'Anteil am Wohnungs-/Teiluntererbbaurechtsanteil', 'Hier wird der Anteil an dem Wohnungs-/Teiluntererbbaurechtsanteil im Grundbuch eingetragen.'),
		('2503', 'Anteil am Erbbaurechtsanteil Par. 3 Abs. 4 GBO', 'Hier wird der Anteil an dem Erbbaurechtsanteil  im Grundbuch eingetragen.'),
		('2504', 'Anteil am anteiligen Recht, Körperschaft', 'Hier wird der Anteil an dem anteiligen Recht Körperschaft im Grundbuch eingetragen.'),
		('2505', 'Anteil am Anteil zum Gebäudeeigentum', 'Hier wird der Anteil an dem Anteil zum Gebäudeeigentum im Grundbuch eingetragen.'),
		('3100', 'Vermerk subjektiv dinglicher Rechte (Par. 9 GBO)', 'Der Vermerk ist ein Hinweis auf eine in Abteilung II des Grundbuchs des dienenden Grundstücks eingetragene Belastung. Er selbst ist kein Recht; seine Eintragung sichert lediglich, dass bei einer Aufhebung des Rechts im Grundbuch des dienenden Grundstücks die Bewilligung derer erforderlich ist, die der Rechtsänderung nach Par. 876 S. 2, 877, 888 BGB zustimmen müssen.'),
		('4100', 'Stockwerkseigentum', ''),
		('5101', 'Von Buchungspflicht befreit Par. 3 Abs. 2 GBO', 'Grundstücke nach Par. 3 Abs. 2 sind von der Buchungspflicht befreit und werden auf dem Katasterblatt gebucht.'),
		('5200', 'Anliegerflurstück', 'Ein Flurstück dessen Teilflächen den anliegenden Flurstücken zugerechnet wird.'),
		('5201', 'Anliegerweg', ''),
		('5202', 'Anliegergraben', ''),
		('5203', 'Anliegerwasserlauf, Anliegergewässer', ''),
		('6101', 'Nicht gebuchtes Fischereirecht', 'Das nicht gebuchte Fischereirecht wird nach Wasserrecht im Fischwasserkataster nachgewiesen und ist im Grundbuch nicht gebucht.');
		CREATE TABLE ax_klassifikation_hierarchiestufe3d_lagefestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_klassifikation_hierarchiestufe3d_lagefestpunkt IS 'Alias: "AX_Klassifikation_Hierarchiestufe3D_Lagefestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_klassifikation_hierarchiestufe3d_lagefestpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Hierarchiestufe A', ''),
		('2000', 'Hierarchiestufe B', ''),
		('3000', 'Hierarchiestufe C', ''),
		('4000', 'Hierarchiestufe D', ''),
		('5000', 'Hierarchiestufe E', ''),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', '');
		CREATE TABLE ax_punktstabilitaet (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet IS 'Alias: "AX_Punktstabilitaet", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr gut', ''),
		('2000', 'gut', ''),
		('3000', 'befriedigend', ''),
		('4000', 'ausreichend', ''),
		('5000', 'mangelhaft (ohne Nennung eines Grundes)', ''),
		('5100', 'mangelhaft (Bergsenkungsgebiet)', ''),
		('5200', 'mangelhaft (in rutschgefährdeter Hanglage)', ''),
		('5300', 'mangelhaft (sehr nahe an Gewässer)', ''),
		('5400', 'mangelhaft (instabiler Untergrund)', ''),
		('6000', 'aus Wiederholungsmessungen nachgewiesene Stabilität der Vermarkung', ''),
		('8000', 'mangelhaft', ''),
		('9998', 'Stabilität der Vermarkung nicht untersucht', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_GeologischeStabilitaet", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_geologischestabilitaet (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr gute geologische Stabilität', ''),
		('2000', 'gute geologische Stabilität', ''),
		('3000', 'befriedigende geologische Stabilität', ''),
		('4000', 'ausreichende geologische Stabilität', ''),
		('5000', 'mangelhafte geologische Stabilität', ''),
		('9998', 'geologische Stabilität nicht untersucht', '');
		CREATE TABLE ax_klassifikation_ordnung_lagefestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_klassifikation_ordnung_lagefestpunkt IS 'Alias: "AX_Klassifikation_Ordnung_Lagefestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_klassifikation_ordnung_lagefestpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'TP (1) - Hauptdreieckspunkt, Zwischenpunkt 1. Ordnung', ''),
		('2000', 'TP (2) - Trigonometrischer Punkt 2. Ordnung', ''),
		('3000', 'TP (3) - Trigonometrischer Punkt 3. Ordnung', ''),
		('4000', 'TP (4) - Trigonometrischer Punkt 4. Ordnung', ''),
		('5000', 'TP (5) - Trigonometrischer Punkt 5. Ordnung', ''),
		('6000', 'ÜL - Übergeordneter Lagefestpunkt (Berlin)', ''),
		('9000', 'LFP, der nur eine interne Bedeutung hat', ''),
		('9998', 'Nach Quellenlage nicht zu spezifieren', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_GueteDesVermarkungstraegers", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_guetedesvermarkungstra (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr gut', ''),
		('2000', 'gut', ''),
		('3000', 'befriedigend', ''),
		('4000', 'ausreichend', ''),
		('5000', 'unzureichend', ''),
		('9998', 'nicht bekannt', '');
		CREATE TABLE ax_ordnung_schwerefestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_ordnung_schwerefestpunkt IS 'Alias: "AX_Ordnung_Schwerefestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_ordnung_schwerefestpunkt (wert,beschreibung,dokumentation) VALUES
		('0500', 'SFP(0) - Schweregrundnetzpunkt', ''),
		('1000', 'SFP(1) - Schwerefestpunkt 1.Ordnung, Hauptschwerenetzpunkt', ''),
		('2000', 'SFP(2) - Schwerefestpunkt 2.Ordnung', ''),
		('3000', 'SFP(3) - Schwerefestpunkt 3.Ordnung', ''),
		('4000', 'SFP(4) - Schwerefestpunkt 4. Ordnung', ''),
		('9000', 'SFP, der nur eine interne Bedeutung hat', ''),
		('9998', 'nach Quellenlage nicht zu spezifizieren', '');
		CREATE TABLE ax_funktion_referenzstationspunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_referenzstationspunkt IS 'Alias: "AX_Funktion_Referenzstationspunkt", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_referenzstationspunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zentrum', ''),
		('2000', 'Exzentrum', '');
		CREATE TABLE ax_funktion_lagefestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_lagefestpunkt IS 'Alias: "AX_Funktion_Lagefestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_lagefestpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zentrum', ''),
		('2000', 'Exzentrum, Stationspunkt, Nebenstand', ''),
		('3000', 'Zwillingspunkt, Orientierungspunkt', ''),
		('4000', 'Versicherungspunkt', '');
		CREATE TABLE ax_skizzenart_skizze (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_skizzenart_skizze IS 'Alias: "AX_Skizzenart_Skizze", UML-Typ: Enumeration';
		INSERT INTO ax_skizzenart_skizze (wert,beschreibung,dokumentation) VALUES
		('1000', 'Lage-/Einmessungsskizze/Ansicht für die Standardausgabe', ''),
		('2000', 'sonstige Lageskizze', ''),
		('2100', 'sonstige Einmessungsskizze', ''),
		('2200', 'sonstige Ansichtszeichnung oder Foto', ''),
		('2300', 'Randzeichnung', ''),
		('3000', 'Ausschnitt aus der Punktübersicht', ''),
		('3100', 'Luftbildausschnitt mit Punkteintrag', ''),
		('4000', 'Diagramm, Tabelle', '');
		CREATE TABLE ax_funktion_schwerefestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_schwerefestpunkt IS 'Alias: "AX_Funktion_Schwerefestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_schwerefestpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zentrum', ''),
		('2000', 'Exzentrum', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_HoehenstabilitaetAusWiederholungsmessungen", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_hoehenstabilitaetauswi (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr gut', ''),
		('2000', 'gut', ''),
		('3000', 'befriedigend', ''),
		('4000', 'bedenklich', ''),
		('5000', 'unzureichend', ''),
		('9998', 'nicht bekannt', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_GueteDesBaugrundes", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_guetedesbaugrundes (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr gute Stabilität des Baugrundes', ''),
		('2000', 'gute Stabilität des Baugrundes', ''),
		('3000', 'befriedigende Güte des Baugrundes', ''),
		('4000', 'ausreichende Güte des Baugrundes', ''),
		('5000', 'mangelhafte Güte des Baugrundes', ''),
		('9998', 'Güte des Baugrundes nicht untersucht', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_Grundwasserschwankung", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_grundwasserschwankung (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr geringe Grundwasserschwankung (kleiner gleich 0,5 m)', ''),
		('2000', 'geringe Grundwasserschwankung (groesser 0,5 m und kleiner gleich 2 m)', ''),
		('3000', 'mäßige Grundwasserschwankung (groesser 2 m und kleiner gleich 5 m)', ''),
		('4000', 'starke Grundwasserschwankung (groesser 5m und kleiner gleich 10 m)', ''),
		('5000', 'sehr starke Grundwasserschwankung (groesser 10 m)', ''),
		('9998', 'Grundwasserschwankung nicht ermittelt', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_TopographieUndUmwelt", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_topographieundumwelt (wert,beschreibung,dokumentation) VALUES
		('1000', 'keine Topographie- und Umwelteinflüsse', ''),
		('2000', 'geringe Topographie- und Umwelteinflüsse', ''),
		('3000', 'mäßige Topographie- und Umwelteinflüsse', ''),
		('4000', 'starke Topographie- und Umwelteinflüsse', ''),
		('5000', 'sehr starke Topographie- und Umwelteinflüsse', ''),
		('9998', 'Topographie- und Umwelteinflüsse nicht untersucht', '');
		CREATE TABLE ax_klassifikation_wertigkeit_lagefestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_klassifikation_wertigkeit_lagefestpunkt IS 'Alias: "AX_Klassifikation_Wertigkeit_Lagefestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_klassifikation_wertigkeit_lagefestpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Fundamentalpunkt (Punkt auf dem die Position (3D), die schwerebezogene Höhe und die Schwere hoch genau bestimmt worden sind)', ''),
		('2000', 'Übergeordneter Festpunkt', ''),
		('3000', 'Geodätischer Grundnetzpunkt', ''),
		('4000', 'Gebrauchsfestpunkt', ''),
		('5000', 'Untergeordneter Festpunkt', ''),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_gnsstauglichkeit (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_gnsstauglichkeit IS 'Alias: "AX_GNSSTauglichkeit", UML-Typ: Enumeration';
		INSERT INTO ax_gnsstauglichkeit (wert,beschreibung,dokumentation) VALUES
		('1000', 'weitgehende Horizontfreiheit, Mehrwegeffekte nicht wahrscheinlich', ''),
		('1001', 'sehr gute Satelliten-Empfangseigenschaften nachgewiesen', ''),
		('3000', 'eingeschränkte Horizontfreiheit', ''),
		('3001', 'eingeschränkte Horizontfreiheit, Tauglichkeit nachgewiesen', ''),
		('3100', 'Mehrwegeffekte möglich', ''),
		('3101', 'Mehrwegeffekte nachgewiesen', ''),
		('5000', 'Festpunkt nicht geeignet für Satellitenmessverfahren', ''),
		('9998', 'GNSS-Tauglichkeit nicht untersucht', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_Grundwasserstand", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_grundwasserstand (wert,beschreibung,dokumentation) VALUES
		('1000', 'Grundwasserstand sehr tief (groesser 10 m)', ''),
		('2000', 'Grundwasserstand tief (groesser 5 m und kleiner gleich 10 m)', ''),
		('3000', 'Grundwasserstand normal (groesser 2 m und kleiner gleich 5 m)', ''),
		('4000', 'Grundwasserstand hoch (groesser 0,5 m und kleiner gleich 2 m)', ''),
		('5000', 'Grundwasserstand sehr hoch (kleiner gleich 0,5 m)', ''),
		('9000', 'Grundwasserstand abgesenkt', ''),
		('9998', 'Grundwasserstand nicht ermittelt', '');
		CREATE TABLE ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili IS 'Alias: "AX_Punktstabilitaet_Hoehenfestpunkt_VermuteteHoehenstabilitaet", UML-Typ: Enumeration';
		INSERT INTO ax_punktstabilitaet_hoehenfestpunkt_vermutetehoehenstabili (wert,beschreibung,dokumentation) VALUES
		('1000', 'sehr gut', ''),
		('2000', 'gut', ''),
		('3000', 'befriedigend', ''),
		('4000', 'ausreichend', ''),
		('5000', 'mangelhaft', ''),
		('9998', 'nicht bekannt', '');
		CREATE TABLE ax_ordnung_hoehenfestpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_ordnung_hoehenfestpunkt IS 'Alias: "AX_Ordnung_Hoehenfestpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_ordnung_hoehenfestpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'NivP(1) - Haupthöhenpunkt, Zwischenlinienpunkt 1.Ordnung', ''),
		('1001', 'NivP(1) - Netzverdichtung GPS (Niedersachsen)', ''),
		('2000', 'NivP(2) - Nivellementpunkt 2. Ordnung', ''),
		('3000', 'NivP(3) - Nivellementpunkt 3. Ordnung', ''),
		('3001', 'NivP(3) - nivellitisch bestimmter Bodenpunkt für Referenzstation', ''),
		('4000', 'NivP(4) - Nivellementpunkt 4. Ordnung', ''),
		('6000', 'ÜH - Übergeordneter Höhenfestpunkt (Berlin)', ''),
		('9000', 'Höhenfestpunkt, der nur eine interne Bedeutung hat', ''),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', '');
		CREATE TABLE ax_horizontfreiheit_grenzpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_horizontfreiheit_grenzpunkt IS 'Alias: "AX_Horizontfreiheit_Grenzpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_horizontfreiheit_grenzpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Uneingeschränkt', ''),
		('2000', 'Eingeschränkt', ''),
		('3000', 'Nicht geeignet', '');
		CREATE TABLE ax_gruendederausgesetztenabmarkung_grenzpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_gruendederausgesetztenabmarkung_grenzpunkt IS 'Alias: "AX_GruendeDerAusgesetztenAbmarkung_Grenzpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_gruendederausgesetztenabmarkung_grenzpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Grenzpunkt durch bauliche Anlage ausreichend gekennzeichnet', ''),
		('2000', 'Grenzpunkt liegt innerhalb einer baulichen Anlage', ''),
		('3000', 'Grenzpunkt in öffentlich-rechtlichem Bodenordnungsverfahren', ''),
		('4000', 'Grenzpunkt liegt innerhalb eines Baugebietes', ''),
		('5000', 'Grenzpunkt liegt innerhalb oder an einem Gewässer', ''),
		('6000', 'Keine Abmarkung aufgrund von anderweitigen Hindernissen', ''),
		('7000', 'Abmarkung würde unzumutbare Schäden verursachen', ''),
		('8000', 'Langfristige gemeinschaftliche Nutzung der angrenzenden  Flurstücke', ''),
		('9000', 'Angrenzende Flurstücke dienen dem Gemeingebrauch', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_bemerkungzurabmarkung_grenzpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bemerkungzurabmarkung_grenzpunkt IS 'Alias: "AX_BemerkungZurAbmarkung_Grenzpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_bemerkungzurabmarkung_grenzpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Abmarkung unterirdisch gesichert', ''),
		('2000', 'Abmarkung exzentrisch gesichert', ''),
		('3000', 'Abmarkung unterirdisch und exzentrisch gesichert', ''),
		('4000', 'Ohne unterirdische oder exzentrische Sicherung', '');
		CREATE TABLE ax_artderflurstuecksgrenze_besondereflurstuecksgrenze (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderflurstuecksgrenze_besondereflurstuecksgrenze IS 'Alias: "AX_ArtDerFlurstuecksgrenze_BesondereFlurstuecksgrenze", UML-Typ: Enumeration';
		INSERT INTO ax_artderflurstuecksgrenze_besondereflurstuecksgrenze (wert,beschreibung,dokumentation) VALUES
		('1000', 'Strittige Grenze', ''),
		('2001', 'Nicht festgestellte Grenze', ''),
		('2002', 'Verwaltungsstreitverfahren (Grenze)', ''),
		('2003', 'Mittellinie in Gewässern', ''),
		('2004', 'Flurstücksgrenze nicht feststellbar', 'Eine "nicht feststellbare Flurstücksgrenze" ist eine bisher nicht festgestellte Flurstücksgrenze, die nach den Daten des Liegenschaftskatasters nicht mit einer für die Grenzfeststellung erforderlichen Genauigkeit und Zuverlässigkeit in die Örtlichkeit übertragen werden kann und für die ein öffentlich-rechtlicher Grenzfeststellungsvertrag zwischen den Eigentümern mangels (positiver) Einigung nicht zustande kommt. Die fehlende Einigung bedeutet nicht zwangsläufig, dass sich die jeweiligen Eigentümer zivilrechtlich streiten.'),
		('2010', 'Topographische Grenzeinrichtung', ''),
		('2100', 'Topographische Gewaesserbegrenzung', ''),
		('2500', 'Grenze der Region', ''),
		('3000', 'Grenze der Flur', ''),
		('7003', 'Grenze der Gemarkung', ''),
		('7101', 'Grenze der Bundesrepublik Deutschland', ''),
		('7102', 'Grenze des Bundeslandes', ''),
		('7103', 'Grenze des Regierungsbezirks', ''),
		('7104', 'Grenze des Landkreises', ''),
		('7106', 'Grenze der Gemeinde', ''),
		('7107', 'Grenze des Gemeindeteils', ''),
		('7108', 'Grenze der Verwaltungsgemeinschaft', '');
		CREATE TABLE ax_horizontfreiheit_netzpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_horizontfreiheit_netzpunkt IS 'Alias: "AX_Horizontfreiheit_Netzpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_horizontfreiheit_netzpunkt (wert,beschreibung,dokumentation) VALUES
		('1000', 'Uneingeschränkt', ''),
		('2000', 'Eingeschränkt', ''),
		('3000', 'Nicht geeignet', '');
		CREATE TABLE ax_marke (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_marke IS 'Alias: "AX_Marke", UML-Typ: Enumeration';
		INSERT INTO ax_marke (wert,beschreibung,dokumentation) VALUES
		('1000', 'Marke, allgemein', ''),
		('1100', 'Stein', ''),
		('1110', 'Stein, Grenzstein', ''),
		('1111', 'Lochstein', ''),
		('1112', 'Vermessungspunktstein', ''),
		('1120', 'Unbehauener Feldstein', ''),
		('1130', 'Gemeinde- und Waldgrenzstein', ''),
		('1131', 'Gemeindegrenzstein', ''),
		('1132', 'Waldgrenzstein, Forstgrenzstein', ''),
		('1140', 'Kunststoffmarke', ''),
		('1160', 'Landesgrenzstein', ''),
		('1190', 'Stein mit Besonderheiten in Form oder Material', ''),
		('1200', 'Rohr', ''),
		('1201', 'Rohr mit Schutzkappe', ''),
		('1202', 'Rohr mit Kopf', ''),
		('1203', 'Rohr mit Bolzen, oberirdisch', ''),
		('1210', 'Eisenrohr', ''),
		('1211', 'Eisenrohr (mit Schutzkappe)', ''),
		('1212', 'Eisenrohr (ohne Schutzkappe)', ''),
		('1220', 'Kunststoffrohr', ''),
		('1221', 'Kunststoffrohr (mit Schutzkappe)', ''),
		('1222', 'Kunststoffrohr (ohne Schutzkappe)', ''),
		('1230', 'Drainrohr', ''),
		('1240', 'Rohr mit Schutzkasten', ''),
		('1250', 'Zementrohr', ''),
		('1260', 'Glasrohr', ''),
		('1290', 'Tonrohr', ''),
		('1300', 'Bolzen/Nagel', ''),
		('1310', 'Bolzen', ''),
		('1311', 'Adapterbolzen', ''),
		('1320', 'Nagel', ''),
		('1400', 'Meißelzeichen (z. B. Kreuz, Kerbe, Anker)', ''),
		('1410', 'Bohrloch', ''),
		('1500', 'Pfahl', ''),
		('1600', 'Sonstige Marke', ''),
		('1610', 'Marke in Schutzbehälter', ''),
		('1620', 'Flasche', ''),
		('1630', 'Platte', ''),
		('1631', 'Klinkerplatte', ''),
		('1632', 'Granitplatte', ''),
		('1635', 'Platte mit Loch', ''),
		('1640', 'Hohlziegel', ''),
		('1650', 'Klebemarke', ''),
		('1655', 'Schlagmarke', ''),
		('1660', 'Kanaldeckel (Kreuz des Gütesiegels auf Rand)', ''),
		('1670', 'Marke besonderer Ausführung', ''),
		('1700', 'Punkt dauerhaft und gut erkennbar festgelegt', ''),
		('1710', 'Punkt der baulichen Anlage', ''),
		('1711', 'Sockel (roh)', ''),
		('1712', 'Sockel (verputzt)', ''),
		('1713', 'Mauerecke (roh)', ''),
		('1714', 'Mauerecke (verputzt)', ''),
		('1720', 'Grenzsäule', ''),
		('1800', 'Pfeiler', ''),
		('1820', 'Kegel', ''),
		('2100', 'Festlegung 1. Ordnung, Kopf 30x30 cm, Bezugspunkt Platte', ''),
		('2101', 'Festlegung 1. Ordnung, Bezugspunkt Kopf 30x30 cm', ''),
		('2102', 'Festlegung STN 1. Ordnung, Pfeiler-kopf 30x30 cm, Bezugspunkt Platte 60x60 cm, Steinwürfel, Tonkegel', ''),
		('2110', 'Festlegung 2. bis 5. Ordnung, Kopf 16x16 oder 12x12 cm, Bezugspunkt Platte 30x30 cm', ''),
		('2111', 'Festlegung 3. bis 5. Ordnung, Bezugspunkt Kopf 16x16 oder 12x12 cm, Platte 30x30 cm', ''),
		('2120', 'Festlegung 2. bis 4. Ordnung, Kopf 20x20 cm, Bezugspunkt Platte', ''),
		('2121', 'Festlegung 2. bis 4. Ordnung, Bezugspunkt Kopf 20x20 cm,', ''),
		('2130', 'Festlegung 2. bis 4. Ordnung, Kopf 25x25 cm, Bezugspunkt Platte', ''),
		('2131', 'Festlegung 2. bis 4. Ordnung, Bezugspunkt Kopf 25x25 cm,', ''),
		('2132', 'Festlegung STN 3. und 5. Ordnung, Pfeilerkopf 16x16 cm, Bezugspunkt Platte 30x30-40x40 cm', ''),
		('2133', 'Festlegung STN 3. und 5. Ordnung, Pfeilerkopf 16x16 cm, Bezugspunkt Bolzen oder Rotgußkappe im Fels', ''),
		('2134', 'Festlegung TP-Feld, Pfeilerkopf 25x25 cm, Bezugspunkt Platte 35x35 cm', ''),
		('2135', 'Festlegung RBP-Feld, Pfeilerkopf 16x16 cm mit Gravur "TP" und "Dreieck", Bezugspunkt Platte 30x30-35x35 cm', ''),
		('2140', 'Plattformbolzen mit Aufschrift TP', ''),
		('2150', 'Turmbolzen mit Aufschrift TP', ''),
		('2160', 'Leuchtschraube oder -bolzen', ''),
		('2161', 'Schraube (vertikal)', ''),
		('2162', 'Messingmarke oder Messingbolzen', ''),
		('2163', 'Keramikbolzen, oberirdisch', ''),
		('2164', 'Bolzen im Fels, unterirdisch', ''),
		('2165', 'Rotgusskappe im Fels, unterirdisch', ''),
		('2166', 'Messingbolzen (gewölbt), Aufschrift TP und Dreieck', ''),
		('2170', 'Turmbolzen, Festlegungsbolzen oder sonstiger Bolzen, keine weiteren Angaben bekannt oder gespeichert', ''),
		('2180', 'Festlegung 2. Ordnung, Kopf 16x16 cm oder 12x12 cm, Bezugspunkt Platte 60x60 cm', ''),
		('2181', 'Festlegung 2. Ordnung, Bezugspunkt Kopf 16x16 cm oder 12x12 cm, Platte 60x60 cm', ''),
		('2190', 'Festlegung 2. bis 3. Ordnung, Kopf 16x16 cm, Bezugspunkt Platte 40x40 cm', ''),
		('2191', 'Festlegung 2. bis 3. Ordnung, Bezugspunkt Kopf 16x16 cm, Platte 40x40 cm', ''),
		('2192', 'Bergpfeiler ca. 60 cm lang, ohne Platte, Kopf 12x12 cm', ''),
		('2200', 'Pfeiler mit Aufschrift AP', ''),
		('2201', 'Pfeiler mit Aufschrift AP mit Platte, Bezugspunkt Kopf', ''),
		('2210', 'Plattformbolzen mit der Aufschrift AP', ''),
		('2220', 'Turmbolzen mit der Aufschrift AP', ''),
		('2230', 'Festlegung der Wasserstraßenverwaltung, Stein mit Rohr und Stehbolzen, Typ 1', ''),
		('2240', 'Festlegung der Wasserstraßenverwaltung, Stein mit Rohr und Stehbolzen, Typ 2', ''),
		('2250', 'Festlegung der Wasserstraßenverwaltung, Stein mit Rohr und Stehbolzen, Typ 3', ''),
		('2260', 'Festlegung Sachsen-Anhalt SANREF (unterirdischer Granitpfeiler mit Kopfbolzen)', ''),
		('2261', '3D-Bolzen einzementiert in Kopffläche einer Festlegung STN 1. Ordnung', ''),
		('2262', '3D-Bolzen in Kopffläche einer Festle-gung TP-Feld', ''),
		('2263', '3D-Bolzen in Kopffläche einer AF Sachsen 3. Ordnung', ''),
		('2264', '3D-Bolzen in Kopffläche einer AF Sachsen 5. Ordnung', ''),
		('2265', '3D-Bolzen in Granitplatte unter Schutzkasten oder Kunststoff- Schutz-rohr', ''),
		('2266', 'Sondervermarkung, einbetonierter Granitpfeiler 40x40x90 cm mit Zentrumsbolzen in Kopffläche', ''),
		('2300', 'Alte Festlegung der Kgl. Generalkommission und von Kurhessen, Rillenstein', ''),
		('2310', 'Alte Festlegung von Nassau, exz., ehemals 2.O.', ''),
		('2320', 'Alte Festlegung von Nassau, exz., ehemals 3.O.', ''),
		('2330', 'Alte Festlegung von Hessen-Darmstadt, 3.O.', ''),
		('2340', 'Alte Festlegung von Hessen-Darmstadt, 4.O.', ''),
		('2350', 'Alte Festlegung von Hessen-Darmstadt, 3.O. exz.', ''),
		('2360', 'Alte Festlegung von Hessen-Darmstadt, 4.O.exz.', ''),
		('2370', 'Alte Festlegung von Westfalen, 2. O. exz.', ''),
		('2400', 'Alte Festlegung von Württemberg, exz.', ''),
		('2410', 'Alte Festlegung von Württemberg', ''),
		('2420', 'Alte Festlegung von Mecklenburg, ehemals 1. bis 2. Ordnung', ''),
		('2430', 'Alte Festlegung von Mecklenburg, ehemals 2. bis 3. Ordnung', ''),
		('2440', 'Alte Festlegung von Mecklenburg, ehemals 4. Ordnung', ''),
		('2450', 'Alte Festlegung Sachsen, 1. und 2. Ordnung (Nagelsche Säule)', ''),
		('2460', 'Alte Festlegung Sachsen, 3. Ordnung, Pfeilerkopf 35x35 cm mit zentrischer Messingmarke', ''),
		('2470', 'Alte Festlegung Sachsen, 5. Ordnung, Pfeilerkopf 25x25 cm mit zentrischer Messingmarke', ''),
		('2500', 'Alte Festlegung von Baden, exz., Typ 1', ''),
		('2510', 'Alte Festlegung von Baden, exz., Typ 2', ''),
		('2540', 'Alte Festlegung von Baden', ''),
		('2550', 'Rohr mit Schutzkasten, Grundständer', ''),
		('2551', 'Grundständer mit Messingmarke unter Schutzkasten', ''),
		('2552', 'Marke (allgemein) unter Schutzkasten', ''),
		('2553', 'Platte unter Schutzkasten', ''),
		('2560', 'Rohr mit Verschlusskappe und aufgesetztem Lochstein', ''),
		('2600', 'Alte Festlegung in Bayern, exz.', ''),
		('2601', 'gebohrter Granitstein (Bezugspunkt) über Tonrohr', ''),
		('2602', 'gebohrter Granitstein über Tonrohr (Bezugspunkt)', ''),
		('2603', 'gebohrter Granitstein (Bezugspunkt) über Platte', ''),
		('2604', 'gebohrter Granitstein über Platte (Bezugspunkt)', ''),
		('2605', 'gebohrter Granitstein (Bezugspunkt) über Eisenrohr', ''),
		('2606', 'gebohrter Granitstein über Eisenrohr (Bezugspunkt)', ''),
		('2607', 'Betonstein (Bezugspunkt) über Platte', ''),
		('2608', 'Betonstein über Platte (Bezugspunkt)', ''),
		('2609', 'Betonstein mit durchgehendem Bohrloch (Einschaltpunkt-Stein in Bayern)', ''),
		('2610', 'Alte Festlegung in Bayern', ''),
		('2611', 'KT-Stein der österreichischen Katastraltriangulation', ''),
		('2612', 'Stein (Bezugspunkt) über Platte', ''),
		('2613', 'Stein über Platte (Bezugspunkt)', ''),
		('2614', 'Stein (Bezugspunkt) über Tonrohr', ''),
		('2615', 'Stein über Tonrohr (Bezugspunkt)', ''),
		('2616', 'Stein mit Eisenrohr', ''),
		('2620', 'Alte Festlegung in Sachsen-Coburg, 3.O.', ''),
		('2630', 'Alte Festlegung in Sachsen-Coburg, 4.O.', ''),
		('2640', 'Alte Festlegung in Bayern und der Pfalz, HDNP-Stein', ''),
		('2700', 'Festlegung MP-Pfeiler', ''),
		('2710', 'Festlegung Orientierungspunkt', ''),
		('2750', 'Steinpfeiler', ''),
		('2760', 'Betonpfeiler', ''),
		('2770', 'Kreuz (gemeisselt)', ''),
		('2800', 'Knopf', ''),
		('2810', 'Mitte', ''),
		('2820', 'Spitze', ''),
		('2830', 'Kreuz (Mitte)', ''),
		('2840', 'Helmstange', ''),
		('2850', 'Fahnenstange', ''),
		('2860', 'Wetterstange', ''),
		('2870', 'Blitzableiter', ''),
		('2880', 'Antenne', ''),
		('2890', 'Rohrstange', ''),
		('2900', 'Platte, unterirdisch', ''),
		('2901', 'Platte einbetoniert mit zentrischem Messingbolzen', ''),
		('2902', 'Platte einbetoniert mit zentrischem Bohrloch', ''),
		('2903', 'Platte einbetoniert mit zentrischer Keramikmarke', ''),
		('2904', 'Platte einbetoniert mit zentrischer Messingmarke', ''),
		('2910', 'Steinwürfel, unterirdisch', ''),
		('2920', 'Steinplatte, unterirdisch', ''),
		('2930', 'Platte, unterirdisch, 60x60 cm', ''),
		('2940', 'Platte, unterirdisch, 30x30 cm', ''),
		('2950', 'Platte, unterirdisch, mit Stehniet', ''),
		('2951', 'Platte unterirdisch mit Kopfbolzen', ''),
		('2960', 'Platte, unterirdisch, mit Schutzrohr', ''),
		('2970', 'Pfeiler 30x30x90 cm, mit Stehniete', ''),
		('2980', 'Platte, mit Bolzen, unterirdisch, im Schacht', ''),
		('3000', 'Unterirdische Festlegung (des RfL)', ''),
		('3010', 'Unterirdische Festlegung Sonderform', ''),
		('3020', 'Unterirdischer Rammpfahl', ''),
		('3030', 'Unterirdischer Pfeilerbolzen', ''),
		('3040', 'Unterirdischer Bolzen', ''),
		('3050', 'Hamburger Flachpunkt', ''),
		('3060', 'Unterirdische Säule', ''),
		('3070', 'Unterirdischer Rammstab', ''),
		('3100', 'Rohrfestpunkt', ''),
		('3110', 'Rohrfestpunkt, Hamburger Bauart', ''),
		('3120', 'Rohrfestpunkt, Oldenburger Bauart', ''),
		('3130', 'Rohrfestpunkt, Eider Bauart', ''),
		('3140', 'Rohrfestpunkt Nordrhein-Westfalen', ''),
		('3150', 'Rohrfestpunkt Nebenpunkt, flach gegründet', ''),
		('3160', 'Rohrfestpunkt, Celler Bauart', ''),
		('3170', 'Unterirdische Festlegung im Boden', ''),
		('3180', 'Unterirdische Festlegung im Schacht', ''),
		('3190', 'kleine unterirdische Festlegung', ''),
		('3200', 'Mauerbolzen', ''),
		('3210', 'Mauerbolzen, horizontal eingebracht (mit Inschrift)', ''),
		('3220', 'Mauerbolzen, vertikal eingebracht (mit Inschrift)', ''),
		('3230', 'Höhenmarke (des RfL)', ''),
		('3240', 'Kugelbolzen', ''),
		('3250', 'Tonnenbolzen', ''),
		('3260', 'Landeshöhenbolzen', ''),
		('3270', 'Stehbolzen bzw. Bolzen vertikal', ''),
		('3280', 'Stehniete', ''),
		('3290', 'sonstiger horizontaler Bolzen', ''),
		('3300', 'Pfeilerbolzen', ''),
		('3301', 'Pfeilerbolzen, Bezugspunkt Pfeileroberfläche', ''),
		('3310', 'Pfeilerbolzen, Naturstein, Bolzen horizontal', ''),
		('3311', 'Pfeilerbolzen, Naturstein, Bolzen horizontal, Bezugspunkt Pfeileroberfläche', ''),
		('3320', 'Pfeilerbolzen, Naturstein, Bolzen vertikal', ''),
		('3330', 'Pfeilerbolzen, Beton, Bolzen, horizontal', ''),
		('3331', 'Pfeilerbolzen, Beton, Bolzen horizontal, Bezugspunkt Pfeileroberfläche', ''),
		('3340', 'Pfeilerbolzen, Beton, Bolzen vertikal', ''),
		('3350', 'Pfeilerniete, Naturstein, Niete vertikal', ''),
		('3400', 'Rammpfahl', ''),
		('3410', 'Rammpfahl, Bolzen horizontal', ''),
		('3420', 'Rammpfahl, Bolzen vertikal', ''),
		('3810', 'Schraubpfahl', ''),
		('3820', 'Hektometerstein', ''),
		('3830', 'Markstein', ''),
		('3840', 'Schraubbolzen', ''),
		('3845', 'Schraubeisen', ''),
		('3850', 'Lochmarke/-bolzen (ohne Höhentafel)', ''),
		('3860', 'Lochmarke/-bolzen mit Höhentafel', ''),
		('3870', 'Festpunktstein', ''),
		('3880', 'Eichpfahl', ''),
		('4100', 'Gravimeterplatte 80 x 80 oder 60 x 60 cm', ''),
		('4110', 'Gravimeternagel', ''),
		('4120', 'Gravimeterpfeiler 20 x 20 x 100 cm', ''),
		('4130', 'Gravimeterpfeiler 16 x 16 x 60 cm', ''),
		('4140', 'Messingscheibe mit zentrischer Wölbung (Durchm. 8 cm)', ''),
		('4150', 'Stehniete, Messing (Durchmesser 3 cm, Aufschrift SFP)', ''),
		('4160', 'Messplakette, Aufschrift', ''),
		('5100', 'Gewindebolzen (höchste Stelle, Mitte) auf Metallplatte (Betonpfeiler mit Fundament im festen Erdboden)', ''),
		('5150', 'Oberfläche der Metallplatte (höchste Stelle, Mitte) (Betonpfeiler mit Fundament im festen Erdboden)', ''),
		('5200', 'Gewindebolzen (höchste Stelle, Mitte) auf Metallplatte (Gemauerter Pfeiler auf einem Bauwerk)', ''),
		('5250', 'Oberfläche der Metallplatte (höchste Stelle, Mitte) (Gemauerter Pfeiler auf einem Bauwerk)', ''),
		('5300', 'Gewindebolzen (höchste Stelle, Mitte) auf Metallplatte (Stahlpfeiler auf einem Bauwerk)', ''),
		('5350', 'Oberfläche der Metallplatte (höchste Stelle, Mitte) (Stahlpfeiler auf einem Bauwerk)', ''),
		('5400', 'Gewindebolzen (höchste Stelle, Mitte) auf Metallplatte (Seitlich befestigtes Stahlrohr am Bauwerk)', ''),
		('5450', 'Oberfläche der Metallplatte (höchste Stelle, Mitte) (Seitlich befestigtes Stahlrohr am Bauwerk)', ''),
		('5500', 'Gewindebolzen (höchste Stelle, Mitte) auf Metallplatte (Antennenträger)', ''),
		('5550', 'Oberfläche der Metallplatte (höchste Stelle, Mitte) (Antennenträger)', ''),
		('5900', 'GNSS-Antennenhalterung, Oberfläche Platte (Loch) (SL)', ''),
		('9000', 'Marke unter Bemerkung näher definiert', ''),
		('9500', 'Ohne Marke', ''),
		('9600', 'Abmarkung zeitweilig ausgesetzt', ''),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_genauigkeitsstufe_punktort (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_genauigkeitsstufe_punktort IS 'Alias: "AX_Genauigkeitsstufe_Punktort", UML-Typ: Enumeration';
		INSERT INTO ax_genauigkeitsstufe_punktort (wert,beschreibung,dokumentation) VALUES
		('0900', 'Standardabweichung S kleiner 1 mm', ''),
		('1000', 'Standardabweichung S kleiner gleich 2 mm', ''),
		('1100', 'Standardabweichung S kleiner gleich 5 mm', ''),
		('1200', 'Standardabweichung S kleiner gleich 1 cm', ''),
		('1300', 'Standardabweichung S kleiner gleich 1,5 cm', ''),
		('2000', 'Standardabweichung S kleiner gleich 2 cm', ''),
		('2050', 'Standardabweichung S kleiner gleich 2,5 cm', ''),
		('2100', 'Standardabweichung S kleiner gleich 3 cm', ''),
		('2200', 'Standardabweichung S kleiner gleich 6 cm', ''),
		('2300', 'Standardabweichung S kleiner gleich 10 cm', ''),
		('3000', 'Standardabweichung S kleiner gleich 30 cm', ''),
		('3100', 'Standardabweichung S kleiner gleich 60 cm', ''),
		('3200', 'Standardabweichung S kleiner gleich 100 cm', ''),
		('3300', 'Standardabweichung S kleiner gleich 500 cm', ''),
		('5000', 'Standardabweichung S groesser 500 cm', '');
		CREATE TABLE ax_messmethode_schwere (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_messmethode_schwere IS 'Alias: "AX_Messmethode_Schwere", UML-Typ: Enumeration';
		INSERT INTO ax_messmethode_schwere (wert,beschreibung,dokumentation) VALUES
		('1000', 'Aus Absolutgravimetermessung ermittelt', ''),
		('2000', 'Aus Relativgravimetermessung ermittelt', '');
		CREATE TABLE ax_koordinatenstatus_punktort (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_koordinatenstatus_punktort IS 'Alias: "AX_Koordinatenstatus_Punktort", UML-Typ: Enumeration';
		INSERT INTO ax_koordinatenstatus_punktort (wert,beschreibung,dokumentation) VALUES
		('1000', 'Amtliche Koordinaten bzw. amtliche Höhe', 'Gültiger Wert in einem amtlichen Bezugssystem'),
		('2000', 'Weitere gültige Koordinaten bzw. weitere gültige Höhe', 'Nicht in einem amtlichen Bezugssystem'),
		('3000', 'Vorläufige Koordinaten bzw. vorläufige Höhe', ''),
		('4000', 'Zu keiner Zeit gültig gewesene Koordinaten bzw. Höhe', ''),
		('5000', 'Historische (nicht mehr gültige)  Koordinaten bzw. Höhe', ''),
		('5100', 'Koordinaten bzw. Höhe, die sich als fehlerhaft herausgestellt haben', '');
		CREATE TABLE ax_datenerhebung_schwere (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_datenerhebung_schwere IS 'Alias: "AX_Datenerhebung_Schwere", UML-Typ: Enumeration';
		INSERT INTO ax_datenerhebung_schwere (wert,beschreibung,dokumentation) VALUES
		('1000', 'Schwerewert direkt gemessen oder mittels Freiluftreduktion über geringe Entfernung abgeleitet (Höhenunterschied bis 1 m, Horizontalabstand bis 5 m)', ''),
		('2000', 'Schwerewert mittels Freiluftreduktion über größere Entfernung abgeleitet', ''),
		('3000', 'Schwerewert mittels Interpolation unter Verwendung einfacher Bougueranomalien ermittelt', ''),
		('3100', 'Schwerewert im System DHSN 96 durch Abschlag ( -19 10-8 m s-2 ) aus Schwerewert im System DHSN 82 ermittelt (alte Bundesländer)', ''),
		('3200', 'Schwerewert DHSN 96 durch Transformation aus System 71 ermittelt (neue Bundesländer)', ''),
		('4000', 'Schwerewert durch andere Methode ermittelt', ''),
		('9998', 'Methode der Datenerhebung Schwere unbekannt', '');
		CREATE TABLE ax_vertrauenswuerdigkeit_schwere (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_vertrauenswuerdigkeit_schwere IS 'Alias: "AX_Vertrauenswuerdigkeit_Schwere", UML-Typ: Enumeration';
		INSERT INTO ax_vertrauenswuerdigkeit_schwere (wert,beschreibung,dokumentation) VALUES
		('1100', 'Vertrauenswürdigkeitsstufe Ausgleichung', 'Die Identität der Anschlusspunkte ist überprüft. Die Zuverlässigkeitskriterien sind durch Ausgleichung und durch mathematisch-statistiche Testverfahren festgestellt.'),
		('1300', 'Vertrauenswürdigkeitsstufe ohne Ausgleichung kontrolliert', 'Die Zuverlässigkeitskriterien sind auf andere Weise (z.B. durch Vergleich von Schwereanomalien) festgestellt.'),
		('1400', 'Vertrauenswürdigkeitsstufe unkontrolliert', 'Der Schwerewert ist nicht unabhängig überprüft.');
		CREATE TABLE ax_schwereanomalie_schwere_art (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_schwereanomalie_schwere_art IS 'Alias: "AX_Schwereanomalie_Schwere_Art", UML-Typ: Enumeration';
		INSERT INTO ax_schwereanomalie_schwere_art (wert,beschreibung,dokumentation) VALUES
		('1000', 'Geländereduktion (Reliefkorrektion)', ''),
		('2000', 'Freiluftanomalie im System Potsdam 1909 - Normalschwere nach Helmert 1901 - Breite auf dem Krassowski-Ellipsoid', ''),
		('3000', 'Freiluftanomalie im System Potsdam 1909 mit Geländereduktion (Fayeanomalie) - Normalschwere nach Helmert 1901 - Breite auf dem Krassowski-Ellipsoid', ''),
		('3100', 'Freiluftanomalie nach Molodenski: Schwere (im System DHSN 96) im Oberflächenpunkt minus Normalschwere (Formel des GRS 80) im Telluroidpunkt', ''),
		('4000', 'Bougueranomalie im System Potsdam 1909 mit Geländereduktion - Normalschwere nach Helmert 1901 - Breite auf dem Krassowski-Ellipsoid', ''),
		('5000', 'Bougueranomalie im System DHSN 82 mit Freiluft- und Plattenreduktion - Normalschwere im GRS 80', ''),
		('6000', 'Bougueranomalie im System DHSN 96 mit Freiluft- und Plattenreduktion - Normalschwere im GRS 80', ''),
		('9998', 'Art der Anomalie bzw. Reduktion unbekannt', '');
		CREATE TABLE ax_vertrauenswuerdigkeit_punktort (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_vertrauenswuerdigkeit_punktort IS 'Alias: "AX_Vertrauenswuerdigkeit_Punktort", UML-Typ: Enumeration';
		INSERT INTO ax_vertrauenswuerdigkeit_punktort (wert,beschreibung,dokumentation) VALUES
		('1100', 'Vertrauenswürdigkeitsstufe Ausgleichung', 'Vertrauenswürdigkeitsstufe Ausgleichung: Die Vertrauenswürdigkeit ist durch Ausgleichung und durch mathematisch-statistische Testverfahren festgestellt. Bei den örtlichen Ver-messungen sind die Anschlusspunkte auf Identität überprüft.'),
		('1200', 'Vertrauenswürdigkeitsstufe Berechnung', 'Vertrauenswürdigkeitsstufe Berechnung: Die Vertrauenswür-digkeit ist durch Berechnung überprüft. Bei den örtlichen Vermessungen sind die Anschlusspunkte auf Identität überprüft. Die Zuverlässigkeit ist durch Programm festgestellt bzw. ergibt sich durch die rechnerisch wirksam kontrollierte Ermittlung der Position (Doppelbestimmung).'),
		('1300', 'Vertrauenswürdigkeitsstufe Bestimmungsverfahren', 'Vertrauenswürdigkeitsstufe Bestimmungsverfahren: Die Vertrauenswürdigkeit ist durch die Art der Bestimmung der Posi-tion überprüft. Bei den örtlichen Vermessungen sind die Anschlusspunkte auf Identität überprüft. Die Position ist durch wirksame Kontrollen überprüft.'),
		('1400', 'Vertrauenswürdigkeitsstufe ohne Kontrollen', 'Vertrauenswürdigkeitsstufe ohne Kontrollen: Die Berechnung ist nicht überprüft.');
		CREATE TABLE ax_schwerestatus_schwere (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_schwerestatus_schwere IS 'Alias: "AX_Schwerestatus_Schwere", UML-Typ: Enumeration';
		INSERT INTO ax_schwerestatus_schwere (wert,beschreibung,dokumentation) VALUES
		('1000', 'amtlicher Schwerewert (gültiger Wert im amtlichen Schweresystem)', ''),
		('2000', 'weiterer gültiger Schwerewert (nicht im amtlichen Schweresystem)', ''),
		('3000', 'vorläufiger Schwerewert', ''),
		('4000', 'zu keiner Zeit gültig gewesener Schwerewert', ''),
		('5000', 'historischer (nicht mehr gültiger) Schwerewert', ''),
		('5100', 'Schwerewert, der sich als fehlerhaft herausgestellt hat', '');
		CREATE TABLE ax_li_processstep_punktort_description (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_li_processstep_punktort_description IS 'Alias: "AX_LI_ProcessStep_Punktort_Description", UML-Typ: Enumeration';
		INSERT INTO ax_li_processstep_punktort_description (wert,beschreibung,dokumentation) VALUES
		('Erhebung', NULL, 'Erhebung beschreibt den Erfassungszeitpunkt "dateTime", aus dem z.B. das Messjahr für Position, Lage oder Höhe der AFIS-Standardausgabe abgeleitet wird.'),
		('Berechnung', NULL, 'Berechnung beschreibt den Auswertezeitpunkt "dateTime", z.B. von Position, Lage oder Höhe der AFIS-Punkte.');
		CREATE TABLE ax_genauigkeitsstufe_schwere (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_genauigkeitsstufe_schwere IS 'Alias: "AX_Genauigkeitsstufe_Schwere", UML-Typ: Enumeration';
		INSERT INTO ax_genauigkeitsstufe_schwere (wert,beschreibung,dokumentation) VALUES
		('1000', 'Standardabweichung S kleiner 20 10-8m s-2', ''),
		('2000', 'Standardabweichung S kleiner gleich 100 10-8m s-2', ''),
		('3000', 'Standardabweichung S groesser 100 10-8m s-2', ''),
		('4000', 'Als Schwereanschlusspunkt ungeeignet', '');
		CREATE TABLE ax_datenerhebung_punktort (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_datenerhebung_punktort IS 'Alias: "AX_Datenerhebung_Punktort", UML-Typ: Enumeration';
		INSERT INTO ax_datenerhebung_punktort (wert,beschreibung,dokumentation) VALUES
		('0100', 'Aus GNSS-Messung', ''),
		('0110', 'Aus langzeitstatischer GNSS-Messung', ''),
		('0120', 'Aus statischer GNSS-Messung', ''),
		('0130', 'Aus Echtzeit-GNSS-Messung', ''),
		('0200', 'Aus trigonometrischer Messung im TP-Netz', ''),
		('0210', 'Aus lokaler trigonometrischer Messung (innerhalb einer Punktgruppe)', ''),
		('0220', 'Aus netzweiser terrestrischer Messung', ''),
		('1000', 'Aus Katastervermessung ermittelt', ''),
		('1010', 'Aus Katastervermessung mit höchster Lagegenauigkeit (NW)', ''),
		('1020', 'Aus Katastervermessung mit hoher Lagegenauigkeit (NW)', ''),
		('1030', 'Aus Katastervermessung mit mittlerer Lagegenauigkeit (NW)', ''),
		('1040', 'Aus Katastervermessung mit unzureichender Lagegenauigkeit (NW)', ''),
		('1060', 'Aus Vermessung mit höchster Positionsgenauigkeit', ''),
		('1070', 'Aus Vermessung mit mittlerer Positionsgenauigkeit', ''),
		('1080', 'Aus Vermessung mit unterer Positionsgenauigkeit', ''),
		('1100', 'Aufgrund Anforderungen mit Netzanschluss ermittelt', ''),
		('1200', 'Aufgrund Anforderungen mit Bezug zur Flurstücksgrenze ermittelt', ''),
		('1300', 'Aufgrund Anforderungen des LiegVermErlasses ermittelt (NI)', ''),
		('1400', 'Aufgrund Anforderungen des Fortführungerlasses II ermittelt (NI)', ''),
		('1500', 'Aufgrund Anforderungen älterer Vorschriften ermittelt', ''),
		('1600', 'Auf einheitlichem und eindeutigem Raumbezug basierend (E-Koord.) (BW)', ''),
		('1610', 'Auf bislang einheitlichem Raumbezug basierend ermittelt (B-Koord.) (BW)', ''),
		('1620', 'Auf früher gültigem Raumbezug basierend ermittelt(T-Koord.) (BW)', ''),
		('1630', 'Aufgrund Anforderungen des AP-Erlasses (M-V)', ''),
		('1640', 'Aufgrund Anforderungen der LiVermA M-V, basierend auf AP-Feld', ''),
		('1650', 'Aufgrund Anforderungen der LiVermA M-V', ''),
		('1700', 'Aufgrund Anforderungen des LiegVermErlasses LSA (LSA)', ''),
		('1710', 'Aufgrund Anforderungen der Anleitung für die Ausführung von Neuvermessungen, 1953, DDR (LSA)', ''),
		('1720', 'Aufgrund Anforderungen der Liegenschaftsvermessungsordnung 112/82, DDR (LSA)', ''),
		('1800', 'Aus Koordinatentransformation ermittelt', ''),
		('1900', 'Aus sonstiger Vermessung ermittelt', ''),
		('2000', 'Aus Luftbildmessung oder Fernerkundungsdaten ermittelt', ''),
		('3000', 'Aus Netzvermessung ermittelt', ''),
		('3100', 'Aufgrund Anforderungen des Festpunktfelderlasses ermittelt (NI, ST)', ''),
		('3200', 'Aufgrund Anforderungen des Polygonpunktfelderlasses ermittelt (NI)', ''),
		('3300', 'Aus Polygonierungsmessung', ''),
		('4000', 'Aus Katasterunterlagen und Karten für graphische Zwecke ermittelt', ''),
		('4100', 'Aus Katasterzahlen für graphische Zwecke ermittelt', ''),
		('4200', 'Aus Katasterkarten digitalisiert', ''),
		('4210', 'Aus Katasterkarten digitalisiert, Kartenmaßstab M größer gleich 1 zu 1000', ''),
		('4220', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 1000 größer M größer gleich 1 zu 2000', ''),
		('4230', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 2000 größer M größer gleich 1 zu 3000', ''),
		('4240', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 3000 größer M größer gleich 1 zu 5000', ''),
		('4250', 'Aus Katasterkarten digitalisiert, Kartenmaßstab 1 zu 5000 größer M', ''),
		('4260', 'Mit  sonstigen geometrischen Bedingungen und/oder Homogenisierung (M größer gleich 1 zu 1000)', ''),
		('4270', 'Mit Berechnung  oder Abstandsbedingung (M größer gleich 1 zu 1000)', ''),
		('4280', 'Mit sonstigen geometrischen  Bedingungen und/oder Homogenisierung (M kleiner 1 zu 1000)', ''),
		('4290', 'Mit  Berechnung oder Abstandsbedingungen (M kleiner 1 zu 1000)', ''),
		('4300', 'Aus sonstigen Unterlagen digitalisiert', ''),
		('4310', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab M größer gleich 1 zu 1000', ''),
		('4320', 'Aus  sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 1000 größer M größer gleich 1 zu 2000', ''),
		('4330', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 2000 größer M größer gleich 1 zu 3000', ''),
		('4340', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 3000 größer M größer gleich 1 zu 5000', ''),
		('4350', 'Aus sonstigen Unterlagen digitalisiert, Kartenmaßstab 1 zu 5000 größer M', ''),
		('4360', 'Aus sonstigen Unterlagen digitalisiert, mit sonstigen geometrischen Bedingungen und/oder Homogenisierung (M größer gleich 1 zu 1000)', ''),
		('4370', 'Aus sonstigen Unterlagen digitalisiert, Mit Berechnung oder Abstandsbedingung (M größer gleich 1 zu 1000)', ''),
		('4380', 'Aus sonstigen Unterlagen digitalisiert, Mit sonstigen geometrischen Bedingungen und/oder Homogenisierung (M kleiner 1 zu 1000)', ''),
		('4390', 'Aus sonstigen Unterlagen digitalisiert, Mit Berechnung oder Abstandsbedingungen (M kleiner 1 zu 1000)', ''),
		('5000', 'Aus Nivellement', ''),
		('5010', 'Aus geometrischem Nivellement', ''),
		('5020', 'Aus trigonometrischer Höhenübertragung', ''),
		('5030', 'Aus trigonometrischer Präzisionshöhenübertragung', ''),
		('5100', 'Aus Präzisionsnivellement', ''),
		('5110', 'Aus Höhenangaben nivellitisch bestimmt, Messgenauigkeit 1.Ordnung', ''),
		('5120', 'Aus  Höhenangaben nivellitisch bestimmt, Messgenauigkeit 2.Ordnung', ''),
		('5130', 'Aus Höhenangaben  nivellitisch bestimmt, Messgenauigkeit 3.Ordnung', ''),
		('5140', 'Aus Höhenangaben nivellitisch bestimmt, Messgenauigkeit 4.Ordnung', ''),
		('5150', 'Aus geometrischem Präzisionsnivellement, Messgenauigkeit  1. Ordnung', ''),
		('5160', 'Aus geometrischem Präzisionsnivellement, Messgenauigkeit  2. Ordnung', ''),
		('5170', 'Aus geometrischem Präzisionsnivellement, Messgenauigkeit  3. Ordnung', ''),
		('5200', 'Aus Präzisionsnivellement (nur eine Messungsrichtung)', ''),
		('5500', 'Höhe aus Laserscannermessung', ''),
		('6000', 'Aus satellitengeodätischer Messung  und Addition einer Undulation', ''),
		('6100', 'Mittels Quasigeoidundulation abgeleitet von gemessener ellipsoidischer Höhe', ''),
		('6200', 'Mittels Quasigeoidundulation abgeleitet von gemessener Normalhöhe', ''),
		('7000', 'Aus trigonometrischer Messung', ''),
		('8000', 'Aus analoger Unterlage abgeleitet', ''),
		('8100', 'Aus näherungsweiser Berechnung oder Transformation', ''),
		('8200', 'Aus Stereo-Auswertung von Luftbildern', ''),
		('8300', 'Graphisch bestimmt', ''),
		('8310', 'Aus der Topographischen Karte 1 zu 10 000 abgegriffen', ''),
		('8320', 'Aus der Topographischen Karte 1 zu 25 000 abgegriffen', ''),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', '');
		CREATE TABLE ax_schweresystem_schwere (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_schweresystem_schwere IS 'Alias: "AX_Schweresystem_Schwere", UML-Typ: Enumeration';
		INSERT INTO ax_schweresystem_schwere (wert,beschreibung,dokumentation) VALUES
		('1000', 'Schwerewert im System des DHSN 82 (System der Landesvermessung)', ''),
		('1100', 'Schwerewert im System des DSGN 62 (auch als DSN 62 bezeichnet)', ''),
		('1200', 'Schwerewert im System des SGN der DDR (auch als System 71 bezeichnet)', ''),
		('1300', 'Schwerewert im System des DHSN 96 (System der Landesvermessung)', ''),
		('4000', 'Schwerewert im System des ISGN 71 (wissenschaftliches System)', ''),
		('4010', 'Schwerewert im System des DSGN 76 (wissenschaftliches System)', ''),
		('4020', 'Schwerewert im System des DSGN 94 (wissenschaftliches System)', ''),
		('6000', 'Potsdamer Schweresystem 1909', ''),
		('6100', 'Schweresystem der Geophysikalischen Reichsaufnahme 1934 - 1943', '');
		CREATE TABLE ax_blattart_historischesflurstueck (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_blattart_historischesflurstueck IS 'Alias: "AX_Blattart_HistorischesFlurstueck", UML-Typ: Enumeration';
		INSERT INTO ax_blattart_historischesflurstueck (wert,beschreibung,dokumentation) VALUES
		('1000', 'Grundbuchblatt', ''),
		('2000', 'Katasterblatt', '');
		CREATE TABLE ax_qualitaet_hauskoordinate (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_qualitaet_hauskoordinate IS 'Alias: "AX_Qualitaet_Hauskoordinate", UML-Typ: Enumeration';
		INSERT INTO ax_qualitaet_hauskoordinate (wert,beschreibung,dokumentation) VALUES
		('1000', 'gebaeudeumring (A)', ''),
		('2000', 'innerhalbFlurstueck (B)', ''),
		('3000', 'interpoliert (C)', '');
		CREATE TABLE ax_art_punktkennung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_punktkennung IS 'Alias: "AX_Art_Punktkennung", UML-Typ: Enumeration';
		INSERT INTO ax_art_punktkennung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Punktkennung - allgemein', ''),
		('1100', 'Punktkennung - Grenzpunkt', ''),
		('1200', 'Punktkennung - Besonderer Gebäudepunkt', ''),
		('1300', 'Punktkennung - Besonderer topographischer Punkt', ''),
		('1400', 'Punktkennung - Aufnahmepunkt', ''),
		('1500', 'Punktkennung - Sicherungspunkt', ''),
		('1600', 'Punktkennung - Sonstiger Vermessungspunkt', ''),
		('1700', 'Punktkennung - Besonderer Bauwerkspunkt', '');
		CREATE TABLE ax_art_reservierung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_reservierung IS 'Alias: "AX_Art_Reservierung", UML-Typ: Enumeration';
		INSERT INTO ax_art_reservierung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Punktkennung', ''),
		('1300', 'Punktkennung - Grenzpunkt', ''),
		('1400', 'Punktkennung - Besonderer Gebäudepunkt', ''),
		('1500', 'Punktkennung - Besonderer topographischer Punkt', ''),
		('1600', 'Punktkennung - Besonderer Bauwerkspunkt', ''),
		('1700', 'Punktkennung - Aufnahmepunkt', ''),
		('1800', 'Punktkennung - Sicherungspunkt', ''),
		('1900', 'Punktkennung - Sonstiger Vermessungspunkt', ''),
		('2000', 'Punktkennung - Lagefestpunkt', ''),
		('2100', 'Punktkennung - Höhenfestpunkt', ''),
		('2200', 'Punktkennung - Schwerefestpunkt', ''),
		('2300', 'Punktkennung - Referenzstationspunkt', ''),
		('3000', 'Flurstückskennzeichen', 'Eine Reservierung von Folgenummern zu einer Nummer darf sich nur auf aktuelle Flustücke 11001 beziehen und nicht auf dauerhaft reservierte ausfallende Nummern, die keine aktuellen Flustücke haben.'),
		('4000', 'FN-Nummer', ''),
		('5000', 'Abmarkungsprotokollnummer', ''),
		('6000', 'Buchungsblattkennzeichen', ''),
		('6100', 'Katasterblatt', ''),
		('6200', 'Pseudoblatt', ''),
		('6300', 'Erwerberblatt', ''),
		('6400', 'Fiktives Blatt', '');
		CREATE TABLE ax_art_adressat_auszug (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_adressat_auszug IS 'Alias: "AX_Art_Adressat_Auszug", UML-Typ: Enumeration';
		INSERT INTO ax_art_adressat_auszug (wert,beschreibung,dokumentation) VALUES
		('1000', 'Privat', ''),
		('2000', 'Notar', ''),
		('3000', 'Grundbuchamt', ''),
		('4000', 'Finanzamt', ''),
		('5000', 'Bauaufsichtsbehörde', ''),
		('6000', 'Weitere Beteiligte', '');
		CREATE TABLE ax_lagezurerdoberflaeche_bauteil (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagezurerdoberflaeche_bauteil IS 'Alias: "AX_LageZurErdoberflaeche_Bauteil", UML-Typ: Enumeration';
		INSERT INTO ax_lagezurerdoberflaeche_bauteil (wert,beschreibung,dokumentation) VALUES
		('1200', 'Unter der Erdoberfläche', 'Unter der Erdoberfläche bedeutet, dass sich das Bauteil unter der Erdoberfläche befindet. Diese Wertart darf nur mit den Bauarten Keller und Tiefgarage vorkommen.'),
		('1400', 'Aufgeständert', 'Aufgeständert bedeutet, dass ein Bauteil auf Stützen steht.');
		CREATE TABLE ax_lagezurerdoberflaeche_gebaeude (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagezurerdoberflaeche_gebaeude IS 'Alias: "AX_LageZurErdoberflaeche_Gebaeude", UML-Typ: Enumeration';
		INSERT INTO ax_lagezurerdoberflaeche_gebaeude (wert,beschreibung,dokumentation) VALUES
		('1200', 'Unter der Erdoberfläche', 'Unter der Erdoberfläche bedeutet, dass sich das Gebäude unter der Erdoberfläche befindet.'),
		('1400', 'Aufgeständert', 'Aufgeständert bedeutet, dass ein Gebäude auf Stützen steht.');
		CREATE TABLE ax_zustand_gebaeude (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_gebaeude IS 'Alias: "AX_Zustand_Gebaeude", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_gebaeude (wert,beschreibung,dokumentation) VALUES
		('1000', 'In behelfsmäßigem Zustand', ''),
		('2000', 'In ungenutztem Zustand', ''),
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass das Gebäude auf Dauer nicht mehr bewohnt oder genutzt wird.'),
		('2200', 'Verfallen, zerstört', 'Verfallen, zerstört bedeutet, dass sich der ursprüngliche Zustand des Gebäudes durch menschliche oder zeitliche Einwirkungen so verändert hat, dass eine Nutzung nicht mehr möglich ist.'),
		('2300', 'Teilweise zerstört', ''),
		('3000', 'Geplant und beantragt', ''),
		('4000', 'Im Bau', '');
		CREATE TABLE ax_dachgeschossausbau_gebaeude (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dachgeschossausbau_gebaeude IS 'Alias: "AX_Dachgeschossausbau_Gebaeude", UML-Typ: Enumeration';
		INSERT INTO ax_dachgeschossausbau_gebaeude (wert,beschreibung,dokumentation) VALUES
		('1000', 'Nicht ausbaufähig', ''),
		('2000', 'Ausbaufähig', ''),
		('3000', 'Ausgebaut', ''),
		('4000', 'Ausbaufähigkeit unklar', '');
		CREATE TABLE ax_dachform (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dachform IS 'Alias: "AX_Dachform", UML-Typ: Enumeration';
		INSERT INTO ax_dachform (wert,beschreibung,dokumentation) VALUES
		('1000', 'Flachdach', ''),
		('2100', 'Pultdach', ''),
		('2200', 'Versetztes Pultdach', ''),
		('3100', 'Satteldach', ''),
		('3200', 'Walmdach', ''),
		('3300', 'Krüppelwalmdach', ''),
		('3400', 'Mansardendach', ''),
		('3500', 'Zeltdach', ''),
		('3600', 'Kegeldach', ''),
		('3700', 'Kuppeldach', ''),
		('3800', 'Sheddach', ''),
		('3900', 'Bogendach', ''),
		('4000', 'Turmdach', ''),
		('5000', 'Mischform', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_bauweise_gebaeude (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauweise_gebaeude IS 'Alias: "AX_Bauweise_Gebaeude", UML-Typ: Enumeration';
		INSERT INTO ax_bauweise_gebaeude (wert,beschreibung,dokumentation) VALUES
		('1100', 'Freistehendes Einzelgebäude', ''),
		('1200', 'Freistehender Gebäudeblock', ''),
		('1300', 'Einzelgarage', ''),
		('1400', 'Doppelgarage', ''),
		('1500', 'Sammelgarage', ''),
		('2100', 'Doppelhaushälfte', ''),
		('2200', 'Reihenhaus', ''),
		('2300', 'Haus in Reihe', ''),
		('2400', 'Gruppenhaus', ''),
		('2500', 'Gebäudeblock in geschlossener Bauweise', ''),
		('4000', 'Offene Halle', 'Unter einer "Offenen Halle" ist eine Halle zu verstehen, bei der alle vier Seiten offen sind.
Hallen, bei denen eine, zwei oder drei Seiten geschlossen sind, werden nur mit der entsprechenden Gebäudefunktion ohne Belegung der Bauweise erfasst. Die offenen Gebäudeseiten sind mit "Besondere Gebäudelinie" Beschaffenheit "Offene Gebäudelinie" zu belegen'),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_gebaeudefunktion (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_gebaeudefunktion IS 'Alias: "AX_Gebaeudefunktion", UML-Typ: Enumeration';
		INSERT INTO ax_gebaeudefunktion (wert,beschreibung,dokumentation) VALUES
		('1000', 'Wohngebäude', 'Wohngebäude ist ein Gebäude, das zum Wohnen genutzt wird.'),
		('1010', 'Wohnhaus', 'Wohnhaus ist ein Gebäude, in dem Menschen ihren Wohnsitz haben.'),
		('1020', 'Wohnheim', 'Wohnheim ist ein Gebäude, das nach seiner baulichen Anlage und Ausstattung zur Unterbringung von Studenten, Arbeitern u.a. bestimmt ist.'),
		('1021', 'Kinderheim', ''),
		('1022', 'Seniorenheim', ''),
		('1023', 'Schwesternwohnheim', ''),
		('1024', 'Studenten-, Schülerwohnheim', ''),
		('1025', 'Schullandheim', 'Schullandheim ist ein Gebäude in ländlicher Region, in dem sich Schulklassen jeweils für einige Tage zur Erholung und zum Unterricht aufhalten.'),
		('1100', 'Gemischt genutztes Gebäude mit Wohnen', 'Gemischt genutztes Gebäude mit Wohnen ist ein Gebäude, in dem sowohl gewohnt wird, als auch Teile des Gebäude zum Anbieten von Dienstleistungen, zur Durchführung von öffentlichen oder privaten Verwaltungsarbeiten, zur gewerblichen oder industriellen Tätigkeit genutzt werden.'),
		('1110', 'Wohngebäude mit Gemeinbedarf', ''),
		('1120', 'Wohngebäude mit Handel und Dienstleistungen', ''),
		('1121', 'Wohn- und Verwaltungsgebäude', ''),
		('1122', 'Wohn- und Bürogebäude', ''),
		('1123', 'Wohn- und Geschäftsgebäude', 'Wohn- und Geschäftsgebäude ist ein Gebäude, in dem gewohnt wird und in dem sich ein oder mehrere Geschäfte befinden, in denen Waren zum Verkauf angeboten werden.'),
		('1130', 'Wohngebäude mit Gewerbe und Industrie', ''),
		('1131', 'Wohn- und Betriebsgebäude', ''),
		('1210', 'Land- und forstwirtschaftliches Wohngebäude', ''),
		('1220', 'Land- und forstwirtschaftliches Wohn- und Betriebsgebäude', ''),
		('1221', 'Bauernhaus', 'Bauernhaus ist das Wohn- und Betriebsgebäude eines Landwirts.'),
		('1222', 'Wohn- und Wirtschaftsgebäude', ''),
		('1223', 'Forsthaus', 'Forsthaus ist ein Gebäude, das gleichzeitig Wohnhaus und Dienststelle der Försterin oder des Försters ist.'),
		('1310', 'Gebäude zur Freizeitgestaltung', ''),
		('1311', 'Ferienhaus', ''),
		('1312', 'Wochenendhaus', 'Wochenendhaus ist ein Gebäude, in dem dauerhaftes Wohnen möglich, aber nicht gestattet ist. Es dient nur zum zeitlich begrenzten Aufenthalt in der Freizeit, beispielsweise am Wochenende oder im Urlaub und steht i.d.R. in einem besonders dafür ausgewiesenen Gebiet (Wochenendhausgebiet).'),
		('1313', 'Gartenhaus', 'Gartenhaus ist ein eingeschossiges Gebäude in einfacher Ausführung, z.B.ohne Feuerstätte und ohne Toilette. Es dient hauptsächlich dem Unterbringen von Gartengeräten oder dem Überwintern von Pflanzen. Stellt es bei der Nutzung des Gartens eine Ruhe oder Erholungsstätte dar, ist es nur zum vorübergehenden Aufenthalt gedacht, nicht jedoch zum Wohnen oder Übernachten.'),
		('2000', 'Gebäude für Wirtschaft oder Gewerbe', 'Gebäude für Wirtschaft oder Gewerbe ist ein Gebäude, das der Produktion von Waren, der Verteilung von Gütern und dem Angebot von Dienstleistungen dient.'),
		('2010', 'Gebäude für Handel und Dienstleistungen', 'Gebäude für Handel und Dienstleistungen ist ein Gebäude, in dem Arbeitsleistungen, die nicht der Produktion von materiellen Gütern dienen, angeboten werden. Dazu gehört u.a. der Handel (Ankauf, Transport, Verkauf) mit Gütern, Kapital oder Wissen.'),
		('2020', 'Bürogebäude', 'Bürogebäude ist ein Gebäude, in dem private Wirtschaftunternehmen ihre Verwaltungsarbeit durchführen.'),
		('2030', 'Kreditinstitut', 'Kreditinstitut ist ein Gebäude, in dem Unternehmen gewerbsmäßig Geldgeschäfte (Verwaltung von Ersparnissen, Vergabe von Krediten) betreiben, die einen kaufmännisch eingerichteten Geschäftsbetrieb erfordern.'),
		('2040', 'Versicherung', 'Versicherung ist ein Gebäude, in dem Versicherungsunternehmen gewerbsmäßige Versicherungsgeschäfte betreiben.'),
		('2050', 'Geschäftsgebäude', 'Geschäftsgebäude ist ein Gebäude, in dem Ein- und Verkauf von Waren stattfindet.'),
		('2051', 'Kaufhaus', 'Kaufhaus ist ein Gebäude, meist mit mehreren Stockwerken, in dem breite Warensortimente zum Kauf angeboten werden.'),
		('2052', 'Einkaufszentrum', 'Einkaufszentrum ist ein Gebäude oder Gebäudekomplex, in dem mehrere Geschäfte untergebracht sind.'),
		('2053', 'Markthalle', ''),
		('2054', 'Laden', ''),
		('2055', 'Kiosk', 'Kiosk ist ein kleines in meist leichter Bauweise errichtetes Gebäude, das als Verkaufseinrichtung für ein beschränktes Warenangebot dient.'),
		('2056', 'Apotheke', 'Apotheke ist ein Geschäft, in dem Arzneimittel hergestellt und verkauft werden.'),
		('2060', 'Messehalle', 'Messehalle ist ein Gebäude, das zur Ausstellung von Kunstgegenständen oder Wirtschaftsgütern dient.'),
		('2070', 'Gebäude für Beherbergung', ''),
		('2071', 'Hotel, Motel, Pension', 'Hotel, Motel, Pension ist ein Gebäude mit Beherbergungs- und/oder Verpflegungsbetrieb nach Service, Ausstattung und Qualität in verschiedene Kategorien eingeteilt. Das Motel ist besonders eingerichtet für Reisende mit Kraftfahrzeug an verkehrsreichen Straßen.'),
		('2072', 'Jugendherberge', 'Jugendherberge ist eine zur Förderung von Jugendreisen dienende Aufenthalts- und Übernachtungsstätte.'),
		('2073', 'Hütte (mit Übernachtungsmöglichkeit)', 'Hütte (mit Übernachtungsmöglichkeit) ist ein Gebäude außerhalb von Ortschaften, meist in den Bergen, in dem Menschen übernachten und Schutz suchen können.'),
		('2074', 'Campingplatzgebäude', ''),
		('2080', 'Gebäude für Bewirtung', ''),
		('2081', 'Gaststätte, Restaurant', 'Gaststätte, Restaurant ist ein Gebäude, in dem gegen Entgelt Mahlzeiten und Getränke zum Verzehr angeboten werden.'),
		('2082', 'Hütte (ohne Übernachtungsmöglichkeit)', ''),
		('2083', 'Kantine', ''),
		('2090', 'Freizeit- und Vergnügungsstätte', 'Freizeit- und Vergnügungsstätte ist ein Gebäude, in dem man in seiner Freizeit bestimmte Angebote wahrnehmen kann.'),
		('2091', 'Festsaal', ''),
		('2092', 'Kino', ''),
		('2093', 'Kegel-, Bowlinghalle', ''),
		('2094', 'Spielkasino', ''),
		('2100', 'Gebäude für Gewerbe und Industrie', 'Gebäude für Gewerbe und Industrie ist ein Gebäude, dass vorwiegend gewerblichen oder industriellen Zwecken dient.'),
		('2110', 'Produktionsgebäude', 'Produktionsgebäude ist ein Gebäude, das zur Herstellung von Wirtschaftsgütern dient.'),
		('2111', 'Fabrik', 'Fabrik ist ein Gebäude mit technischen Anlagen zur Herstellung von Waren in großen Mengen.'),
		('2112', 'Betriebsgebäude', 'Betriebsgebäude ist ein Gebäude, in dem Arbeitskräfte und Produktionsmittel zusammengefasst sind, um Leistungen zu erbringen oder Güter herzustellen.'),
		('2113', 'Brauerei', ''),
		('2114', 'Brennerei', ''),
		('2120', 'Werkstatt', ''),
		('2121', 'Sägewerk', ''),
		('2130', 'Tankstelle', 'Tankstelle ist ein Gebäude, in dem hauptsächlich Kfz-Kraftstoffe, Schmiermittel und Zubehör verkauft werden, meist mit Einrichtungen zur Durchführung von Wartungs- und Pflegearbeiten von Kraftfahrzeugen.'),
		('2131', 'Waschstraße, Waschanlage, Waschhalle', ''),
		('2140', 'Gebäude für Vorratshaltung', ''),
		('2141', 'Kühlhaus', ''),
		('2142', 'Speichergebäude', ''),
		('2143', 'Lagerhalle, Lagerschuppen, Lagerhaus', 'Lagerhalle, Lagerschuppen, Lagerhaus ist ein Gebäude zur Vorratshaltung von Gütern (z. B. Material, Fertigerzeugnissen).'),
		('2150', 'Speditionsgebäude', 'Speditionsgebäude bezeichnet ein Gebäude mit technischen, organisatorischen und wirtschaftlichen Einrichtungen, die der Beförderung von Gütern über räumliche Entfernungen dienen.'),
		('2160', 'Gebäude für Forschungszwecke', 'Gebäude für Forschungszwecke ist ein Gebäude, in dem Forschung betrieben wird.'),
		('2170', 'Gebäude für Grundstoffgewinnung', ''),
		('2171', 'Bergwerk', ''),
		('2172', 'Saline', 'Saline ist eine Anlage zur Gewinnung von Kochsalz.'),
		('2180', 'Gebäude für betriebliche Sozialeinrichtung', ''),
		('2200', 'Sonstiges Gebäude für Gewerbe und Industrie', ''),
		('2210', 'Mühle', ''),
		('2211', 'Windmühle', 'Windmühle ist ein Gebäude, dessen wesentlicher Bestandteil die an einer Achse befestigten Flächen (Flügel, Schaufeln) sind, die von der Windkraft in Drehung versetzt werden. Sie dient zum Mahlen von Getreide, zum Pumpen von Wasser oder zur Erzeugung von Strom.'),
		('2212', 'Wassermühle', 'Wassermühle ist ein Gebäude mit einem Mühlrad, das von Wasser angetrieben wird.'),
		('2213', 'Schöpfwerk', 'Schöpfwerk ist ein Gebäude, in dem Pumpen Wasser einem höher gelegenen Vorfluter zuführen u. a. zur künstlichen Entwässerung von landwirtschaftlich genutzten Flächen.'),
		('2220', 'Wetterstation', 'Wetterstation ist ein Gebäude, in dem meteorologische Daten erfasst und ausgewertet werden.'),
		('2310', 'Gebäude für Handel und Dienstleistung mit Wohnen', ''),
		('2320', 'Gebäude für Gewerbe und Industrie mit Wohnen', ''),
		('2400', 'Betriebsgebäude zu Verkehrsanlagen (allgemein)', ''),
		('2410', 'Betriebsgebäude für Straßenverkehr', ''),
		('2411', 'Straßenmeisterei', 'Straßenmeisterei ist das Verwaltungsgebäude einer Dienststelle, die für den ordnungsgemäßen Zustand von Straßen verantwortlich ist.'),
		('2412', 'Wartehalle', ''),
		('2420', 'Betriebsgebäude für Schienenverkehr', ''),
		('2421', 'Bahnwärterhaus', ''),
		('2422', 'Lokschuppen, Wagenhalle', ''),
		('2423', 'Stellwerk, Blockstelle', 'Stellwerk, Blockstelle ist ein Gebäude, von dem aus die Signale und Weichen im Bahnhof und auf der freien Strecke für die Züge gestellt werden.'),
		('2424', 'Betriebsgebäude des Güterbahnhofs', ''),
		('2430', 'Betriebsgebäude für Flugverkehr', ''),
		('2431', 'Flugzeughalle', 'Flugzeughalle ist ein Gebäude, in dem Flugzeuge abgestellt, inspiziert und repariert werden.'),
		('2440', 'Betriebsgebäude für Schiffsverkehr', ''),
		('2441', 'Werft (Halle)', ''),
		('2442', 'Dock (Halle)', ''),
		('2443', 'Betriebsgebäude zur Schleuse', ''),
		('2444', 'Bootshaus', ''),
		('2450', 'Betriebsgebäude zur Seilbahn', ''),
		('2451', 'Spannwerk zur Drahtseilbahn', ''),
		('2460', 'Gebäude zum Parken', ''),
		('2461', 'Parkhaus', 'Parkhaus ist ein Gebäude, in dem Fahrzeuge auf mehreren Etagen abgestellt werden.'),
		('2462', 'Parkdeck', ''),
		('2463', 'Garage', 'Garage ist ein Gebäude, in dem Fahrzeuge abgestellt werden.'),
		('2464', 'Fahrzeughalle', 'Fahrzeughalle ist ein Gebäude, in dem Fahrzeuge abgestellt, inspiziert und repariert werden.'),
		('2465', 'Tiefgarage', 'Tiefgarage ist ein Bauwerk unter der Erdoberfläche, in dem Fahrzeuge abgestellt werden'),
		('2500', 'Gebäude zur Versorgung', ''),
		('2501', 'Gebäude zur Energieversorgung', ''),
		('2510', 'Gebäude zur Wasserversorgung', ''),
		('2511', 'Wasserwerk', ''),
		('2512', 'Pumpstation', 'Pumpstation ist ein Gebäude an einem Rohrleitungssystem, in dem eine oder mehrere Pumpen eingebaut sind.'),
		('2513', 'Wasserbehälter', 'Wasserbehälter ist ein Gebäude, in dem Wasser gespeichert wird, das zum Ausgleich der Differenz zwischen Wasserzuführung und -abgabe dient.'),
		('2520', 'Gebäude zur Elektrizitätsversorgung', ''),
		('2521', 'Elektrizitätswerk', ''),
		('2522', 'Umspannwerk', ''),
		('2523', 'Umformer', 'Umformer ist ein kleines Gebäude in dem ein Transformator zum Umformen von Gleichstrom in Wechselstrom oder von Gleichstrom in Gleichstrom anderer Spannung untergebracht ist.'),
		('2527', 'Reaktorgebäude', 'Reaktorgebäude ist ein zentrales Gebäude eines Kernkraftwerkes, in dem aus radioaktivem Material mittels Kernspaltung Wärmeenergie erzeugt wird.'),
		('2528', 'Turbinenhaus', 'Turbinenhaus ist ein Gebäude, in dem eine Kraftmaschine die Energie von strömendem Dampf, Gas, Wasser oder Wind unmittelbar in elektrische Energie umsetzt.'),
		('2529', 'Kesselhaus', ''),
		('2540', 'Gebäude für Fernmeldewesen', ''),
		('2560', 'Gebäude an unterirdischen Leitungen', ''),
		('2570', 'Gebäude zur Gasversorgung', ''),
		('2571', 'Gaswerk', ''),
		('2580', 'Heizwerk', ''),
		('2590', 'Gebäude zur Versorgungsanlage', ''),
		('2591', 'Pumpwerk (nicht für Wasserversorgung)', ''),
		('2600', 'Gebäude zur Entsorgung', ''),
		('2610', 'Gebäude zur Abwasserbeseitigung', ''),
		('2611', 'Gebäude der Kläranlage', ''),
		('2612', 'Toilette', ''),
		('2620', 'Gebäude zur Abfallbehandlung', ''),
		('2621', 'Müllbunker', ''),
		('2622', 'Gebäude zur Müllverbrennung', 'Gebäude zur Müllverbrennung ist ein Gebäude in dem Abfälle mit chemisch/physikalischen und biologischen oder thermischen Verfahren oder Kombination dieser Verfahren behandelt werden.'),
		('2623', 'Gebäude der Abfalldeponie', ''),
		('2700', 'Gebäude für Land- und Forstwirtschaft', 'Gebäude für Land- und Forstwirtschaft ist ein Gebäude, das land- und forstwirtschaftlichen Zwecken dient.'),
		('2720', 'Land- und forstwirtschaftliches Betriebsgebäude', ''),
		('2721', 'Scheune', 'Scheune ist ein Gebäude zur Lagerung landwirtschaftlicher Güter (z. B. Stroh, Heu und Getreide).'),
		('2723', 'Schuppen', ''),
		('2724', 'Stall', 'Stall ist ein Gebäude, in dem Tiere untergebracht sind.'),
		('2726', 'Scheune und Stall', ''),
		('2727', 'Stall für Tiergroßhaltung', ''),
		('2728', 'Reithalle', ''),
		('2729', 'Wirtschaftsgebäude', ''),
		('2732', 'Almhütte', 'Almhütte ist ein einfaches, hoch in den Bergen gelegenes Gebäude, das überwiegend weidewirtschaftlichen Zwecken dient und hauptsächlich im Sommer genutzt wird.'),
		('2735', 'Jagdhaus, Jagdhütte', ''),
		('2740', 'Treibhaus, Gewächshaus', 'Treibhaus, Gewächshaus ist ein Gebäude mit lichtdurchlässigem Dach und Wänden, das durch künstliche Klimagestaltung der Aufzucht oder Produktion von Pflanzen dient.'),
		('2741', 'Treibhaus', 'Treibhaus ist ein Gebäude mit lichtdurchlässigem Dach und Wänden, das durch künstliche Klimagestaltung der Aufzucht oder Produktion von Pflanzen dient.'),
		('2742', 'Gewächshaus, verschiebbar', ''),
		('3000', 'Gebäude für öffentliche Zwecke', 'Gebäude für öffentliche Zwecke ist ein Gebäude das der Allgemeinheit dient.'),
		('3010', 'Verwaltungsgebäude', 'Verwaltungsgebäude ist ein Gebäude, in dem Verwaltungstätigkeiten durchgeführt werden.'),
		('3011', 'Parlament', 'Parlament ist ein Gebäude, in dem die gesetzgebende Volksvertretung (Bundestag, Landtag) tagt.'),
		('3012', 'Rathaus', 'Rathaus ist ein Gebäude, in dem der Vorstand einer Gemeinde seinen Amtssitz hat und/oder Teile der Verwaltung untergebracht sind.'),
		('3013', 'Post', 'Post ist ein Gebäude, in dem die Post Dienstleistungen anbietet.'),
		('3014', 'Zollamt', 'Zollamt ist ein Gebäude für die Zollabfertigung an der Staatsgrenze (Grenzzollamt) oder im Inland (Binnenzollamt).'),
		('3015', 'Gericht', 'Gericht ist ein Gebäude, in dem Rechtsprechung und Rechtspflege stattfinden.'),
		('3016', 'Botschaft, Konsulat', 'Botschaft, Konsulat ist ein Gebäude, in dem eine ständige diplomatische Vertretung ersten Rangs eines fremden Staates oder einer  internationalen Organisation untergebracht ist.'),
		('3017', 'Kreisverwaltung', ''),
		('3018', 'Bezirksregierung', ''),
		('3019', 'Finanzamt', ''),
		('3020', 'Gebäude für Bildung und Forschung', 'Gebäude für Bildung und Forschung ist ein Gebäude, in dem durch Ausbildung Wissen und Können auf verschiedenen Gebieten vermittelt werden bzw. wo neues Wissen durch wissenschaftliche Tätigkeit gewonnen wird.'),
		('3021', 'Allgemein bildende Schule', 'Allgemein bildende Schule ist ein Gebäude, in dem Kindern, Jugendlichen und Erwachsenen durch planmäßigen Unterricht Wissen vermittelt wird.'),
		('3022', 'Berufsbildende Schule', ''),
		('3023', 'Hochschulgebäude (Fachhochschule, Universität)', 'Hochschulgebäude (Fachhochschule, Universität) ist ein Gebäude, in dem Wissenschaften gelehrt und Forschung betrieben wird.'),
		('3024', 'Forschungsinstitut', 'Forschungsinstitut ist ein Gebäude, in dem Forschung betrieben wird.'),
		('3030', 'Gebäude für kulturelle Zwecke', 'Gebäude für kulturelle Zwecke ist ein Gebäude, in dem kulturelle Ereignisse stattfinden sowie ein Gebäude von kulturhistorischer Bedeutung.'),
		('3031', 'Schloss', 'Schloss ist ein Gebäude, das als repräsentativer Wohnsitz vor allem des Adels dient oder diente.'),
		('3032', 'Theater, Oper', 'Theater, Oper ist ein Gebäude, in dem Bühnenstücke aufgeführt werden.'),
		('3033', 'Konzertgebäude', 'Konzertgebäude ist ein Gebäude, in dem Musikaufführungen stattfinden.'),
		('3034', 'Museum', 'Museum ist ein Gebäude, in dem Sammlungen von (historischen) Objekten oder Reproduktionen davon ausgestellt werden.'),
		('3035', 'Rundfunk, Fernsehen', 'Rundfunk-, Fernsehen ist ein Gebäude, in dem Radio- und Fernsehprogramme produziert und gesendet werden.'),
		('3036', 'Veranstaltungsgebäude', 'Veranstaltungsgebäude ist ein Gebäude, das hauptsächlich für kulturelle Zwecke wie z.B. Aufführungen, Ausstellungen, Konzerte genutzt wird'),
		('3037', 'Bibliothek, Bücherei', 'Bibliothek, Bücherei ist ein Gebäude, in dem Bücher und Zeitschriften gesammelt, aufbewahrt und ausgeliehen werden.'),
		('3038', 'Burg, Festung', 'Burg, Festung ist ein Gebäude innerhalb einer befestigten Anlage.'),
		('3040', 'Gebäude für religiöse Zwecke', ''),
		('3041', 'Kirche', 'Kirche ist ein Gebäude, in dem sich Christen zu Gottesdiensten versammeln.'),
		('3042', 'Synagoge', ''),
		('3043', 'Kapelle', 'Kapelle ist ein kleines Gebäude (Gebets-, Tauf-, Grabkapelle) für (christliche) gottesdienstliche Zwecke.'),
		('3044', 'Gemeindehaus', ''),
		('3045', 'Gotteshaus', 'Gotteshaus ist ein Gebäude, in dem Gläubige einer nichtchristlichen Religionsgemeinschaft religiöse Handlungen vollziehen.'),
		('3046', 'Moschee', ''),
		('3047', 'Tempel', ''),
		('3048', 'Kloster', ''),
		('3050', 'Gebäude für Gesundheitswesen', 'Gebäude für Gesundheitswesen ist ein Gebäude, das der ambulanten oder stationären Behandlung und Pflege von Patienten dient.'),
		('3051', 'Krankenhaus', 'Krankenhaus ist ein Gebäude, in dem Kranke behandelt und/oder gepflegt werden.'),
		('3052', 'Heilanstalt, Pflegeanstalt, Pflegestation', ''),
		('3053', 'Ärztehaus, Poliklinik', 'Ärztehaus, Poliklinik ist ein Gebäude, in dem mehrere Ärzte unterschiedlicher Fachrichtung Kranke ambulant behandeln und versorgen.'),
		('3060', 'Gebäude für soziale Zwecke', 'Gebäude für soziale Zwecke ist ein Gebäude, in dem ältere Menschen, Obdachlose, Jugendliche oder Kinder betreut werden.'),
		('3061', 'Jugendfreizeitheim', ''),
		('3062', 'Freizeit-, Vereinsheim, Dorfgemeinschafts-, Bürgerhaus', ''),
		('3063', 'Seniorenfreizeitstätte', ''),
		('3064', 'Obdachlosenheim', ''),
		('3065', 'Kinderkrippe, Kindergarten, Kindertagesstätte', 'Kinderkrippe, Kindergarten, Kindertagesstätte ist ein Gebäude, in dem Kinder im Vorschulalter betreut werden.'),
		('3066', 'Asylbewerberheim', ''),
		('3070', 'Gebäude für Sicherheit und Ordnung', 'Gebäude für Sicherheit und Ordnung ist ein Gebäude, das für Personen und Gegenstände dient, die zur Verhütung oder Bekämpfung von Rechtsverletzungen und zum Katastrophenschutz eingesetzt werden, oder zur Unterbringung von Strafgefangenen.'),
		('3071', 'Polizei', 'Polizei ist ein Gebäude für Polizeibedienstete, die in einem bestimmten Gebiet für Sicherheit und Ordnung zuständig sind.'),
		('3072', 'Feuerwehr', 'Feuerwehr ist ein Gebäude der Feuerwehr, in dem Personen und Geräte zur Brandbekämpfung sowie zu anderen Hilfeleistungen untergebracht sind.'),
		('3073', 'Kaserne', 'Kaserne ist ein Gebäude zur ortsfesten Unterbringung von Angehörigen der Bundeswehr und der Polizei sowie deren Ausrüstung.'),
		('3074', 'Schutzbunker', 'Schutzbunker ist ein Gebäude zum Schutz der Zivilbevölkerung vor militärischen Angriffen.'),
		('3075', 'Justizvollzugsanstalt', 'Justizvollzugsanstalt ist ein Gebäude zur Unterbringung von Untersuchungshäftlingen und Strafgefangenen.'),
		('3080', 'Friedhofsgebäude', 'Friedhofsgebäude ist ein Gebäude, das zur Aufrechterhaltung des Friedhofbetriebes dient (z.B. Verwaltung, Leichenhalle, Krematorium).'),
		('3081', 'Trauerhalle', ''),
		('3082', 'Krematorium', ''),
		('3090', 'Empfangsgebäude', 'Empfangsgebäude ist ein Gebäude mit Wartesaal, Fahrkarten- und Gepäckschalter zur Abwicklung des Straßen-, Schienen-, Seilbahn-, Luft- und Schiffsverkehrs.'),
		('3091', 'Bahnhofsgebäude', ''),
		('3092', 'Flughafengebäude', ''),
		('3094', 'Gebäude zum U-Bahnhof', ''),
		('3095', 'Gebäude zum S-Bahnhof', ''),
		('3097', 'Gebäude zum Busbahnhof', 'Gebäude zum Busbahnhof ist ein Gebäude auf dem Busbahnhof, das zur Abwicklung des Busverkehrs dient.'),
		('3098', 'Empfangsgebäude Schifffahrt', ''),
		('3100', 'Gebäude für öffentliche Zwecke mit Wohnen', ''),
		('3200', 'Gebäude für Erholungszwecke', 'Gebäude für Erholungszwecke ist ein Gebäude zur Freizeitgestaltung mit dem Ziel der Erhaltung und Wiederherstellung der Leistungsfähigkeit des Menschen.'),
		('3210', 'Gebäude für Sportzwecke', 'Gebäude für Sportzwecke ist ein Gebäudes, in dem verschiedene Sportarten ausgeübt werden.'),
		('3211', 'Sport-, Turnhalle', 'Sport-, Turnhalle ist ein Gebäude, das für den Turnunterricht und für sportliche Betätigungen in der Freizeit errichtet und dementsprechend ausgestattet ist.'),
		('3212', 'Gebäude zum Sportplatz', ''),
		('3220', 'Badegebäude', ''),
		('3221', 'Hallenbad', 'Hallenbad ist ein Gebäude mit Schwimmbecken und zugehörigen Einrichtungen (z. B. Umkleidekabinen).'),
		('3222', 'Gebäude im Freibad', ''),
		('3230', 'Gebäude im Stadion', ''),
		('3240', 'Gebäude für Kurbetrieb', ''),
		('3241', 'Badegebäude für medizinische Zwecke', ''),
		('3242', 'Sanatorium', 'Sanatorium ist ein Gebäude mit zugehörigen Einrichtungen, das klimagünstig gelegen ist, unter fachärztlicher Leitung steht und zur Behandlung chronisch Kranker und Genesender bestimmt ist, für die kein Krankenhausaufenthalt in Frage kommt.'),
		('3260', 'Gebäude im Zoo', ''),
		('3261', 'Empfangsgebäude des Zoos', ''),
		('3262', 'Aquarium, Terrarium, Voliere', 'Aquarium, Terrarium, Voliere ist ein Gebäude, in dem Fische und Wasserpflanzen, Reptilien und Amphibien oder Vögel gehalten und gezüchtet werden.'),
		('3263', 'Tierschauhaus', ''),
		('3264', 'Stall im Zoo', ''),
		('3270', 'Gebäude im botanischen Garten', ''),
		('3271', 'Empfangsgebäude des botanischen Gartens', ''),
		('3272', 'Gewächshaus (Botanik)', ''),
		('3273', 'Pflanzenschauhaus', ''),
		('3280', 'Gebäude für andere Erholungseinrichtung', ''),
		('3281', 'Schutzhütte', 'Schutzhütte ist ein Gebäude zum Schutz vor Unwetter.'),
		('3290', 'Touristisches Informationszentrum', 'Touristisches Informationszentrum ist eine Auskunftsstelle für Touristen.'),
		('9998', 'Nach Quellenlage nicht zu spezifizieren', 'Nach Quellenlage nicht zu spezifizieren bedeutet, dass keine Aussage über die Werteart gemacht werden kann.');
		CREATE TABLE ax_art_gebaeudepunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_gebaeudepunkt IS 'Alias: "AX_Art_Gebaeudepunkt", UML-Typ: Enumeration';
		INSERT INTO ax_art_gebaeudepunkt (wert,beschreibung,dokumentation) VALUES
		('1100', 'First', ''),
		('1200', 'Traufe', ''),
		('2100', 'Eingang', ''),
		('2200', 'Lichtschacht', '');
		CREATE TABLE ax_weitere_gebaeudefunktion (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_weitere_gebaeudefunktion IS 'Alias: "AX_Weitere_Gebaeudefunktion", UML-Typ: Enumeration';
		INSERT INTO ax_weitere_gebaeudefunktion (wert,beschreibung,dokumentation) VALUES
		('1000', 'Bankfiliale', 'Bankfiliale ist eine Einrichtung in der Geldgeschäfte getätigt werden.'),
		('1010', 'Hotel', 'Hotel ist ein Beherbergungs- und/oder Verpflegungsbetrieb.'),
		('1020', 'Jugendherberge', 'Jugendherberge ist eine zur Förderung von Jugendreisen dienende Aufenthalts- und Übernachtungsstätte.'),
		('1030', 'Gaststätte', 'Gaststätte ist eine Einrichtung, in der gegen Entgelt Mahlzeiten und Getränke zum sofortigen Verzehr angeboten werden.'),
		('1040', 'Kino', 'Kino ist eine Einrichtung, in der alle Arten von Filmen bzw. Lichtspielen für ein Publikum abgespielt werden.'),
		('1050', 'Spielkasino', 'Spielkasino ist eine Einrichtung, in der öffentlich zugänglich staatlich konzessioniertes Glücksspiel betrieben wird.'),
		('1060', 'Tiefgarage', 'Tiefgarage ist ein Bauwerk unterhalb der Erdoberfläche, in dem Fahrzeuge abgestellt werden.'),
		('1070', 'Parkdeck', 'Parkdeck ist eine Fläche auf einem Gebäude, auf der Fahrzeuge abgestellt werden.'),
		('1080', 'Toilette', 'Toilette ist eine Einrichtung mit sanitären Vorrichtungen zum Verrichtung der Notdurft.'),
		('1090', 'Post', 'Post ist eine Einrichtung, von der aus Briefe, Pakete befördert und weitere Dienstleistungen angeboten werden.'),
		('1100', 'Zoll', 'Zoll ist eine Einrichtung der Zollabfertigung.'),
		('1110', 'Theater', 'Theater ist eine Einrichtung, in der Bühnenstücke aufgeführt werden.'),
		('1120', 'Museum', 'Museum ist eine Einrichtung in der Sammlungen von (historischen) Objekten oder Reproduktionen davon ausgestellt werden.'),
		('1130', 'Bibliothek', 'Bibliothek ist eine Einrichtung, in der Bücher und Zeitschriften gesammelt, aufbewahrt und ausgeliehen werden.'),
		('1140', 'Kapelle', 'Kapelle ist eine Einrichtung für (christliche) gottesdienstliche Zwecke .'),
		('1150', 'Moschee', 'Moschee ist ein Einrichtung, in der sich Muslime zu Gottesdiensten versammeln oder zu anderen Zwecken treffen.'),
		('1160', 'Tempel', ''),
		('1170', 'Apotheke', 'Apotheke ist ein Geschäft, in dem Arzneimittel hergestellt und verkauft werden.'),
		('1180', 'Polizeiwache', 'Polizeiwache ist eine Dienststelle der Polizei.'),
		('1190', 'Rettungsstelle', 'Rettungsstelle ist eine Einrichtung zur Aufnahme, Erstbehandlung und gezielten Weiterverlegung von Patienten mit Erkrankungen und Unfällen aller Art.'),
		('1200', 'Touristisches Informationszentrum', 'Touristisches Informationszentrum ist eine Auskunftsstelle für Touristen.'),
		('1210', 'Kindergarten', 'Kindergarten ist eine Einrichtung, in der Kinder im Vorschulalter betreut werden.'),
		('1220', 'Arztpraxis', 'Arztpraxis ist die Arbeitsstätte eines Arztes.'),
		('1230', 'Supermarkt', ''),
		('1240', 'Geschäft', '');
		CREATE TABLE ax_beschaffenheit_besonderegebaeudelinie (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_beschaffenheit_besonderegebaeudelinie IS 'Alias: "AX_Beschaffenheit_BesondereGebaeudelinie", UML-Typ: Enumeration';
		INSERT INTO ax_beschaffenheit_besonderegebaeudelinie (wert,beschreibung,dokumentation) VALUES
		('1000', 'Offene Gebäudelinie', ''),
		('2100', 'Unverputzt', ''),
		('2200', 'Verputzt', ''),
		('2300', 'Verklinkert', ''),
		('3100', 'Holz', ''),
		('3200', 'Sichtbeton', ''),
		('3300', 'Naturstein', ''),
		('3400', 'Glas', ''),
		('4000', 'Trennlinie nicht eindeutig festgelegt', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_bauart_bauteil (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bauart_bauteil IS 'Alias: "AX_Bauart_Bauteil", UML-Typ: Enumeration';
		INSERT INTO ax_bauart_bauteil (wert,beschreibung,dokumentation) VALUES
		('1100', 'Geringergeschossiger Gebäudeteil', ''),
		('1200', 'Höhergeschossiger Gebäudeteil (nicht Hochhaus)', ''),
		('1300', 'Hochhausgebäudeteil', ''),
		('1400', 'Abweichende Geschosshöhe', ''),
		('2000', 'Keller', ''),
		('2100', 'Tiefgarage', 'Tiefgarage ist ein Bauwerk unter der Erdoberfläche, in dem Fahrzeuge abgestellt werden'),
		('2300', 'Loggia', ''),
		('2350', 'Wintergarten', ''),
		('2400', 'Arkade', ''),
		('2500', 'Auskragende/zurückspringende Geschosse', ''),
		('2510', 'Auskragende Geschosse', ''),
		('2520', 'Zurückspringende Geschosse', ''),
		('2610', 'Durchfahrt im Gebäude', ''),
		('2620', 'Durchfahrt an überbauter Verkehrsstraße', 'Durchfahrt an überbauter Verkehrsstraße ist eine Stelle, an der mit Fahrzeugen durch Gebäude gefahren werden kann.'),
		('2710', 'Schornstein im Gebäude', 'Schornstein im Gebäude ist ein über das Dach  hinausragender Abzugskanal für die Rauchgase einer  Feuerungsanlage oder für andere Abgase.'),
		('2720', 'Turm im Gebäude', 'Turm im Gebäude ist ein hochaufragendes Bauwerk innerhalb eines Gebäudes.'),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_nutzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_nutzung IS 'Alias: "AX_Nutzung", UML-Typ: Enumeration';
		INSERT INTO ax_nutzung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zivil', 'Zivil wird für ein Gebäude verwendet, das privaten, öffentlichen oder religiösen Zwecken dient und nicht militärisch genutzt wird.'),
		('1100', 'Privat', 'Privat bezeichnet ein Gebäude, das wohn- oder privatwirtschaftlichen Zwecken dient.'),
		('1200', 'Öffentlich', 'Öffentlich bedeutet, dass in einem Gebäude Aufgaben der öffentlichen Hand wahrgenommen werden oder dass das Gebäude für die Nutzung durch die Allgemeinheit vorgesehen ist.'),
		('1300', 'Religiös', 'Religiös bezeichnet ein Gebäude, das religiösen Zwecken dient.'),
		('2000', 'Militärisch', 'Militärisch bedeutet, dass das Gebäude von Streitkräften genutzt wird.');
		CREATE TABLE ax_art_verbandsgemeinde (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_verbandsgemeinde IS 'Alias: "AX_Art_Verbandsgemeinde", UML-Typ: Enumeration';
		INSERT INTO ax_art_verbandsgemeinde (wert,beschreibung,dokumentation) VALUES
		('1000', 'Samtgemeinde', 'Samtgemeinde umfasst in Niedersachsen das Gebiet einer Samtgemeinde.'),
		('2000', 'Verbandsgemeinde', 'Verbandsgemeinde umfasst in Rheinland-Pfalz das Gebiet einer Verbandsgemeinde, das aus benachbarten Gemeinden desselben Landkreises gebildet wird.'),
		('3000', 'Amt', 'Amt umfasst das Gebiet eines Amtes, das aus Gemeinden desselben Landkreises besteht.');
		CREATE TABLE ax_art_baublock (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_baublock IS 'Alias: "AX_Art_Baublock", UML-Typ: Enumeration';
		INSERT INTO ax_art_baublock (wert,beschreibung,dokumentation) VALUES
		('1000', 'Nettobaublockfläche', ''),
		('2000', 'Bruttobaublockfläche', '');
		CREATE TABLE ax_artdergebietsgrenze_gebietsgrenze (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdergebietsgrenze_gebietsgrenze IS 'Alias: "AX_ArtDerGebietsgrenze_Gebietsgrenze", UML-Typ: Enumeration';
		INSERT INTO ax_artdergebietsgrenze_gebietsgrenze (wert,beschreibung,dokumentation) VALUES
		('7101', 'Grenze der Bundesrepublik Deutschland', 'Grenze der Bundesrepublik Deutschland begrenzt das Gebiet der Bundesrepublik Deutschland oder eines Kondominiums.'),
		('7102', 'Grenze des Bundeslandes', 'Grenze des Bundeslandes begrenzt das Gebiet einer Verwaltungseinheit auf der Bundeslandebene.'),
		('7103', 'Grenze des Regierungsbezirks', 'Grenze des Regierungsbezirks begrenzt das Gebiet einer Verwaltungseinheit auf der Regierungsbezirksebene.'),
		('7104', 'Grenze des Kreises/Region', 'Grenze des Kreises/Region begrenzt das Gebiet einer Verwaltungseinheit auf der Kreisebene.'),
		('7105', 'Grenze der Verwaltungsgemeinschaft', 'Grenze der Verwaltungsgemeinschaft begrenzt das Gebiet einer Verwaltungseinheit auf der Verwaltungsgemeinschaftsebene.'),
		('7106', 'Grenze der Gemeinde', 'Grenze der Gemeinde begrenzt ein kommunales Gebiet auf der Gemeindeebene.'),
		('7107', 'Grenze des Gemeindeteils', 'Grenze des Gemeindeteils begrenzt das Gebiet einer Verwaltungseinheit auf der Gemeindeteilebene.');
		CREATE TABLE ax_sonstigeangaben_bodenschaetzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_sonstigeangaben_bodenschaetzung IS 'Alias: "AX_SonstigeAngaben_Bodenschaetzung", UML-Typ: Enumeration';
		INSERT INTO ax_sonstigeangaben_bodenschaetzung (wert,beschreibung,dokumentation) VALUES
		('1100', 'Nass, zu viel Wasser (Wa+)', ''),
		('1200', 'Trocken, zu wenig Wasser (Wa-)', ''),
		('1300', 'Besonders günstige Wasserverhältnisse (Wa gt)', ''),
		('1400', 'Rieselwasser, künstliche Bewässerung (RiWa)', ''),
		('2100', 'Unbedingtes Wiesenland (W)', ''),
		('2200', 'Streuwiese (Str)', ''),
		('2300', 'Hutung (Hu)', ''),
		('2400', 'Acker-Hackrain (A-Hack)', ''),
		('2500', 'Grünland-Hackrain (Gr-Hack)', ''),
		('2600', 'Garten (G)', ''),
		('3000', 'Neukultur (N)', ''),
		('4000', 'Tiefkultur (T)', ''),
		('5000', 'Geringstland (Ger)', ''),
		('9000', 'Nachschätzung erforderlich', '');
		CREATE TABLE ax_kulturart_musterlandesmusterundvergleichsstueck (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_kulturart_musterlandesmusterundvergleichsstueck IS 'Alias: "AX_Kulturart_MusterLandesmusterUndVergleichsstueck", UML-Typ: Enumeration';
		INSERT INTO ax_kulturart_musterlandesmusterundvergleichsstueck (wert,beschreibung,dokumentation) VALUES
		('1000', 'Ackerland (A)', ''),
		('2000', 'Acker-Grünland (AGr)', ''),
		('3000', 'Grünland (Gr)', ''),
		('4000', 'Grünland-Acker (GrA)', '');
		CREATE TABLE ax_entstehungsartoderklimastufewasserverhaeltnisse_bodensc (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_entstehungsartoderklimastufewasserverhaeltnisse_bodensc IS 'Alias: "AX_EntstehungsartOderKlimastufeWasserverhaeltnisse_Bodenschaetzung", UML-Typ: Enumeration';
		INSERT INTO ax_entstehungsartoderklimastufewasserverhaeltnisse_bodensc (wert,beschreibung,dokumentation) VALUES
		('1000', 'Diluvium (D)', ''),
		('1100', 'Diluvium über Alluvium (DAl)', ''),
		('1200', 'Diluvium über Löß (DLö)', ''),
		('1300', 'Diluvium über Verwitterung (DV)', ''),
		('1400', 'Diluvium, gesteinig (Dg)', ''),
		('1410', 'Diluvium, gesteinig über Alluvium (DgAl)', ''),
		('1420', 'Diluvium, gesteinig über Löß (DgLö)', ''),
		('1430', 'Diluvium, gesteinig über Verwitterung (DgV)', ''),
		('2000', 'Löß (Lö)', ''),
		('2100', 'Löß über Diluvium (LöD)', ''),
		('2110', 'Löß, Diluvium, Gesteinsböden (LöDg)', ''),
		('2120', 'Löß, Diluvium, Verwitterung (LöDV)', ''),
		('2200', 'Löß über Alluvium (LöAl)', ''),
		('2300', 'Löß über Verwitterung (LöV)', ''),
		('2310', 'Löß, Verwitterung, Gesteinsböden (LöVg)', ''),
		('2400', 'Löß über Verwitterung, gesteinig (LöVg)', ''),
		('3000', 'Alluvium (Al)', ''),
		('3100', 'Alluvium über Diluvium (AlD)', ''),
		('3200', 'Alluvium über Löß (AlLö)', ''),
		('3300', 'Alluvium über Verwitterung (AlV)', ''),
		('3400', 'Alluvium, gesteinig (Alg)', ''),
		('3410', 'Alluvium, gesteinig über Diluvium (AlgD)', ''),
		('3420', 'Alluvium, gesteinig über Löß (AlgLö)', ''),
		('3430', 'Alluvium, gesteinig über Verwitterung (AlgV)', ''),
		('3500', 'Alluvium, Marsch (AlMa)', ''),
		('3610', 'Alluvium, Moor (AlMo)', ''),
		('3620', 'Moor, Alluvium (MoAI)', ''),
		('3700', 'Mergel (Me)', ''),
		('4000', 'Verwitterung (V)', ''),
		('4100', 'Verwitterung über Diluvium (VD)', ''),
		('4200', 'Verwitterung über Alluvium (VAl)', ''),
		('4300', 'Verwitterung über Löß (VLö)', ''),
		('4400', 'Verwitterung, Gesteinsböden (Vg)', ''),
		('4410', 'Verwitterung, Gesteinsböden über Diluvium (VgD)', ''),
		('5000', 'Entstehungsart nicht erkennbar (-)', ''),
		('6100', 'Klimastufe 8° C und darüber (a)', ''),
		('6200', 'Klimastufe 7,9° - 7,0° C (b)', ''),
		('6300', 'Klimastufe 6,9° - 5,7° C (c)', ''),
		('6400', 'Klimastufe 5,6° C und darunter (d)', ''),
		('7100', 'Wasserstufe (1)', ''),
		('7200', 'Wasserstufe (2)', ''),
		('7300', 'Wasserstufe (3)', ''),
		('7400', 'Wasserstufe (4)', ''),
		('7410', 'Wasserstufe (4-)', ''),
		('7500', 'Wasserstufe (5)', ''),
		('7510', 'Wasserstufe (5-)', ''),
		('7520', 'Wasserstufe (3-)', ''),
		('7530', 'Wasserstufe (3+4)', '');
		CREATE TABLE ax_sonstigeangaben_musterlandesmusterundvergleichsstueck (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_sonstigeangaben_musterlandesmusterundvergleichsstueck IS 'Alias: "AX_SonstigeAngaben_MusterLandesmusterUndVergleichsstueck", UML-Typ: Enumeration';
		INSERT INTO ax_sonstigeangaben_musterlandesmusterundvergleichsstueck (wert,beschreibung,dokumentation) VALUES
		('1100', 'Nass, zu viel Wasser (Wa+)', ''),
		('1200', 'Trocken, zu wenig Wasser (Wa-)', ''),
		('1300', 'Besonders günstige Wasserverhältnisse (Wa gt)', ''),
		('1400', 'Rieselwasser, künstliche Bewässerung (RiWa)', ''),
		('2100', 'Unbedingtes Wiesenland (W)', ''),
		('2200', 'Streuwiese (Str)', ''),
		('2300', 'Hutung (Hu)', ''),
		('2400', 'Acker-Hackrain (A-Hack)', ''),
		('2500', 'Grünland-Hackrain (Gr-Hack)', ''),
		('2600', 'Garten (G)', ''),
		('5000', 'Geringstland (Ger)', '');
		CREATE TABLE ax_kulturart_bodenschaetzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_kulturart_bodenschaetzung IS 'Alias: "AX_Kulturart_Bodenschaetzung", UML-Typ: Enumeration';
		INSERT INTO ax_kulturart_bodenschaetzung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Ackerland (A)', ''),
		('2000', 'Acker-Grünland (AGr)', ''),
		('3000', 'Grünland (Gr)', ''),
		('4000', 'Grünland-Acker (GrA)', '');
		CREATE TABLE ax_klassifizierung_bewertung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_klassifizierung_bewertung IS 'Alias: "AX_Klassifizierung_Bewertung", UML-Typ: Enumeration';
		INSERT INTO ax_klassifizierung_bewertung (wert,beschreibung,dokumentation) VALUES
		('1110', 'Unbebautes Grundstück', ''),
		('1120', 'Unbebautes Grundstück mit Gebäude von untergeordneter Bedeutung', ''),
		('1130', 'Unbebautes Grundstück mit einem dem Verfall preisgegebenen Gebäude', ''),
		('1140', 'Unbebautes Grundstück für Erholungs- und Freizeitzwecke', ''),
		('1210', 'Einfamilienhausgrundstück', ''),
		('1220', 'Zweifamilienhausgrundstück', ''),
		('1230', 'Mietwohngrundstück', ''),
		('1240', 'Gemischtgenutztes Grundstück', ''),
		('1250', 'Geschäftsgrundstück', ''),
		('1260', 'Sonstiges bebautes Grundstück', ''),
		('1310', 'Einfamilienhaus auf fremdem Grund und Boden', ''),
		('1320', 'Zweifamilienhaus auf fremdem Grund und Boden', ''),
		('1330', 'Mietwohngrundstück, Mietwohngebäude auf fremdem Grund und Boden', ''),
		('1340', 'Gemischtgenutztes Grundstück, gemischtgenutztes Gebäude auf fremdem Grund und Boden', ''),
		('1350', 'Geschäftsgrundstück, Geschäftsgebäude auf fremdem Grund und Boden', ''),
		('1360', 'Sonstige bebaute Grundstücke, sonstige Gebäude auf fremdem Grund und Boden', ''),
		('2110', 'Landwirtschaftliche Nutzung', ''),
		('2120', 'Hopfen', ''),
		('2130', 'Spargel', ''),
		('2190', 'Sonstige Sonderkulturen', ''),
		('2200', 'Holzung', ''),
		('2300', 'Weingarten (allgemein)', ''),
		('2310', 'Weingarten 1', ''),
		('2320', 'Weingarten 2', ''),
		('2330', 'Weingarten 3', ''),
		('2340', 'Weingarten 4', ''),
		('2350', 'Weingarten 5', ''),
		('2360', 'Weingarten 6', ''),
		('2370', 'Weingarten 7', ''),
		('2380', 'Weingarten 8', ''),
		('2390', 'Weingarten 9', ''),
		('2410', 'Gartenland', ''),
		('2420', 'Obstplantage', ''),
		('2430', 'Baumschule', ''),
		('2440', 'Anbaufläche unter Glas', ''),
		('2450', 'Kleingarten', ''),
		('2510', 'Weihnachtsbaumkultur', ''),
		('2520', 'Saatzucht', ''),
		('2530', 'Teichwirtschaft', ''),
		('2610', 'Abbauland der Land- und Forstwirtschaft', ''),
		('2620', 'Geringstland', 'Geringstland sind Flächen geringster Ertragsfähigkeit ohne Wertzahlen nach dem Bodenschätzungsgesetz, das sind unkultivierte Moor- und Heideflächen (sofern nicht gesondert geführt), ehemals bodengeschätzte Flächen und ehemalige Weinbauflächen, die ihren Kulturzustand verloren haben.'),
		('2630', 'Unland', ''),
		('2640', 'Moor', 'Moor ist eine unkultivierte Fläche mit einer (mindestens 20 cm starken) Auflage aus vertorften und vermoorten Pflanzenresten, soweit sie nicht als Torfstich benutzt wird.'),
		('2650', 'Heide', 'Heide ist eine unkultivierte, sandige, überwiegend mit Heidekraut oder Ginster bewachsene Fläche.'),
		('2700', 'Reet', 'Reet ist eine ständig oder zeitweise unter Wasser stehende und mit Reet bewachsene Fläche.'),
		('2710', 'Reet I', 'Reetfläche, deren Nutzung eingestuft ist in Güteklasse I (gut).'),
		('2720', 'Reet II', 'Reetfläche, deren Nutzung eingestuft ist in Güteklasse II (mittel).'),
		('2730', 'Reet III', 'Reetfläche, deren Nutzung eingestuft ist in Güteklasse III (gering).'),
		('2800', 'Nebenfläche des Betriebs der Land- und Forstwirtschaft', ''),
		('2899', 'Noch nicht klassifiziert', '');
		CREATE TABLE ax_merkmal_musterlandesmusterundvergleichsstueck (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_merkmal_musterlandesmusterundvergleichsstueck IS 'Alias: "AX_Merkmal_MusterLandesmusterUndVergleichsstueck", UML-Typ: Enumeration';
		INSERT INTO ax_merkmal_musterlandesmusterundvergleichsstueck (wert,beschreibung,dokumentation) VALUES
		('1000', 'Musterstück (M)', ''),
		('2000', 'Landesmusterstück (L)', ''),
		('3000', 'Vergleichsstück (V)', '');
		CREATE TABLE ax_zustandsstufeoderbodenstufe_bodenschaetzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustandsstufeoderbodenstufe_bodenschaetzung IS 'Alias: "AX_ZustandsstufeOderBodenstufe_Bodenschaetzung", UML-Typ: Enumeration';
		INSERT INTO ax_zustandsstufeoderbodenstufe_bodenschaetzung (wert,beschreibung,dokumentation) VALUES
		('1100', 'Zustandsstufe (1)', ''),
		('1200', 'Zustandsstufe (2)', ''),
		('1300', 'Zustandsstufe (3)', ''),
		('1400', 'Zustandsstufe (4)', ''),
		('1500', 'Zustandsstufe (5)', ''),
		('1600', 'Zustandsstufe (6)', ''),
		('1700', 'Zustandsstufe (7)', ''),
		('1800', 'Zustandsstufe Misch- und Schichtböden sowie künstlichveränderte Böden (-)', ''),
		('2100', 'Bodenstufe (I)', ''),
		('2200', 'Bodenstufe (II)', ''),
		('2300', 'Bodenstufe (III)', ''),
		('2400', 'Bodenstufe Misch- und Schichtböden sowie künstlich veränderte Böden (-)', ''),
		('3100', 'Bodenstufe (II+III)', ''),
		('3200', 'Bodenstufe ("(III)")', ''),
		('3300', 'Bodenstufe (IV)', '');
		CREATE TABLE ax_bedeutung_grablochderbodenschaetzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bedeutung_grablochderbodenschaetzung IS 'Alias: "AX_Bedeutung_GrablochDerBodenschaetzung", UML-Typ: Enumeration';
		INSERT INTO ax_bedeutung_grablochderbodenschaetzung (wert,beschreibung,dokumentation) VALUES
		('1100', 'Grabloch, bestimmend, lagerichtig (innerhalb der Fläche)', ''),
		('1200', 'Grabloch, bestimmend, lagerichtig (außerhalb des Abschnitts)', ''),
		('1300', 'Grabloch, nicht lagerichtig, im Abschnitt nicht vorhanden', ''),
		('2000', 'Grabloch für Muster-, Landesmuster-, Vergleichsstück', ''),
		('3000', 'Grabloch, nicht bestimmend', '');
		CREATE TABLE ax_zustandsstufeoderbodenstufe_musterlandesmusterundvergle (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustandsstufeoderbodenstufe_musterlandesmusterundvergle IS 'Alias: "AX_ZustandsstufeOderBodenstufe_MusterLandesmusterUndVergleichsstueck", UML-Typ: Enumeration';
		INSERT INTO ax_zustandsstufeoderbodenstufe_musterlandesmusterundvergle (wert,beschreibung,dokumentation) VALUES
		('1100', 'Zustandsstufe (1)', ''),
		('1200', 'Zustandsstufe (2)', ''),
		('1300', 'Zustandsstufe (3)', ''),
		('1400', 'Zustandsstufe (4)', ''),
		('1500', 'Zustandsstufe (5)', ''),
		('1600', 'Zustandsstufe (6)', ''),
		('1700', 'Zustandsstufe (7)', ''),
		('1800', 'Zustandsstufe Misch- und Schichtböden sowie künstlichveränderte Böden (-)', ''),
		('2100', 'Bodenstufe (I)', ''),
		('2200', 'Bodenstufe (II)', ''),
		('2300', 'Bodenstufe (III)', ''),
		('2400', 'Bodenstufe Misch- und Schichtböden sowie künstlich veränderte Böden (-)', '');
		CREATE TABLE ax_entstehungsartoderklimastufewasserverhaeltnisse_musterl (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_entstehungsartoderklimastufewasserverhaeltnisse_musterl IS 'Alias: "AX_EntstehungsartOderKlimastufeWasserverhaeltnisse_MusterLandesmusterUndVergleichsstueck", UML-Typ: Enumeration';
		INSERT INTO ax_entstehungsartoderklimastufewasserverhaeltnisse_musterl (wert,beschreibung,dokumentation) VALUES
		('1000', 'Diluvium (D)', ''),
		('1100', 'Diluvium über Alluvium (DAl)', ''),
		('1200', 'Diluvium über Löß (DLö)', ''),
		('1300', 'Diluvium über Verwitterung (DV)', ''),
		('1400', 'Diluvium, gesteinig (Dg)', ''),
		('1410', 'Diluvium, gesteinig über Alluvium (DgAl)', ''),
		('1420', 'Diluvium, gesteinig über Löß (DgLö)', ''),
		('1430', 'Diluvium, gesteinig über Verwitterung (DgV)', ''),
		('2000', 'Löß (Lö)', ''),
		('2100', 'Löß über Diluvium (LöD)', ''),
		('2200', 'Löß über Alluvium (LöAl)', ''),
		('2300', 'Löß über Verwitterung (LöV)', ''),
		('3000', 'Alluvium (Al)', ''),
		('3100', 'Alluvium über Diluvium (AlD)', ''),
		('3200', 'Alluvium über Löß (AlLö)', ''),
		('3300', 'Alluvium über Verwitterung (AlV)', ''),
		('3400', 'Alluvium, gesteinig (Alg)', ''),
		('3410', 'Alluvium, gesteinig über Diluvium (AlgD)', ''),
		('3420', 'Alluvium, gesteinig über Löß (AlgLö)', ''),
		('3430', 'Alluvium, gesteinig über Verwitterung (AlgV)', ''),
		('4000', 'Verwitterung (V)', ''),
		('4100', 'Verwitterung über Diluvium (VD)', ''),
		('4200', 'Verwitterung über Alluvium (VAl)', ''),
		('4300', 'Verwitterung über Löß (VLö)', ''),
		('4400', 'Verwitterung, Gesteinsböden (Vg)', ''),
		('4410', 'Verwitterung, Gesteinsböden über Diluvium (VgD)', ''),
		('5000', 'Entstehungsart nicht erkennbar (-)', ''),
		('6100', 'Klimastufe 8 Grad C und darüber (a)', ''),
		('6200', 'Klimastufe 7,9 Grad - 7,0 Grad C (b)', ''),
		('6300', 'Klimastufe 6,9 Grad - 5,7 Grad C (c)', ''),
		('6400', 'Klimastufe 5,6 Grad C und darunter (d)', ''),
		('7100', 'Wasserstufe (1)', ''),
		('7200', 'Wasserstufe (2)', ''),
		('7300', 'Wasserstufe (3)', ''),
		('7400', 'Wasserstufe (4)', ''),
		('7410', 'Wasserstufe (4-)', ''),
		('7500', 'Wasserstufe (5)', ''),
		('7510', 'Wasserstufe (5-)', '');
		CREATE TABLE ax_bodenart_bodenschaetzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bodenart_bodenschaetzung IS 'Alias: "AX_Bodenart_Bodenschaetzung", UML-Typ: Enumeration';
		INSERT INTO ax_bodenart_bodenschaetzung (wert,beschreibung,dokumentation) VALUES
		('1100', 'Sand (S)', ''),
		('1200', 'Anlehmiger Sand (Sl)', ''),
		('2100', 'Lehmiger Sand (lS)', ''),
		('2200', 'Stark lehmiger Sand (SL)', ''),
		('3100', 'Lehm (L)', ''),
		('3200', 'Sandiger Lehm (sL)', ''),
		('4100', 'Ton (T)', ''),
		('4200', 'Schwerer Lehm (LT)', ''),
		('5000', 'Moor (Mo)', ''),
		('6110', 'Sand, Moor (SMo)', ''),
		('6120', 'Lehmiger Sand, Moor (lSMo)', ''),
		('6130', 'Lehm, Moor (LMo)', ''),
		('6140', 'Ton, Moor (TMo)', ''),
		('6210', 'Moor,Sand (MoS)', ''),
		('6220', 'Moor, Lehmiger Sand (MolS)', ''),
		('6230', 'Moor, Lehm (MoL)', ''),
		('6240', 'Moor, Ton (MoT)', ''),
		('7110', 'Sand auf sandigem Lehm (S/sL)', ''),
		('7120', 'Sand auf Lehm (S/L)', ''),
		('7130', 'Sand auf schwerem Lehm (S/LT)', ''),
		('7140', 'Sand auf Ton (S/T)', ''),
		('7210', 'Anlehmiger Sand auf Lehm (Sl/L)', ''),
		('7220', 'Anlehmiger Sand auf schwerem Lehm (Sl/LT)', ''),
		('7230', 'Anlehmiger Sand auf Ton (Sl/T)', ''),
		('7310', 'Lehmiger Sand auf schwerem Lehm (lS/LT)', ''),
		('7320', 'Lehmiger Sand auf Sand (lS/S)', ''),
		('7330', 'Lehmiger Sand auf Ton (lS/T)', ''),
		('7400', 'Stark lehmiger Sand auf Ton (SL/T)', ''),
		('7510', 'Ton auf stark lehmigen Sand (T/SL)', ''),
		('7520', 'Ton auf lehmigen Sand (T/lS)', ''),
		('7530', 'Ton auf anlehmigen Sand (T/Sl)', ''),
		('7540', 'Ton auf Sand (T/S)', ''),
		('7610', 'Schwerer Lehm auf lehmigen Sand (LT/lS)', ''),
		('7620', 'Schwerer Lehm auf anlehmigen Sand (LT/Sl)', ''),
		('7630', 'Schwerer Lehm auf Sand (LT/S)', ''),
		('7710', 'Lehm auf anlehmigen Sand (L/Sl)', ''),
		('7720', 'Lehm auf Sand (L/S)', ''),
		('7800', 'Sandiger Lehm auf Sand (sL/S)', ''),
		('8110', 'Sand auf Moor (S/Mo)', ''),
		('8120', 'Lehmiger Sand auf Moor (lS/Mo)', ''),
		('8130', 'Lehm auf Moor (L/Mo)', ''),
		('8140', 'Ton auf Moor (T/Mo)', ''),
		('8210', 'Moor auf Sand (Mo/S)', ''),
		('8220', 'Moor auf lehmigen Sand (Mo/lS)', ''),
		('8230', 'Moor auf Lehm (Mo/L)', ''),
		('8240', 'Moor auf Ton (Mo/T)', ''),
		('9120', 'Bodenwechsel vom Lehm zu Moor (L+Mo)', ''),
		('9130', 'Lehmiger Sand mit starkem Steingehalt (lSg)', ''),
		('9140', 'Lehm mit starkem Steingehalt (Lg)', ''),
		('9150', 'lehmiger Sand mit Steinen und Blöcken (lS+St)', ''),
		('9160', 'Lehm mit Steinen und Blöcken L+St)', ''),
		('9170', 'Steine und Blöcke mit  lehmigem Sand (St+lS)', ''),
		('9180', 'Steine und Blöcke mit  Lehm (St+L)', ''),
		('9190', 'lehmiger Sand mit Felsen (lS+Fe)', ''),
		('9200', 'Lehm mit Felsen (L+Fe)', ''),
		('9210', 'Felsen mit lehmigem Sand (Fe+lS)', ''),
		('9220', 'Felsen mit Lehm (Fe+L)', ''),
		('9310', 'Sand auf lehmigen Sand (S/lS)', ''),
		('9320', 'Anlehmiger Sand auf Mergel (Sl/Me)', ''),
		('9330', 'Anlehmiger Sand auf sandigem Lehm (Sl/sL)', ''),
		('9340', 'Lehmiger Sand auf Lehm (lS/L)', ''),
		('9350', 'Lehmiger Sand auf Mergel (lS/Me)', ''),
		('9360', 'Lehmiger Sand auf sandigem Lehm (lS/sL)', ''),
		('9370', 'Lehmiger Sand, Mergel (lSMe)', ''),
		('9380', 'Lehmiger Sand, Moor auf Mergel (lSMo/Me)', ''),
		('9390', 'Anlehmiger Sand, Moor (SlMo)', ''),
		('9410', 'Lehm auf Mergel (L/Me)', ''),
		('9420', 'Lehm, Moor auf Mergel (LMo/Me)', ''),
		('9430', 'Schwerer Lehm auf Moor (LT/Mo)', ''),
		('9440', 'Ton auf Mergel (T/Me)', ''),
		('9450', 'Moor auf Mergel (Mo/Me)', ''),
		('9460', 'Moor, Lehm auf Mergel (MoL/Me)', ''),
		('9470', 'Moor, Mergel (MoMe)', ''),
		('9480', 'LößDiluvium(LöD)', ''),
		('9490', 'AlluviumDiluvium(AlD)', '');
		CREATE TABLE ax_bodenart_musterlandesmusterundvergleichsstueck (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bodenart_musterlandesmusterundvergleichsstueck IS 'Alias: "AX_Bodenart_MusterLandesmusterUndVergleichsstueck", UML-Typ: Enumeration';
		INSERT INTO ax_bodenart_musterlandesmusterundvergleichsstueck (wert,beschreibung,dokumentation) VALUES
		('1100', 'Sand (S)', ''),
		('1200', 'Anlehmiger Sand (Sl)', ''),
		('2100', 'Lehmiger Sand (lS)', ''),
		('2200', 'Stark lehmiger Sand (SL)', ''),
		('3100', 'Lehm (L)', ''),
		('3200', 'Sandiger Lehm (sL)', ''),
		('4100', 'Ton (T)', ''),
		('4200', 'Schwerer Lehm (LT)', ''),
		('5000', 'Moor (Mo)', ''),
		('6110', 'Sand, Moor (SMo)', ''),
		('6120', 'Lehmiger Sand, Moor (lSMo)', ''),
		('6130', 'Lehm, Moor (LMo)', ''),
		('6140', 'Ton, Moor (TMo)', ''),
		('6210', 'Moor, Sand (MoS)', ''),
		('6220', 'Moor, Lehmiger Sand (MolS)', ''),
		('6230', 'Moor, Lehm (MoL)', ''),
		('6240', 'Moor, Ton (MoT)', ''),
		('7110', 'Sand auf sandigem Lehm (S/sL)', ''),
		('7120', 'Sand auf Lehm (S/L)', ''),
		('7130', 'Sand auf schwerem Lehm (S/LT)', ''),
		('7140', 'Sand auf Ton (S/T)', ''),
		('7210', 'Anlehmiger Sand auf Lehm (Sl/L)', ''),
		('7220', 'Anlehmiger Sand auf schwerem Lehm (Sl/LT)', ''),
		('7230', 'Anlehmiger Sand auf Ton (Sl/T)', ''),
		('7310', 'Lehmiger Sand auf schwerem Lehm (lS/LT)', ''),
		('7320', 'Lehmiger Sand auf Ton (lS/T)', ''),
		('7400', 'Stark lehmiger Sand auf Ton (SL/T)', ''),
		('7510', 'Ton auf stark lehmigen Sand (T/SL)', ''),
		('7520', 'Ton auf lehmigen Sand (T/lS)', ''),
		('7530', 'Ton auf anlehmigen Sand (T/Sl)', ''),
		('7540', 'Ton auf Sand (T/S)', ''),
		('7610', 'Schwerer Lehm auf lehmigen Sand (LT/lS)', ''),
		('7620', 'Schwerer Lehm auf anlehmigen Sand (LT/Sl)', ''),
		('7630', 'Schwerer Lehm auf Sand (LT/S)', ''),
		('7710', 'Lehm auf anlehmigen Sand (L/Sl)', ''),
		('7720', 'Lehm auf Sand (L/S)', ''),
		('7800', 'Sandiger Lehm auf Sand (sL/S)', ''),
		('8110', 'Sand auf Moor (S/Mo)', ''),
		('8120', 'Lehmiger Sand auf Moor (lS/Mo)', ''),
		('8130', 'Lehm auf Moor (L/Mo)', ''),
		('8140', 'Ton auf Moor (T/Mo)', ''),
		('8210', 'Moor auf Sand (Mo/S)', ''),
		('8220', 'Moor auf lehmigen Sand (Mo/lS)', ''),
		('8230', 'Moor auf Lehm (Mo/L)', ''),
		('8240', 'Moor auf Ton (Mo/T)', '');
		CREATE TABLE ax_landschaftstyp (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_landschaftstyp IS 'Alias: "AX_Landschaftstyp", UML-Typ: Enumeration';
		INSERT INTO ax_landschaftstyp (wert,beschreibung,dokumentation) VALUES
		('1100', 'Gebirge', 'Gebirge bezeichnet eine zusammenhängende größere Erhebung der Erdoberfläche. Es besteht aus einzelnen Bergen und Hochflächen, die durch Täler und Senken gegliedert sind.'),
		('1200', 'Berg', 'Berg bezeichnet eine über die Umgebung deutlich herausragende Geländeerhebung, einzeln oder als Teil eines Gebirges.'),
		('1300', 'Niederung/Senke', 'Niederung/Senke bezeichnet ein tief liegendes Land an Flussläufen und Küsten.'),
		('1400', 'Tal', 'Tal bezeichnet eine langgestreckte, offene Hohlform der Erdoberfläche.'),
		('1500', 'Tiefebene', 'Tiefebene bezeichnet einen Teil der Erdoberfläche mit fehlenden oder kaum wahrnehmbaren Höhenunterschieden in einer Höhenlage bis etwa 200 m über NN.'),
		('1600', 'Plateau/Hochebene', 'Plateau/Hochebene, bezeichnet einen Teil der Erdoberfläche mit fehlenden oder kaum wahrnehmbaren Höhenunterschieden in einer Höhenlage ab etwa 200 m über NN.'),
		('1700', 'Mündungsgebiet', 'Mündungsgebiet bezeichnet die typische, durch Ablagerung von Schwebestoffen entstandene Landschaft im Bereich der Mündung eines fließenden Gewässers in ein anderes Binnengewässer oder in ein Meer.'),
		('1800', 'Dünenlandschaft', 'Dünenlandschaft ist eine, vom Wind gebildete, durch Sandanhäufungen geprägte Landschaft.'),
		('1900', 'Waldlandschaft', 'Waldlandschaft ist eine größere und zusammenhängende Landschaft, bestehend aus mit Forstpflanzen bestockten Flächen (Wald) sowie mit dem Wald verbundene und ihm dienende Flächen.'),
		('2000', 'Inselgruppe', 'Inselgruppe ist eine Gruppe mehrerer nahe beieinander liegender Inseln geologisch gleicher Entstehung.'),
		('2100', 'Seenlandschaft', 'Seenlandschaft ist eine durch zahlreiche, nahe beieinander liegende Binnenseen geprägte Landschaft.'),
		('2200', 'Siedlungslandschaft', 'Siedlungslandschaft ist eine durch Siedlungen geprägte Landschaft.'),
		('2300', 'Moorlandschaft', 'Moorlandschaft ist eine durch Moore geprägte Landschaft.'),
		('2400', 'Heidelandschaft', 'Heidelandschaft ist eine waldfreie Landschaft der unteren Höhenstufen, die von einer mehr oder weniger lockeren Zwergstrauchformation geprägt wird.'),
		('2500', 'Wattlandschaft', 'Wattlandschaft ist ein ebener, im Wirkungsbereich der Gezeiten liegender Küstenstreifen, der bei Hochwasser überflutet wird, bei Niedrigwasser jedoch trocken fällt, wobei das Wasser durch oft tiefe Furchen (Priele) abfließt.');
		CREATE TABLE ax_art_verband (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_verband IS 'Alias: "AX_Art_Verband", UML-Typ: Enumeration';
		INSERT INTO ax_art_verband (wert,beschreibung,dokumentation) VALUES
		('1000', 'Planungsverband', ''),
		('2000', 'Region', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_behoerde (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_behoerde IS 'Alias: "AX_Behoerde", UML-Typ: Enumeration';
		INSERT INTO ax_behoerde (wert,beschreibung,dokumentation) VALUES
		('1000', 'Grundbuchamt', ''),
		('1100', 'Katasteramt', ''),
		('1200', 'Finanzamt', ''),
		('1300', 'Flurbereinigungsbehörde', ''),
		('1400', 'Forstamt', ''),
		('1500', 'Wasserwirtschaftsamt', ''),
		('1600', 'Straßenbauamt', ''),
		('1700', 'Gemeindeamt', ''),
		('1800', 'Landratsamt', ''),
		('1900', 'Kreis- oder Stadtverwaltung', ''),
		('2000', 'Wasser- und Bodenverband', ''),
		('2100', 'Umlegungsstelle', ''),
		('2200', 'Landesvermessungsverwaltung', 'Landesvermessungsverwaltung ist eine Fachstelle, die für die Aufgaben des amtlichen  Vermessungswesens, insbesondere für die Landesvermessung und des Liegenschaftskatasters in einem Bundesland zuständig ist.'),
		('2300', 'ÖbVI', ''),
		('2400', 'Bundeseisenbahnvermögen', ''),
		('2500', 'Landwirtschaftskammer', '');
		CREATE TABLE ax_administrative_funktion (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_administrative_funktion IS 'Alias: "AX_Administrative_Funktion", UML-Typ: Enumeration';
		INSERT INTO ax_administrative_funktion (wert,beschreibung,dokumentation) VALUES
		('1001', 'Bundesrepublik', ''),
		('2001', 'Land', ''),
		('2002', 'Freistaat', ''),
		('2003', 'Freie und Hansestadt', ''),
		('3001', 'Regierungsbezirk', ''),
		('3002', 'Freie Hansestadt', ''),
		('3003', 'Bezirk', ''),
		('3004', 'Stadt (Bremerhaven)', ''),
		('4001', 'Kreis', ''),
		('4002', 'Landkreis', ''),
		('4003', 'Kreisfreie Stadt', ''),
		('4007', 'Ursprünglich gemeindefreies Gebiet', ''),
		('4008', 'Ortsteil', ''),
		('4009', 'Region', ''),
		('5002', 'Verbandsfreie Gemeinde', ''),
		('5004', 'Große kreisangehörige Stadt', ''),
		('5006', 'Verwaltungsgemeinschaft', ''),
		('5007', 'Amt', ''),
		('5008', 'Samtgemeinde', ''),
		('5012', 'Gemeinde, die sich einer erfüllenden Gemeinde bedient', ''),
		('5013', 'Erfüllende Gemeinde', ''),
		('6001', 'Gemeinde', ''),
		('6002', 'Ortsgemeinde', ''),
		('6003', 'Stadt', ''),
		('6004', 'Kreisangehörige Stadt', ''),
		('6005', 'Große Kreisstadt', ''),
		('6006', 'Amtsangehörige Stadt', ''),
		('6007', 'Amtsangehörige Landgemeinde', ''),
		('6008', 'Amtsangehörige Gemeinde', ''),
		('6009', 'Kreisangehörige Gemeinde', ''),
		('6010', 'Mitgliedsgemeinde einer Verwaltungsgemeinschaft', ''),
		('6011', 'Mitgliedsgemeinde', ''),
		('6012', 'Markt', ''),
		('6013', 'Große kreisangehoerige Stadt', ''),
		('6014', 'Kreisangehörige Gemeinde, die die Bezeichnung Stadt führt', ''),
		('6015', 'Gemeindefreies Gebiet', ''),
		('7001', 'Gemeindeteil', ''),
		('7003', 'Gemarkung', ''),
		('7004', 'Stadtteil', ''),
		('7005', 'Stadtbezirk', ''),
		('7007', 'Ortsteil (Gemeinde)', ''),
		('8001', 'Kondominium', '');
		CREATE TABLE ax_bezeichnung_verwaltungsgemeinschaft (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bezeichnung_verwaltungsgemeinschaft IS 'Alias: "AX_Bezeichnung_Verwaltungsgemeinschaft", UML-Typ: Enumeration';
		INSERT INTO ax_bezeichnung_verwaltungsgemeinschaft (wert,beschreibung,dokumentation) VALUES
		('1000', 'Samtgemeinde', 'Samtgemeinde umfasst in Niedersachsen das Gebiet einer Samtgemeinde.'),
		('2000', 'Verbandsgemeinde', 'Verbandsgemeinde umfasst in Rheinland-Pfalz das Gebiet einer Verbandsgemeinde, das aus benachbarten Gemeinden desselben Landkreises 
		gebildet wird.'),
		('3000', 'Amt', 'Amt umfasst das Gebiet eines Amtes, das aus Gemeinden desselben Landkreises besteht.'),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_funktion_schutzgebietnachwasserrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_schutzgebietnachwasserrecht IS 'Alias: "AX_Funktion_SchutzgebietNachWasserrecht", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_schutzgebietnachwasserrecht (wert,beschreibung,dokumentation) VALUES
		('2111', 'Öffentliche Wasserversorgung', 'Öffentliche Wasserversorgung ist die Aufgabe eines Wasserschutzgebietes. Ziel ist es, die Gewässer der öffentlichen Wasserversorgung vor nachteiligen Einwirkungen zu schützen.'),
		('2112', 'Grundwasseranreicherung', 'Grundwasseranreicherung ist eine Methode, das Grundwasser durch Infiltration (Zugang von Wasser in die Erdrinde) in einem fest definierten Gebiet (Wasserschutzgebiet) anzureichern.'),
		('2113', 'Verhütung von schädlichem Abfluss', 'Verhütung von schädlichem Abfluss ist eine Aufgabe eines Wasserschutzgebietes. In diesem Gebiet soll das schädliche Abfließen von Niederschlagswasser sowie das Abschwemmen und der Eintrag von Bodenbestandteilen, Dünge- oder Pflanzenbehandlungsmitteln in Gewässer verhindert werden.');
		CREATE TABLE ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc IS 'Alias: "AX_ArtDerFestlegung_SchutzgebietNachNaturUmweltOderBodenschutzrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_schutzgebietnachnaturumweltoderbodensc (wert,beschreibung,dokumentation) VALUES
		('1670', 'Nationalpark', 'Nationalpark ist ein rechtsverbindlich festgesetztes einheitlich zu schützendes Gebiet, das großräumig und von besonderer Eigenart ist, im überwiegenden Teil die Voraussetzungen eines Naturschutzgebietes erfüllt und sich in einem vom Menschen nicht oder nur wenig beeinflussten Zustand befindet.'),
		('1680', 'Lärmschutzbereich', ''),
		('1690', 'Biosphärenreservat', 'Biosphärenreservat ist ein rechtsverbindlich festgesetztes einheitlich zu schützendes und zu entwickelndes Gebiet, das 
		1. großräumig und für bestimmte Landschaftstypen charakteristisch ist, 
		2. in wesentlichen Teilen seines Gebietes die Voraussetzungen eines Naturschutzgebietes, im Übrigen überwiegend eines Landschaftsschutzgebietes erfüllt, 
		3. vornehmlich der Erhaltung, Entwicklung oder Wiederherstellung einer durch hergebrachte vielfältige Nutzung geprägten Landschaft und der darin historisch gewachsenen Arten- und Biotopvielfalt, einschließlich Wild- und frühere Kulturformen wirtschaftlich genutzter oder nutzbarer Tier- und Pflanzenarten dient und 
		4. beispielhaft der Entwicklung und Erprobung von Naturgütern besonders schonenden Wirtschaftsweise dient.');
		CREATE TABLE ax_artderfestlegung_anderefestlegungnachstrassenrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_anderefestlegungnachstrassenrecht IS 'Alias: "AX_ArtDerFestlegung_AndereFestlegungNachStrassenrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_anderefestlegungnachstrassenrecht (wert,beschreibung,dokumentation) VALUES
		('1210', 'Bundesfernstraßengesetz', ''),
		('1220', 'Anbauverbot', ''),
		('1230', 'Anbauverbot nach Bundesfernstraßengesetz', ''),
		('1231', 'Anbauverbot (40m)', ''),
		('1232', 'Anbauverbot (20m)', ''),
		('1240', 'Anbaubeschränkung', ''),
		('1241', 'Anbaubeschränkung (100m)', ''),
		('1242', 'Anbaubeschränkung (40m)', ''),
		('1250', 'Veränderungssperre nach Bundesfernstraßengesetz', ''),
		('1260', 'Landesstraßengesetz', ''),
		('1270', 'Anbauverbot nach Landesstraßengesetz', ''),
		('1280', 'Veränderungssperre', '');
		CREATE TABLE ax_artderfestlegung_schutzgebietnachwasserrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_schutzgebietnachwasserrecht IS 'Alias: "AX_ArtDerFestlegung_SchutzgebietNachWasserrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_schutzgebietnachwasserrecht (wert,beschreibung,dokumentation) VALUES
		('1510', 'Wasserschutzgebiet', 'Wasserschutzgebiet ist ein Schutzgebiet von Wassergewinnungsanlagen mit Nutzungsbeschränkungen, um ober- und unterirdisches Wasser vor Verunreinigungen zu schützen.'),
		('1520', 'Heilquellenschutzgebiet', 'Heilquellenschutzgebiet ist ein Schutzgebiet, in dem Handlungen untersagt sind, die den Bestand oder die Beschaffenheit der staatlich anerkannten Heilquellen gefährden.');
		CREATE TABLE ax_besonderefunktion_forstrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_besonderefunktion_forstrecht IS 'Alias: "AX_BesondereFunktion_Forstrecht", UML-Typ: Enumeration';
		INSERT INTO ax_besonderefunktion_forstrecht (wert,beschreibung,dokumentation) VALUES
		('1000', 'Ohne besondere gesetzliche Bindung', ''),
		('1010', 'Ohne besondere gesetzliche Bindung nach LWaldG- Holzboden', ''),
		('2000', 'Schutzwald', ''),
		('2010', 'Schutzwald - Holzboden', ''),
		('3000', 'Erholungswald', ''),
		('3010', 'Erholungswald - Holzboden', ''),
		('4000', 'Bannwald', ''),
		('4010', 'Nationalpark - Holzboden', ''),
		('5010', 'Naturschutzgebiet - Holzboden', ''),
		('6000', 'Schutz- und Erholungswald', ''),
		('6010', 'Schutz- und Erholungswald - Holzboden', ''),
		('7010', 'Nationalpark - Nichtholzboden', ''),
		('8010', 'Naturschutzgebiet - Nichtholzboden', ''),
		('9000', 'Andere Forstbetriebsfläche', ''),
		('9010', 'Nichtholzboden', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_zone_schutzzone (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zone_schutzzone IS 'Alias: "AX_Zone_Schutzzone", UML-Typ: Enumeration';
		INSERT INTO ax_zone_schutzzone (wert,beschreibung,dokumentation) VALUES
		('1010', 'Zone I', 'Zone I ist eine räumlich begrenzte Fläche eines Schutzgebiets, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1020', 'Zone II', 'Zone II ist eine räumlich begrenzte Fläche eines Schutzgebiets, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1030', 'Zone III', 'Zone III ist eine räumlich begrenzte Fläche eines Schutzgebiets, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1031', 'Zone IIIa', 'Zone IIIa ist eine räumlich begrenzte Fläche eines Wasser- oder Heilquellenschutzgebietes, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1032', 'Zone IIIb', 'Zone IIIb ist eine räumlich begrenzte Fläche eines Wasser- oder Heilquellenschutzgebietes, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1040', 'Zone IV', 'Zone IV ist eine räumlich begrenzte Fläche eines Wasser- oder Heilquellenschutzgebietes, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1060', 'Kernzone', 'Kernzone ist eine räumlich begrenzte Fläche eines Biosphärenreservats, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1070', 'Pflegezone', 'Pflegezone ist eine räumlich begrenzte Fläche eines Biosphärenreservates, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1080', 'Entwicklungszone', 'Entwicklungszone ist eine räumlich begrenzte Fläche eines Biosphärenreservates, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('1090', 'Regenerationszone', 'Regenerationszone ist eine räumlich begrenzte Fläche eines Biosphärenreservates, für die die Fachverwaltung besondere Schutzbestimmungen festgelegt hat.'),
		('9997', 'Attribut trifft nicht zu', 'Attribut trifft nicht zu bedeutet, dass keiner der in der Werteliste aufgeführten Attributwerte dem vorliegenden Sachverhalt entspricht.');
		CREATE TABLE ax_artderfestlegung_klassifizierungnachstrassenrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_klassifizierungnachstrassenrecht IS 'Alias: "AX_ArtDerFestlegung_KlassifizierungNachStrassenrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_klassifizierungnachstrassenrecht (wert,beschreibung,dokumentation) VALUES
		('1100', 'Klassifizierung nach Bundes- oder Landesstraßengesetz', ''),
		('1110', 'Bundesautobahn', ''),
		('1120', 'Bundesstraße', ''),
		('1130', 'Landes- oder Staatsstraße', ''),
		('1140', 'Kreisstraße', ''),
		('1150', 'Gemeindestraße', ''),
		('1160', 'Ortsstraße', ''),
		('1170', 'Gemeindeverbindungsstraße', ''),
		('1180', 'Sonstige öffentliche Straße', ''),
		('1190', 'Privatstraße', '');
		CREATE TABLE ax_artderfestlegung_denkmalschutzrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_denkmalschutzrecht IS 'Alias: "AX_ArtDerFestlegung_Denkmalschutzrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_denkmalschutzrecht (wert,beschreibung,dokumentation) VALUES
		('2700', 'Kulturdenkmal', ''),
		('2710', 'Bau- und Kunstdenkmal nach Landesdenkmalschutzgesetz', ''),
		('2711', 'Baudenkmal', 'Baudenkmal ist eine bauliche Anlage, an deren Erhaltung wegen ihrer geschichtlichen Bedeutung ein öffentliches Interesse besteht.'),
		('2712', 'Kunstdenkmal', ''),
		('2713', 'Gartendenkmal', ''),
		('2800', 'Archäologisches Denkmal (auch Bodendenkmal) nach Landesdenkmalschutzgesetz', ''),
		('2810', 'Archäologisches Denkmal', 'Archäologisches Denkmal ist ein Ort, an dem Reste vor-  oder frühgeschichtlicher Kulturen oder früheren menschlichen Wirkens entdeckt wurden.'),
		('2820', 'Bodendenkmal', 'Bodendenkmal ist eine mit dem Boden verbundene oder im Boden verborgene Sache, die von Menschen geschaffen oder bearbeitet wurde oder Aufschluss über menschliches Leben in vergangenen Zeiten gibt und deshalb erhaltenswert ist.'),
		('2900', 'Schutzgebiet oder -bereiche nach Landesdenkmalschutzgesetz', ''),
		('2910', 'Denkmalzone oder -bereich', ''),
		('2920', 'Geschützter Baubereich', ''),
		('2930', 'Grabungsschutzgebiet', ''),
		('3100', 'Befestigungen', ''),
		('3110', 'Befestigung (Burg)', 'Befestigung (Burg) ist eine Anlage zum Schutz der Menschen vor äußerer Bedrohung.'),
		('3111', 'Burg (Fliehburg, Ringwall)', 'Burg (Fliehburg, Ringwall) ist eine ur- oder frühgeschichtliche runde, ovale oder an Gegebenheiten des Geländes (Böschungskanten) angepasste Befestigungsanlage, die aus einem Erdwall mit oder ohne Holzeinbauten besteht.'),
		('3112', 'Erdwerk', ''),
		('3113', 'Ringwall', ''),
		('3114', 'Steinwerk', ''),
		('3115', 'Festung', ''),
		('3116', 'Gräftenanlage', ''),
		('3117', 'Schanze', 'Schanze ist eine mittelalterliche oder neuzeitliche, in der Regel geschlossene, quadratische, rechteckige oder sternförmige Wallanlage mit Außengraben.'),
		('3118', 'Lager', 'Lager ist die Bezeichnung für ein befestigtes Truppenlager in der Römer- oder in der Neuzeit (z.B. bei Belagerungen im 30 jährigen Krieg).'),
		('3120', 'Wachturm (römisch), Warte', 'Wachtturm (römisch), Warte ist ein allein oder in Verbindung mit einem Befestigungssystem (Limes) stehender Beobachtungsturm.'),
		('3121', 'Wachturm', ''),
		('3122', 'Warte', ''),
		('3130', 'Befestigung (Wall, Graben)', 'Befestigung (Wall, Graben) ist ein aus Erde aufgeschütteter Grenz-, Schutz- oder Stadtwall. Zu der Befestigung (Wall) zählen auch Limes und Landwehr.'),
		('3131', 'Grenzwall, Schutzwall', 'Grenzwall, Schutzwall ist ein aus Erde aufgeschütteter Wall.'),
		('3132', 'Limes', ''),
		('3133', 'Landwehr', 'Landwehr ist eine mittelalterliche Verteidigungslinie aus ein- bis mehrfachem Wallgrabensystem.'),
		('3134', 'Stadtwall', ''),
		('3135', 'Historischer Wall', ''),
		('3200', 'Historische Siedlung', 'Historische Siedlung ist eine Stelle, an der sich eine ur- oder frühgeschichtliche Siedlung befunden hat.'),
		('3210', 'Pfahlbau', 'Pfahlbau ist ein auf eingerammte Pfähle gestelltes, frei über dem Untergrund (Wasser oder Land) stehendes Gebäude.'),
		('3220', 'Wüstung', ''),
		('3230', 'Wurt', ''),
		('3240', 'Abri', ''),
		('3250', 'Höhle', ''),
		('3300', 'Historische Bestattung', ''),
		('3310', 'Großsteingrab (Dolmen, Hünenbett)', 'Großsteingrab (Dolmen, Hünenbett) ist ein Grab mit Steineinbau, d.h. es ist ein aus großen Steinen (z.B. Findlingen) errichteter Grabbau.'),
		('3320', 'Grabhügel (Hügelgrab)', 'Grabhügel (Hügelgrab) ist ein meist runder oder ovaler Hügel, der über einer ur- oder frühgeschichtlichen Bestattung aus Erde aufgeschüttet oder aus Plaggen aufgeschichtet wurde.'),
		('3330', 'Grabhügelfeld', 'Grabhügelfeld ist eine abgegrenzte Fläche, auf der sich mehrere Grabhügel befinden.'),
		('3340', 'Urnenfriedhof', ''),
		('3350', 'Körpergräberfeld', ''),
		('3360', 'Reihengräberfriedhof', ''),
		('3400', 'Historisches land- oder forstwirtschaftliches Objekt', ''),
		('3410', 'Historischer Pflanzkamp', ''),
		('3420', 'Historisches Viehgehege', ''),
		('3430', 'Sandfang', ''),
		('3440', 'Historisches Ackersystem', ''),
		('3500', 'Historische Bergbau-, Verhüttungs- oder sonstige Produktionsstätte', ''),
		('3510', 'Historisches Bergbaurelikt', ''),
		('3520', 'Historischer Meiler', ''),
		('3530', 'Historischer Ofen', ''),
		('3540', 'Historischer Verhüttungsplatz', ''),
		('3600', 'Historische Straße oder Weg', ''),
		('3610', 'Heerstraße', ''),
		('3620', 'Hohlweg', ''),
		('3630', 'Moorweg', ''),
		('3640', 'Wegespur', ''),
		('3700', 'Historisches wasserwirtschaftliches Objekt', ''),
		('3710', 'Historische Wasserleitung', 'Historische Wasserleitung ist ein meist offenes System von Gräben, Kunstgräben und Kanälen, in dem Wasser transportiert wird.'),
		('3720', 'Aquädukt', 'Aquädukt ist ein brückenartiges Steinbauwerk zur Überführung von Freispiegel-Wasserleitungen mit natürlichem Gefälle über Täler oder andere Bodenunebenheiten.'),
		('3730', 'Historischer Deich', ''),
		('3740', 'Historischer Damm', ''),
		('3750', 'Historischer Graben', ''),
		('3800', 'Steinmal', 'Steinmal ist eine kultische oder rechtliche Kennzeichnung, bestehend aus einzelnen oder Gruppen von Steinen.'),
		('3810', 'Schalenstein', ''),
		('3820', 'Rillenstein', ''),
		('3830', 'Wetzrillen', ''),
		('3840', 'Kreuzstein', ''),
		('3850', 'Historischer Grenzstein', ''),
		('3860', 'Menhir', '');
		CREATE TABLE ax_artderfestlegung_klassifizierungnachwasserrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_klassifizierungnachwasserrecht IS 'Alias: "AX_ArtDerFestlegung_KlassifizierungNachWasserrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_klassifizierungnachwasserrecht (wert,beschreibung,dokumentation) VALUES
		('1300', 'Klassifizierung nach Bundes- oder Landeswassergesetz', ''),
		('1310', 'Gewässer I. Ordnung - Bundeswasserstraße', ''),
		('1320', 'Gewässer I. Ordnung - nach Landesrecht', ''),
		('1330', 'Gewässer II. Ordnung', ''),
		('1340', 'Gewässer III. Ordnung', '');
		CREATE TABLE ax_rechtszustand_schutzzone (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_rechtszustand_schutzzone IS 'Alias: "AX_Rechtszustand_Schutzzone", UML-Typ: Enumeration';
		INSERT INTO ax_rechtszustand_schutzzone (wert,beschreibung,dokumentation) VALUES
		('1000', 'Amtlich festgestellt', 'Amtlich festgestellt bedeutet, dass der Zustand von Schutzzone durch einen Verwaltungsstelle festgelegt wird.'),
		('2000', 'Im Verfahren befindlich', 'Im Verfahren befindlich bedeutet, dass das Genehmigungsverfahren für die Festlegung eines Wasserschutz-, Heilquellenschutzgebietes begonnen aber noch nicht abgeschlossen ist.'),
		('3000', 'In Planung', 'In Planung bedeutet, dass ein Wasserschutz-, Heilquellenschutzgebiet geplant ist, aber das Genehmigungsverfahren noch nicht begonnen hat.');
		CREATE TABLE ax_artderfestlegung_bauraumoderbodenordnungsrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_bauraumoderbodenordnungsrecht IS 'Alias: "AX_ArtDerFestlegung_BauRaumOderBodenordnungsrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_bauraumoderbodenordnungsrecht (wert,beschreibung,dokumentation) VALUES
		('1700', 'Festlegung nach Baugesetzbuch - Allgemeines Städtebaurecht', ''),
		('1710', 'Bebauungsplan', ''),
		('1720', 'Veränderungssperre nach Baugesetzbuch', ''),
		('1730', 'Vorkaufrechtssatzung', ''),
		('1740', 'Enteignungsverfahren', ''),
		('1750', 'Umlegung nach dem BauGB', ''),
		('1760', 'Bauland', ''),
		('1770', 'Vereinfachte Umlegung', ''),
		('1780', 'Vorhaben- und Erschließungsplan', ''),
		('1790', 'Flächennutzungsplan', ''),
		('1800', 'Festlegung nach Baugesetzbuch - Besonderes Städtebaurecht', ''),
		('1810', 'Städtebauliche Entwicklungsmaßnahme', ''),
		('1811', 'Städtebauliche Entwicklungsmaßnahme (Beschluss zu vorbereitenden Untersuchungen gefasst)', ''),
		('1820', 'Erhaltungssatzung', ''),
		('1821', 'Städtebauliches Erhaltungsgebiet', ''),
		('1822', 'Soziales Erhaltungsgebiet', ''),
		('1823', 'Erhaltungsgebiet zur städtebaulichen Umstrukturierung', ''),
		('1824', 'Soziales Erhaltungsgebiet (Aufstellungsbeschluss gefasst)', ''),
		('1830', 'Städtebauliche Gebote', ''),
		('1840', 'Sanierung', ''),
		('1841', 'Sanierung (Beschluss zu vorbereitenden Untersuchungen gefasst)', ''),
		('1900', 'Wohnungsbauerleichterungsgesetz', ''),
		('2100', 'Flurbereinigungsgesetz', ''),
		('2110', 'Flurbereinigung (Par. 1, 37 FlurbG)', ''),
		('2120', 'Vereinfachtes Flurbereinigungsverfahren (Par. 86 FlurbG)', ''),
		('2130', 'Unternehmensflurbereinigung (nach Par. 87 oder 90 FlurbG)', ''),
		('2140', 'Beschleunigtes Zusammenlegungsverfahren (Par. 91 FlurbG)', ''),
		('2150', 'Freiwilliger Landtausch (Par. 103a FlurbG)', ''),
		('2160', 'Verfahren nach dem Gemeinheitsteilungsgesetz', ''),
		('2170', 'Verfahren nach dem Gemeinschaftswaldgesetz', ''),
		('2180', 'Freiwilliger Nutzungstausch', ''),
		('2200', 'Verfahren nach dem Landwirtschaftsanpassungsgesetz', ''),
		('2210', 'Flurneuordnung', ''),
		('2220', 'Freiwilliger Landtausch (Par. 54 LwAnpG)', ''),
		('2230', 'Bodenordnungsverfahren (Par. 56 LwAnpG)', ''),
		('2240', 'Zusammenführung von Boden- und Gebäudeeigentum (Par. 64 LwAnpG)', ''),
		('2300', 'Bodensonderungsgesetz', ''),
		('2310', 'Unvermessenes Eigentum', ''),
		('2320', 'Unvermessenes Nutzungsrecht', ''),
		('2330', 'Ergänzende Bodenneuordnung', ''),
		('2340', 'Komplexe Bodenneuordnung', ''),
		('2400', 'Vermögenszuordnungsgesetz', ''),
		('2410', 'Vermögenszuordnung nach Plan', ''),
		('2411', 'Vermögenszuordnung nach dem Aufteilungsplan', ''),
		('2412', 'Vermögenszuordnung nach dem Zuordnungsplan', ''),
		('2500', 'Landesraumordnungsgesetz', ''),
		('2510', 'Wasservorranggebiete', ''),
		('2600', 'Bauordnung', ''),
		('2610', 'Baulast', ''),
		('2611', 'Begünstigende Baulast', ''),
		('2612', 'Belastende Baulast', ''),
		('2700', 'Grenzfeststellungsverfahren nach Hamb. Wassergesetz', ''),
		('2800', 'Verkehrsflächenbereinigung', '');
		CREATE TABLE ax_artderfestlegung_anderefestlegungnachwasserrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_anderefestlegungnachwasserrecht IS 'Alias: "AX_ArtDerFestlegung_AndereFestlegungNachWasserrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_anderefestlegungnachwasserrecht (wert,beschreibung,dokumentation) VALUES
		('1410', 'Wasserstraßengesetz', ''),
		('1420', 'Veränderungssperre', ''),
		('1421', 'Veränderungssperre nach Bundeswasserstraßengesetz', ''),
		('1430', 'Landeswassergesetz', ''),
		('1440', 'Überschwemmungsgebiet', 'Überschwemmungsgebiet ist ein durch Rechtsverordnung festgesetztes oder natürliches Gebiet, das bei Hochwasser überschwemmt werden kann bzw. überschwemmt wird.'),
		('1441', 'Festgesetztes Überschwemmungsgebiet', 'Festgesetztes Überschwemmungsgebiet ist ein zur Sicherung des schadlosen Hochwasserabflusses durch Rechtsverordnung festgesetztes Gebiet.'),
		('1442', 'Natürliches Überschwemmungsgebiet', 'Natürliches Überschwemmungsgebiet ist ein Gebiet, das bei Hochwasser überschwemmt wird.'),
		('1450', 'Pegelschutzgebiet', ''),
		('1460', 'Wasser- und Bodenverbandsgebiet', ''),
		('1470', 'Deichgebiet', ''),
		('1480', 'Unterhaltungsverbandsgebiet', '');
		CREATE TABLE ax_artderfestlegung_forstrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_forstrecht IS 'Alias: "AX_ArtDerFestlegung_Forstrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_forstrecht (wert,beschreibung,dokumentation) VALUES
		('3900', 'Klassifizierung nach Bundes- oder Landeswaldgesetz', ''),
		('3910', 'Staatswald Bund', ''),
		('3920', 'Staatswald Land', ''),
		('3930', 'Kommunalwald', ''),
		('3940', 'Anstalts- und Stiftungswald', ''),
		('3950', 'Anderer öffentlicher Wald', ''),
		('3960', 'Privater Gemeinschaftswald', ''),
		('3970', 'Großprivatwald', ''),
		('3980', 'Kleinprivatwald', ''),
		('3990', 'Anderer Privatwald', '');
		CREATE TABLE ax_zustand_naturumweltoderbodenschutzrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_naturumweltoderbodenschutzrecht IS 'Alias: "AX_Zustand_NaturUmweltOderBodenschutzrecht", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_naturumweltoderbodenschutzrecht (wert,beschreibung,dokumentation) VALUES
		('1000', 'Amtlich festgestellt', 'Amtlich festgestellt bedeutet, dass der Zustand für eine dem Natur-, Umwelt- oder Bodenschutzrecht unterliegende Fläche durch eine Verwaltungsstelle festgelegt wird.'),
		('2000', 'Einstweilig sicher gestellt', 'Einstweilig sicher gestellt bedeutet, dass durch die zuständige Fachbehörde eine dem Natur-, Umwelt- oder Bodenschutzrecht unterliegende Fläche eine Veränderungssperre erlassen wurde.');
		CREATE TABLE ax_artderfestlegung_sonstigesrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_sonstigesrecht IS 'Alias: "AX_ArtDerFestlegung_SonstigesRecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_sonstigesrecht (wert,beschreibung,dokumentation) VALUES
		('4100', 'Luftverkehrsgesetz', ''),
		('4110', 'Bauschutzbereich', ''),
		('4120', 'Beschränkter Bauschutzbereich', ''),
		('4200', 'Bundeskleingartengesetz', ''),
		('4210', 'Dauerkleingarten', ''),
		('4300', 'Berggesetz', ''),
		('4301', 'Bodenbewegungsgebiet', 'Bodenbewegungsgebiet ist ein Gebiet, in dem sich die oberen Erdschichten auf Grund verschiedener Einflüsse (z.B. geologische Kräfte, Bergbau) lage- oder höhenmäßig verändern.'),
		('4302', 'Bruchfeld', 'Bruchfeld ist ein durch Bergbau unterhöhltes Gebiet, das teilweise bereits eingebrochen ist oder sich in Absenkung befindet.'),
		('4310', 'Baubeschränkung', ''),
		('4400', 'Reichsheimstättengesetz', ''),
		('4410', 'Reichsheimstätte', ''),
		('4500', 'Schutzbereichsgesetz', ''),
		('4510', 'Schutzbereich', ''),
		('4600', 'Eisenbahnneuordnungsgesetz', ''),
		('4610', 'Übergabebescheidverfahren', ''),
		('4710', 'Baubeschränkungen durch Richtfunkverbindungen', ''),
		('4720', 'Truppenübungsplatz, Standortübungsplatz', 'Truppenübungsplatz, Standortübungsplatz ist ein Gelände zur militärischen Ausbildung.'),
		('4800', 'Vermessungs- und Katasterrecht', ''),
		('4810', 'Schutzfläche Festpunkt', ''),
		('4811', 'Schutzfläche Festpunkt, 1 m Radius', ''),
		('4812', 'Schutzfläche Festpunkt, 2 m Radius', ''),
		('4813', 'Schutzfläche Festpunkt, 5 m Radius', ''),
		('4814', 'Schutzfläche Festpunkt, 10 m Radius', ''),
		('4815', 'Schutzfläche Festpunkt, 30 m Radius', ''),
		('4820', 'Marksteinschutzfläche', ''),
		('4830', 'Liegenschaftskatastererneuerung', ''),
		('4900', 'Fischereirecht', ''),
		('5100', 'Jagdkataster', ''),
		('5200', 'Landesgrundbesitzkataster', ''),
		('5300', 'Bombenblindgängerverdacht', ''),
		('5400', 'Rieselfeld', 'Rieselfeld ist eine Fläche, auf der organisch verunreinigtes Wasser zum Zwecke der biologischen Reinigung verrieselt wird.'),
		('5500', 'Sicherungsstreifen', ''),
		('5600', 'Grenzbereinigung', ''),
		('5700', 'Hochwasserdeich', 'Hochwasserdeich ist die Eigenschaft (Widmung) eines Deiches, die durch die obere Deichbehörde festgelegt wird.'),
		('5710', 'Hauptdeich, 1. Deichlinie', 'Hauptdeich, 1. Deichlinie ist die Eigenschaft (Widmung) eines Deiches, die durch die obere Deichbehörde festgelegt wird.'),
		('5720', '2. Deichlinie', '2. Deichlinie ist die Eigenschaft (Widmung) eines Deiches, die durch die obere Deichbehörde festgelegt wird.'),
		('6000', 'Beregnungsverband', ''),
		('7000', 'Weinlage', ''),
		('7100', 'Weinbausteillage', ''),
		('7200', 'Weinbergsrolle', ''),
		('7300', 'Weinbausteilstlage', ''),
		('8000', 'Benachteiligtes landwirtschaftliches Gebiet', ''),
		('9100', 'Mitverwendung Hochwasserschutz, Oberirdische Anlagen', ''),
		('9200', 'Mitverwendung Hochwasserschutz, Unterirdische Anlagen', ''),
		('9300', 'Hafennutzungsgebiet', ''),
		('9400', 'Hafenerweiterungsgebiet', ''),
		('9450', 'Hafenbecken', 'Hafenbecken ist ein rechtlich definierter Teil eines Gewässers, in dem Schiffe be- und entladen werden'),
		('9500', 'Bohrung verfüllt', ''),
		('9600', 'Zollgrenze', ''),
		('9700', 'Belastung nach §7 Abs. 2 GBO', ''),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass Art der Festlegung bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_artderfestlegung_naturumweltoderbodenschutzrecht (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderfestlegung_naturumweltoderbodenschutzrecht IS 'Alias: "AX_ArtDerFestlegung_NaturUmweltOderBodenschutzrecht", UML-Typ: Enumeration';
		INSERT INTO ax_artderfestlegung_naturumweltoderbodenschutzrecht (wert,beschreibung,dokumentation) VALUES
		('1610', 'Schutzfläche nach Europarecht', ''),
		('1611', 'Flora-Fauna-Habitat-Gebiet', 'Flora-Fauna-Habitat-Gebiet ist ein Schutzgebiet von gemeinschaftlicher Bedeutung im Sinne der Richtlinie 92/43/EWG des Rates der Europäischen Wirtschaftsgemeinschaft zur Erhaltung der natürlichen Lebensräume sowie der wildlebenden Pflanzen und Tiere.'),
		('1612', 'Vogelschutzgebiet', 'Vogelschutzgebiet ist ein besonderes Schutzgebiet (Special Protected Area, SPA) im Sinne Artikel 4 Abs. 1 der Richtlinie 79/409/EWG des Rates der Europäischen Wirtschaftsgemeinschaft über die Erhaltung der wildlebenden Vogelarten (Vogelschutzrichtlinie).'),
		('1620', 'Schutzflächen nach Landesnaturschutzgesetz', ''),
		('1621', 'Naturschutzgebiet', 'Naturschutzgebiet ist ein rechtsverbindlich festgesetztes Gebiet, in dem ein besonderer Schutz von Natur und Landschaft in ihrer Ganzheit oder in einzelnen Teilen zur Erhaltung von Lebensgemeinschaften oder Biotopen bestimmter wildlebender Tier- und Pflanzenarten, aus wissenschaftlichen, naturgeschichtlichen oder landeskundlichen Gründen oder wegen ihrer Seltenheit, besonderen Eigenart oder hervorragenden Schönheit erforderlich ist.'),
		('1622', 'Geschützter Landschaftsbestandteil', 'Geschützter Landschaftsbestandteil ist ein rechtsverbindlich festgesetzter Teil von Natur und Landschaft, dessen besonderer Schutz zur Sicherstellung der Leistungsfähigkeit des Naturhaushalts, zur Belebung, Gliederung oder Pflege des Orts- und Landschaftsbildes oder zur Abwehr schädlicher Einwirkungen erforderlich ist.'),
		('1623', 'Landschaftsschutzgebiet', 'Landschaftsschutzgebiet ist ein rechtsverbindlich festgesetztes Gebiet, in dem ein besonderer Schutz von Natur und Landschaft zur Erhaltung oder Wiederherstellung der Leistungsfähigkeit des Naturhaushalts oder der Nutzungsfähigkeit der Naturgüter, wegen der Vielfalt, Eigenart oder Schönheit des Landschaftsbildes oder wegen der besonderen Bedeutung für die Erholung erforderlich ist.'),
		('1624', 'Naturpark', 'Naturpark ist ein einheitlich zu entwickelndes und zu pflegendes Gebiet, das großräumig ist, überwiegend Landschaftsschutzgebiet oder Naturschutzgebiet ist, sich wegen seiner landschaftlichen Voraussetzungen für die Erholung besonders eignet und nach den Grundsätzen und Zielen der Raumordnung und der Landesplanung für die Erholung oder den Fremdenverkehr vorgesehen ist.'),
		('1630', 'Bundesbodenschutzgesetz', ''),
		('1631', 'Verdachtsfläche auf schädliche Bodenveränderung', ''),
		('1632', 'Schädliche Bodenveränderung', ''),
		('1633', 'Altlastenverdächtige Fläche', ''),
		('1634', 'Altlast', ''),
		('1640', 'Bundesimmisionsschutzgesetz', ''),
		('1641', 'Belastungsgebiet', ''),
		('1642', 'Schutzbedürftiges Gebiet', ''),
		('1643', 'Gefährdetes Gebiet', ''),
		('1650', 'Naturschutzgesetz', ''),
		('1651', 'Besonders geschütztes Biotop', ''),
		('1652', 'Besonders geschütztes Feuchtgrünland', ''),
		('1653', 'Naturdenkmal', 'Naturdenkmal ist eine rechtsverbindlich festgesetzte Einzelschöpfung der Natur, deren besonderer Schutz erforderlich ist (z.B. Baum).'),
		('1654', 'Einstweilige Sicherstellung, Veränderungssperre', ''),
		('1655', 'Vorkaufsrecht', ''),
		('1656', 'Ausgleichs- oder Kompensationsfläche', ''),
		('1660', 'Bodenschutzgesetz', ''),
		('1661', 'Dauerbeobachtungsflächen', ''),
		('1662', 'Bodenschutzgebiet', '');
		CREATE TABLE ax_liniendarstellung_topographischelinie (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_liniendarstellung_topographischelinie IS 'Alias: "AX_Liniendarstellung_TopographischeLinie", UML-Typ: Enumeration';
		INSERT INTO ax_liniendarstellung_topographischelinie (wert,beschreibung,dokumentation) VALUES
		('1000', 'Durchgezogen', ''),
		('2000', 'Lang gestrichelt', ''),
		('3000', 'Kurz gestrichelt', '');
		CREATE TABLE ax_darstellung_gebaeudeausgestaltung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_darstellung_gebaeudeausgestaltung IS 'Alias: "AX_Darstellung_Gebaeudeausgestaltung", UML-Typ: Enumeration';
		INSERT INTO ax_darstellung_gebaeudeausgestaltung (wert,beschreibung,dokumentation) VALUES
		('0251', 'Topographische Begrenzungslinie', ''),
		('0252', 'Überdeckte topographische Begrenzungslinie', ''),
		('1011', 'Hochhausbegrenzungslinie', ''),
		('1012', 'Begrenzungslinie eines öffentlichen Gebäudes', ''),
		('1013', 'Begrenzungslinie eines nicht öffentlichen Gebäudes', ''),
		('1014', 'Offene Begrenzungslinie eines Gebäudes', ''),
		('1016', 'Begrenzungslinie eines unterirdischen Gebäudes', ''),
		('1041', 'Durchfahrt im Gebäude', ''),
		('1042', 'Durchfahrt an überbauter Straße', ''),
		('1043', 'Arkade', ''),
		('1044', 'Freitreppe', ''),
		('1045', 'Überdachung', ''),
		('1046', 'Aufgeständertes Gebäude', ''),
		('1047', 'Begrenzungslinie auskragendes Geschoss, aufgehendes Mauerwerk', ''),
		('1048', 'Rampe', '');
		CREATE TABLE ax_datenformat_benutzer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_datenformat_benutzer IS 'Alias: "AX_Datenformat_Benutzer", UML-Typ: Enumeration';
		INSERT INTO ax_datenformat_benutzer (wert,beschreibung,dokumentation) VALUES
		('1000', 'NAS', ''),
		('2000', 'DXF', ''),
		('3000', 'TIFF', ''),
		('4000', 'GeoTIFF', '');
		CREATE TABLE ax_art_bereichzeitlich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_bereichzeitlich IS 'Alias: "AX_Art_BereichZeitlich", UML-Typ: Enumeration';
		INSERT INTO ax_art_bereichzeitlich (wert,beschreibung,dokumentation) VALUES
		('1000', 'Stichtagsbezogen ohne Historie', ''),
		('1100', 'Stichtagsbezogen mit Historie', ''),
		('3000', 'Fallbezogen ohne Historie', ''),
		('3100', 'Fallbezogen mit Historie', '');
		CREATE TABLE ax_letzteabgabeart (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_letzteabgabeart IS 'Alias: "AX_LetzteAbgabeArt", UML-Typ: Enumeration';
		INSERT INTO ax_letzteabgabeart (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zeitintervall', 'Standardabgabe: LAZ + Abgabeintervall'),
		('1100', 'NBA auf Abruf', 'Zwischenabgabe: LAZ bis heute.'),
		('2000', 'Wiederholungslauf', 'Identische Wiederholung des fehlerhaften Laufs: VAZ bis LAZ.'),
		('2100', 'Aufholungslauf auf Abruf', 'Wiederholungslauf zuzüglich der Änderungsdaten bis heute: VAZ bis heute.'),
		('2200', 'Aufholungslauf bis Intervallende', 'Zusammenfassung mehrerer Abgabeintervalle: VAZ bis Intervallende nach heute.');
		CREATE TABLE ax_ausgabemedium_benutzer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_ausgabemedium_benutzer IS 'Alias: "AX_Ausgabemedium_Benutzer", UML-Typ: Enumeration';
		INSERT INTO ax_ausgabemedium_benutzer (wert,beschreibung,dokumentation) VALUES
		('1000', 'Analog', ''),
		('2000', 'CD-R', ''),
		('3000', 'DVD', ''),
		('4000', 'E-Mail', '');
		CREATE TABLE ax_identifikation (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_identifikation IS 'Alias: "AX_Identifikation", UML-Typ: Enumeration';
		INSERT INTO ax_identifikation (wert,beschreibung,dokumentation) VALUES
		('5400', 'Sicher', ''),
		('5410', 'Unsicher', '');
		CREATE TABLE ax_dqerfassungsmethodemarkantergelaendepunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dqerfassungsmethodemarkantergelaendepunkt IS 'Alias: "AX_DQErfassungsmethodeMarkanterGelaendepunkt", UML-Typ: Enumeration';
		INSERT INTO ax_dqerfassungsmethodemarkantergelaendepunkt (wert,beschreibung,dokumentation) VALUES
		('5000', 'Terrestrische Aufnahme', ''),
		('5010', 'Interaktive photogrammetrische Datenerfassung', ''),
		('5020', 'Laserscanning', ''),
		('5030', 'Digitalisierung analoger Vorlagen', '');
		CREATE TABLE ax_dqerfassungsmethodestrukturiertegelaendepunkte (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dqerfassungsmethodestrukturiertegelaendepunkte IS 'Alias: "AX_DQErfassungsmethodeStrukturierteGelaendepunkte", UML-Typ: Enumeration';
		INSERT INTO ax_dqerfassungsmethodestrukturiertegelaendepunkte (wert,beschreibung,dokumentation) VALUES
		('5000', 'Terrestrische Aufnahme', ''),
		('5010', 'Interaktive photogrammetrische Datenerfassung', ''),
		('5030', 'Digitalisierung analoger Vorlagen', '');
		CREATE TABLE ax_dqerfassungsmethode (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dqerfassungsmethode IS 'Alias: "AX_DQErfassungsmethode", UML-Typ: Enumeration';
		INSERT INTO ax_dqerfassungsmethode (wert,beschreibung,dokumentation) VALUES
		('5000', 'Terrestrische Aufnahme', ''),
		('5010', 'Interaktive photogrammetrische Datenerfassung', ''),
		('5020', 'Laserscanning', ''),
		('5030', 'Digitalisierung analoger Vorlagen', ''),
		('5040', 'Bildkorrelation', ''),
		('5050', 'IfSAR', '');
		CREATE TABLE ax_besonderebedeutung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_besonderebedeutung IS 'Alias: "AX_BesondereBedeutung", UML-Typ: Enumeration';
		INSERT INTO ax_besonderebedeutung (wert,beschreibung,dokumentation) VALUES
		('1600', 'Höhenpunkt auf Wasserfläche', ''),
		('1610', 'Wegepunkt', '');
		CREATE TABLE ax_dqerfassungsmethodebesondererhoehenpunkt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dqerfassungsmethodebesondererhoehenpunkt IS 'Alias: "AX_DQErfassungsmethodeBesondererHoehenpunkt", UML-Typ: Enumeration';
		INSERT INTO ax_dqerfassungsmethodebesondererhoehenpunkt (wert,beschreibung,dokumentation) VALUES
		('5000', 'Terrestrische Aufnahme', ''),
		('5010', 'Interaktive photogrammetrische Aufnahme', ''),
		('5020', 'Laserscanning', ''),
		('5030', 'Digitalisierung analoger Vorlagen', ''),
		('5060', 'Amtliche Festlegung', '');
		CREATE TABLE ax_artdergeripplinie (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdergeripplinie IS 'Alias: "AX_ArtDerGeripplinie", UML-Typ: Enumeration';
		INSERT INTO ax_artdergeripplinie (wert,beschreibung,dokumentation) VALUES
		('1300', 'Muldenlinie', ''),
		('1310', 'Wasserführende Muldenlinie', ''),
		('1320', 'Rückenlinie', '');
		CREATE TABLE ax_artdergelaendekante (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdergelaendekante IS 'Alias: "AX_ArtDerGelaendekante", UML-Typ: Enumeration';
		INSERT INTO ax_artdergelaendekante (wert,beschreibung,dokumentation) VALUES
		('1200', 'Allgemeine Geländekante', 'Allgemeine Geländekante ist die einzelne Kante unterschiedlich geneigter Geländeflächen und keine Obergruppe anderer Geländekanten.'),
		('1210', 'Steilrand, Kliffkante', 'Steilrand, Kliffkante begrenzt den von der Brandung beständig abgetragenen Steilhang einer Küste.'),
		('1220', 'Böschungsoberkante', 'Böschungsoberkante ist der eindeutig identifizierbare Geländeknick an der oberen Kante einer Böschung.'),
		('1230', 'Böschungsunterkante', 'Böschungsunterkante ist der eindeutig identifizierbare Geländeknick an der unteren Kante einer Böschung.'),
		('1240', 'Sonstige Begrenzungskante', 'Sonstige Begrenzungskante sind alle Kanten, die nicht anderen Kanten zugeordnet werden können (z.B. Trennschraffe).');
		CREATE TABLE ax_artderstrukturierung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderstrukturierung IS 'Alias: "AX_ArtDerStrukturierung", UML-Typ: Enumeration';
		INSERT INTO ax_artderstrukturierung (wert,beschreibung,dokumentation) VALUES
		('1700', 'Höhenlinien', ''),
		('1710', 'Dynamisch gemessene Profile', '');
		CREATE TABLE ax_dqerfassungsmethodegewaesserbegrenzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dqerfassungsmethodegewaesserbegrenzung IS 'Alias: "AX_DQErfassungsmethodeGewaesserbegrenzung", UML-Typ: Enumeration';
		INSERT INTO ax_dqerfassungsmethodegewaesserbegrenzung (wert,beschreibung,dokumentation) VALUES
		('5000', 'Terrestrische Aufnahme', ''),
		('5010', 'Interaktive photogrammetrische Datenerfassung', ''),
		('5020', 'Laserscanning', ''),
		('5030', 'Digitalisierung analoger Vorlagen', ''),
		('5040', 'Bildkorrelation', ''),
		('5050', 'IfSAR', ''),
		('5060', 'Amtliche Festlegung', '');
		CREATE TABLE ax_artdernichtgelaendepunkte (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdernichtgelaendepunkte IS 'Alias: "AX_ArtDerNichtGelaendepunkte", UML-Typ: Enumeration';
		INSERT INTO ax_artdernichtgelaendepunkte (wert,beschreibung,dokumentation) VALUES
		('1000', 'Brückenpunkte', ''),
		('1010', 'Gebäudepunkte', ''),
		('1020', 'Vegetationspunkte', '');
		CREATE TABLE ax_artdesmarkantengelaendepunktes (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artdesmarkantengelaendepunktes IS 'Alias: "AX_ArtDesMarkantenGelaendepunktes", UML-Typ: Enumeration';
		INSERT INTO ax_artdesmarkantengelaendepunktes (wert,beschreibung,dokumentation) VALUES
		('1400', 'Kuppenpunkt', ''),
		('1410', 'Kesselpunkt', ''),
		('1420', 'Sattelpunkt', '');
		CREATE TABLE ax_artderaussparung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderaussparung IS 'Alias: "AX_ArtDerAussparung", UML-Typ: Enumeration';
		INSERT INTO ax_artderaussparung (wert,beschreibung,dokumentation) VALUES
		('1500', 'DGM-Aussparung', ''),
		('1510', 'Kartographische Aussparung', '');
		CREATE TABLE ax_besondereartdergewaesserbegrenzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_besondereartdergewaesserbegrenzung IS 'Alias: "AX_BesondereArtDerGewaesserbegrenzung", UML-Typ: Enumeration';
		INSERT INTO ax_besondereartdergewaesserbegrenzung (wert,beschreibung,dokumentation) VALUES
		('1350', 'Gleichzeitig Böschungsunterkante', '');
		CREATE TABLE ax_ursprung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_ursprung IS 'Alias: "AX_Ursprung", UML-Typ: Enumeration';
		INSERT INTO ax_ursprung (wert,beschreibung,dokumentation) VALUES
		('1100', 'Natürlich entstanden', ''),
		('1110', 'Künstlich entstanden', '');
		CREATE TABLE ax_funktion_dammwalldeich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_dammwalldeich IS 'Alias: "AX_Funktion_DammWallDeich", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_dammwalldeich (wert,beschreibung,dokumentation) VALUES
		('3001', 'Hochwasserschutz, Sturmflutschutz', 'Hochwasserschutz, Sturmflutschutz bedeutet, dass Damm, Wall, Deich dem Schutz vor Hochwasser bzw. Sturmflut dient.'),
		('3002', 'Verkehrsführung', 'Verkehrsführung bedeutet, dass auf Damm, Wall, Deich ein Verkehrsweg verläuft.'),
		('3003', 'Hochwasserschutz, Sturmflutschutz zugleich Verkehrsführung', 'Hochwasserschutz, Sturmflutschutz zugleich Verkehrsführung bedeutet, dass Damm, Wall, Deich dem Schutz vor Hochwasser dient und auf dem gleichzeitig ein Verkehrsweg verläuft.'),
		('3004', 'Lärmschutz', 'Lärmschutz bedeutet, dass Damm, Wall, Deich dem Schutz vor Lärmemissionen dient.');
		CREATE TABLE ax_art_dammwalldeich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_dammwalldeich IS 'Alias: "AX_Art_DammWallDeich", UML-Typ: Enumeration';
		INSERT INTO ax_art_dammwalldeich (wert,beschreibung,dokumentation) VALUES
		('1910', 'Hochwasserdeich', 'Hochwasserdeich ist ein Deich an einem Fliessgewässer oder im Küstengebiet, der dem Schutz eines Gebietes vor Hochwasser oder gegen Sturmfluten dient.'),
		('1920', 'Hauptdeich, Landesschutzdeich', 'Hauptdeich, Landesschutzdeich ist ein Deich der ersten Deichlinie zum Schutz der Küsten- und Inselgebiete gegen Sturmflut.'),
		('1930', 'Überlaufdeich', 'Überlaufdeich ist ein Deich vor dem Hauptdeich, der in erster Linie dem Schutz landwirtschaftlich genutzter Flächen gegen leichte Sturmtiden dient und der bei höheren Sturmtiden überströmt wird.'),
		('1940', 'Leitdeich', 'Leitdeich ist ein dammartiges Bauwerk im Watt, um strömendes Wasser in bestimmte Richtungen zu lenken und zum Schutz von Wasserläufen im Watt (Außentiefs) vor Versandung.'),
		('1950', 'Polderdeich', 'Polderdeich ist ein vor dem Hauptdeich liegender Deich, der landwirtschaftlich nutzbares Land (z. B. Marschland) schützt.'),
		('1960', 'Schlafdeich', 'Schlafdeich ist ein ehemaliger Hauptdeich, der infolge einer Vorverlegung der Deichlinie zu einem Binnendeich geworden ist und keine unmittelbare Schutzaufgabe mehr zu erfüllen hat.'),
		('1970', 'Mitteldeich', 'Mitteldeich ist ein Deich der 2. Deichlinie, auch an größeren Flüssen. Er soll Überschwemmungen beim Bruch des Deiches der ersten Deichlinie verhindern.'),
		('1980', 'Binnendeich', 'Binnendeich ist ein Deich an kleineren Flüssen, der Überschwemmungen durch ablaufendes Oberflächenwasser verhindern soll.'),
		('1990', 'Wall', ''),
		('1991', 'Wallkante, rechts', ''),
		('1992', 'Wallkante, links', ''),
		('1993', 'Wallmitte', ''),
		('2000', 'Knick', 'Knick ist ein bewachsener Erdwall, der als Einfriedung und zum Windschutz von Wiesen und Äckern dient.'),
		('2001', 'Knickkante, rechts', ''),
		('2002', 'Knickkante, links', ''),
		('2003', 'Knickmitte', ''),
		('2010', 'Graben mit Wall, rechts', ''),
		('2011', 'Graben mit Wall, links', ''),
		('2012', 'Graben mit Knick, rechts', ''),
		('2013', 'Graben mit Knick, links', '');
		CREATE TABLE ax_funktion_einschnitt (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_einschnitt IS 'Alias: "AX_Funktion_Einschnitt", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_einschnitt (wert,beschreibung,dokumentation) VALUES
		('3002', 'Verkehrsführung', 'Verkehrsführung bedeutet, dass im Einschnitt ein Verkehrsweg verläuft.');
		CREATE TABLE ax_zustand_boeschungkliff (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_boeschungkliff IS 'Alias: "AX_Zustand_BoeschungKliff", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_boeschungkliff (wert,beschreibung,dokumentation) VALUES
		('2400', 'Befestigt', ''),
		('2500', 'Unbefestigt', '');
		CREATE TABLE ax_zustand_hoehleneingang (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_hoehleneingang IS 'Alias: "AX_Zustand_Hoehleneingang", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_hoehleneingang (wert,beschreibung,dokumentation) VALUES
		('2200', 'Verfallen, zerstört', 'Verfallen, zerstört bedeutet, dass sich der ursprüngliche Zustand von Höhleneingang durch menschliche oder zeitliche Einwirkungen so verändert hat, dass eine Nutzung nicht mehr möglich ist.'),
		('4100', 'Offen', 'Offen bedeutet, dass Höhleneingang allgemein zugänglich ist.'),
		('4200', 'Verschlossen', 'Verschlossen bedeutet, dass  Höhleneingang nicht allgemein zugänglich ist.');
		CREATE TABLE ax_berechnungsmethode (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_berechnungsmethode IS 'Alias: "AX_Berechnungsmethode", UML-Typ: Enumeration';
		INSERT INTO ax_berechnungsmethode (wert,beschreibung,dokumentation) VALUES
		('5300', 'Einfache Mittelbildung', ''),
		('5310', 'Gewichtete Mittelbildung', ''),
		('5320', 'Gleitende Schrägebene', ''),
		('5330', 'Prädiktion', ''),
		('5340', 'Polynomasatz', ''),
		('5350', 'Finite Elemente', ''),
		('5360', 'Dreiecksvermaschung', '');
		CREATE TABLE ax_verwendeteobjekte (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_verwendeteobjekte IS 'Alias: "AX_VerwendeteObjekte", UML-Typ: Enumeration';
		INSERT INTO ax_verwendeteobjekte (wert,beschreibung,dokumentation) VALUES
		('5111', 'Geländepunkte', ''),
		('5121', 'Geländekanten', ''),
		('5122', 'Gewässerbegrenzung', ''),
		('5123', 'Geripplinien', ''),
		('5124', 'Markante Geländepunkte', ''),
		('5131', 'Aussparungsflächen', ''),
		('5211', 'DGM-Gitter', '');
		CREATE TABLE ax_berechnungsmethodehoehenlinie (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_berechnungsmethodehoehenlinie IS 'Alias: "AX_BerechnungsmethodeHoehenlinie", UML-Typ: Enumeration';
		INSERT INTO ax_berechnungsmethodehoehenlinie (wert,beschreibung,dokumentation) VALUES
		('5211', 'Aus sekundärem DGM-Gitter', ''),
		('5360', 'Über Dreiecksvermaschung abgeleitet', '');
		CREATE TABLE ax_dqerfassungsmethodesekundaeresdgm (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_dqerfassungsmethodesekundaeresdgm IS 'Alias: "AX_DQErfassungsmethodeSekundaeresDGM", UML-Typ: Enumeration';
		INSERT INTO ax_dqerfassungsmethodesekundaeresdgm (wert,beschreibung,dokumentation) VALUES
		('5000', 'Terrestrische Aufnahme', ''),
		('5010', 'Interaktive photogrammetrische Aufnahme', ''),
		('5020', 'Laserscanning', ''),
		('5030', 'Digitalisierung analoger Vorlagen', ''),
		('5040', 'Bildkorrelation', ''),
		('5050', 'IfSAR', ''),
		('5060', 'Amtliche Festlegung', '');
		CREATE TABLE ax_zustand_kanal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_kanal IS 'Alias: "AX_Zustand_Kanal", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_kanal (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich der Kanal nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('4000', 'Im Bau', 'Im Bau bedeutet, dass der Kanal noch nicht fertiggestellt ist.');
		CREATE TABLE ax_funktion_stehendesgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_stehendesgewaesser IS 'Alias: "AX_Funktion_StehendesGewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_stehendesgewaesser (wert,beschreibung,dokumentation) VALUES
		('8610', 'See', 'See ist eine natürliche oder künstlich angelegte, größere, stehende oder nahezu stehende Wasserfläche, die nach allgemeiner Verkehrsauffassung als See anzuse-hen ist.'),
		('8620', 'Teich', 'Teich ist eine natürliche oder künstlich angelegte, stehende oder nahezu stehende Wasserfläche, die nach allgemeiner Verkehrsauffassung als Teich anzusehen ist.'),
		('8630', 'Stausee', 'Stausee ist eine mit Wasser gefüllte, allseitig umschlossene Hohlform der Landoberfläche ohne unmittelbaren Zusammenhang mit Meer.'),
		('8631', 'Speicherbecken', 'Speicherbecken ist eine zeitweise mit Wasser gefüllte, allseitig umschlossene Hohlform der Landoberfläche ohne unmittelbaren Zusammenhang mit Meer'),
		('8640', 'Baggersee', 'Baggersee ist ein künstlich geschaffenes Gewässer, aus dem Bodenmaterial gefördert wird.'),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_schifffahrtskategorie (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_schifffahrtskategorie IS 'Alias: "AX_Schifffahrtskategorie", UML-Typ: Enumeration';
		INSERT INTO ax_schifffahrtskategorie (wert,beschreibung,dokumentation) VALUES
		('1000', 'Binnenwasserstraße', 'Binnenwasserstraße ist ein oberirdisches Gewässer oder Küstengewässer, das gesetzlich für den Personen- und/oder Güterverkehr mit Schiffen bestimmt ist. Binnengewässer im Küstengebiet sind gegen das Küstengewässer gesetzlich abgegrenzt.'),
		('2000', 'Seewasserstraße', 'Seewasserstraße ist ein als Wasserstraße gesetzlich festgelegter Teil eines Küstengewässers.'),
		('3000', 'Landesgewässer mit Verkehrsordnung', 'Landesgewässer mit Verkehrsordnung ist eine Wasserstraße, die keine Binnenwasserstraße ist. Die Schiffbarkeit wird durch eine Landesverkehrsordnung geregelt.');
		CREATE TABLE ax_hydrologischesmerkmal_fliessgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_fliessgewaesser IS 'Alias: "AX_HydrologischesMerkmal_Fliessgewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_fliessgewaesser (wert,beschreibung,dokumentation) VALUES
		('2000', 'Nicht ständig Wasser führend', 'Nicht ständig Wasser führend heißt, dass ein Gewässer nicht ganzjährig Wasser führt.');
		CREATE TABLE ax_schifffahrtskategorie_kanal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_schifffahrtskategorie_kanal IS 'Alias: "AX_Schifffahrtskategorie_Kanal", UML-Typ: Enumeration';
		INSERT INTO ax_schifffahrtskategorie_kanal (wert,beschreibung,dokumentation) VALUES
		('1000', 'Binnenwasserstraße', 'Binnenwasserstraße ist ein oberirdisches Gewässer oder Küstengewässer, das gesetzlich für den Personen- und/oder Güterverkehr mit Schiffen bestimmt ist. Binnengewässer im Küstengebiet sind gegen das Küstengewässer gesetzlich abgegrenzt.'),
		('2000', 'Seewasserstraße', 'Seewasserstraße ist ein als Wasserstraße gesetzlich festgelegter Teil eines Küstengewässers.'),
		('3000', 'Landesgewässer mit Verkehrsordnung', 'Landesgewässer mit Verkehrsordnung ist eine Wasserstraße, die keine Binnenwasserstraße ist. Die Schiffbarkeit wird durch eine Landesverkehrsordnung geregelt.');
		CREATE TABLE ax_funktion_fliessgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_fliessgewaesser IS 'Alias: "AX_Funktion_Fliessgewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_fliessgewaesser (wert,beschreibung,dokumentation) VALUES
		('8200', 'Fluss', 'Fluss ist ein natürliches, fließendes Gewässer (ggf. auch mit begradigten, kanalisierten Teilstücken), das wegen seiner Größe und Bedeutung nach allgemeiner Verkehrsauffassung als Fluss angesprochen wird.'),
		('8210', 'Altwasser', 'Altwasser ist ein Teil eines Fließgewässers, der bei einer Begradigung vom fließenden Gewässer abgeschnitten wurde und wichtiger Lebensraum für z. B. Wasser-vögel, Amphibien und Libellen ist.'),
		('8220', 'Altarm', ''),
		('8230', 'Flussmündungstrichter', 'Flussmündungstrichter ist der Bereich des Flusses im Übergang zum Meer. Er beginnt dort, wo die bis dahin etwa parallel verlaufenden Ufer des Flusses sich trichterförmig zur offenen See hin erweitern. Die Abgrenzungen der Flussmündungstrichter ergeben sich aus dem Bundeswasserstraßengesetz (meerseitig) und den Bekanntmachungen der Wasser- und Schifffahrtsverwaltung sowie höchst-richterlicher Rechtsprechung (binnenseitig).'),
		('8300', 'Kanal', 'Kanal ist ein für die Schifffahrt angelegter, künstlicher Wasserlauf.'),
		('8400', 'Graben', 'Graben ist ein ständig oder zeitweise fließendes, künstlich angelegtes oder natürliches Gewässer, das nach allgemeiner Verkehrsauffassung als Graben anzusehen ist.'),
		('8410', 'Fleet', ''),
		('8500', 'Bach', 'Bach ist ein natürliches, fließendes Gewässer, das wegen seiner geringen Größe und Bedeutung nach allgemeiner Verkehrsauffassung als Bach anzusehen ist.');
		CREATE TABLE ax_widmung_wasserlauf (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_widmung_wasserlauf IS 'Alias: "AX_Widmung_Wasserlauf", UML-Typ: Enumeration';
		INSERT INTO ax_widmung_wasserlauf (wert,beschreibung,dokumentation) VALUES
		('1310', 'Gewässer I. Ordnung - Bundeswasserstraße', 'Gewässer I. Ordnung - Bundeswasserstraße ist ein Gewässer, das der Zuständigkeit des Bundes obliegt.'),
		('1320', 'Gewässer I. Ordnung - nach Landesrecht', 'Gewässer I. Ordnung - nach Landesrecht ist ein Gewässer, das der Zuständigkeit des Landes obliegt.'),
		('1330', 'Gewässer II. Ordnung', 'Gewässer II. Ordnung ist ein Gewässer, für das die Unterhaltungsverbände zuständig sind.'),
		('1340', 'Gewässer III. Ordnung', 'Gewässer III. Ordnung ist ein Gewässer, das weder zu den Gewässern I. noch II. Ordnung zählt.');
		CREATE TABLE ax_funktion_meer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_meer IS 'Alias: "AX_Funktion_Meer", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_meer (wert,beschreibung,dokumentation) VALUES
		('8710', 'Küstengewässer', 'Küstengewässer ist die Fläche zwischen der Küstenlinie bei mittlerem Hochwasser oder der seewärtigen Begrenzung der oberirdischen Gewässer und der seewärtigen Begrenzung des deutschen Hoheitsgebietes. Dem mittleren Hochwasser ist der mittlere Wasserstand der Ostsee gleichzusetzen.');
		CREATE TABLE ax_hydrologischesmerkmal_gewaesserachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_gewaesserachse IS 'Alias: "AX_HydrologischesMerkmal_Gewaesserachse", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_gewaesserachse (wert,beschreibung,dokumentation) VALUES
		('2000', 'Nicht ständig Wasser führend', 'Nicht ständig Wasser führend heißt, dass ein Gewässer nicht ganzjährig Wasser führt.'),
		('3000', 'Trocken, versiegt', 'Trocken, versiegt heißt, dass ein Gewässerbett ganzjährig kein Wasser führt.');
		CREATE TABLE ax_tidemerkmal_meer (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_tidemerkmal_meer IS 'Alias: "AX_Tidemerkmal_Meer", UML-Typ: Enumeration';
		INSERT INTO ax_tidemerkmal_meer (wert,beschreibung,dokumentation) VALUES
		('1000', 'Mit Tideeinfluss', 'Mit Tideeinfluss sind periodische Änderungen des Wasserspiegels und horizontale Bewegungen des Wassers, hervorgerufen durch die Massenanziehungs- und Fliehkräfte des Systems Sonne, Mond und Erde in Verbindung mit der Erdrotation.');
		CREATE TABLE ax_nutzung_hafenbecken (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_nutzung_hafenbecken IS 'Alias: "AX_Nutzung_Hafenbecken", UML-Typ: Enumeration';
		INSERT INTO ax_nutzung_hafenbecken (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zivil', 'Zivil bedeutet, dass Hafenbecken privaten oder öffentlichen Zwecken dient und nicht militärisch genutzt wird.'),
		('2000', 'Militärisch', 'Militärisch bedeutet, dass Hafenbecken nur von Streitkräften genutzt wird.'),
		('3000', 'Teils zivil, teils militärisch', 'Teils zivil, teils militärisch bedeutet, dass Hafenbecken sowohl zivil als auch militärisch genutzt wird.');
		CREATE TABLE ax_hydrologischesmerkmal_stehendesgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_hydrologischesmerkmal_stehendesgewaesser IS 'Alias: "AX_HydrologischesMerkmal_StehendesGewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_hydrologischesmerkmal_stehendesgewaesser (wert,beschreibung,dokumentation) VALUES
		('2000', 'Nicht ständig Wasser führend', 'Nicht ständig Wasser führend heißt, dass ein Gewässer nicht ganzjährig Wasser führt.');
		CREATE TABLE ax_widmung_stehendesgewaesser (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_widmung_stehendesgewaesser IS 'Alias: "AX_Widmung_StehendesGewaesser", UML-Typ: Enumeration';
		INSERT INTO ax_widmung_stehendesgewaesser (wert,beschreibung,dokumentation) VALUES
		('1310', 'Gewässer I. Ordnung - Bundeswasserstraße', 'Gewässer I. Ordnung - Bundeswasserstraße ist ein Gewässer, das der Zuständigkeit  des Bundes obliegt.'),
		('1320', 'Gewässer I. Ordnung - nach Landesrecht', 'Gewässer I. Ordnung - nach Landesrecht ist ein Gewässer, das der Zuständigkeit des Landes obliegt.'),
		('1330', 'Gewässer II. Ordnung', 'Gewässer II. Ordnung ist ein Gewässer, für das die Unterhaltungsverbände zuständig sind.'),
		('1340', 'Gewässer III. Ordnung', 'Gewässer III. Ordnung ist ein Gewässer, das weder zu den Gewässern I. noch II. Ordnung zählt.');
		CREATE TABLE ax_funktion_gewaesserachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_gewaesserachse IS 'Alias: "AX_Funktion_Gewaesserachse", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_gewaesserachse (wert,beschreibung,dokumentation) VALUES
		('8300', 'Kanal', 'Kanal ist ein für die Schifffahrt angelegter künstlicher Wasserlauf.');
		CREATE TABLE ax_funktion_hafenbecken (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_hafenbecken IS 'Alias: "AX_Funktion_Hafenbecken", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_hafenbecken (wert,beschreibung,dokumentation) VALUES
		('8810', 'Sportboothafenbecken', '');
		CREATE TABLE ax_widmung_kanal (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_widmung_kanal IS 'Alias: "AX_Widmung_Kanal", UML-Typ: Enumeration';
		INSERT INTO ax_widmung_kanal (wert,beschreibung,dokumentation) VALUES
		('1310', 'Gewässer I. Ordnung - Bundeswasserstraße', 'Gewässer I. Ordnung -Bundeswasserstraße ist ein Gewässer, das der Zuständigkeit  des Bundes obliegt.'),
		('1320', 'Gewässer I. Ordnung - nach Landesrecht', 'Gewässer I. Ordnung - nach Landesrecht ist ein Gewässer, das der Zuständigkeit des Landes obliegt.'),
		('1330', 'Gewässer II. Ordnung', 'Gewässer II. Ordnung ist ein Gewässer, für das die Unterhaltungsverbände zuständig sind.'),
		('1340', 'Gewässer III. Ordnung', 'Gewässer III. Ordnung ist ein Gewässer, das weder zu den Gewässern I. noch II. Ordnung zählt.');
		CREATE TABLE ax_zustand_wohnbauflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_wohnbauflaeche IS 'Alias: "AX_Zustand_Wohnbauflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_wohnbauflaeche (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich die Fläche nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_artderbebauung_wohnbauflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderbebauung_wohnbauflaeche IS 'Alias: "AX_ArtDerBebauung_Wohnbauflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_artderbebauung_wohnbauflaeche (wert,beschreibung,dokumentation) VALUES
		('1000', 'Offen', 'Offen beschreibt die Bebauung von Wohnbaufläche, die vorwiegend durch einzelstehende Gebäude charakterisiert wird.'),
		('2000', 'Geschlossen', 'Geschlossen beschreibt die Bebauung von Wohnbaufläche, die vorwiegend durch zusammenhängende Gebäude charakterisiert wird. Die Gebäudeabdeckung ist in der Regel >50 Prozent der Wohnbaufläche.');
		CREATE TABLE ax_zustand_flaechebesondererfunktionalerpraegung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_flaechebesondererfunktionalerpraegung IS 'Alias: "AX_Zustand_FlaecheBesondererFunktionalerPraegung", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_flaechebesondererfunktionalerpraegung (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich die Fläche funktionaler Prägung nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_funktion_flaechegemischternutzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_flaechegemischternutzung IS 'Alias: "AX_Funktion_FlaecheGemischterNutzung", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_flaechegemischternutzung (wert,beschreibung,dokumentation) VALUES
		('2100', 'Gebäude- und Freifläche, Mischnutzung mit Wohnen', ''),
		('2110', 'Wohnen mit Öffentlich', ''),
		('2120', 'Wohnen mit Handel und Dienstleistungen', ''),
		('2130', 'Wohnen mit Gewerbe und Industrie', ''),
		('2140', 'Öffentlich mit Wohnen', ''),
		('2150', 'Handel und Dienstleistungen mit Wohnen', ''),
		('2160', 'Gewerbe und Industrie mit Wohnen', ''),
		('2700', 'Gebäude- und Freifläche Land- und Forstwirtschaft', 'Gebäude- und Freifläche Land- und Forstwirtschaft ist eine Fläche, die der Land- und Forstwirtschaft dient.'),
		('2710', 'Wohnen', ''),
		('2720', 'Betrieb', ''),
		('2730', 'Wohnen und Betrieb', ''),
		('6800', 'Landwirtschaftliche Betriebsfläche', 'Landwirtschaftliche Betriebsfläche ist eine bebaute und unbebaute Fläche, die dem landwirtschaftlichen Betrieb dient.'),
		('7600', 'Forstwirtschaftliche Betriebsfläche', 'Forstwirtschaftliche Betriebsfläche ist eine bebaute und unbebaute Fläche, die dem forstwirtschaftlichen Betrieb dient.');
		CREATE TABLE ax_foerdergut_industrieundgewerbeflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_foerdergut_industrieundgewerbeflaeche IS 'Alias: "AX_Foerdergut_IndustrieUndGewerbeflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_foerdergut_industrieundgewerbeflaeche (wert,beschreibung,dokumentation) VALUES
		('1000', 'Erdöl', 'Erdöl ist ein flüssiges und brennbares Kohlenwasserstoffgemisch, das gefördert wird.'),
		('2000', 'Erdgas', 'Erdgas ist ein in der Erdkruste vorkommendes brennbares Naturgas, das gefördert wird.'),
		('3000', 'Sole, Lauge', 'Sole, Lauge ist ein kochsalzhaltiges Wasser, das gefördert wird.'),
		('4000', 'Kohlensäure', 'Kohlensäure ist eine schwache Säure, die durch Lösung von Kohlendioxid in Wasser entsteht und gefördert wird.'),
		('5000', 'Erdwärme', 'Erdwärme ist eine auf natürlichem Wege sich erneuernde Wärmeenergie, die aus einer geothermisch geringen Tiefenstufe der Erdkruste gefördert wird.');
		CREATE TABLE ax_artderbebauung_flaechegemischternutzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderbebauung_flaechegemischternutzung IS 'Alias: "AX_ArtDerBebauung_FlaecheGemischterNutzung", UML-Typ: Enumeration';
		INSERT INTO ax_artderbebauung_flaechegemischternutzung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Offen', 'Offen beschreibt die Bebauung von Fläche gemischter Nutzung, die vorwiegend durch einzelstehende Gebäude charakterisiert wird.'),
		('2000', 'Geschlossen', 'Geschlossen beschreibt die Bebauung von Fläche gemischter Nutzung, die vorwiegend durch zusammenhängende Gebäude charakterisiert wird. Die Gebäudeabdeckung ist in der Regel > 50 Prozent der Fläche.');
		CREATE TABLE ax_zustand_sportfreizeitunderholungsflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_sportfreizeitunderholungsflaeche IS 'Alias: "AX_Zustand_SportFreizeitUndErholungsflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_sportfreizeitunderholungsflaeche (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', ''),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_funktion_flaechebesondererfunktionalerpraegung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_flaechebesondererfunktionalerpraegung IS 'Alias: "AX_Funktion_FlaecheBesondererFunktionalerPraegung", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_flaechebesondererfunktionalerpraegung (wert,beschreibung,dokumentation) VALUES
		('1100', 'Öffentliche Zwecke', 'Öffentliche Zwecke bezeichnet eine Fläche, die der Erfüllung öffentlicher Aufgaben und der Allgemeinheit dient.'),
		('1110', 'Verwaltung', 'Verwaltung bezeichnet eine Fläche auf der vorwiegend Gebäude der öffentlichen Verwaltung, z. B. Rathaus, Gericht, Kreisverwaltung stehen.'),
		('1120', 'Bildung und Forschung', 'Bildung und Forschung bezeichnet eine Fläche, auf der vorwiegend Gebäude stehen, in denen geistige, kulturelle und soziale Fähigkeiten vermittelt werden und/oder wissenschaftliche Forschung betrieben wird (z.B. Schulen, Universitäten, Forschungsinstitute).'),
		('1130', 'Kultur', 'Kultur bezeichnet eine Fläche auf der vorwiegend Anlagen und Gebäude für kulturelle Zwecke, z.B. Konzert- und Museumsgebäude, Bibliotheken, Theater, Schlösser und Burgen sowie Rundfunk- und Fernsehgebäude stehen.'),
		('1140', 'Religiöse Einrichtung', 'Religiöse Einrichtung bezeichnet eine Fläche auf der vorwiegend religiöse Gebäude stehen.'),
		('1150', 'Gesundheit, Kur', 'Gesundheit, Kur bezeichnet eine Fläche auf der vorwiegend Gebäude des Gesundheitswesens stehen, z.B. Krankenhäuser, Heil- und Pflegeanstalten.'),
		('1160', 'Soziales', 'Soziales bezeichnet eine Fläche auf der vorwiegend Gebäude des Sozialwesens stehen, z. B. Kindergärten, Jugend- und Senioreneinrichtungen, Freizeit-, Fremden- und Obdachlosenheime.'),
		('1170', 'Sicherheit und Ordnung', 'Sicherheit und Ordnung bezeichnet eine Fläche auf der vorwiegend Anlagen und Gebäude der Polizei, der Bundeswehr, der Feuerwehr und der Justizvollzugsbehörden stehen.'),
		('1200', 'Parken', 'Parken bezeichnet eine Fläche auf der vorwiegend Anlagen und Gebäude zum vorübergehenden Abstellen von Fahrzeugen stehen.'),
		('1300', 'Historische Anlage', 'Historische Anlage ist eine Fläche mit historischen Anlagen, z. B. historische Stadtmauern und -türme, Denkmälern und Ausgrabungsstätten.'),
		('1310', 'Burg-, Festungsanlage', ''),
		('1320', 'Schlossanlage', '');
		CREATE TABLE ax_funktion_sportfreizeitunderholungsflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_sportfreizeitunderholungsflaeche IS 'Alias: "AX_Funktion_SportFreizeitUndErholungsflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_sportfreizeitunderholungsflaeche (wert,beschreibung,dokumentation) VALUES
		('4001', 'Gebäude- und Freifläche Sport, Freizeit und Erholung', 'Gebäude- und Freifläche Sport, Freizeit und Erholung ist eine bebaute Fläche, die dem Sport, der Freizeitgestaltung oder der Erholung dient.'),
		('4100', 'Sportanlage', 'Sportanlage ist eine Fläche mit Bauwerken und Einrichtungen, die zur Ausübung von (Wettkampf-)sport und für Zuschauer bestimmt ist.'),
		('4101', 'Gebäude- u. Freifläche Erholung, Sport', ''),
		('4110', 'Golfplatz', 'Golfplatz ist eine Fläche mit Bauwerken und Einrichtungen, die zur Ausübung des Golfsports genutzt wird.'),
		('4120', 'Sportplatz', ''),
		('4130', 'Rennbahn', ''),
		('4140', 'Reitplatz', ''),
		('4150', 'Schießanlage', ''),
		('4160', 'Eis-, Rollschuhbahn', ''),
		('4170', 'Tennisplatz', ''),
		('4200', 'Freizeitanlage', 'Freizeitanlage ist eine Fläche mit Bauwerken und Einrichtungen, die zur Freizeitgestaltung bestimmt ist.'),
		('4210', 'Zoo', 'Zoo ist ein Gelände mit Tierschauhäusern und umzäunten Gehegen, auf dem Tiere gehalten und gezeigt werden.'),
		('4211', 'Gebäude- u. Freifläche Erholung, Zoologie', ''),
		('4220', 'Safaripark, Wildpark', 'Safaripark, Wildpark, ist ein Gelände mit umzäunten Gehegen, in denen Tiere im Freien gehalten und gezeigt werden.'),
		('4230', 'Freizeitpark', 'Freizeitpark ist ein Gelände mit Karussells, Verkaufs- und Schaubuden und/oder Wildgattern, das der Freizeitgestaltung dient.'),
		('4240', 'Freilichttheater', 'Freilichttheater ist eine Anlage mit Bühne und Zuschauerbänken für Theateraufführungen im Freien.'),
		('4250', 'Freilichtmuseum', 'Freilichtmuseum ist eine volkskundliche Museumsanlage, in der Wohnformen oder historische 
Betriebsformen in ihrer natürlichen Umgebung im Freien dargestellt sind.'),
		('4260', 'Autokino, Freilichtkino', 'Autokino, Freilichtkino ist ein Lichtspieltheater im Freien, in dem der Film im Allgemeinen vom Auto aus angesehen wird.'),
		('4270', 'Verkehrsübungsplatz', 'Verkehrsübungsplatz ist eine Fläche, die Übungs- und Erprobungszwecken dient.'),
		('4280', 'Hundeübungsplatz', 'Hundeübungsplatz ist eine Fläche, auf der Übungen mit Hunden durchgeführt werden.'),
		('4290', 'Modellflugplatz', 'Modellflugplatz ist eine Fläche, die zur Ausübung des Modellflugsports dient.'),
		('4300', 'Erholungsfläche', 'Erholungsfläche ist eine Fläche mit Bauwerken und Einrichtungen, die zur Erholung bestimmt ist.'),
		('4301', 'Gebäude- und Freifläche Erholung', ''),
		('4310', 'Wochenend- und Ferienhausfläche', 'Wochenend- und Ferienhausfläche bezeichnet eine extra dafür ausgewiesene Fläche auf der vorwiegend Wochenend- und Ferienhäuser stehen dürfen.'),
		('4320', 'Schwimmbad, Freibad', 'Schwimmbad, Freibad ist eine Anlage mit Schwimmbecken oder Anlage an Ufern von Gewässern für den Badebetrieb und Schwimmsport.'),
		('4321', 'Gebäude- u. Freifläche Erholung, Bad', ''),
		('4330', 'Campingplatz', 'Campingplatz ist eine Fläche für den Aufbau einer größeren Zahl von Zelten oder zum Abstellen und Benutzen von Wohnwagen mit ortsfesten Anlagen und Einrichtungen.'),
		('4331', 'Gebäude- u. Freifläche Erholung, Camping', ''),
		('4400', 'Grünanlage', 'Grünanlage ist eine Anlage mit Bäumen, Sträuchern, Rasenflächen, Blumenrabatten und Wegen, die vor allem der Erholung und Verschönerung des Stadtbildes dient.'),
		('4410', 'Grünfläche', 'Grünfläche ist eine unbebaute Wiese, Rasenfläche und Parkanlage in Städten und Siedlungen.'),
		('4420', 'Park', 'Park ist eine landschaftsgärtnerisch gestaltete Grünanlage, die der Repräsentation und der Erholung dient.'),
		('4430', 'Botanischer Garten', 'Botanischer Garten ist ein der Öffentlichkeit zugänglicher Garten zum Studium der Pflanzenwelt; systematisch geordnete Sammlung in Freiland und Gewächs-häusern (Warmhäuser).'),
		('4431', 'Gebäude- u. Freifläche Erholung, Botanik', ''),
		('4440', 'Kleingarten', 'Kleingarten (Schrebergarten) ist eine Anlage von Gartengrundstücken, die von Vereinen verwaltet und verpachtet werden.'),
		('4450', 'Wochenendplatz', ''),
		('4460', 'Garten', ''),
		('4470', 'Spielplatz, Bolzplatz', 'Spielplatz, Bolzplatz ist ein Platz an dem körperliche oder geistige Tätigkeit aus eigenem Antrieb ohne Zweckbestimmung ausgeübt wird.'),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_lagergut_industrieundgewerbeflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagergut_industrieundgewerbeflaeche IS 'Alias: "AX_Lagergut_IndustrieUndGewerbeflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_lagergut_industrieundgewerbeflaeche (wert,beschreibung,dokumentation) VALUES
		('1000', 'Baustoffe', ''),
		('2000', 'Kohle', ''),
		('3000', 'Öl', ''),
		('4000', 'Erde', ''),
		('5000', 'Schutt', ''),
		('6000', 'Schlacke', ''),
		('7000', 'Abraum', ''),
		('8000', 'Schrott, Altmaterial', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_zustand_halde (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_halde IS 'Alias: "AX_Zustand_Halde", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_halde (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich die Halde nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_zustand_bergbaubetrieb (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_bergbaubetrieb IS 'Alias: "AX_Zustand_Bergbaubetrieb", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_bergbaubetrieb (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich der Bergbaubetrieb nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_abbaugut_tagebaugrubesteinbruch (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_abbaugut_tagebaugrubesteinbruch IS 'Alias: "AX_Abbaugut_TagebauGrubeSteinbruch", UML-Typ: Enumeration';
		INSERT INTO ax_abbaugut_tagebaugrubesteinbruch (wert,beschreibung,dokumentation) VALUES
		('1000', 'Erden, Lockergestein', 'Erden, Lockergestein bedeutet, dass feinkörnige Gesteine abgebaut werden.'),
		('1001', 'Ton', 'Ton ist ein Abbaugut, das aus gelblichem bis grauem Lockergestein besteht und durch Verwitterung älterer Gesteine entsteht.'),
		('1002', 'Bentonit', 'Bentonit ist ein tonartiges Abbaugut, das durch Verwitterung vulkanischer Asche (Tuffe) entstanden ist.'),
		('1003', 'Kaolin', 'Kaolin ist ein Abbaugut, das aus weißem, erdigem Gestein, fast reinem Aluminiumsilikat (kieselsaure Tonerde) besteht.'),
		('1004', 'Lehm', 'Lehm ist ein Abbaugut, das durch Verwitterung entstanden ist und aus gelb bis braun gefärbtem sandhaltigem Ton besteht.'),
		('1005', 'Löß, Lößlehm', 'Löß, Lößlehm ist ein Abbaugut das aus feinsten gelblichen Sedimenten besteht und eine hohe Wasserspeicherfähigkeit aufweist.'),
		('1006', 'Mergel', 'Mergel ist ein Abbaugut das aus kalk- und tonartigem Sedimentgestein besteht.'),
		('1007', 'Kalk, Kalktuff, Kreide', 'Kalk, Kalktuff, Kreide ist ein Abbaugut, das aus erdigem weißen Kalkstein besteht.'),
		('1008', 'Sand', 'Sand ist ein Abbaugut, das aus kleinen, losen Mineralkörnern (häufig Quarz) besteht.'),
		('1009', 'Kies, Kiessand', 'Kies, Kiessand ist ein Abbaugut, das aus vom Wasser rund geschliffenen Gesteinsbrocken besteht.'),
		('1011', 'Farberden', 'Farberden ist ein Abbaugut, das durch Verwitterung entstanden ist und vorrangig aus eisenhaltigem Gestein besteht.'),
		('1012', 'Quarzsand', 'Quarzsand ist ein Abbaugut, das vorwiegend aus kleinen, losen Quarzkörnern besteht.'),
		('1013', 'Kieselerde', 'Kieselerde ist ein Abbaugut, das durch tertiäre Binnenseeablagerungen aus Kieselschalen toter Kieselalgen entstanden ist.'),
		('2000', 'Steine, Gestein, Festgestein', 'Steine, Gestein, Festgestein bedeutet, dass grobkörnige oder feste Gesteine abgebaut werden.'),
		('2001', 'Tonstein', 'Tonstein ist ein gelblich bis graues Abbaugut, das überwiegend aus Tonmineralien besteht.'),
		('2002', 'Schiefer, Dachschiefer', 'Schiefer, Dachschiefer ist ein toniges Abbaugut, das in dünne ebene Platten spaltbar ist.'),
		('2003', 'Metamorpher Schiefer', 'Metamorpher Schiefer ist ein Abbaugut, dessen ursprüngliche Zusammensetzung und Struktur durch Wärme und Druck innerhalb der Erdkruste verändert worden ist.'),
		('2004', 'Mergelstein', 'Mergelstein ist ein Abbaugut, das sich größtenteils aus Ton und Kalk zusammensetzt.'),
		('2005', 'Kalkstein', 'Kalkstein ist ein Abbaugut, das als weit verbreitetes Sedimentgestein überwiegend aus Calciumcarbonat besteht.'),
		('2006', 'Dolomitstein', 'Dolomitstein ist ein Abbaugut, das überwiegend aus calcium- und magnesiumhaltigen Mineralien besteht.'),
		('2007', 'Travertin', 'Travertin ist ein Abbaugut, das aus gelblichen Kiesel- oder Kalktuffen besteht.'),
		('2008', 'Marmor', 'Marmor ist ein Abbaugut, das als rein weißer kristalliner, körniger Kalkstein (Calciumcarbonat) vorkommt.'),
		('2009', 'Sandstein', 'Sandstein ist ein Abbaugut, das aus verfestigtem Sedimentgestein besteht.'),
		('2010', 'Grauwacke', 'Grauwacke ist ein Abbaugut, das aus tonhaltigem Sandstein besteht und mit Gesteinsbruchstücken angereichert sein kann.'),
		('2011', 'Quarzit', 'Quarzit ist ein sehr hartes metamorphes Abbaugut, das vorwiegend aus feinkörnigen Quarzmineralien besteht.'),
		('2012', 'Gneis', 'Gneis ist ein metamorphes Abbaugut mit Schieferung, das aus Feldspat, Quarz und Glimmer besteht.'),
		('2013', 'Basalt, Diabas', 'Basalt, Diabas ist ein Abbaugut, das aus basischem Ergussgestein besteht.'),
		('2014', 'Andesit', 'Andesit ist ein Abbaugut, das aus Ergussgestein besteht.'),
		('2015', 'Porphyr, Quarzporphyr', 'Porphyr, Quarzporphyr ist ein eruptiv entstandenes Abbaugut, das aus einer dichten Grundmasse und groben Einsprenglingen besteht.'),
		('2016', 'Granit', 'Granit ist ein eruptiv entstandenes Abbaugut, das aus körnigem Feldspat, Quarz, Glimmer besteht.'),
		('2017', 'Granodiorit', 'Granodiorit ist ein hell- bis dunkelgraues Abbaugut. Es ist ein mittelkörniges Tiefengestein mit den Hauptbestandteilen Feldspat, Quarz, Hornblende und Biotit.'),
		('2018', 'Tuff-, Bimsstein', 'Tuff-, Bimsstein ist ein helles, sehr poröses Abbaugut, das durch rasches Erstarren der Lava entstanden ist.'),
		('2019', 'Trass', 'Trass ist ein Abbaugut, das aus vulkanischem Aschentuff (Bimsstein) besteht.'),
		('2020', 'Lavaschlacke', 'Lavaschlacke ist ein Abbaugut, das aus ausgestoßenem, geschmolzenen Vulkangestein besteht.'),
		('2021', 'Talkschiefer, Speckstein', 'Talkschiefer, Speckstein ist ein farbloses bis graugrünes, sich fettig anfühlendes Abbaugut, das aus dem weichen Mineral Talk besteht.'),
		('4000', 'Treib- und Brennstoffe', 'Treib- und Brennstoffe bedeutet, dass die in der Natur vorkommenden brennbaren organischen und anorganischen Substanzen abgebaut oder gewonnen werden.'),
		('4010', 'Torf', 'Torf ist ein Abbaugut, das aus der unvollkommenen Zersetzung abgestorbener pflanzlicher Substanz unter Luftabschluss in Mooren entstanden ist.'),
		('4020', 'Kohle', 'Kohle ist ein Abbaugut, das durch Inkohlung (Umwandlungsprozess pflanzlicher Substanzen) entstanden ist.'),
		('4021', 'Braunkohle', 'Braunkohle ist ein Abbaugut, das durch einen bestimmten Grad von Inkohlung (Umwandlungsprozess pflanzlicher Substanzen) entstanden ist.'),
		('4022', 'Steinkohle', 'Steinkohle ist ein Abbaugut, das durch vollständige Inkohlung (Umwandlungsprozess pflanzlicher Substanzen) entstanden ist.'),
		('4030', 'Ölschiefer', 'Ölschiefer ist ein Abbaugut, das aus dunklem, bitumenhaltigen, tonigen Gestein besteht.'),
		('5000', 'Industrieminerale, Salze', 'Industrieminerale, Salze bedeutet, dass die in der Natur vorkommenden Mineralien abgebaut werden.'),
		('5001', 'Gipsstein', 'Gipsstein ist ein natürliches Abbaugut.'),
		('5002', 'Anhydritstein', 'Anhydritstein ist ein Abbaugut, das aus wasserfreiem Gips besteht.'),
		('5005', 'Kalkspat', 'Kalkspat ist ein weißes oder hell gefärbtes Abbaugut (Calciumcarbonat).'),
		('5007', 'Schwerspat', 'Schwerspat ist ein formenreiches, rhombisches weißes bis farbiges Abbaugut.'),
		('5008', 'Quarz', 'Quarz ist ein Abbaugut, das aus verschiedenen Gesteinsarten (Granit, Gneis, Sandstein) gewonnen wird.'),
		('5009', 'Feldspat', 'Feldspat ist ein weiß bis grauweißes gesteinsbildendes Mineral von blättrigem Bruch, das abgebaut wird.'),
		('5010', 'Pegmatitsand', 'Pegmatitsand ist ein Abbaugut, das durch Verwitterung von Granit und Gneis entstanden ist.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass das Abbaugut bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_primaerenergie_industrieundgewerbeflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_primaerenergie_industrieundgewerbeflaeche IS 'Alias: "AX_Primaerenergie_IndustrieUndGewerbeflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_primaerenergie_industrieundgewerbeflaeche (wert,beschreibung,dokumentation) VALUES
		('1000', 'Wasser', 'Wasser bedeutet, dass das Kraftwerk potentielle und kinetische Energie des Wasserkreislaufs in elektrische Energie umwandelt.'),
		('2000', 'Kernkraft', 'Kernkraft bedeutet, dass das Kraftwerk die durch Kernspaltung gewonnene Energie in eine andere Energieform umwandelt.'),
		('3000', 'Sonne', 'Sonne bedeutet, dass das Kraftwerk bzw. Heizwerk Sonnenenergie in eine andere Energieform umwandelt.'),
		('4000', 'Wind', 'Wind bedeutet, dass das Kraftwerk die Strömungsenergie des Windes in elektrische Energie umwandelt.'),
		('5000', 'Gezeiten', 'Gezeiten bedeutet, dass das Kraftwerk die kinetische Energie der Meeresgezeiten in elektrische Energie umwandelt.'),
		('6000', 'Erdwärme', 'Erdwärme bedeutet, dass das Heizwerk die geothermische Energie der Erde nutzt.'),
		('7000', 'Verbrennung', 'Verbrennung bedeutet, dass das Kraftwerk bzw. Heizwerk die durch Verbrennung freiwerdende Energie in eine andere Energieform umwandelt.'),
		('7100', 'Kohle', 'Kohle bedeutet, dass das Kraftwerk bzw. Heizwerk die durch Verbrennung von Kohle freiwerdende Energie in eine andere Energieform umwandelt.'),
		('7200', 'Öl', 'Öl bedeutet, dass das Kraftwerk bzw. Heizwerk die durch Verbrennung von Öl freiwerdende Energie in eine andere Energieform umwandelt.'),
		('7300', 'Gas', 'Gas bedeutet, dass das Kraftwerk bzw. Heizwerk die durch Verbrennung von Gas freiwerdende Energie in eine andere Energieform umwandelt.'),
		('7400', 'Müll, Abfall', 'Müll, Abfall bedeutet, dass das Kraftwerk bzw. Heizwerk die durch Verbrennung von Müll bzw. Abfall freiwerdende Energie in eine andere Energieform umwandelt.');
		CREATE TABLE ax_abbaugut_bergbaubetrieb (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_abbaugut_bergbaubetrieb IS 'Alias: "AX_Abbaugut_Bergbaubetrieb", UML-Typ: Enumeration';
		INSERT INTO ax_abbaugut_bergbaubetrieb (wert,beschreibung,dokumentation) VALUES
		('1000', 'Erden, Lockergestein', 'Erden, Lockergestein bedeutet, dass feinkörnige Gesteine abgebaut werden.'),
		('1001', 'Ton', 'Ton ist ein Abbaugut, das aus gelblichem bis grauem Lockergestein besteht und durch Verwitterung älterer Gesteine entsteht.'),
		('1007', 'Kalk, Kalktuff, Kreide', 'Kalk, Kalktuff, Kreide ist ein Abbaugut, das aus erdigem weißen Kalkstein besteht.'),
		('2000', 'Steine, Gestein, Festgestein', 'Steine, Gestein, Festgestein bedeutet, dass grobkörnige oder feste Gesteine abgebaut werden.'),
		('2002', 'Schiefer, Dachschiefer', 'Schiefer, Dachschiefer ist ein toniges Abbaugut, das in dünne ebene Platten spaltbar ist.'),
		('2003', 'Metamorpher Schiefer', 'Metamorpher Schiefer ist ein Abbaugut, dessen ursprüngliche Zusammensetzung und Struktur durch Wärme und Druck innerhalb der Erdkruste verändert worden ist.'),
		('2005', 'Kalkstein', 'Kalkstein ist ein Abbaugut, das als weit verbreitetes Sedimentgestein überwiegend aus Calciumcarbonat besteht.'),
		('2006', 'Dolomitstein', 'Dolomitstein ist ein Abbaugut, das überwiegend aus calcium- und magnesiumhaltigen Mineralien besteht.'),
		('2013', 'Basalt, Diabas', 'Basalt, Diabas ist ein Abbaugut, das aus basischem Ergussgestein besteht.'),
		('2021', 'Talkschiefer, Speckstein', 'Talkschiefer, Speckstein ist ein farbloses bis graugrünes, sich fettig anfühlendes Abbaugut, das aus dem weichen Mineral Talk besteht.'),
		('3000', 'Erze', 'Erze bedeutet, dass die in der Natur vorkommenden, metallhaltigen Mineralien und Mineralgemische abgebaut oder gespeichert werden.'),
		('3001', 'Eisen', 'Eisen wird als Eisenerz abgebaut und durch Verhüttung gewonnen.'),
		('3002', 'Buntmetallerze', 'Buntmetallerze ist das Abbaugut, das alle Nichteisenmetallerze als Sammelbegriff umfasst.'),
		('3003', 'Kupfer', 'Kupfer wird als Kupfererz abgebaut und durch Verhüttung gewonnen.'),
		('3004', 'Blei', 'Blei wird als Bleierz abgebaut und durch spezielle Verfahren gewonnen.'),
		('3005', 'Zink', 'Zink wird als Zinkerz abgebaut und durch spezielle Verfahren gewonnen.'),
		('3006', 'Zinn', 'Zinn wird als Zinnerz abgebaut und durch spezielle Verfahren gewonnen.'),
		('3007', 'Wismut, Kobalt, Nickel', 'Wismut, Kobalt, Nickel werden als Erze abgebaut und durch spezielle Verfahren gewonnen.'),
		('3008', 'Uran', 'Uran wird als Uranerz abgebaut und durch spezielle Verfahren gewonnen.'),
		('3009', 'Mangan', 'Mangan wird als Manganerz abgebaut und durch spezielle Verfahren gewonnen.'),
		('3010', 'Antimon', 'Antimon wird als Antimonerz abgebaut und durch spezielle Verfahren gewonnen.'),
		('3011', 'Edelmetallerze', 'Edelmetallerze ist das Abbaugut, aus dem Edelmetalle (z. B. Gold, Silber) gewonnen werden.'),
		('4000', 'Treib- und Brennstoffe', 'Treib- und Brennstoffe bedeutet, dass die in der Natur vorkommenden brennbaren organischen und anorganischen Substanzen abgebaut oder gewonnen werden.'),
		('4020', 'Kohle', 'Kohle ist ein Abbaugut, das durch Inkohlung (Umwandlungsprozess pflanzlicher Substanzen) entstanden ist.'),
		('4021', 'Braunkohle', 'Braunkohle ist ein Abbaugut, das durch einen bestimmten Grad von Inkohlung (Umwandlungsprozess pflanzlicher Substanzen) entstanden ist.'),
		('4022', 'Steinkohle', 'Steinkohle ist ein Abbaugut, das durch vollständige Inkohlung (Umwandlungsprozess pflanzlicher Substanzen) entstanden ist.'),
		('4030', 'Ölschiefer', 'Ölschiefer ist ein Abbaugut, das aus dunklem, bitumenhaltigem, tonigem Gestein besteht.'),
		('5000', 'Industrieminerale, Salze', 'Industrieminerale, Salze bedeutet, dass die in der Natur vorkommenden Mineralien abgebaut werden.'),
		('5001', 'Gipsstein', 'Gipsstein ist ein natürliches Abbaugut.'),
		('5002', 'Anhydritstein', 'Anhydritstein ist ein Abbaugut, das aus wasserfreiem Gips besteht.'),
		('5003', 'Steinsalz', 'Steinsalz ist ein Abbaugut, das aus Salzstöcken gewonnen wird und aus Natriumchlorid besteht.'),
		('5004', 'Kalisalz', 'Kalisalz ist ein Abbaugut, das aus Salzstöcken gewonnen wird und aus Chloriden und Sulfaten besteht.'),
		('5005', 'Kalkspat', 'Kalkspat ist ein weißes oder hell gefärbtes Abbaugut (Calciumcarbonat).'),
		('5006', 'Flussspat', 'Flussspat ist ein Abbaugut, das aus Calciumfluorid besteht.'),
		('5007', 'Schwerspat', 'Schwerspat ist ein formenreiches, rhombisches weißes bis farbiges Abbaugut.'),
		('5011', 'Graphit', 'Graphit ist ein bleigraues, weiches, metallglänzendes Abbaugut, das aus fast reinem Kohlenstoff besteht.');
		CREATE TABLE ax_zustand_flaechegemischternutzung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_flaechegemischternutzung IS 'Alias: "AX_Zustand_FlaecheGemischterNutzung", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_flaechegemischternutzung (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', ''),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_zustand_industrieundgewerbeflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_industrieundgewerbeflaeche IS 'Alias: "AX_Zustand_IndustrieUndGewerbeflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_industrieundgewerbeflaeche (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich die Fläche nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_funktion_friedhof (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_friedhof IS 'Alias: "AX_Funktion_Friedhof", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_friedhof (wert,beschreibung,dokumentation) VALUES
		('9401', 'Gebäude- und Freifläche Friedhof', ''),
		('9402', 'Friedhof (ohne Gebäude)', ''),
		('9403', 'Friedhof (Park)', 'Friedhof (Park) ist ein Friedhof, der als Park angelegt ist.'),
		('9404', 'Historischer Friedhof', 'Historischer Friedhof ist ein Friedhof, der als historisch gilt.');
		CREATE TABLE ax_zustand_friedhof (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_friedhof IS 'Alias: "AX_Zustand_Friedhof", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_friedhof (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', ''),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_lagergut_halde (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_lagergut_halde IS 'Alias: "AX_Lagergut_Halde", UML-Typ: Enumeration';
		INSERT INTO ax_lagergut_halde (wert,beschreibung,dokumentation) VALUES
		('1000', 'Baustoffe', ''),
		('2000', 'Kohle', ''),
		('4000', 'Erde', ''),
		('5000', 'Schutt', ''),
		('6000', 'Schlacke', ''),
		('7000', 'Abraum', ''),
		('8000', 'Schrott, Altmaterial', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_funktion_industrieundgewerbeflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_industrieundgewerbeflaeche IS 'Alias: "AX_Funktion_IndustrieUndGewerbeflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_industrieundgewerbeflaeche (wert,beschreibung,dokumentation) VALUES
		('1400', 'Handel und Dienstleistung', 'Handel und Dienstleistung bezeichnet eine Fläche, auf der vorwiegend Gebäude stehen, in denen Handels- und/oder Dienstleistungsbetriebe ansässig sind.'),
		('1410', 'Verwaltung, freie Berufe', ''),
		('1420', 'Bank, Kredit', ''),
		('1430', 'Versicherung', ''),
		('1440', 'Handel', 'Handel bezeichnet Anlagen mit Einzelhandels- und Dienstleistungsbetrieben, die durch einheitliche 
Verwaltung, auf das Einzugsgebiet abgestimmter Anbieter und durch große Parkplatzflächen geprägt sind.'),
		('1450', 'Ausstellung, Messe', 'Ausstellung, Messe bezeichnet eine Fläche mit Ausstellungshallen und sonstigen Einrichtungen zur 
Präsentation von Warenmustern.'),
		('1460', 'Beherbergung', ''),
		('1470', 'Restauration', ''),
		('1480', 'Vergnügung', ''),
		('1490', 'Gärtnerei', 'Gärtnerei bezeichnet eine Fläche mit Gebäuden, Gewächshäusern und sonstigen Einrichtungen, zur Aufzucht von Blumen und Gemüsepflanzen.'),
		('1700', 'Industrie und Gewerbe', 'Industrie und Gewerbe bezeichnet Flächen, auf denen vorwiegend Industrie- und Gewerbebetriebe vorhanden sind.  Darin sind Gebäude- und Freiflächen und die Betriebsläche Lagerplatz enthalten.'),
		('1701', 'Gebäude- und Freifläche Industrie und Gewerbe', 'Darin sind die Gebäude- und Freiflächen der folgenden Differenzierung enthalten ohne die Betriebsfläche Lagerplatz.'),
		('1710', 'Produktion', ''),
		('1720', 'Handwerk', ''),
		('1730', 'Tankstelle', ''),
		('1740', 'Lagerplatz', 'Lagerplatz bezeichnet Flächen, auf denen inner- und außerhalb von Gebäuden wirtschaftliche Güter gelagert werden.'),
		('1750', 'Transport', ''),
		('1760', 'Forschung', ''),
		('1770', 'Grundstoff', ''),
		('1780', 'Betriebliche Sozialeinrichtung', ''),
		('1790', 'Werft', 'Werft ist eine Betriebsfläche mit Bauwerken und sonstigen Einrichtungen zum Bau oder zur Reparatur von Schiffen.'),
		('2500', 'Versorgungsanlage', 'Versorgungsanlage bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur Versorgung der Allgemeinheit mit Elektrizität, Wärme und Wasser vorhanden sind.'),
		('2501', 'Gebäude- und Freifläche Versorgungsanlage', 'Gebäude- und Freifläche Versorgungsanlage bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur Versorgung der Allgemeinheit mit Elektrizität, Wärme und Wasser vorhanden sind.'),
		('2502', 'Betriebsfläche Versorgungsanlage', 'Betriebsfläche Versorgungsanlage bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur Versorgung der Allgemeinheit mit Elektrizität, Wärme und Wasser vorhanden sind.'),
		('2510', 'Förderanlage', 'Förderanlage bezeichnet eine Fläche mit Einrichtungen zur Förderung von Erdöl, Erdgas, Sole, Kohlensäure oder Erdwärme aus dem Erdinneren.'),
		('2520', 'Wasserwerk', 'Wasserwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Gewinnung und/ oder zur Aufbereitung von (Trink-)wasser.'),
		('2521', 'Gebäude- und Freifläche Versorgungsanlage, Wasser', 'Gebäude- und Freifläche Versorgungsanlage, Wasser ist Teil von Wasserwerk. Wasserwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Gewinnung und/ oder zur Aufbereitung von (Trink-)wasser.'),
		('2522', 'Betriebsfläche Versorgungsanlage, Wasser', 'Betriebsfläche Versorgungsanlage, Wasser ist Teil von Wasserwerk. Wasserwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Gewinnung und/ oder zur Aufbereitung von (Trink-)wasser.'),
		('2530', 'Kraftwerk', 'Kraftwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Erzeugung von elektrischer Energie.'),
		('2531', 'Gebäude- und Freifläche Versorgungsanlage, Elektrizität', 'Gebäude- und Freifläche Versorgungsanlage, Elektrizität ist Teil von Kraftwerk. Kraftwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Erzeugung von elektrischer Energie.'),
		('2532', 'Betriebsfläche Versorgungsanlage, Elektrizität', 'Betriebsfläche Versorgungsanlage, Elektrizität ist Teil von Kraftwerk. Kraftwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Erzeu-gung von elektrischer Energie.'),
		('2540', 'Umspannstation', 'Umspannstation bezeichnet eine Fläche mit Gebäuden und sonstigen Einrichtungen, um Strom auf eine andere Spannungsebene zu transformieren.'),
		('2550', 'Raffinerie', 'Raffinerie bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Aufbereitung von Erdöl.'),
		('2551', 'Gebäude- und Freifläche Versorgungsanlage, Öl', 'Gebäude- und Freifläche Versorgungsanlage, Öl ist Teil von Raffinerie. Raffinerie bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Aufbereitung von Erdöl.'),
		('2552', 'Betriebsfläche Versorganlage, Öl', 'Betriebsfläche Versorgungsanlage, Öl ist Teil von Raffinerie. Raffinerie bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Aufbereitung von Erdöl.'),
		('2560', 'Gaswerk', ''),
		('2561', 'Gebäude- und Freifläche Versorgungsanlage, Gas', 'Gebäude- und Freifläche Versorgungsanlage, Gas ist Teil von Gaswerk. Gaswerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Aufbe-reitung von Gas.'),
		('2562', 'Betriebsfläche Versorgungsanlage, Gas', 'Betriebsfläche Versorgungsanlage, Gas ist Teil von Gaswerk. Gaswerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Aufbereitung von Gas.'),
		('2570', 'Heizwerk', 'Heizwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Erzeugung von Wärmeenergie zu Heizzwecken.'),
		('2571', 'Gebäude- und Freifläche Versorgungsanlage, Wärme', 'Gebäude- und Freifläche Versorgungsanlage, Wärme ist Teil von Heizwerk. Heizwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Er-zeugung von Wärmeenergie zu Heizzwecken.'),
		('2572', 'Betriebsfläche Versorgungsanlage, Wärme', 'Betriebsfläche Versorgungsanlage, Wärme ist Teil von Heizwerk. Heizwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Erzeugung von Wärmeenergie zu Heizzwecken.'),
		('2580', 'Funk- und Fernmeldeanlage', 'Funk- und Fernmeldeanlage bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur elektronischen Informationenvermittlung stehen.'),
		('2581', 'Gebäude- und Freifläche Versorgungsanlage, Funk- und Fernmeldewesen', 'Gebäude- und Freifläche Versorgungsanlage, Funk- und Fernmeldewesen ist Teil von Funk- und Fernmeldeanlage. Funk- und Fernmeldeanlage bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur elektronischen Informationenvermittlung stehen.'),
		('2582', 'Betriebsfläche Versorgungsanlage, Funk- und Fernmeldewesen', 'Betriebsfläche Versorgungsanlage, Funk- und Fernmeldewesen ist Teil von Funk- und Fernmeldeanlage. Funk- und Fernmeldeanlage bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur elektronischen Informationenvermittlung stehen.'),
		('2600', 'Entsorgung', 'Entsorgung bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur Verwertung und Entsorgung von Abwasser und festen Abfallstoffen vorhanden sind.'),
		('2601', 'Gebäude- und Freifläche Entsorgungsanlage', 'Gebäude- und Freifläche Entsorgungsanlage ist Teil von Entsorgung. Entsorgung bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur Verwertung und Entsorgung von Abwasser und festen Abfallstoffen vorhanden sind.'),
		('2602', 'Betriebsfläche Entsorgungsanlage', 'Betriebsfläche Entsorgungsanlage ist Teil von Entsorgung. Entsorgung bezeichnet eine Fläche, auf der vorwiegend Anlagen und Gebäude zur Verwertung und Entsorgung von Abwasser und festen Abfallstoffen vorhanden sind.'),
		('2610', 'Kläranlage, Klärwerk', 'Kläranlage, Klärwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Reinigung von Abwasser.'),
		('2611', 'Gebäude- und Freifläche Entsorgungsanlage, Abwasserbeseitigung', 'Gebäude- und Freifläche Entsorgungsanlage, Abwasserbeseitigung ist Teil von Kläranlage, Klärwerk. Kläranlage, Klärwerk bezeichnet eine Fläche mit Bauwer-ken und sonstigen Einrichtungen zur Reinigung von Abwasser.'),
		('2612', 'Betriebsfläche Entsorgungsanlage, Abwasserbeseitigung', 'Betriebsfläche Entsorgungsanlage, Abwasserbeseitigung ist Teil von Kläranlage, Klärwerk. Kläranlage, Klärwerk bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen zur Reinigung von Abwasser.'),
		('2620', 'Abfallbehandlungsanlage', 'Abfallbehandlungsanlage bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen, auf der Abfälle mit chemisch/physikalischen und biologischen oder thermischen Verfahren oder Kombinationen dieser Verfahren behandelt werden.'),
		('2621', 'Gebäude- und Freifläche Entsorgungsanlage, Abfallbeseitigung', 'Gebäude- und Freifläche Entsorgungsanlage, Abfallbeseitigung ist Teil von Abfallbehandlungsanlage. Abfallbehandlungsanlage bezeichnet eine Fläche mit Bau-werken und sonstigen Einrichtungen, auf der Abfälle mit chemisch/physikalischen und biologischen oder thermischen Verfahren oder Kombinationen dieser Ver-fahren behandelt werden.'),
		('2622', 'Betriebsfläche Entsorgungsanlage, Abfallbeseitigung', 'Betriebsfläche Entsorgungsanlage, Abfallbeseitigung ist Teil von Abfallbehandlungsanlage. Abfallbehandlungsanlage bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen, auf der Abfälle mit chemisch/physikalischen und biologischen oder thermischen Verfahren oder Kombinationen dieser Verfahren behan-delt werden.'),
		('2623', 'Betriebsfläche Entsorgungsanlage, Schlamm', 'Betriebsfläche Entsorgungsanlage, Schlamm ist Teil von Abfallbehandlungsanlage. Abfallbehandlungsanlage bezeichnet eine Fläche mit Bauwerken und sonstigen Einrichtungen, auf der Abfälle mit chemisch/physikalischen und biologischen oder thermischen Verfahren oder Kombinationen dieser Verfahren behandelt wer-den.'),
		('2630', 'Deponie (oberirdisch)', 'Deponie (oberirdisch) bezeichnet eine Fläche, auf der oberirdisch Abfallstoffe gelagert werden.'),
		('2640', 'Deponie (untertägig)', 'Deponie (untertägig) bezeichnet eine oberirdische Betriebsfläche, unter der  Abfallstoffe eingelagert werden (Untertagedeponie).');
		CREATE TABLE ax_zustand_tagebaugrubesteinbruch (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_tagebaugrubesteinbruch IS 'Alias: "AX_Zustand_TagebauGrubeSteinbruch", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_tagebaugrubesteinbruch (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich Tagebau, Grube, Steinbruch nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('8000', 'Erweiterung, Neuansiedlung', '');
		CREATE TABLE ax_artderbebauung_siedlungsflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderbebauung_siedlungsflaeche IS 'Alias: "AX_ArtDerBebauung_Siedlungsflaeche", UML-Typ: Enumeration';
		INSERT INTO ax_artderbebauung_siedlungsflaeche (wert,beschreibung,dokumentation) VALUES
		('1000', 'Offen', 'Offen beschreibt die Bebauung von Fläche gemischter Nutzung, die vorwiegend durch einzelstehende Gebäude charakterisiert wird.'),
		('2000', 'Geschlossen', 'Geschlossen beschreibt die Bebauung von Siedlungsfläche, die vorwiegend durch zusammenhängende Gebäude charakterisiert wird. Die Gebäudeabdeckung ist in der Regel  > 50 Prozent der Fläche.');
		CREATE TABLE ax_artderbebauung_flaechebesondererfunktionalerpraegung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_artderbebauung_flaechebesondererfunktionalerpraegung IS 'Alias: "AX_ArtDerBebauung_FlaecheBesondererFunktionalerPraegung", UML-Typ: Enumeration';
		INSERT INTO ax_artderbebauung_flaechebesondererfunktionalerpraegung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Offen', 'Offen beschreibt die Bebauung von Fläche besonderer funktionaler Prägung, die vorwiegend durch einzelstehende Gebäude charakterisiert wird.'),
		('2000', 'Geschlossen', 'Geschlossen beschreibt die Bebauung von Fläche besonderer funktionaler Prägung, die vorwiegend durch zusammenhängende Gebäude charakterisiert wird. Die Gebäudeabdeckung ist in der Regel > 50 Prozent der Fläche.');
		CREATE TABLE ax_vegetationsmerkmal_gehoelz (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_vegetationsmerkmal_gehoelz IS 'Alias: "AX_Vegetationsmerkmal_Gehoelz", UML-Typ: Enumeration';
		INSERT INTO ax_vegetationsmerkmal_gehoelz (wert,beschreibung,dokumentation) VALUES
		('1400', 'Latschenkiefer', '');
		CREATE TABLE ax_vegetationsmerkmal_wald (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_vegetationsmerkmal_wald IS 'Alias: "AX_Vegetationsmerkmal_Wald", UML-Typ: Enumeration';
		INSERT INTO ax_vegetationsmerkmal_wald (wert,beschreibung,dokumentation) VALUES
		('1100', 'Laubholz', 'Laubholz beschreibt den Bewuchs einer Vegetationsfläche mit Laubbäumen.'),
		('1200', 'Nadelholz', 'Nadelholz beschreibt den Bewuchs einer Vegetationsfläche mit Nadelbäumen.'),
		('1300', 'Laub- und Nadelholz', 'Laub- und Nadelholz beschreibt den Bewuchs einer Vegetationsfläche mit Laub- und Nadelbäumen.'),
		('1310', 'Laubwald mit Nadelholz', ''),
		('1320', 'Nadelwald mit Laubholz', '');
		CREATE TABLE ax_vegetationsmerkmal_landwirtschaft (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_vegetationsmerkmal_landwirtschaft IS 'Alias: "AX_Vegetationsmerkmal_Landwirtschaft", UML-Typ: Enumeration';
		INSERT INTO ax_vegetationsmerkmal_landwirtschaft (wert,beschreibung,dokumentation) VALUES
		('1010', 'Ackerland', 'Ackerland ist eine Fläche für den Anbau von Feldfrüchten (z.B. Getreide, Hülsenfrüchte, Hackfrüchte) und Beerenfrüchten (z.B. Erdbeeren). 
Zum Ackerland gehören auch die Rotationsbrachen, Dauerbrachen sowie Flächen, die zur Erlangung der Ausgleichszahlungen der EU stillgelegt worden sind.'),
		('1011', 'Streuobstacker', 'Streuobstacker beschreibt den Bewuchs einer Ackerfläche mit Obstbäumen.'),
		('1012', 'Hopfen', 'Hopfen ist eine mit speziellen Vorrichtungen ausgestattete Agrarfläche für den Anbau von Hopfen.'),
		('1013', 'Spargel', ''),
		('1020', 'Grünland', 'Grünland ist eine Grasfläche, die gemäht oder beweidet wird.'),
		('1021', 'Streuobstwiese', 'Streuobstwiese beschreibt den Bewuchs einer Grünlandfläche mit Obstbäumen.'),
		('1030', 'Gartenland', 'Gartenland ist eine Fläche für den Anbau von Gemüse, Obst und Blumen sowie für die Aufzucht von Kulturpflanzen.'),
		('1031', 'Baumschule', 'Baumschule ist eine Fläche, auf der Holzgewächse aus Samen, Ablegern oder Stecklingen unter mehrmaligem Umpflanzen (Verschulen) gezogen werden.'),
		('1040', 'Weingarten', 'Weingarten ist eine mit speziellen Vorrichtungen ausgestattete Agrarfläche auf der Weinstöcke angepflanzt sind.'),
		('1050', 'Obstplantage', 'Obstplantage ist eine landwirtschaftliche Fläche, die mit Obstbäumen und Obststräuchern bepflanzt ist'),
		('1051', 'Obstbaumplantage', 'Obstbaumplantage ist eine landwirtschaftliche Fläche, die ausschließlich mit Obstbäumen bepflanzt ist.'),
		('1052', 'Obststrauchplantage', ''),
		('1200', 'Brachland', 'Brachland ist eine Fläche der Landwirtschaft, die seit längerem nicht mehr zu Produktionszwecken genutzt wird.');
		CREATE TABLE ax_oberflaechenmaterial_unlandvegetationsloseflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_oberflaechenmaterial_unlandvegetationsloseflaeche IS 'Alias: "AX_Oberflaechenmaterial_UnlandVegetationsloseFlaeche", UML-Typ: Enumeration';
		INSERT INTO ax_oberflaechenmaterial_unlandvegetationsloseflaeche (wert,beschreibung,dokumentation) VALUES
		('1010', 'Fels', 'Fels bedeutet, dass die Erdoberfläche aus einer festen Gesteinsmasse besteht.'),
		('1020', 'Steine, Schotter', 'Steine, Schotter bedeutet, dass die Erdoberfläche mit zerkleinertem Gestein unterschiedlicher Größe bedeckt ist.'),
		('1030', 'Geröll', 'Geröll bedeutet, dass die Erdoberfläche mit durch fließendes Wasser abgerundeten Gesteinen bedeckt ist.'),
		('1040', 'Sand', 'Sand bedeutet, dass die Erdoberfläche mit kleinen, losen Gesteinskörnern bedeckt ist.'),
		('1110', 'Schnee', 'Schnee bedeutet, dass die Erdoberfläche für die größte Zeit des Jahres mit Schnee bedeckt ist.'),
		('1120', 'Eis, Firn', 'Eis, Firn bedeutet, dass die Erdoberfläche mit altem, grobkörnigem, mehrjährigem Schnee im Hochgebirge bedeckt ist, der unter zunehmendem Druck zu Gletschereis wird.');
		CREATE TABLE ax_funktion_unlandvegetationsloseflaeche (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_unlandvegetationsloseflaeche IS 'Alias: "AX_Funktion_UnlandVegetationsloseFlaeche", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_unlandvegetationsloseflaeche (wert,beschreibung,dokumentation) VALUES
		('1000', 'Vegetationslose Fläche', 'Vegetationslose Fläche ist eine Fläche ohne nennenswerten Bewuchs aufgrund besonderer Bodenbeschaffenheit.'),
		('1100', 'Gewässerbegleitfläche', 'Gewässerbegleitfläche bezeichnet eine bebaute oder unbebaute Fläche, die einem Gewässer zugeordnet wird. Die Gewässerbegleitfläche ist nicht Bestandteil der Gewässerfläche.'),
		('1110', 'Bebaute Gewässerbegleitfläche', ''),
		('1120', 'Unbebaute Gewässerbegleitfläche', ''),
		('1200', 'Sukzessionsfläche', 'Sukzessionsfläche ist eine Fläche, die dauerhaft aus der landwirtschaftlichen oder sonstigen bisherigen Nutzung herausgenommen ist und die in den Urzustand z. B. Gehölz, Moor, Heide übergeht.'),
		('1300', 'Naturnahe Fläche', 'Naturnahe Fläche ist eine nicht zum Anbau von Kulturpflanzen genutzte Fläche, die mit Gras, Wildkräutern und anderen Pflanzen bewachsen ist.');
		CREATE TABLE ax_funktion_gehoelz (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_gehoelz IS 'Alias: "AX_Funktion_Gehoelz", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_gehoelz (wert,beschreibung,dokumentation) VALUES
		('1000', 'Windschutz', '');
		CREATE TABLE ax_bahnkategorie (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_bahnkategorie IS 'Alias: "AX_Bahnkategorie", UML-Typ: Enumeration';
		INSERT INTO ax_bahnkategorie (wert,beschreibung,dokumentation) VALUES
		('1100', 'Eisenbahn', 'Eisenbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem im Nah- und Fernverkehr Personen befördert und Güter transportiert werden.'),
		('1102', 'Güterverkehr', 'Güterverkehr ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem im Nah- und Fernverkehr ausschließlich Güter transportiert werden.'),
		('1104', 'S-Bahn', 'S-Bahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, der zur schnellen Personenbeförderung in Ballungsräumen dient und meist auf eigenen Gleisen verläuft.'),
		('1200', 'Stadtbahn', 'Stadtbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem eine elektrisch betriebene Schienenbahn zur Personenbeförderung im öffentlichen Nahverkehr fährt. Sie kann sowohl ober- als auch unterirdisch verlaufen.'),
		('1201', 'Straßenbahn', 'Straßenbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem eine elektrisch betriebene Schienbahn zur Personenbeförderung fährt. Sie verläuft i. d. R. oberirdisch.'),
		('1202', 'U-Bahn', 'U-Bahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem eine elektrisch betriebene Schienenbahn zur Personenbeförderung in Großstädten fährt. Sie verläuft i. d. R. unterirdisch.'),
		('1300', 'Seilbahn, Bergbahn', 'Seilbahn, Bergbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem eine Schienenbahn große Höhenunterschiede überwindet.'),
		('1301', 'Zahnradbahn', 'Zahnradbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem eine Schienenbahn mittels Zahnradantrieb große Höhenunterschiede in stark geneigtem Gelände überwindet.'),
		('1302', 'Standseilbahn', 'Standseilbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem eine Schienenbahn auf einer stark geneigten, meist kurzen und geraden Strecke verläuft. Mit Hilfe eines oder mehrerer Zugseile wird ein Schienenfahrzeug bergauf gezogen und gleichzeitig ein zweites bergab gelassen.'),
		('1400', 'Museumsbahn', 'Museumsbahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem ausschließlich Touristen in alten, meist restaurierten Zügen befördert werden.'),
		('1500', 'Bahn im Freizeitpark', 'Bahn im Freizeitpark ist die Bezeichnung für einen schienengebundenen Verkehrsweg innerhalb eines Freizeitparks.'),
		('1600', 'Magnetschwebebahn', 'Magnetschwebebahn ist die Bezeichnung für einen schienengebundenen Verkehrsweg, auf dem  räderlose Schienenfahrzeuge mit Hilfe von Magnetfeldern an oder auf einer Fahrschiene schwebend entlanggeführt werden.');
		CREATE TABLE ax_funktion_weg (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_weg IS 'Alias: "AX_Funktion_Weg", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_weg (wert,beschreibung,dokumentation) VALUES
		('5210', 'Fahrweg', ''),
		('5211', 'Hauptwirtschaftsweg', ''),
		('5212', 'Wirtschaftsweg', ''),
		('5220', 'Fußweg', 'Fußweg ist ein Weg, der auf Grund seines Ausbauzustandes nur von Fußgängern zu begehen ist.'),
		('5230', 'Gang', ''),
		('5240', 'Radweg', 'Radweg ist ein Weg, der als besonders gekennzeichneter und abgegrenzter Teil einer Straße oder mit selbständiger Linienführung für den Fahrradverkehr bestimmt ist.'),
		('5250', 'Rad- und Fußweg', 'Rad- und Fußweg ist ein Weg, der als besonders gekennzeichneter und abgegrenzter Teil einer Straße oder mit selbständiger Linienführung ausschließlich für den Fahrrad- und Fußgängerverkehr bestimmt ist.'),
		('5260', 'Reitweg', ''),
		('9999', 'Sonstiges', '');
		CREATE TABLE ax_funktion_bahnverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_bahnverkehr IS 'Alias: "AX_Funktion_Bahnverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_bahnverkehr (wert,beschreibung,dokumentation) VALUES
		('2321', 'Gebäude- und Freifläche zu Verkehrsanlagen, Schiene', 'Gebäude- und Freifläche zu Verkehrsanlage, Schiene dient der Abwicklung und Sicherheit des Verkehrs sowie der Unterhaltung der Verkehrsfläche'),
		('2322', 'Verkehrsbegleitfläche Bahnverkehr', 'Verkehrsbegleitfläche Bahnverkehr bezeichnet eine bebaute oder unbebaute, an den Bahnkörper 
angrenzende Fläche, die dem Schienenverkehr dient.');
		CREATE TABLE ax_verkehrsbedeutunginneroertlich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_verkehrsbedeutunginneroertlich IS 'Alias: "AX_VerkehrsbedeutungInneroertlich", UML-Typ: Enumeration';
		INSERT INTO ax_verkehrsbedeutunginneroertlich (wert,beschreibung,dokumentation) VALUES
		('1000', 'Durchgangsverkehr', 'Durchgangsverkehr beschreibt den tatsächlich stattfindenden Verkehr auf einer innerörtlichen Straße (Durchgangsstraße), unabhängig von gesetzlichen Festlegungen (z.B. Landesstraßengesetz) Als Durchgangsstraße wird die Straße bezeichnet, auf der aufgrund des Ausbauzustandes und der örtlichen Verkehrsregelung der überörtliche Verkehr durch ein bebautes Gebiet geleitet wird. Der Durchgangsverkehr kann sowohl auf Gemeindestraßen als auch auf höherwertig klassifizierten Straßen liegen.'),
		('2000', 'Ortsverkehr', 'Ortsverkehr beschreibt den tatsächlich stattfindenden Verkehr auf einer innerörtlichen Straße (Ortsstraße), unabhängig von örtlichen Festlegungen (z.B. Ortssatzungen). Ortsstraße ist in der Regel eine als Gemeindestraße gewidmete Straße, auf der kein Durchgangsverkehr verläuft.'),
		('2001', 'Sammelverkehr', 'Sammelverkehr beschreibt den tatsächlich stattfindenden Verkehr auf einer innerörtlichen Straße (Sammelstraße), unabhängig von örtlichen Festlegungen (z.B. Ortssatzungen). Die Sammelstraße leitet hauptsächlich den innerörtlichen Verkehr von den Anliegerstrassen zur Durchgangsstraße.'),
		('2002', 'Anliegerverkehr', 'Anliegerverkehr beschreibt den tatsächlich stattfindenden Verkehr auf einer innerörtlichen Straße (Anliegerstraße), unabhängig von örtlichen Festlegungen (z.B. Ortssatzungen). Die Anliegerstraße ist eine Straße auf die jeder Straßenanlieger von seinem Anwesen aus freie Zufahrt hat und die nicht die Funktion einer Sammelstraße übernimmt.');
		CREATE TABLE ax_internationalebedeutung_strasse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_internationalebedeutung_strasse IS 'Alias: "AX_InternationaleBedeutung_Strasse", UML-Typ: Enumeration';
		INSERT INTO ax_internationalebedeutung_strasse (wert,beschreibung,dokumentation) VALUES
		('2001', 'Europastraße', 'Europastraßen sind Abschnitte von Bundesfernstraßen, die einen Teil des von der Europäischen Konferenz der Verkehrsminister beschlossenen internationalen europäischen Straßennetzes bilden.');
		CREATE TABLE ax_besonderefahrstreifen (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_besonderefahrstreifen IS 'Alias: "AX_BesondereFahrstreifen", UML-Typ: Enumeration';
		INSERT INTO ax_besonderefahrstreifen (wert,beschreibung,dokumentation) VALUES
		('1000', 'Mit Radweg', 'Mit Radweg bedeutet, dass parallel zur <Objektart> ein Radweg verläuft, der aber nicht als eigenständiges Objekt erfasst wird.'),
		('2000', 'Mit Fußweg', 'Mit Fußweg bedeutet, dass parallel zur <Objektart> ein Fußweg verläuft, der aber nicht als eigenständiges Objekt erfasst wird.'),
		('3000', 'Mit Rad- und Fußweg', 'Mit Rad- und Fußweg bedeutet, dass parallel zur <Objektart> ein Rad- und Fußweg verläuft, der aber nicht als eigenständiges Objekt erfasst wird.');
		CREATE TABLE ax_zustand_bahnverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_bahnverkehr IS 'Alias: "AX_Zustand_Bahnverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_bahnverkehr (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', ''),
		('4000', 'Im Bau', '');
		CREATE TABLE ax_befestigung_fahrwegachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_befestigung_fahrwegachse IS 'Alias: "AX_Befestigung_Fahrwegachse", UML-Typ: Enumeration';
		INSERT INTO ax_befestigung_fahrwegachse (wert,beschreibung,dokumentation) VALUES
		('1000', 'Befestigt', 'Befestigt bedeutet, dass Fahrwegachse mit einem festen Unterbau versehen und ganzjährig befahrbar bzw. begehbar ist.'),
		('2000', 'Unbefestigt', 'Unbefestigt bedeutet, dass Fahrwegachse nicht mit einem festen Unterbau versehen und nicht ganzjährig befahrbar bzw. begehbar ist.');
		CREATE TABLE ax_spurweite (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_spurweite IS 'Alias: "AX_Spurweite", UML-Typ: Enumeration';
		INSERT INTO ax_spurweite (wert,beschreibung,dokumentation) VALUES
		('1000', 'Normalspur (Regelspur, Vollspur)', 'Normalspur (Regelspur, Vollspur) hat eine Spurweite von 1435 mm. Das ist das Innenmaß zwischen den Innenkanten der Schienenköpfe eines Gleises.'),
		('2000', 'Schmalspur', 'Schmalspur ist eine Spurweite, die kleiner ist als 1435 mm.'),
		('3000', 'Breitspur', 'Breitspur ist eine Spurweite, die größer ist als 1435 mm.');
		CREATE TABLE ax_zustand_schiffsverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_schiffsverkehr IS 'Alias: "AX_Zustand_Schiffsverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_schiffsverkehr (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich Schiffsverkehr nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('4000', 'Im Bau', 'Im Bau bedeutet, dass Schiffsverkehr noch nicht fertiggestellt ist.');
		CREATE TABLE ax_funktion_platz (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_platz IS 'Alias: "AX_Funktion_Platz", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_platz (wert,beschreibung,dokumentation) VALUES
		('5130', 'Fußgängerzone', 'Fußgängerzone ist ein dem Fußgängerverkehr vorbehaltener Bereich, in dem ausnahmsweise öffentlicher Personenverkehr, Lieferverkehr oder Fahrradverkehr zulässig sein kann.'),
		('5310', 'Parkplatz', 'Parkplatz ist eine zum vorübergehenden Abstellen von Fahrzeugen bestimmte Fläche.'),
		('5320', 'Rastplatz', 'Rastplatz ist eine Anlage zum Halten, Parken oder Rasten der Verkehrsteilnehmer mit unmittelbarem Anschluss zur Straße ohne Versorgungseinrichtung, ggf. mit Toiletten.'),
		('5330', 'Raststätte', 'Raststätte ist eine Anlage an Verkehrsstraßen mit Bauwerken und Einrichtungen zur Versorgung und 
Erholung von Reisenden.'),
		('5340', 'Marktplatz', ''),
		('5350', 'Festplatz', 'Festplatz ist eine Fläche, auf der zeitlich begrenzte Festveranstaltungen stattfinden.');
		CREATE TABLE ax_art_flugverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_art_flugverkehr IS 'Alias: "AX_Art_Flugverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_art_flugverkehr (wert,beschreibung,dokumentation) VALUES
		('5510', 'Flughafen', 'Flughafen ist eine Anlage mit Gebäuden, Bauwerken, Start- und Landebahnen sowie sonstigen  flugtechnischen Einrichtungen zur Abwicklung des Flugverkehrs.'),
		('5511', 'Internationaler Flughafen', 'Internationaler Flughafen ist ein Flughafen, der in der Luftfahrtkarte 1 : 500000 (ICAO) als solcher ausgewiesen ist.'),
		('5512', 'Regionalflughafen', 'Regionalflughafen ist ein Flughafen der gemäß Raumordnungsgesetz als Regionalflughafen eingestuft ist.'),
		('5520', 'Verkehrslandeplatz', 'Verkehrslandeplatz ist ein Flugplatz, der in der Luftfahrtkarte 1:500000 (ICAO) als solcher ausgewiesen ist.'),
		('5530', 'Hubschrauberflugplatz', 'Hubschrauberflugplatz ist ein Flugplatz, der in der Luftfahrtkarte 1:500000 (ICAO) als solcher ausgewiesen ist.'),
		('5540', 'Landeplatz, Sonderlandeplatz', 'Landeplatz, Sonderlandeplatz ist eine Fläche, die in der Luftfahrtkarte 1:500000 (ICAO) als Landeplatz, Sonderlandeplatz ausgewiesen ist.'),
		('5550', 'Segelfluggelände', 'Segelfluggelände ist eine Fläche, die in der Luftfahrtkarte 1:500000 (ICAO) als Segelfluggelände ausgewiesen ist.');
		CREATE TABLE ax_elektrifizierung (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_elektrifizierung IS 'Alias: "AX_Elektrifizierung", UML-Typ: Enumeration';
		INSERT INTO ax_elektrifizierung (wert,beschreibung,dokumentation) VALUES
		('1000', 'Elektrifiziert', 'Elektrifiziert bedeutet, dass den Schienenfahrzeugen über eine Oberleitung oder eine Stromschiene längs des Fahrweges elektrische Energie zugeführt wird.'),
		('2000', 'Nicht elektrifiziert', 'Nicht elektrifiziert bedeutet, dass die Schienenfahrzeuge ohne elektrische Energie angetrieben werden.');
		CREATE TABLE ax_zustand (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand IS 'Alias: "AX_Zustand", UML-Typ: Enumeration';
		INSERT INTO ax_zustand (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt, verlassen bedeutet, dass sich <Objektart> nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('4000', 'Im Bau', 'Im Bau bedeutet, dass <Objektart> noch nicht fertiggestellt ist.');
		CREATE TABLE ax_fahrbahntrennung_strasse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_fahrbahntrennung_strasse IS 'Alias: "AX_Fahrbahntrennung_Strasse", UML-Typ: Enumeration';
		INSERT INTO ax_fahrbahntrennung_strasse (wert,beschreibung,dokumentation) VALUES
		('2000', 'Getrennt', 'Getrennt bedeutet, dass sich ein Grünstreifen, eine Leitplanke oder ein sonstiges Hindernis zwischen zwei Fahrbahnen befindet.');
		CREATE TABLE ax_funktion_fahrbahnachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_fahrbahnachse IS 'Alias: "AX_Funktion_Fahrbahnachse", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_fahrbahnachse (wert,beschreibung,dokumentation) VALUES
		('1808', 'Fußgängerzone', 'Fußgängerzone ist ein dem Fußgängerverkehr vorbehaltener Bereich, in dem ausnahmsweise öffentlicher Personenverkehr, Lieferverkehr oder Fahrradverkehr zulässig sein kann.');
		CREATE TABLE ax_oberflaechenmaterial_strasse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_oberflaechenmaterial_strasse IS 'Alias: "AX_Oberflaechenmaterial_Strasse", UML-Typ: Enumeration';
		INSERT INTO ax_oberflaechenmaterial_strasse (wert,beschreibung,dokumentation) VALUES
		('1220', 'Beton', 'Beton bedeutet, dass die Oberfläche von <Objektart> aus Beton besteht.'),
		('1230', 'Bitumen, Asphalt', 'Bitumen, Asphalt bedeutet, dass die Oberfläche von <Objektart> aus Bitumen bzw. Asphalt besteht.'),
		('1240', 'Pflaster', 'Pflaster bedeutet, dass die Oberfläche von <Objektart> gepflastert ist.'),
		('1250', 'Gestein, zerkleinert', 'Gestein, zerkleinert bedeutet, dass die Oberfläche von <Objektart> aus Schotter, Splitt, Sand oder aus einem Gemisch dieser Materialen besteht.');
		CREATE TABLE ax_funktion_flugverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_flugverkehr IS 'Alias: "AX_Funktion_Flugverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_flugverkehr (wert,beschreibung,dokumentation) VALUES
		('5501', 'Gebäude- und Freifläche zu Verkehrsanlagen, Luftfahrt', 'Gebäude- und Freifläche zu Verkehrsanlagen, Luftfahrt ist eine besondere Flugverkehrsfläche.');
		CREATE TABLE ax_funktion_wegachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_wegachse IS 'Alias: "AX_Funktion_Wegachse", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_wegachse (wert,beschreibung,dokumentation) VALUES
		('5211', 'Hauptwirtschaftsweg', 'Hauptwirtschaftsweg ist ein Weg mit fester Fahrbahndecke zur Erschließung eines oder mehrerer Grundstücke, der für den Kraftverkehr zu jeder Jahreszeit befahrbar ist.'),
		('5212', 'Wirtschaftsweg', 'Wirtschaftsweg ist ein leicht- oder unbefestigter Weg zur Erschließung  land- und forstwirtschaftlicher Flächen.');
		CREATE TABLE ax_zustand_strasse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_strasse IS 'Alias: "AX_Zustand_Strasse", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_strasse (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', ''),
		('4000', 'Im Bau', '');
		CREATE TABLE ax_markierung_wegachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_markierung_wegachse IS 'Alias: "AX_Markierung_Wegachse", UML-Typ: Enumeration';
		INSERT INTO ax_markierung_wegachse (wert,beschreibung,dokumentation) VALUES
		('1401', 'Gekennzeichneter Wanderweg', 'Gekennzeichneter Wanderweg ist ein als Wanderweg eindeutig markierter Weg.'),
		('1402', 'Gekennzeichneter Rad(wander)weg', 'Gekennzeichneter Rad(wander)weg ist ein als Rad(wander)weg eindeutig markierter Weg.'),
		('1403', 'Gekennzeichnete Skaterstrecke', 'Gekennzeichnete Skaterstrecke ist ein als Skaterstrecke eindeutig markierter Weg.');
		CREATE TABLE ax_zustand_flugverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_zustand_flugverkehr IS 'Alias: "AX_Zustand_Flugverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_zustand_flugverkehr (wert,beschreibung,dokumentation) VALUES
		('2100', 'Außer Betrieb, stillgelegt, verlassen', 'Außer Betrieb, stillgelegt; verlassen bedeutet, dass sich Flugverkehr nicht mehr in regelmäßiger, der Bestimmung entsprechenden Nutzung befindet.'),
		('4000', 'Im Bau', 'Im Bau bedeutet, dass Flugverkehr noch nicht fertiggestellt ist.');
		CREATE TABLE ax_funktion_strassenachse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_strassenachse IS 'Alias: "AX_Funktion_Strassenachse", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_strassenachse (wert,beschreibung,dokumentation) VALUES
		('1808', 'Fußgängerzone', 'Fußgängerzone ist ein dem Fußgängerverkehr vorbehaltener Bereich, in dem ausnahmsweise öffentlicher Personenverkehr, Lieferverkehr oder Fahrradverkehr zulässig sein kann.');
		CREATE TABLE ax_verkehrsbedeutungueberoertlich (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_verkehrsbedeutungueberoertlich IS 'Alias: "AX_VerkehrsbedeutungUeberoertlich", UML-Typ: Enumeration';
		INSERT INTO ax_verkehrsbedeutungueberoertlich (wert,beschreibung,dokumentation) VALUES
		('1000', 'Überörtlicher Verkehr', 'Überörtlicher Verkehr umfasst die Verkehrswege des Fern-, Regional- und Nahverkehrs.'),
		('1001', 'Fernverkehr', 'Fernverkehr findet hauptsächlich auf dem zusammenhängenden Verkehrsnetz der Bundesautobahnen und Bundesstraßen statt.'),
		('1002', 'Regionalverkehr', 'Regionalverkehr findet hauptsächlich auf Landes- und Kreisstraßen statt.'),
		('1003', 'Nahverkehr, zwischenörtlicher Verkehr', 'Nahverkehr, zwischenörtlicher Verkehr findet hauptsächlich auf Kreis- und Gemeindestraßen statt.');
		CREATE TABLE ax_nutzung_flugverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_nutzung_flugverkehr IS 'Alias: "AX_Nutzung_Flugverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_nutzung_flugverkehr (wert,beschreibung,dokumentation) VALUES
		('1000', 'Zivil', 'Zivil bedeutet, dass Flugverkehr privaten, öffentlichen oder religiösen Zwecken dient und nicht militärisch genutzt wird.'),
		('2000', 'Militärisch', 'Militärisch bedeutet, dass Flugverkehr nur von Streitkräften genutzt wird.'),
		('3000', 'Teils zivil, teils militärisch', 'Teils zivil, teils militärisch bedeutet dass Flugverkehr sowohl zivil als auch militärisch genutzt wird.');
		CREATE TABLE ax_funktion_schiffsverkehr (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_schiffsverkehr IS 'Alias: "AX_Funktion_Schiffsverkehr", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_schiffsverkehr (wert,beschreibung,dokumentation) VALUES
		('2341', 'Gebäude und Freifläche zu Verkehrsanlagen, Schifffahrt', 'Gebäude- und Freifläche zu Verkehrsanlagen, Schifffahrt ist eine Fläche, die dem Schiffsverkehr dient.'),
		('5610', 'Hafenanlage (Landfläche)', 'Hafenanlage (Landfläche) bezeichnet die Fläche innerhalb von Hafen, die nicht von Wasser bedeckt ist und die ausschließlich zum Betrieb des Hafens dient.'),
		('5620', 'Schleuse (Landfläche)', 'Schleuse (Landfläche) bezeichnet die Fläche innerhalb von Schleuse, die nicht von Wasser bedeckt ist und die ausschließlich zum Betrieb der Schleuse dient..'),
		('5630', 'Anlegestelle', 'Anlegestelle umfasst mehr als den überlagernden landseitigen Anleger, der eine feste oder schwimmende Einrichtung zum Anlegen von Schiffen ist.'),
		('5640', 'Fähranlage', 'Fähranlage ist eine besondere Landfläche von der in der Regel nach festem Fahrplan über Flüsse, Seen, Kanäle, Meerengen oder Meeresarme ein Schiffsverkehr stattfindet.');
		CREATE TABLE ax_funktion_strasse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktion_strasse IS 'Alias: "AX_Funktion_Strasse", UML-Typ: Enumeration';
		INSERT INTO ax_funktion_strasse (wert,beschreibung,dokumentation) VALUES
		('2311', 'Gebäude- und Freifläche zu Verkehrsanlagen, Straße', 'Gebäude- und Freifläche zu Verkehrsanlagen, Straße ist eine Fläche. die der Abwicklung und Sicherheit des Verkehrs sowie der Unterhaltung der Verkehrsfläche dient.'),
		('2312', 'Verkehrsbegleitfläche Straße', 'Verkehrsbegleitfläche Straße bezeichnet eine bebaute oder unbebaute Fläche, die einer Straße zugeordnet wird. Die Verkehrsbegleitfläche Straße ist nicht Bestandteil der Fahrbahn.'),
		('2313', 'Straßenentwässerungsanlage', ''),
		('5130', 'Fußgängerzone', 'Fußgängerzone ist ein dem Fußgängerverkehr vorbehaltener Bereich, in dem ausnahmsweise öffentlicher Personenverkehr, Lieferverkehr oder Fahrradverkehr zulässig sein kann.');
		CREATE TABLE ax_widmung_strasse (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_widmung_strasse IS 'Alias: "AX_Widmung_Strasse", UML-Typ: Enumeration';
		INSERT INTO ax_widmung_strasse (wert,beschreibung,dokumentation) VALUES
		('1301', 'Bundesautobahn', 'Bundesautobahn ist eine durch Verwaltungsakt zur Bundesautobahn gewidmete Bundesfernstraße.'),
		('1303', 'Bundesstraße', 'Bundesstraße ist eine durch Verwaltungsakt zur Bundesstraße gewidmete Bundesfernstraße.'),
		('1305', 'Landesstraße, Staatsstraße', 'Landesstraße, Staatsstraße ist eine durch Verwaltungsakt zur Landesstraße bzw. Staatsstraße gewidmete Straße.'),
		('1306', 'Kreisstraße', 'Kreisstraße ist eine durch Verwaltungsakt zur Kreisstraße gewidmete Straße.'),
		('1307', 'Gemeindestraße', 'Gemeindestraße ist eine durch Verwaltungsakt zur Gemeindestrasse gewidmete Straße.'),
		('9997', 'Attribut trifft nicht zu', 'Attribut trifft nicht zu bedeutet, dass keiner der in der Werteliste aufgeführten Attributwerte dem vorliegenden Sachverhalt entspricht.'),
		('9999', 'Sonstiges', 'Sonstiges bedeutet, dass die Straße eine Widmung aufweist, die bekannt, aber nicht in der Attributwertliste aufgeführt ist.');
		CREATE TABLE ax_anzahlderstreckengleise (
		  wert integer,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_anzahlderstreckengleise IS 'Alias: "AX_AnzahlDerStreckengleise", UML-Typ: Enumeration';
		INSERT INTO ax_anzahlderstreckengleise (wert,beschreibung,dokumentation) VALUES
		('1000', 'Eingleisig', 'Eingleisig bedeutet, dass für Bahnstrecke nur ein Gleis für beide Fahrtrichtungen zur Verfügung steht.'),
		('2000', 'Zweigleisig', 'Zweigleisig bedeutet, dass für Bahnstrecke je ein Gleis für eine Fahrtrichtung zur Verfügung steht.');
		CREATE TABLE ax_funktionoa_k_tngr_all (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktionoa_k_tngr_all IS 'Alias: "AX_FunktionOA_K_TNGR_all", UML-Typ: Enumeration';
		INSERT INTO ax_funktionoa_k_tngr_all (wert,beschreibung,dokumentation) VALUES
		('Wohnbaufläche', NULL, ''),
		('Industrie- und Gewerbefläche', NULL, ''),
		('Straßenverkehr', NULL, ''),
		('Gehölz', NULL, ''),
		('Wald', NULL, ''),
		('Friedhof', NULL, ''),
		('Halde', NULL, ''),
		('Moor', NULL, ''),
		('Schiffsverkehr', NULL, ''),
		('Sport-, Freizeit- und Erholungsfläche', NULL, ''),
		('Bergbaubetrieb', NULL, ''),
		('Hafenbecken', NULL, ''),
		('Platz', NULL, ''),
		('Flugverkehr', NULL, ''),
		('Heide', NULL, ''),
		('Fließgewässer', NULL, ''),
		('Fläche besonderer funktionaler Prägung', NULL, ''),
		('Meer', NULL, ''),
		('Landwirtschaft', NULL, ''),
		('Tagebau, Grube, Steinbruch', NULL, ''),
		('Sumpf', NULL, ''),
		('Bahnverkehr', NULL, ''),
		('UnlandVegetationslose Fläche', NULL, ''),
		('Weg', NULL, ''),
		('Stehendes Gewässer', NULL, ''),
		('Fläche gemischter Nutzung', NULL, '');
		CREATE TABLE ax_klassifizierunggr_k_bewgr (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_klassifizierunggr_k_bewgr IS 'Alias: "AX_KlassifizierungGr_K_BEWGR", UML-Typ: Enumeration';
		INSERT INTO ax_klassifizierunggr_k_bewgr (wert,beschreibung,dokumentation) VALUES
		('Forstwirtschaftliche Nutzung', NULL, ''),
		('Weinbauliche Nutzung', NULL, ''),
		('Landwirtschaftliche Nutzung', NULL, ''),
		('Gärtnerische Nutzung', NULL, ''),
		('Nebenflächen des Betriebs der Land- und Forstwirtschaft', NULL, ''),
		('Andere Nutzung', NULL, ''),
		('Noch nicht klassifiziert', NULL, ''),
		('Waldflächen', NULL, ''),
		('Gewässerflächen', NULL, ''),
		('Straßenflächen', NULL, ''),
		('Sonstige Land- und forstwirtschaftliche Nutzung', NULL, '');
		CREATE TABLE ax_funktionoa_k_tnfl (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktionoa_k_tnfl IS 'Alias: "AX_FunktionOA_K_TNFL", UML-Typ: Enumeration';
		INSERT INTO ax_funktionoa_k_tnfl (wert,beschreibung,dokumentation) VALUES
		('Fließgewässer', NULL, ''),
		('Wald', NULL, ''),
		('Hafenbecken', NULL, ''),
		('Versorgungsanlage', NULL, ''),
		('Fläche gemischter Nutzung', NULL, ''),
		('Flugverkehr', NULL, ''),
		('Heide', NULL, ''),
		('Wohnbaufläche', NULL, ''),
		('Sport-, Freizeit- und Erholungsfläche', NULL, ''),
		('Industrie- und Gewerbefläche', NULL, ''),
		('Meer', NULL, ''),
		('Friedhof', NULL, ''),
		('Grünanlage', NULL, ''),
		('Straßenverkehr', NULL, ''),
		('Bahnverkehr', NULL, ''),
		('Entsorgung', NULL, ''),
		('Landwirtschaft', NULL, ''),
		('Fläche besonderer funktionaler Prägung', NULL, ''),
		('Halde', NULL, ''),
		('Sumpf', NULL, ''),
		('Weg', NULL, ''),
		('Gehölz', NULL, ''),
		('Bergbaubetrieb', NULL, ''),
		('StehendesGewässer', NULL, ''),
		('UnlandVegetationslose Fläche', NULL, ''),
		('Platz', NULL, ''),
		('Industrie und Gewerbe', NULL, ''),
		('Tagebau, Grube, Steinbruch', NULL, ''),
		('Moor', NULL, ''),
		('Schiffsverkehr', NULL, ''),
		('Handel und Dienstleistungen', NULL, '');
		CREATE TABLE ax_klassifizierungobg_k_bewfl (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_klassifizierungobg_k_bewfl IS 'Alias: "AX_KlassifizierungOBG_K_BEWFL", UML-Typ: Enumeration';
		INSERT INTO ax_klassifizierungobg_k_bewfl (wert,beschreibung,dokumentation) VALUES
		('Grünland-Acker', NULL, ''),
		('Hutung', NULL, ''),
		('Weingarten 4', NULL, ''),
		('Unland', NULL, ''),
		('Baumschule', NULL, ''),
		('Hopfen', NULL, ''),
		('Obstplantage', NULL, ''),
		('Ackerland-Grünland', NULL, ''),
		('Holzung', NULL, ''),
		('Wiese', NULL, ''),
		('Weihnachtsbaumkultur', NULL, ''),
		('Gartenland', NULL, ''),
		('Saatzucht', NULL, ''),
		('Nebenflächen des Betriebs der Land- und Forstwirtschaft', NULL, ''),
		('Anbaufläche unter Glas', NULL, ''),
		('Weingarten (allgemein)', NULL, ''),
		('Weingarten 1', NULL, ''),
		('Weingarten 2', NULL, ''),
		('Ackerland', NULL, ''),
		('Spargel', NULL, ''),
		('Grünland-Hackrain', NULL, ''),
		('Teichwirtschaft', NULL, ''),
		('Weingarten 6', NULL, ''),
		('Geringstland', NULL, ''),
		('Kleingarten', NULL, ''),
		('Acherland-Hackrain', NULL, ''),
		('Grünland', NULL, ''),
		('Weingarten 5', NULL, ''),
		('Weingarten 7', NULL, ''),
		('Weingarten 3', NULL, ''),
		('Abbauland der Land- und Forstwirtschaft', NULL, ''),
		('Streuwiese', NULL, '');
		CREATE TABLE ax_funktionoa_k_tngrerweitert_all (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktionoa_k_tngrerweitert_all IS 'Alias: "AX_FunktionOA_K_TNGRerweitert_all", UML-Typ: Enumeration';
		INSERT INTO ax_funktionoa_k_tngrerweitert_all (wert,beschreibung,dokumentation) VALUES
		('Schleuse (Landfläche)', NULL, ''),
		('Obstbaumplantage', NULL, ''),
		('Campingplatz', NULL, ''),
		('Straßenverkehr', NULL, ''),
		('Betriebsfläche Entsorgungsanlage', NULL, ''),
		('Handel', NULL, ''),
		('Friedhof (ohne Geb äude)', NULL, ''),
		('Freizeitpark', NULL, ''),
		('Verkehrsübungsplatz', NULL, ''),
		('Freizeitanlage', NULL, ''),
		('Gebäude- und Freifläche Sport, Freizeit und Erholung', NULL, ''),
		('Reitweg', NULL, ''),
		('Hopfen', NULL, ''),
		('Soziales', NULL, ''),
		('Flugverkehr', NULL, ''),
		('Gebäude- und Freifläche Erholung, Camping', NULL, ''),
		('Sonstiges (Sport-, Freizeit- und Erholungsfläche)', NULL, ''),
		('Betriebsfläche Entsorgungsanlage, Abwasserbeseitigung', NULL, ''),
		('Kleingarten', NULL, ''),
		('Rastplatz', NULL, ''),
		('Anlegestelle', NULL, ''),
		('Historischer Friedhof', NULL, ''),
		('Tennisplatz', NULL, ''),
		('Grünland', NULL, ''),
		('Burg-, Festungsanlage', NULL, ''),
		('Sonstiges (Weg)', NULL, ''),
		('Nadelwald mit Laubholz', NULL, ''),
		('Gebäude- und Freifläche zu Verkehrsanlagen, Straße', NULL, ''),
		('Autokino, Freilichtkino', NULL, ''),
		('Bergbaubetrieb', NULL, ''),
		('Kläranlage, Klärwerk', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage, Elektrizität', NULL, ''),
		('Wohnen und Betrieb', NULL, ''),
		('Versicherung', NULL, ''),
		('Radweg', NULL, ''),
		('Betrieb', NULL, ''),
		('Deponie (untertägig)', NULL, ''),
		('Raststätte', NULL, ''),
		('Kultur', NULL, ''),
		('See', NULL, ''),
		('Beherbergung', NULL, ''),
		('Betriebsfläche Entsorgungsanlage, Schlamm', NULL, ''),
		('Obststrauchplantage', NULL, ''),
		('Transport', NULL, ''),
		('Abfallbehandlungsanlage', NULL, ''),
		('Versorgungsanlage', NULL, ''),
		('Betriebsfläche Versorgungsanlage', NULL, ''),
		('Golfplatz', NULL, ''),
		('Weingarten', NULL, ''),
		('Windschutz', NULL, ''),
		('Gebäude- und Freifläche Land- und Forstwirtschaft', NULL, ''),
		('Festplatz', NULL, ''),
		('Lagerplatz', NULL, ''),
		('Spargel', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage, Wasser', NULL, ''),
		('Gebäude- und Freifläche Entsorgungsanlage, Abwasserbeseitigung', NULL, ''),
		('Hafenbecken', NULL, ''),
		('Gebäude- und Freifläche Erholung, Botanik', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage', NULL, ''),
		('Wasserwerk', NULL, ''),
		('Sportplatz', NULL, ''),
		('Ackerland', NULL, ''),
		('Funk- und Fernmeldeanlage', NULL, ''),
		('Obstplantage', NULL, ''),
		('Grundstoff', NULL, ''),
		('Fußweg', NULL, ''),
		('Entsorgung', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage, Öl', NULL, ''),
		('Stausee', NULL, ''),
		('Gebäude- und Freifläche Erholung, Zoologie', NULL, ''),
		('Fluss', NULL, ''),
		('Tagebau, Grube, Steinbruch', NULL, ''),
		('Förderanlage', NULL, ''),
		('Gebäude- und Freifläche Erholung', NULL, ''),
		('Hauptwirtschaftsweg', NULL, ''),
		('Gärtnerei', NULL, ''),
		('Modellflugplatz', NULL, ''),
		('Baumschule', NULL, ''),
		('Garten', NULL, ''),
		('Wohnen', NULL, ''),
		('Forschung', NULL, ''),
		('Bildung und Forschung', NULL, ''),
		('Kanal', NULL, ''),
		('Industrie und Gewerbe', NULL, ''),
		('Betriebsfläche Versorgungsanlage, Wärme', NULL, ''),
		('Brachland', NULL, ''),
		('Handwerk', NULL, ''),
		('Flussmündungstrichter', NULL, ''),
		('Altarm', NULL, ''),
		('Verkehrsbegleitfläche Straße', NULL, ''),
		('Wohnbaufläche', NULL, ''),
		('Fortswirtschaftliche Betriebsfläche', NULL, ''),
		('Sporthafenbecken', NULL, ''),
		('Laubholz', NULL, ''),
		('Restauration', NULL, ''),
		('Safaripark, Wildpark', NULL, ''),
		('Gebäude- und Freifläche zu Verkehrsanlagen, Luftfahrt', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage, Funk- und Fernmeldewesen', NULL, ''),
		('Rad- und Fußweg', NULL, ''),
		('Grünanlage', NULL, ''),
		('Raffinerie', NULL, ''),
		('Sukzessionsfläche', NULL, ''),
		('Fähranlage', NULL, ''),
		('Sport-, Freizeit- und Erholungsfläche', NULL, ''),
		('Landwirtschaftliche Betriebsfläche', NULL, ''),
		('Gebäude- und Freifläche zu Verkehrsanlagen, Schifffahrt', NULL, ''),
		('Verwaltung', NULL, ''),
		('Gebäude- und Freifläche, Mischnutzung mit Wohnen', NULL, ''),
		('Reitplatz', NULL, ''),
		('Betriebsfläche Entsorguingsanlage, Abfallbeseitigung', NULL, ''),
		('Gebäude- und Freifläche Entsorgungsanlage, Abfallbeseitigung', NULL, ''),
		('Straßenentwässerungsanlage', NULL, ''),
		('Wald', NULL, ''),
		('Moor', NULL, ''),
		('Heizwerk', NULL, ''),
		('Gewerbe und Industrie mit Wohnen', NULL, ''),
		('Betriebsfläche Versorgungsanlage, Wasser', NULL, ''),
		('Fleet', NULL, ''),
		('Bebaute Gewässerbegleitfläche', NULL, ''),
		('Meer', NULL, ''),
		('Betriebsfläche Versorgungsanlage, Gas', NULL, ''),
		('Wirtschaftsweg', NULL, ''),
		('Gebäude- und Freifläche Friedhof', NULL, ''),
		('Küstengewässer', NULL, ''),
		('Fußgängerzone (Straßenverkehr)', NULL, ''),
		('Betriebsfläche Versorgungsanlage, Öl', NULL, ''),
		('Gang', NULL, ''),
		('Schlossanlage', NULL, ''),
		('Freilichttheater', NULL, ''),
		('Gebäude- und Freifläche Entsorgungsanlage', NULL, ''),
		('Marktplatz', NULL, ''),
		('Gaswerk', NULL, ''),
		('Schießanlage', NULL, ''),
		('Baggersee', NULL, ''),
		('Gebäude- und Freifläche Industrie und Gewerbe', NULL, ''),
		('Deponie (oberirdisch)', NULL, ''),
		('Speicherbecken', NULL, ''),
		('Handel und Dienstleistung', NULL, ''),
		('Vegetationslose Fläche', NULL, ''),
		('Erholungsfläche', NULL, ''),
		('Fließgewässer', NULL, ''),
		('Wochenend- und Ferienhausfläche', NULL, ''),
		('Gesundheit, Kur', NULL, ''),
		('Tankstelle', NULL, ''),
		('Wohnen mit Handel und Dienstleistungen', NULL, ''),
		('Religiöse Einrichtung', NULL, ''),
		('Vergnügen', NULL, ''),
		('Ausstellung, Messe', NULL, ''),
		('Betriebsfläche Versorgungsanlage, Funkt- und Fernmeldewesen', NULL, ''),
		('Handel und Dienstleistungen mit Wohnen', NULL, ''),
		('Historische Anlage', NULL, ''),
		('Weg', NULL, ''),
		('Wohnen mit Gewerbe und Industrie', NULL, ''),
		('Halde', NULL, ''),
		('Produktion', NULL, ''),
		('Betriebsfläche Versorgungsanlage, Elektrizität', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage, Wärme', NULL, ''),
		('Spielplatz, Bolzplatz', NULL, ''),
		('Öffentlich mit Wohnen', NULL, ''),
		('Industrie- und Gewerbefläche', NULL, ''),
		('Streuobstwiese', NULL, ''),
		('Laubwald mit Nadelholz', NULL, ''),
		('Schwimmbad, Freibad', NULL, ''),
		('Gebäude- und Freifläche Versorgungsanlage, Gas', NULL, ''),
		('Betriebliche Sozialeinrichtungen', NULL, ''),
		('Verkehrsbegleitfläche Bahnverkehr', NULL, ''),
		('Fläche gemischter Nutzung', NULL, ''),
		('Sicherheit und Ordnung', NULL, ''),
		('Umspannstation', NULL, ''),
		('Bank, Kredit', NULL, ''),
		('Hafenanlage (Landfläche)', NULL, ''),
		('Werft', NULL, ''),
		('Nadelholz', NULL, ''),
		('Wohnen mit Öffentlich', NULL, ''),
		('Bahnverkehr', NULL, ''),
		('Gebäude- und Freifläche Erholung, Sport', NULL, ''),
		('Gartenland', NULL, ''),
		('Grünfläche', NULL, ''),
		('Sportanlage', NULL, ''),
		('Landwirtschaft', NULL, ''),
		('Heide', NULL, ''),
		('Latschenkiefer', NULL, ''),
		('Gehölz', NULL, ''),
		('Gewässerbegleitfläche', NULL, ''),
		('Teich', NULL, ''),
		('Unbebaute Gewässerbegleitfläche', NULL, ''),
		('Zoo', NULL, ''),
		('Gebäude- und Freifläche Erholung, Bad', NULL, ''),
		('Kraftwerk', NULL, ''),
		('Parkplatz', NULL, ''),
		('Freilichtmuseum', NULL, ''),
		('Parken', NULL, ''),
		('Streuobstacker', NULL, ''),
		('Park', NULL, ''),
		('Hundeübungsplatz', NULL, ''),
		('Fläche besonderer funktionaler Prägung', NULL, ''),
		('Graben', NULL, ''),
		('UnlandVegetationslose Fläche', NULL, ''),
		('Platz', NULL, ''),
		('Bach', NULL, ''),
		('Öffentliche Zwecke', NULL, ''),
		('Fahrweg', NULL, ''),
		('Stehendes Gewässer', NULL, ''),
		('Rennbahn', NULL, ''),
		('Laub- und Nadelholz', NULL, ''),
		('Altwasser', NULL, ''),
		('Friedhof', NULL, ''),
		('Eis-, Rollschuhbahn', NULL, ''),
		('Schiffsverkehr', NULL, ''),
		('Friedhof (Park)', NULL, ''),
		('Sumpf', NULL, ''),
		('Fußgängerzone (Platz)', NULL, ''),
		('Verwaltung, freie Berufe', NULL, ''),
		('Wochenendplatz', NULL, ''),
		('Botanischer Garten', NULL, ''),
		('Gebäude- und Freifläche zu Verkehrtsanlagen, Schiene', NULL, '');
		CREATE TABLE ax_funktionhgr_k_tnhgr (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_funktionhgr_k_tnhgr IS 'Alias: "AX_FunktionHGr_K_TNHGR", UML-Typ: Enumeration';
		INSERT INTO ax_funktionhgr_k_tnhgr (wert,beschreibung,dokumentation) VALUES
		('Gewässer', NULL, ''),
		('Siedlung', NULL, ''),
		('Vegetation', NULL, ''),
		('Verkehr', NULL, '');
		CREATE TABLE ax_wirtschaftsart (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_wirtschaftsart IS 'Alias: "AX_Wirtschaftsart", UML-Typ: Enumeration';
		INSERT INTO ax_wirtschaftsart (wert,beschreibung,dokumentation) VALUES
		('Friedhof', NULL, ''),
		('Historische Anlage', NULL, ''),
		('Waldfläche', NULL, ''),
		('Verkehrsfläche', NULL, ''),
		('Betriebsfläche', NULL, ''),
		('Wasserfläche', NULL, ''),
		('Gebäude- und Freifläche', NULL, ''),
		('Erholungsfläche', NULL, ''),
		('Unland', NULL, ''),
		('Übungsgelände', NULL, ''),
		('Landwirtschaftsfläche', NULL, '');
		CREATE TABLE ax_punktart_k_punkte (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_punktart_k_punkte IS 'Alias: "AX_Punktart_K_PUNKTE", UML-Typ: Enumeration';
		INSERT INTO ax_punktart_k_punkte (wert,beschreibung,dokumentation) VALUES
		('Gebäudepunkt', NULL, ''),
		('Sicherungspunkt', NULL, ''),
		('Lagefestpunkt', NULL, ''),
		('Topographischer Punkt', NULL, ''),
		('Vermessungspunkt', NULL, ''),
		('Bauwerkspunkt', NULL, ''),
		('Aufnahmepunkt', NULL, '');
		CREATE TABLE ax_k_zeile_punktart (
		  wert character varying,
		  beschreibung character varying,
		  dokumentation character varying,
		  PRIMARY KEY (wert)
		);

		COMMENT ON TABLE ax_k_zeile_punktart IS 'Alias: "AX_K_ZEILE_Punktart", UML-Typ: Enumeration';
		INSERT INTO ax_k_zeile_punktart (wert,beschreibung,dokumentation) VALUES
		('Sonstiger Vermessungspunkt', NULL, ''),
		('Sicherungspunkt', NULL, ''),
		('Grenzpunkt', NULL, ''),
		('Aufnahmepunkt', NULL, ''),
		('Besonderer Bauwerkspunkt', NULL, ''),
		('Besonderer Gebäudepunkt', NULL, ''),
		('Besonderer topographischer Punkt', NULL, '');
		CREATE TABLE aa_besonderemeilensteinkategorie (
		  codespace text,
		  id character varying,
		  value text,
		  PRIMARY KEY (id)
		);

		COMMENT ON TABLE aa_besonderemeilensteinkategorie IS 'Alias: "AA_BesondereMeilensteinkategorie", UML-Typ: Code Liste';
		CREATE TABLE aa_anlassart (
		  codespace text,
		  id character varying,
		  value text,
		  PRIMARY KEY (id)
		);

		COMMENT ON TABLE aa_anlassart IS 'Alias: "AA_Anlassart", UML-Typ: Code Liste';
		CREATE TABLE aa_levelofdetail (
		  codespace text,
		  id character varying,
		  value text,
		  PRIMARY KEY (id)
		);

		COMMENT ON TABLE aa_levelofdetail IS 'Alias: "AA_LevelOfDetail", UML-Typ: Code Liste';
		CREATE TABLE aa_anlassart_benutzungsauftrag (
		  codespace text,
		  id character varying,
		  value text,
		  PRIMARY KEY (id)
		);

		COMMENT ON TABLE aa_anlassart_benutzungsauftrag IS 'Alias: "AA_Anlassart_Benutzungsauftrag", UML-Typ: Code Liste';
		CREATE TABLE aa_weiteremodellart (
		  codespace text,
		  id character varying,
		  value text,
		  PRIMARY KEY (id)
		);

		COMMENT ON TABLE aa_weiteremodellart IS 'Alias: "AA_WeitereModellart", UML-Typ: Code Liste';
		CREATE TABLE aa_instanzenthemen (
		  codespace text,
		  id character varying,
		  value text,
		  PRIMARY KEY (id)
		);

		COMMENT ON TABLE aa_instanzenthemen IS 'Alias: "AA_Instanzenthemen", UML-Typ: Code Liste';

		INSERT INTO aa_anlassart (id, value) VALUES ('000000', 'Ersteinrichtung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010307', 'Eintragung des Flurstückes');
		INSERT INTO aa_anlassart (id, value) VALUES ('010308', 'Löschen des Flurstückes');
		INSERT INTO aa_anlassart (id, value) VALUES ('010309', 'Veränderung der Gemeindezugehörigkeit einzelner Flurstücke');
		INSERT INTO aa_anlassart (id, value) VALUES ('010310', 'Veränderung der Gemeindezugehörigkeit');
		INSERT INTO aa_anlassart (id, value) VALUES ('010312', 'Veränderung der Flurstücksnummer ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010317', 'Gebäudeeinmessung nach § 14 Abs. 2 Satz 2 VermGeoG LSA');
		INSERT INTO aa_anlassart (id, value) VALUES ('010401', 'Veränderung der besonderen Flurstücksgrenze');
		INSERT INTO aa_anlassart (id, value) VALUES ('010402', 'Veränderung der Lage');
		INSERT INTO aa_anlassart (id, value) VALUES ('010403', 'Veränderung der tatsächlichen Nutzung mit Änderung der Wirtschaftsart');
		INSERT INTO aa_anlassart (id, value) VALUES ('010404', 'Veränderung des Anliegervermerks');
		INSERT INTO aa_anlassart (id, value) VALUES ('010405', 'Veränderung der tatsächlichen Nutzung ohne Änderung der Wirtschaftsart ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010501', 'Berichtigung der Flächenangabe');
		INSERT INTO aa_anlassart (id, value) VALUES ('010502', 'Berichtigung eines Zeichenfehlers');
		INSERT INTO aa_anlassart (id, value) VALUES ('010503', 'Berichtigung eines Katastrierungsfehlers');
		INSERT INTO aa_anlassart (id, value) VALUES ('300420', 'Veränderung der Bodenschätzung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010511', 'Berichtigung der Flächenangabe mit Veränderung des Flurstückskennzeichens');
		INSERT INTO aa_anlassart (id, value) VALUES ('050300', 'Berichtigung aufgrund Erwerbsvorgängen außerhalb des Grundbuchs');
		INSERT INTO aa_anlassart (id, value) VALUES ('060000', 'Grundstücke buchen');
		INSERT INTO aa_anlassart (id, value) VALUES ('060100', 'Abschreibung');
		INSERT INTO aa_anlassart (id, value) VALUES ('060101', 'Abschreibung auf neues Buchungsblatt');
		INSERT INTO aa_anlassart (id, value) VALUES ('010601', 'Verfahren nach dem Flurbereinigungsgesetz');
		INSERT INTO aa_anlassart (id, value) VALUES ('010602', 'Verfahren nach dem Baugesetzbuch');
		INSERT INTO aa_anlassart (id, value) VALUES ('010611', 'Flurbereinigung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010615', 'Übernahme von wichtigen Terminen im Ablauf eines Bodenordnungsverfahrens ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010616', 'Vermessung der Verfahrensgrenze des Flurbereinigungsgebietes');
		INSERT INTO aa_anlassart (id, value) VALUES ('010619', 'Übernahme von Flurbereinigungsergebnissen');
		INSERT INTO aa_anlassart (id, value) VALUES ('010621', 'Umlegung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010622', 'Umlegung nach § 76 BauGB');
		INSERT INTO aa_anlassart (id, value) VALUES ('010623', 'Vereinfachte Umlegung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010700', 'Katastererneuerung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010701', 'Katastererneuerung - vereinfachte Neuvermessung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010702', 'Erneuerung der Lagekoordinaten ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010703', 'Veränderung aufgrund der Qualitätsverbesserung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010704', 'Qualitätssicherung und Datenpflege');
		INSERT INTO aa_anlassart (id, value) VALUES ('010801', 'Zuschreibung eines Flurstückes (Gebietsreform)');
		INSERT INTO aa_anlassart (id, value) VALUES ('010802', 'Abschreibung eines Flurstückes (Gebietsreform)');
		INSERT INTO aa_anlassart (id, value) VALUES ('010900', 'Grenzfeststellung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010901', 'Grenzvermessung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010902', 'Grenzwiederherstellung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010903', 'Grenzbestimmung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010904', 'Grenzabmarkung ');
		INSERT INTO aa_anlassart (id, value) VALUES ('020100', 'Katasterliche Buchungsdaten fortführen');
		INSERT INTO aa_anlassart (id, value) VALUES ('020102', 'Katasterliche Buchung eines buchungsfreien Grundstücks');
		INSERT INTO aa_anlassart (id, value) VALUES ('020200', 'Namensnummer von katasterlichen Buchungsstellen verändern');
		INSERT INTO aa_anlassart (id, value) VALUES ('020201', 'Katasterliche Namensnummer und Rechtsgemeinschaft fortführen');
		INSERT INTO aa_anlassart (id, value) VALUES ('020301', 'Veränderung der Personendaten');
		INSERT INTO aa_anlassart (id, value) VALUES ('020302', 'Veränderung der Personengruppe');
		INSERT INTO aa_anlassart (id, value) VALUES ('020304', 'Veränderung der Verwaltung');
		INSERT INTO aa_anlassart (id, value) VALUES ('020305', 'Veränderung der Vertretung ');
		INSERT INTO aa_anlassart (id, value) VALUES ('030000', 'Grundbuchblattbezeichnung ändern');
		INSERT INTO aa_anlassart (id, value) VALUES ('030100', 'Umnummerierung (infolge Zuständigkeitsänderungen am Grundbuch)');
		INSERT INTO aa_anlassart (id, value) VALUES ('040000', 'Beschreibung der Buchungsstelle ändern');
		INSERT INTO aa_anlassart (id, value) VALUES ('040100', 'Änderungen am Wohnungseigentum');
		INSERT INTO aa_anlassart (id, value) VALUES ('040200', 'Änderungen am Wohnungserbbaurecht');
		INSERT INTO aa_anlassart (id, value) VALUES ('040500', 'Änderungen an den Beziehungen zwischen den Buchungsstellen');
		INSERT INTO aa_anlassart (id, value) VALUES ('040600', 'Änderungen in der Ergänzung, Beschreibung sowie der Bemerkung zu Buchungen');
		INSERT INTO aa_anlassart (id, value) VALUES ('050000', 'Angaben zu Eigentümer oder Erbbauberechtigten verändern');
		INSERT INTO aa_anlassart (id, value) VALUES ('050100', 'Eigentumsänderung');
		INSERT INTO aa_anlassart (id, value) VALUES ('050101', 'Kompletter Eigentumswechsel im Grundbuchblatt');
		INSERT INTO aa_anlassart (id, value) VALUES ('050102', 'Teilweiser Eigentumswechsel im Grundbuchblatt');
		INSERT INTO aa_anlassart (id, value) VALUES ('050200', 'Berichtigung aufgrund Erbnachweis');
		INSERT INTO aa_anlassart (id, value) VALUES ('060102', 'Abschreibung auf bestehendes Buchungsblatt');
		INSERT INTO aa_anlassart (id, value) VALUES ('060200', 'Teilung');
		INSERT INTO aa_anlassart (id, value) VALUES ('060201', 'Buchung der neuen Grundstücke in einem bestehenden Buchungsblatt');
		INSERT INTO aa_anlassart (id, value) VALUES ('060600', 'Vollziehung einer Verschmelzung');
		INSERT INTO aa_anlassart (id, value) VALUES ('060700', 'Buchung aufgrund Veränderung der Grundstücksbezeichnung oder der Größe (Spalten 3 + 4 des BV)');
		INSERT INTO aa_anlassart (id, value) VALUES ('060801', 'Buchung des Grundstücks auf ein neues Buchungsblatt (1)');
		INSERT INTO aa_anlassart (id, value) VALUES ('060802', 'Buchung des Grundstücks auf ein bestehendes Buchungsblatt der Miteigentümer');
		INSERT INTO aa_anlassart (id, value) VALUES ('060900', 'Aufhebung eines Wohnungseigentums');
		INSERT INTO aa_anlassart (id, value) VALUES ('062000', 'Grundbuchblatt schließen');
		INSERT INTO aa_anlassart (id, value) VALUES ('070000', 'Rechte buchen');
		INSERT INTO aa_anlassart (id, value) VALUES ('070100', 'Erbbaurecht anlegen');
		INSERT INTO aa_anlassart (id, value) VALUES ('070200', 'Erbbaurecht aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('070300', 'Herrschvermerk buchen');
		INSERT INTO aa_anlassart (id, value) VALUES ('070400', 'Herrschvermerk aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('070500', 'Untererbbaurecht anlegen');
		INSERT INTO aa_anlassart (id, value) VALUES ('070600', 'Untererbbaurecht aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('070700', 'Sonstige Rechte anlegen');
		INSERT INTO aa_anlassart (id, value) VALUES ('070800', 'Sonstige Rechte aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('070900', 'Wohnungserbbaurecht aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('071000', 'Wohnungsuntererbbaurecht aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('071100', 'Teilung Herrschvermerk aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('071200', 'Teilung sonstiges Recht aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('080000', 'Anteile buchen');
		INSERT INTO aa_anlassart (id, value) VALUES ('080200', 'Anlegen von Wohnungseigentum');
		INSERT INTO aa_anlassart (id, value) VALUES ('080300', 'Anlegen von Wohnungserbbaurecht');
		INSERT INTO aa_anlassart (id, value) VALUES ('080400', 'Anlegen von Wohnungsuntererbbaurecht');
		INSERT INTO aa_anlassart (id, value) VALUES ('080500', 'Teilung eines Herrschvermerks');
		INSERT INTO aa_anlassart (id, value) VALUES ('080600', 'Teilung eines sonstigen Rechts');
		INSERT INTO aa_anlassart (id, value) VALUES ('080650', 'Auflösung ungetrennter Hofräume');
		INSERT INTO aa_anlassart (id, value) VALUES ('080700', 'Teilung am ungetrennten Hofraum aufheben');
		INSERT INTO aa_anlassart (id, value) VALUES ('080800', 'Teilung einer Buchung § 3 Abs. 4 GBO nach Wohnungseigentumsgesetz');
		INSERT INTO aa_anlassart (id, value) VALUES ('080900', 'Teilung eines Anteils am ungetrennten Hofraum nach Wohnungseigentumsgesetz');
		INSERT INTO aa_anlassart (id, value) VALUES ('090000', 'Ändern in Verzeichnissen ohne Grundbucheintragung');
		INSERT INTO aa_anlassart (id, value) VALUES ('090100', 'Verwaltung eintragen oder ändern');
		INSERT INTO aa_anlassart (id, value) VALUES ('090200', 'Personengruppe eintragen oder ändern');
		INSERT INTO aa_anlassart (id, value) VALUES ('090300', 'Änderung der Anschrift');
		INSERT INTO aa_anlassart (id, value) VALUES ('090400', 'Änderung der Personendaten');
		INSERT INTO aa_anlassart (id, value) VALUES ('090500', 'Entstehung eines Festpunkts (erstmalige Aufnahme in AFIS)');
		INSERT INTO aa_anlassart (id, value) VALUES ('090510', 'Änderung von Koordinaten, Höhe und/oder Schwerewert eines Festpunkts');
		INSERT INTO aa_anlassart (id, value) VALUES ('090520', 'Veränderung der Vermarkung und/oder der beschreibenden Angaben eines Festpunkts');
		INSERT INTO aa_anlassart (id, value) VALUES ('090530', 'Zerstörung der Vermarkung eines Festpunkts');
		INSERT INTO aa_anlassart (id, value) VALUES ('090540', 'Untergang eines Festpunktes');
		INSERT INTO aa_anlassart (id, value) VALUES ('001000', 'Grundaktualisierung');
		INSERT INTO aa_anlassart (id, value) VALUES ('100000', 'Veränderung der Angaben zu den Nutzerprofilen');
		INSERT INTO aa_anlassart (id, value) VALUES ('002000', 'Spitzenaktualisierung');
		INSERT INTO aa_anlassart (id, value) VALUES ('200000', 'Veränderung von Gebäudedaten');
		INSERT INTO aa_anlassart (id, value) VALUES ('200100', 'Eintragen eines Gebäudes');
		INSERT INTO aa_anlassart (id, value) VALUES ('200200', 'Veränderung der Gebäudeeigenschaften');
		INSERT INTO aa_anlassart (id, value) VALUES ('200300', 'Löschen eines Gebäudes');
		INSERT INTO aa_anlassart (id, value) VALUES ('300000', 'Sonstige Daten fortführen');
		INSERT INTO aa_anlassart (id, value) VALUES ('300100', 'Veränderungen der Angaben zum Netzpunkt');
		INSERT INTO aa_anlassart (id, value) VALUES ('300410', 'Veränderung der öffentlich-rechtlichen und sonstigen Festsetzungen');
		INSERT INTO aa_anlassart (id, value) VALUES ('300421', 'Erfassung der Bodenschätzung ');
		INSERT INTO aa_anlassart (id, value) VALUES ('300430', 'Veränderung der Bewertung');
		INSERT INTO aa_anlassart (id, value) VALUES ('300440', 'Veränderung der Gebietseinheiten');
		INSERT INTO aa_anlassart (id, value) VALUES ('300501', 'Veränderung aufgrund der Homogenisierung ');
		INSERT INTO aa_anlassart (id, value) VALUES ('300600', 'Veränderung der Reservierung von Fachkennzeichen');
		INSERT INTO aa_anlassart (id, value) VALUES ('300700', 'Veränderung von Katalogeinträgen ');
		INSERT INTO aa_anlassart (id, value) VALUES ('300800', 'Veränderung von Metadaten');
		INSERT INTO aa_anlassart (id, value) VALUES ('300900', 'Veränderung der Geometrie durch Implizitbehandlung');
		INSERT INTO aa_anlassart (id, value) VALUES ('004100', 'Automatische Kartographische Generalisierung mit Konflikt');
		INSERT INTO aa_anlassart (id, value) VALUES ('005000', 'Interaktive Kartographische Generalisierung');
		INSERT INTO aa_anlassart (id, value) VALUES ('006000', 'Veränderung an der Landesgrenze und des Gebietes');
		INSERT INTO aa_anlassart (id, value) VALUES ('009999', 'Sonstiges');
		INSERT INTO aa_anlassart (id, value) VALUES ('003000', 'Angaben zur Grundbuchfortführung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010101', 'Zerlegung oder Sonderung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010102', 'Verschmelzung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010103', 'Zerlegung und Verschmelzung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010104', 'Flurstückszerlegung mit Eigentumsübergang nach Straßengesetzen');
		INSERT INTO aa_anlassart (id, value) VALUES ('010105', 'Zerlegung ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010107', 'Sonderung ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010199', 'Verschmelzung von Flurstücken auf unterschiedlichen Beständen/Buchungsstellen');
		INSERT INTO aa_anlassart (id, value) VALUES ('010201', 'Veränderung aufgrund der Vorschriften des Straßenrechts');
		INSERT INTO aa_anlassart (id, value) VALUES ('010202', 'Veränderung aufgrund der Vorschriften des Wasserrechts');
		INSERT INTO aa_anlassart (id, value) VALUES ('010203', 'Veränderung am Flurstück mit Änderung der Umfangsgrenzen ');
		INSERT INTO aa_anlassart (id, value) VALUES ('010205', 'Veränderung aufgrund Berichtigung eines Aufnahmefehlers');
		INSERT INTO aa_anlassart (id, value) VALUES ('010206', 'Veränderung aufgrund gerichtlicher Entscheidung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010207', 'Veränderung aufgrund Berichtigung eines Grenzbestimmungsfehlers');
		INSERT INTO aa_anlassart (id, value) VALUES ('010301', 'Veränderung der Flurstücksbezeichnung');
		INSERT INTO aa_anlassart (id, value) VALUES ('010303', 'Veränderung der Gemeindezugehörigkeit ganzer Gemarkungen');
		INSERT INTO aa_anlassart (id, value) VALUES ('010305', 'Veränderung der Flurzugehörigkeit');
		INSERT INTO aa_anlassart (id, value) VALUES ('050400', 'Namensänderung');
		INSERT INTO aa_anlassart (id, value) VALUES ('060901', 'Buchung des Grundstücks auf ein neues Buchungsblatt (2)');
		INSERT INTO aa_anlassart (id, value) VALUES ('060902', 'Buchung des Grundstücks auf ein bestehendes Buchungsblatt');
		INSERT INTO aa_anlassart (id, value) VALUES ('061000', 'Umschreibung des Grundbuchs (§§ 28 ff, 68 GBV)');
		INSERT INTO aa_anlassart (id, value) VALUES ('061100', 'Neufassung des Grundbuchs (§§ 33, 69 GBV)');
		INSERT INTO aa_anlassart (id, value) VALUES ('061200', 'Erstbuchung eines Grundstücks');
		INSERT INTO aa_anlassart (id, value) VALUES ('061400', 'Aufhebung von Anteilen am ungetrennten Hofraum');

		CREATE TABLE ax_benutzer (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  art character varying,
		  direkt character varying,
		  email character varying,
		  http character varying,
		  manuell character varying,
		  ausgabemasstab double precision,
		  ausgabemedium integer,
		  datenformat integer,
		  formatangabe character varying,
		  letzteabgabeart integer,
		  letzteabgabezugriff character(20),
		  nbaquittierungerhalten character(20),
		  nbauebernahmeerfolgreich character varying,
		  profilkennung character varying NOT NULL,
		  vorletzteabgabezugriff character(20),
		  zahlungsweise character varying,
		  zeitlicheberechtigung date,
		  istteilvon character(16)[],
		  ist character(16) NOT NULL,
		  gehoertzu character(16) NOT NULL,
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_benutzer_gml ON ax_benutzer USING btree (gml_id,beginnt);
		CREATE INDEX ax_benutzer_endet ON ax_benutzer USING btree (endet);
		CREATE INDEX ax_benutzer_ist ON ax_benutzer USING btree (ist);
		CREATE INDEX ax_benutzer_gehoertzu ON ax_benutzer USING btree (gehoertzu);
		CREATE INDEX ax_benutzer_istteilvon ON ax_benutzer USING gin (istteilvon);

		COMMENT ON TABLE ax_benutzer IS 'FeatureType: "AX_Benutzer"';
		COMMENT ON COLUMN ax_benutzer.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_benutzer.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_benutzer.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_benutzer.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_benutzer.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_benutzer.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_benutzer.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_benutzer.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_benutzer.art IS 'art   0..1';
		COMMENT ON COLUMN ax_benutzer.direkt IS 'empfaenger|AA_Empfaenger|direkt  Boolean 0..1';
		COMMENT ON COLUMN ax_benutzer.email IS 'empfaenger|AA_Empfaenger|email  URI 0..1';
		COMMENT ON COLUMN ax_benutzer.http IS 'empfaenger|AA_Empfaenger|http  URI 0..1';
		COMMENT ON COLUMN ax_benutzer.manuell IS 'empfaenger|AA_Empfaenger|manuell   0..1';
		COMMENT ON COLUMN ax_benutzer.ausgabemasstab IS 'folgeverarbeitung|AX_FOLGEVA|ausgabemasstab  Real 0..1';
		COMMENT ON COLUMN ax_benutzer.ausgabemedium IS 'folgeverarbeitung|AX_FOLGEVA|ausgabemedium enumeration AX_Ausgabemedium_Benutzer 0..1';
		COMMENT ON COLUMN ax_benutzer.datenformat IS 'folgeverarbeitung|AX_FOLGEVA|datenformat enumeration AX_Datenformat_Benutzer 0..1';
		COMMENT ON COLUMN ax_benutzer.formatangabe IS 'folgeverarbeitung|AX_FOLGEVA|formatangabe   0..1';
		COMMENT ON COLUMN ax_benutzer.letzteabgabeart IS 'letzteAbgabeArt enumeration AX_LetzteAbgabeArt 0..1';
		COMMENT ON COLUMN ax_benutzer.letzteabgabezugriff IS 'letzteAbgabeZugriff  DateTime 0..1';
		COMMENT ON COLUMN ax_benutzer.nbaquittierungerhalten IS 'nBAQuittierungErhalten  DateTime 0..1';
		COMMENT ON COLUMN ax_benutzer.nbauebernahmeerfolgreich IS 'nBAUebernahmeErfolgreich  Boolean 0..1';
		COMMENT ON COLUMN ax_benutzer.profilkennung IS 'profilkennung   1';
		COMMENT ON COLUMN ax_benutzer.vorletzteabgabezugriff IS 'vorletzteAbgabeZugriff  DateTime 0..1';
		COMMENT ON COLUMN ax_benutzer.zahlungsweise IS 'zahlungsweise   0..1';
		COMMENT ON COLUMN ax_benutzer.zeitlicheberechtigung IS 'zeitlicheBerechtigung  Date 0..1';
		COMMENT ON COLUMN ax_benutzer.ist IS 'Assoziation zu: FeatureType AX_Person (ax_person) 1';
		COMMENT ON COLUMN ax_benutzer.gehoertzu IS 'Assoziation zu: FeatureType AX_Benutzergruppe (ax_benutzergruppe) 1';
		CREATE TABLE ax_benutzergruppemitzugriffskontrolle (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying NOT NULL,
		  koordinatenreferenzsystem character varying,
		  land character varying NOT NULL,
		  stelle character varying NOT NULL,
		  selektionskriterien character varying[] NOT NULL,
		  zugriffhistorie character varying NOT NULL,
		  zugriffsartfortfuehrungsanlass character varying[],
		  zugriffsartproduktkennungbenutzung character varying[],
		  zugriffsartproduktkennungfuehrung character varying[],
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_benutzergruppemitzugriffskontrolle_gml ON ax_benutzergruppemitzugriffskontrolle USING btree (gml_id,beginnt);
		CREATE INDEX ax_benutzergruppemitzugriffskontrolle_endet ON ax_benutzergruppemitzugriffskontrolle USING btree (endet);
		CREATE INDEX ax_benutzergruppemitzugriffskontrolle_bestehtaus ON ax_benutzergruppemitzugriffskontrolle USING gin (bestehtaus);
		CREATE INDEX ax_benutzergruppemitzugriffskontrolle_istteilvon ON ax_benutzergruppemitzugriffskontrolle USING gin (istteilvon);

		COMMENT ON TABLE ax_benutzergruppemitzugriffskontrolle IS 'FeatureType: "AX_BenutzergruppeMitZugriffskontrolle"';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.bezeichnung IS 'bezeichnung   1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.koordinatenreferenzsystem IS 'koordinatenreferenzsystem  SC_CRS 0..1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.land IS 'zustaendigeStelle|AX_Dienststelle_Schluessel|land   1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.stelle IS 'zustaendigeStelle|AX_Dienststelle_Schluessel|stelle   1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.selektionskriterien IS 'selektionskriterien   1..*';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zugriffhistorie IS 'zugriffHistorie  Boolean 1';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zugriffsartfortfuehrungsanlass IS 'zugriffsartFortfuehrungsanlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zugriffsartproduktkennungbenutzung IS 'zugriffsartProduktkennungBenutzung codelist AA_Anlassart_Benutzungsauftrag 0..*';
		COMMENT ON COLUMN ax_benutzergruppemitzugriffskontrolle.zugriffsartproduktkennungfuehrung IS 'zugriffsartProduktkennungFuehrung codelist AA_Anlassart 0..*';
		CREATE TABLE ax_benutzergruppenba (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying NOT NULL,
		  koordinatenreferenzsystem character varying,
		  land character varying NOT NULL,
		  stelle character varying NOT NULL,
		  bereichzeitlich_art integer,
		  bereichzeitlich_ersterstichtag date,
		  bereichzeitlich_intervall date,
		  seitenlaenge integer,
		  quittierung character varying,
		  selektionskriterien character varying[] NOT NULL,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_benutzergruppenba_gml ON ax_benutzergruppenba USING btree (gml_id,beginnt);
		CREATE INDEX ax_benutzergruppenba_endet ON ax_benutzergruppenba USING btree (endet);
		CREATE INDEX ax_benutzergruppenba_bestehtaus ON ax_benutzergruppenba USING gin (bestehtaus);
		CREATE INDEX ax_benutzergruppenba_istteilvon ON ax_benutzergruppenba USING gin (istteilvon);

		COMMENT ON TABLE ax_benutzergruppenba IS 'FeatureType: "AX_BenutzergruppeNBA"';
		COMMENT ON COLUMN ax_benutzergruppenba.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_benutzergruppenba.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_benutzergruppenba.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_benutzergruppenba.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.bezeichnung IS 'bezeichnung   1';
		COMMENT ON COLUMN ax_benutzergruppenba.koordinatenreferenzsystem IS 'koordinatenreferenzsystem  SC_CRS 0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.land IS 'zustaendigeStelle|AX_Dienststelle_Schluessel|land   1';
		COMMENT ON COLUMN ax_benutzergruppenba.stelle IS 'zustaendigeStelle|AX_Dienststelle_Schluessel|stelle   1';
		COMMENT ON COLUMN ax_benutzergruppenba.bereichzeitlich_art IS 'bereichZeitlich|AX_BereichZeitlich|art enumeration AX_Art_BereichZeitlich 1';
		COMMENT ON COLUMN ax_benutzergruppenba.bereichzeitlich_ersterstichtag IS 'bereichZeitlich|AX_BereichZeitlich|ersterStichtag  Date 1';
		COMMENT ON COLUMN ax_benutzergruppenba.bereichzeitlich_intervall IS 'bereichZeitlich|AX_BereichZeitlich|intervall  TM_Duration 1';
		COMMENT ON COLUMN ax_benutzergruppenba.seitenlaenge IS 'portionierungsparameter|AX_Portionierungsparameter|seitenlaenge  Integer 1';
		COMMENT ON COLUMN ax_benutzergruppenba.quittierung IS 'quittierung  Boolean 0..1';
		COMMENT ON COLUMN ax_benutzergruppenba.selektionskriterien IS 'selektionskriterien   1..*';
		CREATE TABLE ax_personengruppe (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  namederpersonengruppe character varying NOT NULL,
		  istteilvon character(16)[],
		  bestehtaus character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_personengruppe_gml ON ax_personengruppe USING btree (gml_id,beginnt);
		CREATE INDEX ax_personengruppe_endet ON ax_personengruppe USING btree (endet);
		CREATE INDEX ax_personengruppe_bestehtaus ON ax_personengruppe USING gin (bestehtaus);
		CREATE INDEX ax_personengruppe_istteilvon ON ax_personengruppe USING gin (istteilvon);

		COMMENT ON TABLE ax_personengruppe IS 'FeatureType: "AX_Personengruppe"';
		COMMENT ON COLUMN ax_personengruppe.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_personengruppe.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_personengruppe.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_personengruppe.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_personengruppe.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_personengruppe.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_personengruppe.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_personengruppe.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_personengruppe.namederpersonengruppe IS 'nameDerPersonengruppe   1';
		COMMENT ON COLUMN ax_personengruppe.bestehtaus IS 'Assoziation zu: FeatureType AX_Person (ax_person) 0..*';
		CREATE TABLE ax_skizze (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bemerkungen character varying,
		  skizzenart integer,
		  skizzenname character varying NOT NULL,
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_skizze_gml ON ax_skizze USING btree (gml_id,beginnt);
		CREATE INDEX ax_skizze_endet ON ax_skizze USING btree (endet);
		CREATE INDEX ax_skizze_istteilvon ON ax_skizze USING gin (istteilvon);

		COMMENT ON TABLE ax_skizze IS 'FeatureType: "AX_Skizze"';
		COMMENT ON COLUMN ax_skizze.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_skizze.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_skizze.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_skizze.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_skizze.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_skizze.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_skizze.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_skizze.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_skizze.bemerkungen IS 'bemerkungen   0..1';
		COMMENT ON COLUMN ax_skizze.skizzenart IS 'skizzenart enumeration AX_Skizzenart_Skizze 0..1';
		COMMENT ON COLUMN ax_skizze.skizzenname IS 'skizzenname  URI 1';
		CREATE TABLE ax_schwere (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  aufstellhoehe double precision,
		  hinweise character varying,
		  bestimmungsdatum date,
		  datenerhebung integer,
		  genauigkeitsstufe integer,
		  genauigkeitswert integer,
		  messmethode integer,
		  vertrauenswuerdigkeit integer,
		  schwereanomalie_art integer[],
		  schwereanomalie_wert character varying[],
		  schwerestatus integer,
		  schweresystem integer NOT NULL,
		  schwerewert double precision NOT NULL,
		  ueberpruefungsdatum date,
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_schwere_gml ON ax_schwere USING btree (gml_id,beginnt);
		CREATE INDEX ax_schwere_endet ON ax_schwere USING btree (endet);
		CREATE INDEX ax_schwere_istteilvon ON ax_schwere USING gin (istteilvon);

		COMMENT ON TABLE ax_schwere IS 'FeatureType: "AX_Schwere"';
		COMMENT ON COLUMN ax_schwere.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_schwere.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_schwere.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_schwere.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_schwere.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_schwere.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_schwere.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_schwere.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_schwere.aufstellhoehe IS 'aufstellhoehe  Distance 0..1';
		COMMENT ON COLUMN ax_schwere.hinweise IS 'hinweise   0..1';
		COMMENT ON COLUMN ax_schwere.bestimmungsdatum IS 'qualitaetsangaben|AX_DQSchwere|bestimmungsdatum  Date 0..1';
		COMMENT ON COLUMN ax_schwere.datenerhebung IS 'qualitaetsangaben|AX_DQSchwere|datenerhebung enumeration AX_Datenerhebung_Schwere 0..1';
		COMMENT ON COLUMN ax_schwere.genauigkeitsstufe IS 'qualitaetsangaben|AX_DQSchwere|genauigkeitsstufe enumeration AX_Genauigkeitsstufe_Schwere 0..1';
		COMMENT ON COLUMN ax_schwere.genauigkeitswert IS 'qualitaetsangaben|AX_DQSchwere|genauigkeitswert  Integer 0..1';
		COMMENT ON COLUMN ax_schwere.messmethode IS 'qualitaetsangaben|AX_DQSchwere|messmethode enumeration AX_Messmethode_Schwere 0..1';
		COMMENT ON COLUMN ax_schwere.vertrauenswuerdigkeit IS 'qualitaetsangaben|AX_DQSchwere|vertrauenswuerdigkeit enumeration AX_Vertrauenswuerdigkeit_Schwere 0..1';
		COMMENT ON COLUMN ax_schwere.schwereanomalie_art IS 'schwereanomalie|AX_Schwereanomalie_Schwere|art enumeration AX_Schwereanomalie_Schwere_Art 1';
		COMMENT ON COLUMN ax_schwere.schwereanomalie_wert IS 'schwereanomalie|AX_Schwereanomalie_Schwere|wert  Measure 1';
		COMMENT ON COLUMN ax_schwere.schwerestatus IS 'schwerestatus enumeration AX_Schwerestatus_Schwere 0..1';
		COMMENT ON COLUMN ax_schwere.schweresystem IS 'schweresystem enumeration AX_Schweresystem_Schwere 1';
		COMMENT ON COLUMN ax_schwere.schwerewert IS 'schwerewert  Real 1';
		COMMENT ON COLUMN ax_schwere.ueberpruefungsdatum IS 'ueberpruefungsdatum  Date 0..1';
		CREATE TABLE ax_punktkennungvergleichend (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  antragsnummer character varying NOT NULL,
		  art integer,
		  endgueltigepunktkennung character varying NOT NULL,
		  vorlaeufigepunktkennung character varying NOT NULL,
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_punktkennungvergleichend_gml ON ax_punktkennungvergleichend USING btree (gml_id,beginnt);
		CREATE INDEX ax_punktkennungvergleichend_endet ON ax_punktkennungvergleichend USING btree (endet);
		CREATE INDEX ax_punktkennungvergleichend_istteilvon ON ax_punktkennungvergleichend USING gin (istteilvon);

		COMMENT ON TABLE ax_punktkennungvergleichend IS 'FeatureType: "AX_PunktkennungVergleichend"';
		COMMENT ON COLUMN ax_punktkennungvergleichend.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_punktkennungvergleichend.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.antragsnummer IS 'antragsnummer   1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.art IS 'art enumeration AX_Art_Punktkennung 0..1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.endgueltigepunktkennung IS 'endgueltigePunktkennung   1';
		COMMENT ON COLUMN ax_punktkennungvergleichend.vorlaeufigepunktkennung IS 'vorlaeufigePunktkennung   1';
		CREATE TABLE ax_verband (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  administrativefunktion integer[],
		  art integer,
		  bezeichnung character varying NOT NULL,
		  gemeinde character varying[] NOT NULL,
		  gemeindeteil character varying[],
		  kreis character varying[] NOT NULL,
		  land character varying[] NOT NULL,
		  regierungsbezirk character varying[],
		  schluesselgesamt character varying NOT NULL,
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_verband_gml ON ax_verband USING btree (gml_id,beginnt);
		CREATE INDEX ax_verband_endet ON ax_verband USING btree (endet);
		CREATE INDEX ax_verband_istteilvon ON ax_verband USING gin (istteilvon);

		COMMENT ON TABLE ax_verband IS 'FeatureType: "AX_Verband"';
		COMMENT ON COLUMN ax_verband.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_verband.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_verband.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_verband.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_verband.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_verband.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_verband.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_verband.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_verband.administrativefunktion IS 'administrativeFunktion enumeration AX_Administrative_Funktion 0..*';
		COMMENT ON COLUMN ax_verband.art IS 'art enumeration AX_Art_Verband 0..1';
		COMMENT ON COLUMN ax_verband.bezeichnung IS 'bezeichnung  CharacterString 1';
		COMMENT ON COLUMN ax_verband.gemeinde IS 'enthaelt|AX_Gemeindekennzeichen|gemeinde   1';
		COMMENT ON COLUMN ax_verband.gemeindeteil IS 'enthaelt|AX_Gemeindekennzeichen|gemeindeteil   0..1';
		COMMENT ON COLUMN ax_verband.kreis IS 'enthaelt|AX_Gemeindekennzeichen|kreis   1';
		COMMENT ON COLUMN ax_verband.land IS 'enthaelt|AX_Gemeindekennzeichen|land   1';
		COMMENT ON COLUMN ax_verband.regierungsbezirk IS 'enthaelt|AX_Gemeindekennzeichen|regierungsbezirk   0..1';
		COMMENT ON COLUMN ax_verband.schluesselgesamt IS 'schluesselGesamt  CharacterString 1';
		CREATE TABLE ax_nationalstaat (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  administrativefunktion integer[],
		  bezeichnung character varying NOT NULL,
		  schluessel character varying NOT NULL,
		  schluesselgesamt character varying NOT NULL,
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_nationalstaat_gml ON ax_nationalstaat USING btree (gml_id,beginnt);
		CREATE INDEX ax_nationalstaat_endet ON ax_nationalstaat USING btree (endet);
		CREATE INDEX ax_nationalstaat_istteilvon ON ax_nationalstaat USING gin (istteilvon);

		COMMENT ON TABLE ax_nationalstaat IS 'FeatureType: "AX_Nationalstaat"';
		COMMENT ON COLUMN ax_nationalstaat.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_nationalstaat.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_nationalstaat.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_nationalstaat.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_nationalstaat.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_nationalstaat.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_nationalstaat.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_nationalstaat.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_nationalstaat.administrativefunktion IS 'administrativeFunktion enumeration AX_Administrative_Funktion 0..*';
		COMMENT ON COLUMN ax_nationalstaat.bezeichnung IS 'bezeichnung  CharacterString 1';
		COMMENT ON COLUMN ax_nationalstaat.schluessel IS 'schluessel   1';
		COMMENT ON COLUMN ax_nationalstaat.schluesselgesamt IS 'schluesselGesamt  CharacterString 1';
		CREATE TABLE ax_netzknoten (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying NOT NULL,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_netzknoten_gml ON ax_netzknoten USING btree (gml_id,beginnt);
		CREATE INDEX ax_netzknoten_endet ON ax_netzknoten USING btree (endet);
		CREATE INDEX ax_netzknoten_bestehtaus ON ax_netzknoten USING gin (bestehtaus);
		CREATE INDEX ax_netzknoten_istteilvon ON ax_netzknoten USING gin (istteilvon);

		COMMENT ON TABLE ax_netzknoten IS 'FeatureType: "AX_Netzknoten"';
		COMMENT ON COLUMN ax_netzknoten.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_netzknoten.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_netzknoten.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_netzknoten.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_netzknoten.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_netzknoten.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_netzknoten.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_netzknoten.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_netzknoten.bezeichnung IS 'bezeichnung   1';
		COMMENT ON COLUMN ax_netzknoten.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_netzknoten.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_netzknoten.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_netzknoten.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_netzknoten.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_netzknoten.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_netzknoten.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_referenzstationspunkt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  darstellungshinweis character varying,
		  frueherepunktnummer character varying[],
		  gemarkung_gemarkungsnummer character varying,
		  gemarkung_land character varying,
		  gemeinde character varying NOT NULL,
		  gemeinde_gemeindeteil character varying,
		  kreis character varying NOT NULL,
		  gemeinde_land character varying NOT NULL,
		  gemeinde_regierungsbezirk character varying,
		  internebemerkungen character varying[],
		  katasteramt_land character varying,
		  katasteramt_stelle character varying,
		  land_land character varying NOT NULL,
		  namelagebeschreibung character varying,
		  nutzerspezifischebemerkungen character varying[],
		  punktkennung character varying NOT NULL,
		  punktvermarkung integer NOT NULL,
		  relativehoehe double precision,
		  beginn character(20),
		  ende character(20),
		  funkfrequenz double precision,
		  funktion integer,
		  gnssantenne_abbaudatum character varying,
		  antennenhoehe double precision NOT NULL,
		  antennenreferenzpunkt character varying NOT NULL,
		  aufbaudatum character varying NOT NULL,
		  azimutaleabweichung double precision NOT NULL,
		  gnssantennenundradometyp character varying NOT NULL,
		  hoehenoffsetl1 double precision NOT NULL,
		  hoehenoffsetl2 double precision NOT NULL,
		  seriennummer character varying NOT NULL,
		  gnssantenne_zusaetzlicheinformationen character varying,
		  gnssempfaenger_abbaudatum character varying,
		  gnssempfaenger_aufbaudatum character varying,
		  gnssempfaenger_firmwareversion character varying,
		  gnssempfaenger_gnssempfaengertyp character varying,
		  gnssempfaenger_seriennummer character varying,
		  gnssempfaenger_zusaetzlicheinformationen character varying,
		  isdnnummer character varying,
		  offsetl1_east double precision NOT NULL,
		  offsetl1_height double precision NOT NULL,
		  offsetl1_north double precision NOT NULL,
		  offsetl2_east double precision NOT NULL,
		  offsetl2_height double precision NOT NULL,
		  offsetl2_north double precision NOT NULL,
		  phasenzentrumsvariationl1_zeile character varying[],
		  phasenzentrumsvariationl2_zeile character varying[],
		  befund character varying,
		  gnsstauglichkeit integer,
		  punktstabilitaet integer,
		  ueberwachungsdatum date,
		  tcpipnummer character varying,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  istidentischmitlfp character(16),
		  istidentischmithfp character(16),
		  unterschiedlicherbezugspunktmitsfp character(16),
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_referenzstationspunkt_gml ON ax_referenzstationspunkt USING btree (gml_id,beginnt);
		CREATE INDEX ax_referenzstationspunkt_endet ON ax_referenzstationspunkt USING btree (endet);
		CREATE INDEX ax_referenzstationspunkt_istidentischmitlfp ON ax_referenzstationspunkt USING btree (istidentischmitlfp);
		CREATE INDEX ax_referenzstationspunkt_istidentischmithfp ON ax_referenzstationspunkt USING btree (istidentischmithfp);
		CREATE INDEX ax_r_unterschiedlicherbezugspunktmitsfp ON ax_referenzstationspunkt USING btree (unterschiedlicherbezugspunktmitsfp);
		CREATE INDEX ax_referenzstationspunkt_bestehtaus ON ax_referenzstationspunkt USING gin (bestehtaus);
		CREATE INDEX ax_referenzstationspunkt_istteilvon ON ax_referenzstationspunkt USING gin (istteilvon);

		COMMENT ON TABLE ax_referenzstationspunkt IS 'FeatureType: "AX_Referenzstationspunkt"';
		COMMENT ON COLUMN ax_referenzstationspunkt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_referenzstationspunkt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.darstellungshinweis IS 'darstellungshinweis  Boolean 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.frueherepunktnummer IS 'frueherePunktnummer   0..*';
		COMMENT ON COLUMN ax_referenzstationspunkt.gemarkung_gemarkungsnummer IS 'gemarkung|AX_Gemarkung_Schluessel|gemarkungsnummer   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gemarkung_land IS 'gemarkung|AX_Gemarkung_Schluessel|land   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gemeinde IS 'gemeinde|AX_Gemeindekennzeichen|gemeinde   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gemeinde_gemeindeteil IS 'gemeinde|AX_Gemeindekennzeichen|gemeindeteil   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.kreis IS 'gemeinde|AX_Gemeindekennzeichen|kreis   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gemeinde_land IS 'gemeinde|AX_Gemeindekennzeichen|land   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gemeinde_regierungsbezirk IS 'gemeinde|AX_Gemeindekennzeichen|regierungsbezirk   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.internebemerkungen IS 'interneBemerkungen   0..*';
		COMMENT ON COLUMN ax_referenzstationspunkt.katasteramt_land IS 'katasteramt|AX_Dienststelle_Schluessel|land   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.katasteramt_stelle IS 'katasteramt|AX_Dienststelle_Schluessel|stelle   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.land_land IS 'land|AX_Bundesland_Schluessel|land   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.namelagebeschreibung IS 'nameLagebeschreibung   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.nutzerspezifischebemerkungen IS 'nutzerspezifischeBemerkungen   0..*';
		COMMENT ON COLUMN ax_referenzstationspunkt.punktkennung IS 'punktkennung   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.punktvermarkung IS 'punktvermarkung enumeration AX_Marke 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.relativehoehe IS 'relativeHoehe  Length 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.beginn IS 'beginn  DateTime 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.ende IS 'ende  DateTime 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.funkfrequenz IS 'funkfrequenz  Real 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.funktion IS 'funktion enumeration AX_Funktion_Referenzstationspunkt 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssantenne_abbaudatum IS 'gnssAntenne|AX_GNSSAntenne|abbaudatum   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.antennenhoehe IS 'gnssAntenne|AX_GNSSAntenne|antennenhoehe  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.antennenreferenzpunkt IS 'gnssAntenne|AX_GNSSAntenne|antennenreferenzpunkt   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.aufbaudatum IS 'gnssAntenne|AX_GNSSAntenne|aufbaudatum   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.azimutaleabweichung IS 'gnssAntenne|AX_GNSSAntenne|azimutaleAbweichung  Angle 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssantennenundradometyp IS 'gnssAntenne|AX_GNSSAntenne|gnssAntennenUndRadomeTyp   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.hoehenoffsetl1 IS 'gnssAntenne|AX_GNSSAntenne|hoehenoffsetL1  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.hoehenoffsetl2 IS 'gnssAntenne|AX_GNSSAntenne|hoehenoffsetL2  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.seriennummer IS 'gnssAntenne|AX_GNSSAntenne|seriennummer   1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssantenne_zusaetzlicheinformationen IS 'gnssAntenne|AX_GNSSAntenne|zusaetzlicheInformationen   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssempfaenger_abbaudatum IS 'gnssEmpfaenger|AX_GNSSEmpfaenger|abbaudatum   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssempfaenger_aufbaudatum IS 'gnssEmpfaenger|AX_GNSSEmpfaenger|aufbaudatum   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssempfaenger_firmwareversion IS 'gnssEmpfaenger|AX_GNSSEmpfaenger|firmwareversion   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssempfaenger_gnssempfaengertyp IS 'gnssEmpfaenger|AX_GNSSEmpfaenger|gnssEmpfaengertyp   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssempfaenger_seriennummer IS 'gnssEmpfaenger|AX_GNSSEmpfaenger|seriennummer   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnssempfaenger_zusaetzlicheinformationen IS 'gnssEmpfaenger|AX_GNSSEmpfaenger|zusaetzlicheInformationen   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.isdnnummer IS 'isdnNummer   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.offsetl1_east IS 'offsetL1|AX_Offsetkomponenten_Referenzstationspunkt|east  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.offsetl1_height IS 'offsetL1|AX_Offsetkomponenten_Referenzstationspunkt|height  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.offsetl1_north IS 'offsetL1|AX_Offsetkomponenten_Referenzstationspunkt|north  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.offsetl2_east IS 'offsetL2|AX_Offsetkomponenten_Referenzstationspunkt|east  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.offsetl2_height IS 'offsetL2|AX_Offsetkomponenten_Referenzstationspunkt|height  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.offsetl2_north IS 'offsetL2|AX_Offsetkomponenten_Referenzstationspunkt|north  Distance 1';
		COMMENT ON COLUMN ax_referenzstationspunkt.phasenzentrumsvariationl1_zeile IS 'phasenzentrumsvariationL1|AX_Phasenzentrumsvariation_Referenzstationspunkt|zeile  doubleList *';
		COMMENT ON COLUMN ax_referenzstationspunkt.phasenzentrumsvariationl2_zeile IS 'phasenzentrumsvariationL2|AX_Phasenzentrumsvariation_Referenzstationspunkt|zeile  doubleList *';
		COMMENT ON COLUMN ax_referenzstationspunkt.befund IS 'qualitaetsangaben|AX_DQFestpunkt|befund   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.gnsstauglichkeit IS 'qualitaetsangaben|AX_DQFestpunkt|gnssTauglichkeit enumeration AX_GNSSTauglichkeit 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.punktstabilitaet IS 'qualitaetsangaben|AX_DQFestpunkt|punktstabilitaet enumeration AX_Punktstabilitaet 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.ueberwachungsdatum IS 'qualitaetsangaben|AX_DQFestpunkt|ueberwachungsdatum  Date 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.tcpipnummer IS 'tcpipNummer   0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.istidentischmitlfp IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.istidentischmithfp IS 'Assoziation zu: FeatureType AX_Hoehenfestpunkt (ax_hoehenfestpunkt) 0..1';
		COMMENT ON COLUMN ax_referenzstationspunkt.unterschiedlicherbezugspunktmitsfp IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..1';
		CREATE TABLE ax_lagefestpunkt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  darstellungshinweis character varying,
		  frueherepunktnummer character varying[],
		  gemarkung_gemarkungsnummer character varying,
		  gemarkung_land character varying,
		  gemeinde character varying NOT NULL,
		  gemeinde_gemeindeteil character varying,
		  kreis character varying NOT NULL,
		  gemeinde_land character varying NOT NULL,
		  gemeinde_regierungsbezirk character varying,
		  internebemerkungen character varying[],
		  katasteramt_land character varying,
		  katasteramt_stelle character varying,
		  land_land character varying NOT NULL,
		  namelagebeschreibung character varying,
		  nutzerspezifischebemerkungen character varying[],
		  punktkennung character varying NOT NULL,
		  punktvermarkung integer NOT NULL,
		  relativehoehe double precision,
		  funktion integer,
		  hierarchiestufe3d integer,
		  ordnung integer,
		  wertigkeit integer,
		  abstand double precision,
		  messung date,
		  befund character varying,
		  gnsstauglichkeit integer,
		  punktstabilitaet integer,
		  ueberwachungsdatum date,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  istidentischmitrsp character(16),
		  istexzentrumzu character(16),
		  istzentrumzu character(16)[],
		  unterschiedlicherbezugspunktmithfp character(16),
		  istidentischmithfp character(16),
		  unterschiedlicherbezugspunktmitsfp character(16),
		  istidentischmitsfp character(16),
		  istidentischmitgrenzpunkt character(16),
		  unterschiedlicherbezugspunktmitgrenzpunkt character(16),
		  istidentischmitap character(16),
		  unterschiedlicherbezugspunktmitap character(16),
		  unterschiedlicherbezugspunktmitsvp character(16),
		  istidentischmitsvp character(16),
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_lagefestpunkt_gml ON ax_lagefestpunkt USING btree (gml_id,beginnt);
		CREATE INDEX ax_lagefestpunkt_endet ON ax_lagefestpunkt USING btree (endet);
		CREATE INDEX ax_lagefestpunkt_istidentischmitrsp ON ax_lagefestpunkt USING btree (istidentischmitrsp);
		CREATE INDEX ax_lagefestpunkt_istexzentrumzu ON ax_lagefestpunkt USING btree (istexzentrumzu);
		CREATE INDEX ax_lagefestpunkt_istzentrumzu ON ax_lagefestpunkt USING gin (istzentrumzu);
		CREATE INDEX ax_lagefestpunkt_unterschiedlicherbezugspunktmithfp ON ax_lagefestpunkt USING btree (unterschiedlicherbezugspunktmithfp);
		CREATE INDEX ax_lagefestpunkt_istidentischmithfp ON ax_lagefestpunkt USING btree (istidentischmithfp);
		CREATE INDEX ax_lagefestpunkt_unterschiedlicherbezugspunktmitsfp ON ax_lagefestpunkt USING btree (unterschiedlicherbezugspunktmitsfp);
		CREATE INDEX ax_lagefestpunkt_istidentischmitsfp ON ax_lagefestpunkt USING btree (istidentischmitsfp);
		CREATE INDEX ax_lagefestpunkt_istidentischmitgrenzpunkt ON ax_lagefestpunkt USING btree (istidentischmitgrenzpunkt);
		CREATE INDEX ax_lagefestpunkt_unterschiedlicherbezugspunktmitgrenzpunkt ON ax_lagefestpunkt USING btree (unterschiedlicherbezugspunktmitgrenzpunkt);
		CREATE INDEX ax_lagefestpunkt_istidentischmitap ON ax_lagefestpunkt USING btree (istidentischmitap);
		CREATE INDEX ax_lagefestpunkt_unterschiedlicherbezugspunktmitap ON ax_lagefestpunkt USING btree (unterschiedlicherbezugspunktmitap);
		CREATE INDEX ax_lagefestpunkt_unterschiedlicherbezugspunktmitsvp ON ax_lagefestpunkt USING btree (unterschiedlicherbezugspunktmitsvp);
		CREATE INDEX ax_lagefestpunkt_istidentischmitsvp ON ax_lagefestpunkt USING btree (istidentischmitsvp);
		CREATE INDEX ax_lagefestpunkt_bestehtaus ON ax_lagefestpunkt USING gin (bestehtaus);
		CREATE INDEX ax_lagefestpunkt_istteilvon ON ax_lagefestpunkt USING gin (istteilvon);

		COMMENT ON TABLE ax_lagefestpunkt IS 'FeatureType: "AX_Lagefestpunkt"';
		COMMENT ON COLUMN ax_lagefestpunkt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_lagefestpunkt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_lagefestpunkt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_lagefestpunkt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.darstellungshinweis IS 'darstellungshinweis  Boolean 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.frueherepunktnummer IS 'frueherePunktnummer   0..*';
		COMMENT ON COLUMN ax_lagefestpunkt.gemarkung_gemarkungsnummer IS 'gemarkung|AX_Gemarkung_Schluessel|gemarkungsnummer   1';
		COMMENT ON COLUMN ax_lagefestpunkt.gemarkung_land IS 'gemarkung|AX_Gemarkung_Schluessel|land   1';
		COMMENT ON COLUMN ax_lagefestpunkt.gemeinde IS 'gemeinde|AX_Gemeindekennzeichen|gemeinde   1';
		COMMENT ON COLUMN ax_lagefestpunkt.gemeinde_gemeindeteil IS 'gemeinde|AX_Gemeindekennzeichen|gemeindeteil   0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.kreis IS 'gemeinde|AX_Gemeindekennzeichen|kreis   1';
		COMMENT ON COLUMN ax_lagefestpunkt.gemeinde_land IS 'gemeinde|AX_Gemeindekennzeichen|land   1';
		COMMENT ON COLUMN ax_lagefestpunkt.gemeinde_regierungsbezirk IS 'gemeinde|AX_Gemeindekennzeichen|regierungsbezirk   0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.internebemerkungen IS 'interneBemerkungen   0..*';
		COMMENT ON COLUMN ax_lagefestpunkt.katasteramt_land IS 'katasteramt|AX_Dienststelle_Schluessel|land   1';
		COMMENT ON COLUMN ax_lagefestpunkt.katasteramt_stelle IS 'katasteramt|AX_Dienststelle_Schluessel|stelle   1';
		COMMENT ON COLUMN ax_lagefestpunkt.land_land IS 'land|AX_Bundesland_Schluessel|land   1';
		COMMENT ON COLUMN ax_lagefestpunkt.namelagebeschreibung IS 'nameLagebeschreibung   0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.nutzerspezifischebemerkungen IS 'nutzerspezifischeBemerkungen   0..*';
		COMMENT ON COLUMN ax_lagefestpunkt.punktkennung IS 'punktkennung   1';
		COMMENT ON COLUMN ax_lagefestpunkt.punktvermarkung IS 'punktvermarkung enumeration AX_Marke 1';
		COMMENT ON COLUMN ax_lagefestpunkt.relativehoehe IS 'relativeHoehe  Length 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.funktion IS 'funktion enumeration AX_Funktion_Lagefestpunkt 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.hierarchiestufe3d IS 'klassifikation|AX_Klassifikation_Lagefestpunkt|hierarchiestufe3D enumeration AX_Klassifikation_Hierarchiestufe3D_Lagefestpunkt 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.ordnung IS 'klassifikation|AX_Klassifikation_Lagefestpunkt|ordnung enumeration AX_Klassifikation_Ordnung_Lagefestpunkt 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.wertigkeit IS 'klassifikation|AX_Klassifikation_Lagefestpunkt|wertigkeit enumeration AX_Klassifikation_Wertigkeit_Lagefestpunkt 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.abstand IS 'pfeilerhoehe|AX_Pfeilerhoehe_Lagefestpunkt|abstand  Distance 1';
		COMMENT ON COLUMN ax_lagefestpunkt.messung IS 'pfeilerhoehe|AX_Pfeilerhoehe_Lagefestpunkt|messung  Date 1';
		COMMENT ON COLUMN ax_lagefestpunkt.befund IS 'qualitaetsangaben|AX_DQFestpunkt|befund   0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.gnsstauglichkeit IS 'qualitaetsangaben|AX_DQFestpunkt|gnssTauglichkeit enumeration AX_GNSSTauglichkeit 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.punktstabilitaet IS 'qualitaetsangaben|AX_DQFestpunkt|punktstabilitaet enumeration AX_Punktstabilitaet 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.ueberwachungsdatum IS 'qualitaetsangaben|AX_DQFestpunkt|ueberwachungsdatum  Date 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istidentischmitrsp IS 'Assoziation zu: FeatureType AX_Referenzstationspunkt (ax_referenzstationspunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istexzentrumzu IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istzentrumzu IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..*';
		COMMENT ON COLUMN ax_lagefestpunkt.unterschiedlicherbezugspunktmithfp IS 'Assoziation zu: FeatureType AX_Hoehenfestpunkt (ax_hoehenfestpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istidentischmithfp IS 'Assoziation zu: FeatureType AX_Hoehenfestpunkt (ax_hoehenfestpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.unterschiedlicherbezugspunktmitsfp IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istidentischmitsfp IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istidentischmitgrenzpunkt IS 'Assoziation zu: FeatureType AX_Grenzpunkt (ax_grenzpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.unterschiedlicherbezugspunktmitgrenzpunkt IS 'Assoziation zu: FeatureType AX_Grenzpunkt (ax_grenzpunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istidentischmitap IS 'Assoziation zu: FeatureType AX_Aufnahmepunkt (ax_aufnahmepunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.unterschiedlicherbezugspunktmitap IS 'Assoziation zu: FeatureType AX_Aufnahmepunkt (ax_aufnahmepunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.unterschiedlicherbezugspunktmitsvp IS 'Assoziation zu: FeatureType AX_SonstigerVermessungspunkt (ax_sonstigervermessungspunkt) 0..1';
		COMMENT ON COLUMN ax_lagefestpunkt.istidentischmitsvp IS 'Assoziation zu: FeatureType AX_SonstigerVermessungspunkt (ax_sonstigervermessungspunkt) 0..1';
		CREATE TABLE ax_hoehenfestpunkt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  darstellungshinweis character varying,
		  frueherepunktnummer character varying[],
		  gemarkung_gemarkungsnummer character varying,
		  gemarkung_land character varying,
		  gemeinde character varying NOT NULL,
		  gemeinde_gemeindeteil character varying,
		  kreis character varying NOT NULL,
		  gemeinde_land character varying NOT NULL,
		  gemeinde_regierungsbezirk character varying,
		  internebemerkungen character varying[],
		  katasteramt_land character varying,
		  katasteramt_stelle character varying,
		  land_land character varying NOT NULL,
		  namelagebeschreibung character varying,
		  nutzerspezifischebemerkungen character varying[],
		  punktkennung character varying NOT NULL,
		  punktvermarkung integer NOT NULL,
		  relativehoehe double precision,
		  nivlinie character varying[],
		  ordnung integer,
		  befund character varying,
		  gnsstauglichkeit integer,
		  geologischestabilitaet integer,
		  grundwasserschwankung integer,
		  grundwasserstand integer,
		  guetedesbaugrundes integer,
		  guetedesvermarkungstraegers integer,
		  hoehenstabilitaetauswiederholungsmessungen integer,
		  topographieundumwelt integer,
		  vermutetehoehenstabilitaet integer,
		  ueberwachungsdatum date,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  istidentischmitrsp character(16),
		  unterschiedlicherbezugspunktmitlfp character(16),
		  istidentischmitlfp character(16),
		  istidentischmitsfp character(16),
		  unterschiedlicherbezugspunktmitsfp character(16),
		  unterschiedlicherbezugspunktmitap character(16),
		  unterschiedlicherbezugspunktmitsvp character(16),
		  unterschiedlicherbezugspunktmitsp character(16),
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_hoehenfestpunkt_gml ON ax_hoehenfestpunkt USING btree (gml_id,beginnt);
		CREATE INDEX ax_hoehenfestpunkt_endet ON ax_hoehenfestpunkt USING btree (endet);
		CREATE INDEX ax_hoehenfestpunkt_istidentischmitrsp ON ax_hoehenfestpunkt USING btree (istidentischmitrsp);
		CREATE INDEX ax_hoehenfestpunkt_unterschiedlicherbezugspunktmitlfp ON ax_hoehenfestpunkt USING btree (unterschiedlicherbezugspunktmitlfp);
		CREATE INDEX ax_hoehenfestpunkt_istidentischmitlfp ON ax_hoehenfestpunkt USING btree (istidentischmitlfp);
		CREATE INDEX ax_hoehenfestpunkt_istidentischmitsfp ON ax_hoehenfestpunkt USING btree (istidentischmitsfp);
		CREATE INDEX ax_hoehenfestpunkt_unterschiedlicherbezugspunktmitsfp ON ax_hoehenfestpunkt USING btree (unterschiedlicherbezugspunktmitsfp);
		CREATE INDEX ax_hoehenfestpunkt_unterschiedlicherbezugspunktmitap ON ax_hoehenfestpunkt USING btree (unterschiedlicherbezugspunktmitap);
		CREATE INDEX ax_hoehenfestpunkt_unterschiedlicherbezugspunktmitsvp ON ax_hoehenfestpunkt USING btree (unterschiedlicherbezugspunktmitsvp);
		CREATE INDEX ax_hoehenfestpunkt_unterschiedlicherbezugspunktmitsp ON ax_hoehenfestpunkt USING btree (unterschiedlicherbezugspunktmitsp);
		CREATE INDEX ax_hoehenfestpunkt_bestehtaus ON ax_hoehenfestpunkt USING gin (bestehtaus);
		CREATE INDEX ax_hoehenfestpunkt_istteilvon ON ax_hoehenfestpunkt USING gin (istteilvon);

		COMMENT ON TABLE ax_hoehenfestpunkt IS 'FeatureType: "AX_Hoehenfestpunkt"';
		COMMENT ON COLUMN ax_hoehenfestpunkt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_hoehenfestpunkt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.darstellungshinweis IS 'darstellungshinweis  Boolean 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.frueherepunktnummer IS 'frueherePunktnummer   0..*';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gemarkung_gemarkungsnummer IS 'gemarkung|AX_Gemarkung_Schluessel|gemarkungsnummer   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gemarkung_land IS 'gemarkung|AX_Gemarkung_Schluessel|land   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gemeinde IS 'gemeinde|AX_Gemeindekennzeichen|gemeinde   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gemeinde_gemeindeteil IS 'gemeinde|AX_Gemeindekennzeichen|gemeindeteil   0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.kreis IS 'gemeinde|AX_Gemeindekennzeichen|kreis   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gemeinde_land IS 'gemeinde|AX_Gemeindekennzeichen|land   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gemeinde_regierungsbezirk IS 'gemeinde|AX_Gemeindekennzeichen|regierungsbezirk   0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.internebemerkungen IS 'interneBemerkungen   0..*';
		COMMENT ON COLUMN ax_hoehenfestpunkt.katasteramt_land IS 'katasteramt|AX_Dienststelle_Schluessel|land   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.katasteramt_stelle IS 'katasteramt|AX_Dienststelle_Schluessel|stelle   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.land_land IS 'land|AX_Bundesland_Schluessel|land   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.namelagebeschreibung IS 'nameLagebeschreibung   0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.nutzerspezifischebemerkungen IS 'nutzerspezifischeBemerkungen   0..*';
		COMMENT ON COLUMN ax_hoehenfestpunkt.punktkennung IS 'punktkennung   1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.punktvermarkung IS 'punktvermarkung enumeration AX_Marke 1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.relativehoehe IS 'relativeHoehe  Length 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.nivlinie IS 'nivlinie   0..*';
		COMMENT ON COLUMN ax_hoehenfestpunkt.ordnung IS 'ordnung enumeration AX_Ordnung_Hoehenfestpunkt 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.befund IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|befund   0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.gnsstauglichkeit IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|gnssTauglichkeit enumeration AX_GNSSTauglichkeit 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.geologischestabilitaet IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|geologischeStabilitaet enumeration AX_Punktstabilitaet_Hoehenfestpunkt_GeologischeStabilitaet 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.grundwasserschwankung IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|grundwasserschwankung enumeration AX_Punktstabilitaet_Hoehenfestpunkt_Grundwasserschwankung 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.grundwasserstand IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|grundwasserstand enumeration AX_Punktstabilitaet_Hoehenfestpunkt_Grundwasserstand 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.guetedesbaugrundes IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|gueteDesBaugrundes enumeration AX_Punktstabilitaet_Hoehenfestpunkt_GueteDesBaugrundes 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.guetedesvermarkungstraegers IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|gueteDesVermarkungstraegers enumeration AX_Punktstabilitaet_Hoehenfestpunkt_GueteDesVermarkungstraegers 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.hoehenstabilitaetauswiederholungsmessungen IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|hoehenstabilitaetAusWiederholungsmessungen enumeration AX_Punktstabilitaet_Hoehenfestpunkt_HoehenstabilitaetAusWiederholungsmessungen 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.topographieundumwelt IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|topographieUndUmwelt enumeration AX_Punktstabilitaet_Hoehenfestpunkt_TopographieUndUmwelt 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.vermutetehoehenstabilitaet IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|punktstabilitaet|AX_Punktstabilitaet_Hoehenfestpunkt|vermuteteHoehenstabilitaet enumeration AX_Punktstabilitaet_Hoehenfestpunkt_VermuteteHoehenstabilitaet 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.ueberwachungsdatum IS 'qualitaetsangaben|AX_DQHoehenfestpunkt|ueberwachungsdatum  Date 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.istidentischmitrsp IS 'Assoziation zu: FeatureType AX_Referenzstationspunkt (ax_referenzstationspunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.unterschiedlicherbezugspunktmitlfp IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.istidentischmitlfp IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.istidentischmitsfp IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.unterschiedlicherbezugspunktmitsfp IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.unterschiedlicherbezugspunktmitap IS 'Assoziation zu: FeatureType AX_Aufnahmepunkt (ax_aufnahmepunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.unterschiedlicherbezugspunktmitsvp IS 'Assoziation zu: FeatureType AX_SonstigerVermessungspunkt (ax_sonstigervermessungspunkt) 0..1';
		COMMENT ON COLUMN ax_hoehenfestpunkt.unterschiedlicherbezugspunktmitsp IS 'Assoziation zu: FeatureType AX_Sicherungspunkt (ax_sicherungspunkt) 0..1';
		CREATE TABLE ax_schwerefestpunkt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  darstellungshinweis character varying,
		  frueherepunktnummer character varying[],
		  gemarkung_gemarkungsnummer character varying,
		  gemarkung_land character varying,
		  gemeinde character varying NOT NULL,
		  gemeinde_gemeindeteil character varying,
		  kreis character varying NOT NULL,
		  gemeinde_land character varying NOT NULL,
		  gemeinde_regierungsbezirk character varying,
		  internebemerkungen character varying[],
		  katasteramt_land character varying,
		  katasteramt_stelle character varying,
		  land_land character varying NOT NULL,
		  namelagebeschreibung character varying,
		  nutzerspezifischebemerkungen character varying[],
		  punktkennung character varying NOT NULL,
		  punktvermarkung integer NOT NULL,
		  relativehoehe double precision,
		  funktion integer,
		  ordnung integer,
		  befund character varying,
		  gnsstauglichkeit integer,
		  punktstabilitaet integer,
		  ueberwachungsdatum date,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  unterschiedlicherbezugspunktmitrsp character(16),
		  unterschiedlicherbezugspunktmitlfp character(16),
		  istidentischmitlfp character(16),
		  istidentischmithfp character(16),
		  unterschiedlicherbezugspunktmithfp character(16),
		  istzentrumzu character(16)[],
		  istexzentrumzu character(16),
		  istidentischmitgrenzpunkt character(16),
		  istidentischmitap character(16),
		  istidentischmitsvp character(16),
		  istidentischmitsp character(16),
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_schwerefestpunkt_gml ON ax_schwerefestpunkt USING btree (gml_id,beginnt);
		CREATE INDEX ax_schwerefestpunkt_endet ON ax_schwerefestpunkt USING btree (endet);
		CREATE INDEX ax_schwerefestpunkt_unterschiedlicherbezugspunktmitrsp ON ax_schwerefestpunkt USING btree (unterschiedlicherbezugspunktmitrsp);
		CREATE INDEX ax_schwerefestpunkt_unterschiedlicherbezugspunktmitlfp ON ax_schwerefestpunkt USING btree (unterschiedlicherbezugspunktmitlfp);
		CREATE INDEX ax_schwerefestpunkt_istidentischmitlfp ON ax_schwerefestpunkt USING btree (istidentischmitlfp);
		CREATE INDEX ax_schwerefestpunkt_istidentischmithfp ON ax_schwerefestpunkt USING btree (istidentischmithfp);
		CREATE INDEX ax_schwerefestpunkt_unterschiedlicherbezugspunktmithfp ON ax_schwerefestpunkt USING btree (unterschiedlicherbezugspunktmithfp);
		CREATE INDEX ax_schwerefestpunkt_istzentrumzu ON ax_schwerefestpunkt USING gin (istzentrumzu);
		CREATE INDEX ax_schwerefestpunkt_istexzentrumzu ON ax_schwerefestpunkt USING btree (istexzentrumzu);
		CREATE INDEX ax_schwerefestpunkt_istidentischmitgrenzpunkt ON ax_schwerefestpunkt USING btree (istidentischmitgrenzpunkt);
		CREATE INDEX ax_schwerefestpunkt_istidentischmitap ON ax_schwerefestpunkt USING btree (istidentischmitap);
		CREATE INDEX ax_schwerefestpunkt_istidentischmitsvp ON ax_schwerefestpunkt USING btree (istidentischmitsvp);
		CREATE INDEX ax_schwerefestpunkt_istidentischmitsp ON ax_schwerefestpunkt USING btree (istidentischmitsp);
		CREATE INDEX ax_schwerefestpunkt_bestehtaus ON ax_schwerefestpunkt USING gin (bestehtaus);
		CREATE INDEX ax_schwerefestpunkt_istteilvon ON ax_schwerefestpunkt USING gin (istteilvon);

		COMMENT ON TABLE ax_schwerefestpunkt IS 'FeatureType: "AX_Schwerefestpunkt"';
		COMMENT ON COLUMN ax_schwerefestpunkt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_schwerefestpunkt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_schwerefestpunkt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_schwerefestpunkt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.darstellungshinweis IS 'darstellungshinweis  Boolean 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.frueherepunktnummer IS 'frueherePunktnummer   0..*';
		COMMENT ON COLUMN ax_schwerefestpunkt.gemarkung_gemarkungsnummer IS 'gemarkung|AX_Gemarkung_Schluessel|gemarkungsnummer   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.gemarkung_land IS 'gemarkung|AX_Gemarkung_Schluessel|land   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.gemeinde IS 'gemeinde|AX_Gemeindekennzeichen|gemeinde   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.gemeinde_gemeindeteil IS 'gemeinde|AX_Gemeindekennzeichen|gemeindeteil   0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.kreis IS 'gemeinde|AX_Gemeindekennzeichen|kreis   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.gemeinde_land IS 'gemeinde|AX_Gemeindekennzeichen|land   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.gemeinde_regierungsbezirk IS 'gemeinde|AX_Gemeindekennzeichen|regierungsbezirk   0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.internebemerkungen IS 'interneBemerkungen   0..*';
		COMMENT ON COLUMN ax_schwerefestpunkt.katasteramt_land IS 'katasteramt|AX_Dienststelle_Schluessel|land   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.katasteramt_stelle IS 'katasteramt|AX_Dienststelle_Schluessel|stelle   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.land_land IS 'land|AX_Bundesland_Schluessel|land   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.namelagebeschreibung IS 'nameLagebeschreibung   0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.nutzerspezifischebemerkungen IS 'nutzerspezifischeBemerkungen   0..*';
		COMMENT ON COLUMN ax_schwerefestpunkt.punktkennung IS 'punktkennung   1';
		COMMENT ON COLUMN ax_schwerefestpunkt.punktvermarkung IS 'punktvermarkung enumeration AX_Marke 1';
		COMMENT ON COLUMN ax_schwerefestpunkt.relativehoehe IS 'relativeHoehe  Length 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.funktion IS 'funktion enumeration AX_Funktion_Schwerefestpunkt 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.ordnung IS 'ordnung enumeration AX_Ordnung_Schwerefestpunkt 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.befund IS 'qualitaetsangaben|AX_DQFestpunkt|befund   0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.gnsstauglichkeit IS 'qualitaetsangaben|AX_DQFestpunkt|gnssTauglichkeit enumeration AX_GNSSTauglichkeit 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.punktstabilitaet IS 'qualitaetsangaben|AX_DQFestpunkt|punktstabilitaet enumeration AX_Punktstabilitaet 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.ueberwachungsdatum IS 'qualitaetsangaben|AX_DQFestpunkt|ueberwachungsdatum  Date 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.unterschiedlicherbezugspunktmitrsp IS 'Assoziation zu: FeatureType AX_Referenzstationspunkt (ax_referenzstationspunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.unterschiedlicherbezugspunktmitlfp IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istidentischmitlfp IS 'Assoziation zu: FeatureType AX_Lagefestpunkt (ax_lagefestpunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istidentischmithfp IS 'Assoziation zu: FeatureType AX_Hoehenfestpunkt (ax_hoehenfestpunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.unterschiedlicherbezugspunktmithfp IS 'Assoziation zu: FeatureType AX_Hoehenfestpunkt (ax_hoehenfestpunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istzentrumzu IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..*';
		COMMENT ON COLUMN ax_schwerefestpunkt.istexzentrumzu IS 'Assoziation zu: FeatureType AX_Schwerefestpunkt (ax_schwerefestpunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istidentischmitgrenzpunkt IS 'Assoziation zu: FeatureType AX_Grenzpunkt (ax_grenzpunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istidentischmitap IS 'Assoziation zu: FeatureType AX_Aufnahmepunkt (ax_aufnahmepunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istidentischmitsvp IS 'Assoziation zu: FeatureType AX_SonstigerVermessungspunkt (ax_sonstigervermessungspunkt) 0..1';
		COMMENT ON COLUMN ax_schwerefestpunkt.istidentischmitsp IS 'Assoziation zu: FeatureType AX_Sicherungspunkt (ax_sicherungspunkt) 0..1';
		CREATE TABLE ax_kanal (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  gewaesserkennzahl character varying,
		  identnummer character varying,
		  name character varying,
		  schifffahrtskategorie integer,
		  widmung integer,
		  zweitname character varying,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_kanal_gml ON ax_kanal USING btree (gml_id,beginnt);
		CREATE INDEX ax_kanal_endet ON ax_kanal USING btree (endet);
		CREATE INDEX ax_kanal_bestehtaus ON ax_kanal USING gin (bestehtaus);
		CREATE INDEX ax_kanal_istteilvon ON ax_kanal USING gin (istteilvon);

		COMMENT ON TABLE ax_kanal IS 'FeatureType: "AX_Kanal"';
		COMMENT ON COLUMN ax_kanal.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_kanal.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_kanal.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_kanal.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_kanal.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_kanal.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_kanal.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_kanal.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_kanal.gewaesserkennzahl IS 'gewaesserkennzahl   0..1';
		COMMENT ON COLUMN ax_kanal.identnummer IS 'identnummer   0..1';
		COMMENT ON COLUMN ax_kanal.name IS 'name   0..1';
		COMMENT ON COLUMN ax_kanal.schifffahrtskategorie IS 'schifffahrtskategorie enumeration AX_Schifffahrtskategorie_Kanal 0..1';
		COMMENT ON COLUMN ax_kanal.widmung IS 'widmung enumeration AX_Widmung_Kanal 0..1';
		COMMENT ON COLUMN ax_kanal.zweitname IS 'zweitname   0..1';
		CREATE TABLE ax_wasserlauf (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  gewaesserkennzahl character varying,
		  identnummer character varying,
		  name character varying,
		  schifffahrtskategorie integer,
		  widmung integer,
		  zweitname character varying,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_wasserlauf_gml ON ax_wasserlauf USING btree (gml_id,beginnt);
		CREATE INDEX ax_wasserlauf_endet ON ax_wasserlauf USING btree (endet);
		CREATE INDEX ax_wasserlauf_bestehtaus ON ax_wasserlauf USING gin (bestehtaus);
		CREATE INDEX ax_wasserlauf_istteilvon ON ax_wasserlauf USING gin (istteilvon);

		COMMENT ON TABLE ax_wasserlauf IS 'FeatureType: "AX_Wasserlauf"';
		COMMENT ON COLUMN ax_wasserlauf.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_wasserlauf.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_wasserlauf.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_wasserlauf.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_wasserlauf.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_wasserlauf.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_wasserlauf.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_wasserlauf.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_wasserlauf.gewaesserkennzahl IS 'gewaesserkennzahl   0..1';
		COMMENT ON COLUMN ax_wasserlauf.identnummer IS 'identnummer   0..1';
		COMMENT ON COLUMN ax_wasserlauf.name IS 'name   0..1';
		COMMENT ON COLUMN ax_wasserlauf.schifffahrtskategorie IS 'schifffahrtskategorie enumeration AX_Schifffahrtskategorie 0..1';
		COMMENT ON COLUMN ax_wasserlauf.widmung IS 'widmung enumeration AX_Widmung_Wasserlauf 0..1';
		COMMENT ON COLUMN ax_wasserlauf.zweitname IS 'zweitname   0..1';
		CREATE TABLE ax_strasse (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying[],
		  fahrbahntrennung integer,
		  internationalebedeutung integer,
		  name character varying,
		  strassenschluessel character varying[],
		  widmung integer NOT NULL,
		  zweitname character varying,
		  bestehtaus character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_strasse_gml ON ax_strasse USING btree (gml_id,beginnt);
		CREATE INDEX ax_strasse_endet ON ax_strasse USING btree (endet);
		CREATE INDEX ax_strasse_bestehtaus ON ax_strasse USING gin (bestehtaus);
		CREATE INDEX ax_strasse_istteilvon ON ax_strasse USING gin (istteilvon);

		COMMENT ON TABLE ax_strasse IS 'FeatureType: "AX_Strasse"';
		COMMENT ON COLUMN ax_strasse.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_strasse.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_strasse.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_strasse.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_strasse.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_strasse.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_strasse.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_strasse.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_strasse.bezeichnung IS 'bezeichnung   0..*';
		COMMENT ON COLUMN ax_strasse.fahrbahntrennung IS 'fahrbahntrennung enumeration AX_Fahrbahntrennung_Strasse 0..1';
		COMMENT ON COLUMN ax_strasse.internationalebedeutung IS 'internationaleBedeutung enumeration AX_InternationaleBedeutung_Strasse 0..1';
		COMMENT ON COLUMN ax_strasse.name IS 'name   0..1';
		COMMENT ON COLUMN ax_strasse.strassenschluessel IS 'strassenschluessel   0..*';
		COMMENT ON COLUMN ax_strasse.widmung IS 'widmung enumeration AX_Widmung_Strasse 1';
		COMMENT ON COLUMN ax_strasse.zweitname IS 'zweitname   0..1';
		CREATE TABLE ap_fpo (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  art character varying,
		  darstellungsprioritaet integer,
		  signaturnummer character varying,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  dientzurdarstellungvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ap_fpo_gml ON ap_fpo USING btree (gml_id,beginnt);
		CREATE INDEX ap_fpo_endet ON ap_fpo USING btree (endet);
		PERFORM AddGeometryColumn('ap_fpo', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ap_fpo_wkb_geometry_idx ON ap_fpo USING gist (wkb_geometry);
		CREATE INDEX ap_fpo_dientzurdarstellungvon ON ap_fpo USING gin (dientzurdarstellungvon);
		CREATE INDEX ap_fpo_istabgeleitetaus ON ap_fpo USING gin (istabgeleitetaus);
		CREATE INDEX ap_fpo_traegtbeizu ON ap_fpo USING gin (traegtbeizu);
		CREATE INDEX ap_fpo_hatdirektunten ON ap_fpo USING gin (hatdirektunten);
		CREATE INDEX ap_fpo_istteilvon ON ap_fpo USING gin (istteilvon);

		COMMENT ON TABLE ap_fpo IS 'FeatureType: "AP_FPO"';
		COMMENT ON COLUMN ap_fpo.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ap_fpo.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ap_fpo.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ap_fpo.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ap_fpo.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ap_fpo.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ap_fpo.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ap_fpo.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ap_fpo.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ap_fpo.art IS 'art  CharacterString 0..1';
		COMMENT ON COLUMN ap_fpo.darstellungsprioritaet IS 'darstellungsprioritaet  Integer 0..1';
		COMMENT ON COLUMN ap_fpo.signaturnummer IS 'signaturnummer  CharacterString 0..1';
		COMMENT ON COLUMN ap_fpo.dientzurdarstellungvon IS 'Assoziation zu: FeatureType AA_Objekt (aa_objekt) 0..*';
		CREATE TABLE ax_polder (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  artdespolders integer,
		  funktion character varying,
		  name character varying,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_polder_gml ON ax_polder USING btree (gml_id,beginnt);
		CREATE INDEX ax_polder_endet ON ax_polder USING btree (endet);
		PERFORM AddGeometryColumn('ax_polder', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_polder_wkb_geometry_idx ON ax_polder USING gist (wkb_geometry);
		CREATE INDEX ax_polder_istabgeleitetaus ON ax_polder USING gin (istabgeleitetaus);
		CREATE INDEX ax_polder_traegtbeizu ON ax_polder USING gin (traegtbeizu);
		CREATE INDEX ax_polder_hatdirektunten ON ax_polder USING gin (hatdirektunten);
		CREATE INDEX ax_polder_istteilvon ON ax_polder USING gin (istteilvon);

		COMMENT ON TABLE ax_polder IS 'FeatureType: "AX_Polder"';
		COMMENT ON COLUMN ax_polder.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_polder.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_polder.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_polder.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_polder.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_polder.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_polder.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_polder.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_polder.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_polder.artdespolders IS 'artDesPolders enumeration AX_ArtDesPolders 0..1';
		COMMENT ON COLUMN ax_polder.funktion IS 'funktion enumeration AX_Funktion_Polder 0..1';
		COMMENT ON COLUMN ax_polder.name IS 'name   0..1';
		COMMENT ON COLUMN ax_polder.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_polder.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_polder.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_polder.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_polder.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_polder.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_polder.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_polder.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_kondominium (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_kondominium_gml ON ax_kondominium USING btree (gml_id,beginnt);
		CREATE INDEX ax_kondominium_endet ON ax_kondominium USING btree (endet);
		PERFORM AddGeometryColumn('ax_kondominium', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_kondominium_wkb_geometry_idx ON ax_kondominium USING gist (wkb_geometry);
		CREATE INDEX ax_kondominium_istabgeleitetaus ON ax_kondominium USING gin (istabgeleitetaus);
		CREATE INDEX ax_kondominium_traegtbeizu ON ax_kondominium USING gin (traegtbeizu);
		CREATE INDEX ax_kondominium_hatdirektunten ON ax_kondominium USING gin (hatdirektunten);
		CREATE INDEX ax_kondominium_istteilvon ON ax_kondominium USING gin (istteilvon);

		COMMENT ON TABLE ax_kondominium IS 'FeatureType: "AX_Kondominium"';
		COMMENT ON COLUMN ax_kondominium.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_kondominium.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_kondominium.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_kondominium.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_kondominium.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_kondominium.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_kondominium.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_kondominium.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_kondominium.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		CREATE TABLE ax_aussparungsflaeche (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  artderaussparung integer NOT NULL,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_aussparungsflaeche_gml ON ax_aussparungsflaeche USING btree (gml_id,beginnt);
		CREATE INDEX ax_aussparungsflaeche_endet ON ax_aussparungsflaeche USING btree (endet);
		PERFORM AddGeometryColumn('ax_aussparungsflaeche', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_aussparungsflaeche_wkb_geometry_idx ON ax_aussparungsflaeche USING gist (wkb_geometry);
		CREATE INDEX ax_aussparungsflaeche_istabgeleitetaus ON ax_aussparungsflaeche USING gin (istabgeleitetaus);
		CREATE INDEX ax_aussparungsflaeche_traegtbeizu ON ax_aussparungsflaeche USING gin (traegtbeizu);
		CREATE INDEX ax_aussparungsflaeche_hatdirektunten ON ax_aussparungsflaeche USING gin (hatdirektunten);
		CREATE INDEX ax_aussparungsflaeche_istteilvon ON ax_aussparungsflaeche USING gin (istteilvon);

		COMMENT ON TABLE ax_aussparungsflaeche IS 'FeatureType: "AX_Aussparungsflaeche"';
		COMMENT ON COLUMN ax_aussparungsflaeche.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_aussparungsflaeche.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_aussparungsflaeche.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_aussparungsflaeche.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_aussparungsflaeche.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_aussparungsflaeche.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_aussparungsflaeche.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_aussparungsflaeche.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_aussparungsflaeche.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_aussparungsflaeche.artderaussparung IS 'artDerAussparung enumeration AX_ArtDerAussparung 1';
		CREATE TABLE ax_hafen (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  hafenkategorie integer,
		  name character varying,
		  nutzung integer,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_hafen_gml ON ax_hafen USING btree (gml_id,beginnt);
		CREATE INDEX ax_hafen_endet ON ax_hafen USING btree (endet);
		PERFORM AddGeometryColumn('ax_hafen', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_hafen_wkb_geometry_idx ON ax_hafen USING gist (wkb_geometry);
		CREATE INDEX ax_hafen_istabgeleitetaus ON ax_hafen USING gin (istabgeleitetaus);
		CREATE INDEX ax_hafen_traegtbeizu ON ax_hafen USING gin (traegtbeizu);
		CREATE INDEX ax_hafen_hatdirektunten ON ax_hafen USING gin (hatdirektunten);
		CREATE INDEX ax_hafen_istteilvon ON ax_hafen USING gin (istteilvon);

		COMMENT ON TABLE ax_hafen IS 'FeatureType: "AX_Hafen"';
		COMMENT ON COLUMN ax_hafen.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_hafen.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_hafen.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_hafen.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_hafen.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_hafen.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_hafen.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_hafen.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_hafen.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_hafen.hafenkategorie IS 'hafenkategorie enumeration AX_Hafenkategorie_Hafen 0..1';
		COMMENT ON COLUMN ax_hafen.name IS 'name   0..1';
		COMMENT ON COLUMN ax_hafen.nutzung IS 'nutzung enumeration AX_Nutzung_Hafen 0..1';
		COMMENT ON COLUMN ax_hafen.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_hafen.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_hafen.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_hafen.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_hafen.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_hafen.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_hafen.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_testgelaende (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  name character varying,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_testgelaende_gml ON ax_testgelaende USING btree (gml_id,beginnt);
		CREATE INDEX ax_testgelaende_endet ON ax_testgelaende USING btree (endet);
		PERFORM AddGeometryColumn('ax_testgelaende', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_testgelaende_wkb_geometry_idx ON ax_testgelaende USING gist (wkb_geometry);
		CREATE INDEX ax_testgelaende_istabgeleitetaus ON ax_testgelaende USING gin (istabgeleitetaus);
		CREATE INDEX ax_testgelaende_traegtbeizu ON ax_testgelaende USING gin (traegtbeizu);
		CREATE INDEX ax_testgelaende_hatdirektunten ON ax_testgelaende USING gin (hatdirektunten);
		CREATE INDEX ax_testgelaende_istteilvon ON ax_testgelaende USING gin (istteilvon);

		COMMENT ON TABLE ax_testgelaende IS 'FeatureType: "AX_Testgelaende"';
		COMMENT ON COLUMN ax_testgelaende.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_testgelaende.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_testgelaende.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_testgelaende.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_testgelaende.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_testgelaende.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_testgelaende.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_testgelaende.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_testgelaende.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_testgelaende.name IS 'name   0..1';
		COMMENT ON COLUMN ax_testgelaende.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_testgelaende.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_testgelaende.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_testgelaende.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_testgelaende.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_testgelaende.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_testgelaende.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_schleuse (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying,
		  konstruktionsmerkmalbauart integer,
		  name character varying,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  zustand integer,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_schleuse_gml ON ax_schleuse USING btree (gml_id,beginnt);
		CREATE INDEX ax_schleuse_endet ON ax_schleuse USING btree (endet);
		PERFORM AddGeometryColumn('ax_schleuse', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_schleuse_wkb_geometry_idx ON ax_schleuse USING gist (wkb_geometry);
		CREATE INDEX ax_schleuse_istabgeleitetaus ON ax_schleuse USING gin (istabgeleitetaus);
		CREATE INDEX ax_schleuse_traegtbeizu ON ax_schleuse USING gin (traegtbeizu);
		CREATE INDEX ax_schleuse_hatdirektunten ON ax_schleuse USING gin (hatdirektunten);
		CREATE INDEX ax_schleuse_istteilvon ON ax_schleuse USING gin (istteilvon);

		COMMENT ON TABLE ax_schleuse IS 'FeatureType: "AX_Schleuse"';
		COMMENT ON COLUMN ax_schleuse.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_schleuse.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_schleuse.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_schleuse.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_schleuse.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_schleuse.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_schleuse.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_schleuse.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_schleuse.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_schleuse.bezeichnung IS 'bezeichnung   0..1';
		COMMENT ON COLUMN ax_schleuse.konstruktionsmerkmalbauart IS 'konstruktionsmerkmalBauart enumeration AX_KonstruktionsmerkmalBauart_Schleuse 0..1';
		COMMENT ON COLUMN ax_schleuse.name IS 'name   0..1';
		COMMENT ON COLUMN ax_schleuse.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_schleuse.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_schleuse.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_schleuse.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_schleuse.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_schleuse.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_schleuse.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_schleuse.zustand IS 'zustand enumeration AX_Zustand_Schleuse 0..1';
		CREATE TABLE ax_ortslage (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  name character varying,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  zweitname character varying,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_ortslage_gml ON ax_ortslage USING btree (gml_id,beginnt);
		CREATE INDEX ax_ortslage_endet ON ax_ortslage USING btree (endet);
		PERFORM AddGeometryColumn('ax_ortslage', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_ortslage_wkb_geometry_idx ON ax_ortslage USING gist (wkb_geometry);
		CREATE INDEX ax_ortslage_istabgeleitetaus ON ax_ortslage USING gin (istabgeleitetaus);
		CREATE INDEX ax_ortslage_traegtbeizu ON ax_ortslage USING gin (traegtbeizu);
		CREATE INDEX ax_ortslage_hatdirektunten ON ax_ortslage USING gin (hatdirektunten);
		CREATE INDEX ax_ortslage_istteilvon ON ax_ortslage USING gin (istteilvon);

		COMMENT ON TABLE ax_ortslage IS 'FeatureType: "AX_Ortslage"';
		COMMENT ON COLUMN ax_ortslage.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_ortslage.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_ortslage.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_ortslage.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_ortslage.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_ortslage.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_ortslage.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_ortslage.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_ortslage.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_ortslage.name IS 'name   0..1';
		COMMENT ON COLUMN ax_ortslage.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_ortslage.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_ortslage.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_ortslage.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_ortslage.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_ortslage.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_ortslage.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_ortslage.zweitname IS 'zweitname   0..1';
		CREATE TABLE ax_grenzuebergang (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  name character varying,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_grenzuebergang_gml ON ax_grenzuebergang USING btree (gml_id,beginnt);
		CREATE INDEX ax_grenzuebergang_endet ON ax_grenzuebergang USING btree (endet);
		PERFORM AddGeometryColumn('ax_grenzuebergang', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_grenzuebergang_wkb_geometry_idx ON ax_grenzuebergang USING gist (wkb_geometry);
		CREATE INDEX ax_grenzuebergang_istabgeleitetaus ON ax_grenzuebergang USING gin (istabgeleitetaus);
		CREATE INDEX ax_grenzuebergang_traegtbeizu ON ax_grenzuebergang USING gin (traegtbeizu);
		CREATE INDEX ax_grenzuebergang_hatdirektunten ON ax_grenzuebergang USING gin (hatdirektunten);
		CREATE INDEX ax_grenzuebergang_istteilvon ON ax_grenzuebergang USING gin (istteilvon);

		COMMENT ON TABLE ax_grenzuebergang IS 'FeatureType: "AX_Grenzuebergang"';
		COMMENT ON COLUMN ax_grenzuebergang.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_grenzuebergang.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_grenzuebergang.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_grenzuebergang.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_grenzuebergang.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.name IS 'name   0..1';
		COMMENT ON COLUMN ax_grenzuebergang.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_grenzuebergang.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_grenzuebergang.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_grenzuebergang.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_grenzuebergang.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_insel (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  name character varying,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_insel_gml ON ax_insel USING btree (gml_id,beginnt);
		CREATE INDEX ax_insel_endet ON ax_insel USING btree (endet);
		PERFORM AddGeometryColumn('ax_insel', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_insel_wkb_geometry_idx ON ax_insel USING gist (wkb_geometry);
		CREATE INDEX ax_insel_istabgeleitetaus ON ax_insel USING gin (istabgeleitetaus);
		CREATE INDEX ax_insel_traegtbeizu ON ax_insel USING gin (traegtbeizu);
		CREATE INDEX ax_insel_hatdirektunten ON ax_insel USING gin (hatdirektunten);
		CREATE INDEX ax_insel_istteilvon ON ax_insel USING gin (istteilvon);

		COMMENT ON TABLE ax_insel IS 'FeatureType: "AX_Insel"';
		COMMENT ON COLUMN ax_insel.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_insel.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_insel.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_insel.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_insel.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_insel.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_insel.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_insel.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_insel.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_insel.name IS 'name   0..1';
		CREATE TABLE ax_gewann (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  name character varying NOT NULL,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_gewann_gml ON ax_gewann USING btree (gml_id,beginnt);
		CREATE INDEX ax_gewann_endet ON ax_gewann USING btree (endet);
		PERFORM AddGeometryColumn('ax_gewann', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_gewann_wkb_geometry_idx ON ax_gewann USING gist (wkb_geometry);
		CREATE INDEX ax_gewann_istabgeleitetaus ON ax_gewann USING gin (istabgeleitetaus);
		CREATE INDEX ax_gewann_traegtbeizu ON ax_gewann USING gin (traegtbeizu);
		CREATE INDEX ax_gewann_hatdirektunten ON ax_gewann USING gin (hatdirektunten);
		CREATE INDEX ax_gewann_istteilvon ON ax_gewann USING gin (istteilvon);

		COMMENT ON TABLE ax_gewann IS 'FeatureType: "AX_Gewann"';
		COMMENT ON COLUMN ax_gewann.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_gewann.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_gewann.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_gewann.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_gewann.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_gewann.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_gewann.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_gewann.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_gewann.wkb_geometry IS 'wkb_geometry  GM_Object 0..1';
		COMMENT ON COLUMN ax_gewann.name IS 'name   1';
		CREATE TABLE ax_abschnitt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying NOT NULL,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_abschnitt_gml ON ax_abschnitt USING btree (gml_id,beginnt);
		CREATE INDEX ax_abschnitt_endet ON ax_abschnitt USING btree (endet);
		PERFORM AddGeometryColumn('ax_abschnitt', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_abschnitt_wkb_geometry_idx ON ax_abschnitt USING gist (wkb_geometry);
		CREATE INDEX ax_abschnitt_istabgeleitetaus ON ax_abschnitt USING gin (istabgeleitetaus);
		CREATE INDEX ax_abschnitt_traegtbeizu ON ax_abschnitt USING gin (traegtbeizu);
		CREATE INDEX ax_abschnitt_hatdirektunten ON ax_abschnitt USING gin (hatdirektunten);
		CREATE INDEX ax_abschnitt_istteilvon ON ax_abschnitt USING gin (istteilvon);

		COMMENT ON TABLE ax_abschnitt IS 'FeatureType: "AX_Abschnitt"';
		COMMENT ON COLUMN ax_abschnitt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_abschnitt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_abschnitt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_abschnitt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_abschnitt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_abschnitt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_abschnitt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_abschnitt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_abschnitt.wkb_geometry IS 'wkb_geometry  GM_Curve 0..1';
		COMMENT ON COLUMN ax_abschnitt.bezeichnung IS 'bezeichnung   1';
		COMMENT ON COLUMN ax_abschnitt.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_abschnitt.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_abschnitt.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_abschnitt.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_abschnitt.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_abschnitt.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_abschnitt.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_ast (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  bezeichnung character varying NOT NULL,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_ast_gml ON ax_ast USING btree (gml_id,beginnt);
		CREATE INDEX ax_ast_endet ON ax_ast USING btree (endet);
		PERFORM AddGeometryColumn('ax_ast', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_ast_wkb_geometry_idx ON ax_ast USING gist (wkb_geometry);
		CREATE INDEX ax_ast_istabgeleitetaus ON ax_ast USING gin (istabgeleitetaus);
		CREATE INDEX ax_ast_traegtbeizu ON ax_ast USING gin (traegtbeizu);
		CREATE INDEX ax_ast_hatdirektunten ON ax_ast USING gin (hatdirektunten);
		CREATE INDEX ax_ast_istteilvon ON ax_ast USING gin (istteilvon);

		COMMENT ON TABLE ax_ast IS 'FeatureType: "AX_Ast"';
		COMMENT ON COLUMN ax_ast.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_ast.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_ast.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_ast.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_ast.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_ast.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_ast.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_ast.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_ast.wkb_geometry IS 'wkb_geometry  GM_Curve 0..1';
		COMMENT ON COLUMN ax_ast.bezeichnung IS 'bezeichnung   1';
		COMMENT ON COLUMN ax_ast.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_ast.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_ast.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|positionName  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_phone IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|phone  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_address IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|address  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_onlineresource IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|onlineResource  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_hoursofservice IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|hoursOfService  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_contactinstructions IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|contactInfo|CI_Contact|contactInstructions  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_role IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|role enumeration CI_RoleCode 1';
		COMMENT ON COLUMN ax_ast.processstep_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.processstep_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_ast.processstep_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|source|LI_Source|sourceStep  CharacterString 0..*';
		COMMENT ON COLUMN ax_ast.herkunft_source_source_ax_datenerhebung IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|AX_Datenerhebung  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.herkunft_source_source_scaledenominator IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|scaleDenominator  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.herkunft_source_source_sourcereferencesystem IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceReferenceSystem  CharacterString 0..1';
		COMMENT ON COLUMN ax_ast.herkunft_source_source_sourceextent IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceExtent  CharacterString 0..*';
		COMMENT ON COLUMN ax_ast.herkunft_source_source_sourcestep IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|source|LI_Source|sourceStep  CharacterString 0..*';
		CREATE TABLE ax_geripplinie (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  artdergeripplinie integer,
		  erfassung_datetime character(20),
		  ax_dqerfassungsmethode integer NOT NULL,
		  identifikation integer NOT NULL,
		  hoehengenauigkeit_nameofmeasure character varying[],
		  hoehengenauigkeit_measureidentification character varying,
		  hoehengenauigkeit_measuredescription character varying,
		  hoehengenauigkeit_evaluationmethodtype character varying,
		  hoehengenauigkeit_evaluationmethoddescription character varying,
		  hoehengenauigkeit_evaluationprocedure character varying,
		  hoehengenauigkeit_datetime character(20)[],
		  hoehengenauigkeit_result character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_geripplinie_gml ON ax_geripplinie USING btree (gml_id,beginnt);
		CREATE INDEX ax_geripplinie_endet ON ax_geripplinie USING btree (endet);
		PERFORM AddGeometryColumn('ax_geripplinie', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_geripplinie_wkb_geometry_idx ON ax_geripplinie USING gist (wkb_geometry);
		CREATE INDEX ax_geripplinie_istabgeleitetaus ON ax_geripplinie USING gin (istabgeleitetaus);
		CREATE INDEX ax_geripplinie_traegtbeizu ON ax_geripplinie USING gin (traegtbeizu);
		CREATE INDEX ax_geripplinie_hatdirektunten ON ax_geripplinie USING gin (hatdirektunten);
		CREATE INDEX ax_geripplinie_istteilvon ON ax_geripplinie USING gin (istteilvon);

		COMMENT ON TABLE ax_geripplinie IS 'FeatureType: "AX_Geripplinie"';
		COMMENT ON COLUMN ax_geripplinie.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_geripplinie.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_geripplinie.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_geripplinie.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_geripplinie.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_geripplinie.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_geripplinie.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_geripplinie.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_geripplinie.wkb_geometry IS 'wkb_geometry  GM_MultiCurve 0..1';
		COMMENT ON COLUMN ax_geripplinie.artdergeripplinie IS 'artDerGeripplinie enumeration AX_ArtDerGeripplinie 0..1';
		COMMENT ON COLUMN ax_geripplinie.erfassung_datetime IS 'erfassung|AX_Erfassung_DGM|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_geripplinie.ax_dqerfassungsmethode IS 'erfassung|AX_Erfassung_DGM|AX_DQErfassungsmethode enumeration AX_DQErfassungsmethode 1';
		COMMENT ON COLUMN ax_geripplinie.identifikation IS 'erfassung|AX_Erfassung_DGM|identifikation enumeration AX_Identifikation 1';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_nameofmeasure IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|nameOfMeasure  CharacterString 0..*';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_measureidentification IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|measureIdentification  MD_Identifier 0..1';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_measuredescription IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|measureDescription  CharacterString 0..1';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_evaluationmethodtype IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationMethodType  DQ_EvaluationMethodTypeCode 0..1';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_evaluationmethoddescription IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationMethodDescription  CharacterString 0..1';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_evaluationprocedure IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationProcedure  CI_Citation 0..1';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_datetime IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|dateTime  DateTime 0..*';
		COMMENT ON COLUMN ax_geripplinie.hoehengenauigkeit_result IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|result  DQ_Result 1..*';
		CREATE TABLE ax_gewaesserbegrenzung (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  besondereartdergewaesserbegrenzung integer,
		  erfassunggewaesserbegrenzung_datetime character(20),
		  ax_dqerfassungsmethodegewaesserbegrenzung integer NOT NULL,
		  identifikation integer NOT NULL,
		  hoehengenauigkeit_nameofmeasure character varying[],
		  hoehengenauigkeit_measureidentification character varying,
		  hoehengenauigkeit_measuredescription character varying,
		  hoehengenauigkeit_evaluationmethodtype character varying,
		  hoehengenauigkeit_evaluationmethoddescription character varying,
		  hoehengenauigkeit_evaluationprocedure character varying,
		  hoehengenauigkeit_datetime character(20)[],
		  hoehengenauigkeit_result character varying[],
		  ursprung integer,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_gewaesserbegrenzung_gml ON ax_gewaesserbegrenzung USING btree (gml_id,beginnt);
		CREATE INDEX ax_gewaesserbegrenzung_endet ON ax_gewaesserbegrenzung USING btree (endet);
		PERFORM AddGeometryColumn('ax_gewaesserbegrenzung', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_gewaesserbegrenzung_wkb_geometry_idx ON ax_gewaesserbegrenzung USING gist (wkb_geometry);
		CREATE INDEX ax_gewaesserbegrenzung_istabgeleitetaus ON ax_gewaesserbegrenzung USING gin (istabgeleitetaus);
		CREATE INDEX ax_gewaesserbegrenzung_traegtbeizu ON ax_gewaesserbegrenzung USING gin (traegtbeizu);
		CREATE INDEX ax_gewaesserbegrenzung_hatdirektunten ON ax_gewaesserbegrenzung USING gin (hatdirektunten);
		CREATE INDEX ax_gewaesserbegrenzung_istteilvon ON ax_gewaesserbegrenzung USING gin (istteilvon);

		COMMENT ON TABLE ax_gewaesserbegrenzung IS 'FeatureType: "AX_Gewaesserbegrenzung"';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.wkb_geometry IS 'wkb_geometry  GM_MultiCurve 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.besondereartdergewaesserbegrenzung IS 'besondereArtDerGewaesserbegrenzung enumeration AX_BesondereArtDerGewaesserbegrenzung 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.erfassunggewaesserbegrenzung_datetime IS 'erfassungGewaesserbegrenzung|AX_ErfassungGewaesserbegrenzung|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.ax_dqerfassungsmethodegewaesserbegrenzung IS 'erfassungGewaesserbegrenzung|AX_ErfassungGewaesserbegrenzung|AX_DQErfassungsmethodeGewaesserbegrenzung enumeration AX_DQErfassungsmethodeGewaesserbegrenzung 1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.identifikation IS 'erfassungGewaesserbegrenzung|AX_ErfassungGewaesserbegrenzung|identifikation enumeration AX_Identifikation 1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_nameofmeasure IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|nameOfMeasure  CharacterString 0..*';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_measureidentification IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|measureIdentification  MD_Identifier 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_measuredescription IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|measureDescription  CharacterString 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_evaluationmethodtype IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationMethodType  DQ_EvaluationMethodTypeCode 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_evaluationmethoddescription IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationMethodDescription  CharacterString 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_evaluationprocedure IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationProcedure  CI_Citation 0..1';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_datetime IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|dateTime  DateTime 0..*';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.hoehengenauigkeit_result IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|result  DQ_Result 1..*';
		COMMENT ON COLUMN ax_gewaesserbegrenzung.ursprung IS 'ursprung enumeration AX_Ursprung 0..1';
		CREATE TABLE ax_strukturierterfasstegelaendepunkte (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  aktualisierungsdatum date,
		  artderstrukturierung integer NOT NULL,
		  erfassungstrukturiertegelaendepunkte_datetime character(20),
		  ax_dqerfassungsmethodestrukturiertegelaendepunkte integer NOT NULL,
		  hoehengenauigkeit_nameofmeasure character varying[],
		  hoehengenauigkeit_measureidentification character varying,
		  hoehengenauigkeit_measuredescription character varying,
		  hoehengenauigkeit_evaluationmethodtype character varying,
		  hoehengenauigkeit_evaluationmethoddescription character varying,
		  hoehengenauigkeit_evaluationprocedure character varying,
		  hoehengenauigkeit_datetime character(20)[],
		  hoehengenauigkeit_result character varying[],
		  punktabstand double precision,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_strukturierterfasstegelaendepunkte_gml ON ax_strukturierterfasstegelaendepunkte USING btree (gml_id,beginnt);
		CREATE INDEX ax_strukturierterfasstegelaendepunkte_endet ON ax_strukturierterfasstegelaendepunkte USING btree (endet);
		PERFORM AddGeometryColumn('ax_strukturierterfasstegelaendepunkte', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_strukturierterfasstegelaendepunkte_wkb_geometry_idx ON ax_strukturierterfasstegelaendepunkte USING gist (wkb_geometry);
		CREATE INDEX ax_strukturierterfasstegelaendepunkte_istabgeleitetaus ON ax_strukturierterfasstegelaendepunkte USING gin (istabgeleitetaus);
		CREATE INDEX ax_strukturierterfasstegelaendepunkte_traegtbeizu ON ax_strukturierterfasstegelaendepunkte USING gin (traegtbeizu);
		CREATE INDEX ax_strukturierterfasstegelaendepunkte_hatdirektunten ON ax_strukturierterfasstegelaendepunkte USING gin (hatdirektunten);
		CREATE INDEX ax_strukturierterfasstegelaendepunkte_istteilvon ON ax_strukturierterfasstegelaendepunkte USING gin (istteilvon);

		COMMENT ON TABLE ax_strukturierterfasstegelaendepunkte IS 'FeatureType: "AX_StrukturiertErfassteGelaendepunkte"';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.wkb_geometry IS 'wkb_geometry  GM_MultiCurve 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.aktualisierungsdatum IS 'aktualisierungsdatum  Date 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.artderstrukturierung IS 'artDerStrukturierung enumeration AX_ArtDerStrukturierung 1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.erfassungstrukturiertegelaendepunkte_datetime IS 'erfassungStrukturierteGelaendepunkte|AX_ErfassungStrukturierteGelaendepunkte|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.ax_dqerfassungsmethodestrukturiertegelaendepunkte IS 'erfassungStrukturierteGelaendepunkte|AX_ErfassungStrukturierteGelaendepunkte|AX_DQErfassungsmethodeStrukturierteGelaendepunkte enumeration AX_DQErfassungsmethodeStrukturierteGelaendepunkte 1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_nameofmeasure IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|nameOfMeasure  CharacterString 0..*';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_measureidentification IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|measureIdentification  MD_Identifier 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_measuredescription IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|measureDescription  CharacterString 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_evaluationmethodtype IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationMethodType  DQ_EvaluationMethodTypeCode 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_evaluationmethoddescription IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationMethodDescription  CharacterString 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_evaluationprocedure IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|evaluationProcedure  CI_Citation 0..1';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_datetime IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|dateTime  DateTime 0..*';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.hoehengenauigkeit_result IS 'hoehengenauigkeit|DQ_AbsoluteExternalPositionalAccuracy|result  DQ_Result 1..*';
		COMMENT ON COLUMN ax_strukturierterfasstegelaendepunkte.punktabstand IS 'punktabstand  Length 0..1';
		CREATE TABLE ax_einschnitt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  funktion integer,
		  tiefevoneinschnitt double precision,
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_einschnitt_gml ON ax_einschnitt USING btree (gml_id,beginnt);
		CREATE INDEX ax_einschnitt_endet ON ax_einschnitt USING btree (endet);
		PERFORM AddGeometryColumn('ax_einschnitt', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_einschnitt_wkb_geometry_idx ON ax_einschnitt USING gist (wkb_geometry);
		CREATE INDEX ax_einschnitt_istabgeleitetaus ON ax_einschnitt USING gin (istabgeleitetaus);
		CREATE INDEX ax_einschnitt_traegtbeizu ON ax_einschnitt USING gin (traegtbeizu);
		CREATE INDEX ax_einschnitt_hatdirektunten ON ax_einschnitt USING gin (hatdirektunten);
		CREATE INDEX ax_einschnitt_istteilvon ON ax_einschnitt USING gin (istteilvon);

		COMMENT ON TABLE ax_einschnitt IS 'FeatureType: "AX_Einschnitt"';
		COMMENT ON COLUMN ax_einschnitt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_einschnitt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_einschnitt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_einschnitt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_einschnitt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_einschnitt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_einschnitt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_einschnitt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_einschnitt.wkb_geometry IS 'wkb_geometry  GM_MultiCurve 0..1';
		COMMENT ON COLUMN ax_einschnitt.funktion IS 'funktion enumeration AX_Funktion_Einschnitt 0..1';
		COMMENT ON COLUMN ax_einschnitt.tiefevoneinschnitt IS 'tiefeVonEinschnitt  Length 0..1';
		CREATE TABLE ax_abgeleitetehoehenlinie (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  aktualitaetsstand date NOT NULL,
		  berechnungsdatum date NOT NULL,
		  berechnungsmethodehoehenlinie integer NOT NULL,
		  hoehe double precision NOT NULL,
		  verwendeteobjekte integer[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_abgeleitetehoehenlinie_gml ON ax_abgeleitetehoehenlinie USING btree (gml_id,beginnt);
		CREATE INDEX ax_abgeleitetehoehenlinie_endet ON ax_abgeleitetehoehenlinie USING btree (endet);
		PERFORM AddGeometryColumn('ax_abgeleitetehoehenlinie', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_abgeleitetehoehenlinie_wkb_geometry_idx ON ax_abgeleitetehoehenlinie USING gist (wkb_geometry);
		CREATE INDEX ax_abgeleitetehoehenlinie_istabgeleitetaus ON ax_abgeleitetehoehenlinie USING gin (istabgeleitetaus);
		CREATE INDEX ax_abgeleitetehoehenlinie_traegtbeizu ON ax_abgeleitetehoehenlinie USING gin (traegtbeizu);
		CREATE INDEX ax_abgeleitetehoehenlinie_hatdirektunten ON ax_abgeleitetehoehenlinie USING gin (hatdirektunten);
		CREATE INDEX ax_abgeleitetehoehenlinie_istteilvon ON ax_abgeleitetehoehenlinie USING gin (istteilvon);

		COMMENT ON TABLE ax_abgeleitetehoehenlinie IS 'FeatureType: "AX_AbgeleiteteHoehenlinie"';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.wkb_geometry IS 'wkb_geometry  GM_MultiCurve 0..1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.aktualitaetsstand IS 'aktualitaetsstand  Date 1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.berechnungsdatum IS 'berechnungsdatum  Date 1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.berechnungsmethodehoehenlinie IS 'berechnungsmethodeHoehenlinie enumeration AX_BerechnungsmethodeHoehenlinie 1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.hoehe IS 'hoehe  Length 1';
		COMMENT ON COLUMN ax_abgeleitetehoehenlinie.verwendeteobjekte IS 'verwendeteObjekte enumeration AX_VerwendeteObjekte 0..*';
		CREATE TABLE ax_nullpunkt (
		  ogc_fid serial NOT NULL,
		  gml_id character(16) NOT NULL,
		  anlass character varying[],
		  beginnt character(20) NOT NULL,
		  endet character(20),
		  advstandardmodell character varying[],
		  sonstigesmodell character varying[],
		  zeigtaufexternes_art character varying[],
		  zeigtaufexternes_name character varying[],
		  zeigtaufexternes_uri character varying[],
		  artdesnullpunktes integer NOT NULL,
		  bezeichnung character varying NOT NULL,
		  statement character varying,
		  processstep_ax_li_processstep_mitdatenerhebung_description character varying[],
		  processstep_rationale character varying[],
		  processstep_datetime character(20)[],
		  processstep_individualname character varying[],
		  processstep_organisationname character varying[],
		  processstep_positionname character varying[],
		  processstep_phone character varying[],
		  processstep_address character varying[],
		  processstep_onlineresource character varying[],
		  processstep_hoursofservice character varying[],
		  processstep_contactinstructions character varying[],
		  processstep_role character varying[],
		  processstep_ax_datenerhebung character varying[],
		  processstep_scaledenominator character varying[],
		  processstep_sourcereferencesystem character varying[],
		  processstep_sourceextent character varying[],
		  processstep_sourcestep character varying[],
		  herkunft_source_source_ax_datenerhebung character varying[],
		  herkunft_source_source_scaledenominator character varying[],
		  herkunft_source_source_sourcereferencesystem character varying[],
		  herkunft_source_source_sourceextent character varying[],
		  herkunft_source_source_sourcestep character varying[],
		  istabgeleitetaus character(16)[],
		  traegtbeizu character(16)[],
		  hatdirektunten character(16)[],
		  istteilvon character(16)[],
		  PRIMARY KEY (ogc_fid)
		);

		CREATE UNIQUE INDEX ax_nullpunkt_gml ON ax_nullpunkt USING btree (gml_id,beginnt);
		CREATE INDEX ax_nullpunkt_endet ON ax_nullpunkt USING btree (endet);
		PERFORM AddGeometryColumn('ax_nullpunkt', 'wkb_geometry', find_srid(current_schema()::text,'ax_flurstueck','wkb_geometry'), 'GEOMETRY', 2);
		CREATE INDEX ax_nullpunkt_wkb_geometry_idx ON ax_nullpunkt USING gist (wkb_geometry);
		CREATE INDEX ax_nullpunkt_istabgeleitetaus ON ax_nullpunkt USING gin (istabgeleitetaus);
		CREATE INDEX ax_nullpunkt_traegtbeizu ON ax_nullpunkt USING gin (traegtbeizu);
		CREATE INDEX ax_nullpunkt_hatdirektunten ON ax_nullpunkt USING gin (hatdirektunten);
		CREATE INDEX ax_nullpunkt_istteilvon ON ax_nullpunkt USING gin (istteilvon);

		COMMENT ON TABLE ax_nullpunkt IS 'FeatureType: "AX_Nullpunkt"';
		COMMENT ON COLUMN ax_nullpunkt.anlass IS 'anlass codelist AA_Anlassart 0..*';
		COMMENT ON COLUMN ax_nullpunkt.beginnt IS 'lebenszeitintervall|AA_Lebenszeitintervall|beginnt  DateTime 1';
		COMMENT ON COLUMN ax_nullpunkt.endet IS 'lebenszeitintervall|AA_Lebenszeitintervall|endet  DateTime 0..1';
		COMMENT ON COLUMN ax_nullpunkt.advstandardmodell IS 'modellart|AA_Modellart|advStandardModell enumeration AA_AdVStandardModell 0..1';
		COMMENT ON COLUMN ax_nullpunkt.sonstigesmodell IS 'modellart|AA_Modellart|sonstigesModell codelist AA_WeitereModellart 0..1';
		COMMENT ON COLUMN ax_nullpunkt.zeigtaufexternes_art IS 'zeigtAufExternes|AA_Fachdatenverbindung|art  URI 1';
		COMMENT ON COLUMN ax_nullpunkt.zeigtaufexternes_name IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|name   0..1';
		COMMENT ON COLUMN ax_nullpunkt.zeigtaufexternes_uri IS 'zeigtAufExternes|AA_Fachdatenverbindung|fachdatenobjekt|AA_Fachdatenobjekt|uri  URI 0..1';
		COMMENT ON COLUMN ax_nullpunkt.wkb_geometry IS 'wkb_geometry  GM_Point 0..1';
		COMMENT ON COLUMN ax_nullpunkt.artdesnullpunktes IS 'artDesNullpunktes enumeration AX_ArtDesNullpunktes_Nullpunkt 1';
		COMMENT ON COLUMN ax_nullpunkt.bezeichnung IS 'bezeichnung   1';
		COMMENT ON COLUMN ax_nullpunkt.statement IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|statement  CharacterString 0..1';
		COMMENT ON COLUMN ax_nullpunkt.processstep_ax_li_processstep_mitdatenerhebung_description IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|AX_LI_ProcessStep_MitDatenerhebung_Description enumeration  1';
		COMMENT ON COLUMN ax_nullpunkt.processstep_rationale IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|rationale  CharacterString 0..1';
		COMMENT ON COLUMN ax_nullpunkt.processstep_datetime IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|dateTime  DateTime 0..1';
		COMMENT ON COLUMN ax_nullpunkt.processstep_individualname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|individualName  CharacterString 0..1';
		COMMENT ON COLUMN ax_nullpunkt.processstep_organisationname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|processStep|LI_ProcessStep|processor|CI_ResponsibleParty|organisationName  CharacterString 0..1';
		COMMENT ON COLUMN ax_nullpunkt.processstep_positionname IS 'qualitaetsangaben|AX_DQMitDatenerhebung|herkunft|LI_Lineage|proce