---
name: v.to.rast
description: Converts (rasterize) a vector map into a raster map.
keywords: [ vector, conversion, raster, rasterization ]
---

# v.to.rast

Converts (rasterize) a vector map into a raster map.

=== "Command line"

    **v.to.rast**
    [**-d**]
    **input**=*name*
    [**layer**=*string*]
    [**type**=*string* [,*string*,...]]
    [**cats**=*range*]
    [**where**=*sql_query*]
    **output**=*name*
    **use**=*string*
    [**attribute_column**=*name*]
    [**rgb_column**=*name*]
    [**label_column**=*name*]
    [**value**=*float*]
    [**memory**=*memory in MB*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.to.rast input=name output=name use=attr
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.to.rast***",
        **input**,
        **layer**=*"1"*,
        **type**=*"point,line,area"*,
        **cats**=*None*,
        **where**=*None*,
        **output**,
        **use**,
        **attribute_column**=*None*,
        **rgb_column**=*None*,
        **label_column**=*None*,
        **value**=*1*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.to.rast", input="name", output="name", use="attr")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_to_rast*(**input**,
        **layer**=*"1"*,
        **type**=*"point,line,area"*,
        **cats**=*None*,
        **where**=*None*,
        **output**,
        **use**,
        **attribute_column**=*None*,
        **rgb_column**=*None*,
        **label_column**=*None*,
        **value**=*1*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_to_rast(input="name", output="name", use="attr")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **type**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **cats**=*range*  
    &nbsp;&nbsp;&nbsp;&nbsp;Category values  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: 1,3,7-9,13  
    **where**=*sql_query*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **use**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Source of raster values  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *attr, cat, value, z, dir*  
    &nbsp;&nbsp;&nbsp;&nbsp;**attr**: read values from attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;**cat**: use category values  
    &nbsp;&nbsp;&nbsp;&nbsp;**value**: use value specified by value option  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**: use z coordinate (points or contours only)  
    &nbsp;&nbsp;&nbsp;&nbsp;**dir**: line direction in degrees CCW from east (lines only)  
    **attribute_column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of column for 'attr' parameter (data type must be numeric)  
    **rgb_column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of color definition column (with RRR:GGG:BBB entries)  
    **label_column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of column used as raster category labels  
    **value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster value (for use=value)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **memory**=*memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Create densified lines (default: thin lines)  
    &nbsp;&nbsp;&nbsp;&nbsp;All cells touched by the line will be set, not only those on the render path  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **cats** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Category values  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: 1,3,7-9,13  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, cats, *range*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, sql_query, *sql_query*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **use** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Source of raster values  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *attr, cat, value, z, dir*  
    &nbsp;&nbsp;&nbsp;&nbsp;**attr**: read values from attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;**cat**: use category values  
    &nbsp;&nbsp;&nbsp;&nbsp;**value**: use value specified by value option  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**: use z coordinate (points or contours only)  
    &nbsp;&nbsp;&nbsp;&nbsp;**dir**: line direction in degrees CCW from east (lines only)  
    **attribute_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of column for 'attr' parameter (data type must be numeric)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **rgb_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of color definition column (with RRR:GGG:BBB entries)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **label_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of column used as raster category labels  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster value (for use=value)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create densified lines (default: thin lines)  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;All cells touched by the line will be set, not only those on the render path  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **cats** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Category values  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: 1,3,7-9,13  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, cats, *range*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, sql_query, *sql_query*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **use** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Source of raster values  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *attr, cat, value, z, dir*  
    &nbsp;&nbsp;&nbsp;&nbsp;**attr**: read values from attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;**cat**: use category values  
    &nbsp;&nbsp;&nbsp;&nbsp;**value**: use value specified by value option  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**: use z coordinate (points or contours only)  
    &nbsp;&nbsp;&nbsp;&nbsp;**dir**: line direction in degrees CCW from east (lines only)  
    **attribute_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of column for 'attr' parameter (data type must be numeric)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **rgb_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of color definition column (with RRR:GGG:BBB entries)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **label_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of column used as raster category labels  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster value (for use=value)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create densified lines (default: thin lines)  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;All cells touched by the line will be set, not only those on the render path  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.to.rast* transforms GRASS vector map layers into GRASS raster map
layer format. Optionally, attributes can be converted to raster category
labels.

## NOTES

In order to avoid unexpected results, the type of vector features should
always be specified. The default is to convert all vector features, but
if only e.g. areas should be converted use *type=area* rather than
*type=point,line,area*.

*v.to.rast* will only affect data in areas lying inside the boundaries
of the current geographic region. A grid cell belongs to the area where
the grid cell center falls into.

Before running *v.to.rast*, the user should therefore ensure that the
current geographic region is correctly set and that the region
resolution is at the desired level.

Either the ***column*** parameter or the ***value*** parameter must be
specified. The ***use*** option may be specified alone when using the
*dir* option.

***use*** options are:

- *attr* - read values from attribute table (default)
- *cat* - read values from category
- *value* - use value specified by ***value*** option
- *z* - use z coordinate (points or contours only)
- *dir* - line direction in degrees counterclockwise from east (lines
  only)

The ***column*** parameter uses an existing column from the vector map
database table as the category value in the output raster map. Existing
table columns can be shown by using *[db.describe](db.describe.md)*.

An empty raster map layer will be created if the vector map layer has
not been assigned category/attribute labels (e.g., through use of
[v.category option=add](v.category.md)).

Otherwise:

- Labeled areas and/or centroids will produce filled raster coverages
  with edges that straddle the original area boundary **as long as the
  boundary is NOT labeled**.  
  (Use `v.category option=del type=boundary` to remove.)
- Labeled lines and boundaries will produce lines of raster cells which
  touch the original vector line. This tends to be more aggressive than
  area-only conversions.
- Points and orphaned centroids will be converted into single cells on
  the resultant raster map.

**Line directions** are given in degrees counterclockwise from east.

Raster category labels are supported for all of *use=* except *use=z*.

The **-d** flag applies only to lines and boundaries, the default is to
set only those cells on the render path (thin line).

Boundaries (usually without categories) can be rasterized with

```sh
v.to.rast type=boundary layer=-1 use=value
```

## EXAMPLES

### Convert a vector map and use column SPEED from attribute table

```sh
db.describe -c table=vect_map

ncols:3
Column 1: CAT
Column 2: SPEED
Column 3: WIDTH
```

```sh
v.to.rast input=vect_map output=raster_map attribute_column=SPEED type=line
```

### Calculate stream directions from a river vector map (Spearfish)

```sh
v.to.rast input=streams output=streamsdir use=dir
```

### Calculate slope along path

Using slope and aspect maps, compute slope along a bus route (use full
NC sample dataset):

```sh
g.region raster=elevation -p
r.slope.aspect elevation=elevation slope=slope aspect=aspect

# compute direction of the bus route
v.to.rast input=busroute11 type=line output=busroute11_dir use=dir

# extract steepest slope values and transform them into slope along path
r.mapcalc "route_slope = if(busroute11, slope)"
r.mapcalc "route_slope_dir = abs(atan(tan(slope) * cos(aspect - busroute11_dir)))"
```

![Slope along path](v_to_rast_direction.png)  
*Slope in degrees along bus route*

### Convert a vector polygon map to raster including descriptive labels

In this example, the ZIP code vector map is rasterized (North Carolina
sample dataset):

```sh
# rasterize ZIP codes at 50m raster resolution
g.region vector=zipcodes_wake res=50 -ap
# vector to raster conversion, with category labels
v.to.rast input=zipcodes_wake output=myzipcodes use=attr attribute_column="ZIPNUM" label_column="NAME"
```

### Convert vector points to raster with raster cell binning

In this example, the number of schools per raster cell are counted
(North Carolina sample dataset):

```sh
g.copy vector=schools_wake,myschools_wake

# set computation region for raster binning
g.region vector=myschools_wake res=5000 -p -a

# add new column for counting
v.db.addcolumn myschools_wake column="value integer"
v.db.update myschools_wake column=value value=1

# verify attributes
v.db.select myschools_wake column=cat,value
v.out.ascii input=myschools_wake output=- column=value

# export and import on the fly, use 4th column (value) as input
v.out.ascii input=myschools_wake output=- column=value | r.in.xyz input=- \
            z=4 output=schools_wake_aggreg method=sum

d.mon wx0
d.rast schools_wake_aggreg
d.vect schools_wake
d.grid 5000
```

![Number of schools per raster cell](v_to_rast_binning.png)  
*Number of schools per raster cell*

## SEE ALSO

*[db.describe](db.describe.md), [v.category](v.category.md)*

## AUTHORS

Original code: Michael Shapiro, U.S. Army Construction Engineering
Research Laboratory  
GRASS 6.0 updates: Radim Blazek, ITC-irst, Trento, Italy  
Stream directions: Jaro Hofierka and Helena Mitasova  
GRASS 6.3 code cleanup and label support: Brad Douglas

## SOURCE CODE

Available at: [v.to.rast source code](https://github.com/OSGeo/grass/tree/main/vector/v.to.rast)
([history](https://github.com/OSGeo/grass/commits/main/vector/v.to.rast))  
Latest change: Friday May 09 14:33:40 2025 in commit [b356c7e](https://github.com/OSGeo/grass/commit/b356c7e108fcbeca77474c8e83ebec8f1e7b3e4a)
