---
name: r.gwflow
description: Numerical calculation program for transient, confined and unconfined groundwater flow in two dimensions.
keywords: [ raster, groundwater flow, hydrology ]
---

# r.gwflow

Numerical calculation program for transient, confined and unconfined groundwater flow in two dimensions.

=== "Command line"

    **r.gwflow**
    [**-f**]
    **phead**=*name*
    **status**=*name*
    **hc_x**=*name*
    **hc_y**=*name*
    [**q**=*name*]
    **s**=*name*
    [**recharge**=*name*]
    **top**=*name*
    **bottom**=*name*
    **output**=*name*
    [**vx**=*name*]
    [**vy**=*name*]
    [**budget**=*name*]
    **type**=*string*
    [**river_bed**=*name*]
    [**river_head**=*name*]
    [**river_leak**=*name*]
    [**drain_bed**=*name*]
    [**drain_leak**=*name*]
    **dtime**=*float*
    [**maxit**=*integer*]
    [**maxit**=*integer*]
    [**error**=*float*]
    [**solver**=*name*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.gwflow phead=name status=name hc_x=name hc_y=name s=name top=name bottom=name output=name type=confined dtime=86400
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.gwflow***",
        **phead**,
        **status**,
        **hc_x**,
        **hc_y**,
        **q**=*None*,
        **s**,
        **recharge**=*None*,
        **top**,
        **bottom**,
        **output**,
        **vx**=*None*,
        **vy**=*None*,
        **budget**=*None*,
        **type**=*"confined"*,
        **river_bed**=*None*,
        **river_head**=*None*,
        **river_leak**=*None*,
        **drain_bed**=*None*,
        **drain_leak**=*None*,
        **dtime**=*86400*,
        **maxit**=*10000*,
        **maxit**=*25*,
        **error**=*0.000001*,
        **solver**=*"cg"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.gwflow", phead="name", status="name", hc_x="name", hc_y="name", s="name", top="name", bottom="name", output="name", type="confined", dtime=86400)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_gwflow*(**phead**,
        **status**,
        **hc_x**,
        **hc_y**,
        **q**=*None*,
        **s**,
        **recharge**=*None*,
        **top**,
        **bottom**,
        **output**,
        **vx**=*None*,
        **vy**=*None*,
        **budget**=*None*,
        **type**=*"confined"*,
        **river_bed**=*None*,
        **river_head**=*None*,
        **river_leak**=*None*,
        **drain_bed**=*None*,
        **drain_leak**=*None*,
        **dtime**=*86400*,
        **maxit**=*10000*,
        **maxit**=*25*,
        **error**=*0.000001*,
        **solver**=*"cg"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_gwflow(phead="name", status="name", hc_x="name", hc_y="name", s="name", top="name", bottom="name", output="name", type="confined", dtime=86400)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **phead**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with initial piezometric head in [m]  
    **status**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing boundary condition status: 0-inactive, 1-active, 2-dirichlet  
    **hc_x**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with x-part of the hydraulic conductivity tensor in [m/s]  
    **hc_y**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with y-part of the hydraulic conductivity tensor in [m/s]  
    **q**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with water sources and sinks in [m^3/s]  
    **s**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with storativity for confined or effective porosity for unconfined groundwater flow booth in [-]   
    **recharge**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Recharge input raster map e.g: 6\*10^-9 per cell in [m^3/s\*m^2]  
    **top**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map describing the top surface of the aquifer in [m]  
    **bottom**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map describing the bottom surface of the aquifer in [m]  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map storing the numerical result [m]  
    **vx**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater filter velocity vector part in x direction [m/s]  
    **vy**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater filter velocity vector part in y direction [m/s]  
    **budget**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater budget for each cell [m^3/s]  
    **type**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of groundwater flow  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *confined, unconfined*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *confined*  
    **river_bed**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the height of the river bed in [m]  
    **river_head**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the water level (head) of the river with leakage connection in [m]  
    **river_leak**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the leakage coefficient of the river bed in [1/s].  
    **drain_bed**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the height of the drainage bed in [m]  
    **drain_leak**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the leakage coefficient of the drainage bed in [1/s]  
    **dtime**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The calculation time in seconds  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *86400*  
    **maxit**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iteration used to solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **maxit**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;The maximum number of iterations in the linearization approach  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *25*  
    **error**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Error break criteria for iterative solver  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.000001*  
    **solver**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of solver which should solve the symmetric linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cg, pcg, cholesky*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cg*  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allocate a full quadratic linear equation system, default is a sparse linear equation system.  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **phead** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with initial piezometric head in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **status** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing boundary condition status: 0-inactive, 1-active, 2-dirichlet  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_x** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with x-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_y** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with y-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **q** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with water sources and sinks in [m^3/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **s** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with storativity for confined or effective porosity for unconfined groundwater flow booth in [-]   
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **recharge** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Recharge input raster map e.g: 6\*10^-9 per cell in [m^3/s\*m^2]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **top** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map describing the top surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **bottom** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map describing the bottom surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map storing the numerical result [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vx** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater filter velocity vector part in x direction [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vy** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater filter velocity vector part in y direction [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **budget** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater budget for each cell [m^3/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **type** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of groundwater flow  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *confined, unconfined*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *confined*  
    **river_bed** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the height of the river bed in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **river_head** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the water level (head) of the river with leakage connection in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **river_leak** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the leakage coefficient of the river bed in [1/s].  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **drain_bed** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the height of the drainage bed in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **drain_leak** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the leakage coefficient of the drainage bed in [1/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **dtime** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The calculation time in seconds  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *86400*  
    **maxit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iteration used to solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **maxit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The maximum number of iterations in the linearization approach  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *25*  
    **error** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Error break criteria for iterative solver  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.000001*  
    **solver** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of solver which should solve the symmetric linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cg, pcg, cholesky*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cg*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Allocate a full quadratic linear equation system, default is a sparse linear equation system.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **phead** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with initial piezometric head in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **status** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing boundary condition status: 0-inactive, 1-active, 2-dirichlet  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_x** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with x-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_y** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with y-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **q** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with water sources and sinks in [m^3/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **s** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with storativity for confined or effective porosity for unconfined groundwater flow booth in [-]   
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **recharge** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Recharge input raster map e.g: 6\*10^-9 per cell in [m^3/s\*m^2]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **top** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map describing the top surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **bottom** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map describing the bottom surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map storing the numerical result [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vx** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater filter velocity vector part in x direction [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vy** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater filter velocity vector part in y direction [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **budget** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster map to store the groundwater budget for each cell [m^3/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **type** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of groundwater flow  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *confined, unconfined*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *confined*  
    **river_bed** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the height of the river bed in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **river_head** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the water level (head) of the river with leakage connection in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **river_leak** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the leakage coefficient of the river bed in [1/s].  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **drain_bed** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the height of the drainage bed in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **drain_leak** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map providing the leakage coefficient of the drainage bed in [1/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **dtime** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The calculation time in seconds  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *86400*  
    **maxit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iteration used to solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **maxit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The maximum number of iterations in the linearization approach  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *25*  
    **error** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Error break criteria for iterative solver  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.000001*  
    **solver** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of solver which should solve the symmetric linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cg, pcg, cholesky*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cg*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Allocate a full quadratic linear equation system, default is a sparse linear equation system.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.gwflow* is a numerical program which calculates implicit transient,
confined and unconfined groundwater flow in two dimensions based on
raster maps and the current region settings. All initial and boundary
conditions must be provided as raster maps. The unit of the current
project's coordinate reference system must be meters.

This module is sensitive to mask settings. All cells which are outside
the mask are ignored and handled as no flow boundaries.

![Workflow of r.gwflow](r_gwflow_concept.png)  
*Workflow of r.gwflow*

*r.gwflow* calculates the piezometric head and optionally the water
budget and the filter velocity field, based on the hydraulic
conductivity and the piezometric head. The vector components can be
visualized with paraview if they are exported with *r.out.vtk*.  
  
The groundwater flow will always be calculated transient. For stady
state computation set the timestep to a large number (billions of
seconds) or set the storativity/ effective porosity raster map to
zero.  
  
The water budget is calculated for each non inactive cell. The sum of
the budget for each non inactive cell must be near zero. This is an
indicator of the quality of the numerical result.

## NOTES

The groundwater flow calculation is based on Darcy's law and a numerical
implicit finite volume discretization. The discretization results in a
symmetric and positive definite linear equation system in form of *Ax =
b*, which must be solved. The groundwater flow partial differential
equation is of the following form:

(dh/dt)\*S = div (K grad h) + q

In detail for 2 dimensions:

(dh/dt)\*S = Kxx \* (d^2h/dx^2) + Kyy \* (d^2h/dy^2) + q

- h -- the piezometric head im \[m\]
- dt -- the time step for transient calculation in \[s\]
- S -- the specific storage \[1/m\]
- Kxx -- the hydraulic conductivity tensor part in x direction in
  \[m/s\]
- Kyy -- the hydraulic conductivity tensor part in y direction in
  \[m/s\]
- q - inner source/sink in meter per second \[1/s\]

Confined and unconfined groundwater flow is supported. Be aware that the
storativity input parameter is handled differently in case of unconfined
flow. Instead of the storativity, the effective porosity is expected.

To compute unconfined groundwater flow, a simple Picard based
linearization scheme is used to solve the resulting non-linear equation
system.

Two different boundary conditions are implemented, the Dirichlet and
Neumann conditions. By default the calculation area is surrounded by
homogeneous Neumann boundary conditions. The calculation and boundary
status of single cells must be set with a status map, the following
states are supported:

- 0 == inactive - the cell with status 0 will not be calculated, active
  cells will have a no flow boundary to this cell
- 1 == active - this cell is used for groundwater flow calculation,
  inner sources and recharge can be defined for those cells
- 2 == Dirichlet - cells of this type will have a fixed piezometric head
  value which do not change over the time

Note that all required raster maps are read into main memory.
Additionally the linear equation system will be allocated, so the memory
consumption of this module rapidely grow with the size of the input
maps.  
  
The resulting linear equation system *Ax = b* can be solved with several
solvers. An iterative solvers with sparse and quadratic matrices support
is implemented. The conjugate gradients method with (pcg) and without
(cg) precondition. Additionally a direct Cholesky solver is available.
This direct solver only work with normal quadratic matrices, so be
careful using them with large maps (maps of size 10.000 cells will need
more than one gigabyte of RAM). Always prefer a sparse matrix solver.

## EXAMPLE

Use this small script to create a working groundwater flow area and
data. Make sure you are not in a lat/lon projection. It includes
drainage and river input as well.

```sh
# set the region accordingly
g.region res=25 res3=25 t=100 b=0 n=1000 s=0 w=0 e=1000 -p3

#now create the input raster maps for confined and unconfined aquifers
r.mapcalc expression="phead = if(row() == 1 , 50, 40)"
r.mapcalc expression="status = if(row() == 1 , 2, 1)"
r.mapcalc expression="well = if(row() == 20 && col() == 20 , -0.01, 0)"
r.mapcalc expression="hydcond = 0.00025"
r.mapcalc expression="recharge = 0"
r.mapcalc expression="top_conf = 20.0"
r.mapcalc expression="top_unconf = 70.0"
r.mapcalc expression="bottom = 0.0"
r.mapcalc expression="null = 0.0"
r.mapcalc expression="poros = 0.15"
r.mapcalc expression="s = 0.0001"

# The maps of the river
r.mapcalc expression="river_bed = if(col() == 35 , 48, null())"
r.mapcalc expression="river_head = if(col() == 35 , 49, null())"
r.mapcalc expression="river_leak = if(col() == 35 , 0.0001, null())"

# The maps of the drainage
r.mapcalc expression="drain_bed = if(col() == 5 , 48, null())"
r.mapcalc expression="drain_leak = if(col() == 5 , 0.01, null())"

#confined groundwater flow with cg solver and sparse matrix, river and drain
#do not work with this confined aquifer (top == 20m)
r.gwflow solver=cg top=top_conf bottom=bottom phead=phead status=status \
  hc_x=hydcond hc_y=hydcond q=well s=s recharge=recharge output=gwresult_conf \
  dt=8640000 type=confined vx=gwresult_conf_velocity_x vy=gwresult_conf_velocity_y budget=budget_conf

#unconfined groundwater flow with cg solver and sparse matrix, river and drain are enabled
# We use the effective porosity as storativity parameter
r.gwflow solver=cg top=top_unconf bottom=bottom phead=phead \
  status=status hc_x=hydcond hc_y=hydcond q=well s=poros recharge=recharge \
  river_bed=river_bed river_head=river_head river_leak=river_leak \
  drain_bed=drain_bed drain_leak=drain_leak \
  output=gwresult_unconf dt=8640000 type=unconfined vx=gwresult_unconf_velocity_x \
  budget=budget_unconf vy=gwresult_unconf_velocity_y

# The data can be visulaized with paraview when exported with r.out.vtk
r.out.vtk -p in=gwresult_conf,status vector=gwresult_conf_velocity_x,gwresult_conf_velocity_y,null \
  out=/tmp/gwdata_conf2d.vtk
r.out.vtk -p elevation=gwresult_unconf in=gwresult_unconf,status vector=gwresult_unconf_velocity_x,gwresult_unconf_velocity_y,null \
  out=/tmp/gwdata_unconf2d.vtk

#now load the data into paraview
paraview --data=/tmp/gwdata_conf2d.vtk &
paraview --data=/tmp/gwdata_unconf2d.vtk &
```

## SEE ALSO

*[r.solute.transport](r.solute.transport.md), [r3.gwflow](r3.gwflow.md),
[r.out.vtk](r.out.vtk.md)*

## AUTHOR

Sören Gebbert

This work is based on the [Diploma Thesis of Sören Gebbert](https://grass.osgeo.org/gdp/hydrology/gebbert2007_diplom_stroemung_grass_gis.pdf)
at Technical University Berlin in Germany.

## SOURCE CODE

Available at: [r.gwflow source code](https://github.com/OSGeo/grass/tree/main/raster/r.gwflow)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.gwflow))  
Latest change: Monday Sep 22 11:13:25 2025 in commit [a5c4755](https://github.com/OSGeo/grass/commit/a5c4755b0a03d6c2396fbe584bd44f5c81817575)
