---
name: t.rast.series.py
description: Performs different aggregation algorithms from r.series on all or a subset of raster maps in a space time raster dataset.
keywords: [ temporal, aggregation, series, raster, time ]
---

# t.rast.series.py

Performs different aggregation algorithms from r.series on all or a subset of raster maps in a space time raster dataset.

=== "Command line"

    **t.rast.series.py**
    [**-tn**]
    **input**=*name*
    **method**=*string* [,*string*,...]
    [**quantile**=*float* [,*float*,...]]
    [**order**=*string* [,*string*,...]]
    [**nprocs**=*integer*]
    [**memory**=*memory in MB*]
    [**where**=*sql_query*]
    **output**=*name* [,*name*,...]
    [**file_limit**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    t.rast.series.py input=name method=average output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***t.rast.series.py***",
        **input**,
        **method**=*"average"*,
        **quantile**=*None*,
        **order**=*"start_time"*,
        **nprocs**=*0*,
        **memory**=*300*,
        **where**=*None*,
        **output**,
        **file_limit**=*1000*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("t.rast.series.py", input="name", method="average", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.t_rast_series_py*(**input**,
        **method**=*"average"*,
        **quantile**=*None*,
        **order**=*"start_time"*,
        **nprocs**=*0*,
        **memory**=*300*,
        **where**=*None*,
        **output**,
        **file_limit**=*1000*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.t_rast_series_py(input="name", method="average", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    **method**=*string* [,*string*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Aggregate operation to be performed on the raster maps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *average, count, median, mode, minimum, min_raster, maximum, max_raster, stddev, range, sum, variance, diversity, slope, offset, detcoeff, quart1, quart3, perc90, quantile, skewness, kurtosis*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *average*  
    **quantile**=*float* [,*float*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Quantile to calculate for method=quantile  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0.0-1.0*  
    **order**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort the maps by category  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *id,  name,  creator,  mapset,  creation_time,  modification_time,  start_time,  end_time,  north,  south,  west,  east,  min,  max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *start_time*  
    **nprocs**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory**=*memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **where**=*sql_query*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    **output**=*name* [,*name*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map(s)  
    **file_limit**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;The maximum number of open files allowed for each r.series process  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000*  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Do not assign the space time raster dataset start and end time to the output map  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Propagate NULLs  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **method** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Aggregate operation to be performed on the raster maps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *average, count, median, mode, minimum, min_raster, maximum, max_raster, stddev, range, sum, variance, diversity, slope, offset, detcoeff, quart1, quart3, perc90, quantile, skewness, kurtosis*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *average*  
    **quantile** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quantile to calculate for method=quantile  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0.0-1.0*  
    **order** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort the maps by category  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *id,  name,  creator,  mapset,  creation_time,  modification_time,  start_time,  end_time,  north,  south,  west,  east,  min,  max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *start_time*  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *sql_query*  
    **output** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **file_limit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The maximum number of open files allowed for each r.series process  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*, *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not assign the space time raster dataset start and end time to the output map  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Propagate NULLs  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **method** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Aggregate operation to be performed on the raster maps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *average, count, median, mode, minimum, min_raster, maximum, max_raster, stddev, range, sum, variance, diversity, slope, offset, detcoeff, quart1, quart3, perc90, quantile, skewness, kurtosis*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *average*  
    **quantile** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quantile to calculate for method=quantile  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0.0-1.0*  
    **order** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort the maps by category  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *id,  name,  creator,  mapset,  creation_time,  modification_time,  start_time,  end_time,  north,  south,  west,  east,  min,  max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *start_time*  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *sql_query*  
    **output** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **file_limit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The maximum number of open files allowed for each r.series process  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*, *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not assign the space time raster dataset start and end time to the output map  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Propagate NULLs  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

The input of this module is a single space time raster dataset, the
output is a single raster map layer. A subset of the input space time
raster dataset can be selected using the **where** option. The sorting
of the raster map layer can be set using the **order** option. Be aware
that the order of the maps can significantly influence the result of the
aggregation (e.g.: slope). By default the maps are ordered by
**start_time**.

*t.rast.series* is a simple wrapper for the raster module **r.series**.
It supports a subset of the aggregation methods of **r.series**.

## NOTES

To avoid problems with too many open files, by default, the maximum
number of open files is set to 1000. If the number of input raster files
exceeds this number, the **-z** flag will be invoked. Because this will
slow down processing, the user can set a higher limit with the
**file_limit** parameter. Note that **file_limit** limit should not
exceed the user-specific limit on open files set by your operating
system. See the
[Wiki](https://grasswiki.osgeo.org/wiki/Large_raster_data_processing#Number_of_open_files_limitation)
for more information.

## Performance

To enable parallel processing, the user can specify the number of
threads to be used with the **nprocs** parameter (default 1). The
**memory** parameter (default 300 MB) can also be provided to determine
the size of the buffer in MB for computation. Both parameters are passed
to [r.series](r.series.md). To take advantage of the parallelization,
GRASS needs to be compiled with OpenMP enabled.

## EXAMPLES

### Estimate the average temperature for the whole time series

Here the entire stack of input maps is considered:

```sh
t.rast.series input=tempmean_monthly output=tempmean_average method=average
```

### Estimate the average temperature for a subset of the time series

Here the stack of input maps is limited to a certain period of time:

```sh
t.rast.series input=tempmean_daily output=tempmean_season method=average \
  where="start_time >= '2012-06' and start_time <= '2012-08'"
```

### Climatology: single month in a multi-annual time series

By considering only a single month in a multi-annual time series the
so-called climatology can be computed. Estimate average temperature for
all January maps in the time series:

```sh
t.rast.series input=tempmean_monthly \
    method=average output=tempmean_january \
    where="strftime('%m', start_time)='01'"

# equivalently, we can use
t.rast.series input=tempmean_monthly \
    output=tempmean_january method=average \
    where="start_time = datetime(start_time, 'start of year', '0 month')"

# if we want also February and March averages

t.rast.series input=tempmean_monthly \
    output=tempmean_february method=average \
    where="start_time = datetime(start_time, 'start of year', '1 month')"

t.rast.series input=tempmean_monthly \
    output=tempmean_march method=average \
    where="start_time = datetime(start_time, 'start of year', '2 month')"
```

Generalizing a bit, we can estimate monthly climatologies for all months
by means of different methods

```sh
for i in `seq -w 1 12` ; do
  for m in average stddev minimum maximum ; do
    t.rast.series input=tempmean_monthly method=${m} output=tempmean_${m}_${i} \
    where="strftime('%m', start_time)='${i}'"
  done
done
```

## SEE ALSO

*[r.series](r.series.md), [t.create](t.create.md), [t.info](t.info.md)*

[Temporal data processing
Wiki](https://grasswiki.osgeo.org/wiki/Temporal_data_processing)

## AUTHOR

Sören Gebbert, Thünen Institute of Climate-Smart Agriculture

## SOURCE CODE

Available at: [t.rast.series source code](https://github.com/OSGeo/grass/tree/main/temporal/t.rast.series)
([history](https://github.com/OSGeo/grass/commits/main/temporal/t.rast.series))  
Latest change: Monday Nov 03 07:16:48 2025 in commit [c967967](https://github.com/OSGeo/grass/commit/c96796774942e46bbababcd652d3986d4e295665)
