---
name: r.fillnulls.py
description: Fills no-data areas in raster maps using spline interpolation.
keywords: [ raster, surface, elevation, interpolation, splines, no-data filling ]
---

# r.fillnulls.py

Fills no-data areas in raster maps using spline interpolation.

=== "Command line"

    **r.fillnulls.py**
    **input**=*name*
    **output**=*name*
    **method**=*string*
    [**tension**=*float*]
    [**smooth**=*float*]
    [**edge**=*integer*]
    [**npmin**=*integer*]
    [**segmax**=*integer*]
    [**lambda**=*float*]
    [**memory**=*memory in MB*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.fillnulls.py input=name output=name method=rst
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.fillnulls.py***",
        **input**,
        **output**,
        **method**=*"rst"*,
        **tension**=*40.*,
        **smooth**=*0.1*,
        **edge**=*3*,
        **npmin**=*600*,
        **segmax**=*300*,
        **lambda**=*0.01*,
        **memory**=*300*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.fillnulls.py", input="name", output="name", method="rst")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_fillnulls_py*(**input**,
        **output**,
        **method**=*"rst"*,
        **tension**=*40.*,
        **smooth**=*0.1*,
        **edge**=*3*,
        **npmin**=*600*,
        **segmax**=*300*,
        **lambda**=*0.01*,
        **memory**=*300*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_fillnulls_py(input="name", output="name", method="rst")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **method**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Interpolation method to use  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *bilinear, bicubic, rst*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *rst*  
    **tension**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spline tension parameter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *40.*  
    **smooth**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spline smoothing parameter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **edge**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Width of hole edge used for interpolation (in cells)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **npmin**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum number of points for approximation in a segment (&gt;segmax)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-10000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *600*  
    **segmax**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of points in a segment  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-10000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **lambda**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Tykhonov regularization parameter (affects smoothing)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used in bilinear and bicubic spline interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **memory**=*memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **method** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Interpolation method to use  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *bilinear, bicubic, rst*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *rst*  
    **tension** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spline tension parameter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *40.*  
    **smooth** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spline smoothing parameter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **edge** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Width of hole edge used for interpolation (in cells)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **npmin** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum number of points for approximation in a segment (&gt;segmax)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-10000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *600*  
    **segmax** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of points in a segment  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-10000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **lambda** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Tykhonov regularization parameter (affects smoothing)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used in bilinear and bicubic spline interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **method** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Interpolation method to use  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *bilinear, bicubic, rst*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *rst*  
    **tension** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spline tension parameter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *40.*  
    **smooth** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spline smoothing parameter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **edge** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Width of hole edge used for interpolation (in cells)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **npmin** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum number of points for approximation in a segment (&gt;segmax)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-10000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *600*  
    **segmax** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of points in a segment  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-10000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **lambda** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Tykhonov regularization parameter (affects smoothing)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used in bilinear and bicubic spline interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.fillnulls* fills NULL pixels (no data areas) in input raster map and
stores filled data to a new output raster map. The fill areas are
interpolated from the no data area boundaries buffer using
*[v.surf.rst](v.surf.rst.md)* regularized spline interpolation with
tension (**method=rst**) or *[r.resamp.bspline](r.resamp.bspline.md)*
cubic or linear spline interpolation with Tykhonov regularization.

## NOTES

Each area boundary buffer is set to three times the map resolution to
get nominally three points around the edge. This way the algorithm
interpolates into the hole with a trained slope and curvature at the
edges, in order to avoid that such a flat plane is generated in a hole.
The width of edge area can be adjusted by changing the edge parameter.

During the interpolation following warning may occur when using the RST
method:

```text
Warning: strip exists with insufficient data
Warning: taking too long to find points for interpolation --please change the region to area where your points are
```

This warning is generated if large data holes exist within the surface.
As the idea of *r.fillnulls* is to fill such holes, the user may ignore
the warning. The interpolation will be continued. However, the user may
pay attention to below notes.

If interpolation fails, temporary raster and vector maps are left in
place to allow unfilled map holes (NULL areas) to be identified and
manually repaired.

When using the default RST method, the algorithm is based on
*[v.surf.rst](v.surf.rst.md)* regularized splines with tension
interpolation module which interpolates the raster cell values for NULL
data areas from the boundary values of the NULL data area. An eventual
raster mask is respected during the NULL data area(s) filling. The
interpolated values are patched into the NULL data area(s) of the input
map and saved into a new raster map. Otherwise, either the linear or
cubic spline interpolation with Tykhonov regularization can be selected
(based on *[r.resamp.bspline](r.resamp.bspline.md)*).

## WARNING

Depending on the shape of the NULL data area(s) problems may occur due
to an insufficient number of input cell values for the interpolation
process. Most problems will occur if a NULL data area reaches a large
amount of the map boundary. The user will have to carefully check the
result using *[r.mapcalc](r.mapcalc.md)* (generating a difference map to
the input map and applying the "differences" color table with
*[r.colors](r.colors.md)*) and/or to query individual cell values.

RST method stores temporary maps on hard disk. It will require at least
as much free space as one extra input raster map takes.

## EXAMPLE

In this example, the SRTM elevation map in the North Carolina sample
dataset is filtered for outlier elevation values; missing pixels are
then re-interpolated to obtain a complete elevation map:

```sh
g.region raster=elev_srtm_30m -p
d.mon wx0
d.histogram elev_srtm_30m

# remove SRTM outliers, i.e. SRTM below 50m (esp. lakes), leading to no data areas
r.mapcalc "elev_srtm_30m_filt = if(elev_srtm_30m < 50.0, null(), elev_srtm_30m)"
d.histogram elev_srtm_30m_filt
d.rast elev_srtm_30m_filt

# using the default RST method to fill these holes in DEM
r.fillnulls input=elev_srtm_30m_filt output=elev_srtm_30m_rst tension=20

# using the bilinear method to fill these holes in DEM
r.fillnulls input=elev_srtm_30m_filt output=elev_srtm_30m_bilin method=bilinear

d.histogram elev_srtm_30m_rst
d.rast elev_srtm_30m_rst

d.erase
d.histogram elev_srtm_30m_bilin
d.rast elev_srtm_30m_bilin

r.mapcalc "diff_rst_bilin = elev_srtm_30m_rst - elev_srtm_30m_bilin"
r.colors diff_rst_bilin color=differences

r.univar -e diff_rst_bilin
d.erase
d.rast diff_rst_bilin
d.legend diff_rst_bilin
```

## REFERENCES

- Mitas, L., Mitasova, H., 1999, Spatial Interpolation. In: P.Longley,
  M.F. Goodchild, D.J. Maguire, D.W.Rhind (Eds.), Geographical
  Information Systems: Principles, Techniques, Management and
  Applications, Wiley, pp.481-492
- Mitasova H., Mitas L.,  Brown W.M.,  D.P. Gerdes, I. Kosinovsky,
  Baker, T.1995, Modeling spatially and temporally distributed
  phenomena: New methods and tools for GRASS GIS. *International Journal
  of GIS*, 9 (4), special issue on Integrating GIS and Environmental
  modeling, 433-446.
- [Mitasova H. and Mitas L.
  1993](http://fatra.cnr.ncsu.edu/~hmitaso/gmslab/papers/lmg.rev1.ps):
  Interpolation by Regularized Spline with Tension: I. Theory and
  Implementation, *Mathematical Geology* 25, 641-655.
- [Mitasova H. and Hofierka L.
  1993](http://fatra.cnr.ncsu.edu/~hmitaso/gmslab/papers/hmg.rev1.ps):
  Interpolation by Regularized Spline with Tension: II. Application to
  Terrain Modeling and Surface Geometry Analysis, *Mathematical Geology*
  25, 657-667.

## SEE ALSO

*[r.fill.dir](r.fill.dir.md), [r.mapcalc](r.mapcalc.md),
[r.resamp.bspline](r.resamp.bspline.md),
[v.surf.bspline](v.surf.bspline.md), [v.surf.rst](v.surf.rst.md),
[v.fill.holes](v.fill.holes.md)*

## AUTHORS

Markus Neteler, University of Hannover and Fondazione Edmund Mach  
Improvement by Hamish Bowman, NZ

## SOURCE CODE

Available at: [r.fillnulls source code](https://github.com/OSGeo/grass/tree/main/scripts/r.fillnulls)
([history](https://github.com/OSGeo/grass/commits/main/scripts/r.fillnulls))  
Latest change: Friday Mar 07 23:20:43 2025 in commit [b94776a](https://github.com/OSGeo/grass/commit/b94776ae711b613e98849395a1b4635336af9479)
