---
name: r.what.color
description: Queries colors for a raster map layer.
keywords: [ raster, querying, color table ]
---

# r.what.color

Queries colors for a raster map layer.

=== "Command line"

    **r.what.color**
    [**-i**]
    **input**=*name*
    [**value**=*float* [,*float*,...]]
    [**format**=*string*]
    **color_format**=*name*
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.what.color input=name color_format=hex
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.what.color***",
        **input**,
        **value**=*None*,
        **format**=*"%d:%d:%d"*,
        **color_format**=*"hex"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.what.color", input="name", color_format="hex")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_what_color*(**input**,
        **value**=*None*,
        **format**=*"%d:%d:%d"*,
        **color_format**=*"hex"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_what_color(input="name", color_format="hex")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **value**=*float* [,*float*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Values to query colors for  
    **format**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format ('plain', 'json', or printf-style string)  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format printf-style is deprecated, use 'color_format' option instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *%d:%d:%d*  
    **color_format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format for output values. Applies only when format is set to 'plain' or 'json'.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *rgb, hex, hsv, triplet*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *hex*  
    &nbsp;&nbsp;&nbsp;&nbsp;**rgb**: output color in RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hex**: output color in HEX format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hsv**: output color in HSV format (experimental)  
    &nbsp;&nbsp;&nbsp;&nbsp;**triplet**: output color in colon-separated RGB format  
    **-i**  
    &nbsp;&nbsp;&nbsp;&nbsp;Read values from stdin  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **value** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Values to query colors for  
    **format** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format ('plain', 'json', or printf-style string)  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format printf-style is deprecated, use 'color_format' option instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *%d:%d:%d*  
    **color_format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format for output values. Applies only when format is set to 'plain' or 'json'.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *rgb, hex, hsv, triplet*  
    &nbsp;&nbsp;&nbsp;&nbsp;**rgb**: output color in RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hex**: output color in HEX format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hsv**: output color in HSV format (experimental)  
    &nbsp;&nbsp;&nbsp;&nbsp;**triplet**: output color in colon-separated RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *hex*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *i*  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Read values from stdin  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **value** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Values to query colors for  
    **format** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format ('plain', 'json', or printf-style string)  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format printf-style is deprecated, use 'color_format' option instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *%d:%d:%d*  
    **color_format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format for output values. Applies only when format is set to 'plain' or 'json'.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *rgb, hex, hsv, triplet*  
    &nbsp;&nbsp;&nbsp;&nbsp;**rgb**: output color in RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hex**: output color in HEX format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hsv**: output color in HSV format (experimental)  
    &nbsp;&nbsp;&nbsp;&nbsp;**triplet**: output color in colon-separated RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *hex*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *i*  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Read values from stdin  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.what.color* outputs the color associated with user-specified category
values in a raster input map.

Values may be specified either using the **value=** option, or by
specifying the **-i** flag and passing the values on `stdin`, one per
line.

For each specified value, an output will be generated consisting of the
category value along with the color, e.g.:

```sh
r.what.color input=elevation.dem value=1500
1500: 223:127:31

# In plain format using the triplet color format:
r.what.color input=elevation.dem value=1500 format=plain color_format=triplet
1500: 223:127:31

# In JSON format using the triplet color format:
r.what.color input=elevation.dem value=1500 format=json color_format=triplet
[
    {
        "value": 1500,
        "color": "223:127:31"
    }
]
```

Similarly, other `color_format` options available with `format=json` and
`format=plain` are `hex`, `hsv`, `triplet`, and `rgb`, with `hex` being the
default color format.

If the input map is an integer (CELL) map, the category will be written
as an integer (no decimal point), otherwise it will be written in
floating point format (*printf("%.15g")* format).

If the lookup fails for a value, the color will be output as an
asterisk (or as `null` in JSON), e.g.:

```sh
r.what.color input=elevation.dem value=9999
9999: *

# In plain format:
r.what.color input=elevation.dem value=9999 format=plain
9999: *

# In JSON format:
r.what.color input=elevation.dem value=9999 format=json
[
    {
        "value": 9999,
        "color": null
    }
]
```

If a value cannot be parsed, both the value and the color will be output
as an asterisk (or as `null` in JSON), e.g.:

```sh
r.what.color input=elevation.dem value=bogus
*: *

# In plain format:
r.what.color input=elevation.dem value=bogus format=plain
*: *

# In JSON format:
r.what.color input=elevation.dem value=bogus format=json
[
    {
        "value": null,
        "color": null
    }
]
```

The format can be changed using the **format=** option. The value should
be a *printf()*-style format string containing three conversion
specifiers for the red, green and blue values respectively, e.g.:

```sh
r.what.color input=elevation.dem value=1500 format='%02x:%02x:%02x'
1500: df:7f:1f
```

If your system supports the *%m\$* syntax, you can change the ordering
of the components, e.g.:

```sh
r.what.color input=elevation.dem value=1500 format='%3$02x:%2$02x:%1$02x'
1500: 1f:7f:df
```

Common formats:  

- Tcl/Tk: `format="#%02x%02x%02x"`
- WxPython: `format='"#%02x%02x%02x"'` or `format='"(%d,%d,%d)"'`

NOTE:

Please note that the *printf()*-style output format is deprecated and will be
removed in a future release. Use the `color_format` option instead,
together with `format=plain` or `format=json`.

## Using r.what.color JSON output with python

Print color associated with user-specified category value in JSON format using
Python:

```python
import grass.script as gs

# Run the r.what.color command with rgb option for JSON output format
items = gs.parse_command(
    "r.what.color",
    input="elevation",
    value=[100, 135, 156],
    format="json",
    color_format="rgb",
)

for item in items:
    print(f"{item['value']}: {item['color']}")
```

```text
100: rgb(255, 229, 0)
135: rgb(195, 127, 59)
156: rgb(23, 22, 21)
```

## SEE ALSO

*[r.what](r.what.md)*

## AUTHOR

Glynn Clements

## SOURCE CODE

Available at: [r.what.color source code](https://github.com/OSGeo/grass/tree/main/raster/r.what.color)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.what.color))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
