---
name: r3.flow
description: Computes 3D flow lines and 3D flow accumulation.
keywords: [ raster3d, hydrology, voxel ]
---

# r3.flow

Computes 3D flow lines and 3D flow accumulation.

=== "Command line"

    **r3.flow**
    [**-a**]
    [**input**=*name*]
    [**vector_field**=*name* [,*name*,...]]
    [**seed_points**=*name*]
    [**flowline**=*name*]
    [**flowaccumulation**=*name*]
    [**sampled**=*name*]
    [**unit**=*string*]
    [**step**=*float*]
    [**limit**=*integer*]
    [**max_error**=*float*]
    [**skip**=*integer* [,*integer*,...]]
    [**direction**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r3.flow input=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r3.flow***",
        **input**=*None*,
        **vector_field**=*None*,
        **seed_points**=*None*,
        **flowline**=*None*,
        **flowaccumulation**=*None*,
        **sampled**=*None*,
        **unit**=*"cell"*,
        **step**=*0.25*,
        **limit**=*2000*,
        **max_error**=*1e-5*,
        **skip**=*None*,
        **direction**=*"down"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r3.flow", input="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r3_flow*(**input**=*None*,
        **vector_field**=*None*,
        **seed_points**=*None*,
        **flowline**=*None*,
        **flowaccumulation**=*None*,
        **sampled**=*None*,
        **unit**=*"cell"*,
        **step**=*0.25*,
        **limit**=*2000*,
        **max_error**=*1e-5*,
        **skip**=*None*,
        **direction**=*"down"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r3_flow(input="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input 3D raster map  
    **vector_field**=*name* [,*name*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of three 3D raster maps describing x, y, z components of vector field  
    **seed_points**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector map with points from which flow lines are generated  
    &nbsp;&nbsp;&nbsp;&nbsp;If no map is provided, flow lines are generated from each cell of the input 3D raster  
    **flowline**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for vector map of flow lines  
    **flowaccumulation**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output flowaccumulation 3D raster  
    **sampled**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for 3D raster sampled by flowlines  
    &nbsp;&nbsp;&nbsp;&nbsp;Values of this 3D raster will be stored as attributes of flowlines segments  
    **unit**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Unit of integration step  
    &nbsp;&nbsp;&nbsp;&nbsp;Default unit is cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *time, length, cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;**time**: elapsed time  
    &nbsp;&nbsp;&nbsp;&nbsp;**length**: length in map units  
    &nbsp;&nbsp;&nbsp;&nbsp;**cell**: length in cells (voxels)  
    **step**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Integration step in selected unit  
    &nbsp;&nbsp;&nbsp;&nbsp;Default step is 0.25 cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.25*  
    **limit**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2000*  
    **max_error**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum error of integration  
    &nbsp;&nbsp;&nbsp;&nbsp;Influences step, increase maximum error to allow bigger steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-5*  
    **skip**=*integer* [,*integer*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of cells between flow lines in x, y and z direction  
    **direction**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Compute flowlines upstream, downstream or in both direction.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *up, down, both*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *down*  
    **-a**  
    &nbsp;&nbsp;&nbsp;&nbsp;Create and fill attribute table  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input 3D raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **vector_field** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of three 3D raster maps describing x, y, z components of vector field  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **seed_points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector map with points from which flow lines are generated  
    &nbsp;&nbsp;&nbsp;&nbsp;If no map is provided, flow lines are generated from each cell of the input 3D raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **flowline** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for vector map of flow lines  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **flowaccumulation** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output flowaccumulation 3D raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster_3d, *name*  
    **sampled** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for 3D raster sampled by flowlines  
    &nbsp;&nbsp;&nbsp;&nbsp;Values of this 3D raster will be stored as attributes of flowlines segments  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **unit** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Unit of integration step  
    &nbsp;&nbsp;&nbsp;&nbsp;Default unit is cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *time, length, cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;**time**: elapsed time  
    &nbsp;&nbsp;&nbsp;&nbsp;**length**: length in map units  
    &nbsp;&nbsp;&nbsp;&nbsp;**cell**: length in cells (voxels)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cell*  
    **step** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Integration step in selected unit  
    &nbsp;&nbsp;&nbsp;&nbsp;Default step is 0.25 cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.25*  
    **limit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2000*  
    **max_error** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum error of integration  
    &nbsp;&nbsp;&nbsp;&nbsp;Influences step, increase maximum error to allow bigger steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-5*  
    **skip** : int | list[int] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of cells between flow lines in x, y and z direction  
    **direction** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Compute flowlines upstream, downstream or in both direction.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *up, down, both*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *down*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *a*  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create and fill attribute table  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input 3D raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **vector_field** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of three 3D raster maps describing x, y, z components of vector field  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **seed_points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector map with points from which flow lines are generated  
    &nbsp;&nbsp;&nbsp;&nbsp;If no map is provided, flow lines are generated from each cell of the input 3D raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **flowline** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for vector map of flow lines  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **flowaccumulation** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output flowaccumulation 3D raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster_3d, *name*  
    **sampled** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for 3D raster sampled by flowlines  
    &nbsp;&nbsp;&nbsp;&nbsp;Values of this 3D raster will be stored as attributes of flowlines segments  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **unit** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Unit of integration step  
    &nbsp;&nbsp;&nbsp;&nbsp;Default unit is cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *time, length, cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;**time**: elapsed time  
    &nbsp;&nbsp;&nbsp;&nbsp;**length**: length in map units  
    &nbsp;&nbsp;&nbsp;&nbsp;**cell**: length in cells (voxels)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cell*  
    **step** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Integration step in selected unit  
    &nbsp;&nbsp;&nbsp;&nbsp;Default step is 0.25 cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.25*  
    **limit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2000*  
    **max_error** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum error of integration  
    &nbsp;&nbsp;&nbsp;&nbsp;Influences step, increase maximum error to allow bigger steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-5*  
    **skip** : int | list[int] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of cells between flow lines in x, y and z direction  
    **direction** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Compute flowlines upstream, downstream or in both direction.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *up, down, both*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *down*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *a*  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create and fill attribute table  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

Module *r3.flow* computes 3D flow lines and 3D flow accumulation. It
accepts either three 3D raster maps representing the vector field or one
3D raster map. In case of one map, it computes on-the-fly gradient
field.

### Flow lines

Flow lines are computed either from points (seeds) provided in
**seed_points** vector map, or if there are no seeds, it creates seeds
in a regular grid in the center of voxels (3D raster cells). Parameter
**skip** then controls the step between the regularly distributed seeds.
If skip is not provided, r3.flow decides optimal skip for each dimension
based on current 3D region as one tenth of the number of columns, rows,
and depths. Flow lines can be computed in upstream direction (in the
direction of gradient or vector field), in downstream direction or in
both directions.

### Flow accumulation

Flow accumulation is computed as the number of flow lines traversing
each voxel. Since the flow lines are computed for each voxel, the flow
accumulation computation can be more demanding. Parameter skip does not
influence the flow accumulation computation, parameter direction does.

### Flow line integration

Flow line integration can be influenced by several parameters. Option
**step** controls the integration step and influences the precision and
computational time. The unit of step can be defined either in terms of
the size of the voxel (3D raster cell), length in map units, or as
elapsed time. Option **limit** specifies the maximum number of steps of
each flow line.

### Attributes

Without using flag **a**, no attribute table is created and each flow
line is represented by one vector line with one category. With **a**
flag, an attribute table is created and each category (record)
represents one segment of a flowline, so that attributes specific for
segments can be written. In case of **vector_field** input, only
velocity is written, in case of **input** option, also values of the
input 3D raster are written. Option **sampled** allows sampling (query)
given 3D raster by flow lines (computed from different 3D raster) and
write the values of the given 3D raster as attributes of the flow line
segments. Note that using **a** flag results in longer computation time,
so consider increasing **step** and **max_error** parameter.

## NOTES

r3.flow uses Runge-Kutta with adaptive step size ([Cash-Karp
method](https://en.wikipedia.org/wiki/Cash-Karp_method)).

## EXAMPLES

First we create input data using [example 1](r3.gwflow.md#example-1)
from [r3.gwflow manual page](r3.gwflow.md):

```sh
# set the region accordingly
g.region res=25 res3=25 t=100 b=0 n=1000 s=0 w=0 e=1000 -p3

# now create the input raster maps for a confined aquifer
r3.mapcalc expression="phead = if(row() == 1 && depth() == 4, 50, 40)"
r3.mapcalc expression="status = if(row() == 1 && depth() == 4, 2, 1)"
r3.mapcalc expression="well = if(row() == 20 && col() == 20 && depth() == 2, -0.25, 0)"
r3.mapcalc expression="hydcond = 0.00025"
r3.mapcalc expression="syield = 0.0001"
r.mapcalc  expression="recharge = 0.0"

r3.gwflow solver=cg phead=phead status=status hc_x=hydcond hc_y=hydcond  \
   hc_z=hydcond q=well s=syield r=recharge output=gwresult dt=8640000 vx=vx vy=vy vz=vz budget=budget
```

Then we compute flow lines in both directions and downstream
flowaccumulation.

```sh
r3.flow vector_field=vx,vy,vz flowline=gw_flowlines skip=5,5,2 direction=both
r3.flow vector_field=vx,vy,vz flowaccumulation=gw_flowacc
```

We can visualize the result in 3D view:

![r3flow_flowlines](r3flow_flowlines.png)

We can store velocity values (and values of the input 3D raster map if
we use option **input**) for each segment of flow line in an attribute
table.

```sh
r3.flow -a vector_field=vx,vy,vz flowline=gw_flowlines skip=5,5,2 direction=both
v.colors map=flowlines_color@user1 use=attr column=velocity color=bcyr
```

Again, we visualize the result in 3D view and we check 'use color for
thematic rendering' on 3D view vector page.

![r3flow_flowlines_color](r3flow_flowlines_color.png)

## SEE ALSO

*[r.flow](r.flow.md), [r3.gradient](r3.gradient.md),
[r3.gwflow](r3.gwflow.md)*

## AUTHOR

Anna Petrasova, [NCSU GeoForAll
Lab](https://geospatial.ncsu.edu/geoforall/), developed during GSoC
2014.

## SOURCE CODE

Available at: [r3.flow source code](https://github.com/OSGeo/grass/tree/main/raster3d/r3.flow)
([history](https://github.com/OSGeo/grass/commits/main/raster3d/r3.flow))  
Latest change: Monday Nov 03 07:16:48 2025 in commit [c967967](https://github.com/OSGeo/grass/commit/c96796774942e46bbababcd652d3986d4e295665)
