---
name: r.resamp.interp
description: Resamples raster map to a finer grid using interpolation.
keywords: [ raster, resample, interpolation, nearest neighbor, bilinear, bicubic, lanczos, parallel ]
---

# r.resamp.interp

Resamples raster map to a finer grid using interpolation.

=== "Command line"

    **r.resamp.interp**
    **input**=*name*
    **output**=*name*
    [**method**=*string*]
    [**nprocs**=*integer*]
    [**memory**=*memory in MB*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.resamp.interp input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.resamp.interp***",
        **input**,
        **output**,
        **method**=*"bilinear"*,
        **nprocs**=*0*,
        **memory**=*300*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.resamp.interp", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_resamp_interp*(**input**,
        **output**,
        **method**=*"bilinear"*,
        **nprocs**=*0*,
        **memory**=*300*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_resamp_interp(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **method**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sampling interpolation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *nearest, bilinear, bicubic, lanczos*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *bilinear*  
    &nbsp;&nbsp;&nbsp;&nbsp;**nearest**: Nearest-neighbor interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;**bilinear**: Bilinear interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;**bicubic**: Bicubic interpolation  
    **nprocs**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory**=*memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **method** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sampling interpolation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *nearest, bilinear, bicubic, lanczos*  
    &nbsp;&nbsp;&nbsp;&nbsp;**nearest**: Nearest-neighbor interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;**bilinear**: Bilinear interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;**bicubic**: Bicubic interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *bilinear*  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **method** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sampling interpolation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *nearest, bilinear, bicubic, lanczos*  
    &nbsp;&nbsp;&nbsp;&nbsp;**nearest**: Nearest-neighbor interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;**bilinear**: Bilinear interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;**bicubic**: Bicubic interpolation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *bilinear*  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.resamp.interp* resamples an input raster map by interpolating between
the neighboring cells via a selectable resampling algorithm. All cells
present in the neighborhood of the input raster cell must be non-null to
generate a non-null cell in the output raster map. A choice of four
interpolation methods is available; each uses the weighted values of a
different number of adjacent cells in the input map to determine the
value of each cell in the output map as follows:

- nearest neighbour (1 cell)
- bilinear (4 cells)
- bicubic (16 cells)
- lanczos (25 cells)

This module is intended for reinterpolation of continuous data to a
different resolution rather than for interpolation from scattered data
(use the *v.surf.\** modules for that purpose).

## NOTES

Resampling modules (*r.resample, r.resamp.stats, r.resamp.interp,
r.resamp.rst*) resample the map to match the current region settings.

Note that for bilinear, bicubic and lanczos interpolation, cells of the
output raster that cannot be bounded by the appropriate number of input
cell centers are set to NULL (NULL propagation). This could occur due to
the input cells being outside the current region, being NULL or masked.

For longitude-latitude coordinate reference systems, the interpolation
algorithm is based on degree fractions, not on the absolute distances
between cell centers. Any attempt to implement the latter would violate
the integrity of the interpolation method.

### PERFORMANCE

By specifying the number of parallel processes with **nprocs** option,
*r.resamp.interp* can run significantly faster, see benchmarks below.

![benchmark for number of cells](r_resamp_interp_benchmark_size.png)  
*Figure: Benchmark shows execution time for different number of cells.
See benchmark scripts in source code.*

To reduce the memory requirements to minimum, set option **memory** to
zero. To take advantage of the parallelization, GRASS needs to
compiled with OpenMP enabled.

## EXAMPLE

Resample elevation raster map to a higher resolution (from 500m to 250m;
North Carolina sample dataset):

```sh
g.region raster=elev_state_500m -p
g.region res=250 -ap
r.resamp.interp input=elev_state_500m output=elev_state_250m \
                method=bilinear
```

![Original 500m resolution elevation
map](r_resamp_interp_orig_500m.png)  
Original 500m resolution elevation map

![Resampled 250m resolution elevation
map](r_resamp_interp_new_250m.png)  
Resampled (bilinear) 250m resolution elevation map

## SEE ALSO

*[g.region](g.region.md), [r.resample](r.resample.md),
[r.resamp.filter](r.resamp.filter.md), [r.resamp.rst](r.resamp.rst.md),
[r.resamp.stats](r.resamp.stats.md)*

Overview: [Interpolation and
Resampling](https://grasswiki.osgeo.org/wiki/Interpolation) in GRASS

## AUTHOR

Glynn Clements

## SOURCE CODE

Available at: [r.resamp.interp source code](https://github.com/OSGeo/grass/tree/main/raster/r.resamp.interp)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.resamp.interp))  
Latest change: Friday Aug 08 09:39:19 2025 in commit [ed72c71](https://github.com/OSGeo/grass/commit/ed72c71d8580e872cd6fd345c84a15c324764465)
