---
name: r.solute.transport
description: Numerical calculation program for transient, confined and unconfined solute transport in two dimensions
keywords: [ raster, hydrology, solute transport ]
---

# r.solute.transport

Numerical calculation program for transient, confined and unconfined solute transport in two dimensions

=== "Command line"

    **r.solute.transport**
    [**-fc**]
    **c**=*name*
    **phead**=*name*
    **hc_x**=*name*
    **hc_y**=*name*
    **status**=*name*
    **diff_x**=*name*
    **diff_y**=*name*
    [**q**=*name*]
    [**cin**=*name*]
    **cs**=*name*
    **rd**=*name*
    **nf**=*name*
    **top**=*name*
    **bottom**=*name*
    **output**=*name*
    [**vx**=*name*]
    [**vy**=*name*]
    **dtime**=*float*
    [**maxit**=*integer*]
    [**error**=*float*]
    [**solver**=*name*]
    [**relax**=*float*]
    [**al**=*float*]
    [**at**=*float*]
    [**loops**=*float*]
    [**stab**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.solute.transport c=name phead=name hc_x=name hc_y=name status=name diff_x=name diff_y=name cs=name rd=name nf=name top=name bottom=name output=name dtime=86400
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.solute.transport***",
        **c**,
        **phead**,
        **hc_x**,
        **hc_y**,
        **status**,
        **diff_x**,
        **diff_y**,
        **q**=*None*,
        **cin**=*None*,
        **cs**,
        **rd**,
        **nf**,
        **top**,
        **bottom**,
        **output**,
        **vx**=*None*,
        **vy**=*None*,
        **dtime**=*86400*,
        **maxit**=*10000*,
        **error**=*0.000001*,
        **solver**=*"bicgstab"*,
        **relax**=*1*,
        **al**=*0.0*,
        **at**=*0.0*,
        **loops**=*1*,
        **stab**=*"full"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.solute.transport", c="name", phead="name", hc_x="name", hc_y="name", status="name", diff_x="name", diff_y="name", cs="name", rd="name", nf="name", top="name", bottom="name", output="name", dtime=86400)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_solute_transport*(**c**,
        **phead**,
        **hc_x**,
        **hc_y**,
        **status**,
        **diff_x**,
        **diff_y**,
        **q**=*None*,
        **cin**=*None*,
        **cs**,
        **rd**,
        **nf**,
        **top**,
        **bottom**,
        **output**,
        **vx**=*None*,
        **vy**=*None*,
        **dtime**=*86400*,
        **maxit**=*10000*,
        **error**=*0.000001*,
        **solver**=*"bicgstab"*,
        **relax**=*1*,
        **al**=*0.0*,
        **at**=*0.0*,
        **loops**=*1*,
        **stab**=*"full"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_solute_transport(c="name", phead="name", hc_x="name", hc_y="name", status="name", diff_x="name", diff_y="name", cs="name", rd="name", nf="name", top="name", bottom="name", output="name", dtime=86400)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **c**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The initial concentration in [kg/m^3]  
    **phead**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The piezometric head in [m]  
    **hc_x**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The x-part of the hydraulic conductivity tensor in [m/s]  
    **hc_y**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The y-part of the hydraulic conductivity tensor in [m/s]  
    **status**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The status for each cell, = 0 - inactive cell, 1 - active cell, 2 - dirichlet- and 3 - transfer boundary condition  
    **diff_x**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The x-part of the diffusion tensor in [m^2/s]  
    **diff_y**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The y-part of the diffusion tensor in [m^2/s]  
    **q**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Groundwater sources and sinks in [m^3/s]  
    **cin**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Concentration sources and sinks bounded to a water source or sink in [kg/s]  
    **cs**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Concentration of inner sources and inner sinks in [kg/s] (i.e. a chemical reaction)  
    **rd**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Retardation factor [-]  
    **nf**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Effective porosity [-]  
    **top**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Top surface of the aquifer in [m]  
    **bottom**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Bottom surface of the aquifer in [m]  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The resulting concentration of the numerical solute transport calculation will be written to this map. [kg/m^3]  
    **vx**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate and store the groundwater filter velocity vector part in x direction [m/s]\
  
    **vy**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate and store the groundwater filter velocity vector part in y direction [m/s]\
  
    **dtime**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The calculation time in seconds  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *86400*  
    **maxit**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iteration used to solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **error**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Error break criteria for iterative solver  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.000001*  
    **solver**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of solver which should solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *gauss, lu, jacobi, sor, bicgstab*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *bicgstab*  
    **relax**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;The relaxation parameter used by the jacobi and sor solver for speedup or stabilizing  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **al**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;The longditudinal dispersivity length. [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **at**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;The transversal dispersivity length. [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **loops**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use this number of time loops if the CFL flag is off. The timestep will become dt/loops.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **stab**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Set the flow stabilizing scheme (full or exponential upwinding).  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *full, exp*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *full*  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use a full filled quadratic linear equation system, default is a sparse linear equation system.  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use the Courant-Friedrichs-Lewy criteria for time step calculation  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **c** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The initial concentration in [kg/m^3]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **phead** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The piezometric head in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_x** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The x-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_y** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The y-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **status** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The status for each cell, = 0 - inactive cell, 1 - active cell, 2 - dirichlet- and 3 - transfer boundary condition  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **diff_x** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The x-part of the diffusion tensor in [m^2/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **diff_y** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The y-part of the diffusion tensor in [m^2/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **q** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Groundwater sources and sinks in [m^3/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **cin** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Concentration sources and sinks bounded to a water source or sink in [kg/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **cs** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Concentration of inner sources and inner sinks in [kg/s] (i.e. a chemical reaction)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **rd** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Retardation factor [-]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **nf** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Effective porosity [-]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **top** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Top surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **bottom** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Bottom surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The resulting concentration of the numerical solute transport calculation will be written to this map. [kg/m^3]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vx** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate and store the groundwater filter velocity vector part in x direction [m/s]\
  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vy** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate and store the groundwater filter velocity vector part in y direction [m/s]\
  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **dtime** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The calculation time in seconds  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *86400*  
    **maxit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iteration used to solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **error** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Error break criteria for iterative solver  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.000001*  
    **solver** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of solver which should solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *gauss, lu, jacobi, sor, bicgstab*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *bicgstab*  
    **relax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The relaxation parameter used by the jacobi and sor solver for speedup or stabilizing  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **al** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The longditudinal dispersivity length. [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **at** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The transversal dispersivity length. [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **loops** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use this number of time loops if the CFL flag is off. The timestep will become dt/loops.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **stab** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Set the flow stabilizing scheme (full or exponential upwinding).  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *full, exp*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *full*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use a full filled quadratic linear equation system, default is a sparse linear equation system.  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the Courant-Friedrichs-Lewy criteria for time step calculation  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **c** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The initial concentration in [kg/m^3]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **phead** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The piezometric head in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_x** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The x-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **hc_y** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The y-part of the hydraulic conductivity tensor in [m/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **status** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The status for each cell, = 0 - inactive cell, 1 - active cell, 2 - dirichlet- and 3 - transfer boundary condition  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **diff_x** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The x-part of the diffusion tensor in [m^2/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **diff_y** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The y-part of the diffusion tensor in [m^2/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **q** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Groundwater sources and sinks in [m^3/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **cin** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Concentration sources and sinks bounded to a water source or sink in [kg/s]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **cs** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Concentration of inner sources and inner sinks in [kg/s] (i.e. a chemical reaction)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **rd** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Retardation factor [-]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **nf** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Effective porosity [-]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **top** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Top surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **bottom** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Bottom surface of the aquifer in [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The resulting concentration of the numerical solute transport calculation will be written to this map. [kg/m^3]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vx** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate and store the groundwater filter velocity vector part in x direction [m/s]\
  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **vy** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate and store the groundwater filter velocity vector part in y direction [m/s]\
  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **dtime** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The calculation time in seconds  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *86400*  
    **maxit** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iteration used to solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **error** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Error break criteria for iterative solver  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.000001*  
    **solver** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The type of solver which should solve the linear equation system  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *gauss, lu, jacobi, sor, bicgstab*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *bicgstab*  
    **relax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The relaxation parameter used by the jacobi and sor solver for speedup or stabilizing  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **al** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The longditudinal dispersivity length. [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **at** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The transversal dispersivity length. [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **loops** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use this number of time loops if the CFL flag is off. The timestep will become dt/loops.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **stab** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Set the flow stabilizing scheme (full or exponential upwinding).  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *full, exp*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *full*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use a full filled quadratic linear equation system, default is a sparse linear equation system.  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the Courant-Friedrichs-Lewy criteria for time step calculation  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

This numerical program calculates numerical implicit transient and
steady state solute transport in porous media in the saturated zone of
an aquifer. The computation is based on raster maps and the current
region settings. All initial- and boundary-conditions must be provided
as raster maps. The unit of the coordinate reference system must be
meters.  

This module is sensitive to mask settings. All cells which are outside
the mask are ignored and handled as no flow boundaries.  
This module calculates the concentration of the solution and optional
the velocity field, based on the hydraulic conductivity, the effective
porosity and the initial piecometric heads. The vector components can be
visualized with paraview if they are exported with r.out.vtk.  
  
Use [r.gwflow](r.gwflow.md) to compute the piezometric heights of the
aquifer. The piezometric heights and the hydraulic conductivity are used
to compute the flow direction and the mean velocity of the groundwater.
This is the base of the solute transport computation.  
  
The solute transport will always be calculated transient. For stady
state computation set the timestep to a large number (billions of
seconds).  
  
To reduce the numerical dispersion, which is a consequence of the
convection term and the finite volume discretization, you can use small
time steps and choose between full and exponential upwinding.

## NOTES

The solute transport calculation is based on a diffusion/convection
partial differential equation and a numerical implicit finite volume
discretization. Specific for this kind of differential equation is the
combination of a diffusion/dispersion term and a convection term. The
discretization results in an unsymmetric linear equation system in form
of *Ax = b*, which must be solved. The solute transport partial
differential equation is of the following form:

(dc/dt)\*R = div ( D grad c - uc) + cs -q/nf(c - c_in)

- c -- the concentration \[kg/m^3\]
- u -- vector of mean groundwater flow velocity
- dt -- the time step for transient calculation in seconds \[s\]
- R -- the linear retardation coefficient \[-\]
- D -- the diffusion and dispersion tensor \[m^2/s\]
- cs -- inner concentration sources/sinks \[kg/m^3\]
- c_in -- the solute concentration of influent water \[kg/m^3\]
- q -- inner well sources/sinks \[m^3/s\]
- nf -- the effective porosity \[-\]

Three different boundary conditions are implemented, the Dirichlet,
Transmission and Neumann conditions. The calculation and boundary status
of single cells can be set with the status map. The following states are
supported:

- 0 == inactive - the cell with status 0 will not be calculated, active
  cells will have a no flow boundary to an inactive cell
- 1 == active - this cell is used for sloute transport calculation,
  inner sources can be defined for those cells
- 2 == Dirichlet - cells of this type will have a fixed concentration
  value which do not change over time
- 3 == Transmission - cells of this type should be placed on out-flow
  boundaries to assure the flow of the solute stream out

Note that all required raster maps are read into main memory.
Additionally the linear equation system will be allocated, so the memory
consumption of this module rapidely grow with the size of the input
maps.  
  
The resulting linear equation system *Ax = b* can be solved with several
solvers. Several iterative solvers with unsymmetric sparse and quadratic
matrices support are implemented. The jacobi method, the Gauss-Seidel
method and the biconjugate gradients-stabilized (bicgstab) method.
Additionally a direct Gauss solver and LU solver are available. Those
direct solvers only work with quadratic matrices, so be careful using
them with large maps (maps of size 10.000 cells will need more than one
gigabyte of ram). Always prefer a sparse matrix solver.

## EXAMPLE

Use this small python script to create a working groundwater flow /
solute transport area and data. Make sure you are not in a lat/lon
projection.

```python
#!/usr/bin/env python3
# This is an example script how groundwater flow and solute transport are
# computed within GRASS
import sys
import os
import grass.script as gs

# Overwrite existing maps
gs.run_command("g.gisenv", set="OVERWRITE=1")

gs.message(_("Set the region"))

# The area is 200m x 100m with a cell size of 1m x 1m
gs.run_command("g.region", res=1, res3=1, t=10, b=0, n=100, s=0, w=0, e=200)
gs.run_command("r.mapcalc", expression="phead = if(col() == 1 , 50, 40)")
gs.run_command("r.mapcalc", expression="phead = if(col() ==200  , 45 + row()/40, phead)")
gs.run_command("r.mapcalc", expression="status = if(col() == 1 || col() == 200 , 2, 1)")
gs.run_command("r.mapcalc", expression="well = if((row() == 50 && col() == 175) || (row() == 10 && col() == 135) , -0.001, 0)")
gs.run_command("r.mapcalc", expression="hydcond = 0.00005")
gs.run_command("r.mapcalc", expression="recharge = 0")
gs.run_command("r.mapcalc", expression="top_conf = 20")
gs.run_command("r.mapcalc", expression="bottom = 0")
gs.run_command("r.mapcalc", expression="poros = 0.17")
gs.run_command("r.mapcalc", expression="syield = 0.0001")
gs.run_command("r.mapcalc", expression="null = 0.0")

gs.message(_("Compute a steady state groundwater flow"))

gs.run_command("r.gwflow", solver="cg", top="top_conf", bottom="bottom", phead="phead",\
  status="status", hc_x="hydcond", hc_y="hydcond", q="well", s="syield",\
  recharge="recharge", output="gwresult_conf", dt=8640000000000, type="confined")

gs.message(_("generate the transport data"))
gs.run_command("r.mapcalc", expression="c = if(col() == 15 && row() == 75 , 500.0, 0.0)")
gs.run_command("r.mapcalc", expression="cs = if(col() == 15 && row() == 75 , 0.0, 0.0)")
gs.run_command("r.mapcalc", expression="tstatus = if(col() == 1 || col() == 200 , 3, 1)")
gs.run_command("r.mapcalc", expression="diff = 0.0000001")
gs.run_command("r.mapcalc", expression="R = 1.0")

# Compute the initial state
gs.run_command("r.solute.transport", solver="bicgstab", top="top_conf",\
  bottom="bottom", phead="gwresult_conf", status="tstatus", hc_x="hydcond", hc_y="hydcond",\
  rd="R", cs="cs", q="well", nf="poros", output="stresult_conf_0", dt=3600, diff_x="diff",\
  diff_y="diff", c="c", al=0.1, at=0.01)

# Compute the solute transport for 300 days in 10 day steps
for dt in range(30):
    gs.run_command("r.solute.transport", solver="bicgstab", top="top_conf",\
    bottom="bottom", phead="gwresult_conf", status="tstatus", hc_x="hydcond", hc_y="hydcond",\
    rd="R", cs="cs", q="well", nf="poros", output="stresult_conf_" + str(dt + 1), dt=864000, diff_x="diff",\
    diff_y="diff", c="stresult_conf_" + str(dt), al=0.1, at=0.01)
```

## SEE ALSO

*[r.gwflow](r.gwflow.md)*  
*[r3.gwflow](r3.gwflow.md)*  
*[r.out.vtk](r.out.vtk.md)*  

## AUTHOR

Sören Gebbert

This work is based on the [Diploma Thesis of Sören Gebbert](https://grass.osgeo.org/gdp/hydrology/gebbert2007_diplom_stroemung_grass_gis.pdf)
at Technical University Berlin in Germany.

## SOURCE CODE

Available at: [r.solute.transport source code](https://github.com/OSGeo/grass/tree/main/raster/r.solute.transport)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.solute.transport))  
Latest change: Wednesday May 21 09:11:03 2025 in commit [4510205](https://github.com/OSGeo/grass/commit/45102055f4f205276ad5f85b5775b70c6583a26f)
