---
name: r.surf.contour
description: Generates surface raster map from rasterized contours.
keywords: [ raster, surface, interpolation ]
---

# r.surf.contour

Generates surface raster map from rasterized contours.

=== "Command line"

    **r.surf.contour**
    **input**=*name*
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.surf.contour input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.surf.contour***",
        **input**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.surf.contour", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_surf_contour*(**input**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_surf_contour(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map containing contours  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map containing contours  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map containing contours  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.surf.contour* creates a raster elevation map from a rasterized
contour map. Elevation values are determined using procedures similar to
a manual methods. To determine the elevation of a point on a contour
map, an individual might interpolate its value from those of the two
nearest contour lines (uphill and downhill).

*r.surf.contour* works in a similar way. Initially, a vector map of the
contour lines is made with the elevation of each line as an attribute.
When the program *[v.to.rast](v.to.rast.md)* is run on the vector map,
continuous "lines" of rasters containing the contour line values will be
the input for *r.surf.contour*. For each cell in the input map, either
the cell is a contour line cell (which is given that value), or a flood
fill is generated from that spot until the fill comes to two unique
values. So the *r.surf.contour* algorithm **linearly interpolates**
between contour lines. The flood fill is not allowed to cross over the
rasterized contour lines, thus ensuring that an uphill and downhill
contour value will be the two values chosen. *r.surf.contour*
interpolates from the uphill and downhill values by the true distance.

### Parameters

**input**=*name*  
Name of an existing raster map that contains a set of initial category
values (i.e., some cells contain known elevation values (denoting
contours) while the rest contain NULL values).

**output**=*name*  
Name to be assigned to new output raster map that represents a smooth
(e.g., elevation) surface generated from the known category values in
the input raster map layer.

An existing mask raster map is respected for both reading *input* and
writing *output*.

## NOTES

*r.surf.contour* works well under the following circumstances: 1) the
contour lines extend to the edge of the current region, 2) the
program is run at the same resolution as that of the input map, 3) there
are no disjointed contour lines, and 4) no spot elevation data BETWEEN
contour lines exist. Spot elevations at the tops of hills and the
bottoms of depressions, on the other hand, improve the output greatly.
Violating these constraints will cause non-intuitive anomalies to appear
in the output map. Run *[r.slope.aspect](r.slope.aspect.md)* on
*r.surf.contour* results to locate potential anomalies.

The running of *r.surf.contour* is very sensitive to the resolution of
rasterized vector map. If multiple contour lines go through the same
raster, slight anomalies may occur. The speed of *r.surf.contour* is
dependent on how far "apart" the contour lines are from each other (as
measured in raster cells). Since a flood fill algorithm is used, the
program's running time will grow exponentially with the distance between
contour lines.

## EXAMPLE

Example to create contour lines from elevation model, then recreating
DEM from these contour lines along with differences analysis (North
Carolina sample data set):

```sh
g.region raster=elevation -p

# get minimum elevation value
r.univar elevation

# generate vector contour lines
r.contour input=elevation output=contours_5m step=5 minlevel=50

# rasterize contour lines
v.info -c contours_5m
v.to.rast input=contours_5m output=contours_5m use=attr attribute_column=level

# generate DEM from rasterized contour lines
r.surf.contour input=contours_5m output=elevation_from_cont5m

# calculate difference map
r.mapcalc "diff = elevation - elevation_from_cont5m"
r.colors diff color=differences

# analyze differences statistically
r.univar diff
```

## SEE ALSO

*[r.mapcalc](r.mapcalc.md), [r.slope.aspect](r.slope.aspect.md),
[r.surf.idw](r.surf.idw.md), [wxGUI vector digitizer](wxGUI.vdigit.md),
[v.surf.idw](v.surf.idw.md), [v.surf.rst](v.surf.rst.md),
[v.to.rast](v.to.rast.md)*

Overview: [Interpolation and
Resampling](https://grasswiki.osgeo.org/wiki/Interpolation) in GRASS

## AUTHOR

Chuck Ehlschlaeger, U.S. Army Construction Engineering Research
Laboratory

## SOURCE CODE

Available at: [r.surf.contour source code](https://github.com/OSGeo/grass/tree/main/raster/r.surf.contour)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.surf.contour))  
Latest change: Friday Aug 08 09:39:19 2025 in commit [ed72c71](https://github.com/OSGeo/grass/commit/ed72c71d8580e872cd6fd345c84a15c324764465)
