---
title: Standard Parser Options
author: GRASS Development Team
---


# Standard Parser Options
Standard parser options are simplifying definitions of common
parameters when writing GRASS tools. Their identifiers are the same
across (programming) languages. In Python, place an identifier right
after `# %option` (on the same line separated by a space).
In C, use function `G_define_standard_option()`.
The individual items in the definition can be overridden if the tool
needs, for example, a different key (parameter name) or description.
The use of standard parser options is strongly recommended, with or without
overrides, for all common GRASS parameters of such as raster and vector inputs.
See *[g.parser](g.parser.md)* documentation for details on defining
tool parameters.

## Color

### G_OPT_C

The standard option G_OPT_C has the following default values:

| item | value |
| ---- | ----- |
| key | color |
| label | Color |
| description | Either a standard color name or R:G:B triplet |
| required | NO (`required: no`) |
| answer | DEFAULT_FG_COLOR |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,color,color |
| key_desc | name |

Python usage:

```python
# %option G_OPT_C
# %end
```

To use the value in Python:

```python
color = options["color"]
```

What a tool user (caller) may write:

```python
color="name"
```

### G_OPT_CN

The standard option G_OPT_CN has the following default values:

| item | value |
| ---- | ----- |
| key | color |
| label | Color |
| description | Either a standard color name, R:G:B triplet, or "none" |
| required | NO (`required: no`) |
| answer | DEFAULT_FG_COLOR |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,color_none,color |
| key_desc | name |

Python usage:

```python
# %option G_OPT_CN
# %end
```

To use the value in Python:

```python
color = options["color"]
```

What a tool user (caller) may write:

```python
color="name"
```

### G_OPT_C_FORMAT

The standard option G_OPT_C_FORMAT has the following default values:

| item | value |
| ---- | ----- |
| key | color_format |
| label | Color format |
| description | Color format for output values. |
| required | YES (`required: yes`) |
| multiple | NO (`multiple: no`) |
| answer | hex |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |
| options | rgb,hex,hsv,triplet |

Python usage:

```python
# %option G_OPT_C_FORMAT
# %end
```

To use the value in Python:

```python
color_format = options["color_format"]
```

What a tool user (caller) may write:

```python
color_format="hex"
```

## Database

### G_OPT_DB_SQL

The standard option G_OPT_DB_SQL has the following default values:

| item | value |
| ---- | ----- |
| key | sql |
| label | SQL SELECT statement |
| description | Example: select * from towns where population > 10000 |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| key_desc | sql_query |

Python usage:

```python
# %option G_OPT_DB_SQL
# %end
```

To use the value in Python:

```python
sql = options["sql"]
```

What a tool user (caller) may write:

```python
sql="sql_query"
```

### G_OPT_DB_WHERE

The standard option G_OPT_DB_WHERE has the following default values:

| item | value |
| ---- | ----- |
| key | where |
| label | WHERE conditions of SQL statement without 'where' keyword |
| description | Example: income < 1000 and population >= 10000 |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,sql_query,sql_query |
| key_desc | sql_query |

Python usage:

```python
# %option G_OPT_DB_WHERE
# %end
```

To use the value in Python:

```python
where = options["where"]
```

What a tool user (caller) may write:

```python
where="sql_query"
```

### G_OPT_DB_TABLE

The standard option G_OPT_DB_TABLE has the following default values:

| item | value |
| ---- | ----- |
| key | table |
| description | Name of attribute table |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dbtable,dbtable |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_TABLE
# %end
```

To use the value in Python:

```python
table = options["table"]
```

What a tool user (caller) may write:

```python
table="name"
```

### G_OPT_DB_DRIVER

The standard option G_OPT_DB_DRIVER has the following default values:

| item | value |
| ---- | ----- |
| key | driver |
| description | Name of database driver |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dbdriver,dbdriver |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_DRIVER
# %end
```

To use the value in Python:

```python
driver = options["driver"]
```

What a tool user (caller) may write:

```python
driver="name"
```

### G_OPT_DB_DATABASE

The standard option G_OPT_DB_DATABASE has the following default values:

| item | value |
| ---- | ----- |
| key | database |
| description | Name of database |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dbname,dbname |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_DATABASE
# %end
```

To use the value in Python:

```python
database = options["database"]
```

What a tool user (caller) may write:

```python
database="name"
```

### G_OPT_DB_SCHEMA

The standard option G_OPT_DB_SCHEMA has the following default values:

| item | value |
| ---- | ----- |
| key | schema |
| label | Database schema |
| description | Do not use this option if schemas are not supported by driver/database server |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_SCHEMA
# %end
```

To use the value in Python:

```python
schema = options["schema"]
```

What a tool user (caller) may write:

```python
schema="name"
```

### G_OPT_DB_COLUMN

The standard option G_OPT_DB_COLUMN has the following default values:

| item | value |
| ---- | ----- |
| key | column |
| description | Name of attribute column |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dbcolumn,dbcolumn |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_COLUMN
# %end
```

To use the value in Python:

```python
column = options["column"]
```

What a tool user (caller) may write:

```python
column="name"
```

### G_OPT_DB_COLUMNS

The standard option G_OPT_DB_COLUMNS has the following default values:

| item | value |
| ---- | ----- |
| key | columns |
| description | Name of attribute column(s) |
| required | NO (`required: no`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dbcolumn,dbcolumn |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_COLUMNS
# %end
```

To use the value in Python:

```python
columns = options["columns"]
```

What a tool user (caller) may write:

```python
columns="name"
```

### G_OPT_DB_KEYCOLUMN

The standard option G_OPT_DB_KEYCOLUMN has the following default values:

| item | value |
| ---- | ----- |
| key | key |
| label | Name of key column |
| description | Must refer to an integer column |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | GV_KEY_COLUMN |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |

Python usage:

```python
# %option G_OPT_DB_KEYCOLUMN
# %end
```

To use the value in Python:

```python
key = options["key"]
```

What a tool user (caller) may write:

```python
key="GV_KEY_COLUMN"
```

## File

### G_OPT_F_INPUT

The standard option G_OPT_F_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input file |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,file,file |
| key_desc | name |

Python usage:

```python
# %option G_OPT_F_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_F_BIN_INPUT

The standard option G_OPT_F_BIN_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input file |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,bin,file |
| key_desc | name |

Python usage:

```python
# %option G_OPT_F_BIN_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_F_OUTPUT

The standard option G_OPT_F_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name for output file |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,file,file |
| key_desc | name |

Python usage:

```python
# %option G_OPT_F_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_F_SEP

The standard option G_OPT_F_SEP has the following default values:

| item | value |
| ---- | ----- |
| key | separator |
| label | Field separator |
| description | Special characters: pipe, comma, space, tab, newline |
| required | NO (`required: no`) |
| answer | pipe |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,separator,separator |
| key_desc | character |

Python usage:

```python
# %option G_OPT_F_SEP
# %end
```

To use the value in Python:

```python
separator = options["separator"]
```

What a tool user (caller) may write:

```python
separator="pipe"
```

### G_OPT_F_FORMAT

The standard option G_OPT_F_FORMAT has the following default values:

| item | value |
| ---- | ----- |
| key | format |
| label | Output format |
| required | YES (`required: yes`) |
| answer | plain |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |
| options | plain,json |
| descriptions | plain;Plain text output;json;JSON (JavaScript Object Notation); |

Python usage:

```python
# %option G_OPT_F_FORMAT
# %end
```

To use the value in Python:

```python
format = options["format"]
```

What a tool user (caller) may write:

```python
format="plain"
```

## Imagery

### G_OPT_I_GROUP

The standard option G_OPT_I_GROUP has the following default values:

| item | value |
| ---- | ----- |
| key | group |
| description | Name of input imagery group |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,group,group |
| key_desc | name |

Python usage:

```python
# %option G_OPT_I_GROUP
# %end
```

To use the value in Python:

```python
group = options["group"]
```

What a tool user (caller) may write:

```python
group="name"
```

### G_OPT_I_SUBGROUP

The standard option G_OPT_I_SUBGROUP has the following default values:

| item | value |
| ---- | ----- |
| key | subgroup |
| description | Name of input imagery subgroup |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,subgroup,subgroup |
| key_desc | name |

Python usage:

```python
# %option G_OPT_I_SUBGROUP
# %end
```

To use the value in Python:

```python
subgroup = options["subgroup"]
```

What a tool user (caller) may write:

```python
subgroup="name"
```

## Map (of Any Type)

### G_OPT_MAP_INPUT

The standard option G_OPT_MAP_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| description | Name of the input map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,map,map |
| key_desc | name |

Python usage:

```python
# %option G_OPT_MAP_INPUT
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_MAP_INPUTS

The standard option G_OPT_MAP_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | maps |
| description | Name of the input maps |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,map,map |
| key_desc | name |

Python usage:

```python
# %option G_OPT_MAP_INPUTS
# %end
```

To use the value in Python:

```python
maps = options["maps"]
```

What a tool user (caller) may write:

```python
maps="name"
```

### G_OPT_MAP_TYPE

The standard option G_OPT_MAP_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| description | Type of the input map |
| required | NO (`required: no`) |
| answer | raster |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |
| options | raster,vector,raster_3d |

Python usage:

```python
# %option G_OPT_MAP_TYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="raster"
```

## Miscellaneous

### G_OPT_M_DIR

The standard option G_OPT_M_DIR has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input directory |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dir,dir |
| key_desc | name |

Python usage:

```python
# %option G_OPT_M_DIR
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_M_UNITS

The standard option G_OPT_M_UNITS has the following default values:

| item | value |
| ---- | ----- |
| key | units |
| description | Units |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| options | miles,feet,meters,kilometers,acres,hectares |

Python usage:

```python
# %option G_OPT_M_UNITS
# %end
```

To use the value in Python:

```python
units = options["units"]
```

What a tool user (caller) may write:

```python
units="..."
```

### G_OPT_M_DATATYPE

The standard option G_OPT_M_DATATYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| description | Data type(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| key_desc | datatype |

Python usage:

```python
# %option G_OPT_M_DATATYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="datatype"
```

### G_OPT_M_MAPSET

The standard option G_OPT_M_MAPSET has the following default values:

| item | value |
| ---- | ----- |
| key | mapset |
| label | Name of mapset (default: current search path) |
| description | '.' for current mapset |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,mapset,mapset |
| key_desc | name |

Python usage:

```python
# %option G_OPT_M_MAPSET
# %end
```

To use the value in Python:

```python
mapset = options["mapset"]
```

What a tool user (caller) may write:

```python
mapset="name"
```

### G_OPT_M_LOCATION

The standard option G_OPT_M_LOCATION has the following default values:

| item | value |
| ---- | ----- |
| key | project |
| label | Project (location) name |
| description | Project name (not path to project) |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,location,location |
| key_desc | name |

Python usage:

```python
# %option G_OPT_M_LOCATION
# %end
```

To use the value in Python:

```python
project = options["project"]
```

What a tool user (caller) may write:

```python
project="name"
```

### G_OPT_M_DBASE

The standard option G_OPT_M_DBASE has the following default values:

| item | value |
| ---- | ----- |
| key | dbase |
| label | GRASS database directory |
| description | Default: path to the current GRASS database |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,dbase,dbase |
| key_desc | path |

Python usage:

```python
# %option G_OPT_M_DBASE
# %end
```

To use the value in Python:

```python
dbase = options["dbase"]
```

What a tool user (caller) may write:

```python
dbase="path"
```

### G_OPT_M_COORDS

The standard option G_OPT_M_COORDS has the following default values:

| item | value |
| ---- | ----- |
| key | coordinates |
| description | Coordinates |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_DOUBLE (`type: double`) |
| gisprompt | old,coords,coords |
| key_desc | east,north |

Python usage:

```python
# %option G_OPT_M_COORDS
# %end
```

To use the value in Python:

```python
coordinates = options["coordinates"]
```

What a tool user (caller) may write:

```python
coordinates=east,north
```

### G_OPT_M_COLR

The standard option G_OPT_M_COLR has the following default values:

| item | value |
| ---- | ----- |
| key | color |
| description | Name of color table |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,colortable,colortable |
| key_desc | style |
| options | Computed dynamically by *G_color_rules_options()* |
| descriptions | Computed dynamically by *G_color_rules_description_type()* |

Python usage:

```python
# %option G_OPT_M_COLR
# %end
```

To use the value in Python:

```python
color = options["color"]
```

What a tool user (caller) may write:

```python
color="style"
```

### G_OPT_M_NULL_VALUE

The standard option G_OPT_M_NULL_VALUE has the following default values:

| item | value |
| ---- | ----- |
| key | null_value |
| description | String representing NULL value |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| key_desc | string |

Python usage:

```python
# %option G_OPT_M_NULL_VALUE
# %end
```

To use the value in Python:

```python
null_value = options["null_value"]
```

What a tool user (caller) may write:

```python
null_value="string"
```

### G_OPT_M_REGION

The standard option G_OPT_M_REGION has the following default values:

| item | value |
| ---- | ----- |
| key | region |
| description | Name of saved region |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,windows,region |
| key_desc | name |

Python usage:

```python
# %option G_OPT_M_REGION
# %end
```

To use the value in Python:

```python
region = options["region"]
```

What a tool user (caller) may write:

```python
region="name"
```

### G_OPT_M_NPROCS

The standard option G_OPT_M_NPROCS has the following default values:

| item | value |
| ---- | ----- |
| key | nprocs |
| label | Number of threads for parallel computing |
| description | 0: use OpenMP default; >0: use nprocs; <0: use MAX-nprocs |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | 0 |
| type | TYPE_INTEGER (`type: integer`) |

Python usage:

```python
# %option G_OPT_M_NPROCS
# %end
```

To use the value in Python:

```python
nprocs = options["nprocs"]
```

What a tool user (caller) may write:

```python
nprocs=0
```

### G_OPT_M_SEED

The standard option G_OPT_M_SEED has the following default values:

| item | value |
| ---- | ----- |
| key | seed |
| label | Seed value for the random number generator |
| description | Using the same seed ensures identical results, while a randomly generated seed produces different outcomes in each run. |
| required | NO (`required: no`) |
| type | TYPE_INTEGER (`type: integer`) |

Python usage:

```python
# %option G_OPT_M_SEED
# %end
```

To use the value in Python:

```python
seed = options["seed"]
```

What a tool user (caller) may write:

```python
seed=...
```

## Raster

### G_OPT_R_INPUT

The standard option G_OPT_R_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_R_INPUTS

The standard option G_OPT_R_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input raster map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_INPUTS
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_R_OUTPUT

The standard option G_OPT_R_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name for output raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_R_OUTPUTS

The standard option G_OPT_R_OUTPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name for output raster map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_OUTPUTS
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_R_MAP

The standard option G_OPT_R_MAP has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| description | Name of raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_MAP
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_R_MAPS

The standard option G_OPT_R_MAPS has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| description | Name of raster map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_MAPS
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_R_BASE

The standard option G_OPT_R_BASE has the following default values:

| item | value |
| ---- | ----- |
| key | base |
| description | Name of base raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_BASE
# %end
```

To use the value in Python:

```python
base = options["base"]
```

What a tool user (caller) may write:

```python
base="name"
```

### G_OPT_R_COVER

The standard option G_OPT_R_COVER has the following default values:

| item | value |
| ---- | ----- |
| key | cover |
| description | Name of cover raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_COVER
# %end
```

To use the value in Python:

```python
cover = options["cover"]
```

What a tool user (caller) may write:

```python
cover="name"
```

### G_OPT_R_ELEV

The standard option G_OPT_R_ELEV has the following default values:

| item | value |
| ---- | ----- |
| key | elevation |
| description | Name of input elevation raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_ELEV
# %end
```

To use the value in Python:

```python
elevation = options["elevation"]
```

What a tool user (caller) may write:

```python
elevation="name"
```

### G_OPT_R_ELEVS

The standard option G_OPT_R_ELEVS has the following default values:

| item | value |
| ---- | ----- |
| key | elevation |
| description | Name of input elevation raster map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R_ELEVS
# %end
```

To use the value in Python:

```python
elevation = options["elevation"]
```

What a tool user (caller) may write:

```python
elevation="name"
```

### G_OPT_R_TYPE

The standard option G_OPT_R_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| label | Type of raster map to be created |
| description | Storage type for resultant raster map |
| required | YES (`required: yes`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| options | CELL,FCELL,DCELL |

Python usage:

```python
# %option G_OPT_R_TYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="..."
```

### G_OPT_R_INTERP_TYPE

The standard option G_OPT_R_INTERP_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | method |
| description | Sampling interpolation method |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| options | nearest,bilinear,bicubic |

Python usage:

```python
# %option G_OPT_R_INTERP_TYPE
# %end
```

To use the value in Python:

```python
method = options["method"]
```

What a tool user (caller) may write:

```python
method="..."
```

### G_OPT_R_BASENAME_INPUT

The standard option G_OPT_R_BASENAME_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input basename raster map(s) |
| required | YES (`required: yes`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cell,raster |
| key_desc | basename |

Python usage:

```python
# %option G_OPT_R_BASENAME_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="basename"
```

### G_OPT_R_BASENAME_OUTPUT

The standard option G_OPT_R_BASENAME_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name for output basename raster map(s) |
| required | YES (`required: yes`) |
| multiple | NO (`multiple: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,cell,raster |
| key_desc | basename |

Python usage:

```python
# %option G_OPT_R_BASENAME_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="basename"
```

## Raster 3D

### G_OPT_R3_INPUT

The standard option G_OPT_R3_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input 3D raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,grid3,raster_3d |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R3_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_R3_INPUTS

The standard option G_OPT_R3_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of input 3D raster map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,grid3,raster_3d |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R3_INPUTS
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_R3_OUTPUT

The standard option G_OPT_R3_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name for output 3D raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,grid3,raster_3d |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R3_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_R3_MAP

The standard option G_OPT_R3_MAP has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| description | Name of 3D raster map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,grid3,raster_3d |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R3_MAP
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_R3_MAPS

The standard option G_OPT_R3_MAPS has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| description | Name of 3D raster map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,grid3,raster_3d |
| key_desc | name |

Python usage:

```python
# %option G_OPT_R3_MAPS
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_R3_TYPE

The standard option G_OPT_R3_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| description | Data type used in the output raster3d map |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | default |
| type | TYPE_STRING (`type: string`) |
| options | default,double,float |

Python usage:

```python
# %option G_OPT_R3_TYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="default"
```

### G_OPT_R3_PRECISION

The standard option G_OPT_R3_PRECISION has the following default values:

| item | value |
| ---- | ----- |
| key | precision |
| description | Number of digits used as mantissa in the internal map storage, 0 -23 for float, 0 - 52 for double, max or default |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | default |
| type | TYPE_STRING (`type: string`) |

Python usage:

```python
# %option G_OPT_R3_PRECISION
# %end
```

To use the value in Python:

```python
precision = options["precision"]
```

What a tool user (caller) may write:

```python
precision="default"
```

### G_OPT_R3_COMPRESSION

The standard option G_OPT_R3_COMPRESSION has the following default values:

| item | value |
| ---- | ----- |
| key | compression |
| description | The compression method used in the output raster3d map |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | default |
| type | TYPE_STRING (`type: string`) |
| options | default,zip,none |

Python usage:

```python
# %option G_OPT_R3_COMPRESSION
# %end
```

To use the value in Python:

```python
compression = options["compression"]
```

What a tool user (caller) may write:

```python
compression="default"
```

### G_OPT_R3_TILE_DIMENSION

The standard option G_OPT_R3_TILE_DIMENSION has the following default values:

| item | value |
| ---- | ----- |
| key | tiledimension |
| description | The dimensions of the tiles used in the output raster3d map (XxYxZ or default: 16x16x8) |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | default |
| type | TYPE_STRING (`type: string`) |
| key_desc | XxYxZ |

Python usage:

```python
# %option G_OPT_R3_TILE_DIMENSION
# %end
```

To use the value in Python:

```python
tiledimension = options["tiledimension"]
```

What a tool user (caller) may write:

```python
tiledimension="default"
```

## Temporal

### G_OPT_STDS_INPUT

The standard option G_OPT_STDS_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of the input space time dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,stds,stds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STDS_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_STDS_INPUTS

The standard option G_OPT_STDS_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | inputs |
| description | Name of the input space time datasets |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,stds,stds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STDS_INPUTS
# %end
```

To use the value in Python:

```python
inputs = options["inputs"]
```

What a tool user (caller) may write:

```python
inputs="name"
```

### G_OPT_STDS_OUTPUT

The standard option G_OPT_STDS_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name of the output space time dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,stds,stds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STDS_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_STRDS_INPUT

The standard option G_OPT_STRDS_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of the input space time raster dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,strds,strds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STRDS_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_STRDS_INPUTS

The standard option G_OPT_STRDS_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | inputs |
| description | Name of the input space time raster datasets |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,strds,strds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STRDS_INPUTS
# %end
```

To use the value in Python:

```python
inputs = options["inputs"]
```

What a tool user (caller) may write:

```python
inputs="name"
```

### G_OPT_STRDS_OUTPUT

The standard option G_OPT_STRDS_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name of the output space time raster dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,strds,strds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STRDS_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_STRDS_OUTPUTS

The standard option G_OPT_STRDS_OUTPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | outputs |
| description | Name of the output space time raster datasets |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,strds,strds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STRDS_OUTPUTS
# %end
```

To use the value in Python:

```python
outputs = options["outputs"]
```

What a tool user (caller) may write:

```python
outputs="name"
```

### G_OPT_STVDS_INPUT

The standard option G_OPT_STVDS_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of the input space time vector dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,stvds,stvds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STVDS_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_STVDS_INPUTS

The standard option G_OPT_STVDS_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | inputs |
| description | Name of the input space time vector datasets |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,stvds,stvds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STVDS_INPUTS
# %end
```

To use the value in Python:

```python
inputs = options["inputs"]
```

What a tool user (caller) may write:

```python
inputs="name"
```

### G_OPT_STVDS_OUTPUT

The standard option G_OPT_STVDS_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name of the output space time vector dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,stvds,stvds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STVDS_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_STR3DS_INPUT

The standard option G_OPT_STR3DS_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| description | Name of the input space time raster3d dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,str3ds,str3ds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STR3DS_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_STR3DS_INPUTS

The standard option G_OPT_STR3DS_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | inputs |
| description | Name of the input space time raster3d datasets |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,str3ds,str3ds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STR3DS_INPUTS
# %end
```

To use the value in Python:

```python
inputs = options["inputs"]
```

What a tool user (caller) may write:

```python
inputs="name"
```

### G_OPT_STR3DS_OUTPUT

The standard option G_OPT_STR3DS_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name of the output space time raster3d dataset |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,str3ds,str3ds |
| key_desc | name |

Python usage:

```python
# %option G_OPT_STR3DS_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_STDS_TYPE

The standard option G_OPT_STDS_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| description | Type of the input space time dataset |
| required | NO (`required: no`) |
| answer | strds |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |
| options | strds,stvds,str3ds |

Python usage:

```python
# %option G_OPT_STDS_TYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="strds"
```

### G_OPT_T_SUFFIX

The standard option G_OPT_T_SUFFIX has the following default values:

| item | value |
| ---- | ----- |
| key | suffix |
| description | Suffix to add at basename: set 'gran' for granularity, 'time' for the full time format, 'num' for numerical suffix with a specific number of digits (default %%05) |
| required | NO (`required: no`) |
| answer | gran |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |

Python usage:

```python
# %option G_OPT_T_SUFFIX
# %end
```

To use the value in Python:

```python
suffix = options["suffix"]
```

What a tool user (caller) may write:

```python
suffix="gran"
```

### G_OPT_T_TYPE

The standard option G_OPT_T_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | temporaltype |
| description | The temporal type of the space time dataset |
| required | NO (`required: no`) |
| answer | absolute |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |
| options | absolute,relative |

Python usage:

```python
# %option G_OPT_T_TYPE
# %end
```

To use the value in Python:

```python
temporaltype = options["temporaltype"]
```

What a tool user (caller) may write:

```python
temporaltype="absolute"
```

### G_OPT_T_WHERE

The standard option G_OPT_T_WHERE has the following default values:

| item | value |
| ---- | ----- |
| key | where |
| label | WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework |
| description | Example: start_time > '2001-01-01 12:30:00' |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| key_desc | sql_query |

Python usage:

```python
# %option G_OPT_T_WHERE
# %end
```

To use the value in Python:

```python
where = options["where"]
```

What a tool user (caller) may write:

```python
where="sql_query"
```

### G_OPT_T_SAMPLE

The standard option G_OPT_T_SAMPLE has the following default values:

| item | value |
| ---- | ----- |
| key | sampling |
| description | The method to be used for sampling the input dataset |
| required | NO (`required: no`) |
| multiple | YES (`multiple: yes`) |
| answer | start |
| type | TYPE_STRING (`type: string`) |
| key_desc | name |
| options | start,during,overlap,contain,equal,follows,precedes |

Python usage:

```python
# %option G_OPT_T_SAMPLE
# %end
```

To use the value in Python:

```python
sampling = options["sampling"]
```

What a tool user (caller) may write:

```python
sampling="start"
```

## Vector

### G_OPT_V_INPUT

The standard option G_OPT_V_INPUT has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| label | Name of input vector map |
| description | Or data source for direct OGR access |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,vector,vector |
| key_desc | name |

Python usage:

```python
# %option G_OPT_V_INPUT
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_V_INPUTS

The standard option G_OPT_V_INPUTS has the following default values:

| item | value |
| ---- | ----- |
| key | input |
| label | Name of input vector map(s) |
| description | Or data source(s) for direct OGR access |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,vector,vector |
| key_desc | name |

Python usage:

```python
# %option G_OPT_V_INPUTS
# %end
```

To use the value in Python:

```python
input = options["input"]
```

What a tool user (caller) may write:

```python
input="name"
```

### G_OPT_V_OUTPUT

The standard option G_OPT_V_OUTPUT has the following default values:

| item | value |
| ---- | ----- |
| key | output |
| description | Name for output vector map |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | new,vector,vector |
| key_desc | name |

Python usage:

```python
# %option G_OPT_V_OUTPUT
# %end
```

To use the value in Python:

```python
output = options["output"]
```

What a tool user (caller) may write:

```python
output="name"
```

### G_OPT_V_MAP

The standard option G_OPT_V_MAP has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| label | Name of vector map |
| description | Or data source for direct OGR access |
| required | YES (`required: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,vector,vector |
| key_desc | name |

Python usage:

```python
# %option G_OPT_V_MAP
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_V_MAPS

The standard option G_OPT_V_MAPS has the following default values:

| item | value |
| ---- | ----- |
| key | map |
| description | Name of vector map(s) |
| required | YES (`required: yes`) |
| multiple | YES (`multiple: yes`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,vector,vector |
| key_desc | name |

Python usage:

```python
# %option G_OPT_V_MAPS
# %end
```

To use the value in Python:

```python
map = options["map"]
```

What a tool user (caller) may write:

```python
map="name"
```

### G_OPT_V_TYPE

The standard option G_OPT_V_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| description | Input feature type |
| required | NO (`required: no`) |
| multiple | YES (`multiple: yes`) |
| answer | point,line,boundary,centroid,area |
| type | TYPE_STRING (`type: string`) |
| options | point,line,boundary,centroid,area |

Python usage:

```python
# %option G_OPT_V_TYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="point,line,boundary,centroid,area"
```

### G_OPT_V3_TYPE

The standard option G_OPT_V3_TYPE has the following default values:

| item | value |
| ---- | ----- |
| key | type |
| description | Input feature type |
| required | NO (`required: no`) |
| multiple | YES (`multiple: yes`) |
| answer | point,line,boundary,centroid,area,face,kernel |
| type | TYPE_STRING (`type: string`) |
| options | point,line,boundary,centroid,area,face,kernel |

Python usage:

```python
# %option G_OPT_V3_TYPE
# %end
```

To use the value in Python:

```python
type = options["type"]
```

What a tool user (caller) may write:

```python
type="point,line,boundary,centroid,area,face,kernel"
```

### G_OPT_V_FIELD

The standard option G_OPT_V_FIELD has the following default values:

| item | value |
| ---- | ----- |
| key | layer |
| label | Layer number or name |
| description | Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name. |
| required | NO (`required: no`) |
| answer | 1 |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,layer,layer |

Python usage:

```python
# %option G_OPT_V_FIELD
# %end
```

To use the value in Python:

```python
layer = options["layer"]
```

What a tool user (caller) may write:

```python
layer="1"
```

### G_OPT_V_FIELD_ALL

The standard option G_OPT_V_FIELD_ALL has the following default values:

| item | value |
| ---- | ----- |
| key | layer |
| label | Layer number or name ('-1' for all layers) |
| description | A single vector map can be connected to multiple database tables. This number determines which table to use. When used with direct OGR access this is the layer name. |
| required | NO (`required: no`) |
| answer | -1 |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,layer_all,layer |

Python usage:

```python
# %option G_OPT_V_FIELD_ALL
# %end
```

To use the value in Python:

```python
layer = options["layer"]
```

What a tool user (caller) may write:

```python
layer="-1"
```

### G_OPT_V_CAT

The standard option G_OPT_V_CAT has the following default values:

| item | value |
| ---- | ----- |
| key | cat |
| description | Category value |
| required | NO (`required: no`) |
| type | TYPE_INTEGER (`type: integer`) |
| gisprompt | old,cat,cats |

Python usage:

```python
# %option G_OPT_V_CAT
# %end
```

To use the value in Python:

```python
cat = options["cat"]
```

What a tool user (caller) may write:

```python
cat=...
```

### G_OPT_V_CATS

The standard option G_OPT_V_CATS has the following default values:

| item | value |
| ---- | ----- |
| key | cats |
| label | Category values |
| description | Example: 1,3,7-9,13 |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| gisprompt | old,cats,cats |
| key_desc | range |

Python usage:

```python
# %option G_OPT_V_CATS
# %end
```

To use the value in Python:

```python
cats = options["cats"]
```

What a tool user (caller) may write:

```python
cats="range"
```

### G_OPT_V_ID

The standard option G_OPT_V_ID has the following default values:

| item | value |
| ---- | ----- |
| key | id |
| description | Feature id |
| required | NO (`required: no`) |
| type | TYPE_INTEGER (`type: integer`) |

Python usage:

```python
# %option G_OPT_V_ID
# %end
```

To use the value in Python:

```python
id = options["id"]
```

What a tool user (caller) may write:

```python
id=...
```

### G_OPT_V_IDS

The standard option G_OPT_V_IDS has the following default values:

| item | value |
| ---- | ----- |
| key | ids |
| label | Feature ids |
| description | Example: 1,3,7-9,13 |
| required | NO (`required: no`) |
| type | TYPE_STRING (`type: string`) |
| key_desc | range |

Python usage:

```python
# %option G_OPT_V_IDS
# %end
```

To use the value in Python:

```python
ids = options["ids"]
```

What a tool user (caller) may write:

```python
ids="range"
```

## Other

### G_OPT_MEMORYMB

The standard option G_OPT_MEMORYMB has the following default values:

| item | value |
| ---- | ----- |
| key | memory |
| label | Maximum memory to be used (in MB) |
| description | Cache size for raster rows |
| required | NO (`required: no`) |
| multiple | NO (`multiple: no`) |
| answer | 300 |
| type | TYPE_INTEGER (`type: integer`) |
| key_desc | memory in MB |

Python usage:

```python
# %option G_OPT_MEMORYMB
# %end
```

To use the value in Python:

```python
memory = options["memory"]
```

What a tool user (caller) may write:

```python
memory=300
```
