---
name: g.proj
description: Prints or modifies GRASS projection information files (in various coordinate system descriptions). Can also be used to create new GRASS projects.
keywords: [ general, projection, create project ]
---

# g.proj

Prints or modifies GRASS projection information files (in various coordinate system descriptions).

Can also be used to create new GRASS projects.

=== "Command line"

    **g.proj**
    [**-pgdjfwetc**]
    [**georef**=*file*]
    [**wkt**=*file*]
    [**srid**=*params*]
    [**proj4**=*params*]
    [**epsg**=*code*]
    [**list_codes**=*string*]
    [**datum**=*name*]
    [**datum_trans**=*index*]
    [**project**=*name*]
    **format**=*name*
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    g.proj format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***g.proj***",
        **georef**=*None*,
        **wkt**=*None*,
        **srid**=*None*,
        **proj4**=*None*,
        **epsg**=*None*,
        **list_codes**=*None*,
        **datum**=*None*,
        **datum_trans**=*0*,
        **project**=*None*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("g.proj", format="plain")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.g_proj*(**georef**=*None*,
        **wkt**=*None*,
        **srid**=*None*,
        **proj4**=*None*,
        **epsg**=*None*,
        **list_codes**=*None*,
        **datum**=*None*,
        **datum_trans**=*0*,
        **project**=*None*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.g_proj(format="plain")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **georef**=*file*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of georeferenced data file to read projection information from  
    **wkt**=*file*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of ASCII file containing a WKT projection description  
    &nbsp;&nbsp;&nbsp;&nbsp;'-' for standard input  
    **srid**=*params*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spatial reference ID with authority name and code  
    &nbsp;&nbsp;&nbsp;&nbsp;E.g. EPSG:4326 or urn:ogc:def:crs:EPSG::4326  
    **proj4**=*params*  
    &nbsp;&nbsp;&nbsp;&nbsp;PROJ.4 projection description  
    &nbsp;&nbsp;&nbsp;&nbsp;'-' for standard input  
    **epsg**=*code*  
    &nbsp;&nbsp;&nbsp;&nbsp;EPSG projection code  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-1000000*  
    **list_codes**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;List codes for given authority, e.g. EPSG, and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *EPSG, ESRI, IAU_2015, IGNF, NKG, NRCAN, OGC, PROJ*  
    **datum**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Datum (overrides any datum specified in input coordinate system)  
    &nbsp;&nbsp;&nbsp;&nbsp;Accepts standard GRASS datum codes, or "list" to list and exit  
    **datum_trans**=*index*  
    &nbsp;&nbsp;&nbsp;&nbsp;Index number of datum transform parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;"0" for unspecified or "-1" to list and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *-1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **project**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of new project (location) to create  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, shell, wkt, projjson, proj4*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**wkt**: Well-known text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**projjson**: JSON (JavaScript Object Notation) version of WKT  
    &nbsp;&nbsp;&nbsp;&nbsp;**proj4**: PROJ.4 style text output  
    **-p**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print projection information  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print projection information in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verify datum information and print transformation parameters  
    **-j**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print projection information in PROJ.4 format [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=proj4 instead.  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print 'flat' output with no linebreaks (applies to WKT and PROJ.4 output)  
    **-w**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print projection information in WKT format [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=wkt instead.  
    **-e**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use ESRI-style format (applies to WKT output only)  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force override of datum transformation information in input coordinate system  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Modify current project's projection files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **georef** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of georeferenced data file to read projection information from  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *file*  
    **wkt** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of ASCII file containing a WKT projection description  
    &nbsp;&nbsp;&nbsp;&nbsp;'-' for standard input  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *file*  
    **srid** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spatial reference ID with authority name and code  
    &nbsp;&nbsp;&nbsp;&nbsp;E.g. EPSG:4326 or urn:ogc:def:crs:EPSG::4326  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *params*  
    **proj4** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;PROJ.4 projection description  
    &nbsp;&nbsp;&nbsp;&nbsp;'-' for standard input  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *params*  
    **epsg** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;EPSG projection code  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *code*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-1000000*  
    **list_codes** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;List codes for given authority, e.g. EPSG, and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *EPSG, ESRI, IAU_2015, IGNF, NKG, NRCAN, OGC, PROJ*  
    **datum** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Datum (overrides any datum specified in input coordinate system)  
    &nbsp;&nbsp;&nbsp;&nbsp;Accepts standard GRASS datum codes, or "list" to list and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    **datum_trans** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Index number of datum transform parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;"0" for unspecified or "-1" to list and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *index*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *-1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **project** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of new project (location) to create  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, shell, wkt, projjson, proj4*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**wkt**: Well-known text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**projjson**: JSON (JavaScript Object Notation) version of WKT  
    &nbsp;&nbsp;&nbsp;&nbsp;**proj4**: PROJ.4 style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *p*, *g*, *d*, *j*, *f*, *w*, *e*, *t*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Verify datum information and print transformation parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;**j**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information in PROJ.4 format [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=proj4 instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print 'flat' output with no linebreaks (applies to WKT and PROJ.4 output)  
    &nbsp;&nbsp;&nbsp;&nbsp;**w**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information in WKT format [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=wkt instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use ESRI-style format (applies to WKT output only)  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Force override of datum transformation information in input coordinate system  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Modify current project's projection files  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **georef** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of georeferenced data file to read projection information from  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *file*  
    **wkt** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of ASCII file containing a WKT projection description  
    &nbsp;&nbsp;&nbsp;&nbsp;'-' for standard input  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *file*  
    **srid** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Spatial reference ID with authority name and code  
    &nbsp;&nbsp;&nbsp;&nbsp;E.g. EPSG:4326 or urn:ogc:def:crs:EPSG::4326  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *params*  
    **proj4** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;PROJ.4 projection description  
    &nbsp;&nbsp;&nbsp;&nbsp;'-' for standard input  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *params*  
    **epsg** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;EPSG projection code  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *code*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-1000000*  
    **list_codes** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;List codes for given authority, e.g. EPSG, and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *EPSG, ESRI, IAU_2015, IGNF, NKG, NRCAN, OGC, PROJ*  
    **datum** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Datum (overrides any datum specified in input coordinate system)  
    &nbsp;&nbsp;&nbsp;&nbsp;Accepts standard GRASS datum codes, or "list" to list and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    **datum_trans** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Index number of datum transform parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;"0" for unspecified or "-1" to list and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *index*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *-1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **project** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of new project (location) to create  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, shell, wkt, projjson, proj4*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**wkt**: Well-known text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**projjson**: JSON (JavaScript Object Notation) version of WKT  
    &nbsp;&nbsp;&nbsp;&nbsp;**proj4**: PROJ.4 style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *p*, *g*, *d*, *j*, *f*, *w*, *e*, *t*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Verify datum information and print transformation parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;**j**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information in PROJ.4 format [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=proj4 instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print 'flat' output with no linebreaks (applies to WKT and PROJ.4 output)  
    &nbsp;&nbsp;&nbsp;&nbsp;**w**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print projection information in WKT format [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=wkt instead.  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use ESRI-style format (applies to WKT output only)  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Force override of datum transformation information in input coordinate system  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Modify current project's projection files  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*g.proj* provides a means of converting a coordinate reference system
(CRS) description between various formats.

For an introduction to map projections (with PROJ),see the manual page
of [r.proj](r.proj.md).

If compiled without [OGR](https://gdal.org/) present, the functionality
is limited to:

- Reporting the CRS information for the current project (previously
  called location), either in conventional GRASS (**-p** flag) or PROJ
  (**-j** flag) format
- Changing the datum, or reporting and modifying the datum
  transformation parameters, for the current project

When compiled with OGR, functionality is increased and allows output of
the CRS information in the Well-Known Text (WKT) format popularised by
PROJ and GDAL. In addition, if one of the parameters *georef*, *wkt*,
*proj4* or *epsg* is specified, rather than being read from the current
project, the CRS information is imported from an external source as
follows:

georef=*filename*  
*g.proj* attempts to invoke GDAL and OGR in turn to read a georeferenced
file *filename*. The CRS information will be read from this file. If the
file is not georeferenced or cannot be read, XY (unprojected) will be
used.

wkt=*filename* or **-**  
The file *filename* should contain a CRS description in WKT format with
or without line-breaks (e.g. a '.prj' file). If **-** is given for the
filename, the WKT description will be read from stdin rather than a
file.

proj4=*description* or **-**  
*description* should be a CRS description in [PROJ](https://proj.org/)
format, enclosed in quotation marks if there are any spaces. If **-** is
given for *description*, the PROJ description will be read from stdin
rather than as a directly-supplied command-line parameter.

epsg=*number*  
*number* should correspond to the index number of a valid co-ordinate
system in the [EPSG database](https://epsg.org/search/by-name). EPSG
code support is based upon a local copy of the GDAL CSV co-ordinate
system and datum information files, stored in the directory
`$GISBASE/etc/proj/ogr_csv`. These can be updated if necessary to
support future revisions of the EPSG database.

If datum information is incorrect or missing in the input co-ordinate
system definition (e.g. PROJ descriptions have very limited support for
specifying datum names), a GRASS datum abbreviation can instead be
supplied using the *datum* parameter. This will override any datum
contained in the input co-ordinate system, and discard any datum
transformation parameters. Enter datum=*list* to return a list of all
the datums supported by GRASS. Since any existing datum transformation
parameters will have been discarded, the *datumtrans* parameter should
in general always be used in conjunction with *datum*.

The **-p**, **-j**, **-w**, etc. flags are all functional when importing
CRS information from an external source, meaning that *g.proj* can be
used to convert between representations of the information. It is
**not** required that either the input or output be in GRASS format.

In addition however, if the **-c** flag is specified, *g.proj* will
create new GRASS CRS files (PROJ_INFO, PROJ_UNITS, WIND and
DEFAULT_WIND) based on the imported information. If the *project*
parameter is specified in addition to **-c**, then a new project will be
created. Otherwise the CRS information files in the current project will
be overwritten. The program will **not** warn before doing this.

The final mode of operation of *g.proj* is to report on the datum
information and datum transformation parameters associated with the
co-ordinate system. The **-d** flag will report a human-readable summary
of this.

## NOTES

If the input co-ordinate system contains a datum name but no
transformation parameters, and there is more than one suitable parameter
set available (according to the files datum.table and
datumtransform.table in `$GISBASE/etc/proj`), *g.proj* will check the
value of the *datumtrans* option and act according to the following:

- **-1:** List available parameter sets in a GUI-parsable (but also
  human-readable) format and exit.
- **0 (default):** Continue without specifying parameters - if used when
  creating a project, other GRASS modules will use the "default" (likely
  non-optimum) parameters for this datum if necessary in the future.
- **Any other number less than or equal to the number of parameter sets
  available for this datum:** Choose this parameter set and add it to
  the co-ordinate system description.

If the **-t** flag is specified, the module will attempt to change the
datum transformation parameters using one of the above two methods
**even if** a valid parameter set is already specified in the input
co-ordinate system. This can be useful to change the datum information
for an existing project.

Output is simply based on the input CRS information. g.proj does **not**
attempt to verify that the co-ordinate system thus described matches a
pre-defined existing system in use in the world. In particular, this
means there may be no authority names and codes in the WKT output.

WKT format shows the false eastings and northings in the projected unit
(e.g. meters, feet) but in PROJ format it should always be given in
meters.

PROJJSON format is a JSON version of the WKT format, see the [PROJJSON
specification](https://proj.org/en/stable/specifications/projjson.html)

The maximum size of input WKT or PROJ CRS descriptions is limited to
8000 bytes.

## EXAMPLES

### Print information

Print the CRS information for the current project:  

```sh
g.proj -p
```

Print the CRS information for the current project in PROJJSON format:

```sh
g.proj -p format=projjson
```

Print the CRS information for the current project in shell format:

```sh
g.proj -p format=shell
```

Print the CRS information for the current project in WKT format:

```sh
g.proj -p format=wkt
```

Print the CRS information for the current project in PROJ.4 format (deprecated):

```sh
g.proj -p format=proj4
```

### Create projection (PRJ) file

Create a '.prj' file in ESRI format corresponding to the current
project:  

```sh
g.proj -wef > irish_grid.prj
```

### Read CRS from file

Read the CRS information from a GeoTIFF file and print it in PROJ
format:  

```sh
g.proj -jf georef=ASTER_DEM20020508161837.tif
```

Convert the PROJ CRS description contained in a text file to WKT
format:  

```sh
cat proj4.description | g.proj -w proj4=-
```

### Create new project

Create a new project with the coordinate system referred to by EPSG code
4326 (Latitude-Longitude/WGS84), without explicitly specifying datum
transformation parameters:  

```sh
g.proj -c epsg=4326 project=latlong
```

Create a new project with the coordinate system referred to by EPSG code
3857 ([Pseudo-Mercator
Projection](https://spatialreference.org/ref/epsg/3857/))  

```sh
g.proj -c epsg=3857 project=google
```

Create a new project with the coordinate system referred to by EPSG code
29900 (Irish Grid), selecting datum transformation parameter set no.
2:  

```sh
# list available datums for EPSG code 29900
g.proj -t datumtrans=-1 epsg=29900
g.proj -c epsg=29900 datumtrans=2 project=irish_grid
```

Create a new project with the same coordinate system as the current
project, but forcing a change to datum transformation parameter set no.
1:  

```sh
g.proj -c project=newloc -t datumtrans=1
```

Create a new project with the coordinate system from a WKT definition
stored in a text file:  

```sh
g.proj -c wkt=irish_grid.prj project=irish_grid
```

Create a new project from a PROJ description, explicitly specifying a
datum and using the default datum transformation parameters:  

```sh
g.proj -c project=spain proj4="+proj=utm +zone=30 +ellps=intl" datum=eur50 datumtrans=0
```

### Using g.proj output for GDAL/OGR tools

Reproject external raster map to current GRASS project (does not always
make sense!) using the GDAL 'gdalwarp' tool. We recommend to use the
ERDAS/Img format and not to use the ESRI style of WKT:  

```sh
# example for 30x30 pixel resolution (enforce with -tr to avoid odd values)
gdalwarp -of HFA -tr 30 30 -t_srs "`g.proj -wf`" aster.img aster_tmerc.img
```

Reproject external vector map to current GRASS project using the OGR
'ogr2ogr' tool:  

```sh
ogr2ogr -t_srs "`g.proj -wf`" polbnda_italy_GB_ovest.shp polbnda_italy_LL.shp
```

## REFERENCES

[PROJ](https://proj.org): Projection/datum support library  
[GDAL raster library and toolset](https://gdal.org)  
[OGR vector library and toolset](https://gdal.org/)

### Further reading

- [ASPRS Grids and
  Datum](https://www.asprs.org/asprs-publications/grids-and-datums)
- [MapRef - The Collection of Map Projections and Reference Systems for
  Europe](https://mapref.org)
- [Projections Transform List](http://geotiff.maptools.org/proj_list/)
  (PROJ)

## SEE ALSO

*[m.proj](m.proj.md), [r.proj](r.proj.md), [v.proj](v.proj.md),
[r.import](r.import.md), [r.in.gdal](r.in.gdal.md),
[v.import](v.import.md), [v.in.ogr](v.in.ogr.md)*

## AUTHOR

Paul Kelly

## SOURCE CODE

Available at: [g.proj source code](https://github.com/OSGeo/grass/tree/main/general/g.proj)
([history](https://github.com/OSGeo/grass/commits/main/general/g.proj))  
Latest change: Sunday Feb 15 15:28:21 2026 in commit [5cee43f](https://github.com/OSGeo/grass/commit/5cee43f83d1d6e0f13eb1bf2a265c2512d8ba7be)
