---
name: v.what.rast
description: Uploads raster values at positions of vector points to the table.
keywords: [ vector, sampling, raster, position, querying, attribute table, surface information ]
---

# v.what.rast

Uploads raster values at positions of vector points to the table.

=== "Command line"

    **v.what.rast**
    [**-ip**]
    **map**=*name*
    [**layer**=*string*]
    [**type**=*string* [,*string*,...]]
    **raster**=*name*
    [**column**=*name*]
    [**where**=*sql_query*]
    [**separator**=*character*]
    **format**=*name*
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.what.rast map=name raster=name format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***v.what.rast***",
        **map**,
        **layer**=*"1"*,
        **type**=*"point"*,
        **raster**,
        **column**=*None*,
        **where**=*None*,
        **separator**=*"comma"*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("v.what.rast", map="name", raster="name", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_what_rast*(**map**,
        **layer**=*"1"*,
        **type**=*"point"*,
        **raster**,
        **column**=*None*,
        **where**=*None*,
        **separator**=*"comma"*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_what_rast(map="name", raster="name", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **map**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector points map for which to edit attributes  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **type**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, centroid*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point*  
    **raster**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map to be queried  
    **column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column to be updated with the query result  
    **where**=*sql_query*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    **separator**=*character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator for CSV style output  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *comma*  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-i**  
    &nbsp;&nbsp;&nbsp;&nbsp;Interpolate values from the nearest four cells  
    **-p**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print categories and values instead of updating the database  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector points map for which to edit attributes  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, centroid*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point*  
    **raster** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map to be queried  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column to be updated with the query result  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, sql_query, *sql_query*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator for CSV style output  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *comma*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *i*, *p*  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Interpolate values from the nearest four cells  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print categories and values instead of updating the database  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector points map for which to edit attributes  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, centroid*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point*  
    **raster** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map to be queried  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column to be updated with the query result  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, sql_query, *sql_query*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator for CSV style output  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *comma*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *i*, *p*  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Interpolate values from the nearest four cells  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print categories and values instead of updating the database  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.what.rast* retrieves raster value from a given raster map for each
point or centroid stored in a given vector map. It can update a
**column** in the linked vector attribute table with the retrieved
raster cell value or print it.

The column type needs to be numeric (integer, float, double, ...). If
the column doesn't exist in the vector attribute table than the module
will create the new column of type corresponding with the input raster
map.

If the **-p** flag is used, then the attribute table is not updated and
the results are printed to standard output.

If the **-i** flag is used, then the value to be uploaded to the
database is interpolated from the four nearest raster cells values using
an inverse distance weighting method (IDW). This is useful for cases
when the vector point density is much higher than the raster cell size.

## NOTES

Points and centroid with shared category number cannot be processed. To
solved this, unique categories may be added with
*[v.category](v.category.md)* in a separate layer.

If multiple points have the same category, the attribute value is set to
NULL. If the raster value is NULL, then attribute value is set to NULL.

*v.what.rast* operates on the attribute table. To modify the vector
geometry instead, use *[v.drape](v.drape.md)*.

Categories and values are output unsorted with the print flag. To sort
them pipe the output of this module into the UNIX `sort` tool
(`sort -n`). If you need coordinates, after sorting use
*[v.out.ascii](v.out.ascii.md)* and the UNIX `paste` tool
(`paste -d'|'`). In the case of a NULL result, a "`*`" will be printed
in lieu of the value.

The interpolation flag is only useful for continuous value raster maps,
if a categorical raster is given as input the results will be nonsense.
Since the search window is limited to four raster cells there may still
be raster cell-edge artifacts visible in the results, this compromise
has been made for processing speed. If one or more of the nearest four
raster cells is NULL, then only the raster cells containing values will
be used in the weighted average.

## EXAMPLES

### Transferring raster values into existing attribute table of vector points map

Reading values from raster map at position of vector points, writing
these values into a column of the attribute table connected to the
vector map:

```sh
# work on copy of original geodetic points map
g.copy vector=geodetic_pts,mygeodetic_pts

# set computational region to raster map to be queried
g.region raster=elev_state_500m -p

# query raster cells (a new column will be added to existing table)
v.what.rast map=mygeodetic_pts raster=elev_state_500m column=height

# compare official geodetic heights to those of elevation model
v.db.select map=mygeodetic_pts columns=Z_VALUE,height separator=comma
```

### Transferring raster values into new vector points map

In case of a vector map without attached attribute table, first add a
new attribute table. This table is then populated with values queried
from the raster map:

```sh
# create new random vector points map
v.random pnts n=100

# add new table, link to map
v.db.addtable map=pnts column="height double precision"

# set computational region to raster map to be queried
g.region raster=elevation -p
# query raster map and upload values to vector table into specified column
v.what.rast map=pnts raster=elevation column=height

# verify new attribute table:
v.db.select pnts

# verify statistics of uploaded values:
v.univar map=pnts column=height type=point
```

### Print categories and values using Pandas

```python
import grass.script as gs
import pandas as pd

data = gs.parse_command(
    "v.what.rast", map="bridges", raster="elevation", flags="p", format="json"
)

df = pd.DataFrame(data)
print(df)
```

Possible output:

```text
    category       value
0       4635   86.851212
1       4648   87.883255
2       4688  103.382004
3       4709   67.387856
4       4986   90.972832
5       4991   86.550629
6       5027   84.256355
7       5060   79.500443
8       5071   78.732864
9       5089   85.703636
10      5096  115.967323
11      5130   81.055870
12      5150   92.812927
13      5166   88.149483
14      5184   76.426331
```

The JSON output looks like (trimmed):

```json
[
    {
        "category": 4635,
        "value": 86.851211547851562
    },
    {
        "category": 4648,
        "value": 87.883255004882812
    },
    ...
]
```

## SEE ALSO

*[v.category](v.category.md), [v.db.addtable](v.db.addtable.md),
[v.db.select](v.db.select.md), [v.drape](v.drape.md),
[v.univar](v.univar.md), [v.rast.stats](v.rast.stats.md),
[v.what.vect](v.what.vect.md)*

## AUTHORS

Radim Blazek  
Hamish Bowman (interpolation)

## SOURCE CODE

Available at: [v.what.rast source code](https://github.com/OSGeo/grass/tree/main/vector/v.what.rast)
([history](https://github.com/OSGeo/grass/commits/main/vector/v.what.rast))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
