---
name: r.carve
description: Generates stream channels. Takes vector stream data, transforms it to raster and subtracts depth from the output DEM.
keywords: [ raster, hydrology ]
---

# r.carve

Generates stream channels.

Takes vector stream data, transforms it to raster and subtracts depth from the output DEM.

=== "Command line"

    **r.carve**
    [**-n**]
    **raster**=*name*
    **vector**=*name*
    **output**=*name*
    [**points**=*name*]
    [**width**=*float*]
    [**depth**=*float*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.carve raster=name vector=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.carve***",
        **raster**,
        **vector**,
        **output**,
        **points**=*None*,
        **width**=*None*,
        **depth**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.carve", raster="name", vector="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_carve*(**raster**,
        **vector**,
        **output**,
        **points**=*None*,
        **width**=*None*,
        **depth**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_carve(raster="name", vector="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **raster**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster elevation map  
    **vector**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map containing stream(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **points**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map for adjusted stream points  
    **width**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Stream width (in meters)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is raster cell width  
    **depth**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Additional stream depth (in meters)  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;No flat areas allowed in flow direction  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **raster** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster elevation map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **vector** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map containing stream(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map for adjusted stream points  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **width** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Stream width (in meters)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is raster cell width  
    **depth** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Additional stream depth (in meters)  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;No flat areas allowed in flow direction  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **raster** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster elevation map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **vector** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map containing stream(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map for adjusted stream points  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **width** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Stream width (in meters)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is raster cell width  
    **depth** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Additional stream depth (in meters)  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;No flat areas allowed in flow direction  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.carve* accepts vector stream data as input, transforms them to
raster, and subtracts a default-depth + additional-depth from a DEM. If
the given width is more than 1 cell, it will carve the stream with the
given width. With the **-n** flag it should eliminate all flat cells
within the stream, so when and if the water gets into the stream it will
flow. The *points* option generates x,y,z for points which define the
stream with the z-value of the bottom of the carved-in stream. These
points can then be combined with contours to interpolate a new DEM with
better representation of valleys.

## NOTES

*r.carve* does not create a depressionless DEM because many depressions
are in flat areas and not in the streams.

## EXAMPLE

North Carolina sample dataset:

```sh
# set computational region
g.region raster=elev_lid792_1m -p

# digitize a ditch for the farm pond
echo "L  3 1
 638692.93595422 220198.90026383
 638737.42270627 220149.74706926
 638984.43306379 220148.19158842
 1     1" | v.in.ascii -n input=- output=ditch format=standard

# visualize original data
d.mon wx0
d.rast elev_lid792_1m
d.vect ditch

# carve
r.carve raster=elev_lid792_1m vector=ditch output=carved_dem width=3 depth=0.5

# visualize resulting carved DEM map
d.rast carved_dem

# visualize
r.relief input=elev_lid792_1m output=elev_lid792_1m_shaded
r.relief input=carved_dem output=carved_dem_shaded
d.rast elev_lid792_1m_shaded
d.erase
d.rast carved_dem_shaded

# flow accumulation
r.watershed elevation=elev_lid792_1m accumulation=elev_lid792_1m_accum
r.watershed elevation=carved_dem accumulation=carved_dem_accum
d.rast elev_lid792_1m_accum
d.erase
d.rast carved_dem_accum

# differences
r.mapcalc "accum_diff = elev_lid792_1m_accum - carved_dem_accum"
r.colors accum_diff color=differences
d.erase
d.rast accum_diff
```

![Original DEM with vector streams](r_carve_dem_orig.png)  
*Fig: Original 1m LiDAR based DEM with vector streams map on top*

![Original DEM shaded](r_carve_dem_orig_shaded.png)  
*Fig: Original 1m LiDAR based DEM shown as shaded terrain*

![Carved DEM](r_carve_dem_carved.png)  
*Fig: Carved 1m LiDAR based DEM*

![Carved DEM shaded](r_carve_dem_carved_shaded.png)  
*Fig: Carved 1m LiDAR based DEM shown as shaded terrain*

![Flow accumulation in original DEM](r_carve_dem_orig_accum.png)  
*Fig: Flow accumulation in original 1m LiDAR based DEM*

![Flow accumulation in carved DEM](r_carve_dem_carved_accum.png)  
*Fig: Flow accumulation in carved 1m LiDAR based DEM*

## KNOWN ISSUES

The module does not operate yet in latitude-longitude coordinate
reference system.

## REFERENCES

[Terrain modeling and Soil Erosion Simulations for Fort Hood and Fort
Polk test
areas](https://web.archive.org/web/20240310015553/http://fatra.cnr.ncsu.edu/~hmitaso/gmslab/reports/cerl99/rep99.html),
by Helena Mitasova, Lubos Mitas, William M. Brown, Douglas M. Johnston,
GMSL (Report for CERL 1999)

## SEE ALSO

*[r.flow](r.flow.md), [r.fill.dir](r.fill.dir.md),
[r.watershed](r.watershed.md)*

## AUTHORS

Bill Brown (GMSL)  
GRASS 6 update: Brad Douglas

## SOURCE CODE

Available at: [r.carve source code](https://github.com/OSGeo/grass/tree/main/raster/r.carve)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.carve))  
Latest change: Thursday Feb 13 11:55:34 2025 in commit [ffc0b8c](https://github.com/OSGeo/grass/commit/ffc0b8c9c71c2c78415bb0db0bd9d7daea963353)
