---
name: r.regression.line
description: Calculates linear regression from two raster maps: y = a + b*x.
keywords: [ raster, statistics, regression ]
---

# r.regression.line

Calculates linear regression from two raster maps: y = a + b*x.

=== "Command line"

    **r.regression.line**
    [**-g**]
    **mapx**=*name*
    **mapy**=*name*
    [**output**=*name*]
    **format**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.regression.line mapx=name mapy=name format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***r.regression.line***",
        **mapx**,
        **mapy**,
        **output**=*None*,
        **format**=*"plain"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("r.regression.line", mapx="name", mapy="name", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_regression_line*(**mapx**,
        **mapy**,
        **output**=*None*,
        **format**=*"plain"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_regression_line(mapx="name", mapy="name", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **mapx**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for x coefficient  
    **mapy**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for y coefficient  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **mapx** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for x coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mapy** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for y coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **mapx** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for x coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mapy** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for y coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.regression.line* calculates a linear regression from two raster maps,
according to the formula

```sh
y = a + b*x
```

where

```sh
x
y
```

represent the input raster maps.

Optionally, it saves regression coefficients as a ASCII file. The result
includes the following coefficients: offset/intercept (a) and gain/slope
(b), correlation coefficient (R), number of elements (N), means (medX,
medY), standard deviations (sdX, sdY), and the F test for testing the
significance of the regression model as a whole (F).

## NOTES

The results for offset/intercept (a) and gain/slope (b) are identical to
that obtained from R-stats's lm() function.

## EXAMPLE

Comparison of two DEMs (SRTM and NED, both at 30m resolution), provided
in the North Carolina sample dataset:

```sh
g.region raster=elev_srtm_30m -p
r.regression.line mapx=elev_ned_30m mapy=elev_srtm_30m
 y = a + b*x
   a (Offset): -1.659279
   b (Gain): 1.043968
   R (sumXY - sumX*sumY/N): 0.894038
   N (Number of elements): 225000
   F (F-test significance): 896093.366283
   meanX (Mean of map1): 110.307571
   sdX (Standard deviation of map1): 20.311998
   meanY (Mean of map2): 113.498292
   sdY (Standard deviation of map2): 23.718307
```

Using the shell format option AND *eval* to make results available in the
shell:

```sh
g.region raster=elev_srtm_30m -p
eval `r.regression.line mapx=elev_ned_30m mapy=elev_srtm_30m format=shell`

# print result stored in respective variables
echo $a
-1.659279

echo $b
1.043968

echo $R
0.894038
```

Using the JSON format option and Python to parse the output:

```python
import grass.script as gs

data = gs.parse_command(
    "r.regression.line", mapx="elev_ned_30m", mapy="elev_srtm_30m", format="json"
)
print(data)
```

Possible JSON Output:

```json
{
 "a": -1.6592786233805945,
 "b": 1.0439679629649166,
 "R": 0.8940383063008781,
 "N": 225000,
 "F": 896093.366283,
 "x_mean": 110.30757108713786,
 "x_stddev": 20.311997672696272,
 "y_mean": 113.49829166406644,
 "y_stddev": 23.718306793642626
}
```

## SEE ALSO

*[d.correlate](d.correlate.md),
[r.regression.multi](r.regression.multi.md), [r.stats](r.stats.md)*

## AUTHORS

Dr. Agustin Lobo - alobo at ija.csic.es  
Updated to GRASS 5.7 Michael Barton, Arizona State University  
Script style output Markus Neteler  
Conversion to C module Markus Metz

## SOURCE CODE

Available at: [r.regression.line source code](https://github.com/OSGeo/grass/tree/main/raster/r.regression.line)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.regression.line))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
